/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Michael Nimz <michael.nimz@open-xchange.com>
 */

define([
    'globals/apphelper',
    'globals/sheethelper',
    'io.ox/office/spreadsheet/model/tablecollection'
], function (AppHelper, SheetHelper, TableCollection) {

    'use strict';

    // convenience shortcuts
    var a = SheetHelper.a;
    var r = SheetHelper.r;

    describe('Spreadsheet', function () {

        var OPERATIONS = [
            { name: 'setDocumentAttributes',  attrs: { document: { cols: 16384, rows: 1048576 } } },
            { name: 'insertAutoStyle', type: 'cell', styleId: 'a0', attrs: {}, default: true },
            { name: 'insertSheet', sheet: 0, sheetName: 'Sheet1' },
            { name: 'changeCells', sheet: 0, start: 'B2', contents: [
                ['Name',    'Age', 'City'],
                ['Miri',    30,    'Hamburg'],
                ['Michael', 32,    'Hamburg'],
                ['Dennis',  32,    'Heide'],
                ['Horst',   33,    'Ostrohe'],
                [null, { f: 'SUM(C3:C6)/4', v: 31.75 }]
            ] },
            { name: 'insertTable', sheet: 0, table: 'Table1', start: 'B2', end: 'D7', attrs: { styleId: 'TableStyleMedium9', table: { headerRow: true, footerRow: true, bandsHor: true } } },
            { name: 'changeTableColumn', sheet: 0, table: 'Table1', col: 1, attrs: { filter: { type: 'discrete', entries: ['32', '33'] } } },
            { name: 'changeTableColumn', sheet: 0, table: 'Table1', col: 2, attrs: { sort: { type: 'value' } } }
        ];

        // initialize test document
        var tableCollection = null, tableModel = null, col1Model = null, col2Model = null, col3Model = null;
        AppHelper.createSpreadsheetApp('ooxml', OPERATIONS).done(function (app) {
            tableCollection = app.getModel().getSheetModel(0).getTableCollection();
            tableModel = tableCollection.getTable('Table1');
            col1Model = tableModel.getColumnModel(0);
            col2Model = tableModel.getColumnModel(1);
            col3Model = tableModel.getColumnModel(2);
        });

        // class TableCollection ==============================================

        describe('class TableCollection', function () {

            it('should exist', function () {
                expect(TableCollection).to.be.a('function');
            });

            describe('method "hasTable"', function () {
                it('should exist', function () {
                    expect(tableCollection).to.respondTo('hasTable');
                });
                it('should return the correct boolean-state for asking a tableName', function () {
                    expect(tableCollection.hasTable('Table1')).to.be.equal(true);
                    expect(tableCollection.hasTable('Table2')).to.be.equal(false);
                });
            });

            describe('method "hasAutoFilter"', function () {
                it('should exist', function () {
                    expect(tableCollection).to.respondTo('hasAutoFilter');
                });
                it('should return whether the collection contains a AutoFilter-table', function () {
                    expect(tableCollection.hasAutoFilter()).to.be.equal(false);
                });
            });

            describe('method "getTable"', function () {
                it('should exist', function () {
                    expect(tableCollection).to.respondTo('getTable');
                });
                it('should return the table-model for the given table-name', function () {
                    expect(tableCollection.getTable('Table1')).to.be.an('object');
                });
            });

            describe('method "getAutoFilter"', function () {
                it('should exist', function () {
                    expect(tableCollection).to.respondTo('getAutoFilter');
                });
                it('should return the table-model of the autoFilter-Table', function () {
                    var autoFilterTable = tableCollection.getAutoFilter();
                    expect(autoFilterTable).to.be.equal(null);
                });
            });

            describe('method "getAllTables"', function () {
                it('should exist', function () {
                    expect(tableCollection).to.respondTo('getAllTables');
                });
                it('should return all table-models (without autoFilter-tables)', function () {
                    var arrTables = tableCollection.getAllTables();
                    expect(arrTables).to.have.lengthOf(1);
                });
                it('should return all table-models (with the autoFilter-table)', function () {
                    var arrTables = tableCollection.getAllTables({ autoFilter: true });
                    expect(arrTables).to.have.lengthOf(1);
                });
            });

            describe('method "findTable"', function () {
                it('should exist', function () {
                    expect(tableCollection).to.respondTo('findTable');
                });
                it('should find a table-model for the given address', function () {
                    var result = tableCollection.findTable(a('A1'));
                    expect(result).to.be.equal(null);
                });
                it('should find a table-model for the given address', function () {
                    var result = tableCollection.findTable(a('B4'));
                    expect(result).to.be.an('object');
                });
            });

            describe('method "findTables"', function () {
                it('should exist', function () {
                    expect(tableCollection).to.respondTo('findTables');
                });
                it('should find table-models for the given ranges', function () {
                    var result = tableCollection.findTables(r('A1:C3'));
                    expect(result).to.be.an('array');
                    expect(result).to.have.lengthOf(1);
                });
            });

            describe('method "generateInsertTableOperations"', function () {
                it('should exist', function () {
                    expect(tableCollection).to.respondTo('generateInsertTableOperations');
                });
            });

            describe('method "generateMissingHeaderLabelOperations"', function () {
                it('should exist', function () {
                    expect(tableCollection).to.respondTo('generateMissingHeaderLabelOperations');
                });
            });

        });

        // class TableModel ===================================================

        describe('class TableModel', function () {

            describe('method "isRefreshable"', function () {
                it('should exist', function () {
                    expect(tableModel).to.respondTo('isRefreshable');
                });
                it('should return whether the table is refreshable or not', function () {
                    expect(tableModel.isRefreshable()).to.be.equal(true);
                });
            });

            describe('method "getColumnContents"', function () {
                it('should exist', function () {
                    expect(tableModel).to.respondTo('getColumnContents');
                });
                it('should return a array with the content of the given column', function () {
                    expect(tableModel.getColumnContents(0)).to.have.lengthOf(4);
                });
            });

            describe('method "getColumnModel"', function () {
                it('should exist', function () {
                    expect(tableModel).to.respondTo('getColumnModel');
                });
                it('should return the column model', function () {
                    expect(tableModel.getColumnModel(0)).to.be.an('object');
                });
            });

            describe('method "getSortPrefillOptions"', function () {
                it('should exist', function () {
                    expect(tableModel).to.respondTo('getSortPrefillOptions');
                });
                it('should return the options needed for prefilling the custom sort dialog', function () {
                    var prefillOptions = tableModel.getSortPrefillOptions();
                    expect(prefillOptions).to.be.an('object');
                    expect(prefillOptions).to.have.property('headers', true);
                    expect(prefillOptions).to.have.property('orderRules');
                });
            });

            describe('method "isSorted"', function () {
                it('should exist', function () {
                    expect(tableModel).to.respondTo('isSorted');
                });
                it('should return the boolean sort-state of the table', function () {
                    expect(tableModel.isSorted()).to.be.equal(true);
                });
            });

            describe('method "getSortedColumns"', function () {
                it('should exist', function () {
                    expect(tableModel).to.respondTo('getSortedColumns');
                });
                it('should return a array with the sorted columns', function () {
                    var sortedColumns = tableModel.getSortedColumns();
                    expect(sortedColumns).to.be.an('array').with.length(1);
                });
            });
        });

        // class TableColumnModel =============================================

        describe('class TableColumnModel', function () {

            describe('method "isFiltered"', function () {
                it('should exist', function () {
                    expect(col1Model).to.respondTo('isFiltered');
                });
                it('should return whether the given column is filtered or not', function () {
                    expect(col1Model.isFiltered()).to.be.equal(false);
                    expect(col2Model.isFiltered()).to.be.equal(true);
                    expect(col3Model.isFiltered()).to.be.equal(false);
                });
            });

            describe('method "isSorted"', function () {
                it('should exist', function () {
                    expect(col1Model).to.respondTo('isSorted');
                });
                it('should return whether the given column is sorted or not', function () {
                    expect(col1Model.isSorted()).to.be.equal(false);
                    expect(col2Model.isSorted()).to.be.equal(false);
                    expect(col3Model.isSorted()).to.be.equal(true);
                });
            });

            describe('method "isRefreshable"', function () {
                it('should exist', function () {
                    expect(col1Model).to.respondTo('isRefreshable');
                });
                it('should return whether the table is refreshable or not', function () {
                    expect(col1Model.isRefreshable()).to.be.equal(false);
                    expect(col2Model.isRefreshable()).to.be.equal(true);
                    expect(col3Model.isRefreshable()).to.be.equal(true);
                });
            });
        });

        // ====================================================================
    });
});
