/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Michael Nimz <michael.nimz@open-xchange.com>
 */

define([
    'globals/apphelper',
    'globals/sheethelper',
    'io.ox/office/spreadsheet/utils/sheetutils',
    'io.ox/office/spreadsheet/model/mergecollection'
], function (AppHelper, SheetHelper, SheetUtils, MergeCollection) {

    'use strict';

    // convenience shortcuts
    var a = SheetHelper.a;
    var r = SheetHelper.r;
    var ra = SheetHelper.ra;
    var MergeMode = SheetUtils.MergeMode;
    var Address = SheetUtils.Address;

    // class MergeCollection ==================================================

    describe('Spreadsheet class MergeCollection', function () {

        it('should exist', function () {
            expect(MergeCollection).to.be.a('function');
        });

        // the operations to be applied by the document model
        var charAttrs    = { fontName: 'Arial', fontSize: 11, bold: false, italic: false, underline: false, strike: 'none', color: { type: 'scheme', value: 'dark1' } },
            noBorder     = { style: 'none' },
            singleBorder = { style: 'single', width: 26, color: { type: 'auto' } },
            doubleBorder = { style: 'double', width: 78, color: { type: 'auto' } },
            applyAttrs1  = { font: true, fill: true, border: false, align: false, number: false, protect: false },
            applyAttrs2  = { font: true, fill: true, border: true, align: false, number: false, protect: false };

        var OPERATIONS = [
            { name: 'setDocumentAttributes', attrs: { document: { cols: 16384, rows: 1048576 } } },
            { name: 'insertAutoStyle', type: 'cell', styleId: 'a0', attrs: {}, default: true },
            { name: 'insertAutoStyle', type: 'cell', styleId: 'a1', attrs: { character: { fontName: 'Arial', fontSize: 13, bold: true, italic: false, underline: false, strike: 'none', color: { type: 'scheme', value: 'dark1' } }, apply: applyAttrs1, cell: { alignHor: 'auto', alignVert: 'bottom', wrapText: false, formatId: 0, fillColor: { type: 'rgb', value: '00B050' }, borderLeft: noBorder, borderRight: noBorder, borderTop: noBorder, borderBottom: noBorder, borderDown: noBorder, borderUp: noBorder, unlocked: false, hidden: false }, styleId: 'Standard' } },
            { name: 'insertAutoStyle', type: 'cell', styleId: 'a2', attrs: { character: { fontName: 'Arial', fontSize: 13, bold: true, italic: false, underline: false, strike: 'none', color: { type: 'scheme', value: 'dark1' } }, apply: applyAttrs2, cell: { alignHor: 'auto', alignVert: 'bottom', wrapText: false, formatId: 0, fillColor: { type: 'rgb', value: '00B050' }, borderLeft: noBorder, borderRight: noBorder, borderTop: noBorder, borderBottom: noBorder, borderDown: noBorder, borderUp: noBorder, unlocked: false, hidden: false }, styleId: 'Standard' } },
            // border auto-styles
            // A3 - autostyle: border top/left corner
            { name: 'insertAutoStyle', type: 'cell', styleId: 'a3', attrs: { character: charAttrs, apply: applyAttrs2, cell: { alignHor: 'auto', alignVert: 'bottom', wrapText: false, formatId: 0, fillColor: { type: 'rgb', value: '00B050' },
                borderLeft:     singleBorder,
                borderRight:    noBorder,
                borderTop:      singleBorder,
                borderBottom:   noBorder,
                borderDown:     noBorder,
                borderUp:       noBorder,
                unlocked: false, hidden: false }, styleId: 'Standard' } },
            // A4 - autostyle: border top/right corner
            { name: 'insertAutoStyle', type: 'cell', styleId: 'a4', attrs: { character: charAttrs, apply: applyAttrs2, cell: { alignHor: 'auto', alignVert: 'bottom', wrapText: false, formatId: 0, fillColor: { type: 'rgb', value: '00B050' },
                borderLeft:     noBorder,
                borderRight:    singleBorder,
                borderTop:      singleBorder,
                borderBottom:   noBorder,
                borderDown:     noBorder,
                borderUp:       noBorder,
                unlocked: false, hidden: false }, styleId: 'Standard' } },
            // A5 - autostyle: border bottom/left corner
            { name: 'insertAutoStyle', type: 'cell', styleId: 'a5', attrs: { character: charAttrs, apply: applyAttrs2, cell: { alignHor: 'auto', alignVert: 'bottom', wrapText: false, formatId: 0, formatCode: 'General', fillColor: { type: 'rgb', value: '00B050' },
                borderLeft:     singleBorder,
                borderRight:    noBorder,
                borderTop:      noBorder,
                borderBottom:   singleBorder,
                borderDown:     noBorder,
                borderUp:       noBorder,
                unlocked: false, hidden: false }, styleId: 'Standard' } },
            // A6 - autostyle: border bottom/right corner
            { name: 'insertAutoStyle', type: 'cell', styleId: 'a6', attrs: { character: charAttrs, apply: applyAttrs2, cell: { alignHor: 'auto', alignVert: 'bottom', wrapText: false, formatId: 0, fillColor: { type: 'rgb', value: '00B050' },
                borderLeft:     noBorder,
                borderRight:    singleBorder,
                borderTop:      noBorder,
                borderBottom:   singleBorder,
                borderDown:     noBorder,
                borderUp:       noBorder,
                unlocked: false, hidden: false }, styleId: 'Standard' } },
            // A7 - autostyle: border top
            { name: 'insertAutoStyle', type: 'cell', styleId: 'a7', attrs: { character: charAttrs, apply: applyAttrs2, cell: { alignHor: 'auto', alignVert: 'bottom', wrapText: false, formatId: 0, fillColor: { type: 'rgb', value: '00B050' },
                borderLeft:     noBorder,
                borderRight:    noBorder,
                borderTop:      singleBorder,
                borderBottom:   noBorder,
                borderDown:     noBorder,
                borderUp:       noBorder,
                unlocked: false, hidden: false }, styleId: 'Standard' } },
            // A8 - autostyle: border right
            { name: 'insertAutoStyle', type: 'cell', styleId: 'a8', attrs: { character: charAttrs, apply: applyAttrs2, cell: { alignHor: 'auto', alignVert: 'bottom', wrapText: false, formatId: 0, fillColor: { type: 'rgb', value: '00B050' },
                borderLeft:     noBorder,
                borderRight:    singleBorder,
                borderTop:      noBorder,
                borderBottom:   noBorder,
                borderDown:     noBorder,
                borderUp:       noBorder,
                unlocked: false, hidden: false }, styleId: 'Standard' } },
            // A9 - autostyle: border bottom
            { name: 'insertAutoStyle', type: 'cell', styleId: 'a9', attrs: { character: charAttrs, apply: applyAttrs2, cell: { alignHor: 'auto', alignVert: 'bottom', wrapText: false, formatId: 0, fillColor: { type: 'rgb', value: '00B050' },
                borderLeft:     noBorder,
                borderRight:    noBorder,
                borderTop:      noBorder,
                borderBottom:   singleBorder,
                borderDown:     noBorder,
                borderUp:       noBorder,
                unlocked: false, hidden: false }, styleId: 'Standard' } },
            // A10 - autostyle: border left
            { name: 'insertAutoStyle', type: 'cell', styleId: 'a10', attrs: { character: charAttrs, apply: applyAttrs2, cell: { alignHor: 'auto', alignVert: 'bottom', wrapText: false, formatId: 0, fillColor: { type: 'rgb', value: '00B050' },
                borderLeft:     singleBorder,
                borderRight:    noBorder,
                borderTop:      noBorder,
                borderBottom:   noBorder,
                borderDown:     noBorder,
                borderUp:       noBorder,
                unlocked: false, hidden: false }, styleId: 'Standard' } },
            // A11 - no border
            { name: 'insertAutoStyle', type: 'cell', styleId: 'a11', attrs: { character: charAttrs, apply: applyAttrs2, cell: { alignHor: 'auto', alignVert: 'bottom', wrapText: false, formatId: 0, fillColor: { type: 'rgb', value: '00B050' },
                borderLeft:     noBorder,
                borderRight:    noBorder,
                borderTop:      noBorder,
                borderBottom:   noBorder,
                borderDown:     noBorder,
                borderUp:       noBorder,
                unlocked: false, hidden: false }, styleId: 'Standard' } },
            // A12 - autostyle: special border top
            { name: 'insertAutoStyle', type: 'cell', styleId: 'a12', attrs: { character: charAttrs, apply: { font: true, fill: true, border: true, align: false, number: false, protect: false }, cell: { alignHor: 'auto', alignVert: 'bottom', wrapText: false, formatId: 0, fillColor: { type: 'rgb', value: '00B050' },
                borderLeft:     noBorder,
                borderRight:    noBorder,
                borderTop:      doubleBorder,
                borderBottom:   noBorder,
                borderDown:     noBorder,
                borderUp:       noBorder,
                unlocked: false, hidden: false }, styleId: 'Standard' } },

            { name: 'insertSheet', sheet: 0, sheetName: 'Sheet1' },
            { name: 'changeCells', sheet: 0, start: 'A5', contents: [
                [{ v: 1, s: 'a1' }, 4, 'a'],
                [2, { v: 5, s: 'a2' }, 'b'],
                [3, 6, { v: 'c', s: 'a1' }]
            ] },
            { name: 'changeCells', sheet: 0, start: 'E5', contents: [
                [{ s: 'a1' }, null, 'a'],
                [null, { v: 5, s: 'a2' }, 'b'],
                [3, 6, { v: 'c', s: 'a1' }]
            ] },

            { name: 'insertSheet', sheet: 1, sheetName: 'Sheet2' },
            { name: 'mergeCells', sheet: 1, start: 'A5', end: 'C7' },
            { name: 'mergeCells', sheet: 1, start: 'E5', end: 'G7' },
            { name: 'changeCells', sheet: 1, start: 'E5', contents: [
                [{ s: 'a1' }, { s: 'a1' }, { s: 'a1' }],
                [{ s: 'a1' }, { s: 'a1' }, { s: 'a1' }],
                [{ s: 'a1' }, { s: 'a1' }, { s: 'a1' }]
            ] },

            { name: 'insertSheet', sheet: 2, sheetName: 'Sheet4' },
            { name: 'changeCells', sheet: 2, start: 'E5', contents: [
                [{ v: 1, s: 'a3'  }, { v: 4, s: 'a7' }, { v: 'a', s: 'a4' }],
                [{ v: 2, s: 'a10' }, { v: 5          }, { v: 'b', s: 'a8' }],
                [{ v: 3, s: 'a5'  }, { v: 6, s: 'a9' }, { v: 'c', s: 'a6' }]
            ] },

            { name: 'insertSheet', sheet: 3, sheetName: 'Sheet5' },
            { name: 'changeCells', sheet: 3, start: 'E5', contents: [
                [{ v: 1, s: 'a3'  }, { v: 4, s: 'a12' }, { v: 'a', s: 'a4' }],
                [{ v: 2, s: 'a10' }, { v: 5           }, { v: 'b', s: 'a8' }],
                [{ v: 3, s: 'a5'  }, { v: 6, s: 'a9'  }, { v: 'c', s: 'a6' }]
            ] },

            { name: 'insertSheet', sheet: 4, sheetName: 'Sheet6' },
            { name: 'changeCells', sheet: 4, start: 'E5', contents: [
                [{ v: 1, s: 'a3'  }, { v: 4, s: 'a7'  }, { v: 'a', s: 'a4' }],
                [{ v: 2, s: 'a10' }, { v: 5           }, { v: 'b', s: 'a8' }],
                [{ v: 3, s: 'a10' }, { v: 6, s: 'a10' }, { v: 'c', s: 'a8' }],
                [{ v: 4, s: 'a10' }, { v: 7           }, { v: 'd', s: 'a8' }],
                [{ v: 5, s: 'a5'  }, { v: 8, s: 'a9'  }, { v: 'e', s: 'a6' }]
            ] }
        ];

        // initialize test document
        var docModel = null;
        AppHelper.createSpreadsheetApp('ooxml', OPERATIONS).done(function (app) {
            docModel = app.getModel();
        });

        function getMergeCollection(sheet) {
            return docModel.getSheetModel(sheet).getMergeCollection();
        }

        function expectMergedRanges(sheet, ranges) {
            var mergedRanges = getMergeCollection(sheet).getMergedRanges(r('A1:XFD1048576')).sort();
            expect(mergedRanges).to.stringifyTo(ranges);
        }

        function getCellCollection(sheet) {
            return docModel.getSheetModel(sheet).getCellCollection();
        }

        function expectValueMatrix(sheet, start, values) {
            var cellCollection = getCellCollection(sheet);
            start = a(start);
            values.forEach(function (vector, row) {
                vector.forEach(function (value, col) {
                    var address = new Address(start[0] + col, start[1] + row);
                    expect(cellCollection.getValue(address)).to.equal(value);
                });
            });
        }

        function expectStyleMatrix(sheet, start, styles) {
            var cellCollection = getCellCollection(sheet);
            start = a(start);
            styles.forEach(function (vector, row) {
                vector.forEach(function (style, col) {
                    var address = new Address(start[0] + col, start[1] + row);
                    expect(cellCollection.getStyleId(address)).to.equal(style);
                });
            });
        }

        // public methods -----------------------------------------------------

        describe('method "_getCloneData"', function () {
            it('should exist', function () {
                expect(getMergeCollection(0)).to.respondTo('_getCloneData');
            });
            it('should return the internal map', function () {
                var cloneData = getMergeCollection(0)._getCloneData();
                expect(cloneData).to.have.a.property('mergedRangeSet').that.is.an.instanceof(SheetUtils.RangeSet);
            });
        });

        describe('method "generateMergeCellsOperations"', function () {
            it('should exist', function () {
                expect(getMergeCollection(0)).to.respondTo('generateMergeCellsOperations');
            });

            it('should merge cell ranges and move first value cells', function (done) {
                var sheetModel = docModel.getSheetModel(0);
                var mergeCollection = sheetModel.getMergeCollection();
                var generator = sheetModel.createOperationGenerator({ applyImmediately: true });
                var promise = mergeCollection.generateMergeCellsOperations(generator, ra('A5:C7 E5:G7'), MergeMode.MERGE);
                promise.always(function () {
                    expect(promise.state()).to.equal('resolved');
                    expectMergedRanges(0, 'A5:C7,E5:G7');
                    expectValueMatrix(0, 'A5', [[1, null, null], [null, null, null], [null, null, null]]);
                    expectStyleMatrix(0, 'A5', [['a1', 'a1', 'a1'], ['a1', 'a1', 'a1'], ['a1', 'a1', 'a1']]);
                    expectValueMatrix(0, 'E5', [['a', null, null], [null, null, null], [null, null, null]]);
                    expectStyleMatrix(0, 'E5', [['', '', ''], ['', '', ''], ['', '', '']]);
                    expect(docModel.applyOperations(generator, { undo: true })).to.equal(true);
                    expectMergedRanges(0, '');
                    expectValueMatrix(0, 'A5', [[1, 4, 'a'], [2, 5, 'b'], [3, 6, 'c']]);
                    expectStyleMatrix(0, 'A5', [['a1', '', ''], ['', 'a2', ''], ['', '', 'a1']]);
                    expectValueMatrix(0, 'E5', [[null, null, 'a'], [null, 5, 'b'], [3, 6, 'c']]);
                    expectStyleMatrix(0, 'E5', [['a1', '', ''], ['', 'a2', ''], ['', '', 'a1']]);
                    done();
                });
            });

            it('should merge cell ranges horizontally and move first value cells', function (done) {
                var sheetModel = docModel.getSheetModel(0);
                var mergeCollection = sheetModel.getMergeCollection();
                var generator = sheetModel.createOperationGenerator({ applyImmediately: true });
                var promise = mergeCollection.generateMergeCellsOperations(generator, ra('A5:C7 E5:G7'), MergeMode.HORIZONTAL);
                promise.always(function () {
                    expect(promise.state()).to.equal('resolved');
                    expectMergedRanges(0, 'A5:C5,E5:G5,A6:C6,E6:G6,A7:C7,E7:G7');
                    expectValueMatrix(0, 'A5', [[1, null, null], [2, null, null], [3, null, null]]);
                    expectStyleMatrix(0, 'A5', [['a1', 'a1', 'a1'], ['', '', ''], ['', '', '']]);
                    expectValueMatrix(0, 'E5', [['a', null, null], [5, null, null], [3, null, null]]);
                    expectStyleMatrix(0, 'E5', [['', '', ''], ['a2', 'a2', 'a2'], ['', '', '']]);
                    expect(docModel.applyOperations(generator, { undo: true })).to.equal(true);
                    expectMergedRanges(0, '');
                    expectValueMatrix(0, 'A5', [[1, 4, 'a'], [2, 5, 'b'], [3, 6, 'c']]);
                    expectStyleMatrix(0, 'A5', [['a1', '', ''], ['', 'a2', ''], ['', '', 'a1']]);
                    expectValueMatrix(0, 'E5', [[null, null, 'a'], [null, 5, 'b'], [3, 6, 'c']]);
                    expectStyleMatrix(0, 'E5', [['a1', '', ''], ['', 'a2', ''], ['', '', 'a1']]);
                    done();
                });
            });

            it('should merge cell ranges vertically and move first value cells', function (done) {
                var sheetModel = docModel.getSheetModel(0);
                var mergeCollection = sheetModel.getMergeCollection();
                var generator = sheetModel.createOperationGenerator({ applyImmediately: true });
                var promise = mergeCollection.generateMergeCellsOperations(generator, ra('A5:C7 E5:G7'), MergeMode.VERTICAL);
                promise.always(function () {
                    expect(promise.state()).to.equal('resolved');
                    expectMergedRanges(0, 'A5:A7,B5:B7,C5:C7,E5:E7,F5:F7,G5:G7');
                    expectValueMatrix(0, 'A5', [[1, 4, 'a'], [null, null, null], [null, null, null]]);
                    expectStyleMatrix(0, 'A5', [['a1', '', ''], ['a1', '', ''], ['a1', '', '']]);
                    expectValueMatrix(0, 'E5', [[3, 5, 'a'], [null, null, null], [null, null, null]]);
                    expectStyleMatrix(0, 'E5', [['', 'a2', ''], ['', 'a2', ''], ['', 'a2', '']]);
                    expect(docModel.applyOperations(generator, { undo: true })).to.equal(true);
                    expectMergedRanges(0, '');
                    expectValueMatrix(0, 'A5', [[1, 4, 'a'], [2, 5, 'b'], [3, 6, 'c']]);
                    expectStyleMatrix(0, 'A5', [['a1', '', ''], ['', 'a2', ''], ['', '', 'a1']]);
                    expectValueMatrix(0, 'E5', [[null, null, 'a'], [null, 5, 'b'], [3, 6, 'c']]);
                    expectStyleMatrix(0, 'E5', [['a1', '', ''], ['', 'a2', ''], ['', '', 'a1']]);
                    done();
                });
            });

            it('should unmerge cell ranges', function (done) {
                var sheetModel = docModel.getSheetModel(1);
                var mergeCollection = sheetModel.getMergeCollection();
                var generator = sheetModel.createOperationGenerator({ applyImmediately: true });
                var promise = mergeCollection.generateMergeCellsOperations(generator, r('D6:F6'), MergeMode.UNMERGE);
                promise.always(function () {
                    expect(promise.state()).to.equal('resolved');
                    expectMergedRanges(1, 'A5:C7');
                    expectStyleMatrix(1, 'E5', [['a1', 'a1', 'a1'], ['a1', 'a1', 'a1'], ['a1', 'a1', 'a1']]);
                    expect(docModel.applyOperations(generator, { undo: true })).to.equal(true);
                    expectMergedRanges(1, 'A5:C7,E5:G7');
                    expectStyleMatrix(1, 'E5', [['a1', 'a1', 'a1'], ['a1', 'a1', 'a1'], ['a1', 'a1', 'a1']]);
                    done();
                });
            });

            it('should unmerge cell ranges implicitly when merging', function (done) {
                var sheetModel = docModel.getSheetModel(1);
                var mergeCollection = sheetModel.getMergeCollection();
                var generator = sheetModel.createOperationGenerator({ applyImmediately: true });
                var promise = mergeCollection.generateMergeCellsOperations(generator, r('D6:F6'), MergeMode.MERGE);
                promise.always(function () {
                    expect(promise.state()).to.equal('resolved');
                    expectMergedRanges(1, 'A5:C7,D6:F6');
                    expectStyleMatrix(1, 'E5', [['a1', 'a1', 'a1'], ['', '', 'a1'], ['a1', 'a1', 'a1']]);
                    expect(docModel.applyOperations(generator, { undo: true })).to.equal(true);
                    expectMergedRanges(1, 'A5:C7,E5:G7');
                    expectStyleMatrix(1, 'E5', [['a1', 'a1', 'a1'], ['a1', 'a1', 'a1'], ['a1', 'a1', 'a1']]);
                    done();
                });
            });

            it('should merge with framed border', function (done) {
                var sheet           = 2,
                    sheetModel      = docModel.getSheetModel(sheet),
                    cellCollection  = sheetModel.getCellCollection(),
                    mergeCollection = sheetModel.getMergeCollection(),

                    checkRange      = r('A1:Z99'),
                    doMerge         = r('E5:G7'),
                    mergedRanges    = null,

                    cells = [
                        { adr: a('E5'), before: { v: 1,   s: 'a3'  }, after: { v: 1,    s: 'a3'  } },
                        { adr: a('E6'), before: { v: 2,   s: 'a10' }, after: { v: null, s: 'a10' } },
                        { adr: a('E7'), before: { v: 3,   s: 'a5'  }, after: { v: null, s: 'a5'  } },
                        { adr: a('F5'), before: { v: 4,   s: 'a7'  }, after: { v: null, s: 'a7'  } },
                        { adr: a('F6'), before: { v: 5,   s: ''    }, after: { v: null, s: 'a11' } },
                        { adr: a('F7'), before: { v: 6,   s: 'a9'  }, after: { v: null, s: 'a9'  } },
                        { adr: a('G5'), before: { v: 'a', s: 'a4'  }, after: { v: null, s: 'a4'  } },
                        { adr: a('G6'), before: { v: 'b', s: 'a8'  }, after: { v: null, s: 'a8'  } },
                        { adr: a('G7'), before: { v: 'c', s: 'a6'  }, after: { v: null, s: 'a6'  } }
                    ];

                cells.forEach(function (obj) {
                    // console.log('before: ', obj.adr.toString());
                    expect(cellCollection.getValue(obj.adr)).to.equal(obj.before.v);
                    expect(cellCollection.getStyleId(obj.adr)).to.equal(obj.before.s);
                });

                mergedRanges = mergeCollection.getMergedRanges(checkRange);
                expect(mergedRanges.length).to.equal(0);

                var generator   = sheetModel.createOperationGenerator({ applyImmediately: true });
                var promise     = mergeCollection.generateMergeCellsOperations(generator, doMerge, MergeMode.MERGE);

                promise.always(function () {
                    expect(promise.state()).to.equal('resolved');

                    cells.forEach(function (obj) {
                        // console.log('after: ', obj.adr.toString());
                        expect(cellCollection.getValue(obj.adr)).to.equal(obj.after.v);
                        expect(cellCollection.getStyleId(obj.adr)).to.equal(obj.after.s);
                    });

                    mergedRanges = mergeCollection.getMergedRanges(checkRange);
                    expect(mergedRanges.length).to.equal(1);
                    expect(mergedRanges[0]).to.deep.equal(doMerge);
                    expect(docModel.applyOperations(generator, { undo: true })).to.equal(true);

                    cells.forEach(function (obj) {
                        expect(cellCollection.getValue(obj.adr)).to.equal(obj.before.v);
                        expect(cellCollection.getStyleId(obj.adr)).to.equal(obj.before.s);
                    });

                    mergedRanges = mergeCollection.getMergedRanges(checkRange);
                    expect(mergedRanges.length).to.equal(0);

                    done();
                });
            });

            it('should merge with framed border except the top one', function (done) {
                var sheet           = 3,
                    sheetModel      = docModel.getSheetModel(sheet),
                    cellCollection  = sheetModel.getCellCollection(),
                    mergeCollection = sheetModel.getMergeCollection(),

                    checkRange      = r('A1:Z99'),
                    doMerge         = r('E5:G7'),
                    mergedRanges    = null,

                    cells = [
                        { adr: a('E5'), before: { v: 1,   s: 'a3'  }, after: { v: 1,    s: 'a10' } },
                        { adr: a('E6'), before: { v: 2,   s: 'a10' }, after: { v: null, s: 'a10' } },
                        { adr: a('E7'), before: { v: 3,   s: 'a5'  }, after: { v: null, s: 'a5'  } },
                        { adr: a('F5'), before: { v: 4,   s: 'a12' }, after: { v: null, s: 'a11' } },
                        { adr: a('F6'), before: { v: 5,   s: ''    }, after: { v: null, s: 'a11' } },
                        { adr: a('F7'), before: { v: 6,   s: 'a9'  }, after: { v: null, s: 'a9'  } },
                        { adr: a('G5'), before: { v: 'a', s: 'a4'  }, after: { v: null, s: 'a8'  } },
                        { adr: a('G6'), before: { v: 'b', s: 'a8'  }, after: { v: null, s: 'a8'  } },
                        { adr: a('G7'), before: { v: 'c', s: 'a6'  }, after: { v: null, s: 'a6'  } }
                    ];

                cells.forEach(function (obj) {
                    // console.log('before: ', obj.adr.toString());
                    expect(cellCollection.getValue(obj.adr)).to.equal(obj.before.v);
                    expect(cellCollection.getStyleId(obj.adr)).to.equal(obj.before.s);
                });

                mergedRanges = mergeCollection.getMergedRanges(checkRange);
                expect(mergedRanges.length).to.equal(0);

                var generator   = sheetModel.createOperationGenerator({ applyImmediately: true });
                var promise     = mergeCollection.generateMergeCellsOperations(generator, doMerge, MergeMode.MERGE);

                promise.always(function () {
                    expect(promise.state()).to.equal('resolved');

                    cells.forEach(function (obj) {
                        // console.log('after: ', obj.adr.toString());
                        expect(cellCollection.getValue(obj.adr)).to.equal(obj.after.v);
                        expect(cellCollection.getStyleId(obj.adr)).to.equal(obj.after.s);
                    });

                    mergedRanges = mergeCollection.getMergedRanges(checkRange);
                    expect(mergedRanges.length).to.equal(1);
                    expect(mergedRanges[0]).to.deep.equal(doMerge);
                    expect(docModel.applyOperations(generator, { undo: true })).to.equal(true);

                    cells.forEach(function (obj) {
                        expect(cellCollection.getValue(obj.adr)).to.equal(obj.before.v);
                        expect(cellCollection.getStyleId(obj.adr)).to.equal(obj.before.s);
                    });

                    mergedRanges = mergeCollection.getMergedRanges(checkRange);
                    expect(mergedRanges.length).to.equal(0);

                    done();
                });
            });

            it('should merge with framed border and reset the middle cells', function (done) {
                var sheet           = 4,
                    sheetModel      = docModel.getSheetModel(sheet),
                    cellCollection  = sheetModel.getCellCollection(),
                    mergeCollection = sheetModel.getMergeCollection(),

                    checkRange      = r('A1:Z99'),
                    doMerge         = r('E5:G9'),
                    mergedRanges    = null,

                    cells = [
                        { adr: a('E5'), before: { v: 1,   s: 'a3'  }, after: { v: 1,    s: 'a3'  } },
                        { adr: a('E6'), before: { v: 2,   s: 'a10' }, after: { v: null, s: 'a10' } },
                        { adr: a('E7'), before: { v: 3,   s: 'a10' }, after: { v: null, s: 'a10' } },
                        { adr: a('E8'), before: { v: 4,   s: 'a10' }, after: { v: null, s: 'a10' } },
                        { adr: a('E9'), before: { v: 5,   s: 'a5'  }, after: { v: null, s: 'a5'  } },

                        { adr: a('F5'), before: { v: 4,   s: 'a7'  }, after: { v: null, s: 'a7'  } },
                        { adr: a('F6'), before: { v: 5,   s: ''    }, after: { v: null, s: 'a11' } },
                        { adr: a('F7'), before: { v: 6,   s: 'a10' }, after: { v: null, s: 'a11' } },
                        { adr: a('F8'), before: { v: 7,   s: ''    }, after: { v: null, s: 'a11' } },
                        { adr: a('F9'), before: { v: 8,   s: 'a9'  }, after: { v: null, s: 'a9'  } },

                        { adr: a('G5'), before: { v: 'a', s: 'a4'  }, after: { v: null, s: 'a4'  } },
                        { adr: a('G6'), before: { v: 'b', s: 'a8'  }, after: { v: null, s: 'a8'  } },
                        { adr: a('G7'), before: { v: 'c', s: 'a8'  }, after: { v: null, s: 'a8'  } },
                        { adr: a('G8'), before: { v: 'd', s: 'a8'  }, after: { v: null, s: 'a8'  } },
                        { adr: a('G9'), before: { v: 'e', s: 'a6'  }, after: { v: null, s: 'a6'  } }
                    ];

                cells.forEach(function (obj) {
                    // console.log('before: ', obj.adr.toString());
                    expect(cellCollection.getValue(obj.adr)).to.equal(obj.before.v);
                    expect(cellCollection.getStyleId(obj.adr)).to.equal(obj.before.s);
                });

                mergedRanges = mergeCollection.getMergedRanges(checkRange);
                expect(mergedRanges.length).to.equal(0);

                var generator   = sheetModel.createOperationGenerator({ applyImmediately: true });
                var promise     = mergeCollection.generateMergeCellsOperations(generator, doMerge, MergeMode.MERGE);

                promise.always(function () {
                    expect(promise.state()).to.equal('resolved');

                    cells.forEach(function (obj) {
                        // console.log('after: ', obj.adr.toString());
                        expect(cellCollection.getValue(obj.adr)).to.equal(obj.after.v);
                        expect(cellCollection.getStyleId(obj.adr)).to.equal(obj.after.s);
                    });

                    mergedRanges = mergeCollection.getMergedRanges(checkRange);
                    expect(mergedRanges.length).to.equal(1);
                    expect(mergedRanges[0]).to.deep.equal(doMerge);
                    expect(docModel.applyOperations(generator, { undo: true })).to.equal(true);

                    cells.forEach(function (obj) {
                        expect(cellCollection.getValue(obj.adr)).to.equal(obj.before.v);
                        expect(cellCollection.getStyleId(obj.adr)).to.equal(obj.before.s);
                    });

                    mergedRanges = mergeCollection.getMergedRanges(checkRange);
                    expect(mergedRanges.length).to.equal(0);

                    done();
                });
            });
        });
    });

    // ========================================================================
});
