/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define([
    'io.ox/office/tk/utils',
    'io.ox/office/spreadsheet/utils/errorcode',
    'io.ox/office/spreadsheet/model/formula/utils/mathutils'
], function (Utils, ErrorCode, MathUtils) {

    'use strict';

    // static class MathUtils =================================================

    describe('Spreadsheet module MathUtils', function () {

        it('should exist', function () {
            expect(MathUtils).to.be.an('object');
        });

        // private helpers ----------------------------------------------------

        var sqrt = Math.sqrt;
        var exp = Math.exp;
        var log = Math.log;

        // static methods -----------------------------------------------------

        describe('method "isZero"', function () {
            it('should exist', function () {
                expect(MathUtils).itself.to.respondTo('isZero');
            });
            it('should return true for zero and denormalized numbers', function () {
                expect(MathUtils.isZero(0)).to.equal(true);
                expect(MathUtils.isZero(Utils.MIN_NUMBER / 2)).to.equal(true);
                expect(MathUtils.isZero(-Utils.MIN_NUMBER / 2)).to.equal(true);
            });
            it('should return false for non-zero numbers, infinite, and NaN', function () {
                expect(MathUtils.isZero(Utils.MIN_NUMBER)).to.equal(false);
                expect(MathUtils.isZero(-Utils.MIN_NUMBER)).to.equal(false);
                expect(MathUtils.isZero(1)).to.equal(false);
                expect(MathUtils.isZero(-1)).to.equal(false);
                expect(MathUtils.isZero(Number.POSITIVE_INFINITY)).to.equal(false);
                expect(MathUtils.isZero(Number.NEGATIVE_INFINITY)).to.equal(false);
                expect(MathUtils.isZero(Number.NaN)).to.equal(false);
            });
        });

        describe('method "trunc"', function () {
            it('should exist', function () {
                expect(MathUtils).itself.to.respondTo('trunc');
            });
            it('should round down positive numbers', function () {
                expect(MathUtils.trunc(1.1)).to.equal(1);
                expect(MathUtils.trunc(5.5)).to.equal(5);
                expect(MathUtils.trunc(9.9)).to.equal(9);
            });
            it('should round up negative numbers', function () {
                expect(MathUtils.trunc(-1.1)).to.equal(-1);
                expect(MathUtils.trunc(-5.5)).to.equal(-5);
                expect(MathUtils.trunc(-9.9)).to.equal(-9);
            });
            it('should return integers unmodified', function () {
                expect(MathUtils.trunc(-1)).to.equal(-1);
                expect(MathUtils.trunc(0)).to.equal(0);
                expect(MathUtils.trunc(1)).to.equal(1);
            });
        });

        describe('method "add"', function () {
            it('should exist', function () {
                expect(MathUtils).itself.to.respondTo('add');
            });
            it('should return the sum of two numbers', function () {
                expect(MathUtils.add(3, 2)).to.equal(5);
                expect(MathUtils.add(0, 2)).to.equal(2);
            });
            it('should return the concatenation of two strings', function () {
                expect(MathUtils.add('a', 'b')).to.equal('ab');
            });
        });

        describe('method "sub"', function () {
            it('should exist', function () {
                expect(MathUtils).itself.to.respondTo('sub');
            });
            it('should return the difference of two numbers', function () {
                expect(MathUtils.sub(3, 2)).to.equal(1);
                expect(MathUtils.sub(0, 2)).to.equal(-2);
            });
        });

        describe('method "mul"', function () {
            it('should exist', function () {
                expect(MathUtils).itself.to.respondTo('mul');
            });
            it('should return the product of two numbers', function () {
                expect(MathUtils.mul(3, 2)).to.equal(6);
                expect(MathUtils.mul(-3, 2)).to.equal(-6);
                expect(MathUtils.mul(3, -2)).to.equal(-6);
                expect(MathUtils.mul(-3, -2)).to.equal(6);
                expect(MathUtils.mul(0, 2)).to.equal(0);
                expect(MathUtils.mul(3, 0)).to.equal(0);
            });
        });

        describe('method "div"', function () {
            it('should exist', function () {
                expect(MathUtils).itself.to.respondTo('div');
            });
            it('should return the quotient of two numbers', function () {
                expect(MathUtils.div(3, 2)).to.equal(1.5);
                expect(MathUtils.div(0, 2)).to.equal(0);
            });
            it('should throw the #DIV/0! error, if divisor is 0', function () {
                expect(MathUtils.div.bind(null, 3, 0)).to.throw(ErrorCode).that.equals(ErrorCode.DIV0);
                expect(MathUtils.div.bind(null, 0, 0)).to.throw(ErrorCode).that.equals(ErrorCode.DIV0);
            });
        });

        describe('method "mod"', function () {
            it('should exist', function () {
                expect(MathUtils).itself.to.respondTo('mod');
            });
            var mod = MathUtils.mod;
            it('should return the remainder, if divisor is positive', function () {
                expect(mod(-5, 3)).to.equal(1);
                expect(mod(-4, 3)).to.equal(2);
                expect(mod(-3, 3)).to.equal(0);
                expect(mod(-2, 3)).to.equal(1);
                expect(mod(-1, 3)).to.equal(2);
                expect(mod(0, 3)).to.equal(0);
                expect(mod(1, 3)).to.equal(1);
                expect(mod(2, 3)).to.equal(2);
                expect(mod(3, 3)).to.equal(0);
                expect(mod(4, 3)).to.equal(1);
                expect(mod(5, 3)).to.equal(2);
            });
            it('should return the remainder, if divisor is negative', function () {
                expect(mod(-5, -3)).to.equal(-2);
                expect(mod(-4, -3)).to.equal(-1);
                expect(mod(-3, -3)).to.equal(0);
                expect(mod(-2, -3)).to.equal(-2);
                expect(mod(-1, -3)).to.equal(-1);
                expect(mod(0, -3)).to.equal(0);
                expect(mod(1, -3)).to.equal(-2);
                expect(mod(2, -3)).to.equal(-1);
                expect(mod(3, -3)).to.equal(0);
                expect(mod(4, -3)).to.equal(-2);
                expect(mod(5, -3)).to.equal(-1);
            });
            it('should throw the #DIV/0! error, if divisor is 0', function () {
                expect(mod.bind(null, 3, 0)).to.throw(ErrorCode).that.equals(ErrorCode.DIV0);
                expect(mod.bind(null, 0, 0)).to.throw(ErrorCode).that.equals(ErrorCode.DIV0);
            });
        });

        describe('method "powNaN"', function () {
            it('should exist', function () {
                expect(MathUtils).itself.to.respondTo('powNaN');
            });
            var pow = MathUtils.powNaN;
            it('should return the power of two numbers', function () {
                expect(pow(3, 0)).to.equal(1);
                expect(pow(3, 1)).to.equal(3);
                expect(pow(3, 2)).to.equal(9);
                expect(pow(3, 3)).to.equal(27);
                expect(pow(9, 0.5)).to.almostEqual(3);
                expect(pow(81, 0.25)).to.almostEqual(3);
                expect(pow(4, -1)).to.almostEqual(0.25);
                expect(pow(2, -2)).to.almostEqual(0.25);
                expect(pow(0, 1)).to.equal(0);
                expect(pow(0, 2)).to.equal(0);
                expect(pow(0, 0.5)).to.equal(0);
            });
            it('should throw the #NUM! error, if both numbers are 0', function () {
                expect(pow.bind(null, 0, 0)).to.throw(ErrorCode).that.equals(ErrorCode.NUM);
            });
            it('should throw the #DIV/0! error, if base is 0, and exponent is negative', function () {
                expect(pow.bind(null, 0, -1)).to.throw(ErrorCode).that.equals(ErrorCode.DIV0);
            });
        });

        describe('method "pow1"', function () {
            it('should exist', function () {
                expect(MathUtils).itself.to.respondTo('pow1');
            });
            var pow = MathUtils.pow1;
            it('should return the power of two numbers', function () {
                expect(pow(3, 0)).to.equal(1);
                expect(pow(3, 1)).to.equal(3);
                expect(pow(3, 2)).to.equal(9);
                expect(pow(3, 3)).to.equal(27);
                expect(pow(9, 0.5)).to.almostEqual(3);
                expect(pow(81, 0.25)).to.almostEqual(3);
                expect(pow(4, -1)).to.almostEqual(0.25);
                expect(pow(2, -2)).to.almostEqual(0.25);
                expect(pow(0, 1)).to.equal(0);
                expect(pow(0, 2)).to.equal(0);
                expect(pow(0, 0.5)).to.equal(0);
            });
            it('should return 1, if both numbers are 0', function () {
                expect(pow(0, 0)).to.equal(1);
            });
            it('should throw the #NUM! error, if base is 0, and exponent is negative', function () {
                expect(pow.bind(null, 0, -1)).to.throw(ErrorCode).that.equals(ErrorCode.NUM);
            });
        });

        describe('property "POW"', function () {
            it('should exist', function () {
                expect(MathUtils).to.have.a.property('POW').that.is.an('object');
            });
            it('should contain resolvers for file formats', function () {
                expect(MathUtils.POW).to.have.a.property('ooxml', MathUtils.powNaN);
                expect(MathUtils.POW).to.have.a.property('odf', MathUtils.pow1);
            });
        });

        describe('method "pow10"', function () {
            it('should exist', function () {
                expect(MathUtils).itself.to.respondTo('pow10');
            });
            var pow10 = MathUtils.pow10;
            it('should return the decimal power', function () {
                expect(pow10(0)).to.equal(1);
                expect(pow10(1)).to.equal(10);
                expect(pow10(2)).to.equal(100);
                expect(pow10(3)).to.equal(1000);
                expect(pow10(4)).to.equal(10000);
                expect(pow10(-1)).to.equal(0.1);
                expect(pow10(-2)).to.equal(0.01);
                expect(pow10(-3)).to.equal(0.001);
                expect(pow10(-4)).to.equal(0.0001);
            });
        });

        describe('method "log10"', function () {
            it('should exist', function () {
                expect(MathUtils).itself.to.respondTo('log10');
            });
            var log10 = MathUtils.log10;
            it('should return the decimal logarithm', function () {
                expect(log10(1)).to.equal(0);
                expect(log10(10)).to.almostEqual(1);
                expect(log10(100)).to.almostEqual(2);
                expect(log10(1000)).to.almostEqual(3);
                expect(log10(10000)).to.almostEqual(4);
                expect(log10(0.1)).to.almostEqual(-1);
                expect(log10(0.01)).to.almostEqual(-2);
                expect(log10(0.001)).to.almostEqual(-3);
                expect(log10(0.0001)).to.almostEqual(-4);
            });
        });

        describe('method "atan2"', function () {
            it('should exist', function () {
                expect(MathUtils).itself.to.respondTo('atan2');
            });
            var atan2 = MathUtils.atan2;
            it('should return the arc tangent of two numbers', function () {
                expect(atan2(1, 0)).to.be.almostZero;
                expect(atan2(1, 1)).to.almostEqual(0.25 * Math.PI);
                expect(atan2(0, 1)).to.almostEqual(0.5 * Math.PI);
                expect(atan2(-1, 1)).to.almostEqual(0.75 * Math.PI);
                expect(atan2(-1, 0)).to.almostEqual(Math.PI);
                expect(atan2(-1, -1)).to.almostEqual(-0.75 * Math.PI);
                expect(atan2(0, -1)).to.almostEqual(-0.5 * Math.PI);
                expect(atan2(1, -1)).to.almostEqual(-0.25 * Math.PI);
            });
            it('should throw the #DIV/0! error, if both numbers are 0', function () {
                expect(atan2.bind(null, 0, 0)).to.throw(ErrorCode).that.equals(ErrorCode.DIV0);
            });
        });

        describe('method "sinh"', function () {
            it('should exist', function () {
                expect(MathUtils).itself.to.respondTo('sinh');
            });
            var sinh = MathUtils.sinh;
            it('should return the hyperbolic sine', function () {
                expect(sinh(-2)).to.almostEqual((exp(-2) - exp(2)) / 2);
                expect(sinh(-1)).to.almostEqual((exp(-1) - exp(1)) / 2);
                expect(sinh(0)).to.equal(0);
                expect(sinh(1)).to.almostEqual((exp(1) - exp(-1)) / 2);
                expect(sinh(2)).to.almostEqual((exp(2) - exp(-2)) / 2);
            });
            it('should use the native implementation', function () {
                if (Math.sinh) { expect(sinh).to.equal(Math.sinh); }
            });
        });

        describe('method "cosh"', function () {
            it('should exist', function () {
                expect(MathUtils).itself.to.respondTo('cosh');
            });
            var cosh = MathUtils.cosh;
            it('should return the hyperbolic cosine', function () {
                expect(cosh(-2)).to.almostEqual((exp(-2) + exp(2)) / 2);
                expect(cosh(-1)).to.almostEqual((exp(-1) + exp(1)) / 2);
                expect(cosh(0)).to.equal(1);
                expect(cosh(1)).to.almostEqual((exp(1) + exp(-1)) / 2);
                expect(cosh(2)).to.almostEqual((exp(2) + exp(-2)) / 2);
            });
            it('should use the native implementation', function () {
                if (Math.cosh) { expect(cosh).to.equal(Math.cosh); }
            });
        });

        describe('method "tanh"', function () {
            it('should exist', function () {
                expect(MathUtils).itself.to.respondTo('tanh');
            });
            var tanh = MathUtils.tanh;
            it('should return the hyperbolic tangent', function () {
                expect(tanh(-2)).to.almostEqual((exp(-4) - 1) / (exp(-4) + 1));
                expect(tanh(-1)).to.almostEqual((exp(-2) - 1) / (exp(-2) + 1));
                expect(tanh(0)).to.equal(0);
                expect(tanh(1)).to.almostEqual((exp(2) - 1) / (exp(2) + 1));
                expect(tanh(2)).to.almostEqual((exp(4) - 1) / (exp(4) + 1));
            });
            it('should use the native implementation', function () {
                if (Math.tanh) { expect(tanh).to.equal(Math.tanh); }
            });
        });

        describe('method "asinh"', function () {
            it('should exist', function () {
                expect(MathUtils).itself.to.respondTo('asinh');
            });
            var asinh = MathUtils.asinh;
            it('should return the argument resulting in the hyperbolic sine', function () {
                expect(asinh(-2)).to.almostEqual(log(sqrt(5) - 2));
                expect(asinh(-1)).to.almostEqual(log(sqrt(2) - 1));
                expect(asinh(0)).to.equal(0);
                expect(asinh(1)).to.almostEqual(log(sqrt(2) + 1));
                expect(asinh(2)).to.almostEqual(log(sqrt(5) + 2));
            });
            it('should use the native implementation', function () {
                if (Math.asinh) { expect(asinh).to.equal(Math.asinh); }
            });
        });

        describe('method "acosh"', function () {
            it('should exist', function () {
                expect(MathUtils).itself.to.respondTo('acosh');
            });
            var acosh = MathUtils.acosh;
            it('should return the argument resulting in the hyperbolic cosine', function () {
                expect(acosh(-1)).not.to.be.finite;
                expect(acosh(0)).not.to.be.finite;
                expect(acosh(1)).to.equal(0);
                expect(acosh(2)).to.almostEqual(log(2 + sqrt(3)));
                expect(acosh(3)).to.almostEqual(log(3 + sqrt(8)));
            });
            it('should use the native implementation', function () {
                if (Math.acosh) { expect(acosh).to.equal(Math.acosh); }
            });
        });

        describe('method "atanh"', function () {
            it('should exist', function () {
                expect(MathUtils).itself.to.respondTo('atanh');
            });
            var atanh = MathUtils.atanh;
            it('should return the argument resulting in the hyperbolic tangent', function () {
                expect(atanh(-2)).not.to.be.finite;
                expect(atanh(-1)).not.to.be.finite;
                expect(atanh(-0.5)).to.almostEqual(log(1 / 3) / 2);
                expect(atanh(0)).to.equal(0);
                expect(atanh(0.5)).to.almostEqual(log(3) / 2);
                expect(atanh(1)).not.to.be.finite;
                expect(atanh(2)).not.to.be.finite;
            });
            it('should use the native implementation', function () {
                if (Math.atanh) { expect(atanh).to.equal(Math.atanh); }
            });
        });

        describe('method "acoth"', function () {
            it('should exist', function () {
                expect(MathUtils).itself.to.respondTo('acoth');
            });
            var acoth = MathUtils.acoth;
            it('should return the argument resulting in the hyperbolic tangent', function () {
                expect(acoth(-3)).to.almostEqual(log(1 / 2) / 2);
                expect(acoth(-2)).to.almostEqual(log(1 / 3) / 2);
                expect(acoth(-1)).not.to.be.finite;
                expect(acoth(0)).not.to.be.finite;
                expect(acoth(1)).not.to.be.finite;
                expect(acoth(2)).to.almostEqual(log(3) / 2);
                expect(acoth(3)).to.almostEqual(log(2) / 2);
            });
            it('should use the native implementation', function () {
                if (Math.acoth) { expect(acoth).to.equal(Math.acoth); }
            });
        });

        describe('method "sum"', function () {
            it('should exist', function () {
                expect(MathUtils).itself.to.respondTo('sum');
            });
            it('should return the sum of the passed array', function () {
                expect(MathUtils.sum([])).to.equal(0);
                expect(MathUtils.sum([1])).to.equal(1);
                expect(MathUtils.sum([1, 2])).to.equal(3);
                expect(MathUtils.sum([1, 2, 3])).to.equal(6);
                expect(MathUtils.sum([1, 2, 3, 4])).to.equal(10);
            });
        });

        describe('method "product"', function () {
            it('should exist', function () {
                expect(MathUtils).itself.to.respondTo('product');
            });
            it('should return the product of the passed array', function () {
                expect(MathUtils.product([])).to.equal(1);
                expect(MathUtils.product([1])).to.equal(1);
                expect(MathUtils.product([1, 2])).to.equal(2);
                expect(MathUtils.product([1, 2, 3])).to.equal(6);
                expect(MathUtils.product([1, 2, 3, 4])).to.equal(24);
            });
        });

        describe('method "factorial"', function () {
            it('should exist', function () {
                expect(MathUtils).itself.to.respondTo('factorial');
            });
            var factorial = MathUtils.factorial;
            it('should return the factorials of numbers', function () {
                expect(factorial(0)).to.equal(1);
                expect(factorial(1)).to.equal(1);
                expect(factorial(2)).to.equal(2);
                expect(factorial(3)).to.equal(6);
                expect(factorial(4)).to.equal(24);
                expect(factorial(5)).to.equal(120);
                expect(factorial(6)).to.equal(720);
                expect(factorial(7)).to.equal(5040);
            });
            it('should round the input parameter', function () {
                expect(factorial(4.5)).to.equal(24);
            });
            it('should throw #NUM! error code for invalid numbers', function () {
                expect(factorial.bind(MathUtils, -1)).to.throw(ErrorCode).that.equals(ErrorCode.NUM);
                expect(factorial.bind(MathUtils, 1000)).to.throw(ErrorCode).that.equals(ErrorCode.NUM);
            });
        });

        describe('method "factorial2"', function () {
            it('should exist', function () {
                expect(MathUtils).itself.to.respondTo('factorial2');
            });
            var factorial2 = MathUtils.factorial2;
            it('should return the factorials of numbers', function () {
                expect(factorial2(0)).to.equal(1);
                expect(factorial2(1)).to.equal(1);
                expect(factorial2(2)).to.equal(2);
                expect(factorial2(3)).to.equal(3);
                expect(factorial2(4)).to.equal(8);
                expect(factorial2(5)).to.equal(15);
                expect(factorial2(6)).to.equal(48);
                expect(factorial2(7)).to.equal(105);
            });
            it('should round the input parameter', function () {
                expect(factorial2(4.5)).to.equal(8);
            });
            it('should throw #NUM! error code for invalid numbers', function () {
                expect(factorial2.bind(MathUtils, -1)).to.throw(ErrorCode).that.equals(ErrorCode.NUM);
                expect(factorial2.bind(MathUtils, 1000)).to.throw(ErrorCode).that.equals(ErrorCode.NUM);
            });
        });

        describe('method "binomial"', function () {
            it('should exist', function () {
                expect(MathUtils).itself.to.respondTo('binomial');
            });
            var binomial = MathUtils.binomial;
            it('should return the binomial coefficient', function () {
                expect(binomial(0, 0)).to.equal(1);
                expect(binomial(1, 0)).to.equal(1);
                expect(binomial(2, 0)).to.equal(1);
                expect(binomial(3, 0)).to.equal(1);
                expect(binomial(6, 0)).to.equal(1);
                expect(binomial(6, 1)).to.equal(6);
                expect(binomial(6, 2)).to.equal(15);
                expect(binomial(6, 3)).to.equal(20);
                expect(binomial(6, 4)).to.equal(15);
                expect(binomial(6, 5)).to.equal(6);
                expect(binomial(6, 6)).to.equal(1);
                expect(binomial(20, 0)).to.equal(1);
                expect(binomial(20, 1)).to.equal(20);
                expect(binomial(20, 5)).to.equal(15504);
                expect(binomial(20, 10)).to.equal(184756);
                expect(binomial(20, 15)).to.equal(15504);
                expect(binomial(20, 19)).to.equal(20);
                expect(binomial(20, 20)).to.equal(1);
                expect(binomial(1028, 514)).to.be.above(7.1e307);
                expect(binomial(1083, 400)).to.be.above(1.4e308);
            });
            it('should throw #NUM! error code for invalid parameters', function () {
                expect(binomial.bind(null, -1, 0)).to.throw(ErrorCode).that.equals(ErrorCode.NUM);
                expect(binomial.bind(null, 0, -1)).to.throw(ErrorCode).that.equals(ErrorCode.NUM);
                expect(binomial.bind(null, 2, 3)).to.throw(ErrorCode).that.equals(ErrorCode.NUM);
                expect(binomial.bind(null, 1200, 600)).to.throw(ErrorCode).that.equals(ErrorCode.NUM);
            });
        });

        describe('method "erf"', function () {
            it('should exist', function () {
                expect(MathUtils).itself.to.respondTo('erf');
            });
            var erf = MathUtils.erf;
            it('should return the result of the Gauss error function', function () {
                expect(erf(0.0)).to.equal(0);
                expect(erf(0.5)).to.be.closeTo(0.520499877813046, 1e-15);
                expect(erf(1.0)).to.be.closeTo(0.842700792949714, 1e-15);
                expect(erf(1.5)).to.be.closeTo(0.96610514647531, 1e-15);
                expect(erf(2.0)).to.be.closeTo(0.995322265019265, 1e-15);
                expect(erf(2.5)).to.be.closeTo(0.999593047982555, 1e-15);
                expect(erf(3.0)).to.be.closeTo(0.999977909503001, 1e-15);
                expect(erf(3.5)).to.be.closeTo(0.999999256901627, 1e-15);
                expect(erf(4.0)).to.be.closeTo(0.999999984582742, 1e-15);
                expect(erf(4.5)).to.be.closeTo(0.999999999803383, 1e-15);
                expect(erf(5.0)).to.be.closeTo(0.999999999998462, 1e-15);
                expect(erf(5.5)).to.be.closeTo(0.9999999999999927, 1e-15);
                expect(erf(6.0)).to.equal(1);
            });
            it('should be an odd function', function () {
                expect(erf(-1)).to.equal(-erf(1));
                expect(erf(-2)).to.equal(-erf(2));
                expect(erf(-3)).to.equal(-erf(3));
                expect(erf(-4)).to.equal(-erf(4));
                expect(erf(-5)).to.equal(-erf(5));
                expect(erf(-6)).to.equal(-erf(6));
            });
        });

        describe('method "erfc"', function () {
            it('should exist', function () {
                expect(MathUtils).itself.to.respondTo('erfc');
            });
            var erf = MathUtils.erf;
            var erfc = MathUtils.erfc;
            it('should return the result of the complementary Gauss error function', function () {
                expect(erfc(0.0)).to.equal(1);
                expect(erfc(0.5)).to.be.closeTo(1 - erf(0.5), 1e-15);
                expect(erfc(1.0)).to.be.closeTo(1 - erf(1.0), 1e-15);
                expect(erfc(1.5)).to.be.closeTo(1 - erf(1.5), 1e-15);
                expect(erfc(2.0)).to.be.closeTo(1 - erf(2.0), 1e-15);
                expect(erfc(2.5)).to.be.closeTo(1 - erf(2.5), 1e-15);
                expect(erfc(3.0)).to.be.closeTo(1 - erf(3.0), 1e-15);
                expect(erfc(3.5)).to.be.closeTo(1 - erf(3.5), 1e-15);
                expect(erfc(4.0)).to.be.closeTo(1 - erf(4.0), 1e-15);
                expect(erfc(4.5)).to.be.closeTo(1 - erf(4.5), 1e-15);
                expect(erfc(5.0)).to.be.closeTo(1 - erf(5.0), 1e-15);
                expect(erfc(5.5)).to.be.closeTo(1 - erf(5.5), 1e-15);
                expect(erfc(6.0)).to.be.closeTo(1 - erf(6.0), 1e-15);
            });
            it('should return non-zero results for large input parameters', function () {
                expect(erfc(6)).to.be.above(0); // erf(6) returns exactly 1; but erfc(6) must not return 0
                expect(erfc(7)).to.be.above(0);
                expect(erfc(8)).to.be.above(0);
                expect(erfc(9)).to.be.above(0);
            });
            it('should return correct results for negative numbers', function () {
                expect(erfc(-1)).to.equal(1 - erf(-1));
                expect(erfc(-2)).to.equal(1 - erf(-2));
                expect(erfc(-3)).to.equal(1 - erf(-3));
                expect(erfc(-4)).to.equal(1 - erf(-4));
                expect(erfc(-5)).to.equal(1 - erf(-5));
                expect(erfc(-6)).to.equal(1 - erf(-6));
            });
        });

        describe('method "slope"', function () {
            it('should exist', function () {
                expect(MathUtils).itself.to.respondTo('slope');
            });
        });

        describe('method "devSq"', function () {
            it('should exist', function () {
                expect(MathUtils).itself.to.respondTo('devSq');
            });
        });
    });

    // ========================================================================
});
