/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Michael Nimz <michael.nimz@open-xchange.com>
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define([
    'globals/apphelper',
    'globals/sheethelper'
], function (AppHelper, SheetHelper) {

    'use strict';

    // convenience shortcuts
    var a = SheetHelper.a;
    var r = SheetHelper.r;

    // class CellCollection ===================================================

    // see https://intranet.open-xchange.com/wiki/documents-team:documentation:spreadsheet-auto-sum

    describe('Spreadsheet class CellCollection', function () {

        // private helpers ----------------------------------------------------

        // the operations to be applied by the document model
        var OPERATIONS = [
            { name: 'setDocumentAttributes', attrs: { document: { cols: 16384, rows: 1048576 } } },
            { name: 'insertAutoStyle', type: 'cell', styleId: 'a0', attrs: {}, default: true },
            { name: 'insertSheet', sheet: 0, sheetName: 'Sheet1' }
        ];

        // initialize test document
        var docModel = null;
        AppHelper.createSpreadsheetApp('ooxml', OPERATIONS).done(function (app) {
            docModel = app.getModel();
        });

        function createSheet(data) {

            var ops = [{ name: 'insertSheet', sheet: 1, sheetName: 'Sheet' + (docModel.getSheetCount() + 1) }];

            _.each(data, function (value, position) {
                ops.push({ name: 'changeCells', sheet: 1, start: a(position).toJSON(), contents: [{ c: [{ v: value }] }] });
            });
            docModel.applyOperations(ops);

            return docModel.getSheetModel(1);
        }

        function testAutoFunctionRange(data, range, expData, expRange, expFormulas, done) {

            var sheetModel = createSheet(data);
            var cellCollection = sheetModel.getCellCollection();

            var generator = sheetModel.createOperationGenerator({ applyImmediately: true });
            var promise = cellCollection.generateAutoFormulaOperations(generator, 'SUM', r(range));
            promise.always(function (targetRange) {
                expect(promise.state()).to.equal('resolved');

                expect(targetRange).to.stringifyTo(expRange);

                _.each(expData, function (expValue, address) {
                    var value = cellCollection.getValue(a(address));

                    expect(value).to.equal(expValue);
                });

                _.each(expFormulas, function (expFormula, address) {
                    var formula = cellCollection.getFormula(a(address), 'ui');

                    expect(formula).to.equal(expFormula);
                });

                done();
            });
        }

        function testAutoFormulaSingle(data, address, expected) {
            var sheetModel = createSheet(data);
            var range = sheetModel.getCellCollection().findAutoFormulaRange(a(address));
            expect(range).to.stringifyTo(expected);
        }

        // test auto sum -----------------------------------------------

        describe('autosum', function () {

            it('describes autosum example 1 (column range) with expanding', function (done) {

                var data = {
                    B2: 2,
                    B3: 3
                };
                var range = 'B2:B6';

                var expectedData = {
                    B6: 5
                };

                var expectedRange = 'B2:B6';

                var expectedFormulas = {
                    B6: 'SUMME(B2:B5)'
                };

                testAutoFunctionRange(data, range, expectedData, expectedRange, expectedFormulas, done);
            });

            it('describes autosum example 1 (column range) without expanding', function (done) {

                var data = {
                    B2: 2,
                    B3: 3
                };
                var range = 'B2:B3';

                var expectedData = {
                    B4: 5
                };

                var expectedRange = 'B2:B4';

                var expectedFormulas = {
                    B4: 'SUMME(B2:B3)'
                };

                testAutoFunctionRange(data, range, expectedData, expectedRange, expectedFormulas, done);
            });

            it('describes autosum example 2 (row range) with expanding', function (done) {

                var data = {
                    B2: 2,
                    C2: 3
                };
                var range = 'B2:E2';

                var expectedData = {
                    E2: 5
                };

                var expectedRange = 'B2:E2';

                var expectedFormulas = {
                    E2: 'SUMME(B2:D2)'
                };

                testAutoFunctionRange(data, range, expectedData, expectedRange, expectedFormulas, done);
            });

            it('describes autosum example 2 (row range) without expanding', function (done) {

                var data = {
                    B2: 2,
                    C2: 3
                };
                var range = 'B2:C2';

                var expectedData = {
                    D2: 5
                };

                var expectedRange = 'B2:D2';

                var expectedFormulas = {
                    D2: 'SUMME(B2:C2)'
                };

                testAutoFunctionRange(data, range, expectedData, expectedRange, expectedFormulas, done);
            });

            it('describes autosum example 3 (column range) with auto expanding', function (done) {

                var data = {
                    B2: 2,
                    B3: 3,
                    B4: 4
                };
                var range = 'B2:B3';

                var expectedData = {
                    B5: 5
                };

                var expectedRange = 'B2:B4';

                var expectedFormulas = {
                    B5: 'SUMME(B2:B3)'
                };

                testAutoFunctionRange(data, range, expectedData, expectedRange, expectedFormulas, done);
            });

            it('describes autosum example 4 (row range) without auto expanding', function (done) {

                var data = {
                    B2: 2,
                    C2: 3,
                    D2: 4
                };
                var range = 'B2:C2';

                var expectedData = {
                    E2: 5
                };

                var expectedRange = 'B2:D2';

                var expectedFormulas = {
                    E2: 'SUMME(B2:C2)'
                };

                testAutoFunctionRange(data, range, expectedData, expectedRange, expectedFormulas, done);
            });

            it('describes autosum example 5 (single cell) with strings in active range (column)', function () {
                testAutoFormulaSingle({ B2: 2, B3: 'str', B4: 3, B5: 4 }, 'B6', 'B4:B5');
            });

            it('describes autosum example 6 (single cell) with strings in active range (row)', function () {
                testAutoFormulaSingle({ B2: 2, C2: 'str', D2: 3, E2: 4 }, 'F2', 'D2:E2');
            });

            it('describes autosum example 7 (range) with auto expanding', function (done) {

                var data = {
                    B2: 5,
                    C2: 5,
                    D2: 5,
                    B3: 1,
                    C3: 2,
                    D3: 5,
                    B4: 1,
                    D4: 5
                };
                var range = 'B2:D4';

                var expectedData = {
                    B5: 7,
                    C5: 7,
                    D5: 15
                };

                var expectedRange = 'B2:D5';

                var expectedFormulas = {
                    B5: 'SUMME(B2:B4)'
                };

                testAutoFunctionRange(data, range, expectedData, expectedRange, expectedFormulas, done);
            });

            it('describes autosum example 8 (single cell) with priority for column on top', function () {
                testAutoFormulaSingle({ D2: 3, D3: 7, B4: 2, C4: 3 }, 'D4', 'D2:D3');
            });

            it('describes autosum example 9 (single cell) with priority for closest number group', function () {
                testAutoFormulaSingle({ D2: 3, D3: 2, B5: -2, C5: 10 }, 'D5', 'B5:C5');
            });

            it('describes autosum example 10 (single cell) with priority for number group instad of string group', function () {
                testAutoFormulaSingle({ E2: 3, E3: -10, B5: 2, C5: 3, D5: 'str' }, 'E5', 'E2:E4');
            });

            it('describes autosum example 11 (range) one row with no content inside', function (done) {

                var data = {
                    B2: 2,
                    B3: 4,
                    C2: 3,
                    C3: 5
                };
                var range = 'B4:C4';

                var expectedData = {
                    B4: 6,
                    C4: 8
                };

                var expectedRange = 'B4:C4';

                var expectedFormulas = {
                    B4: 'SUMME(B2:B3)'
                };

                testAutoFunctionRange(data, range, expectedData, expectedRange, expectedFormulas, done);
            });

            it('describes special scenarion described in Bug 46938', function (done) {

                var data = {
                    A1: 3,
                    A2: 5,
                    A3: 2,
                    A4: 3
                };
                var range = 'A2:A3';

                var expectedData = {
                    A5: 7
                };

                var expectedRange = 'A2:A4';

                var expectedFormulas = {
                    A5: 'SUMME(A2:A3)'
                };

                testAutoFunctionRange(data, range, expectedData, expectedRange, expectedFormulas, done);
            });
        });
    });

    // ========================================================================
});
