/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define([
    'io.ox/office/editframework/utils/pattern'
], function (Pattern) {

    'use strict';

    // static class Pattern ===================================================

    describe('EditFramework module Pattern', function () {

        it('should exist', function () {
            expect(Pattern).to.be.an('object');
        });

        // private helpers ----------------------------------------------------

        // converts the passed byte array to a 32-bit array
        function to32(a) {
            var result = [];
            for (var i = 0; i < a.length; i += 4) {
                result.push(0x1000000 * a[i] + 0x10000 * a[i + 1] + 0x100 * a[i + 2] + a[i + 3]);
            }
            return result;
        }

        // static methods -----------------------------------------------------

        describe('method "createDataURL"', function () {
            it('should exist', function () {
                expect(Pattern).itself.to.respondTo('createDataURL');
            });
            it('should create the specified pattern', function (done) {
                var url = Pattern.createDataURL('lgGrid', { type: 'rgb', value: 'ffffff' }, { type: 'rgb', value: 'ff0000' });
                expect(url).to.be.a('string');
                var image = document.createElement('img');
                image.onload = function () {
                    expect(image.width).to.equal(8);
                    expect(image.height).to.equal(8);
                    var canvas = document.createElement('canvas');
                    canvas.width = canvas.height = 8;
                    var context = canvas.getContext('2d');
                    context.drawImage(image, 0, 0);
                    expect(to32(context.getImageData(0, 0, 8, 8).data)).to.deep.equal([
                        0xff0000ff, 0xff0000ff, 0xff0000ff, 0xff0000ff, 0xff0000ff, 0xff0000ff, 0xff0000ff, 0xff0000ff,
                        0xff0000ff, 0xffffffff, 0xffffffff, 0xffffffff, 0xffffffff, 0xffffffff, 0xffffffff, 0xffffffff,
                        0xff0000ff, 0xffffffff, 0xffffffff, 0xffffffff, 0xffffffff, 0xffffffff, 0xffffffff, 0xffffffff,
                        0xff0000ff, 0xffffffff, 0xffffffff, 0xffffffff, 0xffffffff, 0xffffffff, 0xffffffff, 0xffffffff,
                        0xff0000ff, 0xffffffff, 0xffffffff, 0xffffffff, 0xffffffff, 0xffffffff, 0xffffffff, 0xffffffff,
                        0xff0000ff, 0xffffffff, 0xffffffff, 0xffffffff, 0xffffffff, 0xffffffff, 0xffffffff, 0xffffffff,
                        0xff0000ff, 0xffffffff, 0xffffffff, 0xffffffff, 0xffffffff, 0xffffffff, 0xffffffff, 0xffffffff,
                        0xff0000ff, 0xffffffff, 0xffffffff, 0xffffffff, 0xffffffff, 0xffffffff, 0xffffffff, 0xffffffff
                    ]);
                    done();
                };
                image.src = url;
            });
            it('should return the same pattern for equal settings', function () {
                var url1 = Pattern.createDataURL('pct20', { type: 'rgb', value: 'ffffff' }, { type: 'rgb', value: 'ff0000' });
                var url2 = Pattern.createDataURL('pct20', { type: 'preset', value: 'white' }, { type: 'preset', value: 'red' });
                expect(url1).to.equal(url2);
            });
        });

        describe('method "createCanvasPattern"', function () {
            it('should exist', function () {
                expect(Pattern).itself.to.respondTo('createCanvasPattern');
            });
            it('should create the specified pattern', function () {
                var canvas = document.createElement('canvas');
                canvas.width = canvas.height = 8;
                var context = canvas.getContext('2d');
                var pattern = Pattern.createCanvasPattern('smGrid', { type: 'rgb', value: 'ffffff' }, { type: 'rgb', value: 'ff0000' });
                expect(pattern).to.be.an.instanceof(window.CanvasPattern);
                context.fillStyle = pattern;
                context.fillRect(0, 0, 8, 8);
                expect(to32(context.getImageData(0, 0, 8, 8).data)).to.deep.equal([
                    0xff0000ff, 0xff0000ff, 0xff0000ff, 0xff0000ff, 0xff0000ff, 0xff0000ff, 0xff0000ff, 0xff0000ff,
                    0xff0000ff, 0xffffffff, 0xffffffff, 0xffffffff, 0xff0000ff, 0xffffffff, 0xffffffff, 0xffffffff,
                    0xff0000ff, 0xffffffff, 0xffffffff, 0xffffffff, 0xff0000ff, 0xffffffff, 0xffffffff, 0xffffffff,
                    0xff0000ff, 0xffffffff, 0xffffffff, 0xffffffff, 0xff0000ff, 0xffffffff, 0xffffffff, 0xffffffff,
                    0xff0000ff, 0xff0000ff, 0xff0000ff, 0xff0000ff, 0xff0000ff, 0xff0000ff, 0xff0000ff, 0xff0000ff,
                    0xff0000ff, 0xffffffff, 0xffffffff, 0xffffffff, 0xff0000ff, 0xffffffff, 0xffffffff, 0xffffffff,
                    0xff0000ff, 0xffffffff, 0xffffffff, 0xffffffff, 0xff0000ff, 0xffffffff, 0xffffffff, 0xffffffff,
                    0xff0000ff, 0xffffffff, 0xffffffff, 0xffffffff, 0xff0000ff, 0xffffffff, 0xffffffff, 0xffffffff
                ]);
            });
            it('should return the same pattern for equal settings', function () {
                var pattern1 = Pattern.createCanvasPattern('pct50', { type: 'rgb', value: 'ffffff' }, { type: 'rgb', value: 'ff0000' });
                var pattern2 = Pattern.createCanvasPattern('pct50', { type: 'preset', value: 'white' }, { type: 'preset', value: 'red' });
                expect(pattern1).to.equal(pattern2);
            });
        });
    });

    // ========================================================================
});
