/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Peter Seliger <peter.seliger@open-xchange.com>
 */

define([
    'io.ox/office/editframework/utils/gradient'
], function (Gradient) {

    'use strict';

    // class Gradient ============================================================

    describe('EditFramework module Gradient', function () {
        var
            fullDescriptor        = {
                type: 'linear',
                rotation: 66.666,
                isRotateWithShape: true,
                isScaled: true,
                colorStops: [{
                    position: 0.525,
                    color: {
                        type: 'rgb',
                        value: 'ffc000'
                    }
                }, {
                    position: 0.875,
                    color: {
                        type: 'preset',
                        value: 4
                    }
                }]
            },
            feasibleDescriptor    = {
                color: {
                    type: 'preset',
                    value: '6'
                },
                color2: {
                    type: 'rgb',
                    value: 'ccff00'
                }
            },
            emptyDescriptor       = {},

            mockedGradient        = {
                getType:            function () { return ''; },
                getRotation:        function () { return 0; },
                isRotateWithShape:  function () { return true; },
                isScaled:           function () { return true; },
                getColorStops:      function () { return []; }
            },
            disfunctionalGradient = {
                getRotation:        mockedGradient.getRotation,
                isRotateWithShape:  mockedGradient.isRotateWithShape,
                isScaled:           mockedGradient.isScaled,
                getColorStops:      mockedGradient.getColorStops
            },

            minimalLinearGradient       = Gradient.create(Gradient.LINEAR),
            emptyDescriptorGradient     = Gradient.create(emptyDescriptor),

            fullDescriptorGradient      = Gradient.create(fullDescriptor),
            feasibleDescriptorGradient  = Gradient.create(feasibleDescriptor),

            expectedValueOfTheEmptyDescriptorGradient     = {
                type:               'linear',
                rotation:           0,
                isRotateWithShape:  false,
                isScaled:           false,
                colorStops:         [
                    { position: 0, color: { type: 'auto' } },
                    { position: 1, color: { type: 'auto' } }
                ]
            },
            expectedValueOfTheFeasibleDescriptorGradient  = JSON.parse(JSON.stringify(expectedValueOfTheEmptyDescriptorGradient)); // data structure clone

        expectedValueOfTheFeasibleDescriptorGradient.colorStops[0].color = feasibleDescriptor.color;
        expectedValueOfTheFeasibleDescriptorGradient.colorStops[1].color = feasibleDescriptor.color2;

        it('should exist', function () {
            expect(Gradient).to.be.an('object');
        });

        // constants ----------------------------------------------------------

        describe('constant "LINEAR"', function () {

            it('should exist', function () {
                expect(Gradient.LINEAR).to.deep.equal({ type: 'linear' });
            });
        });

        // factory method -----------------------------------------------------

        describe('factory method "create"', function () {

            it('should exist', function () {
                expect(Gradient).to.respondTo('create');
            });

            it('should return the NULL value when arguments are omitted', function () {
                expect(Gradient.create()).to.equal(null);
            });
            it('should return the NULL value when passing the NULL value as sole argument', function () {
                expect(Gradient.create(null)).to.equal(null);
            });
            it('should return the NULL value when passing any non "Object" object as sole argument', function () {
                expect(Gradient.create(RegExp())).to.equal(null);
                expect(Gradient.create([])).to.equal(null);
            });
            it('should create a gradient when passing at least a plain empty "Object" object as sole descriptor argument', function () {
                expect(Gradient.isGradient(emptyDescriptorGradient)).to.equal(true);
            });
            it('should create a gradient when passing e.g. the minimal descriptor object for linear gradient types as sole argument', function () {
                expect(Gradient.isGradient(minimalLinearGradient)).to.equal(true);
            });
        });

        // module methods -----------------------------------------------------

        describe('static method "isGradient"', function () {

            it('should exist', function () {
                expect(Gradient).to.respondTo('isGradient');
            });

            it('should return false for every object that does not match the "Gradient" signature', function () {
                expect(Gradient.isGradient(Gradient.create())).to.equal(false);
                expect(Gradient.isGradient(Gradient.create(null))).to.equal(false);

                expect(Gradient.isGradient(Gradient.create(RegExp()))).to.equal(false);
                expect(Gradient.isGradient(Gradient.create([]))).to.equal(false);

                expect(Gradient.isGradient(disfunctionalGradient)).to.equal(false);
            });
            it('should return true for every object that does match the "Gradient" signature', function () {
              //expect(Gradient.isGradient(mockedGradient)).to.equal(true); // @TODO - make it pass again

                expect(Gradient.isGradient(emptyDescriptorGradient)).to.equal(true);
                expect(Gradient.isGradient(minimalLinearGradient)).to.equal(true);

                expect(Gradient.isGradient(fullDescriptorGradient)).to.equal(true);
                expect(Gradient.isGradient(feasibleDescriptorGradient)).to.equal(true);
            });
        });

        describe('static method "isEqualDescriptors"', function () {

            it('should exist', function () {
                expect(Gradient).to.respondTo('isEqualDescriptors');
            });

            it('should return true in case of both passed arguments create each a gradient object which then again do equal each other', function () {
                expect(Gradient.isEqualDescriptors(emptyDescriptor, emptyDescriptor)).to.equal(true);
                expect(Gradient.isEqualDescriptors(Gradient.LINEAR, Gradient.LINEAR)).to.equal(true);

                expect(Gradient.isEqualDescriptors(emptyDescriptor, Gradient.LINEAR)).to.equal(true);

                expect(Gradient.isEqualDescriptors(emptyDescriptor, fullDescriptor)).to.equal(false);
                expect(Gradient.isEqualDescriptors(emptyDescriptor, feasibleDescriptor)).to.equal(false);
                expect(Gradient.isEqualDescriptors(feasibleDescriptor, fullDescriptor)).to.equal(false);
            });
        });

        // instance methods ---------------------------------------------------

        describe('any gradient type\'s "equals" method', function () {

            it('should exist', function () {
                expect(emptyDescriptorGradient).to.respondTo('equals');
                expect(minimalLinearGradient).to.respondTo('equals');
            });

            it('should return true for equal gradient types', function () {
                expect(minimalLinearGradient.equals(emptyDescriptorGradient)).to.equal(true);
                expect(emptyDescriptorGradient.equals(minimalLinearGradient)).to.equal(true);

                expect(fullDescriptorGradient.equals(minimalLinearGradient)).to.equal(false);
                expect(feasibleDescriptorGradient.equals(minimalLinearGradient)).to.equal(false);
                expect(feasibleDescriptorGradient.equals(fullDescriptorGradient)).to.equal(false);

                // same reference comparison
                expect(fullDescriptorGradient.equals(fullDescriptorGradient)).to.equal(true);
            });
        });

        describe('any gradient type\'s "clone" method', function () {

            it('should exist', function () {
                expect(emptyDescriptorGradient).to.respondTo('equals');
                expect(minimalLinearGradient).to.respondTo('equals');
            });

            it('should return a clone that is not just a reference to the operated gradient', function () {
                expect(minimalLinearGradient.clone()).not.to.equal(minimalLinearGradient);
                expect(minimalLinearGradient.clone() === minimalLinearGradient).to.equal(false);
            });
            it('should return a clone that is a gradient type itself which equals the operated gradient type', function () {
                expect(Gradient.isGradient(minimalLinearGradient.clone())).to.equal(true);
                expect(minimalLinearGradient.clone().equals(minimalLinearGradient)).to.equal(true);

                expect(Gradient.isGradient(fullDescriptorGradient.clone())).to.equal(true);
                expect(fullDescriptorGradient.clone().equals(fullDescriptorGradient)).to.equal(true);

                expect(Gradient.isGradient(feasibleDescriptorGradient.clone())).to.equal(true);
                expect(feasibleDescriptorGradient.clone().equals(feasibleDescriptorGradient)).to.equal(true);

                expect(fullDescriptorGradient.clone().equals(emptyDescriptorGradient)).to.equal(false);
                expect(feasibleDescriptorGradient.clone().equals(emptyDescriptorGradient)).to.equal(false);

                expect(fullDescriptorGradient.clone().equals(feasibleDescriptorGradient)).to.equal(false);
                expect(feasibleDescriptorGradient.clone().equals(fullDescriptorGradient)).to.equal(false);
            });
        });

        describe('any gradient type\'s "valueOf" method', function () {

            it('should exist', function () {
                expect(emptyDescriptorGradient).to.respondTo('valueOf');
                expect(feasibleDescriptorGradient).to.respondTo('valueOf');

                expect(fullDescriptorGradient).to.respondTo('valueOf');
            });

            it('should return a complete standard descriptor representation of its operated gradient type.', function () {
                // complete data transformation round-trip but mocked.
                expect(emptyDescriptorGradient.valueOf()).to.deep.equal(expectedValueOfTheEmptyDescriptorGradient);
                expect(feasibleDescriptorGradient.valueOf()).to.deep.equal(expectedValueOfTheFeasibleDescriptorGradient);

                // complete data transformation round-trip and real.
                //expect(fullDescriptorGradient.valueOf()).to.deep.equal(fullDescriptor);
            });
        });

        describe('any gradient type\'s "toString" method', function () {

            it('should exist', function () {
                expect(emptyDescriptorGradient).to.respondTo('toString');
                expect(feasibleDescriptorGradient).to.respondTo('toString');

                expect(fullDescriptorGradient).to.respondTo('toString');
            });/*

            / @TODO - make it pass again

            it('should return the JSON-style stringified version of a complete standard descriptor representation of its operated gradient type.', function () {
                var
                  //stringifiedEmptyDescriptor,   = '{"type":"linear","rotation":0,"isRotateWithShape":false,"isScaled":false,"colorStops":[{"position":0,"color":{"type":"auto"}},{"position":1,"color":{"type":"auto"}}]}',
                  //stringifiedFeasibleDescriptor = '{"type":"linear","rotation":0,"isRotateWithShape":false,"isScaled":false,"colorStops":[{"position":0,"color":{"type":"preset","value":"6"}},{"position":1,"color":{"type":"rgb","value":"ccff00"}}]}',
                  //stringifiedFullDescriptor,    = '{"type":"linear","rotation":66.666,"isRotateWithShape":true,"isScaled":true,"colorStops":[{"position":0.525,"color":{"type":"rgb","value":"ffc000"}},{"position":0.875,"color":{"type":"preset","value":4}}]}',

                    stringifiedEmptyDescriptor    = JSON.stringify(expectedValueOfTheEmptyDescriptorGradient),
                    stringifiedFeasibleDescriptor = JSON.stringify(expectedValueOfTheFeasibleDescriptorGradient);

                  //stringifiedFullDescriptor     = JSON.stringify(fullDescriptor);

                // complete data transformation round-trip but mocked.
                expect(emptyDescriptorGradient.toString()).to.deep.equal(stringifiedEmptyDescriptor);
                expect(feasibleDescriptorGradient.toString()).to.deep.equal(stringifiedFeasibleDescriptor);

                // complete data transformation round-trip and real.
                //expect(fullDescriptorGradient.toString()).to.deep.equal(stringifiedFullDescriptor);
            });*/
        });
    });

    // ========================================================================
});
