/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

'use strict';

module.exports = function (grunt) {

    // imports ================================================================

    var _ = require('underscore');
    var Utils = require('../utils/gruntutils')(grunt);

    // configuration ==========================================================

    // Root source directory.
    var SRC_ROOT = 'resource/fonts';

    // Path to the fonts definition file.
    var FONTS_JSON_PATH = SRC_ROOT + '/fonts.json';

    // Root destination directory for all generated files.
    var DEST_ROOT = 'build/apps/io.ox/office/tk/render';

    // Path to the generated CSS file.
    var FONTS_CSS_PATH = DEST_ROOT + '/fonts.css';

    // Font style identifiers to CSS properties.
    var FONT_STYLE_MAP = {
        normal:     ['font-weight:normal;', 'font-style:normal;'],
        bold:       ['font-weight:bold;',   'font-style:normal;'],
        italic:     ['font-weight:normal;', 'font-style:italic;'],
        bolditalic: ['font-weight:bold;',   'font-style:italic;']
    };

    // task configuration =====================================================

    // specify source/destination mapping for 'newer' plug-in (every destination file depends on all source files)
    grunt.config('webfonts', {
        all: {
            files: [{ src: [FONTS_JSON_PATH, SRC_ROOT + '/*.woff'], dest: FONTS_CSS_PATH }]
        }
    });

    // task implementation ====================================================

    grunt.registerTask('webfonts', 'Assembles the CSS file from WOFF font files.', function () {

        // ensure existing destination directory
        grunt.file.mkdir(DEST_ROOT);

        // read and validate the font definitions
        var definitions = grunt.file.readJSON(FONTS_JSON_PATH);
        Utils.ensureObject(definitions, 'Invalid font definitions, expecting JSON object.');

        // the contents of the resulting CSS file
        var cssLines = [];

        // load all WOFF source files, and generate the CSS
        _.each(definitions, function (fontDef, fontFamily) {

            // process the definitions of a single font
            Utils.ensureObject(fontDef, 'Invalid font definition, expecting JSON objects.');
            _.each(fontDef, function (path, style) {

                // validate path and style
                Utils.ensureStringNE(path, 'Invalid path to source file, expecting a string.');
                Utils.ensure(style in FONT_STYLE_MAP, 'Invalid font style "' + style + '".');

                // import the WOFF file and convert it to a BASE64 encoded string
                var base64 = grunt.file.read(SRC_ROOT + '/' + path, { encoding: null }).toString('base64');

                // create the CSS code for the font
                cssLines.push('@font-face {');
                cssLines.push('font-family:"' + fontFamily + '";');
                cssLines = cssLines.concat(FONT_STYLE_MAP[style]);
                cssLines.push('src:url(data:application/font-woff;charset=utf-8;base64,' + base64 + ') format("woff");');
                cssLines.push('}');
            });
        });

        // finalize and write the CSS file
        Utils.writeLines(FONTS_CSS_PATH, cssLines);
        grunt.log.ok('Web fonts created successfully.');
    });
};
