/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

'use strict';

module.exports = function (grunt) {

    // imports ================================================================

    var _ = require('underscore');
    var Utils = require('../utils/gruntutils')(grunt);

    // configuration ==========================================================

    // Root source directory.
    var SRC_ROOT = 'resource/formula';

    // Directory name of the source files with formula resources (booleans, error codes, function names).
    var RES_DIR = SRC_ROOT + '/res';

    // Directory name of the source files with localized function help.
    var HELP_DIR = SRC_ROOT + '/help';

    // File name of the directory file with all translated function names used as function help keys.
    var NAMES_DICT_PATH = SRC_ROOT + '/namesdict.json';

    // Root destination directory for all generated files.
    var DEST_ROOT = 'build/apps/io.ox/office/spreadsheet/resource/formula';

    // helper functions =======================================================

    function getResPath(language) { return RES_DIR + '/' + language + '.json'; }
    function getHelpPath(language) { return HELP_DIR + '/' + language + '.json'; }
    function getDestPath(language) { return DEST_ROOT + '/' + language + '.json'; }

    /**
     * Loads the dictionary with function names used as function help keys, and
     * converts the key/name map to a name/key map. The dictionary is needed to
     * assign the original function help data (which is mapped by some
     * localized function names which may differ from the actual translations)
     * to the resource data mapped by the unique resource keys of the
     * functions.
     */
    var loadNamesDirectory = _.once(function () {
        return Utils.collectErrors(function (runner) {

            // load the JSON file
            var namesDictionary = grunt.file.readJSON(NAMES_DICT_PATH);
            Utils.ensureObject(namesDictionary, NAMES_DICT_PATH + ': Invalid resource, expecting an object.');

            // process all function name maps (keyed by locale code or language code), collect error messages
            // for multiple locale entries of the entire dictionary
            runner.forEach(namesDictionary, function (dict, locale) {
                Utils.ensure(/^[a-z]+(_[A-Z]+)?$/.test(locale), NAMES_DICT_PATH + ': Invalid resource key "' + locale + '", expecting a language or a locale code.');
                if (_.isString(dict)) { return; }
                Utils.ensureObject(dict, NAMES_DICT_PATH + ': Invalid resource data for locale "' + locale + '", expecting an object.');

                // swap keys and names, check for duplicates in the localized function names
                var keyMap = {};
                _.each(dict, function (names, key) {
                    if (_.isString(names)) { names = [names]; }
                    Utils.ensureArrayNE(names, NAMES_DICT_PATH + ': Invalid function name at "' + locale + '.' + key + '".');
                    names.forEach(function (name, index) {
                        Utils.ensure(!(name in keyMap), NAMES_DICT_PATH + ': Duplicate function name "' + name + '" in dictionary "' + locale + '".');
                        keyMap[name] = (index === 0) ? key : null;
                    });
                });
                namesDictionary[locale] = keyMap;
            });

            // remap aliases
            runner.forEach(namesDictionary, function (dict, locale) {
                if (_.isString(dict)) {
                    dict = namesDictionary[dict];
                    Utils.ensureObject(dict, NAMES_DICT_PATH + ': Invalid name mapping for locale "' + locale + '", expecting a valid key.');
                    namesDictionary[locale] = dict;
                }
            });

            return namesDictionary;
        });
    });

    // task configuration =====================================================

    // Build single targets for all supported languages, which is needed to
    // establish the correct source file dependencies for the 'newer' plug-in.
    // Each target file (each language resource) depends on multiple language
    // source files, and on common files that are used for all languages.
    //
    var taskConfig = {};
    grunt.file.expand(RES_DIR + '/*.json').forEach(function (path) {

        // extract the language code from the file name
        var matches = /\/([a-z]+)\.json$/.exec(path);
        Utils.ensure(matches, 'Invalid language code in file name ' + path + '.');
        var language = matches[1];

        // build the source/destination file mapping for the current language
        var srcPaths = [getResPath(language), getHelpPath(language), NAMES_DICT_PATH, 'grunt/tasks/formulares.js'];
        var destPath = getDestPath(language);
        taskConfig[language] = { src: srcPaths, dest: destPath };
    });

    // Register the task configuration.
    grunt.config('formulares', taskConfig);

    // task implementation ====================================================

    grunt.registerMultiTask('formulares', 'Assembles the Spreadsheet formula resources.', function () {

        // current language (target of the multi-task), and source file paths
        var language = this.target;
        var resPath = getResPath(language);
        var helpPath = getHelpPath(language);

        // regular expressions for validation
        var localeRE = new RegExp('^' + language + '(_[A-Z]+)?$');
        var keyRE = /^[A-Z][A-Z0-9]*(\.[A-Z0-9]+)*$/;

        // the dictionary with function names used as function help keys
        var namesDictionary = loadNamesDirectory();

        // --------------------------------------------------------------------
        // Load the base resource data (boolean names, error code names,
        // function names) for the current language, and validate the data. The
        // resource data may contain a generic map for the language, and
        // specific maps for different locales, e.g. maps for 'pt', 'pt_PT',
        // and 'pt_BR'.
        //
        var resData = grunt.file.readJSON(resPath);
        Utils.ensureObject(resData, resPath + ': Invalid resource, expecting an object.');

        // process the maps for the different locales
        Utils.collectErrors(function (collector) {

            // a character range for a valid leading character in a localized name
            var LEADING_CHAR = 'A-Z\xc0-\xd6\xd8-\xde\u0100-\u02af\u0386-\u052f\u1e00-\u1ffc';
            // a character class pattern for a valid character at any position in a localized name
            var LEADING_CHAR_PATTERN = '[' + LEADING_CHAR + ']';
            // a character class pattern for a valid character at any position in a localized name
            var OTHER_CHAR_PATTERN = '[' + LEADING_CHAR + '0-9]';
            // a pattern for a valid part of a localized name
            var NAME_PATTERN = LEADING_CHAR_PATTERN + OTHER_CHAR_PATTERN + '*';
            // a regular expression to validate error codes
            var ERROR_RE = new RegExp('^#[\xa1\xbf]?' + NAME_PATTERN + '\\.?([_/]' + OTHER_CHAR_PATTERN + '+\\.?)*[!?]?$');
            // a regular expression to validate functions
            var FUNCTION_RE = new RegExp('^' + NAME_PATTERN + '([._]' + OTHER_CHAR_PATTERN + '+)*\\.?$');
            // a regular expression to validate region names for table ranges
            var REGION_RE = new RegExp('^#.+$');
            // a regular expression to validate boolean names
            var BOOLEAN_RE = new RegExp('^' + NAME_PATTERN + '$');
            // a regular expression to validate the RC prefix characters
            var RC_PREFIX_RE = /^[A-Z]{2}$/;

            // a map with descriptors for all expected resource entries per language
            var RE_DATA_MAP = {
                errors:     { object: true,  re: ERROR_RE,     upper: true },
                regions:    { object: true,  re: REGION_RE,    upper: false },
                functions:  { object: true,  re: FUNCTION_RE,  upper: true },
                cellParams: { object: true,  re: null,         upper: false },
                FALSE:      { object: false, re: BOOLEAN_RE,   upper: true },
                TRUE:       { object: false, re: BOOLEAN_RE,   upper: true },
                RC:         { object: false, re: RC_PREFIX_RE, upper: true }
            };

            // checks the validity of the passed name, according to the regular expression for the passed map entry
            function checkName(regExpData, name, logKey) {
                Utils.ensureStringNE(name, resPath + ': Invalid resource entry, expecting a string value at key "' + logKey + '".');
                Utils.ensure(!regExpData.re || regExpData.re.test(name), resPath + ': Invalid name "' + name + '" for resource entry at key "' + logKey + '".');
                Utils.ensure(!regExpData.upper || (name === name.toUpperCase()), resPath + ': Invalid resource entry, expecting an uppercase name at key "' + logKey + '".');
            }

            collector.forEach(resData, function (localResData, locale) {
                Utils.ensure(localeRE.test(locale), resPath + ': Invalid resource key "' + locale + '", expecting the language or a matching locale code.');
                Utils.ensureObject(localResData, resPath + ': Invalid resource data for locale "' + locale + '", expecting an object.');

                // process the different resource entries (may be single strings,
                // e.g. for boolean names, or key/name maps, e.g. for function names)
                collector.forEach(localResData, function (entry, key1) {
                    var key1Name = locale + '.' + key1;
                    var regExpData = RE_DATA_MAP[key1];
                    Utils.ensureObject(regExpData, resPath + ': Invalid key "' + key1Name + '".');

                    // process the key/name maps for error codes and function names
                    if (regExpData.object) {
                        Utils.ensureObject(entry, resPath + ': Invalid resource entry, expecting a key/name map at key "' + key1Name + '".');

                        // check the validity of the unique resource keys
                        collector.forEach(entry, function (name, key2) {
                            var key2Name = key1Name + '.' + key2;
                            Utils.ensure(keyRE.test(key2), resPath + ': Invalid key "' + key2Name + '".');

                            // entries may be strings, or maps with different names for the file formats
                            if (_.isObject(name)) {
                                collector.forEach(['ooxml', 'odf'], function (format) { checkName(regExpData, name[format], key2Name + '.' + format); });
                            } else {
                                checkName(regExpData, name, key2Name);
                            }
                        });

                        // check for duplicates in the translated names
                        collector.forEach(_.countBy(_.filter(entry, _.isString)), function (count, name) {
                            Utils.ensure(count === 1, resPath + ': Duplicate translated name "' + name + '".');
                        });

                        return;
                    }

                    // all other entries must be simple strings (localized names of boolean values, etc.)
                    checkName(regExpData, entry, key1Name);
                });
            });
        });

        // --------------------------------------------------------------------
        // Load the function help data (descriptions for functions, and their
        // parameters). Function help entries are mapped by localized function
        // names (and sometimes, also by English function names). Find the
        // corresponding unique resource key of the function (using the
        // function name dictionary loaded above), and insert the function help
        // into the resource data.
        //
        if (Utils.check(grunt.file.isFile(helpPath), 'Missing help resources for language "' + language + '".')) {
            var helpData = grunt.file.readJSON(helpPath);
            Utils.ensureObject(helpData, helpPath + ': Invalid resource, expecting an object.');

            // process all help maps of the different locales
            Utils.collectErrors(function (collector) {
                collector.forEach(helpData, function (localHelpData, locale) {
                    Utils.ensureObject(localHelpData, helpPath + ': Invalid help data for locale "' + locale + '", expecting an object.');
                    Utils.ensure(locale in resData, helpPath + ': Invalid help key "' + locale + '", missing resource data.');
                    Utils.ensure(locale in namesDictionary, helpPath + ': Invalid help key "' + locale + '", missing name dictionary.');

                    // the resource data map, and the function name directory  of the current locale
                    var localResData = resData[locale];
                    var localDict = namesDictionary[locale];

                    // process the help entries of all functions
                    var newHelpData = {};
                    var mappedNames = {};
                    collector.forEach(localHelpData, function (entry, name) {
                        var key = localDict[name];
                        if (_.isNull(key)) { return; } // help entries to be ignored
                        var logKey = locale + '.' + key, logName = locale + '.' + name;
                        Utils.ensureStringNE(key, helpPath + ': Entry found for unknown function "' + logName + '".');
                        Utils.ensure(!(key in mappedNames), helpPath + ': Duplicate help entry for "' + logKey + '": ' + mappedNames[key] + ', ' + name + '.');
                        Utils.ensureObject(entry, helpPath + ': Invalid help entry, expecting an object for "' + logName + '".');
                        Utils.checkStringNE(entry.description, helpPath + ': Invalid function description for "' + logName + '".');
                        Utils.ensureArray(entry.params, helpPath + ': Invalid parameter names, expecting an array for "' + logName + '".');
                        Utils.ensureArray(entry.paramshelp, helpPath + ': Invalid parameter description, expecting an array for "' + logName + '".');
                        Utils.ensure(entry.params.length === entry.paramshelp.length, helpPath + ': Parameter array length mismatch for "' + logName + '".');
                        collector.forEach(entry.params, function (param) {
                            Utils.ensureStringNE(param, helpPath + ': Invalid parameter name for "' + logName + '".');
                        });
                        entry.params = entry.params.map(function (param, index) {
                            var desc = entry.paramshelp[index];
                            Utils.checkStringNE(desc, helpPath + ': Invalid parameter description for "' + logName + '[' + index + ']".');
                            return { n: param || '', d: _.isString(desc) ? desc : '' };
                        });
                        newHelpData[key] = { d: entry.description, p: entry.params };
                        mappedNames[key] = name;
                    });

                    // check for missing help data
// disabled for now, too many hits
//                    _.each(localResData.functions, function (name, key) {
//                        Utils.check(key in newHelpData, helpPath + ': Missing help entry for "' + locale + '.' + key + '".');
//                    });

                    // insert the help map as property 'help' into the resource data
                    localResData.help = newHelpData;
                });
            });
        }

        // --------------------------------------------------------------------
        // Write the resulting resource data

        // ensure existing destination directory, write the resulting complete resource file
        grunt.file.mkdir(DEST_ROOT);
        Utils.writeJSON(getDestPath(language), resData);
    });
};
