/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/tk/utils/domconsole', [
    'io.ox/core/event',
    'io.ox/office/tk/config',
    'io.ox/office/tk/forms'
], function (Events, Config, Forms) {

    'use strict';

    // the configuration key for the initial visibility state of the DOM console
    var STORAGE_KEY = 'office:debug:domconsole';

    // icon names for different log levels
    var ICONS = { info: 'fa-info-circle', warn: 'fa-warning', error: 'fa-times-circle' };

    // the root node of the DOM console window (created on demand)
    var rootNode = null;

    // private global functions -----------------------------------------------

    /**
     * Creates the root node of the DOM console.
     */
    function createRootNode() {

        rootNode = $('<div>', { id: 'io-ox-office-console' }).addClass('noI18n collapsed').hide();

        var outputNode = $('<div class="output">').appendTo(rootNode);
        var buttonsNode = $('<tr>').appendTo($('<table class="buttons">').appendTo(rootNode));
        var states = {};

        function scrollDown() {
            if (states.scroll) {
                outputNode.scrollTop(outputNode[0].scrollHeight);
            }
        }

        function writeToConsole(type, args) {
            var rowNode = $('<p class="' + type + '">').appendTo(outputNode).append(
                '<span class="time">' + moment().format('HH:mm:ss.SSS') + '</span>',
                '<i class="fa ' + (ICONS[type] || 'fa-none') + '">'
            );
            args.forEach(function (msg) {
                if (_.isNull(msg) || _.isUndefined(msg)) {
                    rowNode.append('<span style="font-style:italic">' + msg + '</span>');
                } else {
                    var tooltip = _.isObject(msg) ? Forms.createLoggingTooltipMarkup(msg) : null;
                    var span = '<span' + (tooltip ? (' title="' + tooltip + '"') : '') + '>';
                    rowNode.append($(span).text(msg).toggleClass('hover', !!tooltip));
                }
            });
            scrollDown();
        }

        function selectAll() {
            var docRange = window.document.createRange();
            docRange.setStart(rootNode[0], 0);
            docRange.setEnd(rootNode[0], 1);
            window.getSelection().removeAllRanges();
            window.getSelection().addRange(docRange);
        }

        function handleButtonClick(button, callback) {
            button.on({
                click: function () { callback(); return false; },
                dblclick: false
            });
        }

        function createButton(label, tooltip, callback) {
            var button = $('<td>').attr('title', tooltip).text(label).appendTo(buttonsNode);
            handleButtonClick(button, callback);
            return button;
        }

        function updateToggleButton(button, stateName, state, callback) {
            states[stateName] = state;
            button.find('>i').attr('class', 'fa ' + (state ? 'fa-check-square-o' : 'fa-square-o'));
            callback(state);
        }

        function createToggleButton(label, tooltip, stateName, initState, callback) {
            var button = createButton(label, tooltip, function () {
                updateToggleButton(button, stateName, !states[stateName], callback);
            });
            button.prepend('<i>');
            updateToggleButton(button, stateName, initState, callback);
            return button;
        }

        // create and prepare buttons
        createToggleButton('Show Time', 'Show timestamps for all messages', 'time', true, function (state) { outputNode.toggleClass('show-time', state); });
        createToggleButton('Auto Errors', 'Automatically show the console on uncaught JS exceptions', 'error', true, _.noop);
        createToggleButton('Auto Scroll', 'Automatically scroll down after new messages', 'scroll', true, _.noop);
        createButton('Select All', 'Select all messages', selectAll);
        createButton('Clear', 'Remove all messages', function () { outputNode.empty(); });
        createButton('Hide', 'Collapse this console window', function () { rootNode.addClass('collapsed'); });

        // show console when clicking on the collapsed icon
        rootNode.append('<i class="fa fa-bug">');
        handleButtonClick(rootNode, function () {
            rootNode.removeClass('collapsed');
            scrollDown();
        });

        // replace implementations of browser console methods
        ['log', 'info', 'warn', 'error'].forEach(function (methodName) {
            window.console[methodName] = _.wrap(window.console[methodName], function (origMethod) {
                var args = _.toArray(arguments).slice(1);
                writeToConsole(methodName, args);
                if (_.isFunction(origMethod)) { return origMethod.apply(this, args); }
            });
        });

        // global exception handler to forward unhandled exceptions to the DOM console
        $(window).on('error', function () {
            if (states.error) { rootNode.removeClass('collapsed'); }
            writeToConsole('error', _.toArray(arguments));
        });

        // load CSS formatting on demand (not globally before using the DOM console)
        require(['less!io.ox/office/tk/utils/domconsole']).done(function () {
            rootNode.show();
        });
    }

    // static class DOMConsole ================================================

    /**
     * An overlay DOM element that mirrors the debug log of the browser.
     * Especially useful for mobile devices, and for a better stability (some
     * browsers, e.g. Chrome, have problems with very large console logs).
     */
    var DOMConsole = {};

    // Events interface -------------------------------------------------------

    Events.extend(DOMConsole);

    // static methods ---------------------------------------------------------

    /**
     * Returns whether the DOM console is currently visible.
     *
     * @returns {Boolean}
     *  Whether the DOM console is currently visible.
     */
    DOMConsole.isVisible = Config.DEBUG ? function () {
        return localStorage.getItem(STORAGE_KEY) !== null;
    } : _.constant(false);

    /**
     * Shows the DOM console element in the UI.
     */
    DOMConsole.show = Config.DEBUG ? function () {
        if (!rootNode) { createRootNode(); }
        rootNode.appendTo('body');
        localStorage.setItem(STORAGE_KEY, '1');
        DOMConsole.trigger('change:state', true);
    } : _.noop;

    /**
     * Hides the DOM console element from the UI.
     */
    DOMConsole.hide = Config.DEBUG ? function () {
        if (rootNode) { rootNode.detach(); }
        localStorage.removeItem(STORAGE_KEY);
        DOMConsole.trigger('change:state', false);
    } : _.noop;

    /**
     * Toggles the visibility of the DOM console element in the UI.
     *
     * @param {Boolean} state
     *  The new visibility state of the DOM console.
     */
    DOMConsole.toggle = function (state) {
        if (state) {
            DOMConsole.show();
        } else {
            DOMConsole.hide();
        }
    };

    // static initialization ==================================================

    // show the DOM console, if the configuration item has been set in a previous session
    if (DOMConsole.isVisible()) { DOMConsole.show(); }

    // exports ================================================================

    return DOMConsole;

});
