/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/tk/render/rectangle', [
    'io.ox/office/tk/utils'
], function (Utils) {

    'use strict';

    // convenience shortcuts
    var abs = Math.abs;
    var max = Math.max;
    var sin = Math.sin;
    var cos = Math.cos;
    var atan2 = Math.atan2;
    var radius = Utils.radius;

    // private global functions ===============================================

    /**
     * Returns the offset of the right border of an arbitrary JS object that
     * represents a rectangle.
     *
     * @param {Object} rect
     *  An object with the numeric properties 'left' and 'width'.
     *
     * @returns {Number}
     *  The offset of the right border of the passed rectangle.
     */
    function right(rect) {
        return rect.left + rect.width;
    }

    /**
     * Returns the offset of the lower border of an arbitrary JS object that
     * represents a rectangle.
     *
     * @param {Object} rect
     *  An object with the numeric properties 'top' and 'height'.
     *
     * @returns {Number}
     *  The offset of the lower border of the passed rectangle.
     */
    function bottom(rect) {
        return rect.top + rect.height;
    }

    /**
     * Converts cartesian coordiantes to polar coordinates.
     *
     * @param {Number} x
     *  The X coordinate of the point.
     *
     * @param {Number} y
     *  The Y coordinate of the point.
     *
     * @returns {Object}
     *  The polar coordinates, as object with the properties 'r' and 'a'.
     */
    function toPolar(x, y) {
        return { r: radius(x, y), a: atan2(y, x) };
    }

    /**
     * Converts polar coordiantes to cartesian coordinates.
     *
     * @param {Number} r
     *  The radius, i.e. the distance between origin and the point.
     *
     * @param {Number} [a]
     *  The angle between the positive X axis and radius, in radians.
     *
     * @returns {Object}
     *  The cartesian coordinates, as object with the properties 'x' and 'y'.
     */
    function toCartesian(r, a) {
        return { x: r * cos(a), y: r * sin(a) };
    }

    // class Rectangle ========================================================

    /**
     * Helper structure that representa an arbitrary rectangle with a start
     * position, and a size.
     *
     * @constructor
     *
     * @property {Number} left
     *  Horizontal start coordinate of the rectangle. Can be any floating-point
     *  number, including negative offsets.
     *
     * @property {Number} top
     *  Vertical start coordinate of the rectangle. Can be any floating-point
     *  number, including negative offsets.
     *
     * @property {Number} width
     *  The width of the rectangle. Must not be negative. May be zero, if the
     *  rectangle represents a vertical hair line.
     *
     * @property {Number} height
     *  The height of the rectangle. Must not be negative. May be zero, if the
     *  rectangle represents a horizontal hair line.
     */
    function Rectangle(left, top, width, height) {

        this.left = left;
        this.top = top;
        this.width = Math.max(0, width);
        this.height = Math.max(0, height);

    } // class Rectangle

    // static methods ---------------------------------------------------------

    /**
     * Constructs a rectangle from the passed plain JS object looking like a
     * rectangle, including instances of this class (copy construction).
     *
     * @param {Object} obj
     *  An arbitrary object with the numeric properties 'left', 'top', 'width',
     *  and 'height'.
     *
     * @returns {Rectangle}
     *  A new instance of the class Rectangle at the specified location.
     */
    Rectangle.from = function (obj) {
        return new Rectangle(obj.left, obj.top, obj.width, obj.height);
    };

    // public methods ---------------------------------------------------------

    /**
     * Returns a clone of this rectangle instance.
     *
     * @returns {Rectangle}
     *  A clone of this rectangle instance.
     */
    Rectangle.prototype.clone = function () {
        return new Rectangle(this.left, this.top, this.width, this.height);
    };

    /**
     * Returns whether the passed rectangle is equal to this rectangle.
     *
     * @param {Rectangle|Object} rect
     *  The other rectangle to be compared to this rectangle. Instead of an
     *  instance of this class, this parameter can be an arbitrary JS object
     *  containing the properies of a rectangle.
     *
     * @returns {Boolean}
     *  Whether both rectangles are exactly equal.
     */
    Rectangle.prototype.equals = function (rect) {
        return (this.left === rect.left) && (this.top === rect.top) && (this.width === rect.width) && (this.height === rect.height);
    };

    /**
     * Returns whether the passed rectangle differs from this rectangle.
     *
     * @param {Rectangle|Object} rect
     *  The other rectangle to be compared to this rectangle. Instead of an
     *  instance of this class, this parameter can be an arbitrary JS object
     *  containing the properies of a rectangle.
     *
     * @returns {Boolean}
     *  Whether the passed rectangle differs from this rectangle.
     */
    Rectangle.prototype.differs = function (rect) {
        return !this.equals(rect);
    };

    /**
     * Returns the coordinate of the right border of this rectangle.
     *
     * @returns {Number}
     *  The coordinate of the right border of this rectangle.
     */
    Rectangle.prototype.right = function () {
        return right(this);
    };

    /**
     * Returns the coordinate of the lower border of this rectangle.
     *
     * @returns {Number}
     *  The coordinate of the lower border of this rectangle.
     */
    Rectangle.prototype.bottom = function () {
        return bottom(this);
    };

    /**
     * Returns the X coordinate of the center point of this rectangle.
     *
     * @returns {Number}
     *  The X coordinate of the center point of this rectangle.
     */
    Rectangle.prototype.centerX = function () {
        return this.left + this.width / 2;
    };

    /**
     * Returns the Y coordinate of the center point of this rectangle.
     *
     * @returns {Number}
     *  The Y coordinate of the center point of this rectangle.
     */
    Rectangle.prototype.centerY = function () {
        return this.top + this.height / 2;
    };

    /**
     * Returns the length of the diagonals of this rectangle.
     *
     * @returns {Number}
     *  The length of the diagonals of this rectangle.
     */
    Rectangle.prototype.diag = function () {
        return radius(this.width, this.height);
    };

    /**
     * Returns area size of this rectangle.
     *
     * @returns {Number}
     *  The area size of this rectangle.
     */
    Rectangle.prototype.area = function () {
        return this.width * this.height;
    };

    /**
     * Returns whether this rectangle is a square.
     *
     * @returns {Boolean}
     *  Whether this rectangle is a square.
     */
    Rectangle.prototype.square = function () {
        return this.width === this.height;
    };

    /**
     * Returns the position of the top-left corner of this rectangle, as JSON
     * point.
     *
     * @returns {Object}
     *  The position of the top-left corner of this rectangle, as JSON point
     *  with the properties 'x' and 'y'.
     */
    Rectangle.prototype.topLeft = function () {
        return { x: this.left, y: this.top };
    };

    /**
     * Returns the position of the top-right corner of this rectangle, as JSON
     * point.
     *
     * @returns {Object}
     *  The position of the top-right corner of this rectangle, as JSON point
     *  with the properties 'x' and 'y'.
     */
    Rectangle.prototype.topRight = function () {
        return { x: this.right(), y: this.top };
    };

    /**
     * Returns the position of the bottom-left corner of this rectangle, as
     * JSON point.
     *
     * @returns {Object}
     *  The position of the bottom-left corner of this rectangle, as JSON point
     *  with the properties 'x' and 'y'.
     */
    Rectangle.prototype.bottomLeft = function () {
        return { x: this.left, y: this.bottom() };
    };

    /**
     * Returns the position of the bottom-right corner of this rectangle, as
     * JSON point.
     *
     * @returns {Object}
     *  The position of the bottom-right corner of this rectangle, as JSON
     *  point with the properties 'x' and 'y'.
     */
    Rectangle.prototype.bottomRight = function () {
        return { x: this.right(), y: this.bottom() };
    };

    /**
     * Returns the position of the center of the left border of this rectangle,
     * as JSON point.
     *
     * @returns {Object}
     *  The position of the center of the left border of this rectangle, as
     *  JSON point with the properties 'x' and 'y'.
     */
    Rectangle.prototype.leftCenter = function () {
        return { x: this.left, y: this.centerY() };
    };

    /**
     * Returns the position of the center of the right border of this
     * rectangle, as JSON point.
     *
     * @returns {Object}
     *  The position of the center of the right border of this rectangle, as
     *  JSON point with the properties 'x' and 'y'.
     */
    Rectangle.prototype.rightCenter = function () {
        return { x: this.right(), y: this.centerY() };
    };

    /**
     * Returns the position of the center of the top border of this rectangle,
     * as JSON point.
     *
     * @returns {Object}
     *  The position of the center of the top border of this rectangle, as JSON
     *  point with the properties 'x' and 'y'.
     */
    Rectangle.prototype.topCenter = function () {
        return { x: this.centerX(), y: this.top };
    };

    /**
     * Returns the position of the center of the bottom border of this
     * rectangle, as JSON point.
     *
     * @returns {Object}
     *  The position of the center of the bottom border of this rectangle, as
     *  JSON point with the properties 'x' and 'y'.
     */
    Rectangle.prototype.bottomCenter = function () {
        return { x: this.centerX(), y: this.bottom() };
    };

    /**
     * Returns the position of the center of this rectangle, as JSON point.
     *
     * @returns {Object}
     *  The position of the center of this rectangle, as JSON point with the
     *  properties 'x' and 'y'.
     */
    Rectangle.prototype.center = function () {
        return { x: this.centerX(), y: this.centerY() };
    };

    /**
     * Returns whether this rectangle contains a specific X coordinate.
     *
     * @param {Number} x
     *  The X coordinate to be checked.
     *
     * @returns {Boolean}
     *  Whether the passed X coordinate is inside this rectangle.
     */
    Rectangle.prototype.containsX = function (x) {
        return (this.left <= x) && (x <= right(this));
    };

    /**
     * Returns whether this rectangle contains a specific Y coordinate.
     *
     * @param {Number} y
     *  The Y coordinate to be checked.
     *
     * @returns {Boolean}
     *  Whether the passed Y coordinate is inside this rectangle.
     */
    Rectangle.prototype.containsY = function (y) {
        return (this.top <= y) && (y <= bottom(this));
    };

    /**
     * Returns whether this rectangle contains a specific dimensionless (i.e.
     * of size 0x0) point.
     *
     * @param {Object|Number} x
     *  The X coordinate of the point to be checked. This parameter can also be
     *  a JSON object with the numeric properties 'x' and 'y'.
     *
     * @param {Number} [y]
     *  The Y coordinate of the point to be checked. This parameter will be
     *  ignored, if the first parameter is an object.
     *
     * @returns {Boolean}
     *  Whether the specified point is inside this rectangle.
     */
    Rectangle.prototype.containsPoint = function (x, y) {
        if (typeof x === 'object') { y = x.y; x = x.x; }
        return this.containsX(x) && this.containsY(y);
    };

    /**
     * Returns whether this rectangle contains an entire pixel (a rectangle
     * with size 1x1).
     *
     * @param {Object|Number} x
     *  The X coordinate of the left border of the pixel to be checked. This
     *  parameter can also be a JSON object with the numeric properties 'x' and
     *  'y'.
     *
     * @param {Number} [y]
     *  The Y coordinate of the top border of the pixel to be checked. This
     *  parameter will be ignored, if the first parameter is an object.
     *
     * @returns {Boolean}
     *  Whether the specified pixel is inside this rectangle.
     */
    Rectangle.prototype.containsPixel = function (x, y) {
        if (typeof x === 'object') { y = x.y; x = x.x; }
        return (this.left <= x) && (x + 1 <= right(this)) && (this.top <= y) && (y + 1 <= bottom(this));
    };

    /**
     * Returns whether this rectangle completely contains the passed rectangle.
     *
     * @param {Rectangle|Object} rect
     *  The rectangle to be checked. Instead of an instance of this class, this
     *  parameter can be an arbitrary JS object containing the properies of a
     *  rectangle.
     *
     * @returns {Boolean}
     *  Whether the passed rectangle is completely located inside this
     *  rectangle.
     */
    Rectangle.prototype.contains = function (rect) {
        return (this.left <= rect.left) && (right(rect) <= right(this)) && (this.top <= rect.top) && (bottom(rect) <= bottom(this));
    };

    /**
     * Returns whether this rectangle and the passed rectangle overlap each
     * other.
     *
     * @param {Rectangle|Object} rect
     *  The rectangle to be checked. Instead of an instance of this class, this
     *  parameter can be an arbitrary JS object containing the properies of a
     *  rectangle.
     *
     * @returns {Boolean}
     *  Whether the passed rectangle overlaps with this rectangle.
     */
    Rectangle.prototype.overlaps = function (rect) {
        return (this.left < right(rect)) && (rect.left < right(this)) && (this.top < bottom(rect)) && (rect.top < bottom(this));
    };

    /**
     * Returns the bounding rectangle of this rectangle and the passed
     * rectangle (the smallest rectangle that contains both rectangles).
     *
     * @param {Rectangle|Object} rect
     *  The other rectangle. Instead of an instance of this class, this
     *  parameter can be an arbitrary JS object containing the properies of a
     *  rectangle.
     *
     * @returns {Rectangle}
     *  The bounding rectangle containing both rectangles.
     */
    Rectangle.prototype.boundary = function (rect) {

        var l = Math.min(this.left, rect.left);
        var t = Math.min(this.top, rect.top);
        var r = Math.max(right(this), right(rect));
        var b = Math.max(bottom(this), bottom(rect));

        return new Rectangle(l, t, r - l, b - t);
    };

    /**
     * Returns the common area of this rectangle and the passed rectangle.
     *
     * @param {Rectangle|Object} rect
     *  The other rectangle. Instead of an instance of this class, this
     *  parameter can be an arbitrary JS object containing the properies of a
     *  rectangle.
     *
     * @returns {Rectangle|Null}
     *  The location of the rectangle covered by this rectangle and the passed
     *  rectangle, if existing; otherwise null. If one of the rectangles has
     *  zero width or height and is covered by the other rectangle, the
     *  resulting rectangle will have zero width or height too. Otherwise, the
     *  rectangles must overlap in order to return an existing intersection
     *  rectangle.
     */
    Rectangle.prototype.intersect = function (rect) {

        var l = Math.max(this.left, rect.left);
        var t = Math.max(this.top, rect.top);
        var r = Math.min(right(this), right(rect));
        var b = Math.min(bottom(this), bottom(rect));

        if (((l < r) || ((l === r) && ((this.width === 0) || (rect.width === 0)))) &&
            ((t < b) || ((t === b) && ((this.height === 0) || (rect.height === 0))))
        ) {
            return new Rectangle(l, t, r - l, b - t);
        }

        return null;
    };

    /**
     * Returns all areas of this rectangle that are NOT contained in the passed
     * rectangle.
     *
     * @param {Rectangle|Object} rect
     *  The rectangle to be subtracted from this rectangle. Instead of an
     *  instance of this class, this parameter can be an arbitrary JS object
     *  containing the properies of a rectangle.
     *
     * @returns {Array<Rectangle>}
     *  The areas that are contained in this rectangle but not in the passed
     *  rectangle. Will be an empty array, if the passed rectangle completely
     *  covers this rectangle. Will be an array with a clone of this rectangle,
     *  if the passed rectangle is completely outside this rectangle.
     */
    Rectangle.prototype.remaining = function (rect) {

        // check if the passed rectangle covers this rectangle at all
        if (!this.overlaps(rect)) { return [this.clone()]; }

        // the resulting rectangles
        var rectangles = [];

        // shortcuts to the coordinates of this rectangle
        var l1 = this.left, t1 = this.top, w1 = this.width, r1 = right(this), b1 = bottom(this);
        // shortcuts to the coordinates of the other rectangle
        var l2 = rect.left, t2 = rect.top, r2 = right(rect), b2 = bottom(rect);
        // vertical coordinates of resulting areas left/right of rect
        var t = Math.max(t1, t2), b = Math.min(b1, b2), h = b - t;

        // if this starts above rect, extract the upper part of this
        if (t1 < t2) { rectangles.push(new Rectangle(l1, t1, w1, t2 - t1)); }
        // if this starts left of rect, extract the left part of this
        if (l1 < l2) { rectangles.push(new Rectangle(l1, t, l2 - l1, h)); }
        // if this ends right of rect, extract the right part of this
        if (r2 < r1) { rectangles.push(new Rectangle(r2, t, r1 - r2, h)); }
        // if this ends below rect, extract the lower part of this
        if (b2 < b1) { rectangles.push(new Rectangle(l1, b2, w1, b1 - b2)); }

        return rectangles;
    };

    /**
     * Returns the polar coordinates of the specified point, relative to the
     * center of this rectangle.
     *
     * @param {Object|Number} x
     *  The X coordinate of the point. This parameter can also be a JSON object
     *  with the numeric properties 'x' and 'y'.
     *
     * @param {Number} [y]
     *  The Y coordinate of the point. This parameter will be ignored, if the
     *  first parameter is an object.
     *
     * @returns {Object}
     *  The polar coordinates, as object with the following properties:
     *  - {Number} r
     *      The radius, i.e. the distance between the point and the center of
     *      this rectangle.
     *  - {Number} a
     *      The angle between the radius and the positive X axis, in radians.
     *      This value will be positive, if the Y coordinate is 'above' the X
     *      axis, i.e. greater than the Y coordinate of the center point.
     *      Furthermore, this value will be zero, if the passed point is equal
     *      to the center of this rectangle.
     */
    Rectangle.prototype.pointToPolar = function (x, y) {
        if (typeof x === 'object') { y = x.y; x = x.x; }
        return toPolar(x - this.centerX(), y - this.centerY());
    };

    /**
     * Returns the absolute cartesian coordinates of the specified point, given
     * as polar coordinates relative to the center of this rectangle.
     *
     * @param {Object|Number} r
     *  The radius, i.e. the distance between the point and the center of this
     *  rectangle. This parameter can also be a JSON object with the numeric
     *  properties 'r' and 'a'.
     *
     * @param {Number} [a]
     *  The angle between the radius and the positive X axis, in radians. This
     *  parameter will be ignored, if the first parameter is an object.
     *
     * @returns {Object}
     *  The absolute position of the point, as JSON point with the properties
     *  'x' and 'y'.
     */
    Rectangle.prototype.polarToPoint = function (r, a) {
        if (typeof r === 'object') { a = r.a; r = r.r; }
        var point = toCartesian(r, a);
        point.x += this.centerX();
        point.y += this.centerY();
        return point;
    };

    /**
     * Rotates the specified dimensionless (i.e. of size 0x0) point around the
     * center of this rectangle.
     *
     * @param {Number} a
     *  The rotation angle, in radians.
     *
     * @param {Object|Number} x
     *  The X coordinate of the point to be rotated. This parameter can also be
     *  a JSON object with the numeric properties 'x' and 'y'.
     *
     * @param {Number} [y]
     *  The Y coordinate of the point to be rotated. This parameter will be
     *  ignored, if the first parameter is an object.
     *
     * @returns {Boolean}
     *  The absolute position of the rotated point, as JSON point with the
     *  properties 'x' and 'y'.
     */
    Rectangle.prototype.rotatePoint = function (a, x, y) {
        if (typeof x === 'object') { y = x.y; x = x.x; }
        var p = this.pointToPolar(x, y);
        return this.polarToPoint(p.r, a + p.a);
    };

    /**
     * Returns the location of the bounding box covered by this rectangle, if
     * it will be rotated around its center point by the specified angle.
     *
     * @param {Number} a
     *  The rotation angle for this rectangle, in radians.
     *
     * @returns {Rectangle}
     *  The location of the bounding box of the rotated rectangle.
     */
    Rectangle.prototype.rotatedBoundingBox = function (a) {

        // the polar coordinates of the top-left corner relative to the center point
        var p = this.pointToPolar(this.left, this.top);

        // the rotated location of the corner p1, relative to the center
        var p1 = toCartesian(p.r, a + p.a);
        // the rotated location of corner p2 (corner p1 mirrored at X axis), relative to the center
        var p2 = toCartesian(p.r, a - p.a);

        // the maximum distances of the rotated corners from the X/Y axes
        // (half of the size of the resulting bounding raectangle)
        var w2 = max(abs(p1.x), abs(p2.x));
        var h2 = max(abs(p1.y), abs(p2.y));

        // the location of the bounding box
        var center = this.center();
        return new Rectangle(center.x - w2, center.y - h2, 2 * w2, 2 * h2);
    };

    /**
     * Initializes this instance with the passed coordinates.
     *
     * @param {Number} left
     *  The new horizontal start coordinate for this rectangle.
     *
     * @param {Number} top
     *  The new vertical start coordinate for this rectangle.
     *
     * @param {Number} width
     *  The new width for this rectangle.
     *
     * @param {Number} height
     *  The new height for this rectangle.
     *
     * @returns {Rectangle}
     *  A reference to this instance.
     */
    Rectangle.prototype.set = function (left, top, width, height) {
        this.left = left;
        this.top = top;
        this.width = width;
        this.height = height;
        return this;
    };

    /**
     * Sets the contents of this rectangle to the contents of the passed
     * rectangle.
     *
     * @param {Rectangle|Object} rect
     *  The rectangle to be assigned to this rectangle. Instead of an instance
     *  of this class, this parameter can be an arbitrary JS object containing
     *  the properies of a rectangle.
     *
     * @returns {Rectangle}
     *  A reference to this instance.
     */
    Rectangle.prototype.assign = function (rect) {
        this.left = rect.left;
        this.top = rect.top;
        this.width = rect.width;
        this.height = rect.height;
        return this;
    };

    /**
     * Expands the location of this rectangle in-place.
     *
     * @param {Number} dl
     *  The distance to move the left border to the left. Positive numbers will
     *  expand the rectangle (property 'left' will be decreased), and negative
     *  numbers will shrink the rectangle (property 'left' will be increased).
     *
     * @param {Number} [dt]
     *  The distance to move the top border to the top. Positive numbers will
     *  expand the rectangle (property 'top' will be decreased), and negative
     *  numbers will shrink the rectangle (property 'top' will be increased).
     *  If this parameter will be omitted (one parameter passed only), the
     *  value of 'dl' will be used for this border.
     *
     * @param {Number} [dr]
     *  The distance to move the right border to the right. Positive numbers
     *  will expand the rectangle (property 'width' will be increased), and
     *  negative numbers will shrink the rectangle (property 'width' will be
     *  decreased). If this parameter will be omitted (one or two parameters
     *  passed only), the value of 'dl' will be used for this border.
     *
     * @param {Number} [db]
     *  The distance to move the bottom border to the bottom. Positive numbers
     *  will expand the rectangle (property 'height' will be increased), and
     *  negative numbers will shrink the rectangle (property 'height' will be
     *  decreased). If this parameter will be omitted (one, two, or three
     *  parameters passed only), the value of 'dt' will be used for this border
     *  (if 'dt' has been omitted too, 'dl' will be used).
     *
     * @returns {Rectangle}
     *  A reference to this instance.
     */
    Rectangle.prototype.expandSelf = function (dl, dt, dr, db) {
        switch (arguments.length) {
            case 1: dt = dr = db = dl; break;
            case 2: dr = dl; db = dt; break;
            case 3: db = dt; break;
        }
        this.left -= dl;
        this.top -= dt;
        this.width = Math.max(0, this.width + dl + dr);
        this.height = Math.max(0, this.height + dt + db);
        return this;
    };

    /**
     * Moves the location of this rectangle in-place.
     *
     * @param {Number} dx
     *  The horizontal move distance. Negative values will move the rectangle
     *  to the left.
     *
     * @param {Number} dy
     *  The vertical move distance. Negative values will move the rectangle up.
     *
     * @returns {Rectangle}
     *  A reference to this instance.
     */
    Rectangle.prototype.translateSelf = function (dx, dy) {
        this.left += dx;
        this.top += dy;
        return this;
    };

    /**
     * Scales the location of this rectangle in-place. The start position and
     * size of this rectangle will be scaled relative to the origin (0, 0).
     *
     * @param {Number} fx
     *  The horizontal scaling factor. MUST be positive.
     *
     * @param {Number} fy
     *  The vertical scaling factor. MUST be positive.
     *
     * @returns {Rectangle}
     *  A reference to this instance.
     */
    Rectangle.prototype.scaleSelf = function (fx, fy) {
        this.left *= fx;
        this.top *= fy;
        this.width *= fx;
        this.height *= fy;
        return this;
    };

    /**
     * Rotates this rectangle by 90 degrees in-place around its center point.
     *
     * @returns {Rectangle}
     *  A reference to this instance.
     */
    Rectangle.prototype.rotateSelf = function () {
        var diff = this.width - this.height;
        this.left += diff / 2;
        this.top -= diff / 2;
        this.width -= diff;
        this.height += diff;
        return this;
    };

    /**
     * Rounds all properties of this rectangle to integers.
     *
     * @returns {Rectangle}
     *  A reference to this instance.
     */
    Rectangle.prototype.roundSelf = function () {
        this.left = Math.round(this.left);
        this.top = Math.round(this.top);
        this.width = Math.round(this.width);
        this.height = Math.round(this.height);
        return this;
    };

    /**
     * Returns the location of this rectangle as CSS property set.
     *
     * @param {String} [unit='px']
     *  The CSS unit to be added to the CSS property values.
     *
     * @returns {Object}
     *  The location of this rectangle as CSS property set.
     */
    Rectangle.prototype.toCSS = function (unit) {
        unit = unit || 'px';
        return { left: this.left + unit, top: this.top + unit, width: this.width + unit, height: this.height + unit };
    };

    /**
     * Returns the string representation of this rectangle, for debug logging.
     *
     * @returns {String}
     *  The string representation of this rectangle, for debug logging.
     */
    Rectangle.prototype.toString = function () {
        return '{l=' + this.left + ' t=' + this.top + ' w=' + this.width + ' h=' + this.height + '}';
    };

    /**
     * Returns the location of this rectangle as plain JSON object.
     *
     * @returns {Object}
     *  The location of this rectangle as plain JSON object.
     */
    Rectangle.prototype.toJSON = function () {
        return { left: this.left, top: this.top, width: this.width, height: this.height };
    };

    // exports ================================================================

    return Rectangle;

});
