/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/tk/render/font', [
    'io.ox/office/tk/utils',
    'io.ox/office/tk/locale/localedata',
    'io.ox/office/tk/render/canvas'
], function (Utils, LocaleData, Canvas) {

    'use strict';

    // global helper node for calculation of normal line height of a font
    var LINEHEIGHT_NODE = (function () {
        // create the complete HTML mark-up for ten text lines (IE uses fractional line heights internally)
        var helperNode = $('<div class="font-metrics-helper">' + Utils.repeatString('<span>X</span>', 10, '<br>') + '</div>')[0];
        Utils.insertHiddenNodes(helperNode);
        return helperNode;
    }());

    // global helper node for calculation of the base line position of a font
    var BASELINE_NODE = (function () {
        var helperNode = $('<div class="font-metrics-helper"><span>X</span><span style="display:inline-block;width:1px;height:0;"></span></div>')[0];
        Utils.insertHiddenNodes(helperNode);
        return helperNode;
    }());

    // global helper node for splitting text into multiple lines
    var TEXTLINES_NODE = (function () {
        var helperNode = $('<div class="font-metrics-helper" style="white-space:pre-wrap;word-wrap:break-word;vertical-align:top;">')[0];
        Utils.insertHiddenNodes(helperNode);
        return helperNode;
    }());

    // global helper node for detection of writing direction
    var BIDI_NODE = (function () {
        var helperNode = $('<div class="font-metrics-helper" style="width:0;"><span style="position:relative;">X</span></div>')[0];
        Utils.insertHiddenNodes(helperNode);
        return helperNode;
    }());

    // global canvas for font width calculation
    var TEXTWIDTH_CANVAS = (function () {
        var canvas = new Canvas(Canvas.SINGLETON);
        // Chrome reports different/wrong text widths, if the canvas element remains detached... :-O
        Utils.insertHiddenNodes(canvas.getNode());
        return canvas;
    }());

    // global helper node for custom font metrics calculation
    var CUSTOM_NODE = (function () {
        var helperNode = $('<div class="font-metrics-helper">')[0];
        Utils.insertHiddenNodes(helperNode);
        return helperNode;
    }());

    // cache for calculated font metrics, mapped by font keys
    var fontMetricsCache = {};

    // private global functions ===============================================

    /**
     * Fetches, caches, and returns a specific font metrics property. Invokes
     * If a cache key has been specified, and the font metrics cache contains
     * an entry for the passed font and cache key, that cached value will be
     * returned. Otherwise, the callback function will be invoked, and its
     * result will be cached.
     *
     * @param {Font} font
     *  The font settings.
     *
     * @param {Function} callback
     *  The callback function that will be invoked, unless the font metrics
     *  cache already contains a cached value for the font and cache key. Will
     *  be invoked with the passed font instance as calling context. The return
     *  value of this function will be cached if specified (see parameter
     *  'cacheKey' below), and will be returned by this method.
     *
     * @param {String} [cacheKey]
     *  If specified, a map key for the font metrics cache that will be checked
     *  first. An existing cache entry will be returned without invoking the
     *  callback function. Otherwise, the new value returned by the callback
     *  function will be inserted into the cache.
     *
     * @returns {Any}
     *  The return value of the callback function.
     */
    function fetchAndCacheResult(font, callback, cacheKey) {

        // the unique key of the passed font
        var fontKey = font.key();
        // all known cached metrics settings for the font
        var cacheEntry = _.isString(cacheKey) ? (fontMetricsCache[fontKey] || (fontMetricsCache[fontKey] = {})) : null;

        // look for an existing value in the cache
        if (cacheEntry && (cacheKey in cacheEntry)) {
            return cacheEntry[cacheKey];
        }

        // fetch result from the specified callback function
        var result = callback.call(font);

        // insert the result into the cache entry
        if (cacheEntry) { cacheEntry[cacheKey] = result; }
        return result;
    }

    /**
     * Prepares a helper DOM node used to calculate a specific font metrics
     * property, and invokes the passed callback function.
     *
     * @param {Font} font
     *  The font settings for the helper node.
     *
     * @param {HTMLElement} node
     *  The DOM element to be prepared with the passed font settings.
     *
     * @param {Function} callback
     *  The callback function that will be invoked after preparing the passed
     *  helper node. Will be invoked with the passed font instance as calling
     *  context. The return value of this function will be cached if specified
     *  (see parameter 'cacheKey' below), and will be returned by this method.
     *
     * @param {String} [cacheKey]
     *  If specified, a map key for the font metrics cache that will be checked
     *  first. An existing cache entry will be returned without invoking the
     *  callback function. Otherwise, the new value returned by the callback
     *  function will be inserted into the cache.
     *
     * @returns {Any}
     *  The return value of the callback function.
     */
    function calculateFontMetrics(font, node, callback, cacheKey) {
        return fetchAndCacheResult(font, function () {

            // initialize formatting of the helper node
            node.style.fontFamily = font.family;
            node.style.fontSize = font.size + 'pt';
            node.style.fontWeight = font.bold ? 'bold' : 'normal';
            node.style.fontStyle = font.italic ? 'italic' : 'normal';

            // fetch result from the specified callback function
            return callback.call(font, node);

        }, cacheKey);
    }

    /**
     * Prepares the helper canvas used to calculate specific text widths, and
     * invokes the passed callback function.
     *
     * @param {Font} font
     *  The font settings for the canvas.
     *
     * @param {Function} callback
     *  The callback function that will be invoked after preparing the helper
     *  helper canvas. Will be invoked with the passed font instance as calling
     *  context, and receives the prepared rendering context of the canvas
     *  element as first parameter. Must return the resulting text width that
     *  will be returned to the caller of this method, and that will be stored
     *  in the cache if needed.
     *
     * @param {String} [cacheKey]
     *  If specified, a map key for the text width cache that will be checked
     *  first. An existing cache entry will be returned without invoking the
     *  callback function. Otherwise, the new value returned by the callback
     *  function will be inserted into the cache.
     *
     * @returns {Any}
     *  The return value of the callback function.
     */
    function calculateTextWidth(font, callback, cacheKey) {
        return fetchAndCacheResult(font, function () {

            // the result of the callback function
            var result = null;

            // initialize font in the rendering context, fetch callback result
            TEXTWIDTH_CANVAS.render(function (context) {
                context.setFontStyle({ font: font.getCanvasFont() });
                result = callback.call(font, context);
            });

            // insert the result into the cache entry
            return result;

        }, cacheKey);
    }

    // class Font =============================================================

    /**
     * Helper structure used to transport font information, and to receive
     * specific font measurements, such as the normal line height, the position
     * of the font base line, or the exact width of arbitrary text.
     *
     * @constructor
     *
     * @property {String} family
     *  The CSS font family, including all known fall-back fonts.
     *
     * @property {Number} size
     *  The font size, in points, rounded to 1/10 of points. Note that the unit
     *  string 'pt' has to be added to this number when setting this font size
     *  as CSS property.
     *
     * @property {Boolean} bold
     *  Whether the font is set to bold characters.
     *
     * @property {Boolean} italic
     *  Whether the font is set to italic characters.
     */
    function Font(family, size, bold, italic) {

        this.family = family.toLowerCase();
        this.size = Utils.round(size, 0.1);
        this.bold = bold === true;
        this.italic = italic === true;

    } // class Font

    // constants --------------------------------------------------------------

    /**
     * A flag specifying whether the automatic detection of the default writing
     * direction of arbitrary texts is supported by the current browser.
     * Technically, this flag specifies whether the HTML element attribute
     * 'dir' supports the value 'auto'.
     *
     * @constant
     */
    Font.AUTO_BIDI_SUPPORT = (function () {
        try {
            // IE and PhantomJS throw when executing this assignment
            BIDI_NODE.dir = 'auto';
            return true;
        } catch (ex) {
            Utils.warn('Font: no support for automatic detection of writing direction');
        }
        return false;
    }());

    // static methods ---------------------------------------------------------

    /**
     * Determines the default writing direction of the passed text, based on
     * the first strong character in the text (that is significant for
     * bidirectional writing), and the current browser's implementation of the
     * Unicode Bidirectional Algorithm (UBA).
     *
     * @param {String} text
     *  The text to be examined.
     *
     * @returns {String}
     *  The string 'ltr', if the text should be written from left to right; or
     *  'rtl', if the text should be written from right to left.
     */
    Font.getTextDirection = (function () {

        // fall-back if automatic detection is not supported
        if (!Font.AUTO_BIDI_SUPPORT) {
            return _.constant(LocaleData.DIR);
        }

        // the span element containing the text to be examined
        var spanNode = BIDI_NODE.firstChild;

        // the text node to be filled with the actual text
        var textNode = spanNode.firstChild;

        // the actual method getTextDirection() to be returned from local scope
        function getTextDirection(text) {

            // do not touch the DOM for empty strings
            if (text.length === 0) { return LocaleData.DIR; }

            // insert passed text; if the span element floats out of its parent to the left, is is right-to-left
            textNode.nodeValue = text;
            return (spanNode.offsetLeft < 0) ? 'rtl' : 'ltr';
        }

        return getTextDirection;
    }());

    // public methods ---------------------------------------------------------

    /**
     * Returns a clone of this font instance.
     *
     * @returns {Font}
     *  A clone of this font instance.
     */
    Font.prototype.clone = function () {
        return new Font(this.family, this.size, this.bold, this.italic);
    };

    /**
     * Returns a unique string key for this font that can be used for example
     * as key in an associative map.
     *
     * @returns {String}
     *  A unique string key for this font instance.
     */
    Font.prototype.key = function () {
        return this.family + '|' + this.size + '|' + ((this.bold ? 1 : 0) + (this.italic ? 2 : 0));
    };

    /**
     * Converts the font settings in this instance to a string suitable to be
     * set as font style at an HTML canvas element.
     *
     * @returns {String}
     *  The font settings of this instance, as HTML canvas font style.
     */
    Font.prototype.getCanvasFont = function () {
        return (this.bold ? 'bold ' : '') + (this.italic ? 'italic ' : '') + this.size + 'pt ' + this.family;
    };

    /**
     * Calculates the normal line height for this font, as reported by the
     * current browser. Note that this value may vary for the same font in
     * different browsers.
     *
     * @returns {Number}
     *  The normal line height for this font, in pixels (with a precision of
     *  1/10 pixel for browsers using fractional line heights internally).
     */
    Font.prototype.getNormalLineHeight = function () {
        return calculateFontMetrics(this, LINEHEIGHT_NODE, function (node) {
            return node.offsetHeight / 10;
        }, 'int.lineheight');
    };

    /**
     * Calculates the position of the font base line for the specified line
     * height, as reported by the current browser. Note that this value may
     * vary for the same font in different browsers.
     *
     * @param {Number} [lineHeight]
     *  The line height to be used to determine the position of the font base
     *  line. If omitted, the normal line height, as reported by the method
     *  Font.getNormalLineHeight(), will be used.
     *
     * @returns {Number}
     *  The position of the font base line for the specified line height, in
     *  pixels, relative to the top border of a text line.
     */
    Font.prototype.getBaseLineOffset = function (lineHeight) {

        // CSS line height with unit, or 'normal'; also used as cache key
        var cssLineHeight = _.isNumber(lineHeight) ? (Math.round(lineHeight) + 'px') : 'normal';

        return calculateFontMetrics(this, BASELINE_NODE, function (node) {
            node.style.lineHeight = cssLineHeight;
            return node.lastChild.offsetTop;
        }, 'int.baseline.' + cssLineHeight);
    };

    /**
     * Returns the exact width of the passed text, in pixels.
     *
     * @param {String} text
     *  The text whose width will be calculated. If the text consists of one
     *  character only, the calculated text width will be cached internally for
     *  subsequent invocations with the same font settings.
     *
     * @returns {Number}
     *  The exact width of the passed text, in pixels.
     */
    Font.prototype.getTextWidth = function (text) {

        switch (text.length) {
            case 0:
                // empty string: always zero
                return 0;

            case 1:
                // single character: calculate width of 10 characters for higher precision, store in cache
                return calculateTextWidth(this, function (context) {
                    return context.getCharacterWidth(text);
                }, 'int.textwidth.' + text);

            default:
                // multiple characters: calculate text width, do not store in cache
                return calculateTextWidth(this, function (context) {
                    return context.getTextWidth(text);
                });
        }
    };

    /**
     * Returns measures for the digits '0' to '9'.
     *
     * @returns {Array<Number>}
     *  An array with the widths of the digits '0' to '9', in pixels, inserted
     *  as array element at the appropriate index. The array will contain the
     *  following additional properties:
     *  - {Number} minWidth
     *      The minimum width of any of the digits, in pixels.
     *  - {Number} maxWidth
     *      The maximum width of any of the digits.
     *  - {Number} minDigit
     *      The digit with the smallest width; or 0, if all digits have the
     *      same width.
     *  - {Number} maxDigit
     *      The digit with the largest width; or 0, if all digits have the
     *      same width.
     */
    Font.prototype.getDigitWidths = function () {
        return calculateTextWidth(this, function (context) {
            var result = _.map('0123456789', function (digit) {
                return context.getCharacterWidth(digit);
            });
            result.minWidth = Math.min.apply(Math, result);
            result.maxWidth = Math.max.apply(Math, result);
            result.minDigit = result.indexOf(result.minWidth);
            result.maxDigit = result.indexOf(result.maxWidth);
            return result;
        }, 'int.digitwidth');
    };

    /**
     * Splits the passed string to text lines that all fit into the specified
     * pixel width.
     *
     * @param {String} text
     *  The text that will be split into single text lines.
     *
     * @param {Number} width
     *  The maximum width of the text lines, in pixels.
     *
     * @returns {Array<String>}
     *  An array of strings containing the single text lines, all fitting into
     *  the specified text line width.
     */
    Font.prototype.getTextLines = function (text, width) {

        // empty strings, or single-character strings cannot be split
        if (text.length <= 1) { return [text]; }

        // use a DOM helper node to take advantage of text auto-wrapping provided by the browser
        return calculateFontMetrics(this, TEXTLINES_NODE, function (node) {

            // the resulting substrings
            var textLines = [];

            // set the passed width at the helper node
            node.style.width = width + 'px';

            // create the HTML mark-up, wrap each character into its own span element
            // (for performance: with this solution, the DOM will be modified once,
            // and all node offsets can be read without altering the DOM again)
            node.innerHTML = Utils.escapeHTML(text).replace(/&(#\d+|\w+);|./g, '<span>$&</span>');

            // all DOM span elements
            var spanNodes = node.childNodes;
            // shortcut to the number of spans
            var spanCount = spanNodes.length;
            // start index of the current text line
            var lineIndex = 0;
            // vertical position of the spans in the current text line
            var lineOffset = spanNodes[0].offsetTop;
            // vertical position of the spans in the last text line
            var maxOffset = spanNodes[spanCount - 1].offsetTop;

            // returns the index of the first span element located in the next text line
            function findStartOfNextTextLine() {

                // index increment size size while searching forwards for next text lines
                var increment = 8;
                // start of the search interval (index in the current text line)
                var beginIndex = lineIndex;
                // end of the search interval (search until this index is in one of the following text lines)
                var endIndex = Math.min(spanCount, beginIndex + increment);

                // search for a span element located in any of the next text lines,
                // by picking a few following spans with increasing increment value
                while ((endIndex < spanCount) && (spanNodes[endIndex].offsetTop === lineOffset)) {
                    beginIndex = endIndex + 1;
                    increment *= 2;
                    endIndex = Math.min(spanCount, beginIndex + increment);
                }

                // binary search for the first span element in the resulting search interval
                return Utils.findFirstIndex(spanNodes, function (span) {
                    return lineOffset < span.offsetTop;
                }, { begin: beginIndex, end: endIndex, sorted: true });
            }

            // process all text lines but the last text line
            while (lineOffset < maxOffset) {

                // index of the first span element in the next text line
                var nextIndex = findStartOfNextTextLine();

                // append the text of the current text line to the result array
                textLines.push(text.substring(lineIndex, nextIndex));

                // continue with the next text line
                lineIndex = nextIndex;
                lineOffset = spanNodes[lineIndex].offsetTop;
            }

            // push the remaining last text line to the result array
            if (lineIndex < text.length) {
                textLines.push(text.substr(lineIndex));
            }

            return textLines;
        });
    };

    /**
     * Provides the ability to calculate and cache custom font metrics.
     *
     * @param {Function} callback
     *  The callback function that will be invoked after preparing the helper
     *  node.  Receives the following parameters:
     *  (1) {HTMLDivElement} node
     *      The helper element that can be used to calculate the result.
     *  Will be invoked with this font instance as calling context. The return
     *  value of this function will be cached if specified (see parameter
     *  'cacheKey' below), and will be returned by this method.
     *
     * @param {String} [cacheKey]
     *  If specified, a unique map key for the font metrics cache that will be
     *  checked first. An existing cache entry will be returned without
     *  invoking the callback function. Otherwise, the new value returned by
     *  the callback function will be inserted into the cache. If omitted, the
     *  callback function will always be invoked, and the result will not be
     *  inserted into the cache.
     *
     * @returns {Any}
     *  The font metric property from the cache, or returned by the callback.
     */
    Font.prototype.getCustomFontMetrics = function (callback, cacheKey) {
        cacheKey = _.isString(cacheKey) ? ('ext.' + cacheKey) : null;
        return calculateFontMetrics(this, CUSTOM_NODE, function (node) {
            var result = callback.call(this, node);
            node.innerHTML = '';
            node.removeAttribute('style');
            return result;
        }, cacheKey);
    };

    // exports ================================================================

    return Font;

});
