/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/tk/object/triggerobject', [
    'io.ox/core/event',
    'io.ox/office/tk/utils',
    'io.ox/office/tk/object/baseobject'
], function (Events, Utils, BaseObject) {

    'use strict';

    // class TriggerObject ====================================================

    /**
     * A generic base class for all classes that want to notify a collection of
     * listeners about some events.
     *
     * @constructor
     *
     * @extends BaseObject
     * @extends Events
     *
     * @param {String|Object} windowId
     *  The identifier of the root window of the context application owning the
     *  instances of this class, or an object with a method 'getWindowId' that
     *  returns such a window identifier. Used for debugging and logging of
     *  running timers and pending promises in automated test environments. The
     *  constant BaseObject.SINGLETON can be passed if an instance of this
     *  class will be used as global static singleton that is not bound to any
     *  context application.
     */
    var TriggerObject = BaseObject.extend({ constructor: function (windowId) {

        // base constructors --------------------------------------------------

        BaseObject.call(this, windowId);
        Events.extend(this);

        // initialization -----------------------------------------------------

        // destroy the core events mix-in
        this.registerDestructor(function () {
            this.events.destroy();
        });

    } }); // class TriggerObject

    // protected methods ------------------------------------------------------

    /**
     * Forwards all events triggered by the passed event source as events of
     * this instance.
     *
     * @param {Events|jQuery|HTMLElement} source
     *  The event source object. See method BaseObject.listenTo() for more
     *  details about supported object types.
     *
     * @param {Object} [options]
     *  Optional parameters:
     *  - {String} [options.except]
     *      A single event type name, or a space-separated list of event type
     *      names, that will NOT be forwarded to the listeners of this
     *      instance.
     *
     * @returns {TriggerObject}
     *  A reference to this instance.
     */
    TriggerObject.prototype.forwardEvents = function (source, options) {
        var exceptTypeSet = Utils.makeSet(Utils.getTokenListOption(options, 'except', []));
        this.listenTo(source, 'triggered', function (event, type) {
            // skip the specified events
            if (type in exceptTypeSet) { return; }
            // remove first argument (the jQuery event object)
            this.trigger.apply(this, _.toArray(arguments).slice(1));
        }.bind(this));
        return this;
    };

    // exports ================================================================

    return TriggerObject;

});
