/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/tk/container/valueset', [
    'io.ox/office/tk/container/container'
], function (Container) {

    'use strict';

    // private global functions ===============================================

    /**
     * Predefined keyer callback function for resolving instances of the class
     * BaseObject to their UIDs that will be used as keys in value sets. This
     * function exists to improve performance of the sets by saving the time to
     * dynamically resolve the method name via a string variable.
     */
    function getUid(obj) { return obj.getUid(); }

    // class ValueSet =========================================================

    /**
     * A generic set of unique values.
     *
     * @constructor
     *
     * @param {Function|String} keyer
     *  A callback function that will be invoked for each value that will be
     *  inserted into this set, and that must return a unique string key for
     *  that value; or the name of an object property that will be used as key
     *  (e.g. 'id' to use the property 'id' of the objects inserted into this
     *  set); or the name of a public method returning a string, that will be
     *  invoked on the object (with parentheses, e.g. 'getUid()' to invoke the
     *  method 'getUid' of the objects inserted into this set). The string keys
     *  for all values in this set MUST be constant (the callback function or
     *  object method MUST always return the same key, and the string property
     *  MUST not change).
     *
     * @param {Object} [source]
     *  A generic data source to copy elements from. See description of the
     *  static method Container.forEach() for details. If omitted, an empty set
     *  will be created.
     *
     * @param {Function} [generator]
     *  A generator callback function that will be used to deeply clone the
     *  elements of the passed data source. Receives the following parameters:
     *  (1) {Any} value
     *      The value of the data source element to be cloned.
     *  (2) {String} key
     *      The key of the data source element to be cloned.
     *  The return value of this function will be inserted as new element into
     *  this map. If omitted, a flat copy will be created (the elements of the
     *  data source will be copied by reference).
     *
     * @param {Object} [context]
     *  The calling context for the generator callback function.
     */
    var ValueSet = Container.extend({ constructor: function (keyer, source, generator, context) {

        // base constructor
        Container.call(this);

        // convert the passed keyer to a function
        this._key = (function () {
            if (typeof keyer === 'function') { return keyer; }
            if (keyer === 'getUid()') { return getUid; }
            if (keyer.slice(-2) === '()') {
                var method = keyer.slice(0, -2);
                return function (obj) { return obj[method](); };
            }
            return function (obj) { return obj[keyer]; };
        }());

        // copy the values of the passed source list
        if (source) {
            generator = generator || _.identity;
            Container.forEach(source, function (value, key) {
                this.insert(generator.call(context, value, key));
            }, this);
        }

    } }); // class Set

    // static methods ---------------------------------------------------------

    /**
     * Creates a new set with the values of the specified object property from
     * the passed data source as elements.
     *
     * @param {Function|String} keyer
     *  The keyer used to convert values to keys. See description of the class
     *  constructor for details.
     *
     * @param {Object} list
     *  A generic data source to copy elements from. See description of the
     *  static method Container.forEach() for details.
     *
     * @param {String} prop
     *  The name of the property to pluck from the elements of the data source.
     *  The plucked object properties become the values of the new map.
     *
     * @returns {ValueSet}
     *  The new map with the plucked element properties.
     */
    ValueSet.pluck = function (keyer, list, prop) {
        return new ValueSet(keyer, list, function (value) { return value[prop]; });
    };

    // public methods ---------------------------------------------------------

    /**
     * Creates a shallow or deep clone of this set.
     *
     * @param {Function} [generator]
     *  A generator callback function that can be used to deeply clone the
     *  elements of this set. Receives the following parameters:
     *  (1) {Any} value
     *      The value of the element to be cloned.
     *  (2) {String} key
     *      The key of the element to be cloned.
     *  The return value of this function will be inserted as new map element
     *  into the clone. If omitted, the elements will be copied by reference.
     *
     * @param {Object} [context]
     *  The calling context for the generator callback function.
     *
     * @returns {ValueSet}
     *  The clone of this set.
     */
    ValueSet.prototype.clone = function (generator, context) {
        return new ValueSet(this._key, this, generator, context);
    };

    /**
     * Returns the key that will be used by this set for the passed value.
     *
     * @param {Any} value
     *  The value to be converted to its key.
     *
     * @returns {String}
     *  The key for the specified value.
     */
    ValueSet.prototype.key = function (value) {
        return this._key(value);
    };

    /**
     * Returns whether this set contains an element with the specified key.
     *
     * @param {String} key
     *  The key to be checked.
     *
     * @returns {Boolean}
     *  Whether this set contains an element with the specified key.
     */
    ValueSet.prototype.hasKey = function (key) {
        return this._has(key);
    };

    /**
     * Returns whether this set contains the specified value.
     *
     * @param {Any} value
     *  The value to be checked.
     *
     * @returns {Boolean}
     *  Whether this set contains the specified value.
     */
    ValueSet.prototype.has = function (value) {
        return this._has(this._key(value));
    };

    /**
     * Inserts a value into this set.
     *
     * @param {Any} value
     *  The value to be inserted.
     *
     * @returns {Any}
     *  The value passed to this method, for convenience.
     */
    ValueSet.prototype.insert = function (value) {
        return this._insert(this._key(value), value);
    };

    /**
     * Removes a value from this set.
     *
     * @param {Any} value
     *  The value to be removed.
     *
     * @returns {Boolean}
     *  Whether the passed value was part of this set.
     */
    ValueSet.prototype.remove = function (value) {
        return this._remove(this._key(value));
    };

    /**
     * Inserts all elements of the passed list into this set.
     *
     * @param {Object} list
     *  A generic data source. See the description of the static method
     *  Container.forEach() for details.
     *
     * @returns {ValueSet}
     *  A reference to this instance.
     */
    ValueSet.prototype.merge = function (list) {
        Container.forEach(list, this.insert, this);
        return this;
    };

    // exports ================================================================

    return ValueSet;

});
