/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/tk/container/valuemap', [
    'io.ox/office/tk/container/container'
], function (Container) {

    'use strict';

    // convenience shortcuts
    var ContainerProto = Container.prototype;

    // class ValueMap =========================================================

    /**
     * A generic map of values associated to string keys.
     *
     * @constructor
     *
     * @param {Object} [source]
     *  A generic data source to copy elements from. See description of the
     *  static method Container.forEach() for details. If omitted, an empty map
     *  will be created.
     *
     * @param {Function} [generator]
     *  A generator callback function that will be used to deeply clone the
     *  elements of the passed data source. Receives the following parameters:
     *  (1) {Any} value
     *      The value of the data source element to be cloned.
     *  (2) {String} key
     *      The key of the data source element to be cloned.
     *  The return value of this function will be inserted as new element into
     *  this map. If omitted, a flat copy will be created (the elements of the
     *  data source will be copied by reference).
     *
     * @param {Object} [context]
     *  The calling context for the generator callback function.
     */
    var ValueMap = Container.extend({ constructor: function (source, generator, context) {

        // base constructor
        Container.call(this);

        // copy the values of the passed source list
        if (source) {
            generator = generator || _.identity;
            Container.forEach(source, function (value, key) {
                this.insert(key, generator.call(context, value, key));
            }, this);
        }

    } }); // class ValueMap

    // static methods ---------------------------------------------------------

    /**
     * Creates a new map with the values of the specified object property from
     * the passed data source as elements.
     *
     * @param {Object} list
     *  A generic data source to copy elements from. See description of the
     *  static method Container.forEach() for details.
     *
     * @param {String} prop
     *  The name of the property to pluck from the elements of the data source.
     *  The plucked object properties become the values of the new map.
     *
     * @returns {ValueMap}
     *  The new map with the plucked element properties.
     */
    ValueMap.pluck = function (list, prop) {
        return new ValueMap(list, function (value) { return value[prop]; });
    };

    /**
     * Creates a new map with the elements from the passed data source as map
     * elements, keyed by the specified object property.
     *
     * @param {Object} list
     *  A generic data source to copy elements from. See description of the
     *  static method Container.forEach() for details.
     *
     * @param {String} prop
     *  The name of the string property to pluck from the elements of the data
     *  source. The plucked string becomes the key of the map element.
     *
     * @returns {ValueMap}
     *  The new map with the plucked element properties.
     */
    ValueMap.rekey = function (list, prop) {
        var map = new ValueMap();
        Container.forEach(list, function (value) {
            map.insert(value[prop], value);
        });
        return map;
    };

    // public methods ---------------------------------------------------------

    /**
     * Creates a shallow or deep clone of this map.
     *
     * @param {Function} [generator]
     *  A generator callback function that can be used to deeply clone the
     *  elements of this map. Receives the following parameters:
     *  (1) {Any} value
     *      The value of the map element to be cloned.
     *  (2) {String} key
     *      The key of the map element to be cloned.
     *  The return value of this function will be inserted as new map element
     *  into the clone. If omitted, the elements will be copied by reference.
     *
     * @param {Object} [context]
     *  The calling context for the generator callback function.
     *
     * @returns {ValueMap}
     *  The clone of this map.
     */
    ValueMap.prototype.clone = function (generator, context) {
        return new ValueMap(this, generator, context);
    };

    /**
     * Returns whether this map contains an element with the specified key.
     *
     * @param {String} key
     *  The key to be checked.
     *
     * @returns {Boolean}
     *  Whether this map contains an element with the specified key.
     */
    ValueMap.prototype.has = ContainerProto._has;

    /**
     * Inserts an element into this map. An old existing map element will be
     * overwritten.
     *
     * @param {String} key
     *  The key for the map element to be inserted.
     *
     * @param {Any} value
     *  The value to be inserted at the specified key.
     *
     * @returns {Any}
     *  The value passed to this method, for convenience.
     */
    ValueMap.prototype.insert = ContainerProto._insert;

    /**
     * Removes an element from this map.
     *
     * @param {String} key
     *  The key of the map element to be removed.
     *
     * @param {Any} [def]
     *  The default value that will be returned if this map does not contain an
     *  element with the specified key. If omitted, undefined will be returned.
     *
     * @returns {Any}
     *  The value of the map element that has been removed; or the specified
     *  default value, if the map does not contain such an element.
     */
    ValueMap.prototype.remove = function (key, def) {
        var value = this._get(key);
        return this._remove(key) ? value : def;
    };

    /**
     * Replaces an element in this map with a new value, or inserts a new
     * element.
     *
     * @param {String} key
     *  The key of the map element to be updated.
     *
     * @param {Any} def
     *  The value passed to the callback function for a missing map element.
     *
     * @param {Function} callback
     *  The callback function that will be invoked with the current value of
     *  the map element (or the specified default value). The return value of
     *  this function will become the new value of the map element.
     *
     * @param {Object} [context]
     *  The calling context for the callback function.
     *
     * @returns {Any}
     *  The new value of the map element.
     */
    ValueMap.prototype.update = function (key, def, callback, context) {
        return this.insert(key, callback.call(context, this.get(key, def)));
    };

    /**
     * Moves an element in this map to a new key.
     *
     * @param {String} oldKey
     *  The key of the map element to be moved.
     *
     * @param {String} newKey
     *  The new key of the map element.
     *
     * @returns {ValueMap}
     *  A reference to this instance.
     */
    ValueMap.prototype.move = function (oldKey, newKey) {
        if (this.has(oldKey)) { this.insert(newKey, this.remove(oldKey)); }
        return this;
    };

    /**
     * Inserts all elements of the passed list into this map.
     *
     * @param {Object} list
     *  A generic data source. See the description of the static method
     *  Container.forEach() for details. Existing elements in this map will be
     *  overwritten by the elements of the passed list.
     *
     * @returns {ValueMap}
     *  A reference to this instance.
     */
    ValueMap.prototype.merge = function (source) {
        Container.forEachKey(source, this.insert, this);
        return this;
    };

    /**
     * Returns the map element with the specified key. If such an element does
     * not exist, the generator callback function will be invoked to create a
     * new map element.
     *
     * @param {String} key
     *  The key of the map element to be returned.
     *
     * @param {Function} [generator]
     *  A generator callback function for missing map elements. Receives the
     *  following parameters:
     *  (1) {String} key
     *      The key of the map element to be generated.
     *  The return value of this function will be inserted as new map element.
     *
     * @param {Object} [context]
     *  The calling context for the generator callback function.
     *
     * @returns {Any}
     *  The map element for the specified key.
     */
    ValueMap.prototype.getOrCreate = function (key, generator, context) {
        return this._has(key) ? this._get(key) : this.insert(key, generator.call(context, key));
    };

    /**
     * Returns the map element with the specified key. If such an element does
     * not exist, a new element will be created with the passed class
     * constructor.
     *
     * @param {String} key
     *  The key of the map element to be returned.
     *
     * @param {Function} Ctor
     *  A class constructor function for missing map elements. Receives all
     *  following parameters passed to this method.
     *
     * @param {Any} [...]
     *  All following parameters will be passed to the class constructor.
     *
     * @returns {Any}
     *  The map element for the specified key.
     */
    ValueMap.prototype.getOrConstruct = function (key, Ctor) {
        return this._has(key) ? this._get(key) : this.insert(key, (function (args) {

            // performance shortcut for few number of arguments
            switch (args.length) {
                case 2: return new Ctor();
                case 3: return new Ctor(args[2]);
                case 4: return new Ctor(args[2], args[3]);
                case 5: return new Ctor(args[2], args[3], args[4]);
            }

            // construct a new instance with arbitrary number of arguments on the fly
            return (function () {
                function Helper() { return Ctor.apply(this, _.toArray(args).slice(2)); }
                Helper.prototype = Ctor.prototype;
                return new Helper();
            }());
        }(arguments)));
    };

    /**
     * Returns a new map where the values of this map are the keys, and the
     * keys of this map are the values.
     *
     * @returns {ValueMap}
     *  A new map with swapped values and keys.
     */
    ValueMap.prototype.invert = function () {
        var map = new ValueMap();
        this.forEach(map.insert, map);
        return map;
    };

    // exports ================================================================

    return ValueMap;

});
