/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/tk/container/sortedarray', [
    'io.ox/office/tk/utils',
    'io.ox/office/tk/utils/iteratorutils'
], function (Utils, IteratorUtils) {

    'use strict';

    // class ElementDescriptor ================================================

    /**
     * A descriptor object for an element in a sorted array.
     *
     * @constructor
     *
     * @property {SortedArray} array
     *  A reference to the sorted array that contains the array element.
     *
     * @property {Any} value
     *  The array element value represented by this descriptor.
     *
     * @property {Any} index
     *  The sort index of the array element.
     *
     * @property {Number} _ai
     *  The internal array index of the element represented by this descriptor.
     *  Can be used for optimized implementations using the internal array as
     *  returned by the method SortedArray.values().
     */
    function ElementDescriptor(array, value, index, ai) {

        this.array = array;
        this.value = value;
        this.index = index;

        // protected properties
        this._ai = ai;

    } // class ElementDescriptor

    // public methods ---------------------------------------------------------

    /**
     * Removes the element represented by this descriptor from the sorted
     * array.
     */
    ElementDescriptor.prototype.remove = function () {
        this.array._removeElem(this._ai);
    };

    // class SortedArray ======================================================

    /**
     * A generic sorted sparse array of arbitrary values that will be sorted by
     * separate sort index indicators.
     *
     * @constructor
     *
     * @param {Function} sorter
     *  A callback function that must implement to convert the values intended
     *  to be used as element of this array to their sort index. Receives a
     *  single value parameter, and MUST return a value that is comparable with
     *  the built-in JavaScript less-than-or-equals operator (e.g. numbers,
     *  strings, or date objects). The function _.identity can be passed to
     *  sort the array elements by themselves.
     */
    var SortedArray = _.makeExtendable(function (sorter) {

        this._array = [];
        this._sorter = sorter;

    }); // class SortedArray

    // private methods --------------------------------------------------------

    SortedArray.prototype._insertElem = function (ai, value) {
        this._array.splice(ai, 0, value);
    };

    SortedArray.prototype._replaceElem = function (ai, value) {
        this._array[ai] = value;
    };

    SortedArray.prototype._appendElem = function (value) {
        this._array.push(value);
    };

    SortedArray.prototype._removeElem = function (ai) {
        this._array.splice(ai, 1);
    };

    // public methods ---------------------------------------------------------

    /**
     * Returns the number of elements in this array.
     *
     * @returns {Number}
     *  The number of elements in this array.
     */
    SortedArray.prototype.length = function () {
        return this._array.length;
    };

    /**
     * Returns whether this array is empty.
     *
     * @returns {Boolean}
     *  Whether this array is empty.
     */
    SortedArray.prototype.empty = function () {
        return this._array.length === 0;
    };

    /**
     * Removes all elements from this array.
     *
     * @returns {SortedArray}
     *  A reference to this instance.
     */
    SortedArray.prototype.clear = function () {
        this._array = [];
        return this;
    };

    /**
     * Returns a shallow or deep clone of this sorted array.
     *
     * @param {Boolean} [deep=false]
     *  If set to true, a deep clone of the array will be created. All array
     *  elements MUST be objects with a method 'clone()'. The value true will
     *  be passed as first parameter, in order to recursively clone the array
     *  elements deeply. If set to false or omitted, a shallow clone of this
     *  array will be created (the elements of this array will be copied by
     *  reference to the clone).
     *
     * @returns {SortedArray}
     *  A clone of this sorted array.
     */
    SortedArray.prototype.clone = function (deep) {
        var clone = new SortedArray(this._sorter);
        clone._array = deep ? _.invoke(this._array, 'clone', true) : this._array.slice();
        return clone;
    };

    /**
     * Returns the elements of this sorted array instance as plain JavaScript
     * array object.
     *
     * @returns {Array<Any>}
     *  The elements of this sorted array instance as plain JavaScript array
     *  object. This array MUST NOT be changed.
     */
    SortedArray.prototype.values = function () {
        return this._array;
    };

    /**
     * Returns the first array element.
     *
     * @returns {Any|Null}
     *  The first array element; or null if the array is empty.
     */
    SortedArray.prototype.first = function () {
        return this.empty() ? null : this._array[0];
    };

    /**
     * Returns the last array element.
     *
     * @returns {Any|Null}
     *  The last array element; or null if the array is empty.
     */
    SortedArray.prototype.last = function () {
        return this.empty() ? null : _.last(this._array);
    };

    /**
     * Finds an array element by its sort index using a fast binary search, and
     * returns datailed information about the array element if available.
     *
     * @param {Any} index
     *  The sort index of the array element (as returned by the 'sorter'
     *  callback function passed to the constructor).
     *
     * @param {String} [mode='next']
     *  Specifies how to search for an array element. The following modes are
     *  supported:
     *  - 'next' (default):
     *      Searches for an array element with the passed sort index, or the
     *      nearest following array element if there is none with the exact
     *      sort index.
     *  - 'prev':
     *      Searches for an array element with the passed sort index, or the
     *      nearest preceding array element if there is none with the exact
     *      sort index.
     *  - 'exact':
     *      Searches for an array element with the passed sort index only.
     *
     * @returns {ElementDescriptor|Null}
     *  A result descriptor, if an array element has been found for the passed
     *  sort index; otherwise null.
     */
    SortedArray.prototype.find = function (index, mode) {

        // nothing to do in empty arrays
        if (this.empty()) { return null; }

        // whether to find a preceding array element
        var reverse = mode === 'prev';

        // create the appropriate predicate function for binary search (forward or backward)
        var sorter = this._sorter;
        var pred = reverse ?
            function (value) { return sorter(value) <= index; } :
            function (value) { return index <= sorter(value); };

        // fast O(1) limit check before binary search
        if (reverse ? !pred(this.first()) : !pred(this.last())) { return null; }

        // find the next or preceding array element with a binary search
        var findFunc = reverse ? Utils.findLastIndex : Utils.findFirstIndex;
        var ai = findFunc(this._array, pred, { sorted: true });
        if (ai < 0) { return null; }

        // create the array element descriptor, if a matching array element has been found
        var value = this._array[ai], vi = sorter(value);
        return ((index === vi) || (mode !== 'exact')) ? new ElementDescriptor(this, value, vi, ai) : null;
    };

    /**
     * Returns whether this array contains an element with the specified sort
     * index.
     *
     * @param {Number} index
     *  The sort index associated with the array element (as returned by the
     *  'sorter' callback function passed to the constructor).
     *
     * @returns {Boolean}
     *  Whether this array contains an element with the specified sort index.
     */
    SortedArray.prototype.has = function (index) {
        return !!this.find(index, 'exact');
    };

    /**
     * Returns the array element with the specified sort index.
     *
     * @param {Number} index
     *  The sort index associated with the array element (as returned by the
     *  'sorter' callback function passed to the constructor).
     *
     * @returns {Any|Null}
     *  The array element if existing; otherwise null.
     */
    SortedArray.prototype.get = function (index) {
        var desc = this.find(index, 'exact');
        return desc ? desc.value : null;
    };

    /**
     * Returns the array element at the specified sort index, or creates and
     * inserts a new array element.
     *
     * @param {Number} index
     *  The sort index associated with the array element (as returned by the
     *  'sorter' callback function passed to the constructor).
     *
     * @param {Function} generator
     *  A generator function that will be invoked to create a new array element
     *  for the specified sort index.
     *
     * @param {Object} [context]
     *  The calling context for the generator callback function.
     *
     * @returns {Any}
     *  The array element for the passed sort index.
     */
    SortedArray.prototype.getOrCreate = function (index, generator, context) {
        var desc = this.find(index);
        if (desc && (index === desc.index)) { return desc.value; }
        var value = generator.call(context, index);
        if (desc) { this._insertElem(desc._ai, value); } else { this._appendElem(value); }
        return value;
    };

    /**
     * Returns all elements contained in the passed sort index interval.
     *
     * @param {Any} first
     *  The sort index of the first array element to be returned.
     *
     * @param {Any} last
     *  The sort index of the last array element to be returned (closed
     *  interval).
     *
     * @returns {Array<Any>}
     *  An array with all elements contained in the specified index interval.
     */
    SortedArray.prototype.slice = function (first, last) {
        var desc1 = this.find(first, 'next');
        var desc2 = this.find(last, 'prev');
        return (desc1 && desc2 && (desc1._ai <= desc2._ai)) ? this._array.slice(desc1._ai, desc2._ai + 1) : [];
    };

    /**
     * Invokes the passed callback for each element in this array.
     *
     * @param {Function} callback
     *  The callback function to be invoked for each array element. Receives
     *  the current element as first parameter, and the sort index of the array
     *  element as second parameter.
     *
     * @param {Object} [context]
     *  The calling context for the callback function.
     *
     * @returns {SortedArray}
     *  A reference to this instance.
     */
    SortedArray.prototype.forEach = function (callback, context) {
        var sorter = this._sorter;
        this._array.forEach(function (value) {
            callback.call(context, value, sorter(value));
        });
        return this;
    };

    /**
     * Creates an iterator for all array elements.
     *
     * @param {Boolean} [reverse=false]
     *  Whether to visit the array elements in reversed order.
     *
     * @returns {Object}
     *  An iterator object that implements the standard EcmaScript iterator
     *  protocol, i.e. it provides the method next() that returns a result
     *  object with the following properties:
     *  - {Boolean} done
     *      If set to true, the elements have been visited completely. No more
     *      array elements are available; this result object does not contain
     *      any other properties!
     *  - {Any} value
     *      The array element currently visited.
     *  - {Any} index
     *      The sort index of the array element (as returned by the 'sorter'
     *      callback function passed to the constructor) currently visited by
     *      the iterator (NOT the internal array index).
     */
    SortedArray.prototype.iterator = function (reverse) {

        // create an array iterator that visits all elements in this array, and provides the sort index instead of array index
        var iterator = IteratorUtils.createArrayIterator(this._array, reverse ? { reverse: true } : null);
        var sorter = this._sorter;
        return IteratorUtils.createTransformIterator(iterator, function (value, result) {
            result.index = sorter(value);
            return result;
        }, this);
    };

    /**
     * Creates an element iterator for the passed index interval.
     *
     * @param {Any} first
     *  The sort index of the first array element to be visited by the returned
     *  iterator.
     *
     * @param {Any} last
     *  The sort index of the last array element to be visited by the returned
     *  iterator (closed interval).
     *
     * @param {Boolean} [reverse=false]
     *  Whether to visit the array elements in reversed order. The element with
     *  the sort index passed to the parameter 'last' will be visited first.
     *
     * @returns {Object}
     *  An iterator object that implements the standard EcmaScript iterator
     *  protocol, i.e. it provides the method next() that returns a result
     *  object with the following properties:
     *  - {Boolean} done
     *      If set to true, the interval has been visited completely. No more
     *      array elements are available; this result object does not contain
     *      any other properties!
     *  - {Any} value
     *      The array element currently visited.
     *  - {Any} index
     *      The sort index of the array element (as returned by the 'sorter'
     *      callback function passed to the constructor) currently visited by
     *      the iterator (NOT the internal array index).
     */
    SortedArray.prototype.intervalIterator = function (first, last, reverse) {

        // find the first array element to be visited (fast binary search)
        var desc = this.find(reverse ? last : first, reverse ? 'prev' : 'next');
        if (!desc || (reverse ? (desc.index < first) : (last < desc.index))) {
            return IteratorUtils.ALWAYS_DONE_ITERATOR;
        }

        // create an array iterator that visits all elements in this array until the iterator leaves the passed interval
        var iterator = IteratorUtils.createArrayIterator(this._array, { reverse: reverse, begin: desc._ai });
        var sorter = this._sorter;
        return IteratorUtils.createTransformIterator(iterator, function (value) {
            var index = sorter(value);
            var done = reverse ? (index < first) : (last < index);
            return done ? { done: true } : { value: value, index: index };
        });
    };

    /**
     * Returns the sort index associated to the passed value.
     *
     * @param {Any} value
     *  The value whose sort index will be returned. The value does not need to
     *  exist in this array.
     *
     * @returns {Any}
     *  The sort index associated to the passed value (the return value of the
     *  'sorter' callback function passed to the constructor).
     */
    SortedArray.prototype.index = function (value) {
        return this._sorter(value);
    };

    /**
     * Inserts a new value into this array according to its sort index, or
     * replaces an existing array element with the same sort index.
     *
     * @param {Any} value
     *  The value to be inserted into this array. MUST NOT be null or
     *  undefined.
     *
     * @param {Boolean} [replace=false]
     *  If set to true, an existing array element with the same sort index as
     *  the passed value will be replaced with the new value (e.g. useful for
     *  different objects with the same sort index). By default, the existing
     *  array element will be kept, and the passed value will be ignored.
     *
     * @returns {Any}
     *  The value of the inserted array element. This may not be the same as
     *  the passed value, e.g. if an array element with the sort index of the
     *  passed value exists already in this array, and the flag 'replace' has
     *  not been set.
     */
    SortedArray.prototype.insert = function (value, replace) {
        var index = this._sorter(value);
        var desc = this.find(index);
        if (!desc) {
            this._appendElem(value);
        } else if (desc.index !== index) {
            this._insertElem(desc._ai, value);
        } else if (replace) {
            this._replaceElem(desc._ai, value);
        } else {
            value = desc.value;
        }
        return value;
    };

    /**
     * Removes the specified value from this array. The value fill be searched
     * by its associated sort index.
     *
     * @param {Any} value
     *  The value to be removed from this array. May not exist in the array.
     *  MUST NOT be null or undefined.
     *
     * @returns {Any|Null}
     *  The value of the removed array element; or null, if this array does not
     *  contain an element with the sort index of the passed value.
     */
    SortedArray.prototype.remove = function (value) {
        var desc = this.find(this._sorter(value), 'exact');
        if (!desc) { return null; }
        desc.remove();
        return desc.value;
    };

    // exports ================================================================

    return SortedArray;

});
