/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Michael Nimz <michael.nimz@open-xchange.com>
 */

define('io.ox/office/textframework/view/popup/universalcontextmenu', [
    'io.ox/office/tk/utils',
    'io.ox/office/editframework/utils/attributeutils',
    'io.ox/office/textframework/utils/dom',
    'io.ox/office/textframework/utils/position',
    'io.ox/office/textframework/view/controls',
    'io.ox/office/textframework/view/labels',
    'io.ox/office/baseframework/view/popup/contextmenu',
    'io.ox/office/editframework/view/control/languagepicker',
    'io.ox/office/drawinglayer/view/drawingframe',
    'gettext!io.ox/office/textframework/main'
], function (Utils, AttributeUtils, Dom, Position, Controls, Labels, ContextMenu, LanguagePicker, DrawingFrame, gt) {

    'use strict';

    // convenience shortcuts
    var Button = Controls.Button;
    var CompoundButton = Controls.CompoundButton;

    // class UniversalContextMenu ================================================

    /**
     * A context menu for normal text. Provides menu actions to manipulate
     * entire text.
     *
     * @constructor
     *
     * @extends ContextMenu
     */
    function UniversalContextMenu(docView, initOptions) {

        var // self reference
            self = this,

            // document model
            docModel = docView.getDocModel(),
            editDiv = docModel.getNode(),

            // selection
            selection = docModel.getSelection(),
            // spell checker
            spellChecker = docModel.getSpellChecker(),

            // field manager instance
            fieldManager = docModel.getFieldManager(),

            // the clicked node
            node = null,
            // the type of the clicked node (drawing, table, ...)
            eleType = null,

            // is a range is selected or not
            isRangeSelected = false,
            isDrawingSelected = false,
            // xox position of the cursor
            resetSelectionTo = null,

            // needs to use special behavior for osx chrome
            chromeOnOSX = (_.browser.MacOS && _.browser.Chrome),

            enableKeys = ['document/editable'];

        // base constructor ---------------------------------------------------

        ContextMenu.call(this, docView, editDiv, Utils.extendOptions({
            delay: Utils.getNumberOption(initOptions, 'delay', 0)
        }, initOptions));

        // private methods ----------------------------------------------------

        /**
         * A handler that should be invoked on contextmenu event (see in global 'contextmenu.js')
         */
        function contextMenuEventHander(event) {
            if (chromeOnOSX) {
                var oxoPosition = Position.getOxoPositionFromPixelPosition(editDiv, event.pageX, event.pageY),
                    selectionStartPos = selection.getStartPosition(),
                    selectionEndPos = selection.getEndPosition();

                if (!_.isNull(oxoPosition)) {
                    // click into table
                    if (Position.isPositionInTable(editDiv, oxoPosition.start)) {

                        var reset = false;

                        if (Dom.isParagraphNode(event.target) && Dom.isEmptyParagraph(event.target)) { reset = true; }
                        if (Dom.isSpan(event.target) && Dom.isEmptySpan(event.target)) { reset = true; }
                        if (Dom.isCellNode(event.target)) { reset = true; oxoPosition.start.push(0); }

                        if (reset) {
                            resetSelectionTo = oxoPosition.start;
                            selection.setTextSelection(resetSelectionTo);
                            return;
                        }
                    }

                    // click into another paragraph
                    if (_.first(oxoPosition.start) !== _.first(selectionStartPos) && _.first(oxoPosition.start) !== _.first(selectionEndPos)) {
                        resetSelectionTo = oxoPosition.start;
                        selection.setTextSelection(resetSelectionTo);

                        // click into the same paragraph
                    } else {
                        // don't click on the selection
                        if (!Position.isNodePositionInsideRange(oxoPosition.start, _.last(selection.getStartPosition()), _.last(selection.getEndPosition()))) {
                            resetSelectionTo = oxoPosition.start;
                            selection.setTextSelection(resetSelectionTo);
                        }
                    }
                }
            }
        }

        /**
         * A handler that should be invoked on detect (see in global 'contextmenu.js')
         */
        function detectHandler() {
            isRangeSelected = selection.hasRange();
        }

        /**
         * A handler that should be invoked on reset (see in global 'contextmenu.js')
         * The following options are supported:
         * @param {Object} [options]
         *  @param {Boolean} [options.resetTextSelection=false]
         *      Resets the text selection, if the resetSelectionTo variable
         *      isn't null
         */
        function resetHandler(options) {
            if (Utils.getBooleanOption(options, 'resetTextSelection', false) && !_.isNull(resetSelectionTo)) {
                selection.setTextSelection(resetSelectionTo);
            }
            if (!_.isNull(resetSelectionTo)) {
                resetSelectionTo = null;
            }
        }

        function initMenu() {
            var controller = docView.getApp().getController();

            // remove all existing controls from context menu
            self.destroyAllGroups();
            self.setToolTip('');

            if (enableKeys.every(controller.isItemEnabled, controller)) {

                // Is Drawing
                if (Dom.isDrawingFrame(eleType)) {
                    if (!Dom.isInCommentNode(node)) {
                        prepareDrawingMenu();
                        self.updateAllGroups(); //Bug 47352
                        return;
                    }

                    // Is Table
                } else if (Dom.isTableNode(eleType) || Dom.isCellNode(eleType)) {
                    if (Dom.isCellNode(eleType)) {
                        if (Dom.isSpan(node) && !Dom.isEmptyCell(Dom.getCellContentNode($(node).parent()))) {
                            prepareTextMenu();
                            self.updateAllGroups(); //Bug 47352
                            return;
                        }
                    }
                    prepareTableMenu();
                    self.updateAllGroups(); //Bug 47352
                    return;

                    // Is Text
                } else {
                    // Is TextFrame
                    if (selection.isAdditionalTextframeSelection() && !Dom.isTextSpan(node)) {
                        prepareDrawingMenu();
                        self.updateAllGroups(); //Bug 47352
                        return;
                    } else if (selection.isSlideSelection()) {
                        self.prepareSlideMenu();
                        self.updateAllGroups(); //Bug 47352
                        return;
                    }
                    prepareTextMenu();
                    self.updateAllGroups(); //Bug 47352
                    return;
                }
            } else {
                prepareHyperlinkMenu();
                self.updateAllGroups(); //Bug 47352
            }
        }
        // preparation of context menus ---------------------------------------
        /**
         * Prepares the context menu for text
         */
        function prepareTextMenu() {

            var characterAttrs = AttributeUtils.getExplicitAttributes(node, { family: 'character', direct: true });
            var effectiveUrl = isRangeSelected ? null : characterAttrs.url;

            if (Dom.isChangeTrackNode(node) && !isRangeSelected) {
                self.addGroup('acceptSelectedChangeTracking', new Button(docView, Labels.ACCEPT_CURRENT_OPTIONS))
                    .addGroup('rejectSelectedChangeTracking', new Button(docView, Labels.REJECT_CURRENT_OPTIONS))
                    .addGroup('acceptChangeTracking', new Button(docView, Labels.ACCEPT_ALL_OPTIONS))
                    .addGroup('rejectChangeTracking', new Button(docView, Labels.REJECT_ALL_OPTIONS));

            } else if (fieldManager.isHighlightState() && !docModel.useSlideMode()) {
                if (fieldManager.isSupportedFieldType()) {
                    if (characterAttrs.anchor) {
                        self.addGroup('goToAnchor', new Button(docView, Labels.GO_TO_ANCHOR_LABEL));
                    }
                    if (fieldManager.supportsFieldFormatting()) {
                        self.addGroup('editField', new Button(docView, Labels.EDIT_FIELD_OPTIONS));
                    }
                    if (fieldManager.isTableOfContentsHighlighted()) {
                        self.addGroup('document/inserttoc', new Controls.TocPicker(docView, { autoCloseParent: false, autoHideGroups: true, updateCaptionMode: 'none', label: gt('Table of contents layouts'), tooltip: gt('Change current layout'), icon: 'fa-chevron-right', iconPos: 'trailing', caret: false, anchorBorder: 'right left' }));
                    }
                    self.addGroup('updateField', new Button(docView, Labels.UPDATE_FIELD_OPTIONS))
                    .addSeparator();
                    self.addGroup('updateAllFields', new Button(docView, Labels.UPDATE_ALL_FIELDS_OPTIONS));
                }
            } else {
                // Spellcheck
                if (characterAttrs.spellerror && !isRangeSelected && spellChecker.isOnlineSpelling()) {

                    var spellResult = spellChecker.getSpellErrorWord();

                    spellChecker.selectMisspelledWord(self);

                    if (spellResult.replacements && spellResult.replacements.length > 0) {
                        _.each(spellResult.replacements, function (replace) {
                            self.addGroup('document/spelling/replace', new Button(docView, { label: replace, value: replace }));
                        });
                    }

                    if (spellResult.word && spellChecker.hasSpellerrorClass(spellResult.node)) {
                        self.addSeparator()
                            .addGroup('document/spelling/ignoreword', new Button(docView, { label: /*#. Context menu entry to ignore a word from spellchecking. */ gt('Ignore all'), value: { word: spellResult.word, start: spellResult.start, end: spellResult.end, nodes: spellResult.nodes, target: docModel.getActiveTarget() } }))
                            .addGroup('document/spelling/userdictionary', new Button(docView, { label: /*#. Context menu entry to add a word into the user dictionary. */ gt('Add to dictionary'), value: { word: spellResult.word, start: spellResult.start, end: spellResult.end, nodes: spellResult.nodes, target: docModel.getActiveTarget(), userDictionary: true } }));
                    }

                    addLanguageGroupToMenu();

                    // Hyperlink
                } else if (effectiveUrl) {
                    self.addGroup('character/hyperlink/dialog', new Button(docView, { label: Labels.EDIT_HYPERLINK_LABEL }))
                        .addGroup('character/hyperlink/remove', new Button(docView, { label: Labels.REMOVE_HYPERLINK_LABEL }))
                        .addGroup('character/hyperlink/valid', new Button(docView, { label: Labels.OPEN_HYPERLINK_LABEL, href: characterAttrs.url }));

                    self.setToolTip(effectiveUrl);

                    // Normal Text
                } else {
                    buildNormalTextMenu();
                    if (!Utils.SMALL_DEVICE && !docModel.useSlideMode()) {
                        self.addGroup(null, new Controls.InTextStyleContextSubMenu(docView, { autoCloseParent: false, autoHideGroups: true, anchorBorder: 'right left', icon: 'fa-chevron-right', iconPos: 'trailing', caret: false }));
                    }
                    addLanguageGroupToMenu();
                }
            }

            self.refreshImmediately();
        }

        function prepareHyperlinkMenu() {

            var characterAttrs = AttributeUtils.getExplicitAttributes(node, { family: 'character', direct: true });
            var effectiveUrl = isRangeSelected ? null : characterAttrs.url;

            if (effectiveUrl) {
                self.addGroup('character/hyperlink/valid', new Button(docView, { label: Labels.OPEN_HYPERLINK_LABEL, href: characterAttrs.url }));
                self.setToolTip(effectiveUrl);
            }

            self.refreshImmediately();
        }

        /**
         * Build compound control group and fill it with groups for normal text.
         */
        function buildNormalTextMenu() {

            var normalTextBtn = new CompoundButton(docView, { autoCloseParent: false, autoHideGroups: true, label: Labels.INSERT_HEADER_LABEL, anchorBorder: 'right left', icon: 'fa-chevron-right', iconPos: 'trailing', caret: false });

            normalTextBtn
                .addGroup('image/insert/dialog',        new Button(docView, Labels.INSERT_IMAGE_OPTIONS))
                .addGroup('textframe/insert',           new Controls.InsertTextFrameButton(docView, { icon: null }))
                .addGroup('comment/insert',             new Button(docView, Labels.INSERT_COMMENT_OPTIONS))
                .addGroup('character/hyperlink/dialog', new Button(docView, Labels.INSERT_HYPERLINK_OPTIONS));

            if (docModel.useSlideMode()) {
                normalTextBtn
                    .addGroup('insertSlideNumber', new Button(docView, Labels.INSERT_SLIDE_NUMBER_OPTIONS))
                    .addGroup('insertDateTime',    new Button(docView, Labels.INSERT_DATE_TIME_OPTIONS));
            }

            self.addGroup(null, normalTextBtn);
        }

        /**
         * Add the Language Group to the context menu.
         */
        function addLanguageGroupToMenu() {
            self.addSeparator()
                .addGroup('character/language', new LanguagePicker(docView, { autoCloseParent: false, autoHideGroups: true, updateCaptionMode: 'none', label: gt('Language'), icon: 'fa-chevron-right', iconPos: 'trailing', caret: false, showListIcons: false, anchorBorder: 'right left', showAllLanguages: false }));
        }

        /**
         * Prepares the context menu for drawings
         */
        function prepareDrawingMenu() {
            self.addGroup('drawing/delete', new Button(docView, { label: Labels.DELETE_DRAWING_LABEL, tooltip: Labels.DELETE_DRAWING_TOOLTIP }))
                .addSeparator()
                .addGroup('drawing/group', new Button(docView, { label: Labels.GROUP_DRAWING_LABEL, tooltip: Labels.GROUP_DRAWING_TOOLTIP }))
                .addGroup('drawing/ungroup', new Button(docView, { label: Labels.UNGROUP_DRAWING_LABEL, tooltip: Labels.UNGROUP_DRAWING_TOOLTIP }))
                .addSeparator()
                .addGroup('drawing/order', new Controls.DrawingOrderPicker(docView, { caret: false, icon: 'fa-chevron-right', iconPos: 'trailing', anchorBorder: 'right left' }));
        }

        /**
         * Prepares the context menu for tables
         */
        function prepareTableMenu() {
            self.addGroup('table/insert/row',    new Button(docView, { label: gt('Insert row') }))
                .addGroup('table/delete/row',    new Button(docView, { label: gt('Delete row') }))
                .addGroup('table/insert/column', new Button(docView, { label: gt('Insert column') }))
                .addGroup('table/delete/column', new Button(docView, { label: gt('Delete column') }))
                .addGroup('table/delete',        new Button(docView, { label: gt('Delete table') }))
                .addSeparator()
                .addGroup(null, new CompoundButton(docView, { autoCloseParent: false, autoHideGroups: true, label: Labels.INSERT_HEADER_LABEL, anchorBorder: 'right left', icon: 'fa-chevron-right', iconPos: 'trailing', caret: false })
                    .addGroup('image/insert/dialog',        new Button(docView, Labels.INSERT_IMAGE_OPTIONS))
                    .addGroup('textframe/insert',           new Controls.InsertTextFrameButton(docView, { icon: null }))
                    .addGroup('comment/insert',             new Button(docView, Labels.INSERT_COMMENT_OPTIONS))
                    .addGroup('character/hyperlink/dialog', new Button(docView, Labels.INSERT_HYPERLINK_OPTIONS))
                );
        }

        /**
         * Prepares the context menu.
         * Checks if a table, drawing or something else was clicked
         * and initialize the correct menu
         */
        function contextMenuPrepareHandler(event) {
            var sourceEvent     = event.sourceEvent;
            var sourceTarget    = (sourceEvent) ? sourceEvent.target : null;
            var controller      = docView.getApp().getController();

            selection           = docModel.getSelection();
            isDrawingSelected   = docModel.isDrawingSelected();
            isRangeSelected     = !isDrawingSelected && selection.hasRange();

            if (!sourceTarget && !isDrawingSelected) { return; }

            if (isDrawingSelected) {
                node = (selection.isMultiSelectionSupported() && selection.isMultiSelection()) ? selection.getFirstSelectedDrawingNode()[0] : selection.getSelectedDrawing()[0];
            } else {
                node = (_.browser.IE && $(sourceTarget).is('.page')) ? $(window.getSelection().focusNode.parentNode) : sourceTarget;
            }

            var arrDomTree = Dom.getDomTree(node, {
                endNodes: [
                    DrawingFrame.isTextFrameNode,
                    Dom.isDrawingFrame,
                    Dom.isCellNode,
                    Dom.isTableNode
                ]
            });

            eleType = _.last(arrDomTree);

            if (!enableKeys.every(controller.isItemEnabled, controller)) {
                if (isRangeSelected) {
                    event.preventDefault();
                    return;
                }
                var characterAttrs = AttributeUtils.getExplicitAttributes(node, { family: 'character', direct: true });
                if (!characterAttrs.url) {
                    event.preventDefault();
                    return;
                }
            }
        }

        function selfHide() { self.hide(); }

        // public methods ----------------------------------------------------
        /**
         * Returns whether a range is selected or not
         *
         * @returns {Boolean}
         *   Is a range selected or not
         */
        this.isRangeSelected = function () {
            return isRangeSelected;
        };

        this.isDrawingSelected = function () {
            return isDrawingSelected;
        };

        this.getXY = function (event) {
            if (isDrawingSelected) {
                return {
                    pageX: event.pageX,
                    pageY: event.pageY
                };

            } else if (_.browser.IE && $(event.target).is('.page')) {
                var focusNode = $(window.getSelection().focusNode.parentNode),
                    xy = focusNode.offset();

                return {
                    pageX: xy.left,
                    pageY: xy.top
                };

            } else {
                return (event.sourceEvent) ? {
                    pageX: event.sourceEvent.pageX,
                    pageY: event.sourceEvent.pageY
                } : null;
            }
        };

        // initialization -----------------------------------------------------

        // register some special handler for text
        this.registerContextMenuHandler(contextMenuEventHander);
        this.registerDetectHandler(detectHandler);
        this.registerResetHandler(resetHandler);

        // preprocessing before the context menu will be shown
        this.on('contextmenu:prepare', contextMenuPrepareHandler);

        this.on('popup:beforeshow', initMenu);

        this.listenTo(docView.getContentRootNode(), 'scroll', selfHide);
        this.listenTo(docView.getContentRootNode(), 'activedragging', selfHide);

        // destroy all class members
        this.registerDestructor(function () {
            self = docView = docModel = editDiv = initOptions = null;
            selection = spellChecker = fieldManager = null;
        });

    } // class UniversalContextMenu

    // exports ================================================================

    // derive this class from class ContextMenu
    return ContextMenu.extend({ constructor: UniversalContextMenu });

});
