/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Ingo Schmidt-Rosbiegal <ingo.schmidt-rosbiegal@open-xchange.com>
 */
define('io.ox/office/textframework/utils/listutils', [
    'io.ox/office/tk/utils',
    'io.ox/office/tk/locale/formatter'
], function (Utils, Formatter) {

    'use strict';

    // static class ListUtils =================================================

    var ListUtils = {};

    // constants --------------------------------------------------------------

    /**
     * A constant that will be used in the GUI to represent the default style
     * of bullet lists or numbered lists.
     *
     * @type String
     * @constant
     */
    ListUtils.DEFAULT_LIST_STYLE_ID = '__DEFAULT__';

    // static methods ---------------------------------------------------------

    /**
     * Formatting a specified number value into a string of specified format.
     * Supported values for the parameter 'format' are 'lowerletter', 'upperletter',
     * 'lowerroman' and 'upperroman'.
     *
     * @param {Number} value
     *  One-based number to be converted.
     *
     * @param {String} format
     *  The format, in which the number shall be converted.
     *
     * @returns {String}
     *  The converted specified value. Or the empty string, if no conversion
     *  was done.
     */
    ListUtils.formatNumber = function (value, format) {

        // checking the specified format
        switch (format.toLowerCase()) {
            case 'decimal':
                return value.toString();
            case 'lowerletter':
                return Formatter.formatAlphabetic(value, 'repeat', { lower: true });
            case 'upperletter':
                return Formatter.formatAlphabetic(value, 'repeat');
            case 'lowerroman':
                return Formatter.formatRoman(value, 4, { lower: true });
            case 'upperroman':
                return Formatter.formatRoman(value, 4);
        }

        Utils.error('Lists.formatNumber() - unknown number format: ' + format);

        return '';
    };

    /**
     * Auto detection of lists from a specified text.
     *
     * @param {String} text
     *  Possible numbering label text
     *
     * @returns {Object}
     *  An object containing the properties 'numberFormat', 'listStartValue'
     *  (numbered lists only) and symbol (bullet lists only). Allowed values
     *  for the numberFormat are 'bullet', 'decimal', 'upperRoman' and
     *  'lowerRoman'.
     *  If no list can be detected, an empty object is returned.
     */
    ListUtils.detectListSymbol = function (text) {

        // the return object used for collecting list information
        var ret = {};

        /**
         * Parsing a specified text to detect a list with Roman number format.
         *
         * @param {String} text
         *  Possible numbering label text
         *
         * @returns {Object}
         *  An object containing the properties 'numberFormat' and 'startnumber'.
         *  Allowed values for the numberFormat are 'upperRoman' and 'lowerRoman'.
         *  If no list can be detected, an empty object is returned.
         */
        function parseRoman(text) {

            var romanSmallArr = ['m', 'd', 'c', 'l', 'x', 'v', 'i'],
                romanValArr = [1000, 500, 100,  50,  10, 5, 1],
                localRet = {},
                lowerText = text.toLowerCase(),
                startValue = 0,
                index = 0,
                lastValue = 1000;

            localRet.caps = (lowerText !== text);

            for (; index < text.length; ++index) {

                var char = lowerText.charAt(index);
                if (char === '.') {
                    break;
                }
                var position = $.inArray(char, romanSmallArr);
                if (position < 0) {
                    return {};
                }
                var value = romanValArr[position];
                if (lastValue < value) {
                    startValue = startValue - lastValue + (value - lastValue);
                } else {
                    startValue += value;
                }
                lastValue = value;
            }

            if (startValue > 0) {
                localRet.startnumber = startValue;
                localRet.numberFormat = lowerText !== text ? 'upperRoman' : 'lowerRoman';
            }

            return localRet;
        }

        // searching in the specified string for '-', '*' or '.'

        if (text.length === 1 && (text === '-' || text === '*')) {
            // bullet
            ret.numberFormat = 'bullet';
            ret.symbol = text;
        } else if (text.substring(text.length - 1) === '.' || text.substring(text.length - 1) === ')') {
            var first = text.substring(0, 1);
            var sub = text.substring(0, text.length - 1);
            if (first === '(') { sub = text.substring(1, text.length - 1); }
            // 'sub' should only contain numbers or letters
            var startnumber = parseInt(sub, 10);
            if (startnumber > 0) {
                if (/^\d+$/.test(sub)) { // only numbers allowed in brackets or before dot
                    ret.numberFormat = 'decimal';
                    ret.listStartValue = startnumber;
                    ret.leftString = (first === '(') ? first : '';
                    ret.rightString = text.substring(text.length - 1);
                }
            } else {
                var roman = parseRoman(text);
                if (roman.startnumber > 0) {
                    ret.numberFormat = roman.numberFormat;
                    ret.listStartValue = roman.startnumber;
                    ret.leftString = '';
                    ret.rightString = text.substring(text.length - 1);
                }
            }
        }

        return ret;
    };

    // exports ================================================================

    return ListUtils;

});
