/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Stefan Eckert <stefan.eckert@open-xchange.com>
 */

define('io.ox/office/textframework/selection/androidselection', [
    'io.ox/office/textframework/utils/textutils',
    'io.ox/office/tk/keycodes',
    'io.ox/office/tk/forms',
    'io.ox/office/tk/render/font',
    'io.ox/office/textframework/utils/operations',
    'io.ox/office/textframework/utils/position',
    'io.ox/office/textframework/utils/dom'
], function (Utils, KeyCodes, Forms, Font, Operations, Position, DOM) {

    'use strict';

    var NBSP = '\xa0';
    var DUMMYLEN = 50;

    //first letter is workaround for Camel-Case-Behavior on some devices (Bug 40066)
    var INITTEXT = 'G' + Utils.repeatString(' ', DUMMYLEN - 1);

    var ENDCHARS = Position.getWordBoundEndChars();

    var MONO_FONT = new Font('monospace', 20); // size in points
    var LETTER_WIDTH = MONO_FONT.getTextWidth('A');

    // mix-in class AndroidSelection ==========================================

    /**
     *
     * @constructor
     */
    function AndroidSelection(docModel, updateLogicalPosition) {

        var self = this;

        // the application instance
        var app = docModel.getApp();

        // slide mode for spreadsheet/presentation
        var slideMode = docModel.useSlideMode();

        var blinkDeferred = null;
        var positionOx = null;

        var wordFrom = null;
        var wordTo = null;
        var word = null;

        var div = $('<div class="edit-overlay">');
        var textarea = null;
        var caret = null;
        var highlight = null;
        var singleArray = [0];

        var restoreFN = $.noop;

        // private methods ----------------------------------------------------

        function updateHighlight(from, to) {
            var rootNode = docModel.getSelection().getRootNode();
            var imeRect = getRectangleFromPoints(Position.getDOMPosition(rootNode, from), Position.getDOMPosition(rootNode, to));

            highlight.css({
                top: imeRect.top + imeRect.height + 1,
                left: imeRect.left,
                width: imeRect.width
            });

        }
        function isSoftKeyboardOpen() {
            return (Utils.getInnerWindowHeight() / Utils.getScreenHeight()) < 0.7;
        }

        function getRectangleFromPoints(point1, point2) {
            // the root node of the document
            var rootNode = docModel.getNode();

            var offsetTop = parseFloat(rootNode.css('margin-top'));
            // use only top because page transform-origin is "center top"

            var rect = Utils.getCSSRectangleFromPoints(point1, point2, rootNode, 1, true);
            rect.left +=  (app.getView().getZoomFactor() < 0.99 ? 3 : 0); // magic number '3' is needed because of 'recalculateDocumentMargin' in presentation/view.js
            rect.top += offsetTop;
            return rect;
        }

        function splitTextSpan(para, pos) {
            singleArray[0] = _.last(pos);

            if (singleArray[0] === 0) {
                var firstSpan = para.children().first();
                if (!firstSpan.is('span') || !firstSpan[0].innerHTML.length) {
                    return firstSpan[0];
                } else {
                    var newSpan = firstSpan.clone(true);
                    fillTextInSpan(newSpan[0], '');
                    para.prepend(newSpan);
                    return newSpan[0];
                }
            }

            var domPos = Position.getDOMPosition(para[0], singleArray);
            if (!domPos) {
                Utils.error('no result', para.children().length, pos, para[0]);
                return null;
            }

            if (domPos.node.nodeValue.length === domPos.offset) {
                return domPos.node.parentElement;
            } else {
                return DOM.splitTextSpan(domPos.node.parentElement, domPos.offset)[0];
            }
        }

        function fillTextInSpan(span, text) {
            span.innerHTML = '';
            span.appendChild(document.createTextNode(text));
        }

        function changeSelection(event, selectionInfo) {
            //console.log('fff changeSelection', selectionInfo);
            var start = _.clone(selectionInfo.start);
            var end = _.clone(selectionInfo.end);
            // the selection object
            var selection = docModel.getSelection();

            positionOx = null;

//            var visibility = 'hidden';
            var imeVis = 'hidden';

            if (app.isEditable()) {

                var selType = window.getSelection().type;
                if (!selType || selType === 'None') {
                    selType = isSoftKeyboardOpen();
                }

                if (_.isEqual(start, end) && selType) {

                    var oldPara = $(Position.getDOMPosition(selectionInfo.rootNode, _.initial(start)).node);
                    var wordSelection = Position.getWordSelection(oldPara, _.last(start), ENDCHARS, { fillPlaceholders: true });
                    if (wordSelection) {
                        var i = 0;
                        for (i = 0; i < wordSelection.text.length; i++) {
                            if (wordSelection.text[0] === ' ') {
                                wordSelection.text = wordSelection.text.substring(1, wordSelection.text.length);
                                wordSelection.start++;
                            } else {
                                break;
                            }
                        }
                        for (i = wordSelection.text.length - 1; i >= 0; i--) {
                            if (wordSelection.text[wordSelection.text.length - 1] === ' ') {
                                wordSelection.text = wordSelection.text.substring(0, wordSelection.text.length - 1);
                                wordSelection.end--;
                            } else {
                                break;
                            }
                        }

                        wordSelection.text = wordSelection.text.replace(/\s/g, '\u200B');

                        wordFrom = _.clone(start);
                        wordFrom[wordFrom.length - 1] = wordSelection.start;

                        wordTo = _.clone(start);
                        wordTo[wordTo.length - 1] = wordSelection.end;

                        if (_.last(wordFrom) < _.last(wordTo)) {
                            word = wordSelection.text;

                            imeVis = 'visible';
                            updateHighlight(wordFrom, wordTo);
                        } else {
                            wordFrom = _.clone(start);
                            wordTo =  _.clone(start);
                            word = '';
                        }
                    } else {
                        wordFrom = _.clone(start);
                        wordTo =  _.clone(start);
                        word = '';
                    }

                    positionOx = { start: start, end: end, para: oldPara, rootNode: selectionInfo.rootNode };

                    selectTextArea();

                    // if (slideMode || !app.getView().isSoftkeyboardEditModeOff()) {
                    //     selectTextArea();
                    // }

                    handleTextAreaPos();
        //            visibility = 'visible';
                }

                // no blinking cursor, if a drawing is selected or this is a slide selection (52922)
                if (docModel.useSlideMode()) { caret.toggleClass('noblinkingcursor', selection.isSlideSelection() || selection.isDrawingSelection()); }

            }

        //    textarea.css('visibility', visibility);
            highlight.css('visibility', imeVis);
        }

        function handleTextAreaPos() {
            // not edit mode -> return
            if (positionOx === null) { return; }

            var start = Position.getDOMPosition(positionOx.rootNode, positionOx.start);
            if (!start) {
                Utils.warn(positionOx.start + ' does not lead to a DomPos!');
                return;
            }

            var end = null;
            if (_.last(positionOx.start) > 0) {
                // workaround for empty paragraph -> lead to zero size range rectangle
                end = start;
            }
            var rect = getRectangleFromPoints(start, end);

            caret.css({
                left: rect.left,
                top: rect.top,
                width: 2,
                height: rect.height
            });
            textarea.css({
                left: rect.left - 4,
                top: rect.top - 10,
                width: 8,
                minWidth: 8,
                height: rect.height + 20,
                zIndex: 100
            });
        }

        function isPassThrough(event, press) {
            if (event.ctrlKey || event.altKey || event.metaKey) { return true; }

            if (press) {
                return event.keyCode > 0  && event.keyCode !== KeyCodes.IME_INPUT;
            } else if (event.keyCode === KeyCodes.TAB) {
                event.preventDefault();
                return true;
            } else if (event.shiftKey && DOM.isCursorKey(event.keyCode)) {
                restoreFN();
                event.preventDefault();
                return true;
            } else if (event.keyCode === KeyCodes.BACKSPACE) {
                event.preventDefault();
                return true;
            } else if (DOM.isCursorKey(event.keyCode) || event.keyCode === KeyCodes.ENTER || event.keyCode === KeyCodes.DELETE) {
                return true;
            } else if (event.keyCode > 0  && event.keyCode !== KeyCodes.IME_INPUT) {
                return true;
            } else {
                return false;
            }
        }

        function passThrough(event) {
            if (isPassThrough(event, true)) {
                handlePassThrough(event);
            }
        }

        function handlePassThrough(event) {
            var rootNode = docModel.getNode();

            if (!event.shiftKey && (event.keyCode === KeyCodes.UP_ARROW || event.keyCode === KeyCodes.DOWN_ARROW)) {
                event.preventDefault();
                /*
                FIXME
                if (event.type === 'keydown' && oldPara) {
                    var offset = event.keyCode === KeyCodes.UP_ARROW ? -1 : 1;

                    var cursorSpan = oldPara.find('.' + CURSOR);
                    var w = $(window);

                    var height = cursorSpan.height();
                    offset *= height * 1.5;

                    var pos = cursorSpan.offset();

                    var top = pos.top - w.scrollTop();
                    var left = pos.left + cursorSpan.width() - w.scrollLeft();

                    var newOxo = Position.getOxoPositionFromPixelPosition(rootNode, left, top + height / 2 + offset);

                    if (!newOxo) {
                        if (oldStart.length > 2) {
                            return;
                        }
                        newOxo = { start: [] };
                        if (event.keyCode === KeyCodes.UP_ARROW) {
                            newOxo.start[0] = Math.max(0, oldStart[0] - 1);
                        } else {
                            newOxo.start[0] = oldStart[0] + 1;
                        }
                        newOxo.start.push(0);
                    }

                    self.setTextSelection(newOxo.start);
                }*/
            } else if (!event.shiftKey && (event.keyCode === KeyCodes.LEFT_ARROW || event.keyCode === KeyCodes.RIGHT_ARROW)) {
                docModel.handleAndroidTyping();
                event.preventDefault();
                event.preventDefault = $.noop;
                rootNode.trigger(event);
            } else {
                docModel.handleAndroidTyping();
                event.preventDefault = $.noop;
                rootNode.trigger(event);
            }
            return true;
        }

        function simulateEvent(event, keyCode) {

            updateLogicalPosition(positionOx.start, { start: true });
            updateLogicalPosition(positionOx.end, { start: false });

            textarea.val(INITTEXT);
            Forms.setInputSelection(textarea, DUMMYLEN);

            if (event.type === 'keyup') {
                handlePassThrough(new $.Event('keydown', { keyCode: keyCode }));
                handlePassThrough(new $.Event('keypress', { keyCode: keyCode, charCode: keyCode }));
                handlePassThrough(new $.Event('keyup', { keyCode: keyCode }));
            }
        }

        function changedText(event) {

            var cursorIndex = getCursorIndex();
            // console.log('');
            // console.warn('space?', cursorIndex, event.type, event.keyCode, event.charCode);

            if (isPassThrough(event)) {
                handlePassThrough(event);
                restoreFN();
                return;
            } else {
                var text = getText();
                // console.warn('text', text.length, '"' + text + '"');

                if (cursorIndex < 0) {
                    // backspace!
                    // console.warn('sim BACKSPACE');
                    simulateEvent(event, KeyCodes.BACKSPACE);
                    return;
                } else if (!text.length) {
                    docModel.handleAndroidTyping();
                    // console.warn('RETURN');
                    return;
                } else if (text === ' ' && cursorIndex === 1) {
                    // space!
                    // console.warn('sim SPACE');
                    simulateEvent(event, KeyCodes.SPACE);
                    return;
                } else if (text.indexOf('\n') >= 0) {
                    // enter!
                    // console.warn('sim ENTER');
                    simulateEvent(event, KeyCodes.ENTER);
                    return;
                }

                if (text.length > 0) {

                    var preAttrs = docModel.getPreselectedAttributes() || undefined;
                    var attrs = preAttrs;
                    var changeTrack = docModel.getChangeTrack();
                    if (changeTrack.isActiveChangeTracking()) {
                        if (!attrs) {
                            attrs = {};
                        } else {
                            attrs = _.copy(attrs, true);
                        }
                        attrs.changes = { inserted: changeTrack.getChangeTrackInfo() };
                    }

                    var undoOperations = [];
                    var redoOperations = [];

                    if (DOM.isImplicitParagraphNode(positionOx.para[0])) {
                        positionOx.para.data('implicit', false);
                        redoOperations.push({ name: Operations.PARA_INSERT, start: _.initial(wordFrom) });
                        makeDelOps(undoOperations, _.initial(wordFrom));
                    }

                    var insertOp = { name: Operations.TEXT_INSERT, start: _.clone(wordTo), text: text, attrs: attrs };
                    redoOperations.push(insertOp);

                    var startWord = _.clone(wordFrom);
                    var endWord = null;
                    var offset = 0;

                    if (_.last(wordFrom) !== _.last(wordTo)) {
                        endWord = Position.increaseLastIndex(wordTo, -1);
                        offset = makeDelOps(redoOperations, startWord, endWord);

                        var insStart = Position.increaseLastIndex(wordFrom, text.length);
                        var undoDeleteOp = { name: Operations.TEXT_INSERT, start: insStart, text: word, attrs: preAttrs };
                        undoOperations.push(undoDeleteOp);
                    } else {
                        endWord = _.clone(wordFrom);
                    }

                    makeDelOps(undoOperations, startWord, endWord);

                    sendOperations(docModel, changeTrack, undoOperations, redoOperations);

                    var leftNoIME = _.contains(ENDCHARS, text[cursorIndex - 1]);

                    applyOperations(docModel, redoOperations);

                    // var pos = _.last(wordTo) - text.length + cursorIndex;
                    var pos = _.last(wordFrom) + cursorIndex;
                    positionOx.start[positionOx.start.length - 1] = pos;
                    positionOx.end[positionOx.end.length - 1] = pos;

                    updateLogicalPosition(positionOx.start, { start: true });
                    updateLogicalPosition(positionOx.end, { start: false });

                    if (leftNoIME) {
                        word = text.substring(cursorIndex);
                        wordFrom[wordFrom.length - 1] = pos;
                        wordTo[wordTo.length - 1] = _.last(wordFrom) + word.length;

                        selectTextArea();
                    } else {
                        word = text;
                        wordTo[wordTo.length - 1] = _.last(wordFrom) + text.length;
                    }

                    updateHighlight(wordFrom, wordTo);
                    handleTextAreaPos();

                    // console.warn('change', 'start', positionOx.start, 'wordFrom', wordFrom, 'wordTo', wordTo);

                    if (attrs) {
                        docModel.getStyleCollection('paragraph').updateElementFormatting(positionOx.para);
                    }
                    if (offset > 0) {
                        changeSelection(null, { start:  _.clone(positionOx.start), end: _.clone(positionOx.end), rootNode: docModel.getSelection().getRootNode() });
                    }
                }
                docModel.handleAndroidTyping();
            }
        }

        function makeDelOps(ops, start, end) {
            var offset = 0;
            if (end) {
                var startPos = getDOMPosition(start);
                var endPos = getDOMPosition(end);
                if (startPos.nodeIndex === endPos.nodeIndex) {
                    ops.push({ name: Operations.DELETE, start: start, end: end });
                } else {

                    for (var i = _.last(end); i >= _.last(start); i--) {
                        var pos = _.initial(start);
                        pos.push(i);

                        var domPos = getDOMPosition(pos);
                        if (DOM.isRangeMarkerNode(domPos.node) || DOM.isSubstitutionPlaceHolderNode(domPos.node)) {
                            offset++;
                        } else {
                            ops.push({ name: Operations.DELETE, start: pos, end: pos });
                        }
                    }
                }
            } else {
                ops.push({ name: Operations.DELETE, start: start });
            }
            return offset;
        }

        function sendOperations(docModel, changeTrack, undoOperations, redoOperations) {
            if (changeTrack.isActiveChangeTracking()) {
                var info = changeTrack.getChangeTrackInfo();
                _.each(undoOperations, function (op) {
                    addChangeTrackInfo(op, info);
                });
                _.each(redoOperations, function (op) {
                    addChangeTrackInfo(op, info);
                });
            }
            docModel.sendOperations(redoOperations);
            docModel.getUndoManager().addUndo(undoOperations, redoOperations);
        }

        function getDOMPosition(pos) {
            if (!pos) { return; }

            singleArray[0] = _.last(pos);
            var node = Position.getDOMPosition(positionOx.para, singleArray, true);
            if (node) {
//                node.node = node.node.parentElement;
                node.nodeIndex = $(node.node).index();
                return node;
            }
            throw new Error('nothing found');
        }

        function applyOperations(docModel, ops) {
            _.each(ops, function (op) {

                var start = getDOMPosition(op.start);
                var text;

                if (op.name === Operations.DELETE) {
                    var end = getDOMPosition(op.end);

                    if (start.nodeIndex === end.nodeIndex) {
                        text = start.node.innerText;
                        text = text.substring(0, start.offset) + text.substring(end.offset + 1);
                        fillTextInSpan(start.node, text);
                    } else {
                        var startText = start.node.innerText;
                        fillTextInSpan(start.node, startText.substring(0, start.offset));
                        var endText = end.node.innerText;
                        fillTextInSpan(end.node, endText.substring(end.offset + 1));
                        var children = positionOx.para.children();
                        if ((end.nodeIndex - start.nodeIndex) > 1) {
                            for (var ni = start.nodeIndex + 1; ni < end.nodeIndex; ni++) {
                                positionOx.para[0].removeChild(children[ni]);
                            }
                        }
                    }

                } else if (op.name ===  Operations.TEXT_INSERT) {
                    var opText = op.text;
                    if (opText[opText.length - 1] === ' ') {
                        opText = opText.substring(0, opText.length - 1) + NBSP;
                    }

                    if (!start.offset && start.nodeIndex > 0 && DOM.isTextSpan(start.node.previousSibling)) {
                        start.nodeIndex--;
                        start.node = start.node.previousSibling;
                        start.offset = start.node.innerHTML.length;
                    }

                    var attrs = op.attrs;
                    text = start.node.innerText;
                    text = text.substring(0, start.offset) + opText + text.substring(start.offset);
                    fillTextInSpan(start.node, text);

                    var splitStart = _.last(op.start);
                    var splitEnd = splitStart - start.offset;

                    // console.warn('splitStart', splitStart, 'splitEnd', splitEnd);

                    var counter = start.offset;
                    for (var i = splitStart; i >= splitEnd; i--) {
                        counter--;
                        if (_.contains(ENDCHARS, text[counter])) {

                            var before = splitTextSpan(positionOx.para, [i]);
                            var beforeText = before.innerText;

                            if (_.last(beforeText) === NBSP) {
                                fillTextInSpan(before, beforeText.substring(0, beforeText.length - 1) + ' ');
                            }
                            break;
                        }
                    }

                    handlePreselectedAttrs(attrs, start); //fix me wrong node
                }
            });

        }

        function addChangeTrackInfo(op, info) {
            var attrs = op.attrs;
            if (attrs) {
                if (attrs.changes) {
                    return;
                }
                attrs = _.copy(attrs, true);
            } else {
                attrs = {};
            }

            var name;
            if (op.name === Operations.TEXT_INSERT) {
                name = 'inserted';
            } else {
                name = 'deleted';
            }
            attrs.changes = {};
            attrs.changes[name] = info;

        }

        function handlePreselectedAttrs(attrs, domPos) {

            if (attrs) {
                // if (domPos.offset > 0) {   //TODO must be splitted
                //     DOM.splitTextSpan(domPos.node, domPos.offset);
                // }
                attrs = { attributes: attrs };
                $(domPos.node).data(attrs);
            }
        }

        textarea = $('<textarea class="edit-textarea">');
        caret = $('<div class="edit-caret"></div>');
        highlight = $('<div class="edit-highlight"></div>');

        function selectTextArea() {
            //console.error('fff selectTextArea');

            // not edit mode -> return
            if (positionOx === null) { return; }

            if (document.activeElement !== textarea[0]) {
                Utils.setFocus(textarea);
            }
            textarea.val(INITTEXT + word);
            Forms.setInputSelection(textarea, DUMMYLEN + (_.last(positionOx.start) - _.last(wordFrom)));
        }

        function getCursorIndex() {
            return textarea[0].selectionStart - DUMMYLEN;
        }

        function getText() {
            return textarea.val().substring(DUMMYLEN).replace(/\\u200B/g, '');
        }

        var processSelectUpdate = (function () {
        //    var lastEvent = null;

            var always = function () {
                if (!docModel) { return; }

                var selecttype = window.getSelection().type;
        //        console.log('always');
                // selecting
                var rootNode = docModel.getSelection().getRootNode();
//                console.log('always1', Utils.getActiveElement(), rootNode[0]);
                if (Utils.getActiveElement() === rootNode[0] || (slideMode && $(Utils.getActiveElement()).is('.drawing'))) {
//                    console.log('in....');
                    // listener hangs directly on document, but we only want the events of editdiv
                    if (selecttype !== 'None') {
//                        console.log('always2', selecttype);
                        self.updateSelectionAfterBrowserSelection({ preserveFocus: false, forceTrigger: false }); // prpbrowserevent
        //               self.processBrowserEvent(lastEvent);

                    }

                    // hide caret when a selection range is visible
                    if (selecttype === 'Range') {
            //            console.log('RANGE');
                        caret.css('visibility', 'hidden');
                        highlight.css('visibility', 'hidden');

                        // close keyboard when a range is selected and the keyboard is open -> ime backspace problem
                        if (isSoftKeyboardOpen()) {
    //                        console.log('CLOSE KEYBOARD');
                            docModel.trigger('selection:possibleKeyboardClose', { forceClose: true });
                        }

                    } else {
                        caret.css('visibility', 'visible');
                    }

                // writing
                } else if (Utils.getActiveElement() === textarea[0]) {
                    if (window.getSelection().anchorNode === div[0]) {
                        //focus really in textarea
                        if (selecttype === 'Range') {
    //                        console.log('TEXTAREA');
                            if (_.last(wordFrom) < _.last(wordTo)) {
                                restoreFN();
                                self.setTextSelection(wordFrom, wordTo);
                            }
                        }
                    }
                }
            };
            var notAlways =  _.throttle(always, 250);

            return function (evt) {

                var selecttype = window.getSelection().type;

            //    console.log('processSelectUpdate', selecttype);
    //            lastEvent = evt;

                if (evt.type === 'selectstart') {
                    if (selecttype === 'Range') {
                        ///User seems to call "select all", but android chrome does not really support that featurem when contenteditable=true and begins with scrolling :(
                        evt.preventDefault();
                        return;
                    }
                    always();
                } else {
                    notAlways();
                }

                // if (selecttype === 'Range') {
                //     if (isSoftKeyboardOpen()) {
                //         docModel.trigger('selection:possibleKeyboardClose');
                //     }
                // }

            };
        }());

        // initialization -----------------------------------------------------

        textarea.on('keyup keydown', changedText);
        textarea.on('keypress', passThrough);

        self.on('changecursorinfo', changeSelection);

        blinkDeferred = app.repeatDelayed(function () {
            caret.toggleClass('androidblink');
        }, 'AndroidSelection.blinkDeferred', 500);

        this.waitForImportSuccess(function () {
            var docView = app.getView();

            restoreFN = _.bind(self.restoreBrowserSelection, self);
            /*
            self.restoreBrowserSelection = function () {
                //sideEffect of passThrough, with original restoreFN textarea-focus gets lost
                if (!_.isEqual(self.getStartPosition(), self.getEndPosition())) {
                    restoreFN();
                }
            };
            */

            textarea.on('cut copy paste', function (evt) {
                // TODO maybe update oxoposition to be sure that it's correct?
                docModel[evt.type](evt);
            });

            textarea.css({
                letterSpacing: -LETTER_WIDTH + 'px',
                fontFamily: MONO_FONT.family,
                fontSize: MONO_FONT.size + 'pt',
                lineHeight: MONO_FONT.size + 'pt'
            });

            textarea.addClass('transparent-selection');

            div.append(textarea);
            div.append(caret);
            div.append(highlight);

            docModel.getNode().parent().append(div);

            docModel.listenTo($(document), 'selectstart selectionchange select', processSelectUpdate);

            app.registerWindowResizeHandler(function () {

                //console.warn('resize', 'soft off:   ', app.getWindowNode().hasClass('soft-keyboard-off'));
                //console.warn('resize', 'isPopupOpen:', BasePopup.isPopupOpen());
                //console.warn('resize', 'keyboard op:', isSoftKeyboardOpen());

                // too early
                if (positionOx === null) { return; }

                handleTextAreaPos();
    //            console.log('ANDROID RESIZE');
                if (app.getWindowNode().hasClass('soft-keyboard-off')) {
                    // do nothing
                } else {
                    if ($('.io-ox-office-main.popup-container:not(.context-menu)').length > 0) {
                        // do nothing, wait for popup close
                    } else if (!isSoftKeyboardOpen()) {
                        docModel.trigger('selection:possibleKeyboardClose');
                    }
                }
            });

            docModel.on('selection:focusToEditPage', selectTextArea);
            docModel.listenTo(docView, 'change:zoom', function () {
                if (wordFrom && wordTo) { updateHighlight(wordFrom, wordTo); }
                handleTextAreaPos();
            });

        });

        this.listenTo(app, 'docs:beforequit', function () {
            textarea.remove();
            div.remove();
            blinkDeferred.abort();
        });

        // destroy all class members on destruction
        this.registerDestructor(function () {
            self = docModel = app = blinkDeferred = textarea = div = null;
        });

    } // class AndroidSelection

    // exports ================================================================

    return AndroidSelection;

});
