/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Stefan Eckert <stefan.eckert@open-xchange.com>
 * @author Ingo Schmidt-Rosbiegal <ingo.schmidt-rosbiegal@open-xchange.com>
 * @author York Richter <york.richter@open-xchange.com>
 */

define('io.ox/office/textframework/components/spellcheck/spellchecker', [

    'io.ox/office/tk/utils',

    'io.ox/office/tk/locale/localedata',
    'io.ox/office/tk/object/triggerobject',

    'io.ox/office/settings/userdictionary',
    'io.ox/office/settings/spellchecking',
    'io.ox/office/settings/spellchecklanguagenotification',

    'io.ox/office/baseframework/app/appobjectmixin',

    'io.ox/office/textframework/utils/config',
    'io.ox/office/textframework/utils/dom',
    'io.ox/office/textframework/utils/position',
    'io.ox/office/textframework/utils/textutils',

    'gettext!io.ox/office/textframework/main'

], function (Utils, LocaleData, TriggerObject, UserDictionary, SpellChecking, SpellcheckNotification, AppObjectMixin, Config, DOM, Position, TextUtils, gt) {

    'use strict';

    // class SpellChecker =====================================================

    /**
     * The spell checker object.
     *
     * @constructor
     *
     * @extends TriggerObject
     * @extends AppObjectMixin
     *
     * @param {Editor} model
     *  The text editor model.
     */
    function SpellChecker(model, initOptions) {

        var // self reference for local functions
            self = this,
            // the application instance
            app = model.getApp(),
            // online spelling mode off (there are no app specific spell-check user settings anymore due to DOCS-630)
            spellingSupported = Config.SPELLING_ENABLED && Utils.getBooleanOption(initOptions, 'spellingSupported', false),
            // the scroll node
            scrollNode = null,
            // collects replacements of misspelled words by locale
            spellReplacementsByLocale = { },
            // the cache for the modified paragraphs, that need to be spell checked
            paragraphCache = $(),
            //supported languages to check the spelling
            supportedLanguages,
            // list for the unsupported language without a notification
            unsupportedLanguageWithoutNotification = [],
            // list of unsupported languages with a notification
            unsupportedLanguages = [],

            // in case a page does start as empty it will flag whether it always stod empty or if it, at one point, did contain real text content.
            hasPageAlwaysBeenEmpty = true, // the `true` value default will be overwritten exactly once as soon as the page starts carrying content.

            // list of all languages that are not supposed to be notified about theirs missing spell-check(er) support.
            spellcheckLanguageBlacklist = [],
            // local list of misspelled words which are ignored by the user
            ignoredMisspelledWords = _.clone(UserDictionary.getDictionary()),
            // The undo manager
            undoManager = model.getUndoManager(),
            // ignore actions for add or remove a ignore word, it will be used in the 'implStartAddOrRemoveIgnoreWords' function
            ignoreWordActions = [],
            // counter to set a unique request id
            requestIDCounter = 0,
            // Set to true if the User Dictionary is changed, but the app is not visible
            userDictionaryChanges = false,
            // If a word is misspelled select the whole word, on closing the contextmenu the selection must be restored
            restoreStartPosition,
            // handle if the app get visible
            appVisibleHandler = function () {
                // userDictionaryChanges is true if the user change the dictionary while the spellchecker document is not visible.
                if (userDictionaryChanges) {
                    userDictionaryChanges = false;
                    implStartOnlineSpellingDebounced();
                }
            };

        // base constructor ---------------------------------------------------

        TriggerObject.call(this, model);
        AppObjectMixin.call(this, app);

        // private methods ----------------------------------------------------

        /**
         * Helper function for receiving character styles
         */
        function getCharacterStyles() {
            return model.getStyleCollection('character');
        }

        /**
         * Starting spell checking by calling the worker function 'implCheckSpelling'
         */
        function implStartOnlineSpelling() {
            // # 51390: Spellchecking can interrupt an active composition by manipulating
            // DOM nodes. Make sure that this doesn't happen.
            if (self.isOnlineSpelling() && app.isEditable() && (_.isFunction(model.isImeActive) && !model.isImeActive()) && self.isImportFinished()) {
                implCheckSpelling();
            }
        }

        /**
         * A debounced function for starting spell checking by calling 'implStartOnlineSpelling'
         */
        var implStartOnlineSpellingDebounced = model.createDebouncedMethod('SpellChecker.implStartOnlineSpellingDebounced', null, implStartOnlineSpelling, { delay: 1000 });

        /**
         * Receiving a list of replacement words for a specified word in a specified language from
         * the global replacement collection.
         *
         * @param {String} word
         *  The word to be replaced.
         *
         * @param {String} language
         *  The language of the word to be replaced.
         *
         * @returns {String[]}
         *  An array containing all replacement words for the specified word in the specified language.
         *  If there is no replacement word, the array will be empty.
         */
        function getSpellReplacements(word, language) {
            return spellReplacementsByLocale[language] ? spellReplacementsByLocale[language][word] : [];
        }

        /**
         * Checking, whether two different spell results are identical. A spell check result is an array containing
         * spell check error objects. One object has the following format:
         * Example: {"start":0,"length":2,"replacements":["da","dB","DSL"],"locale":"de_DE","word":"ds"}
         *
         * @param {Object[]} result1
         *  An spell check result container.
         *
         * @param {Object[]} result2
         *  Another spell check result container.
         *
         * @returns {Number}
         *  The number of the first difference. If number is '-1', there is no difference.
         */
        function sameSpellResults(result1, result2) {

            var sameResults = 0,
                counter = 0,
                length1 = result1.length,
                length2 = result2.length,
                minLength = Math.min(length1, length2),
                sameError = minLength > 0;

            // comparing two single result objects
            function compareResultObjects(res1, res2) {
                return res1.start === res2.start && res1.length === res2.length && res1.word === res2.word && res1.locale === res2.locale;
            }

            while (sameError && result1[counter] && result2[counter]) {

                sameError = compareResultObjects(result1[counter], result2[counter]);

                if (sameError) {
                    // comparing the next error objects
                    counter++;
                } else {
                    // finding the first different position
                    sameResults = Math.min(result1[counter].start, result2[counter].start);
                }
            }

            // reaching the end of at least one error array
            if (sameError) {
                if (length1 === length2) {
                    // result arrays are identical
                    sameResults = -1;
                } else {
                    // using the previous error
                    counter--;
                    // increasing the minimum position for the same results
                    sameResults = result1[counter].start + result1[counter].length;
                }
            }

            return sameResults;
        }

        /**
         * Try to merge a span, that is no longer marked with spell error attribute. This can
         * happen for example after inserting a missing letter or after removing a superfluous
         * letter, so that several different text span can be merged.
         *
         * @param {Node[]|jQuery} spans
         *  The list of spans, that need to be checked, if they can be merged with their neighbors.
         */
        function mergeNonSpellErrorSpans(spans) {
            if (spans && _.isNumber(spans.length) && spans.length > 0) {
                _.each(spans, function (span) {
                    if (!DOM.isSpellerrorNode(span)) {
                        TextUtils.mergeSiblingTextSpans(span);
                        TextUtils.mergeSiblingTextSpans(span, true);
                    }
                });
            }
        }

        /**
         * Removing all spell errors inside a paragraph
         *
         * @param {Node|jQuery} paragraph
         *  The DOM node from which the class shall be removed.
         */
        function removeAllSpellErrorsInParagraph(paragraph) {

            var // a container for all spans with spell error
                allSpans = $(paragraph).find('span.spellerror');

            _.each(allSpans, function (span) {
                self.clearSpellErrorAttributes(span); // remove spell attributes - if there are any
            });

            // try to merge with neighbors after removing the spell check attribute
            mergeNonSpellErrorSpans(allSpans);
        }

        /**
         * Helper function that applies the results received from the server to the DOM.
         *
         * @param {Object[]} spellResult
         *  A collection of spell results for the specified paragraph received from the server.
         *
         * @param {Node[]|jQuery} textSpans
         *  The collected text spans inside the specified paragraph.
         *
         * @param {Number} minPos
         *  The minimum position that will be evaluated inside the paragraph.
         */
        function applySpellcheckResults(spellResult, textSpans, minPos) {

            var // a collector for all 'error spans'
                errorSpanArray = [],
                // the offset of the span inside the paragraph
                spanOffset = 0,
                // the length of a text span
                spanLength,
                // a collector for text spans, that can maybe be merged later
                allClearedSpans = [];

            // saving the results in the global collector for all replacements
            _(spellResult).each(function (result) {
                if (result.replacements && result.locale && result.word) {
                    if (!spellReplacementsByLocale[result.locale]) {
                        spellReplacementsByLocale[result.locale] = { };
                    }
                    spellReplacementsByLocale[result.locale][result.word] = result.replacements;
                }
            });

            // iterate over all text spans to generate an array for the error spans
            _(textSpans).each(function (currentSpan) {

                var result,
                    localErrors = [],
                    ignoreError = false;

                spanLength = currentSpan.firstChild.nodeValue.length;

                // ignore all spans at positions left of the minimal position
                if (spanOffset + spanLength < minPos) {
                    ignoreError = true;
                }

                if (!ignoreError) {

                    // removing an existing spell-error attribute from the text span
                    if ($(currentSpan).hasClass(DOM.SPELLERRORNODE_CLASS)) {
                        // clearing existing spell-error attributes from text spans behind 'minpos'
                        self.clearSpellErrorAttributes(currentSpan);
                        // collecting this span to enable later merging of text spans
                        allClearedSpans.push(currentSpan);
                    }

                    // iterating over all errors, to find precise position inside the current text span
                    for (result = 0; result < spellResult.length; ++result) {

                        var currErrorStart = spellResult[result].start,
                            currErrorLength = spellResult[result].length,
                            maxError = currErrorStart + currErrorLength,
                            minError = currErrorStart;

                        if (maxError <= spanOffset) {
                            continue;
                        }

                        if (minError >= spanOffset + spanLength) {
                            break;
                        }

                        if (spellResult[result].word && self.isIgnoredWord(spellResult[result].word.slice(0, currErrorLength))) {
                            continue;
                        }

                        if (minError < spanOffset) {
                            currErrorLength -= spanOffset - minError;
                            minError = 0;
                        }

                        currErrorStart = (minError <= spanOffset) ? 0 : (minError - spanOffset);

                        if (maxError > spanOffset + spanLength) {
                            currErrorLength = spanLength - (currErrorStart - spanOffset);
                        }

                        // saving start and length of error range
                        localErrors.push({ start: currErrorStart, length: currErrorLength });
                    }

                    // collecting all error ranges for the current span
                    if (localErrors.length > 0) {
                        errorSpanArray.push({ span: currentSpan, errors: localErrors });
                    }
                }

                spanOffset += spanLength;
            });

            // iterating over all error spans, to make them visible
            _.each(errorSpanArray, function (errorSpan) {

                var spanPositionOffset = 0,
                    newSpan,
                    currentSpan = errorSpan.span;

                _.each(errorSpan.errors, function (localError) {

                    newSpan = null;

                    if (localError.start > spanPositionOffset && (localError.start - spanPositionOffset) < currentSpan.textContent.length) {
                        DOM.splitTextSpan(currentSpan, localError.start - spanPositionOffset);
                        spanPositionOffset += localError.start - spanPositionOffset;
                    }

                    // split end of text span NOT covered by the error
                    if (localError.length > 0 && currentSpan.textContent.length > localError.length) {
                        newSpan = DOM.splitTextSpan(currentSpan, localError.length, { append: true });
                        spanPositionOffset += localError.length;
                    }

                    // setting the 'spellerror' attribute to the span, so that it will be underlined
                    getCharacterStyles().setElementAttributes(currentSpan, { character: { spellerror: true } }, { special: true });

                    if (newSpan !== null) {
                        currentSpan = newSpan[0];
                    }
                });
            });

            // try to merge collected text spans with their neighbors after removing the spell check attribute
            mergeNonSpellErrorSpans(allClearedSpans);
        }

        /**
         * The handler function, that evaluates the results received from the server.
         *
         * @param {Object[]} spellResult
         *  A collection of spell results for the specified paragraph received from the server.
         *
         * @param {Node[]|jQuery} textSpans
         *  The collected text spans inside the specified paragraph.
         *
         * @param {Node} paragraph
         *  The paragraph, to which the spell results will be applied.
         */
        var spellResultHandler = Utils.profileMethod('SpellChecker: Applying spell check results: ', function (spellResult, textSpans, paragraph) {

            var // the first position with a changed spell check error
                firstChangedPos = 0,
                // the spell check results of a previous spell check process
                oldSpellResult = null,
                // the selection object
                selection = model.getSelection();

            // convert paragraph to jQuery Object
            paragraph = $(paragraph);

            // do nothing, if the paragraph was modified in the meantime (before the answer was received)
            if (!paragraph.hasClass('p_spelled')) {
                return;
            }

            // checking for the old spelling results
            oldSpellResult = paragraph.data(SpellChecker.SPELL_RESULT_CACHE);

            // handling empty result array (will be handled in fail handler)
            if (spellResult.length === 0) {
                if (oldSpellResult) {
                    // no errors now, but some errors before
                    removeAllSpellErrorsInParagraph(paragraph);
                    self.clearSpellResultCache(paragraph);
                } else {
                    // no errors now and no errors before
                    return;
                }
            }

            // finding differences to a previous spell check process
            if (oldSpellResult) {
                // do nothing, if the spell results have not changed
                firstChangedPos = sameSpellResults(spellResult, oldSpellResult);
                if (firstChangedPos === -1) {
                    return;
                }
            }

            // applying the results received from the server
            // -> the array with textSpans is still valid, because 'p_spelled' is still set at the paragraph
            applySpellcheckResults(spellResult, textSpans, firstChangedPos);

            // saving spell results at the paragraph to compare it with the next spell check results
            paragraph.data(SpellChecker.SPELL_RESULT_CACHE, spellResult);

            // invalidate an existing cached text point
            selection.setInsertTextPoint(null);

            // restore the selection, but not if a drawing or a table cell is selected (Task 26214)
            if (app.getView().hasAppFocus() && model.isTextOnlySelected()) {
                selection.restoreBrowserSelection({ preserveFocus: true });
            }

        });

        var wordIterator = function (paraSpans, textSpans) {

            // - provide/get a list of all languages that are not supposed to be notified about theirs missing spell-check(er) support.
            // - a user though, is able of editing this very list from within one's document settings.
            spellcheckLanguageBlacklist = SpellcheckNotification.getLanguageBlacklist();

            return function (node) {

                // iterating over all text spans
                DOM.iterateTextSpans(node, function (span) {
                    var charAttributes = null;

                    if (!DOM.isListLabelNode(span.parentNode)) {

                        charAttributes = getCharacterStyles().getElementAttributes(span);
                        if (charAttributes.character.url === '') {

                            // only supported language spans will be send to the server
                            if (_.contains(supportedLanguages, charAttributes.character.language)) {
                                textSpans.push(span);
                                paraSpans.push({ word: span.firstChild.nodeValue, locale: charAttributes.character.language });
                            } else {
                                self.clearSpellErrorAttributes(span); // remove spell attributes - if there are any
                                addUnsupportedLanguageForNotification(charAttributes.character.language);
                            }

                        }
                    }
                });

            };
        };

        /**
         * The worker function, that checks the paragraphs, generates server calls and
         * applies the response from the server to the paragraphs.
         */
        function implCheckSpelling() {

            var // the vertical offset of the specific node
                topOffset = 0,
                // the maximum offset for paragraphs that will be spell checked (one and a half screen downwards)
                maxOffset = 1.5 * (scrollNode.height() + scrollNode.offset().top),
                // the minimum offset for paragraphs that will be spell checked (a half screen upwards)
                minOffset = -0.3 * maxOffset,
                // container of the paragraph if presentation only the paragraphs of the active slide
                paragraphContainer = model.useSlideMode() ? model.getSlideById(model.getActiveSlideId()) : model.getNode(),
                // iterating over all paragraphs, that are not marked with class 'p_spelled'. If the paragraph cache was filled, use it
                allParagraphs = paragraphContainer === null ? [] : paragraphCache.length > 0 ? paragraphCache : paragraphContainer.find(DOM.PARAGRAPH_NODE_SELECTOR).not('.p_spelled');

            // Slicing, so that there are not too many parallel server calls
            // -> it is useful to profile the sliced function using 'Utils.profileMethod' in following performance tests.
            var promise = model.iterateArraySliced(allParagraphs, function (paragraph) {

                var // all text spans in the paragraph, as array
                    textSpans,
                    // all spans of the paragraph - each span is a word/language object
                    paraSpans,
                    // requestID
                    requestID;
                if ($(paragraph).hasClass(DOM.NO_SPELLCHECK_CLASSNAME) || $(paragraph).hasClass(DOM.PARAGRAPH_NODE_LIST_EMPTY_CLASS)) {
                    removeAllSpellErrorsInParagraph(paragraph);
                } else if (!$(paragraph).hasClass('p_spelled')) {

                    // calculating if this paragraph is in or near the visible region (for performance reasons)
                    topOffset = Utils.round($(paragraph).offset().top, 1);

                    // TODO: This check is not sufficient for long paragraphs
                    if (topOffset < maxOffset && minOffset < topOffset) {

                        textSpans = [];
                        paraSpans = [];
                        // marking paragraph immediately as spell checked, so that directly following changes can be recognized
                        $(paragraph).addClass('p_spelled');

                        // collect all non-empty text spans in the paragraph
                        Position.iterateParagraphChildNodes(paragraph, wordIterator(paraSpans, textSpans), undefined, { allNodes: true });

                        // preparing server call for the paragraph, if text spans were found
                        if (paraSpans.length > 0) {
                            requestID = requestIDCounter++;
                            $(paragraph).data(SpellChecker.SPELL_REQUEST_ID, requestID);
                            Utils.logSelenium('spellchecker/apicall', 'pending');
                            self.sendRequest('spellchecker', { action: 'spellparagraph', paragraph: JSON.stringify(paraSpans) }, { method: 'POST' })
                                .done(function (data) {
                                    var spellResult;
                                    if ($(paragraph).data(SpellChecker.SPELL_REQUEST_ID) !== requestID) {
                                        return false;
                                    }
                                    // remove unused data from paragraph
                                    $(paragraph).removeData(SpellChecker.SPELL_REQUEST_ID);

                                    spellResult = Utils.getArrayOption(data, 'spellResult', []);
                                    if (spellResult.length === 0) {
                                        if ($(paragraph).data(SpellChecker.SPELL_RESULT_CACHE)) {
                                            // no errors now, but some errors before
                                            removeAllSpellErrorsInParagraph(paragraph);
                                            self.clearSpellResultCache(paragraph);
                                        }
                                    } else {
                                        spellResultHandler(spellResult, textSpans, paragraph);
                                    }
                                    Utils.logSelenium('spellchecker/apicall', 'resolved');
                                })
                                .fail(function () {
                                    Utils.logSelenium('spellchecker/apicall', 'rejected');
                                });
                        } else {
                            removeAllSpellErrorsInParagraph(paragraph);
                        }

                        // - trigger dialog if unsupported languages without a notification exist.
                        // - always prevent dialog if app runs within an automated test environment.
                        if (!Config.AUTOTEST && !_.isEmpty(unsupportedLanguageWithoutNotification)) {
                            showUnsupportedLanguageNotificationDebounced();
                        }
                    }
                }
            }, 'Spellchecker.implCheckSpelling');

            promise.always(function () {
                paragraphCache = $();
            });
        }

        /**
         * Initialization of spell checker after document is loaded successfully.
         */
        function documentLoaded() {
            // setting scroll node and register listener for 'scroll' events
            scrollNode = app.getView().getContentRootNode();

            if (self.isOnlineSpelling()) {
                self.setOnlineSpelling(true, true);
            }
            SpellcheckNotification.registerEditSettingsAction(app);
        }

        /**
         * Called if the 'save' UserDictionary event occurred.
         * @param {jQuery.Event} event
         * @param {String} newWords the which are added to the dictionary
         * @param {boolean} removedWords the words which are removed from the dictionary
         */
        function onSaveUserDictionary(event, newWords, removedWords) {
            _.each(newWords, function (word) {
                if (!self.isIgnoredWord(word)) {
                    ignoredMisspelledWords.push(word.toLowerCase());
                    changeParagraphSpelledAttributeForIgnoreWords(word, false);

                    userDictionaryChanges = true;
                }
            });
            _.each(removedWords, function (word) {
                if (self.isIgnoredWord(word)) {
                    ignoredMisspelledWords = _.without(ignoredMisspelledWords, word);
                    changeParagraphSpelledAttributeForIgnoreWords(word, true);

                    userDictionaryChanges = true;
                }

            });
        }

        /**
         * Start adding or removing ignored words.
         */
        function implStartAddOrRemoveIgnoreWords() {
            if (self.isOnlineSpelling() && app.isEditable() && self.isImportFinished()) {
                while (ignoreWordActions.length > 0) {
                    implAddWordToIgnoreSpellcheck(ignoreWordActions.shift());
                }
            }
        }

        // A debounced function for starting adding or removing ignored words
        var implStartAddOrRemoveIgnoreWordsDebounced = this.createDebouncedMethod('Spellchecker.implStartAddOrRemoveIgnoreWordsDebounced', null, implStartAddOrRemoveIgnoreWords, { delay: 500 });

        /**
         * Add or remove a word to/from the ignore word list.
         * ignoreWordAction.node or ignoreWordAction.start/ignoreWordAction.end are optional BUT at least one must be set.
         * The 'spellerror' class of selected word (ignoreWordAction.node or ignoreWordAction.start/ignoreWordAction.end) will be added or removed synchronously,
         * the other word occurrence will be changed asynchronous.
         * @param {Object} ignoreWordAction The action for add or remove a ignore word.
         * @param {String} ignoreWordAction.word The word for add or remove.
         * @param {Boolean} [ignoreWordAction.remove=false] If true remove the word from the igonre words from spellcheck list.
         * @param {Node[]} [ignoreWordAction.nodes] The nodes of the selected word. It is set if the function is called from the ContextMenu.
         * @param {Array<Number>} [ignoreWordAction.start] The logical start index of the selected word. It is set if the function is called from Undo/Redo
         * @param {Array<Number>} [ignoreWordAction.end] The logical end index of the selected word. It is set if the function is called from Undo/Redo
         * @param {String} ignoreWordAction.target the target id of the word element(s)
         */
        function implAddWordToIgnoreSpellcheck(ignoreWordAction) {

            var remove = Utils.getBooleanOption(ignoreWordAction, 'remove', false),
                word = ignoreWordAction.word.trim(),
                paragraph,
                startNode,
                userDictionary = Utils.getBooleanOption(ignoreWordAction, 'userDictionary', false);

            if (!word || (!ignoreWordAction.nodes && !ignoreWordAction.start && !ignoreWordAction.end)) {
                return;
            }

            if (remove && self.isIgnoredWord(word)) {
                ignoredMisspelledWords = _.without(ignoredMisspelledWords, word);
                if (userDictionary) {
                    UserDictionary.removeWord(word, true);
                }
            } else if (!remove && !self.isIgnoredWord(word)) {
                ignoredMisspelledWords.push(word);
                if (userDictionary) {
                    UserDictionary.addWord(word, true);
                }
            } else {
                return;
            }

            if (ignoreWordAction.nodes) {
                _.each(ignoreWordAction.nodes, function (node) {

                    $(node).toggleClass(DOM.SPELLERRORNODE_CLASS, remove);

                    if (!remove) {
                        self.clearSpellErrorAttributes(node);
                    }
                });
            } else {

                startNode = Position.getDOMPosition(model.getRootNode(ignoreWordAction.target), ignoreWordAction.start, true);

                if (startNode) {

                    paragraph = startNode.node.parentNode;

                    self.resetDirectly(paragraph);
                    self.clearSpellResultCache(paragraph);

                    Position.iterateParagraphChildNodes(paragraph, function (node) {
                        $(node).toggleClass(DOM.SPELLERRORNODE_CLASS, remove);
                        if (!remove) {
                            self.clearSpellErrorAttributes(node);
                        }
                    }, undefined, {
                        allNodes: false,
                        start: _(ignoreWordAction.start).last(),
                        end: _(ignoreWordAction.end).last(),
                        split: true
                    });

                }
            }
            changeParagraphSpelledAttributeForIgnoreWords(word, remove);
        }

        /**
         * Remove the p_spelled class from paragraphs which includes the given word to test it with the spellchecker again.
         * @param {String} word the word to check
         * @param {Boolean} remove true if the word is removed from the ignore list, otherwise it is added to the ignore list.
         */
        function changeParagraphSpelledAttributeForIgnoreWords(ignoreWord, remove) {
            var word = ignoreWord.toLowerCase(),
                paragraphs = model.getNode().find(DOM.PARAGRAPH_NODE_SELECTOR + '.p_spelled');

            var promise = model.iterateArraySliced(paragraphs, function (paragraph) {
                if ((remove || self.hasSpellResultCache(paragraph)) && $(paragraph).text().toLowerCase().indexOf(word) >= 0) {
                    self.resetDirectly(paragraph);
                    self.clearSpellResultCache(paragraph);
                }
            }, 'Spellchecker.implAddWordToIgnoreSpellcheck');

            promise.always(function () {
                paragraphCache = $();
                implStartOnlineSpellingDebounced();
            });
        }

        /**
         * Add a ignore word add or remove action to the ignore words stack.
         * action.start and action.end or action.nodes must be set.
         * @param {Object} action The action for add or remove a ignore word.
         * @param {String} action.word The word for add or remove.
         * @param {Boolean} [action.remove=false] If true remove the word from the igonre words from spellcheck list.
         * @param {Node[]} [action.nodes] The nodes of the selected word. It is set if the function is called from the ContextMenu.
         * @param {Array<Number>} [action.start] The logical start index of the selected word. It is set if the function is called from Undo/Redo
         * @param {Array<Number>} [action.end] The logical end index of the selected word. It is set if the function is called from Undo/Redo
         * @param {String} action.target the target id of the word element(s)
         * @param {Boolean} [immediate = false] execute the action immediate if true, otherwise debounced.
         */
        function addIgnoreWordAction(action, immediate) {
            ignoreWordActions.push(action);
            if (immediate) {
                implStartAddOrRemoveIgnoreWords();
            } else {
                implStartAddOrRemoveIgnoreWordsDebounced();
            }
        }

        /**
         * Add the language to the language list without a notification. This languages will be shown in a notification.
         * @param {String} lang the language local e.g. en-US
         */
        function addUnsupportedLanguageForNotification(lang) {
            if (
                !_.contains(unsupportedLanguages, lang) &&
                !_.contains(unsupportedLanguageWithoutNotification, lang) &&
                !_.contains(spellcheckLanguageBlacklist, lang)
            ) {
                unsupportedLanguageWithoutNotification.push(lang);
            }
        }

        /**
         * Shows a notification with the unsupported but used languages.
         * The dialog appears only once for each language.
         */
        function showUnsupportedLanguageNotification() {
            var
                msg,
                langText,

                addComma = false,
                showDialog = false,

                // see: https://bugs.open-xchange.com/show_bug.cgi?id=50880
                //
                // - does prevent raising the dialog right after having opened an entirely empty page.
                // - does work with guarding flag since `model.isEmptyPage()` is quite expensive and will be invoked otherwise every second.
                isPageConsideredEmpty = (hasPageAlwaysBeenEmpty && (hasPageAlwaysBeenEmpty = model.isEmptyPage())),

                isLanguageNotification = !_.isEmpty(unsupportedLanguageWithoutNotification);

            if (!isPageConsideredEmpty && isLanguageNotification) {
                msg = gt('This document contains text in languages which couldn\'t be proofed: ');

                _.each(unsupportedLanguageWithoutNotification, function (lang) {

                    langText = LocaleData.getLanguageName(lang.replace('-', '_'));

                    if (langText) {
                        if (addComma) {
                            msg += ', ';
                        }
                        msg += langText;

                        addComma = true;
                        showDialog = true;
                    }
                    unsupportedLanguages.push(lang);
                });

                // immediately save any newly detected language into the spell-check language whitelist.
                SpellcheckNotification.saveToLanguageWhitelist(unsupportedLanguages);

                if (showDialog) {
                    app.getView().yell({
                        message: msg,
                        action: {
                            itemKey: 'document/settings/editspellchecknotification',
                            label: /*#. Button label for open the user spell-check languages dialog */gt('Set language notification.'),
                            icon: 'fa-cog'
                        }
                    });
                }
                unsupportedLanguageWithoutNotification = [];
            }
        }

        // show the Notification if unsupported languages exist
        var showUnsupportedLanguageNotificationDebounced = this.createDebouncedMethod('SpellChecker.showUnsupportedLanguageNotificationDebounced', null, showUnsupportedLanguageNotification, { delay: 1000 });

        function restoreTextSelection() {
            var selection;
            if (restoreStartPosition) {
                selection = model.getSelection();
                selection.setTextSelection(restoreStartPosition, restoreStartPosition);
                restoreStartPosition = null;
                app.getView().grabFocus();
            }
        }

        /**
         * Set or remove the event listeners
         */
        function setEventListeners() {
            if (self.isOnlineSpelling()) {
                // after receiving edit privileges it is necessary to check the document
                self.listenTo(app, 'docs:editmode:enter', implStartOnlineSpellingDebounced);
                self.listenTo(model, 'operations:success', implStartOnlineSpellingDebounced);

                self.listenTo(scrollNode, 'scroll', implStartOnlineSpellingDebounced);

                // listen if the user update the user dictionary
                self.listenTo(UserDictionary, 'save', onSaveUserDictionary);

                // listen if the app will be visible. It's needed if e.g. the user change the dictionary in the settings and go back to the document,
                // now the spellchecker must start to check the text with the updated dictionary.
                self.listenTo(app.getWindow(), 'show', appVisibleHandler);
            } else {
                self.stopListeningTo(app, 'docs:editmode:enter', implStartOnlineSpellingDebounced);
                self.stopListeningTo(model, 'operations:success', implStartOnlineSpellingDebounced);
                self.stopListeningTo(scrollNode, 'scroll', implStartOnlineSpellingDebounced);
                self.stopListeningTo(UserDictionary, 'save', onSaveUserDictionary);
                self.stopListeningTo(app.getWindow(), 'show', appVisibleHandler);
            }
        }

        /**
         * After the event 'document:reloaded' was fired by the model, some cached nodes and caches
         * need to be invalidated.
         */
        function handleDocumentReload() {
            // invalidating paragraph cache, after the document was reloaded
            paragraphCache = $();
        }

        // public methods -----------------------------------------------------

        this.getSpellErrorWord = function () {

            var // the selection object
                selection = model.getSelection(),
                // the documents character styles
                characterStyles = getCharacterStyles(),
                // the returning object containing the spellchecker infos
                result = { url: null, spellError: false, word: '', replacements: [] },
                // the selected text span
                span = null,
                // an object containing an expanded word selection
                newSelection = null,
                // the locale of the selected text span
                locale = null,
                // the character attributes of the selected text span
                attributes = null,
                // the logical start position of the selection
                startPosition = null,
                // the node info object at the logical start position
                obj = null,
                // whether the logical start position is at the beginning of the paragraph
                isParaStart = false,
                // Chrome on osx makes a text-range-selection on right mouseclick
                chromeOnOSX = (_.browser.MacOS && _.browser.Chrome),
                // the logical start position of the paragraph
                paraPos;

            if (!selection.hasRange() || chromeOnOSX) {
                // find out a possible URL set for the current position
                startPosition = selection.getStartPosition();
                paraPos = _.initial(_.clone(startPosition));
                // on this special case, we need to count up the start-position. Otherwise,
                // the startposition of the selection breaks the detection of wrong spelled words
                if (chromeOnOSX) {
                    startPosition = Position.increaseLastIndex(startPosition);
                }
                obj = Position.getDOMPosition(model.getCurrentRootNode(), startPosition);
                isParaStart = startPosition[startPosition.length - 1] === 0;

                if (!isParaStart && obj && obj.node && DOM.isTextSpan(obj.node.parentNode)) {

                    attributes = characterStyles.getElementAttributes(obj.node.parentNode).character;

                    if (attributes.url.length > 0) {
                        // Now we have to check some edge cases to prevent to show
                        // the popup for a paragraph which contains only an empty span
                        // having set the url attribute.
                        span = $(obj.node.parentNode);
                        if ((span.text().length > 0) || (span.next().length > 0) || DOM.isTextSpan(span.next())) {
                            result.url = attributes.url;
                        }
                    } else if ($(obj.node.parentNode).hasClass(DOM.SPELLERRORNODE_CLASS) === true) {
                        // Now we have to check some edge cases to prevent to show
                        // the popup for a paragraph which contains only an empty span
                        // having set the url attribute.
                        span = $(obj.node.parentNode);
                        if ((span.text().length > 0) || (span.prev().length > 0) || DOM.isTextSpan(span.next())) {
                            result.spellError = true;
                        }

                        newSelection = Position.getWordSelection(selection.getEnclosingParagraph(), startPosition[startPosition.length - 1], null, { returnAllTextSpans: true, returnTextSpan: true });

                        if (newSelection) {
                            locale = attributes.language.replace(/-/, '_');
                            result.word = newSelection.text;
                            result.start = _.clone(paraPos).concat(newSelection.start);
                            result.end = _.clone(paraPos).concat(newSelection.end - 1);

                            result.replacements = getSpellReplacements(result.word, locale);
                            result.node = newSelection.node;
                            result.nodes = newSelection.nodes;
                        }
                    }
                }
            }

            return result;
        };

        this.replaceWord = function (paraPosition, startEndIndex, replacement) {
            restoreTextSelection();
            var // the selection object
                selection = model.getSelection(),
                // the undo manager
                undoManager = model.getUndoManager(),
                startPosition = _.clone(paraPosition),
                endPosition = _.clone(paraPosition),
                newEndPosition = _.clone(paraPosition);

            startPosition[endPosition.length - 1] = startEndIndex.start;
            endPosition[endPosition.length - 1] = startEndIndex.end;

            undoManager.enterUndoGroup(function () {
                selection.setTextSelection(startPosition, endPosition);
                model.deleteSelected();  // this selection can never contain unrecoverable content -> no deferred required
                model.insertText(replacement, selection.getStartPosition(), model.getPreselectedAttributes()); // restoring start position is required!

                selection = model.getSelection();
                paraPosition = selection.getStartPosition();
                startEndIndex = Position.getWordSelection(selection.getEnclosingParagraph(), paraPosition[paraPosition.length - 1]);
                newEndPosition[endPosition.length - 1] = startEndIndex.end;
                selection.setTextSelection(newEndPosition);
            });
        };

        /**
         *
         */
        this.reset = function (attributes, paragraph) {
            //reset spelling status on language changes
            if (attributes.character && attributes.character.language) {
                this.resetDirectly(paragraph);
            }
        };

        /**
         *
         */
        this.resetClosest = function (newAttributes, element) {
            // invalidate spell result if language attribute changes
            if (newAttributes.character && newAttributes.character.language) {
                this.resetDirectly($(element).closest(DOM.PARAGRAPH_NODE_SELECTOR));
            }
        };
        /**
         * Removing the class marker for spell-checked paragraphs
         *
         * @param {Node|jQuery} paragraph
         *  The DOM node from which the class shall be removed.
         */
        this.resetDirectly = function (paragraph) {
            // reset to 'not spelled'
            $(paragraph).removeClass('p_spelled');
            // ... and caching paragraph for performance reasons
            if (!model.useSlideMode()) {
                paragraphCache = paragraphCache.add(paragraph);
            }
            $(paragraph).removeData(SpellChecker.SPELL_REQUEST_ID);
        };

        /**
         * Removing classes & runtimeclasses, multi selection possible.
         * Supported are div elements and text spans.
         *
         * @param {jQuery} node
         *  The jQuerified DOM node from which the class shall be removed.
         */
        this.clearSpellcheckHighlighting = function (node) {
            // removing classes, multi selection possible -> no 'else if'
            if (node.is('span.' + DOM.SPELLERRORNODE_CLASS)) {
                // remove spellcheck attribute and class
                self.clearSpellErrorAttributes(node);
            } else if (node.is('div')) {
                // remove runtime classes and cached result data
                node.removeClass('selected p_spelled');
                self.clearSpellResultCache(node);
            }
        };

        /**
         * Removes the character attribute 'spellerror', so that
         * the dotted underline of a text span disappears.
         *
         * @param {Node|jQuery} span
         *  The DOM node from which the attribute shall be removed.
         */
        this.clearSpellErrorAttributes = function (span) {

            var // the character styles of the document
                characterStyles = getCharacterStyles();

            // remove spell attributes - if there are any
            characterStyles.setElementAttributes(span, { character: { spellerror: null } }, { special: true });
        };

        /**
         * Toggle automatic spell checking in the document.
         */
        this.setOnlineSpelling = function (state, noConfig) {
            if (!spellingSupported) { return; }
            var rootNode = model.getNode(),
                stateChange = state !== self.isOnlineSpelling();
            if (stateChange || noConfig === true) {
                if (noConfig !== true) {
                    // there are no app specific spell-check user settings anymore due to DOCS-630
                    SpellChecking.setOnlineSpelling(state);
                }
                rootNode.toggleClass('spellerror-visible', state);

                setEventListeners();

                if (self.isOnlineSpelling()) {
                    // deleting the paragraph cache, so that all paragraphs are checked
                    paragraphCache = $();
                    // start timeout handler
                    implStartOnlineSpellingDebounced();
                } else {
                    var allParagraphs = model.getNode().find(DOM.PARAGRAPH_NODE_SELECTOR + '.p_spelled');
                    model.iterateArraySliced(allParagraphs, function (paragraph) {
                        removeAllSpellErrorsInParagraph(paragraph);
                        $(paragraph).removeClass('p_spelled');
                        self.clearSpellResultCache(paragraph);
                    }, 'Spellchecker.setOnlineSpelling');
                    paragraphCache = $();
                }
            }
        };

        /**
         * Checking, if spell checking is enabled.
         *
         * @returns {Boolean}
         *  Whether spellcheck in config and in user-hud is enabled
         */
        this.isOnlineSpelling = function () {
            return spellingSupported && SpellChecking.isOnlineSpelling();
        };

        /**
         * Checking, whether a specified paragraph has cached spell results
         *
         * @param {Node|jQuery} paragraph
         *  The paragraph node, that is checked for existing spell results.
         *
         * @returns {Boolean}
         *  Whether a specified paragraph has cached spell results
         */
        this.hasSpellResultCache = function (paragraph) {
            return $(paragraph).data(SpellChecker.SPELL_RESULT_CACHE);
        };

        /**
         * Clearing the spell result cache at a specified paragraph
         *
         * @param {Node|jQuery} paragraph
         *  The paragraph node, whose cached spell results will be removed.
         */
        this.clearSpellResultCache = function (paragraph) {
            $(paragraph).removeData(SpellChecker.SPELL_RESULT_CACHE);
        };

        /**
         * Add the word to the list of words which are ignored by the spellchecker.
         * ignoreWordAction.start and ignoreWordAction.end or ignoreWordAction.nodes must be set.
         * To add the word to the dictionary, the "igonreWordAction.userDictionary" option can be
         * set to true, to not only save the word temporarly.
         * @param {Object} ignoreWordAction The action for add or remove a ignore word.
         * @param {String} ignoreWordAction.word The word for add or remove.
         * @param {Node[]} [ignoreWordAction.nodes] The nodes of the selected word. It is set if the function is called from the ContextMenu.
         * @param {Array<Number>} [ignoreWordAction.start] The logical start index of the selected word. It is set if the function is called from Undo/Redo
         * @param {Array<Number>} [ignoreWordAction.end] The logical end index of the selected word. It is set if the function is called from Undo/Redo
         * @param {String} ignoreWordAction.target the target id of the word element(s)
         * @param {Boolean} [igonreWordAction.userDictionary = false] if the word is for the user dictionary, otherwise ignore the word temporarly.
         */
        this.addWordToIgnoreSpellcheck = function (ignoreWordAction) {
            var word,
                target,
                userDictionary;
            restoreTextSelection();
            if (ignoreWordAction) {
                word = Utils.getStringOption(ignoreWordAction, 'word', '');
                word = word.toLowerCase().trim();
                target = Utils.getStringOption(ignoreWordAction, 'target', null);
                userDictionary = Utils.getBooleanOption(ignoreWordAction, 'userDictionary', false);

                if (word && !self.isIgnoredWord(word) && target !== null && ignoreWordAction.nodes && ignoreWordAction.start && ignoreWordAction.end) {

                    addIgnoreWordAction({ word: word, nodes: ignoreWordAction.nodes, target: target, userDictionary: userDictionary }, true);

                    undoManager.addUndo(function () {
                        addIgnoreWordAction({ word: word, remove: true, start: ignoreWordAction.start, end: ignoreWordAction.end, target: target, userDictionary: userDictionary });
                    }, function () {
                        addIgnoreWordAction({ word: word, remove: false, start: ignoreWordAction.start, end: ignoreWordAction.end, target: target, userDictionary: userDictionary });
                    });
                }
            }
        };

        /**
         * Check if the word is in the list of ignored misspelled words.
         * @param {String} word the word to check
         * @returns {Boolean} true if the word is on the list, otherwise false
         */
        this.isIgnoredWord = function (word) {
            return _.contains(ignoredMisspelledWords, word.toLowerCase());
        };

        /**
         * Test if the node contains the given class.
         * @param {Node} node for test.
         * @returns {Boolean} true if the node contains the class, otherwise false.
         */
        this.hasSpellerrorClass = function (node) {
            return $(node).hasClass(DOM.SPELLERRORNODE_CLASS);
        };

        this.selectMisspelledWord = function (contextmenu) {
            var selection = model.getSelection(),
                paraPosition = selection.getStartPosition(),
                startEndIndex = Position.getWordSelection(selection.getEnclosingParagraph(), paraPosition[paraPosition.length - 1]),
                startPosition = _.clone(paraPosition),
                endPosition = _.clone(paraPosition);

            restoreStartPosition = _.clone(selection.getStartPosition());

            startPosition[endPosition.length - 1] = startEndIndex.start;
            endPosition[endPosition.length - 1] = startEndIndex.end;

            selection.setTextSelection(startPosition, endPosition);

            contextmenu.one('popup:beforehide', function () {
                restoreTextSelection();
            });
        };

        // initialization -----------------------------------------------------

        Config.getLocalesWithDictionary().done(function (supportedLocales) {
            // add the supported languages to list to show notifications for used but unsupported languages
            supportedLanguages = _.map(supportedLocales, function (lang) {
                return lang.replace('_', '-');
            });

            self.waitForImportSuccess(documentLoaded);
        });

        app.onInit(function () {
            // updating the model after the document was reloaded (after cancelling long running actions)
            self.listenTo(model, 'document:reloaded', handleDocumentReload);
        });

    } // class SpellChecker

    // constants --------------------------------------------------------------

    /**
     * The name of the jQuery data key that is used as connection between a comment thread
     * and its bubble node.
     */
    SpellChecker.SPELL_RESULT_CACHE = 'spellresult';

    /**
     * The name of the jQuery data key that is used to add a request sendID at the paragraph.
     * After changing the paragraph while the request is not complete the id will be
     * removed or a second request change the id then is the request is not valid. If another
     * request is started for the paragraph the will be changed and the requests before are
     * no more valid.
     */
    SpellChecker.SPELL_REQUEST_ID = '';

    // exports ================================================================

    // derive this class from class TriggerObject
    return TriggerObject.extend({ constructor: SpellChecker });

});
