/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Ingo Schmidt-Rosbiegal <ingo.schmidt-rosbiegal@open-xchange.com>
 */

define('io.ox/office/text/format/tablecellstyles', [
    'io.ox/office/tk/utils',
    'io.ox/office/editframework/utils/color',
    'io.ox/office/editframework/utils/border',
    'io.ox/office/editframework/model/stylecollection',
    'io.ox/office/textframework/utils/dom'
], function (Utils, Color, Border, StyleCollection, DOM) {

    'use strict';

    var // definitions for table cell attributes
        DEFINITIONS = {

            /**
             * The number of grid columns spanned by the table cell.
             */
            gridSpan: {
                def: 1,
                scope: 'element'
            },

            /**
             * Fill color of the table cell.
             */
            fillColor: {
                def: Color.AUTO
            },

            /**
             * Style, width and color of the left table cell border.
             */
            borderLeft: {
                def: Border.NONE
            },

            /**
             * Style, width and color of the right table cell border.
             */
            borderRight: {
                def: Border.NONE
            },

            /**
             * Style, width and color of the top table cell border.
             */
            borderTop: {
                def: Border.NONE
            },

            /**
             * Style, width and color of the bottom table cell border.
             */
            borderBottom: {
                def: Border.NONE
            },

            /**
             * Inner horizontal table cell borders, used in table style sheets
             * to format inner borders of specific table areas (first/last
             * column, inner vertical bands, ...).
             */
            borderInsideHor: {
                def: Border.NONE,
                scope: 'style'
            },

            /**
             * Inner vertical table cell borders, used in table style sheets to
             * format inner borders of specific table areas (first/last row,
             * inner horizontal bands, ...).
             */
            borderInsideVert: {
                def: Border.NONE,
                scope: 'style'
            },

            /**
             * Top padding of the table cell
             */
            paddingTop: {
                def: 0
            },

            /**
             * Bottom padding of the table cell
             */
            paddingBottom: {
                def: 0
            },

            /**
             * Left padding of the table cell
             */
            paddingLeft: {
                def: 0
            },

            /**
             * Right padding of the table cell
             */
            paddingRight: {
                def: 0
            },

            /**
             * Vertical merge of table cells.
             */
            mergeVert: {
                def: 'noMerge'
            }
        },

        // parent families with parent element resolver functions
        PARENT_RESOLVERS = {
            table: function (cell) { return cell.closest(DOM.TABLE_NODE_SELECTOR); }
        };

    // private global functions ===============================================

    /**
     * Converts the passed padding from 1/100 of millimeters to pixels.
     *
     * @param {Number} padding
     *  The padding, in 1/100 of millimeters.
     *
     * @returns {Number}
     *  The padding, in pixels.
     */
    function getPadding(padding) {
        // minimum 2px, required for cell resizing
        return Math.max(2, Utils.convertHmmToLength(padding, 'px', 1));
    }

    // class TableCellStyles ==================================================

    /**
     * Contains the style sheets for table cell formatting attributes. The CSS
     * formatting will be read from and written to <th> and <td> elements.
     *
     * @constructor
     *
     * @extends StyleCollection
     *
     * @param {TextModel} docModel
     *  The text document model containing instance.
     */
    var TableCellStyles = StyleCollection.extend({ constructor: function (docModel) {

        // base constructor ---------------------------------------------------

        StyleCollection.call(this, docModel, 'cell', {
            styleSheetSupport: false,
            families: 'paragraph character changes',
            parentResolvers: PARENT_RESOLVERS
        });

        // private methods ----------------------------------------------------

        /**
         * Will be called for every table cell for table preview generation.
         * Formats the table cell according to the passed attributes.
         *
         * @param {jQuery} cell
         *  The table cell element whose table attributes need to be formatted, as
         *  jQuery object.
         *
         * @param {Object} mergedAttributes
         *  A map of attribute value maps (name/value pairs), keyed by
         *  attribute family, containing the effective attribute values merged
         *  from style sheets and explicit attributes.
         *
         * @param {Object} [options]
         *  Optional parameters passed to Border.getCssBorder().
         */
        function updateGenericFormatting(cell, mergedAttributes, options) {

            // the cell attributes
            var cellAttrs = mergedAttributes.cell;

            cell.css({
                backgroundColor: docModel.getCssColor(cellAttrs.fillColor, 'fill'),
                borderLeft: docModel.getCssBorder(cellAttrs.borderLeft, null, options),
                borderRight: docModel.getCssBorder(cellAttrs.borderRight, null, options),
                borderTop: docModel.getCssBorder(cellAttrs.borderTop, null, options),
                borderBottom: docModel.getCssBorder(cellAttrs.borderBottom, null, options)
            });
        }

        /**
         * Handling vertically merged cells, so that they are not visible to the user.
         *
         * @param {jQuery} cell
         *  The table cell element whose table attributes need to be formatted, as
         *  jQuery object.
         *
         * @param {Object} cellAttrs
         *  The 'cell' family attributes of the merged cells.
         */
        function handleVerticallyMergedTableCells(cell, cellAttrs) {

            var cellContentNode = DOM.getCellContentNode(cell);
            var mergeVert = cellAttrs.mergeVert;

            if (mergeVert === 'continue') {
                cellContentNode.attr('contenteditable', false);
                cellContentNode.css('visibility', 'hidden');
            }

            cell.attr('mergeVert', mergeVert); // setting marker attribute at cell
        }

        /**
         * Will be called for every table cell element whose attributes have
         * been changed. Reformats the table cell according to the passed
         * attributes.
         *
         * @param {jQuery} cell
         *  The table cell element whose table attributes have been changed, as
         *  jQuery object.
         *
         * @param {Object} mergedAttributes
         *  A map of attribute value maps (name/value pairs), keyed by
         *  attribute family, containing the effective attribute values merged
         *  from style sheets and explicit attributes.
         */
        function updateTableCellFormatting(cell, mergedAttributes) {

            // the paragraph style sheet container
            var paragraphStyles = docModel.getStyleCollection('paragraph');
            // the cell attributes
            var cellAttrs = mergedAttributes.cell;

            // horizontal span size
            cell.attr('colspan', cellAttrs.gridSpan);

            // Hiding vertically merged cells to avoid impression of data loss (56291)
            handleVerticallyMergedTableCells(cell, cellAttrs);

            // update CSS attributes
            updateGenericFormatting(cell, mergedAttributes);
            cell.css({
                paddingTop: getPadding(cellAttrs.paddingTop),
                paddingBottom: getPadding(cellAttrs.paddingBottom),
                paddingLeft: getPadding(cellAttrs.paddingLeft),
                paddingRight: getPadding(cellAttrs.paddingRight)
            });

            // Center cell content vertically. This solution is not dynamic
            // -> changing row height or inserting text destroys center position.
            // var cellContentNode = DOM.getCellContentNode(cell);
            // cellContentNode.css('padding-top', cell.height() / 2 - cellContentNode.height() / 2);

            // update all paragraphs in the table cell
            Utils.iterateSelectedDescendantNodes(DOM.getCellContentNode(cell), DOM.PARAGRAPH_NODE_SELECTOR, function (paragraph) {
                paragraphStyles.updateElementFormatting(paragraph, { baseAttributes: mergedAttributes });
            }, undefined, { children: true });

            // change track attribute handling
            docModel.getChangeTrack().updateChangeTrackAttributes(cell, mergedAttributes);
        }

        /**
         * Will be called for table cells used as preview elements in the GUI.
         *
         * @param {jQuery} cell
         *  The preview table cell node, as jQuery object.
         *
         * @param {Object} mergedAttributes
         *  A complete attribute set, containing the effective attribute values
         *  merged from style sheets and explicit attributes.
         */
        function updatePreviewFormatting(cell, mergedAttributes) {
            updateGenericFormatting(cell, mergedAttributes, { preview: true });
        }

        // initialization -----------------------------------------------------

        // register the attribute definitions for the style family
        docModel.registerAttributes('cell', DEFINITIONS);

        // register the formatting handlers for DOM elements
        this.registerFormatHandler(updateTableCellFormatting);
        this.registerPreviewHandler(updatePreviewFormatting);

        // destroy all class members on destruction
        this.registerDestructor(function () {
            docModel = null;
        });

    } }); // class TableCellStyles

    // exports ================================================================

    return TableCellStyles;

});
