/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Oliver Specht <oliver.specht@open-xchange.com>
 */

define('io.ox/office/text/format/listcollection', [
    'io.ox/office/baseframework/model/modelobject',
    'io.ox/office/textframework/utils/operations',
    'io.ox/office/textframework/utils/listutils',
    'io.ox/office/textframework/utils/textutils',
    'io.ox/office/textframework/utils/position',
    'io.ox/office/textframework/utils/dom',
    'gettext!io.ox/office/text/main'
], function (ModelObject, Operations, ListUtils, TextUtils, Position, DOM, gt) {

    'use strict';

    // the default bullet list definition with different bullets per list level
    var DEFAULT_BULLET_LIST_DEFINITION = {
        listLevel0: { textAlign: 'left', indentLeft: 1270,     numberFormat: 'bullet', listStartValue: 1, fontName: 'Symbol', levelText: '\uf0b7', indentFirstLine: -635 },
        listLevel1: { textAlign: 'left', indentLeft: 2 * 1270, numberFormat: 'bullet', listStartValue: 1, fontName: 'Times New Roman', levelText: '\u25CB', indentFirstLine: -635 },
        listLevel2: { textAlign: 'left', indentLeft: 3 * 1270, numberFormat: 'bullet', listStartValue: 1, fontName: 'Times New Roman', levelText: '\u25A0', indentFirstLine: -635 },
        listLevel3: { textAlign: 'left', indentLeft: 4 * 1270, numberFormat: 'bullet', listStartValue: 1, fontName: 'Symbol', levelText: '\uf0b7', indentFirstLine: -635 },
        listLevel4: { textAlign: 'left', indentLeft: 5 * 1270, numberFormat: 'bullet', listStartValue: 1, fontName: 'Times New Roman', levelText: '\u25CB', indentFirstLine: -635 },
        listLevel5: { textAlign: 'left', indentLeft: 6 * 1270, numberFormat: 'bullet', listStartValue: 1, fontName: 'Times New Roman', levelText: '\u25A0', indentFirstLine: -635 },
        listLevel6: { textAlign: 'left', indentLeft: 7 * 1270, numberFormat: 'bullet', listStartValue: 1, fontName: 'Symbol', levelText: '\uf0b7', indentFirstLine: -635 },
        listLevel7: { textAlign: 'left', indentLeft: 8 * 1270, numberFormat: 'bullet', listStartValue: 1, fontName: 'Times New Roman', levelText: '\u25CB', indentFirstLine: -635 },
        listLevel8: { textAlign: 'left', indentLeft: 9 * 1270, numberFormat: 'bullet', listStartValue: 1, fontName: 'Times New Roman', levelText: '\u25A0', indentFirstLine: -635 }
    };

    // the default numbered list definition with different number formats per list level (as used in MSO2010)
    var DEFAULT_NUMBERING_LIST_DEFINITION = {
        listLevel0: { numberFormat: 'decimal',     listStartValue: 1, indentLeft: 1270,     indentFirstLine: -635, textAlign: 'left',  levelText: '%1.' },
        listLevel1: { numberFormat: 'lowerLetter', listStartValue: 1, indentLeft: 2 * 1270, indentFirstLine: -635, textAlign: 'left',  levelText: '%2.' },
        listLevel2: { numberFormat: 'upperLetter', listStartValue: 1, indentLeft: 3 * 1270, indentFirstLine: -635, textAlign: 'right', levelText: '%3.' },
        listLevel3: { numberFormat: 'lowerRoman',  listStartValue: 1, indentLeft: 4 * 1270, indentFirstLine: -635, textAlign: 'left',  levelText: '%4.' },
        listLevel4: { numberFormat: 'upperRoman',  listStartValue: 1, indentLeft: 5 * 1270, indentFirstLine: -635, textAlign: 'left',  levelText: '%5.' },
        listLevel5: { numberFormat: 'decimal',     listStartValue: 1, indentLeft: 6 * 1270, indentFirstLine: -635, textAlign: 'right', levelText: '%6.' },
        listLevel6: { numberFormat: 'lowerLetter', listStartValue: 1, indentLeft: 7 * 1270, indentFirstLine: -635, textAlign: 'left',  levelText: '%7.' },
        listLevel7: { numberFormat: 'upperLetter', listStartValue: 1, indentLeft: 8 * 1270, indentFirstLine: -635, textAlign: 'left',  levelText: '%8.' },
        listLevel8: { numberFormat: 'lowerRoman',  listStartValue: 1, indentLeft: 9 * 1270, indentFirstLine: -635, textAlign: 'right', levelText: '%9.' }
    };

    // all predefined list styles for the GUI controls, mapped by list identifier
    var PREDEFINED_LISTSTYLES = (function () {

        /**
         * Creates and returns a style definition for a bullet list with
         * the same bullet character in all list levels.
         *
         * @param {String} listKey
         *  A unique key for the list style.
         *
         * @param {String} listLabel
         *  The button label for the list style used in GUI from controls.
         *
         * @param {Number} sortIndex
         *  The sorting index used in drop-down lists in GUI controls.
         *
         * @param {String} tooltip
         *  A tool tip for the list style used in GUI from controls.
         *
         * @param {Function} levelGenerator
         *  A callback function that generates a style definition for a single
         *  list level. Receives the list level in the first parameter.
         *
         * @returns {Object}
         *  An object with an attribute 'definition' and the string
         *  attributes 'listKey', 'listLabel', and 'tooltip'. The list
         *  definition object containing the attributes 'listLevel0' to
         *  'listLevel8' referring to list style definitions with an
         *  indentation according to the respective list level.
         */
        function createListStyle(numberFormat, listKey, listLabel, sortIndex, tooltip, levelGenerator) {
            var definition = {};
            _(9).times(function (level) {
                definition['listLevel' + level] = _.extend({
                    numberFormat: numberFormat,
                    listStartValue: 1
                }, levelGenerator(level));
            });
            return { definition: definition, format: numberFormat, listKey: listKey, listLabel: listLabel, sortIndex: sortIndex, tooltip: tooltip };
        }

        /**
         * Creates and returns a style definition for a bullet list with
         * the same bullet character in all list levels.
         *
         * @param {String} bulletKey
         *  A unique key for the bullet list style.
         *
         * @param {String} bulletCharacter
         *  The bullet character for all list levels.
         *
         * @param {Number} sortIndex
         *  The sorting index used in drop-down lists in GUI controls.
         *
         * @param {String} tooltip
         *  A tool tip for the list style used in GUI from controls.
         *
         * @returns {Object}
         *  An object with an attribute 'definition' and the string
         *  attributes 'listLabel' and 'tooltip'. The list definition
         *  object containing the attributes 'listLevel0' to 'listLevel8'
         *  referring to list style definitions with an indentation
         *  according to the respective list level.
         */
        function createBulletListStyle(bulletKey, bulletCharacter, sortIndex, tooltip) {
            return createListStyle('bullet', bulletKey, bulletCharacter, sortIndex, tooltip, function (level) {
                return {
                    indentLeft: (level + 1) * 1270,
                    indentFirstLine: -635,
                    textAlign: 'left',
                    levelText: bulletCharacter,
                    fontName: 'Times New Roman'
                };
            });
        }

        /**
         * Creates and returns a style definition for a numbered list with
         * the same numbering style in all list levels.
         *
         * @param {String} numberFormat
         *  One of the predefined number formats 'decimal', 'lowerLetter',
         *  'upperLetter', 'lowerRoman', or 'upperRoman'.
         *
         * @param {String} textBefore
         *  Fixed text inserted before each numeric list label.
         *
         * @param {String} textAfter
         *  Fixed text inserted after each numeric list label.
         *
         * @param {Number} sortIndex
         *  The sorting index used in drop-down lists in GUI controls.
         *
         * @param {String} tooltip
         *  A tool tip for the list style used in GUI from controls.
         *
         * @returns {Object}
         *  An object with an attribute 'definition' and the string
         *  attributes 'listLabel' and 'tooltip'. The list definition
         *  object containing the attributes 'listLevel0' to 'listLevel8'
         *  referring to list style definitions with an indentation
         *  according to the respective list level.
         */
        function createNumberedListStyle(numberFormat, textBefore, textAfter, sortIndex, tooltip, indentLeftVal, indentFirstLineVal, textAlignVal) {

            // default value for left indent in 1/100 mm
            var defaultIndentLeftValue = 1270;
            // default value for first line indent in 1/100 mm
            var defaultIndentFirstLineValue = -635;
            // default value for alignment of list numbers
            var defaultTextAlignValue = 'left';
            // value for left indent in 1/100 mm
            var indentLeftValue = indentLeftVal || defaultIndentLeftValue;
            // value for first line indent in 1/100 mm
            var indentFirstLineValue = indentFirstLineVal || defaultIndentFirstLineValue;
            // value for alignment of list numbers (only left supported yet)
            var textAlignValue = textAlignVal || defaultTextAlignValue;
            // unique key for the list style
            var listKey = textBefore + numberFormat + textAfter;
            // GUI label for the list style
            var listLabel = textBefore + ListUtils.formatNumber(1, numberFormat) + textAfter;

            return createListStyle(numberFormat, listKey, listLabel, sortIndex, tooltip, function (level) {
                return {
                    indentLeft: (level + 1) * indentLeftValue,
                    indentFirstLine: indentFirstLineValue,
                    textAlign: textAlignValue,
                    levelText: textBefore + '%' + (level + 1) + textAfter
                };
            });
        }

        // generate and return the array with all predefined list styles
        return {
            L20000: createBulletListStyle('filled-circle',       '\u25CF', 11, /*#. symbol in bullet lists (filled) */ gt('Filled circle')),
            L20001: createBulletListStyle('small-filled-circle', '\u2022', 12, /*#. symbol in bullet lists (filled) */ gt('Small filled circle')),
            L20002: createBulletListStyle('circle',              '\u25CB', 13, /*#. symbol in bullet lists (not filled) */ gt('Circle')),
            L20003: createBulletListStyle('small-circle',        '\u25E6', 14, /*#. symbol in bullet lists (not filled) */ gt('Small circle')),
            L20004: createBulletListStyle('filled-square',       '\u25A0', 21, /*#. symbol in bullet lists (filled) */ gt('Filled square')),
            L20005: createBulletListStyle('small-filled-square', '\u25AA', 22, /*#. symbol in bullet lists (filled) */ gt('Small filled square')),
            L20006: createBulletListStyle('square',              '\u25A1', 23, /*#. symbol in bullet lists (not filled) */ gt('Square')),
            L20007: createBulletListStyle('small-square',        '\u25AB', 24, /*#. symbol in bullet lists (not filled) */ gt('Small square')),
            L20008: createBulletListStyle('filled-diamond',      '\u2666', 31, /*#. symbol in bullet lists (filled) */ gt('Filled diamond')),
            L20009: createBulletListStyle('diamond',             '\u25CA', 32, /*#. symbol in bullet lists (not filled) */ gt('Diamond')),
            L20010: createBulletListStyle('filled-triangle',     '\u25BA', 41, /*#. symbol in bullet lists (filled) */ gt('Filled triangle')),
            L20011: createBulletListStyle('chevron',             '>',      42, /*#. symbol in bullet lists */ gt('Chevron')),
            L20012: createBulletListStyle('arrow',               '\u2192', 51, /*#. symbol in bullet lists */ gt('Arrow')),
            L20013: createBulletListStyle('dash',                '\u2013', 52, /*#. symbol in bullet lists */ gt('Dash')),
            L20014: createBulletListStyle('none',                ' ',      99, /*#. bullet lists without symbols */ gt('No symbol')),

            L30000: createNumberedListStyle('decimal',     '',  '',  11, /*#. label style in numbered lists */ gt('Decimal numbers')),
            L30001: createNumberedListStyle('decimal',     '',  '.', 12, /*#. label style in numbered lists */ gt('Decimal numbers')),
            L30002: createNumberedListStyle('decimal',     '',  ')', 13, /*#. label style in numbered lists */ gt('Decimal numbers')),
            L30003: createNumberedListStyle('decimal',     '(', ')', 14, /*#. label style in numbered lists */ gt('Decimal numbers')),
            L30010: createNumberedListStyle('lowerLetter', '',  '',  21, /*#. label style in numbered lists */ gt('Small Latin letters')),
            L30011: createNumberedListStyle('lowerLetter', '',  '.', 22, /*#. label style in numbered lists */ gt('Small Latin letters')),
            L30012: createNumberedListStyle('lowerLetter', '',  ')', 23, /*#. label style in numbered lists */ gt('Small Latin letters')),
            L30013: createNumberedListStyle('lowerLetter', '(', ')', 24, /*#. label style in numbered lists */ gt('Small Latin letters')),
            L30020: createNumberedListStyle('upperLetter', '',  '',  31, /*#. label style in numbered lists */ gt('Capital Latin letters')),
            L30021: createNumberedListStyle('upperLetter', '',  '.', 32, /*#. label style in numbered lists */ gt('Capital Latin letters')),
            L30022: createNumberedListStyle('upperLetter', '',  ')', 33, /*#. label style in numbered lists */ gt('Capital Latin letters')),
            L30023: createNumberedListStyle('upperLetter', '(', ')', 34, /*#. label style in numbered lists */ gt('Capital Latin letters')),
            L30030: createNumberedListStyle('lowerRoman',  '',  '',  41, /*#. label style in numbered lists */ gt('Small Roman numbers')),
            L30031: createNumberedListStyle('lowerRoman',  '',  '.', 42, /*#. label style in numbered lists */ gt('Small Roman numbers')),
            L30032: createNumberedListStyle('lowerRoman',  '',  ')', 43, /*#. label style in numbered lists */ gt('Small Roman numbers')),
            L30033: createNumberedListStyle('lowerRoman',  '(', ')', 44, /*#. label style in numbered lists */ gt('Small Roman numbers')),
            L30040: createNumberedListStyle('upperRoman',  '',  '',  51, /*#. label style in numbered lists */ gt('Capital Roman numbers')),
            L30041: createNumberedListStyle('upperRoman',  '',  '.', 52, /*#. label style in numbered lists */ gt('Capital Roman numbers')),
            L30042: createNumberedListStyle('upperRoman',  '',  ')', 53, /*#. label style in numbered lists */ gt('Capital Roman numbers')),
            L30043: createNumberedListStyle('upperRoman',  '(', ')', 54, /*#. label style in numbered lists */ gt('Capital Roman numbers'), 1905, -1270)
        };

    }()); // end of PREDEFINED_LISTSTYLES local scope

    // static private function ================================================

    function isLevelEqual(defaultLevel, compareLevel) {
        var ret = defaultLevel !== undefined && compareLevel !== undefined &&
        defaultLevel.numberFormat === compareLevel.numberFormat &&
        defaultLevel.indentLeft === compareLevel.indentLeft &&
        defaultLevel.indentFirstLine === compareLevel.indentFirstLine &&
        defaultLevel.textAlign === compareLevel.textAlign &&
        defaultLevel.levelText === compareLevel.levelText &&
        defaultLevel.fontName === compareLevel.fontName &&
        defaultLevel.listStartValue === compareLevel.listStartValue;
        return ret;
    }

    function isDefinitionEqual(defaultDefinition, compareDefinition) {
        return isLevelEqual(defaultDefinition.listLevel0, compareDefinition.listLevel0) &&
            isLevelEqual(defaultDefinition.listLevel1, compareDefinition.listLevel1) &&
            isLevelEqual(defaultDefinition.listLevel2, compareDefinition.listLevel2) &&
            isLevelEqual(defaultDefinition.listLevel3, compareDefinition.listLevel3) &&
            isLevelEqual(defaultDefinition.listLevel4, compareDefinition.listLevel4) &&
            isLevelEqual(defaultDefinition.listLevel5, compareDefinition.listLevel5) &&
            isLevelEqual(defaultDefinition.listLevel6, compareDefinition.listLevel6) &&
            isLevelEqual(defaultDefinition.listLevel7, compareDefinition.listLevel7) &&
            isLevelEqual(defaultDefinition.listLevel8, compareDefinition.listLevel8);
    }

    function isArrayDefinitionEqual(arrayDefinition, compareDefinition) {
        return isLevelEqual(arrayDefinition.listLevels[0], compareDefinition.listLevel0) &&
            isLevelEqual(arrayDefinition.listLevels[1], compareDefinition.listLevel1) &&
            isLevelEqual(arrayDefinition.listLevels[2], compareDefinition.listLevel2) &&
            isLevelEqual(arrayDefinition.listLevels[3], compareDefinition.listLevel3) &&
            isLevelEqual(arrayDefinition.listLevels[4], compareDefinition.listLevel4) &&
            isLevelEqual(arrayDefinition.listLevels[5], compareDefinition.listLevel5) &&
            isLevelEqual(arrayDefinition.listLevels[6], compareDefinition.listLevel6) &&
            isLevelEqual(arrayDefinition.listLevels[7], compareDefinition.listLevel7) &&
            isLevelEqual(arrayDefinition.listLevels[8], compareDefinition.listLevel8);
    }

    // class ListCollection ===================================================

    /**
     * Contains the definitions of lists.
     *
     * @constructor
     *
     * @extends ModelObject
     *
     * @param {TextModel} docModel
     *  The document model containing this instance.
     */
    function ListCollection(docModel) {

        var // list definitions
            lists = [],

            // listIds used in insertList operation
            insertedListIds = {},

            // saving the base styles id for every list style
            // -> key value pairs: listStyleId: baseStyleId
            baseStyleIDs = {},

            // saving all list styles, that share the same base style
            // -> key value pairs: baseStyleId: [listStyleId1, listStyleId2, ...]
            allListStyleIDs = {},

            // defaults
            defaultNumberingNumId,

            defaultBulletNumId,

            odfFormat = docModel.getApp().isODF();

        // base constructor ---------------------------------------------------

        ModelObject.call(this, docModel);

        // private methods ----------------------------------------------------

        /**
         * Creates the label text of the current numbered paragraph.
         * For picture numbered bullet list it returns the URI of the picture
         *
         * @param {Array<Number>} levelIndexes
         *  array of indexes of all numbering levels
         *
         * @param {Number} ilvl
         *  current indentation level
         *
         * @param {Object} listDefinition
         *  properties of the list used
         *
         * @returns {Object}
         *  text element contains label text
         *  imgsrc element contains URI of the picture used
         *  In case imgsrc is set then text will be empty
         */
        function formatNumberType(levelIndexes, ilvl, listDefinition, listParagraphIndex) {
            var ret = {},
                seqNo = 0,
                end = (listDefinition.listUnifiedNumbering === true) ? ilvl : 0,
                topLevelformat = listDefinition.listLevels[ilvl],
                levelText = topLevelformat.levelText ? topLevelformat.levelText : '',
                lvl = ilvl;
            ret.labelFollowedBy = listDefinition.listLevels[ilvl].labelFollowedBy;
            ret.tabpos = listDefinition.listLevels[ilvl].tabpos;

            for (; lvl >= end; --lvl) {
                var levelFormat = listDefinition.listLevels[lvl];
                if (listDefinition.listUnifiedNumbering === true) {
                    seqNo += listParagraphIndex;
                    if (listDefinition.listLevels[0] && listDefinition.listLevels[0].listStartValue !== undefined) {
                        seqNo += listDefinition.listLevels[0].listStartValue - 1;
                    }
                } else {
                    seqNo = (levelIndexes === undefined) ? 0 :
                        levelIndexes[lvl] + (levelFormat && levelFormat.listStartValue !== undefined ? levelFormat.listStartValue - 1 : 0);
                }
                var levelToken = '%' + (lvl + 1);
                var indexpos = levelText.indexOf(levelToken);
                if (indexpos < 0 && levelFormat.numberFormat !== 'bullet') {
                    continue;
                }
                var replacetext = '';
                switch (levelFormat.numberFormat) {
                    case 'bullet':
                        if (lvl === ilvl) { // bullets in 'upper' levels are ignored
                            if (levelFormat.levelPicBulletUri) {
                                ret.imgsrc = levelFormat.levelPicBulletUri;
                                ret.imgwidth = levelFormat.width;
                                replacetext = '';
                            } else {
                                var charCode = levelFormat.levelText ? levelFormat.levelText.charCodeAt(0) : -1;
                                if (charCode > 0 && (charCode < 0xE000 || charCode > 0xF8FF)) {
                                    replacetext = levelFormat.levelText;
                                } else {
                                    replacetext = charCode === 0xf0b7 ? '\u2022' : '\u25cf';
                                }
                            }
                        }
                        break;
                    case 'none':
                        replacetext = '';
                        break;
                    default:
                        replacetext = ListUtils.formatNumber(seqNo, levelFormat.numberFormat);
                }

                if (topLevelformat.levelPicBulletUri) {
                    levelText = '';
                    break;
                }
                if (topLevelformat.numberFormat === 'bullet') {
                    levelText = replacetext;
                    break;
                }
                levelText = levelText.replace(levelToken, replacetext);
            }
            ret.color = topLevelformat.color;
            if (!ret.color && topLevelformat.styleId) {
                var characterStyles = docModel.getStyleCollection('character');
                ret.color = characterStyles.getStyleAttributeSet(topLevelformat.styleId).character.color;
            }
            ret.text = levelText;
            return ret;
        }

        // methods ------------------------------------------------------------

        /**
         * Adds a new list to this container. An existing list definition
         * with the specified identifier will be replaced.
         *
         * @param {String} listIdentifier
         *  The name of of the new list definition.
         *
         * @param {Object} listDefinition
         *  The attributes of the list definition.
         *
         * @returns {ListCollection}
         *  A reference to this instance.
         */
        this.insertList = function (operation) {

            var list = {};

            lists.push(list);
            list.listIdentifier = operation.listStyleId;
            list.listUnifiedNumbering = operation.listUnifiedNumbering;
            list.listLevels = [];

            // handling the base style id (required for task 40792)
            if (operation.baseStyleId) {
                // saving base style direct in the list object
                list.baseStyleId = operation.baseStyleId;
                // creating key value pair from list style to base style
                baseStyleIDs[operation.listStyleId] = operation.baseStyleId;
                // creating list with all list styles belonging to one base style
                if (!allListStyleIDs[operation.baseStyleId]) { allListStyleIDs[operation.baseStyleId] = []; }
                allListStyleIDs[operation.baseStyleId].push(operation.listStyleId);
            }

            if (operation.listDefinition) {
                list.listLevels[0] = operation.listDefinition.listLevel0;
                list.listLevels[1] = operation.listDefinition.listLevel1;
                list.listLevels[2] = operation.listDefinition.listLevel2;
                list.listLevels[3] = operation.listDefinition.listLevel3;
                list.listLevels[4] = operation.listDefinition.listLevel4;
                list.listLevels[5] = operation.listDefinition.listLevel5;
                list.listLevels[6] = operation.listDefinition.listLevel6;
                list.listLevels[7] = operation.listDefinition.listLevel7;
                list.listLevels[8] = operation.listDefinition.listLevel8;
                if (operation.listDefinition.listLevel9) {
                    list.listLevels[9] = operation.listDefinition.listLevel9;
                }
                if (operation.listDefinition.defaultlist) {
                    if (operation.listDefinition.defaultlist === 'bullet') {
                        defaultBulletNumId = operation.listStyleId;
                    } else {
                        defaultNumberingNumId = operation.listStyleId;
                    }
                } else {
                    if (defaultBulletNumId === undefined) {
                        if (isDefinitionEqual(DEFAULT_BULLET_LIST_DEFINITION, operation.listDefinition) === true) {
                            defaultBulletNumId = operation.listStyleId;
                        }
                    }
                    if (defaultNumberingNumId === undefined) {
                        if (isDefinitionEqual(DEFAULT_NUMBERING_LIST_DEFINITION, operation.listDefinition) === true) {
                            defaultNumberingNumId = operation.listStyleId;
                        }
                    }
                }

                //wingdings bullets, had problem in clipborad and now also in the filter!
                //so we use mario clipboard replace utils to fix it!
                //getUnicodeReplacementForChar() from clipboardmixin
                //Bug 38545
                _.each(list.listLevels, function (listDefinition, index) {
                    var levelText = listDefinition.levelText;
                    var replace = docModel.getUnicodeReplacementForChar(listDefinition.levelText, listDefinition.fontName);
                    if (replace !== levelText && _.isObject(replace)) {
                        var newDef = _.clone(listDefinition);
                        newDef.levelText = replace.levelText;
                        newDef.fontName = replace.fontFamily;
                        list.listLevels[index] = newDef;
                    }
                });
            }

            // notify listeners
            this.trigger('insert:list', operation.listStyleId);
            return this;
        };

        /**
         * Remove an existing list
         *
         * @param {String} listIdentifier
         *  The name of of the list to be removed.
         *
         * @returns {ListCollection}
         *  A reference to this instance.
         */
        this.deleteList = function (listIdentifier) {
            var index = 0;
            for (; index < lists.length; ++index) {
                if (lists[index].listIdentifier === listIdentifier) {
                    lists.splice(index, 1);
                    if (defaultNumberingNumId === listIdentifier) {
                        defaultNumberingNumId = undefined;
                    }
                    if (defaultBulletNumId === listIdentifier) {
                        defaultBulletNumId = undefined;
                    }
                    break;
                }
            }

            // also removing the list id from the already inserted lists
            if (listIdentifier in insertedListIds) {
                delete insertedListIds[listIdentifier];
            }

            // notify listeners
            this.trigger('delete:list', listIdentifier);
            return this;
        };

        /**
         * Gives access to a single list definition.
         *
         * @param {String} name
         *  the name of the list to return.
         */
        this.getList = function (name) {
            var index = 0,
                ret;
            for (; index < lists.length; ++index) {
                if (lists[index].listIdentifier === name) {
                    ret = lists[index];
                    break;
                }
            }
            return ret;
        };

        /**
         * Gives access to all list definitions.
         */
        this.getLists = function () {
            return lists;
        };

        /**
         * Gives access to a single listlevel definition.
         *
         * @param {String} name
         *  The name of the list.
         *
         * @param {Number} level
         *  The list level to be returnd.
         *
         * @returns {Object|Null}
         *  The list level describing object, or Null, if it is not defined.
         */
        this.getListLevel = function (name, level) {
            var index = 0,
                ret = null,
                length = lists.length;

            for (; index < length; ++index) {
                if (lists[index].listIdentifier === name) {
                    ret = lists[index];
                    if (ret.listLevels && ret.listLevels[level]) {
                        ret = ret.listLevels[level];
                        break;
                    }
                    break;
                }
            }

            return ret;
        };

        /**
         * @param {String} type
         *  either bullet or numbering
         * @returns {integer}
         *  the Id of a default bullet or numbered numbering. If this default numbering definition is not available then it will be created
         */
        this.getDefaultNumId = function (type) {
            return type === 'bullet' ? defaultBulletNumId : defaultNumberingNumId;
        };

        /**
         * creates a list id not used, yet
         * @returns {String}
         *  new list id
         */
        this.getFreeListId = function () {
            var freeId = (lists.length + 1),
                sFreeId = 'L' + freeId;

            while (this.hasListStyleId(sFreeId)) {
                ++freeId;
                sFreeId = 'L' + freeId;
            }
            return sFreeId;
        };

        /**
         * Returns true if the given listStyleId is in use
         *
         * @param {String} listStyleId
         *  the list style id
         *
         * @returns {Boolean}
         *  return true if the listStyleId is in use
         */
        this.hasListStyleId = function (listStyleId) {
            return _.any(lists, function (element) {
                return (element.listIdentifier === listStyleId);
            });
        };

        /**
         * Returns the default list definition for bullet or numbering lists.
         *
         * @param {String} [type='numbering']
         *  'bullet' or 'character' for bullet lists, 'numbering' for numbered lists.
         *
         * @returns {Object}
         *  The default list definition.
         */
        this.getDefaultListDefinition = function (type) {
            var listDefinition = null;

            if (type === 'bullet' || type === 'character') {
                listDefinition = _.copy(DEFAULT_BULLET_LIST_DEFINITION, true);
                //replace ooxml special bullets
                if (odfFormat) {
                    listDefinition.listLevel0.levelText = '\u2022';
                    listDefinition.listLevel3.levelText = '\u2022';
                    listDefinition.listLevel6.levelText = '\u2022';
                }
            } else {
                listDefinition = _.copy(DEFAULT_NUMBERING_LIST_DEFINITION, true);
            }

            return listDefinition;
        };

        /**
         * @param {String} type
         *  either bullet or numbering
         * @param {Object} options
         *  can contain symbol - the bullet symbol
         *              listStartValue - start index of an ordered list
         * @returns {Object}
         *  the operation that creates the requested list
         *
         */
        this.getDefaultListOperation = function (type, options) {
            var sFreeId = this.getFreeListId();
            var newOperation = null;
            var target = null; // needed for odt handling (44393)
            var fullListStyleId = sFreeId; // needed for odt handling (44393)

            if (docModel.getApp().isODF() && docModel.isHeaderFooterEditState()) { // odt handling (44393)
                target = docModel.getActiveTarget();
                if (target) {
                    fullListStyleId = sFreeId + '_' + target;
                }
            }

            newOperation = { name: Operations.INSERT_LIST, listStyleId: fullListStyleId };

            if (target) { newOperation.target = target; } // 44393

            newOperation.listDefinition = this.getDefaultListDefinition(type);

            if (type === 'bullet') {
                if (options && options.symbol && options.symbol !== '*') {
                    newOperation.listDefinition.listLevel0.levelText = options.symbol;
                } else {
                    newOperation.listDefinition.defaultlist = type;
                }
            } else {
                var defaultlist = true;
                if (options) {
                    if (options.listStartValue) {
                        newOperation.listDefinition.listLevel0.listStartValue = options.listStartValue;
                        defaultlist = false;
                    }
                    if (options.numberFormat) {
                        newOperation.listDefinition.listLevel0.numberFormat = options.numberFormat;
                        defaultlist = false;
                    }
                }
                if (defaultlist) {
                    newOperation.listDefinition.defaultlist = type;
                }
            }
            return newOperation;
        };

        /**
         * Generates an Insert List operation from an array of html list types.
         *
         * @param {Array} htmlTypes
         *  The HTML list types.
         *      The array index represents the list level and the value
         *      the corresponding list type.
         *
         * @returns {Object}
         *  The operation that creates the requested list.
         *
         */
        this.getListOperationFromHtmlListTypes = function (htmlTypes) {

            var listLevel,
                levelCount,
                sFreeId = this.getFreeListId(),
                newOperation = { name: Operations.INSERT_LIST, listStyleId: sFreeId },
                type;

            if (_.isArray(htmlTypes)) {
                // set list definition for either bullet or numbering
                type = ListCollection.getListTypeFromHtmlListType(htmlTypes[0]);
                newOperation.listDefinition = _.copy((type === 'numbering') ? DEFAULT_NUMBERING_LIST_DEFINITION : DEFAULT_BULLET_LIST_DEFINITION, true);
                if (odfFormat) {
                    newOperation.listDefinition.listLevel9 =  { textAlign: 'left', indentLeft: 10 * 1270, indentFirstLine: -635, listStartValue: 1 };
                }
                levelCount = Math.min(htmlTypes.length, _.size(newOperation.listDefinition));

                for (listLevel = 0; listLevel < levelCount; listLevel++) {

                    switch (htmlTypes[listLevel]) {

                        case 'decimal':
                        case '1':
                            newOperation.listDefinition['listLevel' + listLevel].numberFormat = 'decimal';
                            newOperation.listDefinition['listLevel' + listLevel].levelText = '%' + (listLevel + 1) + '.';
                            break;

                        case 'decimal-leading-zero':
                            newOperation.listDefinition['listLevel' + listLevel].numberFormat = 'decimal';
                            newOperation.listDefinition['listLevel' + listLevel].levelText = '0%' + (listLevel + 1) + '.';
                            break;

                        case 'A':
                        case 'upper-alpha':
                        case 'upper-latin':
                            newOperation.listDefinition['listLevel' + listLevel].numberFormat = 'upperLetter';
                            newOperation.listDefinition['listLevel' + listLevel].levelText = '%' + (listLevel + 1) + '.';
                            break;

                        case 'a':
                        case 'lower-alpha':
                        case 'lower-latin':
                            newOperation.listDefinition['listLevel' + listLevel].numberFormat = 'lowerLetter';
                            newOperation.listDefinition['listLevel' + listLevel].levelText = '%' + (listLevel + 1) + '.';
                            break;

                        case 'I':
                        case 'upper-roman':
                            newOperation.listDefinition['listLevel' + listLevel].numberFormat = 'upperRoman';
                            newOperation.listDefinition['listLevel' + listLevel].levelText = '%' + (listLevel + 1) + '.';
                            break;

                        case 'i':
                        case 'lower-roman':
                            newOperation.listDefinition['listLevel' + listLevel].numberFormat = 'lowerRoman';
                            newOperation.listDefinition['listLevel' + listLevel].levelText = '%' + (listLevel + 1) + '.';
                            break;

                        case 'disc':
                            newOperation.listDefinition['listLevel' + listLevel].numberFormat = 'bullet';
                            newOperation.listDefinition['listLevel' + listLevel].fontName = 'Symbol';
                            newOperation.listDefinition['listLevel' + listLevel].levelText = '\uf0b7';
                            break;

                        case 'circle':
                            newOperation.listDefinition['listLevel' + listLevel].numberFormat = 'bullet';
                            newOperation.listDefinition['listLevel' + listLevel].fontName = 'Times New Roman';
                            newOperation.listDefinition['listLevel' + listLevel].levelText = '\u25CB';
                            break;

                        case 'square':
                            newOperation.listDefinition['listLevel' + listLevel].numberFormat = 'bullet';
                            newOperation.listDefinition['listLevel' + listLevel].fontName = 'Times New Roman';
                            newOperation.listDefinition['listLevel' + listLevel].levelText = '\u25A0';
                            break;

                        case 'none':
                            newOperation.listDefinition['listLevel' + listLevel].numberFormat = 'bullet';
                            newOperation.listDefinition['listLevel' + listLevel].fontName = 'Times New Roman';
                            newOperation.listDefinition['listLevel' + listLevel].levelText = ' ';
                            break;

                        default:
                            //newOperation.listDefinition['listLevel' + listLevel].numberFormat = 'decimal';
                            //newOperation.listDefinition['listLevel' + listLevel].levelText = '%1.';
                            break;
                    }
                }
            }

            return newOperation;
        };

        /**
         * Generates an Insert List operation from a MS list defintion (generated by the clipboard parser).
         *
         * @param {Object} definition
         *          {Object} level
         *              {String} numberFormat
         *                  The number format of the list level
         *              {String} levelText
         *                  The text for the list level
         *              {String} fontFamily
         *                  The level text font
         *              {Number} indentFirstLine
         *                  The first line text indent in 1/100 of millimeters.
         *              {Number} indentLeft
         *                  The left indent in 1/100 of millimeters.
         *              {String} numberPosition
         *                  The number position
         *
         * @returns {Object}
         *  The operation that creates the requested list.
         */
        this.getListOperationFromListDefinition = function (definition) {

            var sFreeId = this.getFreeListId(),
                newOperation = { name: Operations.INSERT_LIST, listStyleId: sFreeId },
                type = definition && definition[0] && definition[0].numberFormat;

            // set list definition for either bullet or numbering
            newOperation.listDefinition = _.copy((type === 'bullet') ? DEFAULT_BULLET_LIST_DEFINITION : DEFAULT_NUMBERING_LIST_DEFINITION, true);

            _.each(definition, function (listItem, listLevel) {

                if (listItem.numberFormat) {
                    newOperation.listDefinition['listLevel' + listLevel].numberFormat = listItem.numberFormat;
                }
                if (listItem.levelText) {
                    newOperation.listDefinition['listLevel' + listLevel].levelText = listItem.levelText;
                }
                if (listItem.fontFamily) {
                    newOperation.listDefinition['listLevel' + listLevel].fontName = listItem.fontFamily;
                }
                if (listItem.indentFirstLine) {
                    newOperation.listDefinition['listLevel' + listLevel].indentFirstLine = listItem.indentFirstLine;
                }
                if (listItem.indentLeft) {
                    newOperation.listDefinition['listLevel' + listLevel].indentLeft = listItem.indentLeft;
                }
                if (listItem.numberPosition) {
                    newOperation.listDefinition['listLevel' + listLevel].textAlign = listItem.numberPosition;
                }
            });

            return newOperation;
        };

        /**
         * Generates an Insert List operation from a list style id.
         *
         * @param {String} listStyleId
         *  The list style id.
         *
         * @returns {Object}
         *  The operation that creates the requested list.
         *
         */
        this.getListOperationFromListStyleId = function (listStyleId) {

            var newOperation = null,
                list = this.getList(listStyleId);

            if (list) {

                newOperation = { name: Operations.INSERT_LIST, listDefinition: {}, listStyleId: list.listIdentifier };

                _(list.listLevels).each(function (levelData, id) {
                    newOperation.listDefinition['listLevel' + id] = levelData;
                });

                // also adding the base style id, if available
                if (list.baseStyleId) { newOperation.baseStyleId = list.baseStyleId; }
            }

            return newOperation;
        };

        /**
         * @param {String} listId
         *  selected list style id
         *
         * @param {String} target
         *  An optional target that is used for generating specific list IDs that are
         *  used in lists in ODT headers or footers (44393).
         *
         * @param {String} predefinedStyleId the styleId to get the predefined liststyle
         *
         * @returns {Object|Null}
         *  the operation that creates the requested list
         */
        this.getSelectedListStyleOperation = function (listId, target, predefinedStyleId) {

            var fullListId = target ? (listId + '_' + target) : listId;
            var newOperation = { name: Operations.INSERT_LIST, listStyleId: fullListId };

            if (target && docModel.getApp().isODF() && docModel.isHeaderFooterEditState()) { newOperation.target = target; } // odt handling (44393)

            if ((predefinedStyleId in PREDEFINED_LISTSTYLES) && !(fullListId in insertedListIds)) {
                newOperation.listDefinition = _.copy(PREDEFINED_LISTSTYLES[predefinedStyleId].definition, true);
                insertedListIds[fullListId] = true;  // avoiding inserting list more than once
            } else {
                newOperation = null;
            }
            return newOperation;
        };

        /**
         * Merge a level of a list definition into an existing listStyle and create an insertList operation
         * (if no such list exists) and return that operation and it's listId
         *
         * @param {String} listId
         *  selected list style id
         * @param {Object} options
         *  can contain symbol - the bullet symbol
         *              listStartValue - start index of an ordered list
         * @returns {Object}
         *  containing the operation that creates the requested list (optional)
         *  and the id of the merged list
         *
         */
        this.mergeListStyle = function (existingListStyle, modifyingStyle, level) {

            var operationAndStyle = { listOperation: null, listStyleId: null },
                sourceLevel = null,
                sourceListDefinition;

            if ((modifyingStyle in PREDEFINED_LISTSTYLES)) {
                operationAndStyle.listOperation = { name: Operations.INSERT_LIST, listStyleId: this.getFreeListId() };
                operationAndStyle.listOperation.listDefinition = _.copy(this.getList(existingListStyle), true);
                sourceListDefinition = PREDEFINED_LISTSTYLES[modifyingStyle].definition;
                _(9).times(function (l) {
                    sourceLevel = l === level ? sourceListDefinition['listLevel' + l] : operationAndStyle.listOperation.listDefinition.listLevels[l];
                    operationAndStyle.listOperation.listDefinition['listLevel' + l] = _.copy(sourceLevel, true);

                });

                //search for the new list definition in the existing list styles to prevent creation of identical styles
                operationAndStyle.listStyleId = operationAndStyle.listOperation.listStyleId;
                _.each(lists, function (element, key, list) {
                    if (operationAndStyle.listOperation && isArrayDefinitionEqual(element, operationAndStyle.listOperation.listDefinition)) {
                        operationAndStyle.listStyleId = list[key].listIdentifier;
                        operationAndStyle.listOperation = null;
                    }
                });
            } else {
                operationAndStyle.listStyleId = existingListStyle;
            }
            return operationAndStyle;
        };

        /**
         * Checks if the given list definition matches an existing on a predefined list style
         * and returns the corresponding list style id, otherwise returns null.
         *
         * @param {Object} listDefinition
         *  the list definition to get the list style id for.
         *
         * @returns {String|null}
         *  the list style id for the given list definition or null.
         */
        this.getListStyleIdForListDefinition = function (listDefinition) {

            var listStyleId = null;

            // search for the list definition in the predefined list styles
            for (listStyleId in PREDEFINED_LISTSTYLES) {
                if (isDefinitionEqual(PREDEFINED_LISTSTYLES[listStyleId].definition, listDefinition)) {
                    return listStyleId;
                }
            }

            // search for the list definition in the existing list styles
            for (listStyleId = 0; listStyleId < lists.length; listStyleId++) {
                if (isArrayDefinitionEqual(lists[listStyleId], listDefinition)) {
                    return lists[listStyleId].listIdentifier;
                }
            }

            return null;
        };

        /**
         * Returns whether the specified list is a bullet list in all levels.
         *
         * @param {String} listStyleId
         *  The identifier of a list.
         *
         * @returns {Boolean}
         *  Whether the specified list identifier points to a list, that contains in all levels
         *  bullets (or the levels are undefined).
         */
        this.isAllLevelsBulletsList = function (listStyleId) {

            var currentList = this.getList(listStyleId);

            if (!currentList) { return false; }

            return _.every(_.range(10), function (level) {
                var listLevelDef = currentList.listLevels[level];
                return (!listLevelDef || ((listLevelDef.numberFormat === 'bullet') || (listLevelDef.numberFormat === 'none')));
            });
        };

        /**
         * Returns whether the specified list is a bullet list.
         *
         * @param {String} listStyleId
         *  The identifier of a list.
         *
         * @param {Number} listLevel
         *  the indent level of the list in the paragraph.
         *
         * @returns {Boolean}
         *  Whether the specified list identifier points to a bullet list.
         */
        this.isBulletsList = function (listStyleId, listLevel) {

            var currentList = this.getList(listStyleId),
                listLevelDef = currentList ? currentList.listLevels[listLevel] : null;

            return listLevelDef && ((listLevelDef.numberFormat === 'bullet') || (listLevelDef.numberFormat === 'none'));
        };

        /**
         * Returns whether the specified list is a numbered list.
         *
         * @param {String} listStyleId
         *  The identifier of a list.
         *
         * @param {Number} listLevel
         *  the indent level of the list in the paragraph.
         *
         * @returns {Boolean}
         *  Whether the specified list identifier points to a numbered list.
         */
        this.isNumberingList = function (listStyleId, listLevel) {

            var currentList = this.getList(listStyleId),
                listLevelDef = currentList ? currentList.listLevels[listLevel] : null;

            return listLevelDef && (listLevelDef.numberFormat !== 'bullet') && (listLevelDef.numberFormat !== 'none');
        };

        /**
         * Generates the numbering Label for the given paragraph
         *
         * @param {String} listId
         *  identifier of the applied numbering definition
         *
         * @param {Number} ilvl
         *  indent level, zero based
         *
         * @param {Array<Number>} levelIndexes
         *  Array of sequential position of the current paragraph, with ilvl+1
         *  elements that determines the sequential position of the current
         *  paragraph within the numbering.
         *
         * @returns {Object}
         *          indent
         *          labelwidth
         *          text
         *          tbd.
         */
        this.formatNumber = function (listStyleId, ilvl, levelIndexes, listParagraphIndex) {
            var ret = {};
            var currentList = this.getList(listStyleId);
            if (currentList === undefined) {
                return '?';
            }
            var levelFormat = currentList.listLevels[ilvl];
            if (levelFormat === undefined) {
                return '??';
            }
            var format = formatNumberType(levelIndexes, ilvl, currentList, listParagraphIndex);
            _.extend(ret, format);
            ret.indent = levelFormat.indentLeft;
            ret.firstLine = levelFormat.indentFirstLine ? levelFormat.indentLeft + levelFormat.indentFirstLine : levelFormat.indentLeft;
            return ret;
        };

        this.findIlvl = function (listStyleId, paraStyle) {
            var list = this.getList(listStyleId);
            if (list === undefined) {
                return -1;
            }
            var ilvl = 0;
            for (; ilvl < 9; ++ilvl) {
                var levelFormat = list.listLevels[ilvl];
                if (levelFormat.paraStyle === paraStyle) {
                    return ilvl;
                }
            }
            return -1;
        };

        this.findPrevNextStyle = function (listStyleId, paraStyle, prev) {
            var list = this.getList(listStyleId);
            if (list === undefined) {
                return '';
            }
            var ilvl = 0;
            for (; ilvl < 9; ++ilvl) {
                var levelFormat = list.listLevels[ilvl];
                if (levelFormat.paraStyle === paraStyle) {
                    var ret = '';
                    if (prev) {
                        if (ilvl > 0 && list.listLevels[ilvl - 1].paraStyle) {
                            ret = list.listLevels[ilvl - 1].paraStyle;
                        }
                    } else if (ilvl < 8 && list.listLevels[ilvl + 1].paraStyle) {
                        ret = list.listLevels[ilvl + 1].paraStyle;
                    }
                    return ret;
                }
            }
            return -1;
        };

        /**
         * Receiving a list of list style ids, that have the same base style id as a specified list
         * style. If there is no base style at all, undefined is returned.
         *
         * @param {String} listStyleId
         *  The list style id for that the list of all list styles with same base style id shall be returned.
         *
         * @returns {String[]|undefined}
         *  Returns an array with all list style IDs, that share the same base style as the specified
         *  list style. Or undefined, if there is no common base style at all.
         */
        this.getAllListStylesWithSameBaseStyle = function (listStyleId) {
            return baseStyleIDs[listStyleId] && allListStyleIDs[baseStyleIDs[listStyleId]];
        };

        /**
         * Receiving the base style id for a specified list style id. If there is no base style at all,
         * undefined is returned.
         *
         * @param {String} listStyleId
         *  The list style id for that the base style id shall be returned.
         *
         * @returns {String|undefined}
         *  Returns the base style ID for the specified list style id. Or undefined, if there is no base
         *  style at all.
         */
        this.getBaseStyleIdFromListStyle = function (listStyleId) {
            return baseStyleIDs[listStyleId];
        };

        /**
         * Getter for all predefined list styles.
         *
         * @returns {Object}
         *  A map with list style identifiers as keys, and objects as values
         *  containing the attributes 'definition' with the list style definition,
         *  'listlabel' containing a string with the number 1 formatted according
         *  to the list style, and 'tooltip' containing a GUI tool tip string for
         *  the list style.
         */
        this.getPredefinedListStyles = function () {
            return PREDEFINED_LISTSTYLES;
        };

        /**
         * Get the listId for a liststyle. If the liststyle is used in the previous paragrap, return the previous
         * listId, if the previous not empty paragraph has no liststyle or the given id is in use return null to create a new
         * listId.
         * @param {Object} listDefinition The default list definition.
         * @param {String} id the liststyleId returned if the previous not empty paragraph has not liststyle and if this id is not in use
         * @returns {String} the liststyleId to create a liststyle
         */
        this.getListId = function (listDefinition, id) {
            var listId = this.hasListStyleId(id) ? null : id;
            if (listDefinition) {
                var rootNode = docModel.getCurrentRootNode();
                var startPosition = docModel.getSelection().getStartPosition();
                startPosition.pop();
                var paragraph = Position.getParagraphElement(rootNode, startPosition);
                var prevParagraph = TextUtils.findPreviousNode(rootNode, paragraph, DOM.PARAGRAPH_NODE_SELECTOR);
                while (prevParagraph) {
                    if (DOM.isEmptyParagraph(prevParagraph)) {
                        prevParagraph = TextUtils.findPreviousNode(rootNode, prevParagraph, DOM.PARAGRAPH_NODE_SELECTOR);
                    } else {
                        var attrs = docModel.getParagraphStyles().getElementAttributes(prevParagraph);
                        if (attrs.paragraph.listStyleId !== '' && isArrayDefinitionEqual(this.getList(attrs.paragraph.listStyleId), listDefinition)) {
                            listId = attrs.paragraph.listStyleId;
                        } else {
                            listId = null;
                        }
                        prevParagraph = null;
                    }
                }
            }

            return listId;
        };

        // initialization -----------------------------------------------------

        this.registerDestructor(function () {
            docModel = lists = null;
        });

    } // class ListCollection

    // static methods ---------------------------------------------------------

    ListCollection.getListTypeFromHtmlListType = function (htmlType) {
        switch (htmlType) {
            case 'disc':
            case 'square':
            case 'circle':
            case 'none':
                return 'bullet';

            default:
                return 'numbering';
        }
    };

    // exports ================================================================

    // derive this class from class ModelObject
    return ModelObject.extend({ constructor: ListCollection });

});
