/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/spreadsheet/view/render/drawingrenderer', [
    'io.ox/office/tk/utils',
    'io.ox/office/tk/forms',
    'io.ox/office/tk/utils/iteratorutils',
    'io.ox/office/tk/container/valueset',
    'io.ox/office/drawinglayer/view/drawingframe',
    'io.ox/office/editframework/utils/attributeutils',
    'io.ox/office/textframework/utils/dom',
    'io.ox/office/spreadsheet/utils/config',
    'io.ox/office/spreadsheet/model/drawing/framenodemanager',
    'io.ox/office/spreadsheet/model/drawing/text/textframeutils',
    'io.ox/office/spreadsheet/view/render/renderutils',
    'io.ox/office/spreadsheet/view/render/rendererbase',
    'io.ox/office/spreadsheet/view/popup/drawingcontextmenu'
], function (Utils, Forms, IteratorUtils, ValueSet, DrawingFrame, AttributeUtils, DOMUtils, Config, FrameNodeManager, TextFrameUtils, RenderUtils, RendererBase, DrawingContextMenu) {

    'use strict';

    // private global functions ===============================================

    /**
     * Returns whether the DOM drawing frame of the passed drawing model needs
     * to be refreshed after it has been created or changed.
     *
     * @param {DrawingModel} drawingModel
     *  The model of a drawing object.
     *
     * @returns {Boolean}
     *  Whether the DOM drawing frame of the passed drawing model needs to be
     *  refreshed.
     */
    function supportsFormatting(drawingModel) {
        return drawingModel.getType() !== 'group';
    }

    // class DrawingRenderer ==================================================

    /**
     * Renders the drawing objects of the active sheet into the DOM drawing
     * layer shown in a single grid pane.
     *
     * @constructor
     *
     * @extends RendererBase
     *
     * @param {GridPane} gridPane
     *  The grid pane instance that owns this drawing layer renderer.
     */
    var DrawingRenderer = RendererBase.extend({ constructor: function (gridPane) {

        // self reference
        var self = this;

        // the spreadsheet application, model, and view
        var docView = gridPane.getDocView();
        var docModel = docView.getDocModel();
        var app = docModel.getApp();

        // the root layer node containing the page node
        var layerNode = gridPane.createLayerNode('drawing-layer');

        // the page node serving as root node for the text framework
        var pageRootNode = DOMUtils.createPageNode().appendTo(layerNode)[0];

        // the content node of the page node (child of page node)
        var pageContentNode = DOMUtils.getPageContentNode(pageRootNode)[0];

        // the manager for the DOM drawing frames of the drawing objects to be rendered
        var frameNodeManager = new FrameNodeManager(docModel);

        // the container node for all top-level drawing frames
        var frameRootNode = frameNodeManager.getRootNode();

        // the context menu for drawing objects
        var contextMenu = new DrawingContextMenu(gridPane, layerNode);

        // current layer rectangle covered by this renderer
        var layerRectangle = null;

        // all drawing models whose drawing frames need to be rendered
        var pendingModelSet = new ValueSet('getUid()');

        // all drawing frames selected by a remote user
        var remoteDrawingFrames = [];

        // base constructor ---------------------------------------------------

        RendererBase.call(this, gridPane);

        // private methods ----------------------------------------------------

        /**
         * Returns all selected DOM drawing frame nodes.
         *
         * @returns {jQuery}
         *  The selected drawing frames, as jQuery object.
         */
        function getSelectedDrawingFrames() {
            return $(frameRootNode).find(DrawingFrame.NODE_SELECTOR + Forms.SELECTED_SELECTOR);
        }

        /**
         * Renders a selection frame into the passed drawing frame.
         *
         * @param {jQuery} drawingFrame
         *  The drawing frame to be selected.
         *
         * @param {Boolean} [force=false]
         *  If set to true, an existing selection frame will be removed.
         */
        function updateSelectionFrame(drawingFrame, force) {

            var editMode = docView.isEditable() && !docView.isSheetLocked();
            var drawingModel = DrawingFrame.getModel(drawingFrame);

            if (force) { DrawingFrame.clearSelection(drawingFrame); }

            DrawingFrame.drawSelection(drawingFrame, {
                movable: editMode,
                resizable: editMode,
                rotatable: editMode && !app.isODF() && drawingModel.isDeepRotatable(), // bug 52930: disable rotation in ODS
                rotation: drawingModel.getRotationDeg()
            });
        }

        /**
         * Updates the dirty flags of the passed drawing frame according to the
         * passed options.
         */
        function updateDirtyFlags(drawingFrame, options) {
            var dirtyFormat = !!drawingFrame.data('dirtyFormat') || Utils.getBooleanOption(options, 'format', false);
            var dirtyText = !!drawingFrame.data('dirtyText') || Utils.getBooleanOption(options, 'text', false);
            drawingFrame.data({ dirtyFormat: dirtyFormat, dirtyText: dirtyText });
        }

        /**
         * Updates the CSS position of the passed text frame node.
         *
         * @param {DrawingModel} drawingModel
         *  The model of a drawing object.
         *
         * @param {jQuery} textFrame
         *  The DOM text frame of the passed drawing model.
         *
         * @param {Object} textPosition
         *  The original position of the text frame at 100% zoom level without
         *  flipping or rotation, as received from the rendering cache.
         */
        function updateTextFramePosition(drawingModel, textFrame/*, textPosition*/) {

            // the current zoom factor of the active sheet
            var zoom = docView.getZoomFactor();
            // text needs to be flipped horizontally, if horizontal/vertical flipping does not match
            var flipText = drawingModel.getEffectiveFlipping().reverseRot;
            // reverse horizontal scaling to flip the text back
            var scaleX = flipText ? -1 : 1;

            textFrame.css({ fontSize: Math.round(16 * zoom) + 'px', transform: 'scaleX(' + scaleX + ')' });
        }

        /**
         * Returns the drawing frame associated to the passed drawing model, if
         * it needs to be rendered, i.e. it covers the current layer rectangle
         * of this renderer, or it is selected (the latter is needed for move
         * tracking to be able to see the move box independent of the source
         * position of the drawing object).
         *
         * @param {DrawingModel} drawingModel
         *  The drawing model to be checked.
         *
         * @returns {DrawingFrame|Null}
         *  The drawing frame associated to the passed drawing model, if it
         *  needs to be rendered.
         */
        function getRenderDrawingFrame(drawingModel) {

            // get drawing frame from collection
            var drawingFrame = frameNodeManager.getDrawingFrame(drawingModel);
            if (!drawingFrame) { return null; }

            // always render selected drawing frames
            if (DrawingFrame.isSelected(drawingFrame)) { return drawingFrame; }

            // resolve absolute location of the drawing object
            var rectangle = drawingModel.getRectangle({ absolute: true });
            if (rectangle && layerRectangle.overlaps(rectangle)) { return drawingFrame; }

            // hide drawing frames in the DOM that are not in the visible area
            drawingFrame.addClass('hidden');
            return null;
        }

        /**
         * Registers the passed drawing model for deferred rendering of its
         * formatting, and starts a background task on demand that refreshes
         * the formatting of all registered pending drawing frames.
         *
         * @param {DrawingModel} drawingModel
         *  The model of the drawing object to be rendered.
         */
        var updateDrawingFrameFormatting = (function () {

            // the current background task for refreshing drawing frames
            var refreshTimer = null;

            // direct callback: insert the passed drawing model into the set
            function registerDrawingModel(drawingModel) {
                pendingModelSet.insert(drawingModel);
            }

            // deferred callback: start the background loop if needed
            var startBackgroundLoop = RenderUtils.profileMethod('DrawingRenderer.startBackgroundLoop()', function () {

                // nothing to do, if the timer is running, or if there are no dirty drawings
                if (refreshTimer || pendingModelSet.empty()) { return; }

                // start a background loop that will process all drawing models in 'pendingModelSet'
                refreshTimer = self.repeatSliced(function () {

                    // extract a drawing model from the pending set (exit loop when map is empty)
                    var drawingModel = pendingModelSet.getAny();
                    if (!drawingModel) { return Utils.BREAK; }

                    // remove processed drawing models from the set
                    pendingModelSet.remove(drawingModel);

                    RenderUtils.takeTime('DrawingRenderer.updateDrawingFrameFormatting()', function () {
                        RenderUtils.withLogging(function () {
                            RenderUtils.log('uid=' + drawingModel.getUid() + ' type=' + drawingModel.getType() + ' range=' + drawingModel.getRange());
                        });

                        // do not render the drawing frame if it has been scrolled/moved outside in the meantime
                        var drawingFrame = getRenderDrawingFrame(drawingModel);
                        if (!drawingFrame) { return; }

                        // the model frame (drawing frame used as source for the text contents)
                        var frameSettings = drawingModel.refreshModelFrame();
                        if (!frameSettings) { return; }
                        var modelFrame = frameSettings.drawingFrame;

                        // whether to refresh the formatting or text contents of the drawing frame
                        var dirtyFormat = drawingFrame.data('dirtyFormat');
                        var dirtyText = drawingFrame.data('dirtyText');

                        // remove the special display for uninitialized drawing frames, and the dirty markers
                        drawingFrame.removeClass('uninitialized').data({ dirtyFormat: false, dirtyText: false });

                        // text contents need to be copied for new drawing frames and for changed text contents
                        var textFrame = dirtyText ? TextFrameUtils.copyTextFrame(modelFrame, drawingFrame) : dirtyFormat ? TextFrameUtils.getTextFrame(drawingFrame) : null;

                        // reset all explicit CSS formatting of the textbox before formatting the drawing frame
                        if (textFrame) {
                            textFrame.css('transform', '');
                        }

                        // update formatting of the drawing frame
                        if (dirtyFormat) {
                            RenderUtils.takeTime('update formatting', function () {
                                var attrSet = drawingModel.getMergedAttributeSet(true);
                                DrawingFrame.updateFormatting(app, drawingFrame, attrSet, { skipFlipping: true });
                            });
                        }

                        // update paragraph/character formatting of the text contents
                        if (textFrame && frameSettings.textPosition) {
                            RenderUtils.takeTime('update text', function () {
                                updateTextFramePosition(drawingModel, textFrame, frameSettings.textPosition);
                                frameNodeManager.updateTextFormatting(drawingModel);
                            });
                        }

                        // notify all listeners
                        self.trigger('render:drawingframe', drawingFrame, drawingModel);
                    });
                }, 'DrawingRenderer.startBackgroundLoop', { delay: 200, slice: 200, interval: 200 });

                // clean up after the timer has finished
                refreshTimer.always(function () { refreshTimer = null; });
            });

            return app.createDebouncedActionsMethodFor(self, 'DrawingRenderer.updateDrawingFrameFormatting', registerDrawingModel, startBackgroundLoop);
        }());

        /**
         * Updates the position and formatting of the drawing frame that
         * represents the passed drawing model. The position and visibility of
         * the drawing frame will be updated immediately, the (expensive)
         * formatting will be updated in a background loop.
         *
         * @param {DrawingModel} drawingModel
         *  The drawing model instance whose drawing frame will be updated.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  - {Boolean} [options.hidden=false]
         *      Whether the drawing frame or any of its ancestors were hidden,
         *      and therefore need to be fully updated.
         *  - {Boolean} [options.zoom=false]
         *      Whether the zoom factor has been changed, i.e. whether all
         *      drawing frames need to be updated completely regardless of
         *      their current state.
         *  - {Boolean} [options.format=false]
         *      If set to true, the formatting of the drawing frame will always
         *      be updated. By default, the formatting will only be updated, if
         *      the visibility or size of the drawing frame has been changed.
         *  - {Boolean} [options.text=false]
         *      If set to true, the text contents of the drawing frame will
         *      always be updated. By default, the text contents will only be
         *      updated after updating the formatting of the drawing frame.
         */
        function updateDrawingFrame(drawingModel, options) {

            // the DOM drawing frame (skip drawing objects that are not inside the viewport)
            var drawingFrame = getRenderDrawingFrame(drawingModel);
            if (!drawingFrame) { return; }

            // settings for the model frame (drawing frame used as source for the text contents)
            var frameSettings = drawingModel.refreshModelFrame();
            if (!frameSettings) { return; }

            // copy explicit formatting attributes (expected by text framework, and drawing framework)
            AttributeUtils.setExplicitAttributes(drawingFrame, frameSettings.explicitAttrSet);

            // old location of the drawing frame in pixels (unless it was hidden)
            var oldHidden = Utils.getBooleanOption(options, 'hidden', false) || drawingFrame.hasClass('hidden');
            var oldRectangle = oldHidden ? null : drawingFrame.data('rectangle');
            drawingFrame.removeClass('hidden');

            // new location of the drawing frame in pixels (cannot be null due to preparations above)
            var newRectangle = drawingModel.getRectangle();
            drawingFrame.data('rectangle', newRectangle);

            // special display for new drawing frames that have not been formatted yet
            var formatSupported = supportsFormatting(drawingModel);
            if (!oldRectangle && formatSupported) { drawingFrame.addClass('uninitialized'); }

            // formatting and text contents need to be updated if specified, or if shown initially
            updateDirtyFlags(drawingFrame, oldRectangle ? options : { format: true, text: true });

            // set current position and size of the drawing frame as CSS properties
            var cssRectangle = drawingModel.isEmbedded() ? newRectangle : gridPane.convertToLayerRectangle(newRectangle);
            cssRectangle.left = Utils.minMax(cssRectangle.left, -Utils.MAX_NODE_SIZE, Utils.MAX_NODE_SIZE);
            cssRectangle.top = Utils.minMax(cssRectangle.top, -Utils.MAX_NODE_SIZE, Utils.MAX_NODE_SIZE);
            drawingFrame.css(cssRectangle.toCSS());

            // update flipping attributes manually
            var oldFlipH = DrawingFrame.isFlippedHorz(drawingFrame);
            var oldFlipV = DrawingFrame.isFlippedVert(drawingFrame);
            var newFlipH = drawingModel.isFlippedH();
            var newFlipV = drawingModel.isFlippedV();
            var isRotatable = drawingModel.isDeepRotatable();
            var flipData = drawingModel.getEffectiveFlipping();
            drawingFrame.toggleClass(DrawingFrame.FLIPPED_HORIZONTALLY_CLASSNAME, isRotatable ? newFlipH : flipData.flipH);
            drawingFrame.toggleClass(DrawingFrame.FLIPPED_VERTICALLY_CLASSNAME, isRotatable ? newFlipV : flipData.flipV);

            // apply rotation transformation
            var degrees = drawingModel.getRotationDeg();
            DrawingFrame.setCssTransform(drawingFrame, degrees, newFlipH, newFlipV);
            drawingFrame.toggleClass(DrawingFrame.ROTATED_DRAWING_CLASSNAME, degrees !== 0);

            // the selection needs to be redrawn, if the flip state of a connector drawing has changed
            if (((oldFlipH !== newFlipH) || (oldFlipV !== newFlipV)) && DrawingFrame.isSelected(drawingFrame) && DrawingFrame.isConnectorDrawingFrame(drawingFrame)) {
                updateSelectionFrame(drawingFrame, true);
            }

            // the resulting dirty flags
            var zoomChanged = Utils.getBooleanOption(options, 'zoom', false);
            var dirtyFormat = drawingFrame.data('dirtyFormat');
            var dirtyText = drawingFrame.data('dirtyText');

            // fast refresh of DOM positions and size while modifying or zooming visible drawings
            if (formatSupported && oldRectangle && (zoomChanged || dirtyFormat)) {

                // fast refresh of the canvas node size of shapes and connectors
                var canvasNode = DrawingFrame.getCanvasNode(drawingFrame);
                if (canvasNode.length === 1) {
                    var widthScale = newRectangle.width / oldRectangle.width;
                    var heightScale = newRectangle.height / oldRectangle.height;
                    canvasNode.css({
                        left: Math.round(Utils.convertCssLength(canvasNode.css('left'), 'px') * widthScale),
                        top: Math.round(Utils.convertCssLength(canvasNode.css('top'), 'px') * heightScale),
                        width: Math.round(canvasNode.width() * widthScale),
                        height: Math.round(canvasNode.height() * heightScale)
                    });
                }

                // set correct padding of the text frame according to current zoom factor
                if (frameSettings.textPosition) {
                    TextFrameUtils.withTextFrame(drawingFrame, function (textFrame) {
                        updateTextFramePosition(drawingModel, textFrame, frameSettings.textPosition);
                    });
                }
            }

            // immediately notify all listeners after the position or size has been changed
            self.trigger('render:drawingframe', drawingFrame, drawingModel);

            // start deferred update of formatting and text contents
            if (formatSupported && (dirtyFormat || dirtyText)) {
                updateDrawingFrameFormatting(drawingModel);
            } else {
                pendingModelSet.remove(drawingModel);
            }

            // propagate dirty formatting to all embedded frames
            if (oldHidden) { options = _.extend({}, options, { hidden: true }); }
            if (dirtyFormat) { options = _.extend({}, options, { format: true }); }

            // update existing embedded drawing frames too
            drawingModel.forEachChildModel(function (childModel) {
                updateDrawingFrame(childModel, options);
            });
        }

        /**
         * Creates and inserts a new DOM drawing frame node that represents the
         * passed drawing model. The position and visibility of the drawing
         * frame will be initialized immediately, the (expensive) formatting
         * will be updated in a background loop.
         *
         * @param {DrawingModel} drawingModel
         *  The drawing model instance to create a DOM drawing frame for.
         */
        function createDrawingFrame(drawingModel) {

            // the new drawing representation, as jQuery object
            var drawingFrame = frameNodeManager.createDrawingFrame(drawingModel);

            // update position and formatting of the drawing frame
            if (drawingFrame) {
                updateDrawingFrame(drawingModel, { format: true, text: true });
            }
        }

        /**
         * Updates the selection of all drawing frames, and triggers the event
         * 'render:drawingselection', if the drawing selection has changed.
         */
        var renderDrawingSelection = this.createDebouncedMethod('DrawingRenderer.renderDrawingSelection', null, function () {

            // all drawing frames currently selected in the DOM
            var oldSelectedFrames = getSelectedDrawingFrames();
            // all drawing frames to be selected in the DOM, according to current selection state
            var newSelectedFrames = self.getDrawingFrames(docView.getSelectedDrawings());

            // create or update the selection frame for all selected drawings
            newSelectedFrames.get().forEach(updateSelectionFrame);

            // remove selection border from all drawing frames not selected anymore
            oldSelectedFrames.not(newSelectedFrames).get().forEach(DrawingFrame.clearSelection);

            // notify listeners
            if (!Utils.equalArrays(oldSelectedFrames.get(), newSelectedFrames.get())) {
                self.trigger('render:drawingselection', newSelectedFrames);
            }
        });

        /**
         * Updates the remote selection of all drawing frames.
         */
        var renderRemoteSelection = Config.SHOW_REMOTE_SELECTIONS ? this.createDebouncedMethod('DrawingRenderer.renderRemoteSelection', null, function () {

            // nothing to do, if all columns/rows are hidden (method may be called debounced)
            if (!gridPane.isVisible()) { return; }

            // the drawing collection of the active sheet
            var drawingCollection = docView.getDrawingCollection();

            // remove the old remote selection nodes from the drawing frames
            remoteDrawingFrames.forEach(function (drawingFrame) {
                drawingFrame.removeAttr('data-remote-user data-remote-color').children('.remote-selection').remove();
            });
            remoteDrawingFrames = [];

            // generate selections of all remote users on the same sheet
            docView.iterateRemoteSelections(function (userName, colorIndex, ranges, drawings) {
                if (!drawings) { return; }

                // add a remote selection node to all selected drawings of the user
                drawings.forEach(function (position) {
                    frameNodeManager.withDrawingFrame(drawingCollection.getModel(position), function (drawingFrame) {

                        // do not insert multiple selection nodes into a drawing frame
                        if (drawingFrame.children('.remote-selection').length > 0) { return; }

                        // create the child node to display the border around the frame, and the data attributes for the user badge
                        drawingFrame.prepend('<div class="remote-selection">').attr({
                            'data-remote-user': _.noI18n(userName),
                            'data-remote-color': colorIndex
                        });
                        remoteDrawingFrames.push(drawingFrame);
                    });
                });
            });
        }, { delay: 100 }) : _.noop;

        // protected methods --------------------------------------------------

        /**
         * Changes the drawing layer according to the passed layer range.
         */
        this.setLayerRange = RenderUtils.profileMethod('DrawingRenderer.setLayerRange()', function (layerSettings) {

            // the drawing collection of the active sheet
            var drawingCollection = docView.getDrawingCollection();
            // whether existing drawing frames need to be updated
            var isVisible = layerRectangle !== null;

            // store new layer rectangle in internal member
            layerRectangle = layerSettings.rectangle;

            // repaint all visible drawing frames; or create all drawing frames if
            // not done yet (e.g. active sheet changed, enabled split in current sheet)
            var iterator = drawingCollection.createModelIterator();

            // update all existing drawing frames, if the layer range has changed
            if (isVisible) {
                var options = (layerSettings.zoomScale !== 1) ? { format: true, zoom: true } : null;
                IteratorUtils.forEach(iterator, function (drawingModel) {
                    updateDrawingFrame(drawingModel, options);
                });
                return;
            }

            // prepare the leading empty container nodes for the preceding sheets (needed by the
            // text framework to be able to generate correct text positions in document operations)
            pageContentNode.removeChild(frameRootNode);
            var activeSheet = docView.getActiveSheet();
            var childNodes = pageContentNode.childNodes;
            for (var missingNodes = activeSheet - childNodes.length; missingNodes > 0; missingNodes -= 1) {
                pageContentNode.appendChild(FrameNodeManager.createRootNode());
            }
            pageContentNode.insertBefore(frameRootNode, childNodes.item(activeSheet));

            // create all DOM drawing frames in this sheet, schedule for deferred rendering
            IteratorUtils.forEach(iterator, createDrawingFrame);

            // paint the entire drawing selection
            renderDrawingSelection();
        });

        /**
         * Resets this renderer, and clears the DOM layer node (called e.g.
         * before switching the active sheet).
         */
        this.hideLayerRange = function () {

            // clear the cache of pending drawing frames
            pendingModelSet.clear();

            // reset internal settings
            layerRectangle = null;
            remoteDrawingFrames = [];

            // safely destroy all image nodes to prevent memory problems on iPad
            frameNodeManager.removeAllDrawingFrames();
        };

        // public methods -----------------------------------------------------

        /**
         * Returns the DOM drawing frame node at the passed document position.
         *
         * @param {Array<Number>} position
         *  The document position of the drawing model. May specify a position
         *  inside another drawing object.
         *
         * @returns {jQuery|Null}
         *  The DOM drawing frame, as jQuery object; or null, if the drawing
         *  frame does not exist.
         */
        this.getDrawingFrame = function (position) {
            var drawingModel = docView.getDrawingCollection().getModel(position);
            return drawingModel ? frameNodeManager.getDrawingFrame(drawingModel) : null;
        };

        /**
         * Returns the DOM drawing frame nodes for all specified document
         * positions.
         *
         * @param {Array<Array<Number>>} positions
         *  The document positions of the drawing models.
         *
         * @returns {jQuery}
         *  The DOM nodes of the drawing frames, as jQuery object.
         */
        this.getDrawingFrames = function (positions) {
            return _.reduce(positions, function (drawingFrames, position) {
                return drawingFrames.add(this.getDrawingFrame(position));
            }, $(), this);
        };

        /**
         * Returns the DOM drawing frame that represents the passed drawing
         * model.
         *
         * @param {DrawingModel} drawingModel
         *  The drawing model instance to return the DOM drawing frame for.
         *
         * @returns {jQuery|Null}
         *  The DOM drawing frame for the passed drawing model; or null, if no
         *  drawing frame could be found.
         */
        this.getDrawingFrameForModel = function (drawingModel) {
            return frameNodeManager.getDrawingFrame(drawingModel);
        };

        /**
         * Returns the drawing frame to be selected or manipulated by the
         * passed tracking event.
         *
         * @param {jQuery.Event} event
         *  The tracking event to be evaluated.
         *
         * @returns {jQuery}
         *  The drawing frame to be selected or manipulated by the passed
         *  tracking event if existing; otherwise an empty jQuery collection.
         */
        this.getDrawingFrameForEvent = function (event) {

            // ensure that a drawing frame is targeted
            var drawingFrame = $(Utils.findClosest(frameRootNode, event.target, DrawingFrame.NODE_SELECTOR));
            if (drawingFrame.length === 0) { return drawingFrame; }

            // ignore the empty area of group objects, unless the group object is selected
            if (!DrawingFrame.isSelected(drawingFrame) && (DrawingFrame.getDrawingType(drawingFrame) === 'group')) {
                return $();
            }

            // always track the root parent of grouped drawing objects (TODO: handle activated group objects)
            return $(Utils.findFarthest(frameRootNode, drawingFrame, DrawingFrame.NODE_SELECTOR));
        };

        // initialization -----------------------------------------------------

        // insert the frame root node into the page
        pageContentNode.appendChild(frameRootNode);

        // repaint drawing selection (also when document edit mode changes, e.g. existence of resize handles)
        this.listenToWhenVisible(app, 'docs:editmode', renderDrawingSelection);
        this.listenToWhenVisible(docView, 'change:selection change:sheet:attributes', renderDrawingSelection);

        // trigger a 'render:drawingselection' event if active pane changes
        // (no actual rendering of the selection needed, border colors will change via CSS)
        this.listenToWhenVisible(docView, 'change:sheet:viewattributes', function (event, attributes) {
            if ('activePane' in attributes) {
                self.trigger('render:drawingselection', getSelectedDrawingFrames());
            }
        });

        // repaint remote drawing selections
        this.listenToWhenVisible(docView, 'change:viewattributes', function (event, attributes) {
            if ('remoteClients' in attributes) {
                renderRemoteSelection();
            }
        });

        // process new drawing objects inserted into the drawing collection
        this.listenToWhenVisible(docView, 'insert:drawing', function (event, drawingModel) {
            createDrawingFrame(drawingModel);
            renderDrawingSelection();
            renderRemoteSelection();
        });

        // process drawing objects removed from the drawing collection
        this.listenToWhenVisible(docView, 'delete:drawing', function (event, drawingModel) {
            frameNodeManager.removeDrawingFrame(drawingModel);
            pendingModelSet.remove(drawingModel);
            renderDrawingSelection();
            renderRemoteSelection();
        });

        // process drawing objects that have been changed in any way (position, size, attributes)
        this.listenToWhenVisible(docView, 'change:drawing', function (event, drawingModel) {
            updateDrawingFrame(drawingModel, { format: true, text: true });
        });

        // process drawing objects whose text contents have been changed
        this.listenToWhenVisible(docView, 'change:drawing:text', function (event, drawingModel) {
            updateDrawingFrame(drawingModel, { text: true });
        });

        // process drawing objects that have been moved to a new document position (Z order)
        this.listenToWhenVisible(docView, 'move:drawing', function (event, drawingModel) {
            frameNodeManager.moveDrawingFrame(drawingModel);
            renderDrawingSelection();
            renderRemoteSelection();
        });

        // bug 31479: suppress double-clicks for drawings
        layerNode.on('dblclick', false);

        // destroy all class members on destruction
        this.registerDestructor(function () {
            contextMenu.destroy();
            frameNodeManager.destroy();
            pendingModelSet.clear();
            self = gridPane = docModel = docView = app = null;
            layerNode = pageRootNode = pageContentNode = null;
            frameNodeManager = frameRootNode = null;
            contextMenu = pendingModelSet = null;
            remoteDrawingFrames = null;
        });

    } }); // class DrawingRenderer

    // exports ================================================================

    return DrawingRenderer;

});
