/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/spreadsheet/view/popup/headercontextmenu', [
    'io.ox/office/tk/utils',
    'io.ox/office/baseframework/view/popup/contextmenu',
    'io.ox/office/spreadsheet/view/controls',
    'gettext!io.ox/office/spreadsheet/main'
], function (Utils, ContextMenu, Controls, gt) {

    'use strict';

    // convenience shortcuts
    var Button = Controls.Button;

    // class HeaderContextMenu ================================================

    /**
     * A context menu for header panes. Provides menu actions to manipulate
     * entire columns/rows in the active sheet.
     *
     * @constructor
     *
     * @extends ContextMenu
     *
     * @param {HeaderPane} headerPane
     *  The header pane that contains this instance.
     *
     * @param {Boolean} columns
     *  Whether this context menu is for columns (true), or for rows (false).
     */
    function HeaderContextMenu(headerPane, columns) {

        // self reference
        var self = this;

        // the document model and view
        var docView = headerPane.getDocView();
        var docModel = docView.getDocModel();

        // base constructor ---------------------------------------------------

        ContextMenu.call(this, docView, headerPane.getNode(), {
            selector: '.cell', // only on cell header nodes (not on resizers)
            // do not show context menu at all in locked sheets, or during cell edit mode
            enableKey: 'sheet/operation/unlocked/cell',
            delay: 200
        });

        // private methods ----------------------------------------------------

        /**
         * Inserts column-specific menu entries into the context menu.
         */
        function initColumnMenu() {

            var multiCols = docView.getSelectedRanges().colIntervals().size() > 1;

            self.destroyAllGroups();

            self.addGroup('column/insert', new Controls.ColRowOpLabelButton(docView, true, true, multiCols))
                .addGroup('column/delete', new Controls.ColRowOpLabelButton(docView, false, true, multiCols))
                .addSeparator()
                .addGroup('column/width/optimal', new Button(docView, { label: gt('Set optimal column width') }))
                .addGroup('column/hide', new Button(docView, { label: multiCols ? gt('Hide columns') : gt('Hide column') }))
                .addGroup('column/show', new Button(docView, { label: gt('Show hidden columns') }));

            self.updateAllGroups();
        }

        /**
         * Inserts row-specific menu entries into the context menu.
         */
        function initRowMenu() {

            var multiRows = docView.getSelectedRanges().rowIntervals().size() > 1;

            self.destroyAllGroups();

            self.addGroup('row/insert', new Controls.ColRowOpLabelButton(docView, true, false, multiRows))
                .addGroup('row/delete', new Controls.ColRowOpLabelButton(docView, false, false, multiRows))
                .addSeparator()
                .addGroup('row/height/optimal', new Button(docView, { label: gt('Set optimal row height') }))
                .addGroup('row/hide', new Button(docView, { label: multiRows ? gt('Hide rows') : gt('Hide row') }))
                .addGroup('row/show', new Button(docView, { label: gt('Show hidden rows') }));

            self.updateAllGroups();
        }

        /**
         * Selects the column/row that has been clicked with the right mouse
         * button, before the context menu will be shown.
         */
        function contextMenuPrepareHandler(event) {

            // the source event that caused to open the context menu
            var sourceEvent = event.sourceEvent;
            // the index of the column/row to be selected
            var index = Utils.getElementAttributeAsInteger(sourceEvent.target, 'data-index', -1);

            // functor to check whether the column/row index is contained in a range address
            function rangeContainsIndex(range) {
                return docModel.isFullRange(range, columns) && range.containsIndex(index, columns);
            }

            // Do nothing if no valid cell node has been clicked
            if ((index < 0)) {
                event.preventDefault();
                return;
            }

            // select the clicked column/row if it is not contained in the selection
            if (!docView.getSelectedRanges().some(rangeContainsIndex)) {
                docView.grabFocus();
                headerPane.trigger('select:start', index, 'select');
                headerPane.trigger('select:end', index, true);
            }
        }

        // public methods ----
        this.getXY = function (event) {

            if (event.sourceEvent.pageX && event.sourceEvent.pageY) {
                return {
                    pageX: event.sourceEvent.pageX,
                    pageY: event.sourceEvent.pageY
                };
            }

            var offset = $(event.sourceEvent.target).offset();
            return {
                pageX: offset.left + $(event.sourceEvent.target).width() / 2,
                pageY: offset.top + $(event.sourceEvent.target).height() / 2
            };
        };

        // initialization -----------------------------------------------------

        // lazy initialization of the menu items according to the orientation of the header pane
        this.on('popup:beforeshow', columns ? initColumnMenu : initRowMenu);

        // preprocessing before the context menu will be shown
        this.on('contextmenu:prepare', contextMenuPrepareHandler);

        // hide contextmenu if the user start scrolling
        this.listenTo(docView, 'change:scrollpos', function () { self.hide(); });

        // destroy all class members
        this.registerDestructor(function () {
            self = headerPane = docModel = docView = null;
        });

    } // class HeaderContextMenu

    // exports ================================================================

    // derive this class from class ContextMenu
    return ContextMenu.extend({ constructor: HeaderContextMenu });

});
