/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 * @author Ingo Schmidt-Rosbiegal <ingo.schmidt-rosbiegal@open-xchange.com>
 */

define('io.ox/office/spreadsheet/view/mixin/gridtrackingmixin', [
    'io.ox/office/tk/utils',
    'io.ox/office/tk/container/valueset',
    'io.ox/office/textframework/utils/dom',
    'io.ox/office/drawinglayer/view/drawingframe',
    'io.ox/office/spreadsheet/utils/sheetutils',
    'io.ox/office/spreadsheet/utils/paneutils',
    'io.ox/office/spreadsheet/model/drawing/text/textframeutils',
    'io.ox/office/drawinglayer/utils/drawingutils',
    'io.ox/core/tk/nodetouch' // activates $.enableTouch() function, not used otherwise (keep at end of list)
], function (Utils, ValueSet, DOMUtils, DrawingFrame, SheetUtils, PaneUtils, TextFrameUtils, DrawingUtils) {

    'use strict';

    // mathematical constants
    var PI_180 = Math.PI / 180;

    // convenience shortcuts
    var Address = SheetUtils.Address;
    var Range = SheetUtils.Range;

    // mix-in class GridTrackingMixin =========================================

    /**
     * Mix-in class for the class GridPane that implements all kinds of mouse
     * and touch tracking, for example selection, or manipulation of drawing
     * objects.
     *
     * @constructor
     */
    function GridTrackingMixin() {

        // self reference (the GridPane instance)
        var self = this;

        // the spreadsheet model and view
        var docView = this.getDocView();
        var docModel = docView.getDocModel();

        // the root node of all rendering layers
        var layerRootNode = this.getLayerRootNode();

        // special behavior in OOXML documents
        var ooxml = docModel.getApp().isOOXML();

        // private methods ----------------------------------------------------

        /**
         * Returns the address of the cell covering the page coordinates in the
         * passed tracking event.
         *
         * @param {jQuery.Event} event
         *  The tracking event to be evaluated, expected to contain the numeric
         *  properties 'pageX' and 'pageY'.
         *
         * @param {Object} [options]
         *  Optional parameters that will be forwarded to the method
         *  ColRowCollection.getEntryByOffset().
         *
         * @returns {Address}
         *  The address of the cell containing the tracking position.
         */
        function getCellAddress(event, options) {
            return self.getCellDataForEvent(event, options).address;
        }

        /**
         * Returns the address of the anchor cell for a tracking cell range,
         * according to the passed tracking position.
         *
         * @param {jQuery.Event} event
         *  The initial 'tracking:start' event.
         *
         * @param {Range} origRange
         *  The address of the cell range to be tracked.
         *
         * @param {String} trackerPos
         *  The position identifier of the tracker node.
         *
         * @param {Boolean} resize
         *  Whether resize tracking (true), or border tracking (false) is
         *  active. For border tracking, the tracker position passed in the
         *  parameter 'trackerPos' must be 't', 'b', 'l', or 'r'.
         *
         * @returns {Address}
         *  The address of the anchor cell in the passed range.
         */
        function getAnchorCell(event, origRange, trackerPos, resize) {
            switch (trackerPos) {
                case 'l':
                    return resize ? origRange.end.clone() : new Address(origRange.start[0], getCellAddress(event)[1]);
                case 'r':
                    return resize ? origRange.start.clone() : new Address(origRange.end[0], getCellAddress(event)[1]);
                case 't':
                    return resize ? origRange.end.clone() : new Address(getCellAddress(event)[0], origRange.start[1]);
                case 'b':
                    return resize ? origRange.start.clone() : new Address(getCellAddress(event)[0], origRange.end[1]);
                case 'tl':
                    return origRange.end.clone();
                case 'tr':
                    return new Address(origRange.start[0], origRange.end[1]);
                case 'bl':
                    return new Address(origRange.end[0], origRange.start[1]);
                case 'br':
                    return origRange.start.clone();
            }
        }

        /**
         * Recalculates the address of the tracked range according to the
         * passed tracking position.
         *
         * @param {jQuery.Event} event
         *  The tracking event.
         *
         * @param {Range} origRange
         *  The address of the original cell range currently tracked.
         *
         * @param {Address} anchorCell
         *  The anchor cell inside the passed original range, as has been
         *  returned by the method GridTrackingMixin.getAnchorCell().
         *
         * @param {String} trackerPos
         *  The position identifier of the tracker node.
         *
         * @param {Boolean} resize
         *  Whether resize tracking (true), or border tracking (false) is
         *  active. For border tracking, the tracker position passed in the
         *  parameter 'trackerPos' must be 't', 'b', 'l', or 'r'.
         *
         * @returns {Range}
         *  The address of the cell range currently tracked.
         */
        function recalcTrackedRange(event, origRange, anchorCell, trackerPos, resize) {

            // current column/row entries
            var trackingData = self.getCellDataForEvent(event, { outerHidden: true });
            // the cell address currently tracked
            var address = trackingData.address;
            // the resulting range
            var range = null;

            // adjusts column/row index according to the offset inside the covered cell
            function adjustTrackedBorder(columns) {

                // the maximum column/row index
                var maxIndex = docModel.getMaxIndex(columns);
                // the column/row descriptor
                var entryData = columns ? trackingData.colDesc : trackingData.rowDesc;
                // whether the leading border is tracked
                var leadingBorder = resize ? (address.get(columns) < anchorCell.get(columns)) : (trackerPos === (columns ? 'l' : 't'));
                // whether the trailing border is tracked
                var trailingBorder = resize ? (address.get(columns) > anchorCell.get(columns)) : (trackerPos === (columns ? 'r' : 'b'));
                // the offset ratio in the tracked column/row
                var ratio = (entryData.size > 0) ? (entryData.relOffset / entryData.size) : Number.NaN;

                if (leadingBorder && (address.get(columns) < maxIndex) && (ratio > 0.5)) {
                    // jump to next column/row if tracking leading border, and covering second half of the tracked column/row ...
                    address.move(1, columns);
                } else if (trailingBorder && (address.get(columns) > 0) && (ratio < 0.5)) {
                    // ... or to previous column/row if tracking trailing border, and covering first half of the tracked column/row
                    address.move(-1, columns);
                }
            }

            // adjust column/row index according to the offset inside the tracked cell
            adjustTrackedBorder(false);
            adjustTrackedBorder(true);

            // build and return the new range according to the current position of the tracked cell
            if (resize) {
                // resize mode: build range between anchor cell and tracked cell
                range = Range.createFromAddresses(anchorCell, address);
                if ((trackerPos === 't') || (trackerPos === 'b')) {
                    // resize vertically: restore row indexes to original range
                    range.start[0] = origRange.start[0];
                    range.end[0] = origRange.end[0];
                } else if ((trackerPos === 'l') || (trackerPos === 'r')) {
                    // resize horizontally: restore row indexes to original range
                    range.start[1] = origRange.start[1];
                    range.end[1] = origRange.end[1];
                }
            } else {
                // border mode: move entire range according to distance between tracked cell and anchor cell
                range = docModel.getBoundedMovedRange(origRange, address[0] - anchorCell[0], address[1] - anchorCell[1]);
            }

            return range;
        }

        // cell selection (mouse) ---------------------------------------------

        /**
         * Handles all tracking events for cell selection originating from
         * mouse events.
         */
        var mouseSelectionTrackingHandler = (function () {

            // current address (prevent updates while tracking over the same cell)
            var currentAddress = null;

            // updates the current address according to the passed tracking event
            function updateCurrentAddress(event) {
                var address = getCellAddress(event, { outerHidden: true });
                if (address.differs(currentAddress)) {
                    currentAddress = address;
                    self.trigger('select:move', currentAddress);
                }
            }

            // return the actual mouseSelectionTrackingHandler() method
            return function (event, endPromise) {
                switch (event.type) {
                    case 'tracking:start':
                        currentAddress = getCellAddress(event);
                        self.trigger('select:start', currentAddress, PaneUtils.getSelectionMode(event));
                        self.scrollToCell(currentAddress);
                        break;
                    case 'tracking:move':
                        updateCurrentAddress(event);
                        break;
                    case 'tracking:scroll':
                        self.scrollRelative(event.scrollX, event.scrollY);
                        updateCurrentAddress(event);
                        break;
                    case 'tracking:end':
                    case 'tracking:cancel':
                        endPromise.always(function (success) {
                            self.trigger('select:end', currentAddress, success);
                            self.scrollToCell(currentAddress);
                        });
                        break;
                }
            };
        }()); // end of mouseSelectionTrackingHandler() local scope

        // cell selection (touch) ---------------------------------------------

        /**
         * Handles all tracking events for cell selection originating from
         * touch events.
         */
        var touchSelectionTrackingHandler = (function () {

            // cell address where tracking has started
            var startAddress = null;
            // whether the touch point has been moved
            var moved = false;

            function finalizeTracking(event) {

                // do nothing, if the touch point has been moved
                if (moved) { return; }

                // current selection
                var selection = docView.getSelection();
                // array index of the range that contains a tapped cell
                var rangeIndex = 0;

                if (event.duration <= 750) {
                    // short-tap: select a single cell, or start cell edit mode
                    if (!docView.hasDrawingSelection() && selection.isActive(startAddress)) {
                        docView.enterTextEditMode('cell');

                    } else if (!docView.hasDrawingSelection() && selection.containsAddress(startAddress)) {
                        rangeIndex = Utils.findFirstIndex(selection.ranges, function (range) { return range.containsAddress(startAddress); });
                        self.trigger('select:active', rangeIndex, startAddress);

                    } else {
                        self.trigger('select:start', startAddress, 'select');
                        self.trigger('select:end', startAddress, true);
                        self.scrollToCell(startAddress);
                    }
                } else {
                    if (!selection.isActive(startAddress) && selection.containsAddress(startAddress)) {
                        rangeIndex = Utils.findFirstIndex(selection.ranges, function (range) { return range.containsAddress(startAddress); });
                        self.trigger('select:active', rangeIndex, startAddress);

                    } else if (!selection.containsAddress(startAddress)) {
                        self.trigger('select:start', startAddress, 'select');
                        self.trigger('select:end', startAddress, true);
                        self.scrollToCell(startAddress);
                    }

                    $('html').trigger('contextmenu');
                }
            }

            // return the actual touchSelectionTrackingHandler() method
            return function (event, endPromise) {
                switch (event.type) {
                    case 'tracking:start':
                        startAddress = getCellAddress(event);
                        moved = false;
                        break;
                    case 'tracking:move':
                        moved = moved || (Utils.radius(event.offsetX, event.offsetY) >= 6);
                        if (moved) { self.scrollRelative(-event.moveX, -event.moveY); }
                        break;
                    case 'tracking:end':
                    case 'tracking:cancel':
                        event.preventDefault(); // prevent native focus setting
                        endPromise.done(function () { finalizeTracking(event); });
                        break;
                }
            };
        }()); // end of touchSelectionTrackingHandler() local scope

        // resize selection range (touch) -------------------------------------

        /**
         * Handles tracking events to resize selected ranges on touch devices.
         */
        var resizeSelectionTrackingHandler = (function () {

            // the position identifier of the tracked node
            var trackerPos = null;
            // the array index of the tracked range
            var rangeIndex = 0;
            // the original address of the tracked range
            var origRange = null;
            // the anchor cell in the original range
            var anchorCell = null;
            // the current address of the tracked range
            var currRange = null;

            // initializes the highlighted range according to the passed tracking event
            function initializeTracking(event) {

                // the handle node currently tracked
                var trackerNode = $(event.target);
                // the root node of the range that will be tracked
                var rangeNode = trackerNode.closest('.range');

                // extract range address
                rangeIndex = Utils.getElementAttributeAsInteger(rangeNode, 'data-index', -1);
                origRange = docView.getSelectedRanges()[rangeIndex];

                // initialize additional data needed while tracking
                trackerPos = trackerNode.attr('data-pos');
                currRange = origRange.clone();
                anchorCell = getAnchorCell(event, origRange, trackerPos, true);
            }

            // updates the highlighted range according to the passed tracking event
            function updateTracking(event) {

                // the new range address
                var newRange = recalcTrackedRange(event, origRange, anchorCell, trackerPos, true);

                // update the tracked range in the respective token array, this will
                // trigger change events which will cause refreshing all grid panes
                if (currRange.differs(newRange)) {
                    currRange = newRange;
                    self.trigger('select:range', rangeIndex, currRange);
                }
            }

            // return the actual resizeSelectionTrackingHandler() method
            return function (event) {
                switch (event.type) {
                    case 'tracking:start':
                        initializeTracking(event);
                        break;
                    case 'tracking:move':
                        updateTracking(event);
                        break;
                    case 'tracking:scroll':
                        self.scrollRelative(event.scrollX, event.scrollY);
                        updateTracking(event);
                        break;
                }
            };
        }()); // end of resizeSelectionTrackingHandler() local scope

        // move/resize highlighted ranges -------------------------------------

        /**
         * Handles all tracking events for highlighted ranges.
         */
        var highlightTrackingHandler = (function () {

            // the position identifier of the tracked node (border or corner handle)
            var trackerPos = null;
            // whether resize tracking is active
            var resize = false;
            // the array index of the tracked highlighted range
            var rangeIndex = null;
            // the original range address
            var origRange = null;
            // the anchor cell in the original range
            var anchorCell = null;
            // the current address of the tracked range
            var currRange = null;

            // initializes the highlighted range according to the passed tracking event
            function initializeTracking(event) {

                // the border/handle node currently tracked
                var trackerNode = $(event.target);
                // the root node of the highlighted range that will be tracked
                var rangeNode = trackerNode.closest('.range');

                // the array index and range address of the tracked range
                rangeIndex = Utils.getElementAttributeAsInteger(rangeNode, 'data-index', -1);
                origRange = docView.getHighlightedRange(rangeIndex);
                if (!origRange) {
                    Utils.warn('GridTrackingMixin.initializeTracking(): cannot find tracking range');
                    event.cancelTracking();
                    return;
                }

                // initialize additional data needed while tracking
                trackerPos = trackerNode.attr('data-pos');
                resize = trackerNode.parent().hasClass('resizers');
                origRange = origRange.toRange(); // convert to simple range without sheet indexes
                currRange = origRange.clone();
                anchorCell = getAnchorCell(event, origRange, trackerPos, resize);

                // render the tracked range with special glow effect
                docView.trackHighlightedRange(rangeIndex);
            }

            // updates the highlighted range according to the passed tracking event
            function updateTracking(event) {

                // the new range address
                var newRange = recalcTrackedRange(event, origRange, anchorCell, trackerPos, resize);

                // update the tracked range in the respective token array, this will
                // trigger change events which will cause refreshing all grid panes
                if (currRange.differs(newRange)) {
                    currRange = newRange;
                    docView.changeHighlightedRange(rangeIndex, currRange);
                }
            }

            // finalizes the highlighted range
            function finalizeTracking() {
                docView.trackHighlightedRange(-1);
            }

            // return the actual highlightTrackingHandler() method
            return function (event, endPromise) {
                switch (event.type) {
                    case 'tracking:start':
                        initializeTracking(event);
                        event.preventDefault();
                        break;
                    case 'tracking:move':
                        updateTracking(event);
                        // prevent Chrome from modifying the text area selection while moving mouse
                        event.preventDefault();
                        break;
                    case 'tracking:scroll':
                        self.scrollRelative(event.scrollX, event.scrollY);
                        updateTracking(event);
                        break;
                    case 'tracking:end':
                    case 'tracking:cancel':
                        endPromise.always(function (success) {
                            finalizeTracking(event);
                            if (!success && origRange) {
                                docView.changeHighlightedRange(rangeIndex, origRange);
                            }
                        });
                        break;
                }
            };
        }()); // end of highlightTrackingHandler() local scope

        // auto fill ----------------------------------------------------------

        /**
         * Handles all tracking events for the auto-fill handle.
         */
        var autoFillTrackingHandler = (function () {

            // shortcut to the model of the active sheet
            var sheetModel = null;
            // the sheet range without leading/trailing hidden columns/rows
            var visibleRange = null;
            // the initial selection range
            var origRange = null;
            // the sheet rectangle covered by the initial selection range
            var origRectangle = null;
            // start position of tracking (bottom-right corner of selection range)
            var startOffset = null;
            // whether to restrict tracking direction (column/row ranges)
            var fixedColumns = null;
            // resulting maximum number of columns that can be filled at the leading/trailing border
            var maxLeadingCols = 0;
            var maxTrailingCols = 0;
            // resulting maximum number of rows that can be filled at the leading/trailing border
            var maxLeadingRows = 0;
            var maxTrailingRows = 0;
            // whether the original range contains at least one merged range
            var hasMergedRange = false;

            // initializes the auto fill handling according to the passed tracking event
            function initializeTracking(event) {

                // cancel custom selection when starting auto-fill
                docView.cancelCustomSelectionMode();

                // the model of the active sheet
                sheetModel = docView.getSheetModel();

                // do not expand auto-fill range into leading/trailing hidden columns/rows
                visibleRange = sheetModel.shrinkRangeToVisible(docModel.getSheetRange());
                if (!visibleRange) { event.cancelTracking(); return; }

                // store address and position of the original selection range
                origRange = docView.getSelectedRanges().first();
                origRectangle = docView.getRangeRectangle(origRange);
                startOffset = { left: origRectangle.right(), top: origRectangle.bottom() };

                // special behavior for column/row ranges
                fixedColumns = docModel.isColRange(origRange) ? true : docModel.isRowRange(origRange) ? false : null;

                // restrict number of columns/rows according to selection mode
                var maxCols = 0, maxRows = 0;
                if (fixedColumns === true) {
                    maxCols = SheetUtils.MAX_AUTOFILL_COL_ROW_COUNT;
                } else if (fixedColumns === false) {
                    maxRows = SheetUtils.MAX_AUTOFILL_COL_ROW_COUNT;
                } else {
                    maxCols = Math.floor(SheetUtils.MAX_FILL_CELL_COUNT / origRange.rows());
                    maxRows = Math.floor(SheetUtils.MAX_FILL_CELL_COUNT / origRange.cols());
                }

                // calculate maximum number of columns/rows that can be filled at once
                maxLeadingCols = Math.min(maxCols, Math.max(0, origRange.start[0] - visibleRange.start[0]));
                maxTrailingCols = Math.min(maxCols, Math.max(0, visibleRange.end[0] - origRange.end[0]));
                maxLeadingRows = Math.min(maxRows, Math.max(0, origRange.start[1] - visibleRange.start[1]));
                maxTrailingRows = Math.min(maxRows, Math.max(0, visibleRange.end[1] - origRange.end[1]));

                // check whether the original range contains at least one merged range
                hasMergedRange = docView.getMergeCollection().rangesOverlapAnyMergedRange(origRange);

                // render selected range with modified auto-fill style
                sheetModel.setViewAttribute('autoFillData', { border: 'bottom', count: 0 });
            }

            // updates the auto fill area according to the passed tracking event
            function updateTracking(event) {

                // current tracking position
                var currOffset = self.getSheetOffsetForEvent(event);
                // whether to delete parts of the selected range
                var deleteMode = origRectangle.containsPixel(currOffset.left, currOffset.top);
                // the adjusted tracking position to determine the direction
                var left = (deleteMode || (currOffset.left >= startOffset.left)) ? currOffset.left : Math.min(currOffset.left + origRectangle.width, startOffset.left);
                var top = (deleteMode || (currOffset.top >= startOffset.top)) ? currOffset.top : Math.min(currOffset.top + origRectangle.height, startOffset.top);
                // whether to expand horizontally or vertically
                var columns = (fixedColumns === null) ? (Math.abs(top - startOffset.top) < Math.abs(left - startOffset.left)) : fixedColumns;
                // whether to expand the leading or trailing border
                var leading = columns ? (left < startOffset.left) : (top < startOffset.top);
                // the tracker position to calculate the target rectangle
                var trackerPos = columns ? (leading ? 'l' : 'r') : (leading ? 't' : 'b');
                // simulate tracking at a specific border of the selected range
                var range = recalcTrackedRange(event, origRange, leading ? origRange.start : origRange.end, trackerPos, false);
                // the new value for the 'autoFillData' attribute
                var autoFillData = {};

                // calculate the auto-fill data from target range
                if (deleteMode) {
                    // delete mode: always delete from trailing border
                    autoFillData.border = columns ? 'right' : 'bottom';
                    autoFillData.count = range.getStart(columns) - origRange.getEnd(columns) - 1;
                } else if (leading) {
                    // expand leading range border
                    autoFillData.border = columns ? 'left' : 'top';
                    autoFillData.count = Math.min(origRange.getStart(columns) - range.getStart(columns), columns ? maxLeadingCols : maxLeadingRows);
                } else {
                    // expand trailing range border
                    autoFillData.border = columns ? 'right' : 'bottom';
                    autoFillData.count = Math.min(range.getEnd(columns) - origRange.getEnd(columns), columns ? maxTrailingCols : maxTrailingRows);
                }

                // if the source range covers a merged range, expand only in multiples of the source range size
                if (!deleteMode && hasMergedRange) {
                    var origSize = origRange.size(columns);
                    var maxCount = leading ? origRange.getStart(columns) : (docModel.getMaxIndex(columns) - origRange.getEnd(columns));
                    autoFillData.count = Math.min(Utils.roundUp(autoFillData.count, origSize), Utils.roundDown(maxCount, origSize));
                }

                sheetModel.setViewAttribute('autoFillData', autoFillData);

                // set correct mouse pointer according to direction
                event.cursor = ((fixedColumns !== null) || (autoFillData.count === 0)) ? null : columns ? 'ew-resize' : 'ns-resize';
            }

            // finalizes the selection tracking
            function finalizeTracking(event, apply) {

                // apply auto-fill operation
                if (apply) {
                    var autoFillData = sheetModel.getViewAttribute('autoFillData');
                    autoFillData.invertType = event.altKey || false;
                    // bug 39533: via controller for busy screen etc.
                    docView.executeControllerItem('cell/autofill', autoFillData);
                }

                // reset the view attribute
                sheetModel.setViewAttribute('autoFillData', null);
                sheetModel = null;
            }

            // return the actual autoFillTrackingHandler() method
            return function (event, endPromise) {

                switch (event.type) {
                    case 'tracking:start':
                        initializeTracking(event);
                        break;
                    case 'tracking:move':
                        updateTracking(event);
                        break;
                    case 'tracking:scroll':
                        self.scrollRelative(event.scrollX, event.scrollY);
                        updateTracking(event);
                        break;
                    case 'tracking:end':
                    case 'tracking:cancel':
                        endPromise.always(function (success) {
                            finalizeTracking(event, success);
                        });
                        break;
                }
            };
        }()); // end of autoFillTrackingHandler() local scope

        // select/move/resize drawings ----------------------------------------

        /**
         * Handles all tracking events for drawing frames.
         */
        var drawingTrackingHandler = (function () {

            // the drawing collection of the active sheet
            var drawingCollection = null;
            // the total size of the sheet, in pixels
            var sheetWidth = 0, sheetHeight = 0;
            // information for all selected drawing objects
            var drawingDescSet = new ValueSet('key');
            // descriptor for the active (tracked) drawing object
            var activeDesc = null;
            // bounding rectangle of all selected drawing objects (in pixels)
            var boundRect = null;
            // tracking start position (absolute position in the sheet, in pixels)
            var startOffset = null;
            // whether the mouse or touch point has really been moved (with threshold distance)
            var mouseMoved = false;
            // whether the drawing object will be moved (false for resize/rotate mode)
            var moveMode = false;
            // position of the tracked resizer handle (null for move mode and rotate mode)
            var resizerPos = null;
            // whether the rotate handle will be tracked (false for move/resize mode)
            var rotateMode = false;
            // whether horizontal/vertical tracking will be processed at all
            var activeX = false, activeY = false;
            // whether resizing in reversed direction (from trailing border) horizontally/vertically
            var reverseX = false, reverseY = false;
            // current move distance in horizontal/vertical direction
            var moveX = 0, moveY = 0;
            // temorary normalized move distance in horizontal/vertical direction used while moving
            var tempMoveX = 0, tempMoveY = 0;
            // minimum and maximum move distances in horizontal direction
            var minMoveX = 0, maxMoveX = 0;
            // minimum and maximum move distances in vertical direction
            var minMoveY = 0, maxMoveY = 0;
            // current resize factor in horizontal/vertical direction
            var scaleX = 1, scaleY = 1;
            // angle of the shape before rotation transformation
            var startAngle = 0;
            // difference between start angle, and angle after rotation applied
            var diffAngle = 0;

            // returns the minimum scaling factor for resizing a specific drawing frame, according to location and sheet size
            function getMinScale(frameOffset, frameSize, sheetSize, reverse) {
                return reverse ? ((frameOffset - sheetSize) / frameSize + 1) : (-frameOffset / frameSize);
            }

            // returns the maximum scaling factor for resizing a specific drawing frame, according to location and sheet size
            function getMaxScale(frameOffset, frameSize, sheetSize, reverse) {
                return reverse ? (frameOffset / frameSize + 1) : ((sheetSize - frameOffset) / frameSize);
            }

            // returns the current location of the passed drawing object, according to the current tracking state
            function getDrawingRectangle(drawingDesc, event) {

                // new offset position and size of the drawing frame
                var rect = drawingDesc.rectangle.clone();
                // if function is triggered on tracking end event
                var isEndEvent = event.type === 'tracking:end';
                // flipping and rotation of the drawing model
                var rotation = drawingDesc.model.getRotationDeg();
                var flipH = drawingDesc.model.isFlippedH();
                var flipV = drawingDesc.model.isFlippedV();

                // adjust offset and size for move mode
                if (moveMode) {
                    rect.left += isEndEvent ? moveX : tempMoveX;
                    rect.top += isEndEvent ? moveY : tempMoveY;
                }

                // adjust offset and size for resize mode
                if (resizerPos) {

                    // effective absolute scaling factors
                    var maxAbsScaleX = activeX ? Math.abs((scaleX < 0) ? -drawingDesc.minScaleX : drawingDesc.maxScaleX) : 1;
                    var maxAbsScaleY = activeY ? Math.abs((scaleY < 0) ? -drawingDesc.minScaleY : drawingDesc.maxScaleY) : 1;
                    var absScaleX = activeX ? Math.min(maxAbsScaleX, Math.abs(scaleX)) : 1;
                    var absScaleY = activeY ? Math.min(maxAbsScaleY, Math.abs(scaleY)) : 1;
                    if (activeX && activeY && (event.shiftKey || drawingDesc.model.isAspectLocked())) {
                        absScaleX = absScaleY = Math.min(maxAbsScaleX, maxAbsScaleY, Math.max(absScaleX, absScaleY));
                    }

                    // scale the drawing size
                    var minSize = drawingDesc.model.getMinSize();
                    rect.width = Math.max(minSize, Math.round(rect.width * absScaleX));
                    rect.height = Math.max(minSize, Math.round(rect.height * absScaleY));

                    // adjust offset (leading/trailing resizers)
                    if (reverseX) { rect.left += drawingDesc.rectangle.width; }
                    if (reverseX !== (scaleX < 0)) { rect.left -= rect.width; }
                    if (reverseY) { rect.top += drawingDesc.rectangle.height; }
                    if (reverseY !== (scaleY < 0)) { rect.top -= rect.height; }

                    // resizing of rotated shape needs position correction
                    if (isEndEvent && ((rotation !== 0) || flipH || flipV)) {
                        DrawingFrame.toggleTracking(drawingDesc.frameNode, true); // temporary enable visibility of tracking node to measure rect positions
                        var diff = DrawingFrame.getPositionDiffAfterResize(drawingDesc.frameNode, drawingDesc.trackerNode, rotation, flipH, flipV);
                        DrawingFrame.toggleTracking(drawingDesc.frameNode, false); // and disable back

                        rect.left = drawingDesc.rectangle.left - diff.leftDiff;
                        rect.top = drawingDesc.rectangle.top - diff.topDiff;
                    }
                }

                return rect;
            }

            // initializes the drawing according to the passed tracking event
            function initializeTracking(event) {

                // ensure that a drawing frame has been hit
                var drawingFrame = self.getDrawingFrameForEvent(event);
                if (drawingFrame.length === 0) {
                    event.cancelTracking();
                    return;
                }

                // position of the tracked resizer handle (null for move/rotate mode, i.e. click into the drawing)
                resizerPos = DrawingFrame.getResizerHandleType(event.target);
                // whether rotation handle is grabbed
                rotateMode = DrawingFrame.isRotateHandle(event.target);
                // whether the drawing will be moved
                moveMode = !resizerPos && !rotateMode;

                // the model of the clicked drawing object
                var activeModel = DrawingFrame.getModel(drawingFrame);
                // whether the clicked drawing frame is already selected
                var selected = DrawingFrame.isSelected(drawingFrame);
                // whether to toggle the selection of the clicked drawing frame
                var toggle = moveMode && (event.shiftKey || event.ctrlKey || event.metaKey);

                var textFrame = TextFrameUtils.getTextFrame(drawingFrame);

                // Bug #53426
                if (_.browser.Firefox && textFrame && (Utils.containsNode(textFrame, event.target) || textFrame[0] === event.target)) {
                    event.preventDefault();
                }

                // clicked into the text area of a selected drawing: start text edit mode (!!! not for android at the moment !!!)
                if (moveMode && selected && !toggle && !_.browser.Android && (docView.getSelectedDrawings().length === 1)) {
                    // BUG #52956:
                    //      you need to click on the text-span, to start text-edit-mode. Otherwise, you will always
                    //      move the drawing.
                    if (textFrame && event.target.nodeName === 'SPAN' && Utils.containsNode(textFrame, event.target)) {
                        event.preventDefault(); // important to prevent that browser "clicks" into the text contents by itself
                        event.cancelTracking();
                        self.grabFocus();
                        docView.enterTextEditMode('drawing', { restart: true });
                        return;
                    }
                }

                // do not modify the selection, if clicked on a selected drawing without modifier key
                if (!selected || toggle) {
                    var drawingPos = activeModel.getPosition();
                    docView.selectDrawing(drawingPos, { toggle: toggle });
                    docView.scrollToDrawingFrame(drawingPos);
                }

                // no tracking after deselecting a selected drawing frame
                if (selected && toggle) {
                    event.cancelTracking();
                    return;
                }

                // check edit mode and sheet protection (after updating the drawing selection!)
                if (!docView.isEditable() || docView.isSheetLocked()) {
                    event.cancelTracking();
                    return;
                }

                // initialize information for all selected drawing objects
                drawingCollection = docView.getDrawingCollection();
                sheetWidth = docView.getSheetWidth();
                sheetHeight = docView.getSheetHeight();
                drawingDescSet.clear();
                boundRect = null;
                diffAngle = tempMoveX = tempMoveY = 0;

                // collect information for all visible drawing objects to be tracked
                docView.getSelectedDrawings().forEach(function (position) {

                    // the model of the drawing object
                    var drawingModel = drawingCollection.getModel(position);
                    // the rectangle (in pixels); or null for hidden drawing objects
                    var rectangle = drawingModel.getRectangle();
                    // the DOM drawing frame for the model
                    var drawingFrame = self.getDrawingFrame(position);

                    // nothing to do for hidden drawing objects, or for missing DOM frames
                    if (!rectangle || !drawingFrame) { return; }

                    // create and store the descriptor for the drawing object
                    drawingDescSet.insert({
                        key: drawingModel.getUid(),
                        model: drawingModel,
                        position: position,
                        rectangle: rectangle,
                        frameNode: drawingFrame,
                        rotatable: drawingModel.isDeepRotatable()
                    });

                    // take rotation of the drawing object into account, update the resulting bounding rectangle
                    rectangle = rectangle.rotatedBoundingBox(drawingModel.getRotationRad());
                    boundRect = boundRect ? boundRect.boundary(rectangle) : rectangle;
                });

                // get the descriptor for the drawing object that will actually be tracked
                activeDesc = drawingDescSet.get(activeModel.getUid());

                // nothing to do without a valid selection
                if (drawingDescSet.empty() || !activeDesc) {
                    event.cancelTracking();
                    return;
                }

                // initialize further tracking data
                startOffset = self.getSheetOffsetForEvent(event);
                mouseMoved = false;
                moveX = moveY = 0;
                scaleX = scaleY = 1;
                startAngle = activeDesc.model.getRotationDeg();

                // decide whether horizontal/vertical tracking is active (always in move/rotate mode)
                activeX = !resizerPos || /[lr]/.test(resizerPos);
                activeY = !resizerPos || /[tb]/.test(resizerPos);

                // decide whether to resize in reversed direction (from trailing border)
                reverseX = resizerPos && /l/.test(resizerPos);
                reverseY = resizerPos && /t/.test(resizerPos);

                // restrict movement of the drawing objects to the sheet area
                if (moveMode) {
                    minMoveX = -boundRect.left;
                    maxMoveX = sheetWidth - boundRect.right();
                    minMoveY = -boundRect.top;
                    maxMoveY = sheetHeight - boundRect.bottom();
                }

                // restrict resizing the drawing objects individually
                if (resizerPos) {
                    drawingDescSet.forEach(function (drawingDesc) {
                        var rect = drawingDesc.rectangle;
                        var drawingModel = drawingDesc.model;
                        var skipRestricting = (drawingModel.getRotationDeg() !== 0) || drawingModel.isFlippedH() || drawingModel.isFlippedV();
                        var minSkipValue = -50000; // TODO: arbitrary big values for min and max, smarter algorithm is required for rotated/flipped shapes
                        var maxSkipValue = 50000; // see bug #52636 for more
                        drawingDesc.minScaleX = skipRestricting ? minSkipValue : getMinScale(rect.left, rect.width, sheetWidth, reverseX);
                        drawingDesc.maxScaleX = skipRestricting ? maxSkipValue : getMaxScale(rect.left, rect.width, sheetWidth, reverseX);
                        drawingDesc.minScaleY = skipRestricting ? minSkipValue : getMinScale(rect.top, rect.height, sheetHeight, reverseY);
                        drawingDesc.maxScaleY = skipRestricting ? maxSkipValue : getMaxScale(rect.top, rect.height, sheetHeight, reverseY);
                    });
                }
            }

            // updates the position of the drawing frames according to the passed tracking event
            function updateTracking(event) {

                // current tracking position (absolute sheet position, in pixels)
                var currOffset = self.getSheetOffsetForEvent(event);
                var deltaX = currOffset.left - startOffset.left;
                var deltaY = currOffset.top - startOffset.top;
                // shift distance in both directions
                var shiftX = activeX ? deltaX : 0;
                var shiftY = activeY ? deltaY : 0;

                if (resizerPos) {
                    var signX = activeDesc.model.isFlippedH() ? -1 : 1;
                    var signY = activeDesc.model.isFlippedV() ? -1 : 1;
                    var normalizedCoords = DrawingFrame.getNormalizedResizeDeltas(deltaX, deltaY, activeX, activeY, (reverseX ? -1 : 1), (reverseY ? -1 : 1), startAngle);
                    shiftX = normalizedCoords.x * signX;
                    shiftY = normalizedCoords.y * signY;
                    scaleX = shiftX / activeDesc.rectangle.width + 1;
                    scaleY = shiftY / activeDesc.rectangle.height + 1;
                }

                // start tracking after a threshold of 3 pixels
                if (!mouseMoved && (Utils.radius(shiftX, shiftY) < 3)) { return; }

                // show tracking frames if mouse has moved the first time (even if the drawings will
                // not really move due to restrictions of the bounding rectangle in the sheet area)
                if (!mouseMoved) {
                    drawingDescSet.forEach(function (drawingDesc) {
                        // skip non-rotatable shapes on rotation
                        if (rotateMode && !drawingDesc.rotatable) { return; }
                        DrawingFrame.toggleTracking(drawingDesc.frameNode, true);
                        drawingDesc.trackerNode = DrawingFrame.getTrackerNode(drawingDesc.frameNode);
                    });
                    mouseMoved = true;
                }

                // restrict to allowed move distances
                if (moveMode) {
                    moveX = Utils.minMax(shiftX, minMoveX, maxMoveX);
                    moveY = Utils.minMax(shiftY, minMoveY, maxMoveY);
                }

                // update the rotate angle according to current tracking shift
                if (rotateMode) {
                    var offsetX = currOffset.left - activeDesc.rectangle.centerX();
                    var offsetY = currOffset.top - activeDesc.rectangle.centerY();
                    var newAngle = Math.round(Math.atan2(offsetY, offsetX) / PI_180) + 90;
                    // make 15 deg step with shift key
                    if (event.shiftKey) { newAngle = Utils.round(newAngle, 15); }
                    if (activeDesc.model.isFlippedV()) { newAngle += 180; }
                    diffAngle = newAngle - startAngle;
                }

                // move the tracker nodes relative to the drawing frames
                drawingDescSet.forEach(function (drawingDesc) {

                    var drawingModel = drawingDesc.model;
                    var flipH = drawingModel.isFlippedH();
                    var flipV = drawingModel.isFlippedV();
                    var trackerAngle = (flipH !== flipV) ? -diffAngle : diffAngle;
                    var transform = '';
                    var rectangle = null;

                    if (moveMode) {
                        var normalizedCoords = DrawingFrame.getNormalizedMoveCoordinates(moveX, moveY, drawingModel.getRotationDeg(), flipH, flipV);
                        tempMoveX = normalizedCoords.x;
                        tempMoveY = normalizedCoords.y;
                    }

                    if (rotateMode) {
                        if (!drawingDesc.rotatable) { return; } // skip non-rotatable shapes on rotation
                        transform = DrawingUtils.getCssTransform(trackerAngle, false, false);
                    }

                    rectangle = getDrawingRectangle(drawingDesc, event);
                    drawingDesc.trackerNode.css({
                        left: rectangle.left - drawingDesc.rectangle.left,
                        top: rectangle.top - drawingDesc.rectangle.top,
                        width: rectangle.width,
                        height: rectangle.height,
                        transform: transform
                    });
                });
            }

            // finalizes the drawing tracking
            function finalizeTracking(event, apply) {

                // deinitialize all tracking nodes
                drawingDescSet.forEach(function (drawingDesc) {
                    DrawingFrame.toggleTracking(drawingDesc.frameNode, false);
                });

                // only generate document operations if the drawing was really moved
                if (apply && mouseMoved) {

                    if (rotateMode) {
                        docView.executeControllerItem('drawing/transform/rotate', diffAngle);
                        return;
                    }

                    var toggleFlipH = scaleX < 0;
                    var toggleFlipV = scaleY < 0;
                    docView.executeControllerItem('drawing/change/explicit', drawingDescSet.map(function (drawingDesc) {

                        var newAttrs = {
                            rotation: drawingDesc.model.getRotationDeg(),
                            flipH: toggleFlipH !== drawingDesc.model.isFlippedH(),
                            flipV: toggleFlipV !== drawingDesc.model.isFlippedV()
                        };

                        // adjust the anchor position according to the rotation angle in OOXML
                        var rectangle = getDrawingRectangle(drawingDesc, event);
                        if (ooxml && drawingDesc.rotatable && DrawingUtils.hasSwappedDimensions(newAttrs)) {
                            rectangle.rotateSelf();
                        }

                        var attrSet = drawingCollection.getAttributeSetForRectangle(rectangle);
                        if (drawingDesc.rotatable) {
                            if (toggleFlipH) { attrSet.drawing.flipH = newAttrs.flipH; }
                            if (toggleFlipV) { attrSet.drawing.flipV = newAttrs.flipV; }
                        }
                        return { position: drawingDesc.position, attrs: attrSet };
                    }));
                }

                // open context menu after a long-tap without move events
                if (Utils.TOUCHDEVICE && apply && !mouseMoved && (event.duration > 750)) {
                    activeDesc.frameNode.trigger('contextmenu');
                }
            }

            // return the actual drawingTrackingHandler() method
            return function (event, endPromise) {

                switch (event.type) {
                    case 'tracking:start':
                        initializeTracking(event);
                        break;
                    case 'tracking:move':
                        updateTracking(event);
                        break;
                    case 'tracking:scroll':
                        self.scrollRelative(event.scrollX, event.scrollY);
                        updateTracking(event);
                        break;
                    case 'tracking:end':
                    case 'tracking:cancel':
                        endPromise.always(function (success) {
                            finalizeTracking(event, success);
                        });
                        break;
                }
            };
        }()); // end of drawingTrackingHandler() local scope

        // --------------------------------------------------------------------

        /**
         * Handles pinch events triggered by the touch framework.
         */
        var pinchHandler = (function () {

            var startDistance;
            var transformScale;
            var originX;
            var originY;
            var originOffsetX;
            var originOffsetY;
            var translateX;
            var translateY;
            var visibleRectangle;
            var scrollSizeNode = layerRootNode.parent();
            var lastPhase;
            var scale;

            function getZoom() {
                return docView.getSheetModel().getZoom();
            }

            function minMaxTransformScale(scale) {
                var currZoom = getZoom();
                var newZoom = PaneUtils.getValidZoom(scale * currZoom);
                return newZoom / currZoom;
            }

            function roundTransformScale(scale) {
                var currZoom = getZoom();
                var newZoom = Utils.round(scale * currZoom, 0.05);
                return newZoom / currZoom;
            }

            function calcOffset(translate, scale, refSize, refMin) {
                var newRefSize = refSize / scale;
                var newRefMin = (refSize - newRefSize) / 2;
                return refMin + newRefMin - (translate * scale);
            }

            function handlePinchEvent(phase, event, distance, midPoint) {
                // workaround for IOS
                if (phase === 'cancel' && lastPhase === 'move') { phase = 'end'; }

                lastPhase = phase;

                switch (phase) {

                    case 'start':
                        event.preventDefault();
                        startDistance = distance;
                        originX = midPoint.x;
                        originY = midPoint.y;

                        var offset = scrollSizeNode.offset();
                        originOffsetX = originX - offset.left;
                        originOffsetY = originY - offset.top;

                        //default values, if no move event is triggered before end event
                        translateX = 0;
                        translateY = 0;
                        transformScale = 1;

                        visibleRectangle = docView.getActiveGridPane().getVisibleRectangle();
                        break;

                    case 'move':
                        event.preventDefault();
                        transformScale = distance / startDistance;
                        transformScale = Utils.minMax(transformScale, 0.5, 2.0);
                        transformScale = minMaxTransformScale(transformScale);
                        translateX = midPoint.x - originX;
                        translateY = midPoint.y - originY;
                        break;

                    case 'end':
                        event.preventDefault();

                        scale = roundTransformScale(transformScale);
                        //round a second time because of imprecise float stuff
                        var zoom = Utils.round(scale * getZoom(), 0.05);

                        if (zoom < 0.1) {
                            //should never be here :)
                            Utils.error('pinch to zoom is broken scale: ' + scale + ' transformScale: ' + transformScale + ' zoom: ' + getZoom());
                            return;
                        }

                        var sheetModel = docView.getSheetModel();
                        var colCollection = sheetModel.getColCollection();
                        var rowCollection = sheetModel.getRowCollection();

                        var attributes = {};

                        var offsetX = calcOffset(translateX, scale, visibleRectangle.width, visibleRectangle.left);
                        var offsetY = calcOffset(translateY, scale, visibleRectangle.height, visibleRectangle.top);

                        attributes.anchorRight = colCollection.getScrollAnchorByOffset(offsetX, { pixel: true });
                        attributes.anchorBottom = rowCollection.getScrollAnchorByOffset(offsetY, { pixel: true });
                        attributes.zoom = zoom;

                        sheetModel.setViewAttributes(attributes);
                        break;

                    case 'cancel':
                        break;
                }
            }

            function updateCssTransformation() {
                switch (lastPhase) {
                    case 'start':
                        Utils.setCssAttributeWithPrefixes(scrollSizeNode, 'transform-origin', originOffsetX + 'px ' + originOffsetY + 'px');
                        Utils.setCssAttributeWithPrefixes(scrollSizeNode, 'transform', 'translate3d(0px,0px,0) scale(1)');
                        break;

                    case 'move':
                        // if move phase comes too fast, start phase is skipped, so we always set origin (because we use 3 event types in one event)
                        Utils.setCssAttributeWithPrefixes(scrollSizeNode, 'transform-origin', originOffsetX + 'px ' + originOffsetY + 'px');
                        Utils.setCssAttributeWithPrefixes(scrollSizeNode, 'transform', 'translate3d(' + translateX + 'px,' + translateY + 'px,0) scale(' + transformScale + ')');
                        break;

                    case 'end':
                    case 'cancel':
                        Utils.setCssAttributeWithPrefixes(scrollSizeNode, 'transform', '');
                        Utils.setCssAttributeWithPrefixes(scrollSizeNode, 'transform-origin', '');
                        startDistance = transformScale = null;
                        originX = originY = originOffsetX = originOffsetY = translateX = translateY = null;
                        visibleRectangle = lastPhase = scale = null;
                        break;
                }
            }

            return self.createDebouncedMethod('GridTrackingMixin.pinchHandler', handlePinchEvent, updateCssTransformation, { animFrame: true });
        }());

        // initialization -----------------------------------------------------

        // set new selection, or extend current selection with mouse and modifier keys (prevent scrolling, Chrome modifies the text area selection while moving mouse)
        this.registerTrackingHandler('mouseselect', function (event, lastType) { return lastType !== 'touchselect'; }, mouseSelectionTrackingHandler, { readOnly: true, keepTextEdit: true, preventScroll: true });
        // touch devices: scroll, or set new selection
        this.registerTrackingHandler('touchselect', function (event) { return event.trackingType === 'touch'; }, touchSelectionTrackingHandler, { readOnly: true, keepTextEdit: true, preventScroll: true });
        // change current selection range with resizer handles on touch devices
        this.registerTrackingHandler('resizeselect', '.selection-layer .select [data-pos]', resizeSelectionTrackingHandler, { readOnly: true, keepTextEdit: true, preventScroll: true });
        // auto-fill: resize active selection range for auto-fill and auto-delete
        this.registerTrackingHandler('autofill', '.selection-layer .autofill [data-pos]', autoFillTrackingHandler);
        // highlight layer: move and resize highlighted cell ranges
        this.registerTrackingHandler('highlight', '.highlight-layer [data-pos]', highlightTrackingHandler, { keepTextEdit: true });
        // drawing layer: select, move, and resize drawing objects
        this.registerTrackingHandler('drawing', function (event) { return self.getDrawingFrameForEvent(event).length > 0; }, drawingTrackingHandler, { readOnly: true, preventScroll: true });

        // handle double clicks triggered by the tracking framework
        this.listenToWhenVisible(layerRootNode, 'tracking:dblclick', function () {
            if (self.getLastTrackingType() === 'mouseselect') {
                self.trigger('select:dblclick');
            }
        });

        // suppress native browser events
        layerRootNode.on('dblclick dragenter dragexit dragover dragleave', false);

        layerRootNode.enableTouch({ pinchHandler: pinchHandler });

        // destroy all class members on destruction
        this.registerDestructor(function () {
            layerRootNode.disableTouch();
            self = docModel = docView = layerRootNode = null;
        });

    } // class GridTrackingMixin

    // exports ================================================================

    return GridTrackingMixin;

});
