/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Stefan Eckert <stefan.eckert@open-xchange.com>
 */

define('io.ox/office/spreadsheet/view/chartcreator', [
    'io.ox/office/tk/utils',
    'io.ox/office/tk/render/rectangle',
    'io.ox/office/spreadsheet/utils/sheetutils',
    'io.ox/office/spreadsheet/utils/operations',
    'io.ox/office/spreadsheet/model/formula/tokenarray',
    'io.ox/office/drawinglayer/view/chartstyleutil',
    'gettext!io.ox/office/spreadsheet/main'
], function (Utils, Rectangle, SheetUtils, Operations, TokenArray, ChartStyleUtil, gt) {

    'use strict';

    // convenience shortcuts
    var Address = SheetUtils.Address;
    var Range = SheetUtils.Range;
    var Range3D = SheetUtils.Range3D;
    var Range3DArray = SheetUtils.Range3DArray;

    // default width of new chart objects
    var CHART_DEFAULT_WIDTH = Utils.convertHmmToLength(15000, 'px', 1);

    // default height of new chart objects
    var CHART_DEFAULT_HEIGHT = Utils.convertHmmToLength(10000, 'px', 1);

    var DEFAULT_LINECOLOR = { transformations: [{ type: 'lumMod', value: 15000 }, { type: 'lumOff', value: 85000 }], type: 'scheme', value: 'text1' };
    var STANDARD_SHAPE = { type: 'solid', color: DEFAULT_LINECOLOR, width: 50 };
    var NONE_SHAPE = { type: 'none', color: null, width: null };
    var HEAD_CHAR = { fontSize: 16, bold: false };

    var INSERT_DEFAULTS = (function () {

        var AXIS_ENABLED =  { axis: { label: true },  line: NONE_SHAPE };
        var AXIS_DISABLED = { axis: { label: false }, line: NONE_SHAPE };
        var GRID_ENABLED =  { line: STANDARD_SHAPE };
        var GRID_DISABLED = { line: NONE_SHAPE };

        return {
            normal: {
                xAxis: AXIS_ENABLED,
                xGrid: GRID_DISABLED,
                yAxis: AXIS_ENABLED,
                yGrid: GRID_ENABLED
            },
            xy: {
                xAxis: AXIS_ENABLED,
                xGrid: GRID_ENABLED,
                yAxis: AXIS_ENABLED,
                yGrid: GRID_ENABLED
            },
            pie: {
                xAxis: AXIS_DISABLED,
                xGrid: GRID_DISABLED,
                yAxis: AXIS_DISABLED,
                yGrid: GRID_DISABLED
            }
        };
    }());

    function getFormula(docModel, sheet, chart, from, to) {
        var range = new Range(from, to);

        if (chart) {
            // workaround for Bug 52802

            var formula = null;

            _.times(chart.getSeriesCount(), function (i) {
                var seriesData = chart.getSeriesModel(i);
                seriesData.iterateTokenArrays(function (tokenArray, linkKey) {
                    if (formula) { return; }

                    var oldRangeList = seriesData.resolveRanges(linkKey);
                    if (oldRangeList.empty()) { return; }
                    var oldRange = oldRangeList.first();

                    if (oldRange.start.equals(range.start) && oldRange.end.equals(range.end)) {
                        formula = tokenArray.getFormula('op');
                    }
                });
            });

            if (formula) {
                return formula;
            }
        }

        var tokenArray = new TokenArray(docModel, 'chart');
        tokenArray.appendRange(range, { sheet: sheet, abs: true });
        return tokenArray.getFormula('op');
    }

    function addDataSeries(docModel, generator, options) {

        var sourceSheet = options.sourceSheet;
        var position = options.position;
        var series = options.series;
        var keyFrom = options.keyFrom;
        var keyTo = options.keyTo;
        var title = options.title;
        var valueFrom = options.valueFrom;
        var valueTo = options.valueTo;
        var bubbleFrom = options.bubbleFrom;
        var bubbleTo = options.bubbleTo;
        var bubbleArray = options.bubbleArray;
        var lineAttrs = options.lineAttrs;
        var seriesData = options.seriesData;
        var colors = options.colors;
        var sourceChart = options.sourceChart;

        var insert = {};
        insert.values = getFormula(docModel, sourceSheet, sourceChart, valueFrom, valueTo);
        insert.type = seriesData.type;
        insert.dataLabel = seriesData.dataLabel;

        if (title) {
            insert.title = getFormula(docModel, sourceSheet, sourceChart, title);
        }
        if (keyFrom) {
            insert.names = getFormula(docModel, sourceSheet, sourceChart, keyFrom, keyTo);
        }
        if (bubbleFrom) {
            insert.bubbles = getFormula(docModel, sourceSheet, sourceChart, bubbleFrom, bubbleTo);
        } else if (bubbleArray) {
            insert.bubbles = bubbleArray;
        }

        var properties = {
            series: series,
            attrs: { series: insert, fill: { type: 'solid' } }
        };

        // TODO: _.clone( should not be needed, but it is, why?!
        if (lineAttrs) {
            properties.attrs.line = _.clone(lineAttrs);
        } else if (seriesData.markersOnly) {
            properties.attrs.line = _.copy(NONE_SHAPE, true);
        } else {
            properties.attrs.line = _.copy(STANDARD_SHAPE, true);
        }

        if (colors) {
            if (colors.length > 1) {
                var dataPoints = [];
                colors.forEach(function (color) {
                    dataPoints.push({
                        fill: { type: 'solid', color: color },
                        line: { type: 'solid', color: color }
                    });
                });
                properties.attrs.series.dataPoints = dataPoints;
            }
            properties.attrs.line.color = colors[0];
            properties.attrs.fill.color = colors[0];
        }
        generator.generateDrawingOperation(Operations.INSERT_CHART_DATASERIES, position, properties);
    }

    function generateSeriesOperations(options) {

        var docModel = options.docModel;
        var sourceRange = options.sourceRange;
        var sourceSheet = options.sourceSheet;
        var generator = options.generator;
        var position = options.position;
        var cells = options.cells;
        var axis = options.axis;
        var chart = options.chart;
        var seriesData = options.seriesData;
        var lineAttrs = options.lineAttrs;
        var sourceChart = options.sourceChart;

        if (!seriesData.colors.length) {
            chart.varyColors = true;
        }

        var start = sourceRange.start;
        var end = sourceRange.end;

        var useAxis;
        if (axis !== 0 && axis !== 1) {
            if (cells[0] && !cells[1]) {
                useAxis = 0;
            } else if (!cells[0] && cells[1]) {
                useAxis = 1;
            } else {
                var width = end[0] - start[0];
                var height = end[1] - start[1];
                if (width > height) {
                    useAxis = 1;
                } else {
                    useAxis = 0;
                }
            }
        } else {
            useAxis = axis;
        }

        var seriesCount;
        if (seriesData.type.indexOf('bubble') === 0) {
            seriesCount = generateBubbleSeriesOperations(docModel, sourceRange, sourceSheet, generator, position, cells, useAxis, seriesData, lineAttrs, sourceChart);
        } else {
            seriesCount = generateStandardSeriesOperations(docModel, sourceRange, sourceSheet, generator, position, cells, useAxis, seriesData, lineAttrs, sourceChart);
        }

        // because of the curved attribute it has to be sent AFTER the series
        generator.generateDrawingOperation(Operations.CHANGE_DRAWING, position, { attrs: { chart: chart } });

        return seriesCount;
    }

    /**
     * makes an address array, needed for the code, which does not know if its row or column oriented
     */
    function makeAddress(activeAxis, otherAxis, activeValue, otherValue) {
        var address = Address.A1.clone();
        address[activeAxis] = activeValue;
        address[otherAxis] = otherValue;
        return address;
    }

    function prepareSeriesColors(docModel, sourceColors, seriesCount, dataPointsCount, colorSetIndex, pieDonut) {

        var count = 0;
        var single = seriesCount === 1 || pieDonut;

        var varyColors = true;
        if (single) {
            count = dataPointsCount;
            if (count > 1 && sourceColors.length > 1 && !ChartStyleUtil.isAutoColor(sourceColors[0]) && _.isEqual(sourceColors[0], sourceColors[1])) {
                varyColors = false;
            }
        } else {
            count = seriesCount;
        }

        var colorSet = null;
        var newColors = null;
        if (Utils.isFiniteNumber(colorSetIndex)) {
            colorSet = ChartStyleUtil.getColorSet()[colorSetIndex];
            newColors = [];
        } else {
            colorSet = ChartStyleUtil.getColorSet()[1];
            newColors = sourceColors.slice(0);
        }

        for (var index = newColors.length; index < count; index++) {
            var color = null;
            if (varyColors) {
                color = ChartStyleUtil.getColorOfPattern('cycle', colorSet.type, index, colorSet.colors, count, docModel);
            } else {
                color = ChartStyleUtil.getColorOfPattern('cycle', colorSet.type, 1, colorSet.colors, 3, docModel);
            }
            newColors.push(color);
        }

        var result = [];
        if (single) {
            for (var i = 0; i < seriesCount; i++) {
                result.push(newColors);
            }
        } else {
            newColors.forEach(function (color) { result.push([color]); });
        }
        return result;
    }

    /**
     * generate all series from start to end.
     * Depending if first row/column contains a String,
     * it is defined as Names (datapoint) and Titles (dataseries)
     *
     * @param {Array<Boolean>} isNumber
     *  first is a Boolean for the cell under the first cell
     *  second is a Boolean for the cell right of the first cell
     *
     * @param {Number} activeAxis
     *  [1 or 0] column or row for the direction of any series data
     */
    function generateStandardSeriesOperations(docModel, sourceRange, sourceSheet, generator, position, cells, activeAxis, seriesData, lineAttrs, sourceChart) {

        var start = sourceRange.start;
        var end = sourceRange.end;

        var max = end[activeAxis] - start[activeAxis];
        var otherAxis = 1 - activeAxis;

        var keyFrom, keyTo;

        if (!cells[activeAxis]) {
            keyFrom = null;
            keyTo = null;
            start[activeAxis] -= 1;
            max += 1;
        } else {
            if (!cells[otherAxis]) {
                keyFrom = start;
            } else {
                keyFrom = makeAddress(activeAxis, otherAxis, start[activeAxis], start[otherAxis] + 1);
            }
            keyTo = makeAddress(activeAxis, otherAxis, start[activeAxis], end[otherAxis]);
        }

        var seriesColors = prepareSeriesColors(docModel, seriesData.colors, max, (end[otherAxis] - start[otherAxis]) + 1, seriesData.colorSet, /^(pie|donut)/.test(seriesData.type));

        for (var i = 0; i < max; i++) {
            var activeIndex = start[activeAxis] + 1 + i;

            var title;
            var valueFrom;
            if (!cells[otherAxis]) {
                title = null;
                valueFrom = makeAddress(activeAxis, otherAxis, activeIndex, start[otherAxis]);
            } else {
                title = makeAddress(activeAxis, otherAxis, activeIndex, start[otherAxis]);
                valueFrom = makeAddress(activeAxis, otherAxis, activeIndex, start[otherAxis] + 1);
            }
            var valueTo = makeAddress(activeAxis, otherAxis, activeIndex, end[otherAxis]);

            addDataSeries(docModel, generator, {
                sourceSheet: sourceSheet,
                position: position,
                series: i,
                keyFrom: keyFrom,
                keyTo: keyTo,
                title: title,
                valueFrom: valueFrom,
                valueTo: valueTo,
                seriesData: seriesData,
                lineAttrs: lineAttrs,
                colors: seriesColors[i],
                sourceChart: sourceChart
            });
        }
        return max;
    }

    /**
     * generate all bubble-series from start to end.
     * Every second row/columns are values, the others are bubbles (sizes).
     * Depending if first row/column contains a String,
     * it is defined as Titles (dataseries).
     * If the count of the series is odd,
     * the first row/column is defined as Names (datapoints)
     *
     * @param {Array<Boolean>} isNumber
     *  first is a Boolean for the cell under the first cell
     *  second is a Boolean for the cell right of the first cell
     *
     * @param {Number} activeAxis
     *  [1 or 0] column or row for the direction of any series data
     */
    function generateBubbleSeriesOperations(docModel, sourceRange, sourceSheet, generator, position, cells, activeAxis, seriesData, lineAttrs, sourceChart) {

        var start = sourceRange.start;
        var end = sourceRange.end;

        var max = end[activeAxis] - start[activeAxis];
        var otherAxis = (activeAxis - 1) * -1;

        var keyFrom, keyTo;

        if (!cells[activeAxis]) {
            keyFrom = null;
            keyTo = null;
            start[activeAxis] -= 1;
            max += 1;
        } else {
            if (!cells[otherAxis]) {
                keyFrom = start;
            } else {
                keyFrom = makeAddress(activeAxis, otherAxis, start[activeAxis], start[otherAxis] + 1);
            }
            keyTo = makeAddress(activeAxis, otherAxis, start[activeAxis], end[otherAxis]);
        }

        var size = 0;

        var seriesColors = prepareSeriesColors(docModel, seriesData.colors, max / 2, (end[otherAxis] - start[otherAxis]) + 1, seriesData.colorSet);

        for (var i = 0; i < max; i += 2) {
            var activeIndex = start[activeAxis] + 1 + i;

            var title;
            var valueFrom;
            if (!cells[otherAxis]) {
                title = null;
                valueFrom = makeAddress(activeAxis, otherAxis, activeIndex, start[otherAxis]);
            } else {
                title = makeAddress(activeAxis, otherAxis, activeIndex, start[otherAxis]);
                valueFrom = makeAddress(activeAxis, otherAxis, activeIndex, start[otherAxis] + 1);
            }
            var valueTo = makeAddress(activeAxis, otherAxis, activeIndex, end[otherAxis]);

            var bubbleFrom = null;
            var bubbleTo = null;
            var bubbleArray = null;

            if (i === max - 1) {
                if (docModel.getApp().isODF()) {
                    // keep old bubble from (bubbleArray not supported yet in ODF)
                } else {
                    bubbleArray = [];
                    var arraySize = 1 + (valueTo[otherAxis] - valueFrom[otherAxis]);
                    for (var j = 0; j < arraySize; j++) {
                        bubbleArray.push(1);
                    }
                }
            } else {
                bubbleFrom = makeAddress(activeAxis, otherAxis, valueFrom[activeAxis] + 1, valueFrom[otherAxis]);
                bubbleTo = makeAddress(activeAxis, otherAxis, valueTo[activeAxis] + 1, valueTo[otherAxis]);
            }

            addDataSeries(docModel, generator, {
                sourceSheet: sourceSheet,
                position: position,
                series: size,
                keyFrom: keyFrom,
                keyTo: keyTo,
                title: title,
                valueFrom: valueFrom,
                valueTo: valueTo,
                bubbleFrom: bubbleFrom,
                bubbleTo: bubbleTo,
                bubbleArray: bubbleArray,
                seriesData: seriesData,
                lineAttrs: lineAttrs,
                colors: seriesColors[i / 2],
                sourceChart: sourceChart
            });
            size++;
        }
        return size;
    }

    function getContentForRange(docModel, docView, sourceRange, sourceSheet, axis, forceTitle, forceNames) {

        var sheetModel = docModel.getSheetModel(sourceSheet);

        var start = sourceRange.start;
        var cols = sourceRange.cols();
        var rows = sourceRange.rows();

        if (cols > 50 || rows > 50) {
            docView.yell({ type: 'warning', message: gt('It is not possible to create a chart out of more than 50 input cells.') });
            return $.Deferred().reject();
        }

        var rightEntry = sheetModel.getColCollection().getNextVisibleEntry(start[0] + 1);
        var bottEntry = sheetModel.getRowCollection().getNextVisibleEntry(start[1] + 1);

        var rightAddress = new Address(rightEntry ? rightEntry.index : start[0], start[1]);
        var bottAddress = new Address(start[0], bottEntry ? bottEntry.index : start[1]);

        var ranges = new Range3DArray(
            Range3D.createFromAddress(bottAddress, sourceSheet),
            Range3D.createFromAddress(rightAddress, sourceSheet),
            Range3D.createFromAddress(start, sourceSheet)
        );

        // first array element of the result is an array with the contents of the three cells
        var res = docModel.getRangeContents(ranges, { blanks: true, attributes: true, display: true });
        var bottomCell = res[0];
        var rightCell = res[1];
        var startCell = res[2];

        function cellIsLabel(cell) {
            if (!startCell) { return false; }
            if (!startCell.display) { return true; } // empty strings (blank cells) or null values (invalid number format)
            return cell.format.isAnyDateTime() || !_.isNumber(cell.value);
        }

        var bottom = false;
        var right = false;

        if (cols === 1 || rows === 1) {
            if (axis !== 0 && axis !== 1) {
                if (cols === 1) {
                    axis = 0;
                } else {
                    axis = 1;
                }
            }
            var firstString = !_.isNumber(startCell.value);
            if (axis === 1) {
                if (firstString) {
                    bottom = true;
                }
            } else {
                if (firstString) {
                    right = true;
                }
            }
        } else {
            bottom = cellIsLabel(bottomCell);
            right = cellIsLabel(rightCell);
        }

        if (!_.isUndefined(forceTitle)) {
            if (axis === 0) {
                right = forceTitle;
            } else if (axis === 1) {
                bottom = forceTitle;
            }
        }
        if (!_.isUndefined(forceNames)) {
            if (axis === 0) {
                bottom = forceNames;
            } else if (axis === 1) {
                right = forceNames;
            }
        }
        if (cols === 1) {
            bottom = false;
        }
        if (rows === 1) {
            right = false;
        }
        return $.when({ cells: [bottom, right], axis: axis });
    }

    function prepareFallbackValueAttrs(docModel, attrs) {
        if (!attrs) {
            return;
        }
        prepareFallbackValueShape(docModel, attrs.line);
        prepareFallbackValueShape(docModel, attrs.fill);

        return attrs;
    }

    function prepareFallbackValueShape(docModel, shape) {
        if (!shape) {
            return;
        }
        if (shape.color && shape.color.type === 'scheme') {
            shape.color.fallbackValue = docModel.parseAndResolveColor(shape.color, 'fill').hex;
        }
        return shape;
    }

    function generatePieUndoRedoOperation(generator, name, position, axis, attrs, undoAttrs) {
        if (_.isEmpty(undoAttrs)) {
            undoAttrs = attrs;
        }
        generator.generateDrawingOperation(name, position, { axis: axis, attrs: attrs });
        generator.generateDrawingOperation(name, position, { axis: axis, attrs: undoAttrs }, { undo: true });
    }

    function generatePieAxisOperations(generator, position, axId, clonedAxes) {
        var cloneAxis = clonedAxes[axId];

        generatePieUndoRedoOperation(generator, Operations.SET_CHART_AXIS_ATTRIBUTES, position, axId, INSERT_DEFAULTS.pie[axId + 'Axis'], cloneAxis.axis);

        if (!_.isEmpty(cloneAxis.grid)) {
            generatePieUndoRedoOperation(generator, Operations.SET_CHART_GRIDLINE_ATTRIBUTES, position, axId, INSERT_DEFAULTS.pie[axId + 'Grid'], cloneAxis.grid);
        }

        if (!_.isEmpty(cloneAxis.title)) {
            generator.generateDrawingOperation(Operations.SET_CHART_TITLE_ATTRIBUTES, position, { axis: axId, attrs: { text: { link: null } } });
            generator.generateDrawingOperation(Operations.SET_CHART_TITLE_ATTRIBUTES, position, { axis: axId, attrs: cloneAxis.title }, { undo: true });
        }
    }

    // static class ChartCreator ==============================================

    var ChartCreator = {};

    // static methods ---------------------------------------------------------

    /**
     * Creates and inserts a new chart into the active sheet.
     *
     * @returns {jQuery.Promise}
     *  A promise that will be resolved after the chart has been created and
     *  inserted successfully, or that will be rejected on any error.
     */
    ChartCreator.createChart = function (app, chartData) {

        var chartAttrs = {
            stacking: chartData.stacking,
            curved: chartData.curved
        };

        var seriesData = {
            type: chartData.series.type,
            markersOnly: chartData.series.markersOnly,
            colors: []
        };

        var docModel = app.getModel();
        var docView = app.getView();
        var gridPane = docView.getActiveGridPane();

        var drawingCollection = docView.getDrawingCollection();

        var range = docView.getActiveRange();

        var forceNames;
        var axis;

        if (seriesData.type.indexOf('bubble') === 0 && Math.min(range.cols(), range.rows()) > 2) {
            forceNames = true;
            if (range.cols() > range.rows()) {
                axis = 1;
            } else {
                axis = 0;
            }
        }

        var promise = getContentForRange(docModel, docView, range, docView.getActiveSheet(), axis, undefined, forceNames);

        promise = promise.then(function (data) {

            var visibleRect = gridPane.getVisibleRectangle();

            var chartRect = new Rectangle(
                Math.max(0, Math.round(visibleRect.left + (visibleRect.width - CHART_DEFAULT_WIDTH) / 2)),
                Math.max(0, Math.round(visibleRect.top + (visibleRect.height - CHART_DEFAULT_HEIGHT) / 2)),
                CHART_DEFAULT_WIDTH,
                CHART_DEFAULT_HEIGHT
            );

            var drawingDesc = { type: 'chart', attrs: drawingCollection.getAttributeSetForRectangle(chartRect) };
            drawingDesc.attrs.fill = { type: 'solid', color: { type: 'scheme', value: 'light1' } };
            drawingDesc.attrs.line = { type: 'none' };

            return app.getController().execInsertDrawings([drawingDesc], function (generator, sheet, position) {

                if (!chartAttrs.chartStyleId) {
                    chartAttrs.chartStyleId = 2;
                }

                generateSeriesOperations({ docModel: docModel, sourceRange: range, sourceSheet: sheet, generator: generator, position: position, cells: data.cells, axis: data.axis, chart: chartAttrs, seriesData: seriesData });

                generator.generateDrawingOperation(Operations.SET_CHART_TITLE_ATTRIBUTES, position, { axis: 'main', attrs: { character: { fontSize: 1 }, text: { link: [' '] } } });
                generator.generateDrawingOperation(Operations.SET_CHART_LEGEND_ATTRIBUTES, position, { attrs: { legend: { pos:  'bottom' } } });

                var defaults;
                var type = seriesData.type.replace('2d', '').replace('3d', '');
                switch (type) {
                    case 'bubble':
                    case 'scatter':
                        defaults = INSERT_DEFAULTS.xy;
                        break;
                    case 'pie':
                    case 'donut':
                        defaults = INSERT_DEFAULTS.pie;
                        break;
                    default:
                        defaults = INSERT_DEFAULTS.normal;
                }

                generator.generateDrawingOperation(Operations.SET_CHART_AXIS_ATTRIBUTES, position, { axis: 'x', attrs: prepareFallbackValueAttrs(docModel, defaults.xAxis) });
                generator.generateDrawingOperation(Operations.SET_CHART_AXIS_ATTRIBUTES, position, { axis: 'y', attrs: prepareFallbackValueAttrs(docModel, defaults.yAxis) });
                generator.generateDrawingOperation(Operations.SET_CHART_GRIDLINE_ATTRIBUTES, position, { axis: 'x', attrs: prepareFallbackValueAttrs(docModel, defaults.xGrid) });
                generator.generateDrawingOperation(Operations.SET_CHART_GRIDLINE_ATTRIBUTES, position, { axis: 'y', attrs: prepareFallbackValueAttrs(docModel, defaults.yGrid) });
            });
        });

        return docView.yellOnFailure(promise);
    };

    /**
     *
     * @param {Object} options
     *  @param {Range} [options.range]
     *      whole start to end area of the new series
     *
     *  @param {Number} [options.sheet]
     *
     *  @param {ChartModel} [options.chartModel]
     *      is used for the position and id in the document and as fallback for chart-type-checks
     *
     *  @param {String} [options.axis]
     *      series direction
     *
     *  @param {Object} [options.chartData]
     *      is used for changing other attributes of the chart and for the type info (bubble or not)
     *
     *  @param {Boolean} [options.forceTitle]
     *      is set, it overwrites the internal logic for title data
     *
     *  @param {Boolean} [options.forceNames]
     *      is set, it overwrites the internal logic for names data
     *
     * @returns {jQuery.Promise}
     *  A promise that will be resolved after the chart has been changed
     *  successfully, or that will be rejected on any error.
     */
    ChartCreator.updateSeries = function (options) {

        var app         = Utils.getOption(options, 'app');
        var sourceRange = Utils.getOption(options, 'sourceRange');
        var sourceSheet = Utils.getOption(options, 'sourceSheet');
        var chartModel  = Utils.getOption(options, 'chartModel');
        var axis        = Utils.getOption(options, 'axis');
        var chartData   = Utils.getOption(options, 'chartData');
        var forceTitle  = Utils.getOption(options, 'forceTitle');
        var forceNames  = Utils.getOption(options, 'forceNames');
        // var typeChanged = Utils.getOption(options, 'typeChanged');

        var chartAttrs = null;
        var seriesData = null;
        var mergedAttributes = chartModel.getMergedAttributeSet(true);

        if (!chartData) {
            //CURVED is a workaround, filter has not all the information it needs!
            chartAttrs = {
                stacking: mergedAttributes.chart.stacking,
                curved: mergedAttributes.chart.curved
            };
            seriesData = {
                type: chartModel.getChartType().split(' ')[0],
                markersOnly: chartModel.isMarkerOnly()
            };
        } else {
            chartAttrs = {
                stacking: chartData.stacking,
                curved: chartData.curved
            };
            seriesData = chartData.series;
        }

        seriesData.colors = chartModel.getSeriesColorInfo();
        seriesData.dataLabel = chartModel.getDataLabel();
        chartAttrs.chartStyleId = mergedAttributes.chart.chartStyleId;
        if (chartAttrs.chartStyleId) {
            seriesData.colorSet = chartModel.getColorSet().replace('cs', '') | 0;
        } else {
            seriesData.colorSet = null;
        }

        var clonedData = chartModel.getCloneData();
        var undoAttrs = chartModel.getUndoAttributeSet({ chart: chartAttrs });

        var docModel = app.getModel();
        var docView = app.getView();

        var promise = getContentForRange(docModel, docView, sourceRange, sourceSheet, axis, forceTitle, forceNames);

        promise = promise.then(function (data) {

            //only overwritten if source range height is 1 oder width is 1
            if (data.axis === 0 || data.axis === 1) {
                axis = data.axis;
            }
            var position = chartModel.getPosition();
            var lineAttrs = null;

            return chartModel.getSheetModel().createAndApplyOperations(function (generator) {

                var oldCount = chartModel.getSeriesCount();
                var seriesCount = generateSeriesOperations({ docModel: docModel, sourceRange: sourceRange, sourceSheet: sourceSheet, generator: generator, position: position, cells: data.cells, axis: axis, chart: chartAttrs, seriesData: seriesData, lineAttrs: lineAttrs, sourceChart: chartModel });

                _.times(oldCount, function () {
                    generator.generateDrawingOperation(Operations.DELETE_CHART_DATASERIES, position, { series: seriesCount });
                });

                // UNDO ---
                generator.generateDrawingOperation(Operations.CHANGE_DRAWING, position, { attrs: undoAttrs }, { undo: true });

                _.each(clonedData.series, function (obj, index) {
                    generator.generateDrawingOperation(Operations.INSERT_CHART_DATASERIES, position, { series: index, attrs: obj }, { undo: true });
                });

                _.times(seriesCount, function () {
                    generator.generateDrawingOperation(Operations.DELETE_CHART_DATASERIES, position, { series: oldCount }, { undo: true });
                });

                if (/^(pie|donut)/.test(seriesData.type)) {
                    generatePieAxisOperations(generator, position, 'x', clonedData.axes);
                    generatePieAxisOperations(generator, position, 'y', clonedData.axes);
                }

                // callback MUST return a promise
                return $.when();
            }, { storeSelection: true });
        });

        return promise;
    };

    ChartCreator.generateOperationsFromModelData = function (generator, position, chartModelData) {

        var axes = chartModelData.axes;
        var series = chartModelData.series;
        var title = chartModelData.title;
        var legend = chartModelData.legend;

        _.each(series, function (serie, index) {
            var properties = {
                series: index,
                attrs: serie
            };
            generator.generateDrawingOperation(Operations.INSERT_CHART_DATASERIES, position, properties);
        });

        _.each(axes, function (axisData, name) {
            if (!_.isEmpty(axisData.axis)) {
                generator.generateDrawingOperation(Operations.SET_CHART_AXIS_ATTRIBUTES, position, { axis: name, attrs: axisData.axis });
            }
            if (!_.isEmpty(axisData.grid)) {
                generator.generateDrawingOperation(Operations.SET_CHART_GRIDLINE_ATTRIBUTES, position, { axis: name, attrs: axisData.grid });
            }
            if (!_.isEmpty(axisData.title)) {
                generator.generateDrawingOperation(Operations.SET_CHART_TITLE_ATTRIBUTES, position, { axis: name, attrs: axisData.title });
            }
        });

        if (!_.isEmpty(title)) {
            generator.generateDrawingOperation(Operations.SET_CHART_TITLE_ATTRIBUTES, position, { axis: 'main', attrs: title });
        }

        if (!_.isEmpty(legend)) {
            generator.generateDrawingOperation(Operations.SET_CHART_LEGEND_ATTRIBUTES, position, { attrs: legend });
        }

    };

    ChartCreator.getStandardShape = function (docModel) {
        return prepareFallbackValueShape(docModel, STANDARD_SHAPE);
    };

    ChartCreator.getHeadChar = function (/*docModel*/) {
        return HEAD_CHAR;
    };

    ChartCreator.getNoneShape = function () {
        return NONE_SHAPE;
    };

    // exports ================================================================

    return ChartCreator;

});
