/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/spreadsheet/utils/rangeset', [
    'io.ox/office/tk/container/valueset',
    'io.ox/office/spreadsheet/utils/rangearray',
    'io.ox/office/spreadsheet/utils/intervalset'
], function (ValueSet, RangeArray, IntervalSet) {

    'use strict';

    // convenience shortcuts
    var SetProto = ValueSet.prototype;

    // private global functions ===============================================

    /**
     * Returns the key of the passed index interval.
     *
     * @param {Interval} interval
     *  An index interval.
     *
     * @returns {String}
     *  The key of the passed index interval.
     */
    function getKey(interval) {
        return interval.key();
    }

    // class RangeSet =========================================================

    /**
     * A unique set of cell range addresses. This class provides optimized
     * methods for insertion, deletion, and access of cell range addresses,
     * while keeping the set unique (two different range addresses in this set
     * will never have the same start end end address).
     *
     * @constructor
     *
     * @extends ValueSet
     */
    var RangeSet = ValueSet.extend({ constructor: function () {

        // base constructor
        ValueSet.call(this, getKey);

        // the interval set storing the column intervals of all cell ranges in this set
        this._colSet = new IntervalSet();

    } }); // class RangeSet

    // public methods ---------------------------------------------------------

    /**
     * Removes all elements from this set.
     *
     * @returns {RangeSet}
     *  A reference to this instance.
     */
    RangeSet.prototype.clear = function () {
        SetProto.clear.call(this);
        this._colSet.clear();
        return this;
    };
    /**
     * Returns all cell range addresses contained in this set as an array.
     *
     * @returns {RangeArray}
     *  All cell range addresses contained in this set as an array, in no
     *  specific order.
     */
    RangeSet.prototype.values = function () {
        var ranges = new RangeArray();
        this.forEach(function (range) { ranges.push(range); });
        return ranges;
    };

    /**
     * Returns a shallow or deep clone of this range set.
     *
     * @param {Boolean|Function} [deep=false]
     *  If set to true or to a function, a deep clone will be created (all
     *  ranges in this set will be cloned); otherwise a shallow clone will be
     *  created (the ranges in this set will be inserted by reference into the
     *  clone). If a function has been passed, it will be invoked for each
     *  cloned cell range, receiving the new cloned cell range inserted into
     *  the cloned set as first parameter, and the old cell range of this set
     *  as second parameter.
     *
     * @param {Object} [context]
     *  The calling context used if the 'deep' parameter is a callback.
     *
     * @returns {RangeSet}
     *  The cloned range set.
     */
    RangeSet.prototype.clone = function (deep, context) {

        // start with a new empty range set
        var clone = new RangeSet();
        // user callback for deep mode
        var callback = (typeof deep === 'function') ? deep : _.noop;

        // create a deep clone of the interval sets (shallow mode only for the final ranges!)
        clone._colSet = this._colSet.clone(function (newColInterval, oldColInterval) {
            newColInterval.rowSet = oldColInterval.rowSet.clone(function (newRowInterval, oldRowInterval) {
                var oldRange = oldRowInterval.range;
                var newRange = deep ? oldRange.clone() : oldRange;
                callback.call(context, newRange, oldRange);
                newRowInterval.range = newRange;
                clone._insert(newRange.key(), newRange);
            });
        });

        return clone;
    };

    /**
     * Returns the cell range address from this set that has the exact position
     * of the passed range address.
     *
     * @param {Range} range
     *  A cell range address to search for.
     *
     * @returns {Range|Null}
     *  An range object from this set with the same start and end address as
     *  the passed range address; otherwise null.
     */
    RangeSet.prototype.get = function (range) {
        return this._get(range.key()) || null;
    };

    /**
     * Returns whether this set contains a cell range covering the passed cell
     * address.
     *
     * @param {Address} address
     *  The cell address used to look for a cell range in this set.
     *
     * @returns {Boolean}
     *  Whether this set contains a cell range covering the passed address.
     */
    RangeSet.prototype.containsAddress = function (address) {
        var colIntervals = this._colSet.findByIndex(address[0]);
        return colIntervals.some(function (colInterval) {
            return colInterval.rowSet.containsIndex(address[1]);
        });
    };

    /**
     * Returns all cell ranges in this set that cover the passed cell address.
     *
     * @param {Address} address
     *  The cell address used to look for the cell ranges in this set.
     *
     * @returns {RangeArray}
     *  An array with all cell ranges in this set that cover the passed cell
     *  address, in no specific order.
     */
    RangeSet.prototype.findByAddress = function (address) {
        var ranges = new RangeArray();
        var colIntervals = this._colSet.findByIndex(address[0]);
        colIntervals.forEach(function (colInterval) {
            var rowIntervals = colInterval.rowSet.findByIndex(address[1]);
            rowIntervals.forEach(function (rowInterval) {
                ranges.push(rowInterval.range);
            });
        });
        return ranges;
    };

    /**
     * Returns whether this set contains a range that overlaps with the passed
     * cell ranges.
     *
     * @param {RangeArray|Range} ranges
     *  An array of cell range addresses, or a single cell range address, used
     *  to look for overlapping ranges in this set.
     *
     * @returns {Boolean}
     *  Whether this set contains a cell range overlapping with the passed cell
     *  ranges.
     */
    RangeSet.prototype.overlaps = function (ranges) {

        // quick check for an exact match
        ranges = RangeArray.get(ranges);
        if (ranges.some(this.has, this)) { return true; }

        // check each range if it overlaps somehow
        return ranges.some(function (range) {
            var rowInterval = range.rowInterval();
            var colIntervals = this._colSet.findIntervals(range.colInterval());
            return colIntervals.some(function (colInterval) {
                return colInterval.rowSet.overlaps(rowInterval);
            }, this);
        }, this);
    };

    /**
     * Returns all ranges in this set that overlap with the passed cell range.
     *
     * @param {Range} range
     *  The cell range used to look for the ranges in this set.
     *
     * @returns {RangeArray}
     *  An array with all ranges in this set overlapping with the passed cell
     *  range, in no specific order.
     */
    RangeSet.prototype.findRanges = function (range) {
        var ranges = new RangeArray();
        var rowInterval = range.rowInterval();
        var colIntervals = this._colSet.findIntervals(range.colInterval());
        colIntervals.forEach(function (colInterval) {
            var rowIntervals = colInterval.rowSet.findIntervals(rowInterval);
            rowIntervals.forEach(function (interval) {
                ranges.push(interval.range);
            });
        });
        return ranges;
    };

    /**
     * Inserts a new cell range address into this set.
     *
     * @param {Range} range
     *  The cell range address to be inserted into this set.
     *
     * @param {Boolean} [replace=false]
     *  If set to true, an existing cell range with equal start and end address
     *  as the passed cell range will be replaced with the new cell range. By
     *  default, the existing cell range will be kept, and the passed cell
     *  range will be ignored. May be important in situations where the ranges
     *  in the set contain additional user settings in arbitrary properties.
     *
     * @returns {Range}
     *  A reference to the cell range contained in this set. This may not be
     *  the same as the passed cell range, e.g. if an equal cell range already
     *  exists in this set, and the flag 'replace' has not been set.
     */
    RangeSet.prototype.insert = function (range, replace) {
        var colInterval = this._colSet.insert(range.colInterval());
        var rowSet = colInterval.rowSet || (colInterval.rowSet = new IntervalSet());
        var rowInterval = rowSet.insert(range.rowInterval());
        if (replace || !rowInterval.range) { rowInterval.range = range; }
        return this._insert(range.key(), rowInterval.range);
    };

    /**
     * Removes the specified cell range address from this set.
     *
     * @param {Range} range
     *  The cell range address to be removed from this set.
     *
     * @returns {Range|Null}
     *  A reference to the removed cell range; or null, if this set does not
     *  contain a cell range that is equal to the passed cell range.
     */
    RangeSet.prototype.remove = function (range) {
        var key = range.key();
        range = this._get(key);
        if (!range) { return null; }
        this._remove(key);
        var colInterval = this._colSet.get(range.colInterval());
        var rowSet = colInterval.rowSet;
        rowSet.remove(range.rowInterval());
        if (rowSet.empty()) { this._colSet.remove(colInterval); }
        return range;
    };

    /**
     * Inserts all cell ranges of the passed set into this set.
     *
     * @param {Object} list
     *  A generic data source with cell range addresses as elements. See static
     *  method Container.forEach() for details.
     *
     * @param {Boolean} [replace=false]
     *  If set to true, cell ranges in the passed list will replace equal cell
     *  ranges in this set. By default, the existing cell ranges of this set
     *  will be preferred.
     *
     * @returns {RangeSet}
     *  A reference to this instance.
     */
    RangeSet.prototype.merge = function (list, replace) {
        ValueSet.forEach(list, function (range) { this.insert(range, replace); }, this);
        return this;
    };

    // exports ================================================================

    return RangeSet;

});
