/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/spreadsheet/model/presetformattable', [
    'io.ox/office/tk/utils',
    'io.ox/office/tk/container/valuemap',
    'io.ox/office/tk/locale/localedata'
], function (Utils, ValueMap, LocaleData) {

    'use strict';

    // the global registry for format table generators
    var FORMAT_TABLE_REGISTRY = {};

    // the name of the global base table, used as implicit parent
    var BASE_TABLE_NAME = '_BASE_';

    // the name of the base table for CJK locales, used as implicit parent
    var CJK_BASE_TABLE_NAME = '_CJK_';

    // currency/accounting meta codes, needed to generate format codes for different locales
    var CURRENCY_META_CODES = {

        // minus-symbol-number variant, e.g.: -$42.00
        '-$0': {
            currency: '{MODIFIER}{SYMBOL}{SPACE}{NUMBER};{RED}{MODIFIER}-{SYMBOL}{SPACE}{NUMBER}',
            accounting: '_-{SYMBOL}{SPACE}* {NUMBER}_-;-{SYMBOL}{SPACE}* {NUMBER}_-;_-{SYMBOL}{SPACE}* "-"{ZERO}_-;_-@_-'
        },

        // symbol-minus-number variant, e.g.: $-42.00
        '$-0': {
            currency: '{MODIFIER}{SYMBOL}{SPACE}{NUMBER};{RED}{MODIFIER}{SYMBOL}{SPACE}-{NUMBER}',
            accounting: '_ {SYMBOL}{SPACE}* {NUMBER}_ ;_ {SYMBOL}{SPACE}* -{NUMBER}_ ;_ {SYMBOL}{SPACE}* "-"{ZERO}_ ;_ @_ '
        },

        // symbol-number-minus variant, e.g.: $42.00-
        '$0-': {
            currency: '{MODIFIER}{SYMBOL}{SPACE}{NUMBER}_-;{RED}{MODIFIER}{SYMBOL}{SPACE}{NUMBER}-',
            accounting: '_-{SYMBOL}{SPACE}* {NUMBER}_-;_-{SYMBOL}{SPACE}* {NUMBER}-;_-{SYMBOL}{SPACE}* "-"{ZERO}_-;_-@_-'
        },

        // minus-number-symbol variant, e.g.: -42.00$
        '-0$': {
            currency: '{MODIFIER}{NUMBER}{SPACE}{SYMBOL};{RED}{MODIFIER}-{NUMBER}{SPACE}{SYMBOL}',
            accounting: '_-* {NUMBER}{SPACE}{SYMBOL}_-;-* {NUMBER}{SPACE}{SYMBOL}_-;_-* "-"{ZERO}{SPACE}{SYMBOL}_-;_-@_-'
        },

        // number-symbol-minus variant, e.g.: 42.00$-
        '0$-': {
            currency: '{MODIFIER}{NUMBER}{SPACE}{SYMBOL}_-;{RED}{MODIFIER}{NUMBER}{SPACE}{SYMBOL}-',
            accounting: '_-* {NUMBER}{SPACE}{SYMBOL}_-;_-* {NUMBER}{SPACE}{SYMBOL}-;_-* "-"{ZERO}{SPACE}{SYMBOL}_-;_-@_-'
        },

        // symbol-number-with-parentheses variant, e.g.: ($42.00)
        '($0)': {
            currency: '{MODIFIER}{SYMBOL}{SPACE}{NUMBER}_);{RED}({MODIFIER}{SYMBOL}{SPACE}{NUMBER})',
            accounting: '_({SYMBOL}{SPACE}* {NUMBER}_);_({SYMBOL}{SPACE}* ({NUMBER});_({SYMBOL}{SPACE}* "-"{ZERO}_);_(@_)'
        },

        // number-symbol-with-parentheses variant, e.g.: (42.00$)
        '(0$)': {
            currency: '{MODIFIER}{NUMBER}{SPACE}{SYMBOL}_);{RED}({MODIFIER}{NUMBER}{SPACE}{SYMBOL})',
            accounting: '_ * {NUMBER}_){SPACE}{SYMBOL}_ ;_ * ({NUMBER}){SPACE}{SYMBOL}_ ;_ * "-"{ZERO}_){SPACE}{SYMBOL}_ ;_ @_ '
        }
    };

    // characters for CJK date/time format code templates
    var CJK_DATE_TIME_CHARACTERS = {
        traditional: { Y: '\u5e74', M: '\u6708', D: '\u65e5', h: '\u6642', m: '\u5206', s: '\u79d2' },
        simplified:  { Y: '\u5e74', M: '\u6708', D: '\u65e5', h: '\u65f6', m: '\u5206', s: '\u79d2' },
        hangul:      { Y: '\ub144', M: '\uc6d4', D: '\uc77c', h: '\uc2dc', m: '\ubd84', s: '\ucd08' }
    };

    // private global functions ===============================================

    /**
     * Escapes the passed currency symbol on demand by preceding a single
     * character with a backslash, or by enclosing it into double quotes; or
     * creates the blind currency symbol by preceding each character with an
     * underscore.
     */
    function encodeCurrencySymbol(currencySymbol, options) {

        // blind currency symbol: precede underscores, e.g. 'EUR' => '_E_U_R'
        if (options && options.blind) {
            return currencySymbol.replace(/./g, '_$&');
        }

        // currency symbol with letters or special characters: escape with literal text markers
        if (/[a-zA-Z.,/]/.test(currencySymbol)) {
            return (currencySymbol.length === 1) ? ('\\' + currencySymbol) : ('"' + currencySymbol + '"');
        }

        // no special characters: return original currency symbol
        return currencySymbol;
    }

    /**
     * Generates a currency format code from the passed meta code and settings.
     * See public method TableGeneratorContext.createCurrencyFormat() for more
     * details.
     */
    function getCurrencyCode(metaCode, encodedCurrency, options) {
        var numberCode = '#,##0' + ((options && options.int) ? '' : '.00');
        var zeroCode = (options && options.int) ? '' : '??';
        return metaCode
            .replace(/{NUMBER}/g, numberCode)
            .replace(/{ZERO}/g, zeroCode)
            .replace(/{SYMBOL}/g, encodedCurrency)
            .replace(/{MODIFIER}/g, (options && options.modifier) || '')
            .replace(/{SPACE}/g, (options && options.space) ? ' ' : '')
            .replace(/{RED}/g, (options && options.red) ? '[Red]' : '');
    }

    /**
     * Registers a format table for a specific locale.
     *
     * @param {String} localeId
     *  The identifier of the locale for the new format table.
     *
     * @param {String} [parentLocaleId]
     *  The locale identifier of the parent format table, that will be cloned
     *  and extended for the specified locale. If omitted, an appropriate
     *  default parent table will be selected for the locale (a special table
     *  will be used for CJK locales).
     *
     * @param {Function} [generatorCallback]
     *  A callback function that will initialize the format table. Will be
     *  invoked on first access of the format table. The format table to be
     *  initialized will be set as function context (symbol 'this'). Can be
     *  omitted to create an exact clone of another format table (the parameter
     *  'parentLocaleId' MUST be specified in that case).
     */
    function registerFormatTable(localeId, parentLocaleId, generatorCallback) {
        FORMAT_TABLE_REGISTRY[localeId] = _.isFunction(parentLocaleId) ?
            { parent: LocaleData.get(localeId).cjk ? CJK_BASE_TABLE_NAME : BASE_TABLE_NAME, generator: parentLocaleId } :
            { parent: parentLocaleId, generator: generatorCallback };
    }

    // class TableGeneratorContext ============================================

    /**
     * A helper class that will be passed as calling context for the generator
     * callback functions of new format tables. Provides useful helper methods
     * to generate single format codes, or entire groups of format codes.
     *
     * @constructor
     *
     * @param {LocaleDataEntry} lcData
     *  The configuration of the locale.
     *
     * @param {Array<Any>} entries
     *  An array that will be filled by the public methods of an instance of
     *  this class.
     */
    function TableGeneratorContext(lcData, entries) {

        this._lcData = lcData;
        this._entries = entries;

    } // class TableGeneratorContext

    // public methods ---------------------------------------------------------

    /**
     * Defines a new number format entry for the format table wrapped by this
     * instance.
     *
     * @param {Number} id
     *  The number format identifier.
     *
     * @param {String|Number} format
     *  A literal format code as string, or another number format identifier to
     *  create a clone of another predefined number format.
     *
     * @returns {TableGeneratorContext}
     *  A reference to this instance.
     */
    TableGeneratorContext.prototype.createFormat = function (id, format) {
        this._entries[id] = format;
        return this;
    };

    /**
     * Defines the date and time formats 14 to 22.
     *
     * @param {String} systemDateCode
     *  Complete short system date code (for number formats 14 and 22).
     *
     * @param {String} fixedDateCode
     *  Fixed date code (for number formats 15 to 17). MUST be in strict order
     *  day-month-year, with some separator characters inbetween.
     *
     * @param {String} [hourMode='short']
     *  Format of the hour token. If specified, must be one of the following:
     *  - 'short' (default): Use 'h' for all time formats 18 to 22.
     *  - 'long': Use 'hh' for all time formats 18 to 22.
     *  - 'long24': Use 'h' for 12-hour time formats (formats 18 and 19), and
     *      'hh' for 24-hour time formats (formats 20 to 22).
     *
     * @returns {TableGeneratorContext}
     *  A reference to this instance.
     */
    TableGeneratorContext.prototype.createAllDateTimeFormats = function (systemDateCode, fixedDateCode, hourMode) {
        var hour12 = (hourMode !== 'long') ? 'h' : 'hh';
        var hour24 = ((hourMode === 'long') || (hourMode === 'long24')) ? 'hh' : 'h';
        return this
            .createFormat(14, systemDateCode)
            .createFormat(15, fixedDateCode)
            .createFormat(16, fixedDateCode.replace(/[^M]+Y+$/, ''))
            .createFormat(17, fixedDateCode.replace(/^D+[^M]+/, ''))
            .createFormat(18, hour12 + ':mm AM/PM')
            .createFormat(19, hour12 + ':mm:ss AM/PM')
            .createFormat(20, hour24 + ':mm')
            .createFormat(21, hour24 + ':mm:ss')
            .createFormat(22, systemDateCode + ' ' + hour24 + ':mm'); // always 24-hours format
    };

    /**
     * Defines a time format for CJK locales.
     *
     * @param {Number} id
     *  The number format identifier for the format code.
     *
     * @param {String} metaCode
     *  The date/time meta code. May contain the meta tokens {YEAR}, {MONTH},
     *  {DAY}, {HOUR}, {MINUTE}, and {SECOND}, which will be replaced by the
     *  respective words according to the passed CJK script type (language);
     *  and the meta token {H} that will be replaced by an hour token according
     *  to the passed options.
     *
     * @param {String} script
     *  The CJK script type used for the meta tokens (see parameter 'metaCode')
     *  added to the generated format code. MUST be one of:
     *  - 'simplified': CJK Unified Ideographs (Simplified Chinese)
     *  - 'traditional': CJK Unified Ideographs (Traditional Chinese, Japanese)
     *  - 'hangul': Hangul alphabet (Korean).
     *
     * @param {Object} [options]
     *  Optional parameters:
     *  @param {Boolean} [options.ampm=false]
     *      If set to true, the token 'AM/PM' will be added in front of the
     *      hour token inserted for the meta token {H}.
     *  @param {Boolean} [options.long=false]
     *      If set to true, the hour token inserted for the meta token {H} will
     *      be 'hh', otherwise 'h'.
     *
     * @returns {TableGeneratorContext}
     *  A reference to this instance.
     */
    TableGeneratorContext.prototype.createCJKDateTimeFormat = function (id, metaCode, script, options) {
        var chars = CJK_DATE_TIME_CHARACTERS[script];
        return this.createFormat(id, metaCode
            .replace(/{H}/g, ((options && options.ampm) ? 'AM/PM' : '') + ((options && options.long) ? 'hh' : 'h'))
            .replace(/{YEAR}/g, chars.Y)
            .replace(/{MONTH}/g, chars.M)
            .replace(/{DAY}/g, chars.D)
            .replace(/{HOUR}/g, chars.h)
            .replace(/{MINUTE}/g, chars.m)
            .replace(/{SECOND}/g, chars.s)
        );
    };

    /**
     * Defines two time formats without, one trailing seconds, and one with
     * trailing seconds, for CJK locales.
     *
     * @param {Number} id
     *  First number format identifier for the format code without seconds.
     *  This method generates format codes for 'id' (without seconds), and for
     *  'id+1' (with seconds).
     *
     * @param {String} script
     *  The CJK script type used to generate the time formats. See method
     *  PresetFormatTable.createCJKDateTimeFormat() for details.
     *
     * @param {Object} [options]
     *  Optional parameters. See method PresetFormatTable.createCJKDateTimeFormat()
     *  for details.
     *
     * @returns {TableGeneratorContext}
     *  A reference to this instance.
     */
    TableGeneratorContext.prototype.createCJKTimeFormats = function (id, script, options) {
        var metaCode = '{H}"{HOUR}"mm"{MINUTE}"';
        return this
            .createCJKDateTimeFormat(id, metaCode, script, options)
            .createCJKDateTimeFormat(id + 1, metaCode + 'ss"{SECOND}"', script, options);
    };

    /**
     * Defines a currency/accounting format according to the passed meta code.
     *
     * @param {Number} id
     *  The number format identifier for the currency format code.
     *
     * @param {String} metaCode
     *  The currency meta code used to build the number format.
     *
     * @param {String} encodedCurrency
     *  The encoded currency symbol to be inserted into the format code for the
     *  {SYMBOL} meta token.
     *
     * @param {Object} options
     *  Optional parameters:
     *  @param {Boolean} [options.int=false]
     *      Whether to create an integral currency format without fractional
     *      digits.
     *  @param {Boolean} [options.space=false]
     *      Whether to insert a space character for the {SPACE} meta token.
     *  @param {Boolean} [options.red=false]
     *      Whether to insert the color name 'Red' for the {RED} meta token.
     *  @param {String} [options.modifier='']
     *      The code snippet for a number system modifier to be inserted into
     *      the format code for the {MODIFIER} meta token.
     *
     * @returns {TableGeneratorContext}
     *  A reference to this instance.
     */
    TableGeneratorContext.prototype.createCurrencyFormat = function (id, metaCode, encodedCurrency, options) {
        return this.createFormat(id, getCurrencyCode(metaCode, encodedCurrency, options));
    };

    /**
     * Defines four currency formats according to the passed meta code and
     * currency symbol.
     *
     * @param {Number} id
     *  Number format identifier for the first currency format.
     *
     * @param {String} metaCodeId
     *  The identifier of the currency meta code used to build the number
     *  formats.
     *
     * @param {String} currencySymbol
     *  The original currency symbol to be inserted into the format code for
     *  the {SYMBOL} meta token.
     *
     * @param {Object} [options]
     *  Optional parameters:
     *  @param {Boolean} [options.blind=false]
     *      Whether to convert the currency symbol to a blind currency symbol.
     *  @param {Boolean} [options.space=false]
     *      Whether to replace the {SPACE} meta token with a space character.
     *  @param {String} [options.modifier='']
     *      Modifier for type of digits, will be added in the beginning of each
     *      format section.
     *
     * @returns {TableGeneratorContext}
     *  A reference to this instance.
     */
    TableGeneratorContext.prototype.createCurrencyFormats = function (id, metaCodeId, currencySymbol, options) {

        // resolve the meta code for currency formats
        var currencyMetaCode = CURRENCY_META_CODES[metaCodeId].currency;
        // quote currency symbol on demand, or create the blind currency symbol (precede each character with an underscore)
        var encodedCurrency = encodeCurrencySymbol(currencySymbol, options);

        return this
            .createCurrencyFormat(id,     currencyMetaCode, encodedCurrency, _.extend({}, options, { int: true,  red: false }))
            .createCurrencyFormat(id + 1, currencyMetaCode, encodedCurrency, _.extend({}, options, { int: true,  red: true }))
            .createCurrencyFormat(id + 2, currencyMetaCode, encodedCurrency, _.extend({}, options, { int: false, red: false }))
            .createCurrencyFormat(id + 3, currencyMetaCode, encodedCurrency, _.extend({}, options, { int: false, red: true }));
    };

    /**
     * Defines the currency/accounting formats 5 to 8 (with currency symbol),
     * 37 to 40 (blind currency symbol), and 41 to 44 (accounting).
     *
     * @param {String} metaCodeId
     *  The identifier of the currency/accounting meta codes used to build the
     *  number formats.
     *
     * @param {String} currencySymbol
     *  The currency symbol to be inserted into the format code for the
     *  {SYMBOL} meta token.
     *
     * @param {Object} [options]
     *  Optional parameters:
     *  @param {Boolean} [options.space=false]
     *      Whether to replace the {SPACE} meta token with a space character.
     *  @param {String} [options.modifier='']
     *      Modifier for type of digits, will be added in the beginning of each
     *      format section.
     *
     * @returns {TableGeneratorContext}
     *  A reference to this instance.
     */
    TableGeneratorContext.prototype.createAllCurrencyFormats = function (metaCodeId, options) {

        // the currency symbol of the locale
        var currencySymbol = this._lcData.currency;
        // do not insert a blind currency symbol at all, if it is located before the number
        var blindSymbol = /0.*\$/.test(metaCodeId) ? currencySymbol : '';

        // format codes 5 to 8: currency formats with currency symbol
        this.createCurrencyFormats(5, metaCodeId, currencySymbol, _.extend({}, options, { blind: false }));
        // format codes 37 to 40: currency formats with blind currency symbol
        this.createCurrencyFormats(37, metaCodeId, blindSymbol, _.extend({}, options, { blind: true }));

        // format codes 41 to 44: accounting formats
        var accountingMetaCode = CURRENCY_META_CODES[metaCodeId].accounting;
        var encodedCurrency = encodeCurrencySymbol(currencySymbol);
        var encodedBlind = encodeCurrencySymbol(blindSymbol, { blind: true });
        return this
            .createCurrencyFormat(41, accountingMetaCode, encodedBlind,    _.extend({}, options, { int: true }))
            .createCurrencyFormat(42, accountingMetaCode, encodedCurrency, _.extend({}, options, { int: true }))
            .createCurrencyFormat(43, accountingMetaCode, encodedBlind,    _.extend({}, options, { int: false }))
            .createCurrencyFormat(44, accountingMetaCode, encodedCurrency, _.extend({}, options, { int: false }));
    };

    // class PresetFormatTable ================================================

    /**
     * A table with predefined number format codes for a specific locale.
     *
     * @constructor
     */
    function PresetFormatTable(localeId) {

        this._lcData = LocaleData.get(localeId);
        this._baseEntries = null;
        this._formatCodes = null;
        this._invMap = new ValueMap();

        // initialization -----------------------------------------------------

        // the parent and generator callback registered for the locale
        var registryEntry = FORMAT_TABLE_REGISTRY[localeId];

        // resolve the parent format table, create a clone of the format codes
        if (_.isString(registryEntry.parent)) {
            var parentTable = PresetFormatTable.create(registryEntry.parent);
            this._baseEntries = _.clone(parentTable._baseEntries);
        } else {
            this._baseEntries = [];
        }

        // invoke the generator callback with a generator context instance
        if (_.isFunction(registryEntry.generator)) {
            var context = new TableGeneratorContext(this._lcData, this._baseEntries);
            registryEntry.generator.call(context);
        }

        // resolve all place-holder formats
        this._formatCodes = this._baseEntries.map(function (format) {
            return _.isNumber(format) ? this._baseEntries[format] : format;
        }, this);

        // create the inverse map (format codes to identifiers), use first available identifier only
        this._formatCodes.forEach(function (format, id) {
            if (!this._invMap.has(format)) {
                this._invMap.insert(format, id);
            }
        }, this);

    } // class PresetFormatTable

    // constants --------------------------------------------------------------

    /**
     * The identifier of the general format code.
     *
     * @constant
     */
    PresetFormatTable.GENERAL_ID = 0;

    /**
     * The identifier of the system date format.
     *
     * @constant
     */
    PresetFormatTable.SYSTEM_DATE_ID = 14;

    /**
     * The identifier of the system date/time format.
     *
     * @constant
     */
    PresetFormatTable.SYSTEM_DATETIME_ID = 22;

    /**
     * The identifier of the default text format.
     *
     * @constant
     */
    PresetFormatTable.TEXT_ID = 49;

    /**
     * The identifier of the first user-defined number format, as used in the
     * OOXML file format.
     *
     * @constant
     */
    PresetFormatTable.FIRST_USER_FORMAT_ID = 164;

    /**
     * The identifiers of locale-dependent currency formats that need to be
     * written into the spreadsheet document.
     *
     * @constant
     */
    PresetFormatTable.CURRENCY_FORMAT_IDS = [5, 6, 7, 8, 37, 38, 39, 40, 41, 42, 43, 44];

    // static methods ---------------------------------------------------------

    /**
     * Returns the predefined format codes for the specified locale. The table
     * will be cached on first access, and the same instance will be returned
     * for subsequent calls with the same locale identifier.
     *
     * @param {String} localeId
     *  The identifier of the locale.
     *
     * @returns {PresetFormatTable}
     *  The predefined format codes for the specified locale.
     */
    PresetFormatTable.create = _.memoize(function (localeId) {

        // fall-back to base table for unknown locales
        if (!(localeId in FORMAT_TABLE_REGISTRY)) {
            Utils.warn('PresetFormatTable.create(): unknown locale "' + localeId + '"');
            localeId = BASE_TABLE_NAME;
        }

        // create a new format table (initializes itself internally)
        return new PresetFormatTable(localeId);
    });

    /**
     * Returns the format codes of all locale-dependent currency number formats
     * for the specified locale.
     *
     * @param {String} localeId
     *  The identifier of the locale.
     *
     * @returns {Object}
     *  The format codes of all locale-dependent currency number formats for
     *  the specified locale, mapped by the integer format identifiers.
     */
    PresetFormatTable.getInitialFormats = _.memoize(function (localeId) {

        var initialFormats = {};
        var presetTable = PresetFormatTable.create(localeId);
        PresetFormatTable.CURRENCY_FORMAT_IDS.forEach(function (presetId) {
            initialFormats[presetId] = presetTable.getFormatCode(presetId);
        });

        return initialFormats;
    });

    /**
     * Returns the identifier of a preset number format for decimal numbers
     * with a fixed count of decimal places, and optionally with group
     * separators.
     *
     * @param {Boolean} int
     *  Whether to return the identifier of a number format for integers
     *  (true), or of a number format with two decimal places (false).
     *
     * @param {Boolean} group
     *  Whether to return the identifier of a number format with group
     *  separators.
     *
     * @returns {Number}
     *  The identifier of a number format for decimal numbers.
     */
    PresetFormatTable.getDecimalId = function (int, group) {
        return (int ? 1 : 2) + (group ? 2 : 0);
    };

    /**
     * Returns the identifier of a preset number format for numbers in
     * scientific notation.
     *
     * @param {Boolean} long
     *  Whether to return the identifier of the number format with two decimal
     *  places (true), or with one decimal place (false).
     *
     * @returns {Number}
     *  The identifier of a number format for scientific notation.
     */
    PresetFormatTable.getScientificId = function (long) {
        return long ? 11 : 48;
    };

    /**
     * Returns the identifier of a preset number format for percentages with a
     * fixed count of decimal places.
     *
     * @param {Boolean} int
     *  Whether to return the identifier of a number format for integers
     *  (true), or of a number format with two decimal places (false).
     *
     * @returns {Number}
     *  The identifier of a number format for percentages.
     */
    PresetFormatTable.getPercentId = function (int) {
        return int ? 9 : 10;
    };

    /**
     * Returns the identifier of a preset number format for fractions.
     *
     * @param {Boolean} long
     *  Whether to return the identifier of the number format for fractions
     *  with a two-digit denominator (true), or with a one-digit denominator
     *  (false).
     *
     * @returns {Number}
     *  The identifier of a number format for fractions.
     */
    PresetFormatTable.getFractionId = function (long) {
        return long ? 13 : 12;
    };

    /**
     * Returns the identifier of a preset currency format.
     *
     * @param {Boolean} int
     *  Whether to return the identifier of a number format for integers
     *  (true), or of a number format with two decimal places (false).
     *
     * @param {Boolean} red
     *  Whether negative currencies will be formatted with red text color.
     *
     * @param {Boolean} blind
     *  Whether to replace the currency symbol in the format code with
     *  white-space (true), or to use the actual currency symbol (false).
     *
     * @returns {Number}
     *  The identifier of a number format for currencies.
     */
    PresetFormatTable.getCurrencyId = function (int, red, blind) {
        return (blind ? 37 : 5) + (int ? 0 : 2) + (red ? 1 : 0);
    };

    /**
     * Returns the identifier of a preset number format for times.
     *
     * @param {Boolean|Null} hours24
     *  Specifies how to show the hours of the time. If set to true, hours will
     *  be shown as number from 0 to 23. If set to false, hours will be shown
     *  as number from 1 to 12, and the format code will include the AM/PM
     *  token. If set to null, the hour mode will be selected as preferred by
     *  the current UI language.
     *
     * @param {Boolean} seconds
     *  Whether the number format includes the seconds of the time.
     *
     * @returns {Number}
     *  The identifier of a number format for percentages.
     */
    PresetFormatTable.getTimeId = function (hours24, seconds) {
        if (hours24 === null) { hours24 = LocaleData.HOURS24; }
        return 18 + (seconds ? 1 : 0) + (hours24 ? 2 : 0);
    };

    /**
     * Generates the format code for a currency number format.
     *
     * @param {String} metaCodeId
     *  The identifier of the currency meta code used to build the currency
     *  number format. MUST be one of the following values:
     *  - '-$0' for the variant minus-symbol-number, e.g. -$42.00.
     *  - '$-0' for the variant symbol-minus-number, e.g. $-42.00.
     *  - '$0-' for the variant symbol-number-minus, e.g. $42.00-.
     *  - '-0$' for the variant minus-number-symbol, e.g. -42.00$.
     *  - '0$-' for the variant number-symbol-minus, e.g. 42.00$-.
     *  - '($0)' for the variant symbol-number-with-parentheses, e.g. ($42.00).
     *  - '(0$)' for the variant number-symbol-with-parentheses, e.g. (42.00$).
     *
     * @param {String} currencySymbol
     *  The currency symbol to be inserted into the format code.
     *
     * @param {Object} [options]
     *  Optional parameters:
     *  @param {Boolean} [options.int=false]
     *      Whether to create an integral currency format without fractional
     *      digits.
     *  @param {Boolean} [options.space=false]
     *      Whether to insert a space character between the number and the
     *      currency symbol.
     *  @param {Boolean} [options.red=false]
     *      Whether to format negative currencies with red text color.
     *
     * @returns {String}
     *  The format code for a currency number format.
     */
    PresetFormatTable.getCurrencyCode = function (metaCodeId, currencySymbol, options) {
        var metaCode = CURRENCY_META_CODES[metaCodeId].currency;
        var encodedCurrency = encodeCurrencySymbol(currencySymbol);
        return getCurrencyCode(metaCode, encodedCurrency, options);
    };

    /**
     * Generates the format code for a fraction with the sepcified number of
     * digits in the numerator and denominator.
     *
     * @param {Number} digits
     *  The number of digits for the numerator and denominator.
     *
     * @returns {String}
     *  The format code for a fraction.
     */
    PresetFormatTable.getFractionCode = function (digits) {
        var pattern = Utils.repeatString('?', digits);
        return '# ' + pattern + '/' + pattern;
    };

    // public methods ---------------------------------------------------------

    /**
     * Returns the predefined format code for the specified format identifier.
     *
     * @param {Number} formatId
     *  The identifier of a predefined format code.
     *
     * @returns {String|Null}
     *  The predefined format code for the specified format identifier, if
     *  available; otherwise null.
     */
    PresetFormatTable.prototype.getFormatCode = function (formatId) {
        var formatCode = this._formatCodes[formatId];
        return _.isString(formatCode) ? formatCode : null;
    };

    /**
     * Returns the predefined identifier of the specified format code.
     *
     * @param {String} formatCode
     *  The format code to be converted to a format identifier.
     *
     * @returns {Number|Null}
     *  The predefined identifier of the passed format code, if available;
     *  otherwise null.
     */
    PresetFormatTable.prototype.getFormatId = function (formatCode) {
        return this._invMap.get(formatCode, null);
    };

    // static table initialization ============================================

    // Default format table. Last parent of all other tables, used for unknown locales.
    registerFormatTable(BASE_TABLE_NAME, null, function () {

            // 0 to 4: numbers
        this.createFormat(0, 'General')
            .createFormat(1, '0')
            .createFormat(2, '0.00')
            .createFormat(3, '#,##0')
            .createFormat(4, '#,##0.00')

            // 5 to 8, 37 to 44: currency/accounting formats
            .createAllCurrencyFormats('($0)')

            // 9 to 13: percentage, scientific, fractions
            .createFormat(9, '0%')
            .createFormat(10, '0.00%')
            .createFormat(11, '0.00E+00')
            .createFormat(12, PresetFormatTable.getFractionCode(1))
            .createFormat(13, PresetFormatTable.getFractionCode(2))

            // 14 to 22: date and time formats
            .createAllDateTimeFormats('M/D/YYYY', 'D-MMM-YY')

            // 23 to 26: CJK currency formats
            .createFormat(23, 0)
            .createFormat(24, 0)
            .createFormat(25, 0)
            .createFormat(26, 0)

            // 27 to 36: CJK date/time formats
            .createFormat(27, 14)
            .createFormat(28, 14)
            .createFormat(29, 14)
            .createFormat(30, 14)
            .createFormat(31, 14)
            .createFormat(32, 21)
            .createFormat(33, 21)
            .createFormat(34, 21)
            .createFormat(35, 21)
            .createFormat(36, 14)

            // 45 to 49: additional formats
            .createFormat(45, 'mm:ss')
            .createFormat(46, '[h]:mm:ss')
            .createFormat(47, 'mm:ss.0')
            .createFormat(48, '0.0E+0')
            .createFormat(49, '@')

            // 50 to 58: CJK date/time formats
            .createFormat(50, 14)
            .createFormat(51, 14)
            .createFormat(52, 14)
            .createFormat(53, 14)
            .createFormat(54, 14)
            .createFormat(55, 14)
            .createFormat(56, 14)
            .createFormat(57, 14)
            .createFormat(58, 14)

            // 59 to 81: Thai formats with 't' modifier
            .createFormat(59, 1)
            .createFormat(60, 2)
            .createFormat(61, 3)
            .createFormat(62, 4)
            .createFormat(63, 5)
            .createFormat(64, 6)
            .createFormat(65, 7)
            .createFormat(66, 8)
            .createFormat(67, 9)
            .createFormat(68, 10)
            .createFormat(69, 12)
            .createFormat(70, 13)
            .createFormat(71, 14)
            .createFormat(72, 14)
            .createFormat(73, 15)
            .createFormat(74, 16)
            .createFormat(75, 17)
            .createFormat(76, 20)
            .createFormat(77, 21)
            .createFormat(78, 22)
            .createFormat(79, 45)
            .createFormat(80, 46)
            .createFormat(81, 47);
    });

    // Base table for CJK locales.
    registerFormatTable(CJK_BASE_TABLE_NAME, function () {
        this.createFormat(29, 28)
            .createFormat(36, 27)
            .createFormat(50, 27)
            .createFormat(51, 28)
            .createFormat(52, 34)
            .createFormat(53, 35)
            .createFormat(54, 28)
            .createFormat(55, 34)
            .createFormat(56, 35)
            .createFormat(57, 27)
            .createFormat(58, 28);
    });

    // Afrikaans, South Africa
    registerFormatTable('af_ZA', 'en_ZA');

    // Arabic, U.A.E.
    registerFormatTable('ar_AE', function () {
        this.createAllDateTimeFormats('DD/MM/YYYY', 'DD-MMM-YY', 'long')
            .createAllCurrencyFormats('$0-', { space: true });
    });

    // Arabic, Bahrain
    registerFormatTable('ar_BH', function () {
        this.createAllDateTimeFormats('DD/MM/YYYY', 'DD-MMM-YY', 'long')
            .createAllCurrencyFormats('$0-', { space: true });
    });

    // Arabic, Algeria
    registerFormatTable('ar_DZ', function () {
        this.createAllDateTimeFormats('DD-MM-YYYY', 'DD-MMM-YY')
            .createAllCurrencyFormats('$0-', { space: true });
    });

    // Arabic, Egypt
    registerFormatTable('ar_EG', function () {
        this.createAllDateTimeFormats('DD/MM/YYYY', 'DD-MMM-YY', 'long')
            .createAllCurrencyFormats('$0-', { space: true });
    });

    // Arabic, Iraq
    registerFormatTable('ar_IQ', function () {
        this.createAllDateTimeFormats('DD/MM/YYYY', 'DD-MMM-YY', 'long')
            .createAllCurrencyFormats('$0-', { space: true });
    });

    // Arabic, Jordan
    registerFormatTable('ar_JO', function () {
        this.createAllDateTimeFormats('DD/MM/YYYY', 'DD-MMM-YY', 'long')
            .createAllCurrencyFormats('$0-', { space: true });
    });

    // Arabic, Kuwait
    registerFormatTable('ar_KW', function () {
        this.createAllDateTimeFormats('DD/MM/YYYY', 'DD-MMM-YY', 'long')
            .createAllCurrencyFormats('$0-', { space: true });
    });

    // Arabic, Lebanon
    registerFormatTable('ar_LB', function () {
        this.createAllDateTimeFormats('DD/MM/YYYY', 'DD-MMM-YY', 'long')
            .createAllCurrencyFormats('$0-', { space: true });
    });

    // Arabic, Libya
    registerFormatTable('ar_LY', function () {
        this.createAllDateTimeFormats('DD/MM/YYYY', 'DD-MMM-YY', 'long')
            .createAllCurrencyFormats('$0-', { space: true });
    });

    // Arabic, Morocco
    registerFormatTable('ar_MA', function () {
        this.createAllDateTimeFormats('DD-MM-YYYY', 'DD-MMM-YY')
            .createAllCurrencyFormats('$0-', { space: true });
    });

    // Arabic, Oman
    registerFormatTable('ar_OM', function () {
        this.createAllDateTimeFormats('DD/MM/YYYY', 'DD-MMM-YY', 'long')
            .createAllCurrencyFormats('$0-', { space: true });
    });

    // Arabic, Qatar
    registerFormatTable('ar_QA', function () {
        this.createAllDateTimeFormats('DD/MM/YYYY', 'DD-MMM-YY', 'long')
            .createAllCurrencyFormats('$0-', { space: true });
    });

    // Arabic, Saudi Arabia
    registerFormatTable('ar_SA', function () {
        this.createAllDateTimeFormats('DD/MM/YYYY', 'DD-MMM-YY', 'long')
            .createAllCurrencyFormats('$0-', { space: true });
    });

    // Arabic, Syria
    registerFormatTable('ar_SY', function () {
        this.createAllDateTimeFormats('DD/MM/YYYY', 'DD-MMM-YY', 'long')
            .createAllCurrencyFormats('$0-', { space: true });
    });

    // Arabic, Tunisia
    registerFormatTable('ar_TN', function () {
        this.createAllDateTimeFormats('DD-MM-YYYY', 'DD-MMM-YY')
            .createAllCurrencyFormats('$0-', { space: true });
    });

    // Arabic, Yemen
    registerFormatTable('ar_YE', function () {
        this.createAllDateTimeFormats('DD/MM/YYYY', 'DD-MMM-YY', 'long')
            .createAllCurrencyFormats('$0-', { space: true });
    });

    // Belarusian, Belarus
    registerFormatTable('be_BY', function () {
        this.createAllDateTimeFormats('DD.MM.YYYY', 'DD.MMM.YY')
            .createAllCurrencyFormats('-0$', { space: true });
    });

    // Bulgarian, Bulgaria
    registerFormatTable('bg_BG', function () {
        this.createAllDateTimeFormats('D.M.YYYY', 'D.MMM.YY')
            .createAllCurrencyFormats('-0$', { space: true });
    });

    // Bengali, India
    registerFormatTable('bn_IN', function () {
        this.createAllDateTimeFormats('DD-MM-YY', 'DD-MMM-YY', 'long24')
            .createAllCurrencyFormats('$-0', { space: true });
    });

    // Catalan, Spain
    registerFormatTable('ca_ES', 'es_ES');

    // Czech, Czech Republic
    registerFormatTable('cs_CZ', function () {
        this.createAllDateTimeFormats('D.M.YYYY', 'D.MMM.YY')
            .createAllCurrencyFormats('-0$', { space: true });
    });

    // Welsh, United Kingdom
    registerFormatTable('cy_GB', 'en_GB');

    // Danish, Denmark
    registerFormatTable('da_DK', function () {
        this.createAllDateTimeFormats('DD-MM-YYYY', 'DD-MMM-YY', 'long24')
            .createAllCurrencyFormats('$-0', { space: true });
    });

    // German, Austria
    registerFormatTable('de_AT', function () {
        this.createAllDateTimeFormats('DD.MM.YYYY', 'DD.MMM.YY', 'long24')
            .createAllCurrencyFormats('-$0', { space: true });
    });

    // German, Switzerland
    registerFormatTable('de_CH', function () {
        this.createAllDateTimeFormats('DD.MM.YYYY', 'DD. MMM YY', 'long24')
            .createAllCurrencyFormats('$-0', { space: true });
    });

    // German, Germany
    registerFormatTable('de_DE', function () {
        this.createAllDateTimeFormats('DD.MM.YYYY', 'DD. MMM YY', 'long24')
            .createAllCurrencyFormats('-0$', { space: true });
    });

    // German, Liechtenstein
    registerFormatTable('de_LI', function () {
        this.createAllDateTimeFormats('DD.MM.YYYY', 'DD. MMM YY', 'long24')
            .createAllCurrencyFormats('$-0', { space: true });
    });

    // German, Luxembourg
    registerFormatTable('de_LU', function () {
        this.createAllDateTimeFormats('DD.MM.YYYY', 'DD.MMM.YY', 'long24')
            .createAllCurrencyFormats('-0$', { space: true });
    });

    // Divehi, Maldives
    registerFormatTable('div_MV', function () {
        this.createAllDateTimeFormats('DD/MM/YY', 'DD-MMM-YY', 'long24')
            .createAllCurrencyFormats('0$-', { space: true });
    });

    // Greek, Greece
    registerFormatTable('el_GR', function () {
        this.createAllDateTimeFormats('D/M/YYYY', 'D-MMM-YY')
            .createAllCurrencyFormats('-0$', { space: true });
    });

    // English, Australia
    registerFormatTable('en_AU', function () {
        this.createAllDateTimeFormats('D/MM/YYYY', 'D-MMM-YY')
            .createAllCurrencyFormats('-$0');
    });

    // English, Belize
    registerFormatTable('en_BZ', function () {
        this.createAllDateTimeFormats('DD/MM/YYYY', 'DD-MMM-YY', 'long')
            .createAllCurrencyFormats('($0)');
    });

    // English, Canada
    registerFormatTable('en_CA', function () {
        this.createAllDateTimeFormats('DD/MM/YYYY', 'DD-MMM-YY')
            .createAllCurrencyFormats('-$0');
    });

    // English, Caribbean
    registerFormatTable('en_CB', function () {
        this.createAllDateTimeFormats('MM/DD/YYYY', 'DD-MMM-YY')
            .createAllCurrencyFormats('-$0');
    });

    // English, United Kingdom
    registerFormatTable('en_GB', function () {
        this.createAllDateTimeFormats('DD/MM/YYYY', 'DD-MMM-YY', 'long24')
            .createAllCurrencyFormats('-$0');
    });

    // English, Ireland
    registerFormatTable('en_IE', function () {
        this.createAllDateTimeFormats('DD/MM/YYYY', 'DD-MMM-YY', 'long24')
            .createAllCurrencyFormats('-$0');
    });

    // English, Jamaica
    registerFormatTable('en_JM', function () {
        this.createAllDateTimeFormats('DD/MM/YYYY', 'DD-MMM-YY', 'long')
            .createAllCurrencyFormats('-$0');
    });

    // English, New Zealand
    registerFormatTable('en_NZ', function () {
        this.createAllDateTimeFormats('D/MM/YYYY', 'D-MMM-YY')
            .createAllCurrencyFormats('-$0');
    });

    // English, Philippines
    registerFormatTable('en_PH', function () {
        this.createAllDateTimeFormats('M/D/YYYY', 'D-MMM-YY')
            .createAllCurrencyFormats('($0)');
    });

    // English, Trinidad and Tobago
    registerFormatTable('en_TT', function () {
        this.createAllDateTimeFormats('DD/MM/YYYY', 'DD-MMM-YY', 'long')
            .createAllCurrencyFormats('($0)');
    });

    // English, USA
    registerFormatTable('en_US', function () {
        this.createAllDateTimeFormats('M/D/YYYY', 'D-MMM-YY')
            .createAllCurrencyFormats('($0)');
    });

    // English, South Africa
    registerFormatTable('en_ZA', function () {
        this.createAllDateTimeFormats('YYYY/MM/DD', 'DD-MMM-YY', 'long')
            .createAllCurrencyFormats('$-0', { space: true });
    });

    // English, Zimbabwe
    registerFormatTable('en_ZW', function () {
        this.createAllDateTimeFormats('M/D/YYYY', 'D-MMM-YY')
            .createAllCurrencyFormats('($0)');
    });

    // Spanish, Argentina
    registerFormatTable('es_AR', function () {
        this.createAllDateTimeFormats('DD/MM/YYYY', 'DD-MMM-YY', 'long')
            .createAllCurrencyFormats('$-0', { space: true });
    });

    // Spanish, Bolivia
    registerFormatTable('es_BO', function () {
        this.createAllDateTimeFormats('DD/MM/YYYY', 'DD-MMM-YY', 'long')
            .createAllCurrencyFormats('($0)', { space: true });
    });

    // Spanish, Chile
    registerFormatTable('es_CL', function () {
        this.createAllDateTimeFormats('DD-MM-YYYY', 'DD-MMM-YY')
            .createAllCurrencyFormats('-$0', { space: true });
    });

    // Spanish, Colombia
    registerFormatTable('es_CO', function () {
        this.createAllDateTimeFormats('DD/MM/YYYY', 'DD-MMM-YY', 'long')
            .createAllCurrencyFormats('($0)', { space: true });
    });

    // Spanish, Costa Rica
    registerFormatTable('es_CR', function () {
        this.createAllDateTimeFormats('DD/MM/YYYY', 'DD-MMM-YY', 'long')
            .createAllCurrencyFormats('($0)');
    });

    // Spanish, Dominican Republic
    registerFormatTable('es_DO', function () {
        this.createAllDateTimeFormats('DD/MM/YYYY', 'DD-MMM-YY', 'long')
            .createAllCurrencyFormats('($0)');
    });

    // Spanish, Ecuador
    registerFormatTable('es_EC', function () {
        this.createAllDateTimeFormats('DD/MM/YYYY', 'DD-MMM-YY')
            .createAllCurrencyFormats('($0)', { space: true });
    });

    // Spanish, Spain
    registerFormatTable('es_ES', function () {
        this.createAllDateTimeFormats('DD/MM/YYYY', 'DD-MMM-YY')
            .createAllCurrencyFormats('-0$', { space: true });
    });

    // Spanish, Guatemala
    registerFormatTable('es_GT', function () {
        this.createAllDateTimeFormats('DD/MM/YYYY', 'DD-MMM-YY', 'long')
            .createAllCurrencyFormats('($0)');
    });

    // Spanish, Honduras
    registerFormatTable('es_HN', function () {
        this.createAllDateTimeFormats('DD/MM/YYYY', 'DD-MMM-YY', 'long')
            .createAllCurrencyFormats('$-0', { space: true });
    });

    // Spanish, Mexico
    registerFormatTable('es_MX', function () {
        this.createAllDateTimeFormats('DD/MM/YYYY', 'DD-MMM-YY', 'long')
            .createAllCurrencyFormats('-$0');
    });

    // Spanish, Nicaragua
    registerFormatTable('es_NI', function () {
        this.createAllDateTimeFormats('DD/MM/YYYY', 'DD-MMM-YY', 'long')
            .createAllCurrencyFormats('($0)', { space: true });
    });

    // Spanish, Panama
    registerFormatTable('es_PA', function () {
        this.createAllDateTimeFormats('DD/MM/YYYY', 'DD-MMM-YY', 'long')
            .createAllCurrencyFormats('($0)', { space: true });
    });

    // Spanish, Peru
    registerFormatTable('es_PE', function () {
        this.createAllDateTimeFormats('DD/MM/YYYY', 'DD-MMM-YY', 'long')
            .createAllCurrencyFormats('$-0', { space: true });
    });

    // Spanish, Puerto Rico
    registerFormatTable('es_PR', function () {
        this.createAllDateTimeFormats('DD/MM/YYYY', 'DD-MMM-YY', 'long')
            .createAllCurrencyFormats('($0)', { space: true });
    });

    // Spanish, Paraguay
    registerFormatTable('es_PY', function () {
        this.createAllDateTimeFormats('DD/MM/YYYY', 'DD-MMM-YY', 'long')
            .createAllCurrencyFormats('($0)', { space: true });
    });

    // Spanish, El Salvador
    registerFormatTable('es_SV', function () {
        this.createAllDateTimeFormats('DD/MM/YYYY', 'DD-MMM-YY', 'long')
            .createAllCurrencyFormats('($0)');
    });

    // Spanish, Uruguay
    registerFormatTable('es_UY', function () {
        this.createAllDateTimeFormats('DD/MM/YYYY', 'DD-MMM-YY', 'long')
            .createAllCurrencyFormats('($0)', { space: true });
    });

    // Spanish, Venezuela
    registerFormatTable('es_VE', function () {
        this.createAllDateTimeFormats('DD/MM/YYYY', 'DD-MMM-YY', 'long')
            .createAllCurrencyFormats('$-0', { space: true });
    });

    // Estonian, Estonia
    registerFormatTable('et_EE', function () {
        this.createAllDateTimeFormats('D.MM.YYYY', 'D.MMM.YY')
            .createAllCurrencyFormats('-0$', { space: true });
    });

    // Farsi, Iran
    registerFormatTable('fa_IR', function () {
        this.createAllDateTimeFormats('YYYY/MM/DD', 'DD-MMM-YY', 'long')
            .createAllCurrencyFormats('$0-', { space: true });
    });

    // Finnish, Finland
    registerFormatTable('fi_FI', function () {
        this.createFormat(9, '0 %')
            .createFormat(10, '0.00 %')
            .createAllDateTimeFormats('D.M.YYYY', 'D.MMM.YY')
            .createAllCurrencyFormats('-0$', { space: true });
    });

    // Faroese, Faroe Islands
    registerFormatTable('fo_FO', function () {
        this.createAllDateTimeFormats('DD-MM-YYYY', 'DD-MMM-YY', 'long24')
            .createAllCurrencyFormats('$-0', { space: true });
    });

    // French, Belgium
    registerFormatTable('fr_BE', function () {
        this.createAllDateTimeFormats('D/MM/YYYY', 'D-MMM-YY')
            .createAllCurrencyFormats('-0$', { space: true });
    });

    // French, Canada
    registerFormatTable('fr_CA', function () {
        this.createAllDateTimeFormats('YYYY-MM-DD', 'DD-MMM-YY', 'long24')
            .createAllCurrencyFormats('(0$)', { space: true });
    });

    // French, Switzerland
    registerFormatTable('fr_CH', function () {
        this.createAllDateTimeFormats('DD.MM.YYYY', 'DD.MMM.YY', 'long24')
            .createAllCurrencyFormats('$-0', { space: true });
    });

    // French, France
    registerFormatTable('fr_FR', function () {
        this.createAllDateTimeFormats('DD/MM/YYYY', 'DD-MMM-YY', 'long24')
            .createAllCurrencyFormats('-0$', { space: true });
    });

    // French, Luxembourg
    registerFormatTable('fr_LU', function () {
        this.createAllDateTimeFormats('DD/MM/YYYY', 'DD-MMM-YY', 'long24')
            .createAllCurrencyFormats('-0$', { space: true });
    });

    // French, Monaco
    registerFormatTable('fr_MC', function () {
        this.createAllDateTimeFormats('DD/MM/YYYY', 'DD-MMM-YY', 'long24')
            .createAllCurrencyFormats('-0$', { space: true });
    });

    // Galizian, Spain
    registerFormatTable('gl_ES', function () {
        this.createAllDateTimeFormats('DD/MM/YY', 'DD-MMM-YY')
            .createAllCurrencyFormats('-$0', { space: true });
    });

    // Gujarati, India
    registerFormatTable('gu_IN', function () {
        this.createAllDateTimeFormats('DD-MM-YY', 'DD-MMM-YY', 'long24')
            .createAllCurrencyFormats('$-0', { space: true });
    });

    // Hebrew, Israel
    registerFormatTable('he_IL', function () {
        this.createAllDateTimeFormats('DD/MM/YYYY', 'DD-MMM-YY', 'long24')
            .createAllCurrencyFormats('$-0', { space: true });
    });

    // Hindi, India
    registerFormatTable('hi_IN', function () {
        this.createAllDateTimeFormats('DD-MM-YYYY', 'DD-MMM-YY', 'long24')
            .createAllCurrencyFormats('$-0', { space: true });
    });

    // Croatian, Bosnia and Herzegowina
    registerFormatTable('hr_BA', function () {
        this.createAllDateTimeFormats('D.M.YYYY', 'D.MMM.YY')
            .createAllCurrencyFormats('-0$', { space: true });
    });

    // Croatian, Croatia
    registerFormatTable('hr_HR', function () {
        this.createAllDateTimeFormats('D.M.YYYY', 'D.MMM.YY')
            .createAllCurrencyFormats('-0$', { space: true });
    });

    // Hungarian, Hungary
    registerFormatTable('hu_HU', function () {
        this.createAllDateTimeFormats('YYYY.MM.DD', 'DD.MMM.YY')
            .createAllCurrencyFormats('-0$', { space: true });
    });

    // Armenian, Armenia
    registerFormatTable('hy_AM', function () {
        this.createAllDateTimeFormats('DD.MM.YYYY', 'DD.MMM.YY')
            .createAllCurrencyFormats('-0$', { space: true });
    });

    // Indonesian, Indonesia
    registerFormatTable('id_ID', function () {
        this.createAllDateTimeFormats('DD/MM/YYYY', 'DD-MMM-YY')
            .createAllCurrencyFormats('($0)');
    });

    // Icelandic, Iceland
    registerFormatTable('is_IS', function () {
        this.createAllDateTimeFormats('D.M.YYYY', 'D.MMM.YY', 'long24')
            .createAllCurrencyFormats('-0$', { space: true });
    });

    // Italian, Switzerland
    registerFormatTable('it_CH', function () {
        this.createAllDateTimeFormats('DD.MM.YYYY', 'DD.MMM.YY', 'long24')
            .createAllCurrencyFormats('$-0', { space: true });
    });

    // Italian, Italy
    registerFormatTable('it_IT', function () {
        this.createAllDateTimeFormats('DD/MM/YYYY', 'DD-MMM-YY')
            .createAllCurrencyFormats('-$0', { space: true });
    });

    // Japanese, Japan
    registerFormatTable('ja_JP', function () {
        this.createAllDateTimeFormats('YYYY/MM/DD', 'DD-MMM-YY')
            .createAllCurrencyFormats('-$0')
            .createCurrencyFormats(23, '($0)', '$')
            .createFormat(27, '[$-411]GE.MM.DD')
            .createCJKDateTimeFormat(28, '[$-411]GGGE"{YEAR}"MM"{MONTH}"DD"{DAY}"', 'traditional')
            .createFormat(30, 'MM/DD/YY')
            .createCJKDateTimeFormat(31, 'YYYY"{YEAR}"MM"{MONTH}"DD"{DAY}"', 'traditional')
            .createCJKTimeFormats(32, 'traditional')
            .createCJKDateTimeFormat(34, 'YYYY"{YEAR}"MM"{MONTH}"', 'traditional')
            .createCJKDateTimeFormat(35, 'MM"{MONTH}"DD"{DAY}"', 'traditional');
    });

    // Georgian, Georgia
    registerFormatTable('ka_GE', function () {
        this.createAllDateTimeFormats('DD.MM.YYYY', 'DD.MMM.YY')
            .createAllCurrencyFormats('-0$', { space: true });
    });

    // Kazakh, Kazakhstan
    registerFormatTable('kk_KZ', function () {
        this.createAllDateTimeFormats('DD.MM.YYYY', 'DD.MMM.YY')
            .createAllCurrencyFormats('-$0');
    });

    // Kannada, India
    registerFormatTable('kn_IN', function () {
        this.createAllDateTimeFormats('DD-MM-YY', 'DD-MMM-YY', 'long24')
            .createAllCurrencyFormats('$-0', { space: true });
    });

    // Korean, South Korea
    registerFormatTable('ko_KR', function () {
        this.createAllDateTimeFormats('YYYY-MM-DD', 'DD-MMM-YY')
            .createAllCurrencyFormats('-$0')
            .createCurrencyFormats(23, '($0)', '$')
            .createCJKDateTimeFormat(27, 'YYYY{YEAR} MM{MONTH} DD{DAY}', 'traditional')
            .createFormat(28, 'MM-DD')
            .createFormat(30, 'MM-DD-YY')
            .createCJKDateTimeFormat(31, 'YYYY{YEAR} MM{MONTH} DD{DAY}', 'hangul')
            .createCJKTimeFormats(32, 'hangul')
            .createFormat(34, 'YYYY/MM/DD')
            .createFormat(35, 14);
    });

    // Konkani, India
    registerFormatTable('kok_IN', 'hi_IN');

    // Kyrgyz, Kyrgyzstan
    registerFormatTable('ky_KG', function () {
        this.createAllDateTimeFormats('DD.MM.YY', 'DD.MMM.YY')
            .createAllCurrencyFormats('-0$', { space: true });
    });

    // Lithuanian, Lithuania
    registerFormatTable('lt_LT', function () {
        this.createAllDateTimeFormats('YYYY.MM.DD', 'DD.MMM.YY', 'long24')
            .createAllCurrencyFormats('-0$', { space: true });
    });

    // Latvian, Latvia
    registerFormatTable('lv_LV', function () {
        this.createAllDateTimeFormats('YYYY.MM.DD', 'DD.MMM.YY')
            .createAllCurrencyFormats('-$0', { space: true });
    });

    // Maori, New Zealand
    registerFormatTable('mi_NZ', 'en_NZ');

    // Malayalam, India
    registerFormatTable('ml_IN', function () {
        this.createAllDateTimeFormats('DD-MM-YY', 'DD-MMM-YY', 'long24')
            .createAllCurrencyFormats('$-0', { space: true });
    });

    // Mongolian, Mongolia
    registerFormatTable('mn_MN', function () {
        this.createAllDateTimeFormats('YY.MM.DD', 'DD.MMM.YY')
            .createAllCurrencyFormats('-0$');
    });

    // Marathi, India
    registerFormatTable('mr_IN', 'hi_IN');

    // Malay, Brunei Darussalam
    registerFormatTable('ms_BN', function () {
        this.createAllDateTimeFormats('DD/MM/YYYY', 'DD-MMM-YY')
            .createAllCurrencyFormats('($0)');
    });

    // Malay, Malaysia
    registerFormatTable('ms_MY', function () {
        this.createAllDateTimeFormats('DD/MM/YYYY', 'DD-MMM-YY')
            .createAllCurrencyFormats('($0)');
    });

    // Maltese, Malta
    registerFormatTable('mt_MT', function () {
        this.createAllDateTimeFormats('DD/MM/YYYY', 'DD-MMM-YY', 'long24')
            .createAllCurrencyFormats('-$0');
    });

    // Norwegian (Bokmal), Norway
    registerFormatTable('nb_NO', function () {
        this.createAllDateTimeFormats('DD.MM.YYYY', 'DD.MMM.YY', 'long24')
            .createAllCurrencyFormats('$-0', { space: true });
    });

    // Dutch, Belgium
    registerFormatTable('nl_BE', function () {
        this.createAllDateTimeFormats('D/MM/YYYY', 'D/MMM/YY')
            .createAllCurrencyFormats('-0$', { space: true });
    });

    // Dutch, Netherlands
    registerFormatTable('nl_NL', function () {
        this.createAllDateTimeFormats('D-M-YYYY', 'D-MMM-YY')
            .createAllCurrencyFormats('$0-', { space: true });
    });

    // Norwegian (Nynorsk), Norway
    registerFormatTable('nn_NO', 'nb_NO');

    // Northern Sotho, South Africa
    registerFormatTable('nso_ZA', 'en_ZA');

    // Punjabi, India
    registerFormatTable('pa_IN', function () {
        this.createAllDateTimeFormats('DD-MM-YY', 'DD-MMM-YY', 'long')
            .createAllCurrencyFormats('$-0', { space: true });
    });

    // Polish, Poland
    registerFormatTable('pl_PL', function () {
        this.createAllDateTimeFormats('YYYY-MM-DD', 'DD-MMM-YY', 'long24')
            .createAllCurrencyFormats('-0$', { space: true });
    });

    // Portugese, Brazil
    registerFormatTable('pt_BR', function () {
        this.createAllDateTimeFormats('D/M/YYYY', 'D/MMM/YY', 'long24')
            .createAllCurrencyFormats('($0)', { space: true });
    });

    // Portugese, Portugal
    registerFormatTable('pt_PT', function () {
        this.createAllDateTimeFormats('DD-MM-YYYY', 'DD-MMM-YY')
            .createAllCurrencyFormats('-0$', { space: true });
    });

    // Quechua, Bolivia
    registerFormatTable('qu_BO', 'es_BO');

    // Quechua, Ecuador
    registerFormatTable('qu_EC', 'es_EC');

    // Quechua, Peru
    registerFormatTable('qu_PE', 'es_PE');

    // Romanian, Romania
    registerFormatTable('ro_RO', function () {
        this.createAllDateTimeFormats('DD.MM.YYYY', 'DD.MMM.YY', 'long24')
            .createAllCurrencyFormats('-0$', { space: true });
    });

    // Russian, Russian Federation
    registerFormatTable('ru_RU', function () {
        this.createAllDateTimeFormats('DD.MM.YYYY', 'DD.MMM.YY')
            .createAllCurrencyFormats('-0$');
    });

    // Sanskrit, India
    registerFormatTable('sa_IN', 'hi_IN');

    // Sami, Finland
    registerFormatTable('se_FI', 'fi_FI');

    // Sami, Norway
    registerFormatTable('se_NO', 'nb_NO');

    // Sami, Sweden
    registerFormatTable('se_SE', 'sv_SE');

    // Slovak, Slovakia
    registerFormatTable('sk_SK', function () {
        this.createAllDateTimeFormats('D.M.YYYY', 'D.MMM.YY')
            .createAllCurrencyFormats('-0$', { space: true });
    });

    // Slovenian, Slovenia
    registerFormatTable('sl_SI', function () {
        this.createAllDateTimeFormats('D.M.YYYY', 'D.MMM.YY')
            .createAllCurrencyFormats('-0$', { space: true });
    });

    // Swedish, Finland
    registerFormatTable('sv_FI', function () {
        this.createFormat(9, '0 %')
            .createFormat(10, '0.00 %')
            .createAllDateTimeFormats('D.M.YYYY', 'D.MMM.YY', 'long24')
            .createAllCurrencyFormats('-0$', { space: true });
    });

    // Swedish, Sweden
    registerFormatTable('sv_SE', function () {
        this.createAllDateTimeFormats('YYYY-MM-DD', 'DD-MMM-YY', 'long24')
            .createAllCurrencyFormats('-0$', { space: true });
    });

    // Swahili, Tanzania
    registerFormatTable('sw_TZ', function () {
        this.createAllDateTimeFormats('M/D/YYYY', 'D-MMM-YY')
            .createAllCurrencyFormats('($0)');
    });

    // Syriac, Syria
    registerFormatTable('syr_SY', 'ar_SY');

    // Syriac, Turkey
    registerFormatTable('syr_TR', 'tr_TR');

    // Tamil, India
    registerFormatTable('ta_IN', function () {
        this.createAllDateTimeFormats('DD-MM-YYYY', 'DD-MMM-YY', 'long24')
            .createAllCurrencyFormats('$-0', { space: true });
    });

    // Telugu, India
    registerFormatTable('te_IN', function () {
        this.createAllDateTimeFormats('DD-MM-YY', 'DD-MMM-YY', 'long24')
            .createAllCurrencyFormats('$-0', { space: true });
    });

    // Thai, Thailand
    registerFormatTable('th_TH', function () {
        this.createAllDateTimeFormats('D/M/YYYY', 'D-MMM-YY')
            .createAllCurrencyFormats('-$0')
            .createFormat(59, 't0')
            .createFormat(60, 't0.00')
            .createFormat(61, 't#,##0')
            .createFormat(62, 't#,##0.00')
            .createCurrencyFormats(63, '($0)', this._lcData.currency, { modifier: 't' })
            .createFormat(67, 't0%')
            .createFormat(68, 't0.00%')
            .createFormat(69, 't' + PresetFormatTable.getFractionCode(1))
            .createFormat(70, 't' + PresetFormatTable.getFractionCode(2))
            .createFormat(71, 'tD/M/EE')
            .createFormat(72, 'tD-MMM-E')
            .createFormat(73, 'tD-MMM')
            .createFormat(74, 'tMMM-E')
            .createFormat(75, 'th:mm')
            .createFormat(76, 'th:mm:ss')
            .createFormat(77, 'tD/M/EE h:mm')
            .createFormat(78, 'tmm:ss')
            .createFormat(79, 't[h]:mm:ss')
            .createFormat(80, 'tmm:ss.0')
            .createFormat(81, 'D/M/E');
    });

    // Tswana, South Africa
    registerFormatTable('tn_ZA', 'en_ZA');

    // Turkish, Turkey
    registerFormatTable('tr_TR', function () {
        this.createAllDateTimeFormats('DD.MM.YYYY', 'DD.MMM.YY', 'long24')
            .createAllCurrencyFormats('-0$', { space: true });
    });

    // Tatar, Russian Federation
    registerFormatTable('tt_RU', function () {
        this.createAllDateTimeFormats('DD.MM.YYYY', 'DD.MMM.YY')
            .createAllCurrencyFormats('-0$', { space: true });
    });

    // Ukrainian, Ukraine
    registerFormatTable('uk_UA', function () {
        this.createAllDateTimeFormats('DD.MM.YYYY', 'DD.MMM.YY')
            .createAllCurrencyFormats('-0$', { space: true });
    });

    // Urdu, Pakistan
    registerFormatTable('ur_PK', function () {
        this.createAllDateTimeFormats('DD/MM/YYYY', 'DD-MMM-YY')
            .createAllCurrencyFormats('$0-');
    });

    // Vietnamese, Viet Nam
    registerFormatTable('vi_VN', function () {
        this.createAllDateTimeFormats('DD/MM/YYYY', 'DD-MMM-YY')
            .createAllCurrencyFormats('-0$', { space: true });
    });

    // Xhosa, South Africa
    registerFormatTable('xh_ZA', 'en_ZA');

    // Chinese, China
    registerFormatTable('zh_CN', function () {
        this.createAllDateTimeFormats('YYYY-M-D', 'D-MMM-YY')
            .createAllCurrencyFormats('$-0')
            .createCurrencyFormats(23, '($0)', '$')
            .createCJKDateTimeFormat(27, 'YYYY"{YEAR}"M"{MONTH}"', 'simplified')
            .createCJKDateTimeFormat(28, 'M"{MONTH}"D"{DAY}"', 'simplified')
            .createFormat(30, 'M-D-YY')
            .createCJKDateTimeFormat(31, 'YYYY"{YEAR}"M"{MONTH}"D"{DAY}"', 'simplified')
            .createCJKTimeFormats(32, 'simplified')
            .createCJKTimeFormats(34, 'simplified', { ampm: true })
            .createFormat(52, 27)
            .createFormat(53, 28);
    });

    // Chinese, Hong Kong
    registerFormatTable('zh_HK', function () {
        this.createAllDateTimeFormats('D/M/YYYY', 'D-MMM-YY')
            .createAllCurrencyFormats('($0)')
            .createCurrencyFormats(23, '($0)', 'US$')
            .createFormat(27, '[$-404]D/M/E')
            .createCJKDateTimeFormat(28, '[$-404]D"{DAY}"M"{MONTH}"E"{YEAR}"', 'traditional')
            .createFormat(30, 'M/D/YY')
            .createCJKDateTimeFormat(31, 'D"{DAY}"M"{MONTH}"YYYY"{YEAR}"', 'traditional')
            .createCJKTimeFormats(32, 'traditional')
            .createCJKTimeFormats(34, 'traditional', { ampm: true });
    });

    // Chinese, Macau
    registerFormatTable('zh_MO', function () {
        this.createAllDateTimeFormats('D/M/YYYY', 'D-MMM-YY')
            .createAllCurrencyFormats('($0)')
            .createCurrencyFormats(23, '($0)', 'US$')
            .createFormat(27, '[$-404]D/M/E')
            .createCJKDateTimeFormat(28, '[$-404]D"{DAY}"M"{MONTH}"E"{YEAR}"', 'traditional')
            .createFormat(30, 'M/D/YY')
            .createCJKDateTimeFormat(31, 'D"{DAY}"M"{MONTH}"YYYY"{YEAR}"', 'traditional')
            .createCJKTimeFormats(32, 'traditional')
            .createCJKTimeFormats(34, 'traditional', { ampm: true });
    });

    // Chinese, Singapore
    registerFormatTable('zh_SG', function () {
        this.createAllDateTimeFormats('D/M/YYYY', 'D-MMM-YY')
            .createAllCurrencyFormats('($0)')
            .createCurrencyFormats(23, '($0)', '$')
            .createCJKDateTimeFormat(27, 'YYYY"{YEAR}"M"{MONTH}"', 'simplified')
            .createCJKDateTimeFormat(28, 'M"{MONTH}"D"{DAY}"', 'simplified')
            .createFormat(30, 'M/D/YY')
            .createCJKDateTimeFormat(31, 'D"{DAY}"M"{MONTH}"YYYY"{YEAR}"', 'simplified')
            .createCJKTimeFormats(32, 'simplified')
            .createCJKTimeFormats(34, 'simplified', { ampm: true });
    });

    // Chinese, Taiwan
    registerFormatTable('zh_TW', function () {
        this.createAllDateTimeFormats('YYYY/M/D', 'D-MMM-YY', 'long')
            .createAllCurrencyFormats('-$0')
            .createCurrencyFormats(23, '($0)', 'US$')
            .createFormat(27, '[$-404]E/M/D')
            .createCJKDateTimeFormat(28, '[$-404]E"{YEAR}"M"{MONTH}"D"{DAY}"', 'traditional')
            .createFormat(30, 'M/D/YY')
            .createCJKDateTimeFormat(31, 'YYYY"{YEAR}"M"{MONTH}"D"{DAY}"', 'traditional')
            .createCJKTimeFormats(32, 'traditional', { long: true })
            .createCJKTimeFormats(34, 'traditional', { long: true, ampm: true });
    });

    // Zulu, South Africa
    registerFormatTable('zu_ZA', 'en_ZA');

    // exports ================================================================

    return PresetFormatTable;

});
