/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/spreadsheet/model/formula/utils/scalar', [
    'io.ox/office/tk/utils',
    'io.ox/office/spreadsheet/utils/sheetutils',
    'io.ox/office/spreadsheet/model/formula/utils/mathutils'
], function (Utils, SheetUtils, MathUtils) {

    'use strict';

    // convenience shortcuts
    var ErrorCode = SheetUtils.ErrorCode;

    // shortcuts to mathematical functions
    var round = Math.round;
    var abs = Math.abs;
    var pow = Math.pow;
    var min = Math.min;
    var isZero = MathUtils.isZero;

    // static private functions ===============================================

    /**
     * Returns the neutral value for the passed scalar value.
     *
     * @param {Number|String|Boolean|ErrorCode|Null} value
     *  Any scalar value used in spreadsheet formulas.
     *
     * @returns {Number|String|Boolean|Null}
     *  The neutral value for the passed scalar value, if available. Returns
     *  the number zero for any number, the empty string for any string, or the
     *  value false for a boolean value. Otherwise, null will be returned
     *  (especially for error codes which do not have a neutral value).
     */
    function getNeutralValue(value) {
        switch (typeof value) {
            case 'number': return 0;
            case 'string': return '';
            case 'boolean': return false;
        }
        return null; // null for error codes
    }

    /**
     * Compares the passed integers, and returns a signed indicator number.
     *
     * @param {Number} int1
     *  The first integer for comparison. MUST be finite.
     *
     * @param {Number} int2
     *  The second integer for comparison. MUST be finite.
     *
     * @returns {Number}
     *  The number -1, if int1 is less than int2; or the number 1, if int1 is
     *  greater than int2; or the number 0, if both integers are equal.
     */
    function compareIntegers(int1, int2) {
        return (int1 === int2) ? 0 : (int1 < int2) ? -1 : 1;
    }

    /**
     * Rounds the passed number to 15 significant digits.
     *
     * @param {Number} number
     *  The number to be rounded. MUST be finite.
     *
     * @returns {Number}
     *  The rounded number.
     */
    function round15(number) {
        var norm = Utils.normalizeNumber(number);
        return round(norm.mant * 1e14) * pow(10, norm.exp - 14);
    }

    /**
     * Returns whether the passed numbers are considered to be equal, allowing
     * a slight difference for equally signed numbers.
     *
     * @param {Number} number1
     *  The first number for comparison. MUST be finite.
     *
     * @param {Number} number2
     *  The second number for comparison. MUST be finite.
     *
     * @returns {Boolean}
     *  Whether the passed numbers are considered to be equal.
     */
    function equalNumbers(number1, number2) {

        // shortcut: true, if the numbers are exactly equal
        if (number1 === number2) { return true; }

        // check whether at least one number is zero or denormalized
        var zero1 = isZero(number1);
        var zero2 = isZero(number2);
        if (zero1 || zero2) { return zero1 === zero2; }

        // false, if the numbers have different signs
        if ((number1 < 0) !== (number2 < 0)) { return false; }

        // performance: false, if the numbers have a decent difference
        var quot = abs((number1 / number2) - 1);
        if (!isFinite(quot) || (quot > 1e-14)) { return false; }

        // round the numbers to 15 significant digits for comparison
        return round15(number1) === round15(number2);
    }

    /**
     * Returns whether the passed strings are considered to be equal.
     *
     * @param {String} string1
     *  The first string for comparison.
     *
     * @param {String} string2
     *  The second string for comparison.
     *
     * @param {Boolean} withCase
     *  If set to true, the strings will be compared case-sensitively.
     *
     * @returns {Boolean}
     *  Whether the passed strings are considered to be equal.
     */
    function equalStrings(string1, string2, withCase) {
        return withCase ? (string1 === string2) : (string1.toUpperCase() === string2.toUpperCase());
    }

    // static class Scalar ====================================================

    /**
     * Static helper methods for dealing with scalar values in formulas. Scalar
     * values are numbers, strings, boolean values, error codes, and the value
     * null that represents a blank cell, or an empty function parameter.
     */
    var Scalar = {};

    // constants --------------------------------------------------------------

    /**
     * Type identifiers for different types of scalar values used in formulas.
     * The values of the type identifiers represent a natural order for
     * different data types, as used in spreadsheet formulas or for sorting
     * (for example, strings are always considered to be greater than numbers).
     *
     * @constant
     * @type {Object<Number>}
     *
     * @property {Number} NULL
     *  Type specifier for the special value null representing a blank cell or
     *  an empty function parameter. Used in situations when null values will
     *  be considered to be less than any other scalar value.
     *
     * @property {Number} NUMBER
     *  Type specifier for floating-point numbers.
     *
     * @property {Number} STRING
     *  Type specifier for strings.
     *
     * @property {Number} BOOLEAN
     *  Type specifier for boolean values.
     *
     * @property {Number} ERROR
     *  Type specifier for error codes (instances of class ErrorCode).
     *
     * @property {Number} NULL_MAX
     *  Type specifier for the special value null representing a blank cell or
     *  an empty function parameter. Used in situations when null values will
     *  be considered to be greater than any other scalar value.
     */
    Scalar.Type = {
        NULL: 0,
        NUMBER: 1,
        STRING: 2,
        BOOLEAN: 3,
        ERROR: 4,
        NULL_MAX: 5
    };

    // static methods ---------------------------------------------------------

    /**
     * Returns the effective result of a cell formula. If the formula evaluates
     * to the empty value (null, e.g. reference to a blank cell), the number
     * zero will be returned instead.
     *
     * @param {Number|String|Boolean|ErrorCode|Null} value
     *  The scalar value to be checked.
     *
     * @returns {Number|String|Boolean|ErrorCode}
     *  The number zero, if the passed value is null; otherwise the passed
     *  value itself.
     */
    Scalar.getCellValue = function (value) {
        return (value === null) ? 0 : value;
    };

    // comparison -------------------------------------------------------------

    /**
     * Returns a numeric type identifier for the passed scalar value. The type
     * identifier represents a natural order of different data types, as used
     * in spreadsheet formulas (for example, strings are always considered to
     * be greater than numbers).
     *
     * @param {Number|String|Boolean|ErrorCode|Null} value
     *  The scalar value to be checked.
     *
     * @param {Boolean} [nullGreater=false]
     *  Whether to consider the null value to be greater than any other value.
     *  By default, the null value is less than any other value.
     *
     * @returns {Number}
     *  A numeric type identifier for the passed scalar value:
     *  - Scalar.Type.NUMBER for a number (including infinite numbers, and NaN).
     *  - Scalar.Type.STRING for a string.
     *  - Scalar.Type.BOOLEAN for boolean values.
     *  - Scalar.Type.ERROR for error codes.
     *  - Scalar.Type.NULL for the value null, if the option 'nullGreater' has
     *      NOT been set to true.
     *  - Scalar.Type.NULL_MAX for the value null, if the option 'nullGreater'
     *      has been set to true.
     */
    Scalar.getType = function (value, nullGreater) {
        switch (typeof value) {
            case 'number':  return Scalar.Type.NUMBER;
            case 'string':  return Scalar.Type.STRING;
            case 'boolean': return Scalar.Type.BOOLEAN;
        }
        return (value instanceof ErrorCode) ? Scalar.Type.ERROR :
            nullGreater ? Scalar.Type.NULL_MAX : Scalar.Type.NULL;
    };

    /**
     * Returns whether the passed numbers are considered to be equal, allowing
     * a slight difference for equally signed numbers. This method may be
     * faster than the method Scalar.compareNumbers() in some situations.
     *
     * @param {Number} number1
     *  The first number for comparison.
     *
     * @param {Number} number2
     *  The second number for comparison.
     *
     * @returns {Boolean}
     *  Whether the passed numbers are both finite, and considered to be equal.
     */
    Scalar.equalNumbers = function (number1, number2) {
        // false, if one of the numbers is not finite
        return isFinite(number1) && isFinite(number2) && equalNumbers(number1, number2);
    };

    /**
     * Returns whether the passed strings are considered to be equal. This
     * method may be faster than the method Scalar.compareStrings() in some
     * situations.
     *
     * @param {String} string1
     *  The first string for comparison.
     *
     * @param {String} string2
     *  The second string for comparison.
     *
     * @param {Object} [options]
     *  Optional parameters:
     *  @param {Boolean} [options.withCase=false]
     *      If set to true, the strings will be compared case-sensitively.
     *
     * @returns {Boolean}
     *  Whether the passed strings are considered to be equal.
     */
    Scalar.equalStrings = function (string1, string2, options) {
        var withCase = Utils.getBooleanOption(options, 'withCase', false);
        return equalStrings(string1, string2, withCase);
    };

    /**
     * Returns whether the passed scalar values are considered to be equal.
     * This method may be faster than the method Scalar.compare() in some
     * situations.
     *
     * @param {Number|String|Boolean|ErrorCode|Null} value1
     *  The first scalar value for comparison.
     *
     * @param {Number|String|Boolean|ErrorCode|Null} value2
     *  The second scalar value for comparison.
     *
     * @param {Object} [options]
     *  Optional parameters:
     *  @param {Boolean} [options.withCase=false]
     *      If set to true, strings will be compared case-sensitively. By
     *      default, the character case of strings will be ignored.
     *  @param {Boolean} [options.nullMode='convert']
     *      Specifies what happens, if one of the passed values is null, and
     *      the other value is not null (two null values are always equal):
     *      - 'convert' (default): The null value will be replaced with the
     *          neutral value of the data type of the other value (zero, empty
     *          string, or false), and these values will be compared. If the
     *          other value is an error code, the null value will not be
     *          converted, but will be considered different to the error code.
     *      - 'less': The null value is different to the other value.
     *      - 'greater': The null value is different to the other value.
     *      The options 'less' and 'greater' are supported for consistency with
     *      the method Scalar.compare().
     *
     * @returns {Boolean}
     *  Whether the passed scalar values are considered to be equal, according
     *  to the passed options.
     */
    Scalar.equal = function (value1, value2, options) {

        // whether to convert a null value to the neutral value of the other parameter
        var convertNull = Utils.getStringOption(options, 'nullMode', 'convert') === 'convert';

        // convert null values if specified
        if (convertNull) {
            if (value1 === null) { value1 = getNeutralValue(value2); }
            if (value2 === null) { value2 = getNeutralValue(value1); }
        }

        // shortcut: simple check for strict equality
        if (value1 === value2) { return true; }

        // get type identifiers (values of different types are never equal)
        var type1 = Scalar.getType(value1);
        var type2 = Scalar.getType(value2);
        if (type1 !== type2) { return false; }

        // compare numbers and strings using the helper methods
        switch (type1) {
            case Scalar.Type.NUMBER: return Scalar.equalNumbers(value1, value2);
            case Scalar.Type.STRING: return Scalar.equalStrings(value1, value2, options);
        }

        // equal booleans, error codes, and null values would have been found above already
        return false;
    };

    /**
     * Compares the passed numbers, and returns a signed indicator number,
     * allowing a slight difference for equally signed numbers.
     *
     * @param {Number} number1
     *  The first number for comparison.
     *
     * @param {Number} number2
     *  The second number for comparison.
     *
     * @returns {Number}
     *  The number -1, if number1 is less than number2; or the number 1, if
     *  number1 is greater than number2; or the number 0, if both numbers are
     *  equal. If the difference of the numbers is less than Utils.MIN_NUMBER
     *  (e.g., due to rounding errors), the numbers are also considered to be
     *  equal. If any of the numbers is NaN or infinite, the result will be the
     *  special value NaN.
     */
    Scalar.compareNumbers = function (number1, number2) {

        // return NaN for any invalid numbers
        if (!isFinite(number1) || !isFinite(number2)) { return Number.NaN; }

        // return comparison specifier for the (finite) numbers
        return equalNumbers(number1, number2) ? 0 : (number1 < number2) ? -1 : 1;
    };

    /**
     * Compares the passed strings lexicographically, and returns a signed
     * indicator number.
     *
     * @param {Boolean} string1
     *  The first string for comparison.
     *
     * @param {Boolean} string2
     *  The second string for comparison.
     *
     * @param {Object} [options]
     *  Optional parameters:
     *  @param {Boolean} [options.withCase=false]
     *      If set to true, the strings will be compared case-sensitively.
     *      Capital letters will be considered greater than their lower-case
     *      variants. By default, the character case of the strings will be
     *      ignored.
     *
     * @returns {Number}
     *  The number -1, if string1 is lexicographically less than string2; or
     *  the number 1, if string1 is lexicographically greater than string2;
     *  otherwise the number 0 (the strings are equal).
     */
    Scalar.compareStrings = function (string1, string2, options) {

        // quick check for exact equality
        if (string1 === string2) { return 0; }

        // compare the strings regarding the character case
        if (Utils.getBooleanOption(options, 'withCase', false)) {

            // the lengths of the passed strings
            var len1 = string1.length;
            var len2 = string2.length;

            // Quick check if the longer string starts exactly with the shorter string. In this case,
            // the longer string is considered greater than the shorter string. This includes the
            // special case that both strings are exactly equal.
            var minLen = min(len1, len2);
            if (string1.substr(0, minLen) === string2.substr(0, minLen)) {
                return compareIntegers(len1, len2);
            }

            // difference in character case (less priority than 'really different' characters)
            var caseDiff = 0;

            // the strings differ in at least one character, check the single characters in a loop
            for (var i = 0; i < minLen; i += 1) {

                // extract the characters at the current position; continue with next character,
                // if both characters are equal
                var c1 = string1[i], c2 = string2[i];
                if (c1 === c2) { continue; }

                // if the upper-case versions of the characters are different, just compare them
                var u1 = c1.toUpperCase(), u2 = c2.toUpperCase();
                if (u1 < u2) { return -1; }
                if (u1 > u2) { return 1; }

                // characters differ in character case only, cache the first difference only
                if ((caseDiff === 0) && (c1 !== c2)) {
                    // the first character is less than than second, if and only if the first character
                    // is lower-case, AND the second character is upper-case
                    caseDiff = ((c1 !== u1) && (c2 === u2)) ? -1 : 1;
                }
            }

            // the strings differ only in character case: ignore character case completely,
            // if the length of the strings are different
            return (len1 === len2) ? caseDiff : compareIntegers(len1, len2);
        }

        // compare the strings case-insensitively
        string1 = string1.toUpperCase();
        string2 = string2.toUpperCase();
        return (string1 < string2) ? -1 : (string1 > string2) ? 1 : 0;
    };

    /**
     * Compares the passed boolean values, and returns a signed indicator
     * number.
     *
     * @param {Boolean} bool1
     *  The first boolean value for comparison.
     *
     * @param {Boolean} bool2
     *  The second boolean value for comparison.
     *
     * @returns {Number}
     *  The number -1, if bool1 is FALSE, and bool2 is TRUE (bool1 is 'less
     *  than' bool2); or the number 1, if bool1 is TRUE, and bool2 is FALSE
     *  (bool1 is 'greater than' bool2); otherwise the number 0 (the boolean
     *  values are equal).
     */
    Scalar.compareBooleans = function (bool1, bool2) {
        // subtraction operator converts booleans to numbers implicitly
        return bool1 - bool2;
    };

    /**
     * Compares the internal indexes of built-in error codes, and returns a
     * signed indicator number.
     *
     * @param {ErrorCode} errorCode1
     *  The first error code for comparison.
     *
     * @param {ErrorCode} errorCode2
     *  The second error code for comparison.
     *
     * @returns {Number}
     *  The value NaN, if any of the passed error codes does not represent a
     *  built-in error (the property 'num' of the error code is not finite);
     *  the number -1, if the index of the first error code is less than the
     *  index of the second error code; the number 1, if the index of the first
     *  error code is greater than the index of the second error code, or the
     *  number 0, if both error codes have equal indexes.
     */
    Scalar.compareErrorCodes = function (errorCode1, errorCode2) {
        return (isFinite(errorCode1.num) && isFinite(errorCode2.num)) ? compareIntegers(errorCode1.num, errorCode2.num) : Number.NaN;
    };

    /**
     * Compares the passed scalar values, and returns a signed indicator
     * number.
     *
     * @param {Number|String|Boolean|ErrorCode|Null} value1
     *  The first scalar value for comparison.
     *
     * @param {Number|String|Boolean|ErrorCode|Null} value2
     *  The second scalar value for comparison.
     *
     * @param {Object} [options]
     *  Optional parameters:
     *  @param {Boolean} [options.withCase=false]
     *      If set to true, strings will be compared case-sensitively. Capital
     *      letters will be considered greater than their lower-case variants.
     *      By default, the character case of strings will be ignored.
     *  @param {Boolean} [options.nullMode='convert']
     *      Specifies what happens, if one of the passed values is null, and
     *      the other value is not null (two null values are always equal):
     *      - 'convert' (default): The null value will be replaced with the
     *          neutral value of the data type of the other value (zero, empty
     *          string, or false), and these values will be compared. If the
     *          other value is an error code, the null value will not be
     *          converted, but will be considered less than the error code.
     *      - 'less': The null value is less than the other value.
     *      - 'greater': The null value is greater than the other value.
     *
     * @returns {Number}
     *  The number -1, if value1 is considered less than value2; or the number
     *  1, if value1 is considered greater than value2; otherwise the number 0
     *  (the values are equal). Numbers are always less than strings, strings
     *  are always less than boolean values, and boolean values are always less
     *  than error codes. The null value will be treated according to the
     *  option 'nullMode'. Values of the same type will be compared as
     *  described for the methods Scalar.compareNumbers(),
     *  Scalar.compareStrings(), Scalar.compareBooleans(), and
     *  Scalar.compareErrorCodes().
     */
    Scalar.compare = function (value1, value2, options) {

        // behavior for null values
        var nullMode = Utils.getStringOption(options, 'nullMode', 'convert');
        // whether the null value is considered greater than any other value (not for 'convert' mode)
        var nullGreater = nullMode === 'greater';

        // convert null values if specified
        if (nullMode === 'convert') {
            if (value1 === null) { value1 = getNeutralValue(value2); }
            if (value2 === null) { value2 = getNeutralValue(value1); }
        }

        // get type identifiers, compare different types (ignore the actual values)
        var type1 = Scalar.getType(value1, nullGreater);
        var type2 = Scalar.getType(value2, nullGreater);
        if (type1 !== type2) { return compareIntegers(type1, type2); }

        // compare values of equal types
        switch (type1) {
            case Scalar.Type.NUMBER:  return Scalar.compareNumbers(value1, value2);
            case Scalar.Type.STRING:  return Scalar.compareStrings(value1, value2, options);
            case Scalar.Type.BOOLEAN: return Scalar.compareBooleans(value1, value2);
            case Scalar.Type.ERROR:   return Scalar.compareErrorCodes(value1, value2);
        }

        // both values are null
        return 0;
    };

    /**
     * Returns the string representation of the passed scalar value for debug
     * logging.
     *
     * @attention
     *  DO NOT USE this method for GUI related code where correct locale
     *  dependent representation of the passed value is required.
     *
     * @param {Any} value
     *  The scalar value to be converted to the string representation.
     *
     * @returns {String}
     *  The string representation of the passed value.
     */
    Scalar.toString = function (value) {
        return (typeof value === 'string') ? ('"' + value.replace(/"/g, '""') + '"') :
            (value instanceof Date) ? value.toISOString() :
            (value === null) ? 'null' :
            (value === true) ? 'TRUE' :
            (value === false) ? 'FALSE' :
            value.toString();
    };

    // exports ================================================================

    return Scalar;

});
