/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/spreadsheet/model/formula/parser/tokens', [
    'io.ox/office/tk/utils',
    'io.ox/office/spreadsheet/utils/sheetutils',
    'io.ox/office/spreadsheet/model/formula/formulautils'
], function (Utils, SheetUtils, FormulaUtils) {

    'use strict';

    // convenience shortcuts
    var ErrorCode = SheetUtils.ErrorCode;
    var Interval = SheetUtils.Interval;
    var Address = SheetUtils.Address;
    var Range = SheetUtils.Range;
    var Range3D = SheetUtils.Range3D;
    var RangeArray = SheetUtils.RangeArray;
    var IntervalArray = SheetUtils.IntervalArray;
    var Range3DArray = SheetUtils.Range3DArray;
    var MoveDescriptor = SheetUtils.MoveDescriptor;
    var Scalar = FormulaUtils.Scalar;
    var CellRef = FormulaUtils.CellRef;
    var SheetRef = FormulaUtils.SheetRef;

    // class BaseToken ========================================================

    /**
     * Base class for all formula tokens used in the formula parser.
     *
     * @constructor
     *
     * @param {String} type
     *  The token type identifier.
     *
     * @param {Function} generator
     *  A callback function that implements converting the current value of
     *  this token to its text representation according to a specific formula
     *  grammar. Receives all parameters passed to the public method
     *  BaseToken.getText(), and must return a string. Will be invoked in the
     *  context of this token instance.
     *
     * @param {Function} toString
     *  A callback function that implements converting the current value of
     *  this token to its text representation for debug logging. Does not
     *  receive any parameter, and must return a string
     */
    var BaseToken = _.makeExtendable(function (type) {

        // private properties
        this._type = type;

    }); // class BaseToken

    // public methods ---------------------------------------------------------

    /**
     * Returns the type identifier of this token.
     *
     * @returns {String}
     *  The type identifier of this token.
     */
    BaseToken.prototype.getType = function () {
        return this._type;
    };

    /**
     * Returns whether the type of this token matches the passed type.
     *
     * @param {String} test
     *  A type identifier that must match the type of this token exactly.
     *
     * @returns {Boolean}
     *  Whether the type of this token matches the passed type.
     */
    BaseToken.prototype.isType = function (test) {
        return test === this._type;
    };

    /**
     * Returns whether the type identifier of this token matches the passed
     * regular expression.
     *
     * @param {RegExp} regExp
     *  A regular expression that will be tested against the type of this
     *  token.
     *
     * @returns {Boolean}
     *  Whether the type of this token matches the regular expression.
     */
    BaseToken.prototype.matchesType = function (regExp) {
        return regExp.test(this._type);
    };

    /**
     * All sub classes MUST overwrite this dummy method to generate the text
     * representation of this token, according to the passed grammar
     * configuration.
     *
     * @param {FormulaGrammar} formulaGrammar
     *  The formula grammar containing specific settings for formula tokens.
     *
     * @param {Object} [options]
     *  Optional parameters for relocation of range addresses, or for other
     *  token-specific special behavior. See descriptions in the subclasses for
     *  more details.
     *
     * @returns {String}
     *  The text representation of this token.
     */
    BaseToken.prototype.getText = function (/*formulaGrammar, options*/) {
        Utils.error('BaseToken.getText(): missing implementation');
        return '';
    };

    /**
     * All sub classes MUST overwrite this dummy method to generate a text
     * description of the contents of this token for debugging (!) purposes.
     *
     * @returns {String}
     *  The debug text representation of the contents of this token.
     */
    BaseToken.prototype.debugText = function () {
        Utils.error('BaseToken.debugText(): missing implementation');
        return '';
    };

    /**
     * Generates a text description of this token for debugging (!) purposes.
     *
     * @returns {String}
     *  The complete debug text representation of this token.
     */
    BaseToken.prototype.toString = function () {
        var txt = this.debugText();
        return this._type + (txt ? ('[' + txt + ']') : '');
    };

    // 'virtual' methods ------------------------------------------------------

    /**
     * Derived classes may overwrite this method to generate a new text
     * representation of this token, after a sheet has been deleted in the
     * spreadsheet document.
     *
     * @returns {String|Null}
     *  The new text representation of this token, if available.
     */
    BaseToken.prototype.resolveDeletedSheet = function (/*formulaGrammar, sheet*/) {
        return null;
    };

    /**
     * Derived classes may overwrite this method to generate a new text
     * representation of this token, after a sheet has been renamed in the
     * spreadsheet document.
     *
     * @returns {String|Null}
     *  The new text representation of this token, if available.
     */
    BaseToken.prototype.resolveRenamedSheet = function (/*formulaGrammar, sheet, sheetName, tableNamesMap*/) {
        return null;
    };

    /**
     * Derived classes may overwrite this method to generate a new text
     * representation of this token, after the label of a defined name has been
     * changed in the spreadsheet document.
     *
     * @returns {String|Null}
     *  The new text representation of this token, if available.
     */
    BaseToken.prototype.resolveRelabeledName = function (/*formulaGrammar, sheet, oldLabel, newLabel*/) {
        return null;
    };

    /**
     * Derived classes may overwrite this method to generate a new text
     * representation of this token, before cell ranges will be moved in a
     * specific sheet of the spreadsheet document.
     *
     * @returns {String|Null}
     *  The new text representation of this token, if available.
     */
    BaseToken.prototype.resolveMovedCells = function (/*formulaGrammar, sheet, moveDescs, options*/) {
        return null;
    };

    /**
     * Derived classes may overwrite this method to generate a new text
     * representation of this token, after cell ranges have been cut and pasted
     * in a specific sheet of the spreadsheet document.
     *
     * @returns {String|Null}
     *  The new text representation of this token, if available.
     */
    BaseToken.prototype.resolveCutPasteCells = function (/*formulaGrammar, sourceRange, targetRange, options*/) {
        return null;
    };

    // class FixedToken =======================================================

    /**
     * A formula token of an arbitrary type with a fixed text representation
     * independent from a formula grammar.
     *
     * @constructor
     *
     * @extends BaseToken
     *
     * @param {String} type
     *  The token type identifier.
     *
     * @param {String} text
     *  The fixed text representation of the token.
     */
    var FixedToken = BaseToken.extend({ constructor: function (type, text) {

        // base constructor
        BaseToken.call(this, type);

        // private properties
        this._text = text;

    } }); // class FixedToken

    // public methods ---------------------------------------------------------

    /**
     * Returns the current text of this token.
     *
     * @returns {String}
     *  The current text of this token.
     */
    FixedToken.prototype.getValue = function () {
        return this._text;
    };

    /**
     * Replaces the text of this token.
     *
     * @param {String} newText
     *  The new text to be inserted into this token.
     *
     * @returns {Boolean}
     *  Whether the text has actually changed.
     */
    FixedToken.prototype.setValue = function (newText) {
        if (this._text === newText) { return false; }
        this._text = newText;
        return true;
    };

    /**
     * Appends more text to this token.
     *
     * @param {String} newText
     *  The new text to be appended to this token.
     *
     * @returns {Boolean}
     *  Whether the token has actually changed (the passed text was not empty).
     */
    FixedToken.prototype.appendValue = function (newText) {
        if (newText.length === 0) { return false; }
        this._text += newText;
        return true;
    };

    /**
     * Generates the text representation of this token.
     */
    FixedToken.prototype.getText = function () {
        return this._text;
    };

    /**
     * Generates a text description of this token for debugging (!) purposes.
     */
    FixedToken.prototype.debugText = function () {
        return this._text;
    };

    // class OperatorToken ====================================================

    /**
     * A formula token that represents a unary or binary operator in a formula.
     *
     * @constructor
     *
     * @extends BaseToken
     *
     * @param {String} value
     *  The grammar-independent operator identifier.
     */
    var OperatorToken = BaseToken.extend({ constructor: function (value) {

        // base constructor
        BaseToken.call(this, 'op');

        // private properties
        this._value = value;

    } }); // class OperatorToken

    // public methods ---------------------------------------------------------

    /**
     * Returns the grammar-independent identifier of the operator.
     *
     * @returns {String}
     *  The grammar-independent identifier of the operator.
     */
    OperatorToken.prototype.getValue = function () {
        return this._value;
    };

    /**
     * Generates the text representation of this token.
     *
     * @param {FormulaGrammar} formulaGrammar
     *  The formula grammar containing specific settings for formula tokens.
     */
    OperatorToken.prototype.getText = function (formulaGrammar) {
        return formulaGrammar.getOperatorName(this._value) || '?';
    };

    /**
     * Generates a text description of this token for debugging (!) purposes.
     */
    OperatorToken.prototype.debugText = function () {
        return this._value;
    };

    // class SeparatorToken ===================================================

    /**
     * A formula token that represents the range list operator, or a separator
     * in a function parameter list.
     *
     * @constructor
     *
     * @extends BaseToken
     */
    var SeparatorToken = BaseToken.extend({ constructor: function () {

        // base constructor
        BaseToken.call(this, 'sep');

    } }); // class SeparatorToken

    // public methods ---------------------------------------------------------

    /**
     * Generates the text representation of this token.
     *
     * @param {FormulaGrammar} formulaGrammar
     *  The formula grammar containing specific settings for formula tokens.
     */
    SeparatorToken.prototype.getText = function (formulaGrammar) { return formulaGrammar.SEP; };

    /**
     * Generates a text description of this token for debugging (!) purposes.
     */
    SeparatorToken.prototype.debugText = _.constant('');

    // class ParenthesisToken =================================================

    /**
     * A formula token that represents an opening or a closing parenthesis.
     *
     * @constructor
     *
     * @extends FixedToken
     *
     * @param {Boolean} open
     *  Whether this token represents an opening parenthesis (true), or a
     *  closing parenthesis (false).
     */
    var ParenthesisToken = FixedToken.extend({ constructor: function (open) {

        // base constructor
        FixedToken.call(this, open ? 'open' : 'close', open ? '(' : ')');

    } }); // class ParenthesisToken

    // public methods ---------------------------------------------------------

    /**
     * Generates a text description of this token for debugging (!) purposes.
     */
    ParenthesisToken.prototype.debugText = _.constant('');

    // class LiteralToken =====================================================

    /**
     * A formula token that represents a literal value (a number, a string, a
     * boolean value, or an error code) in a formula.
     *
     * @constructor
     *
     * @extends BaseToken
     *
     * @param {Number|String|Boolean|ErrorCode|Null} value
     *  The value for this literal token. The special value null is reserved
     *  for empty parameter in a function call, e.g. in the formula =SUM(1,,2).
     */
    var LiteralToken = BaseToken.extend({ constructor: function (value) {

        // base constructor
        BaseToken.call(this, 'lit');

        // private properties
        this._value = value;

    } }); // class LiteralToken

    // public methods ---------------------------------------------------------

    /**
     * Returns the current value of this literal token.
     *
     * @returns {Number|String|Boolean|ErrorCode|Null}
     *  The current value of this literal token.
     */
    LiteralToken.prototype.getValue = function () {
        return this._value;
    };

    /**
     * Changes the current value of this token.
     *
     * @param {Number|String|Boolean|ErrorCode|Null} newValue
     *  The new value for this token.
     *
     * @returns {Boolean}
     *  Whether the value of the token has actually been changed.
     */
    LiteralToken.prototype.setValue = function (newValue) {
        if (this._value === newValue) { return false; }
        this._value = newValue;
        return true;
    };

    /**
     * Generates the text representation of this token.
     *
     * @param {FormulaGrammar} formulaGrammar
     *  The formula grammar containing specific settings for formula tokens.
     */
    LiteralToken.prototype.getText = function (formulaGrammar) {
        return formulaGrammar.formatScalar(this._value);
    };

    /**
     * Generates the text representation for debug logging.
     */
    LiteralToken.prototype.debugText = function () {
        return Scalar.toString(this._value);
    };

    // class MatrixToken ======================================================

    /**
     * A formula token that represents a constant matrix literal in a formula.
     *
     * @constructor
     *
     * @extends BaseToken
     *
     * @param {Matrix} matrix
     *  The matrix literal represented by the token.
     */
    var MatrixToken = BaseToken.extend({ constructor: function (matrix) {

        // base constructor
        BaseToken.call(this, 'mat');

        // private properties
        this._matrix = matrix;

    } }); // class MatrixToken

    // public methods ---------------------------------------------------------

    /**
     * Returns the matrix literal contained in this token.
     *
     * @returns {Matrix}
     *  The matrix literal of this token.
     */
    MatrixToken.prototype.getMatrix = function () {
        return this._matrix;
    };

    /**
     * Generates the text representation of this token.
     *
     * @param {FormulaGrammar} formulaGrammar
     *  The formula grammar containing specific settings for formula tokens.
     */
    MatrixToken.prototype.getText = function (formulaGrammar) {
        var text = formulaGrammar.MAT_OPEN;
        this._matrix.forEach(function (value, row, col) {
            if (col > 0) {
                text += formulaGrammar.MAT_COL;
            } else if (row > 0) {
                text += formulaGrammar.MAT_ROW;
            }
            text += formulaGrammar.formatScalar(value);
        });
        return text + formulaGrammar.MAT_CLOSE;
    };

    /**
     * Generates the text representation for debug logging.
     */
    MatrixToken.prototype.debugText = function () {
        return this._matrix.toString();
    };

    // class FunctionToken ====================================================

    /**
     * A formula token that represents a built-in function in a formula.
     *
     * @constructor
     *
     * @extends BaseToken
     *
     * @param {String} funcKey
     *  The unique resource key of the built-in sheet function.
     */
    var FunctionToken = BaseToken.extend({ constructor: function (funcKey) {

        // base constructor
        BaseToken.call(this, 'func');

        // private properties
        this._value = funcKey;

    } }); // class FunctionToken

    // public methods ---------------------------------------------------------

    /**
     * Returns the unique resource key of the function, as passed to the
     * constructor.
     *
     * @returns {String}
     *  The unique resource key of the function.
     */
    FunctionToken.prototype.getValue = function () {
        return this._value;
    };

    /**
     * Generates the text representation of this token.
     *
     * @param {FormulaGrammar} formulaGrammar
     *  The formula grammar containing specific settings for formula tokens.
     */
    FunctionToken.prototype.getText = function (formulaGrammar) {
        return formulaGrammar.getFunctionName(this._value) || this._value.toUpperCase();
    };

    /**
     * Returns a text description of this token for debug logging.
     */
    FunctionToken.prototype.debugText = function () {
        return this._value;
    };

    // class SheetRefToken ====================================================

    /**
     * Base class for formula tokens containing one or two sheet references.
     *
     * @constructor
     *
     * @extends BaseToken
     *
     * @param {Boolean} type
     *  The type of the token.
     *
     * @param {SpreadsheetModel} docModel
     *  The spreadsheet document model. Needed to resolve sheet names.
     *
     * @param {SheetRef|Null} sheet1Ref
     *  The first sheet reference structure (e.g. 'Sheet1' in the expression
     *  Sheet1:Sheet2!A1). If null or undefined, the token represents a local
     *  reference into the containing sheet (e.g. A1:B2).
     *
     * @param {SheetRef|Null} sheet2Ref
     *  The second sheet reference structure (e.g. 'Sheet2' in the expression
     *  Sheet1:Sheet2!A1). If null or undefined, the token represents a single
     *  sheet reference (e.g. Sheet1!A1).
     *
     * @param {Number|Null} extDocRef
     *  The index of the external document, e.g. [1]!extname or [1]Sheet1!A1.
     *  The value zero represents a global reference into the own document,
     *  e.g. in [0]!globalname. If null or undefined, the token represents a
     *  reference in the own document (e.g. globalname).
     */
    var SheetRefToken = BaseToken.extend({ constructor: function (type, docModel, sheet1Ref, sheet2Ref, extDocRef) {

        // base constructor
        BaseToken.call(this, type);

        // private properties
        this._docModel = docModel;
        this._sheet1Ref = sheet1Ref;
        this._sheet2Ref = sheet2Ref;
        this._extDocRef = extDocRef;

        // swap sheet indexes (and absolute flags!), if sheet references are not in order
        if (sheet1Ref && sheet1Ref.valid() && sheet2Ref && sheet2Ref.valid() && (sheet2Ref.sheet < sheet1Ref.sheet)) {
            var abs = sheet2Ref.abs;
            sheet2Ref.abs = sheet1Ref.abs;
            sheet1Ref.abs = abs;
            var sheet = sheet2Ref.sheet;
            sheet2Ref.sheet = sheet1Ref.sheet;
            sheet1Ref.sheet = sheet;
        }

    } }); // class SheetRefToken

    // protected methods ------------------------------------------------------

    SheetRefToken.prototype._isOwnSheetRef = function (sheetRef) {
        return this.isOwnDoc() && _.isObject(sheetRef);
    };

    /**
     * Returns the sheet references of this token, and allows to oveeride the
     * own sheet references with arbitrary sheet reference structures, as
     * offered by various public methods of this class, and other subclasses.
     *
     * @param {Object} [options]
     *  Optional parameters:
     *  @param {SheetRef|Null} [options.sheet1Ref]
     *      A sheet reference for the first sheet to be returned instead of the
     *      own first sheet reference.
     *  @param {SheetRef|Null} [options.sheet2Ref]
     *      A sheet reference for the second sheet to be returned instead of
     *      the own second sheet reference.
     *
     * @returns {Object}
     *  A result object with the following properties:
     *  - {SheetRef|Null} s1
     *      The first sheet reference. The value null indicates a token without
     *      sheet reference (e.g. the sheet-local cell address 'A1').
     *  - {SheetRef|Null} s2
     *      The second sheet reference of a sheet interval. The value null
     *      indicates a token without second sheet reference.
     */
    SheetRefToken.prototype._resolveSheetRefs = function (options) {
        return {
            s1: Utils.getOption(options, 'sheet1Ref', this._sheet1Ref),
            s2: Utils.getOption(options, 'sheet2Ref', this._sheet2Ref)
        };
    };

    /**
     * Returns the sheet interval referred by this token.
     *
     * @param {Object} [options]
     *  Optional parameters:
     *  @param {Number} [options.refSheet]
     *      The index of the reference sheet used to resolve references without
     *      sheets. If omitted, and this token does not contain its own sheet
     *      reference, this method will return null.
     *
     * @returns {Interval|Null}
     *  The interval of sheets referred by this token. If the token does not
     *  contain a valid sheet interval (according to the passed reference
     *  sheet), null will be returned.
     */
    SheetRefToken.prototype._resolveSheetInterval = function (options) {
        if (!this.isOwnDoc()) { return null; }
        var sheet1 = this._sheet1Ref ? this._sheet1Ref.sheet : Utils.getIntegerOption(options, 'refSheet', null);
        var sheet2 = this._sheet2Ref ? this._sheet2Ref.sheet : sheet1;
        return ((typeof sheet1 === 'number') && (sheet1 >= 0) && (typeof sheet2 === 'number') && (sheet2 >= 0)) ? Interval.create(sheet1, sheet2) : null;
    };

    // public methods ---------------------------------------------------------

    /**
     * Returns whether this token refers into its own document.
     *
     * @returns {Boolean}
     *  Returns whether this token refers into its own document (true), or to
     *  another external spreadsheet document (false).
     */
    SheetRefToken.prototype.isOwnDoc = function () {
        return !this._extDocRef; // either null or zero
    };

    /**
     * Returns whether this token contains a sheet reference, regardless if it
     * is valid.
     *
     * @returns {Boolean}
     *  Whether this token contains a sheet reference.
     */
    SheetRefToken.prototype.hasSheetRef = function () {
        return !!this._sheet1Ref;
    };

    /**
     * Returns whether a sheet reference of this token is invalid (it exists,
     * AND it points to a non-existing sheet).
     *
     * @returns {Boolean}
     *  Whether a sheet reference of this token is invalid.
     */
    SheetRefToken.prototype.hasSheetError = function () {
        // sheet error: sheet reference must exist and must be invalid
        return (!!this._sheet1Ref && !this._sheet1Ref.valid()) || (!!this._sheet2Ref && !this._sheet2Ref.valid());
    };

    /**
     * Tries to replace unresolved sheet names with existing sheet indexes in
     * the spreadsheet document. Intended to be used after document import to
     * refresh all token arrays that refer to sheets that did not exist during
     * their creation.
     *
     * @returns {Boolean}
     *  Whether the token has been changed, i.e. whether it has been refreshed
     *  from at least one unresolved sheet name to an existing sheet index.
     */
    SheetRefToken.prototype.refreshSheets = function () {
        var changed1 = this._isOwnSheetRef(this._sheet1Ref) && this._sheet1Ref.refresh(this._docModel);
        var changed2 = this._isOwnSheetRef(this._sheet2Ref) && this._sheet2Ref.refresh(this._docModel);
        return changed1 || changed2;
    };

    /**
     * Refreshes the sheet references after a sheet has been inserted, deleted,
     * or moved in the document.
     *
     * @returns {Boolean}
     *  Whether the token has been changed.
     */
    SheetRefToken.prototype.transformSheet = function (fromSheet, toSheet) {
        var changed1 = this._isOwnSheetRef(this._sheet1Ref) && this._sheet1Ref.transform(fromSheet, toSheet);
        var changed2 = this._isOwnSheetRef(this._sheet2Ref) && this._sheet2Ref.transform(fromSheet, toSheet);
        return changed1 || changed2;
    };

    /**
     * Returns a text description of the sheet names for debug logging.
     */
    SheetRefToken.prototype.debugSheetPrefix = function () {

        // the external document refernce
        var sheetName = (this._extDocRef === null) ? '' : ('[' + this._extDocRef + ']');

        // convert first sheet name
        if (this._sheet1Ref) {
            sheetName += this._sheet1Ref.toString();
        }

        // convert second sheet name
        if (this._sheet1Ref && this._sheet2Ref && !this._sheet1Ref.equals(this._sheet2Ref)) {
            sheetName += ':' + this._sheet2Ref.toString();
        }

        return sheetName ? (sheetName + '!') : '';
    };

    // 'virtual' methods ------------------------------------------------------

    /**
     * Resolves the new text representation of this token, before a sheet will
     * be deleted in the spreadsheet document.
     *
     * @param {FormulaGrammar} formulaGrammar
     *  The formula grammar containing specific settings for formula tokens.
     *
     * @param {Number} sheet
     *  The zero-based index of the deleted sheet.
     *
     * @param {Number} [refSheet]
     *  The index of the reference sheet used to resolve relative references.
     *
     * @returns {String|Null}
     *  The text representation of this sheet reference token, if it has been
     *  changed; otherwise null.
     */
    SheetRefToken.prototype.resolveDeletedSheet = function (formulaGrammar, sheet, refSheet) {

        // nothing to do for external references
        if (!this.isOwnDoc()) { return null; }

        // resolve the own sheet range (nothing to do on reference errors)
        var sheetInterval = this._sheet1Ref ? this._resolveSheetInterval({ refSheet: refSheet }) : null;
        if (!sheetInterval) { return null; }

        // if this token refers to the deleted sheet, it becomes a #REF! error
        if (sheetInterval.single()) {
            return (sheet === sheetInterval.first) ? this.getText(formulaGrammar, { sheet1Ref: new SheetRef(-1, true) }) : null;
        }

        // if this token refers to multiple sheets, and the deleted sheet is located at the start, shorten the range
        if (sheet === sheetInterval.first) {
            return this.getText(formulaGrammar, {
                sheet1Ref: new SheetRef(sheetInterval.first + 1, this._sheet1Ref.abs),
                sheet2Ref: new SheetRef(sheetInterval.last, this._sheet2Ref.abs)
            });
        }

        // if this token refers to multiple sheets, and the deleted sheet is located at the end, shorten the range
        if (sheet === sheetInterval.last) {
            return this.getText(formulaGrammar, {
                sheet1Ref: new SheetRef(sheetInterval.first, this._sheet1Ref.abs),
                sheet2Ref: new SheetRef(sheetInterval.last - 1, this._sheet2Ref.abs)
            });
        }

        return null;
    };

    /**
     * Resolves the new text representation of this token, before a sheet will
     * be renamed in the spreadsheet document, or after an existing sheet has
     * been copied.
     *
     * @param {FormulaGrammar} formulaGrammar
     *  The formula grammar containing specific settings for formula tokens.
     *
     * @param {Number} sheet
     *  The zero-based index of the renamed sheet.
     *
     * @param {String} sheetName
     *  The new name of the sheet.
     *
     * @returns {String|Null}
     *  The text representation of this sheet reference token, if it has been
     *  changed; otherwise null.
     */
    SheetRefToken.prototype.resolveRenamedSheet = function (formulaGrammar, sheet, sheetName) {

        // nothing to do for external references
        if (!this.isOwnDoc()) { return null; }

        // create the string representation only if this token refers to the renamed sheet
        var sheet1Ref = (this._sheet1Ref && (this._sheet1Ref.sheet === sheet)) ? new SheetRef(sheetName, this._sheet1Ref.abs) : this._sheet1Ref;
        var sheet2Ref = (this._sheet2Ref && (this._sheet2Ref.sheet === sheet)) ? new SheetRef(sheetName, this._sheet2Ref.abs) : this._sheet2Ref;
        if ((sheet1Ref !== this._sheet1Ref) || (sheet2Ref !== this._sheet2Ref)) {
            return this.getText(formulaGrammar, { sheet1Ref: sheet1Ref, sheet2Ref: sheet2Ref });
        }

        return null;
    };

    // class ReferenceToken ===================================================

    /**
     * A formula token that represents a cell range reference in a formula.
     *
     * @constructor
     *
     * @extends SheetRefToken
     *
     * @param {SpreadsheetModel} docModel
     *  The spreadsheet document model containing this token instance.
     *
     * @param {CellRef|Null} cell1Ref
     *  The first cell reference structure (e.g. the cell address 'A1' in the
     *  reference A1:B2). If null or omitted, the token will represent a
     *  reference error (e.g. Sheet1!#REF!, after deleting the referenced
     *  column or row).
     *
     * @param {CellRef|Null} cell2Ref
     *  The second cell reference structure (e.g. the cell address 'B2' in the
     *  reference A1:B2). If null or omitted, the token will represent a single
     *  cell (e.g. Sheet1!A1).
     *
     * @param {SheetRef|Null} sheet1Ref
     *  The first sheet reference structure (e.g. 'Sheet1' in the reference
     *  Sheet1:Sheet2!A1). If null or undefined, the token will represent a
     *  local cell range reference into the passed sheet (e.g. A1:B2).
     *
     * @param {SheetRef|Null} sheet2Ref
     *  The second sheet reference structure (e.g. 'Sheet2' in the reference
     *  Sheet1:Sheet2!A1). If null or undefined, the token will represent a
     *  single sheet reference (e.g. Sheet1!A1).
     */
    var ReferenceToken = SheetRefToken.extend({ constructor: function (docModel, cell1Ref, cell2Ref, sheet1Ref, sheet2Ref) {

        // base constructor
        SheetRefToken.call(this, 'ref', docModel, sheet1Ref, sheet2Ref, null);

        // private properties
        this._cell1Ref = cell1Ref;
        this._cell2Ref = cell2Ref;

        // swap column indexes (and absolute flags!), if cell references are not in order
        if (cell1Ref && cell2Ref && (cell2Ref.col < cell1Ref.col)) {
            var absCol = cell2Ref.absCol;
            cell2Ref.absCol = cell1Ref.absCol;
            cell1Ref.absCol = absCol;
            var col = cell2Ref.col;
            cell2Ref.col = cell1Ref.col;
            cell1Ref.col = col;
        }

        // swap row indexes (and absolute flags!), if old cell references are not in order
        if (this._cell1Ref && this._cell2Ref && (this._cell2Ref.row < this._cell1Ref.row)) {
            var absRow = cell2Ref.absRow;
            cell2Ref.absRow = cell1Ref.absRow;
            cell1Ref.absRow = absRow;
            var row = cell2Ref.row;
            cell2Ref.row = cell1Ref.row;
            cell1Ref.row = row;
        }

    } }); // class ReferenceToken

    // private methods --------------------------------------------------------

    /**
     * Returns whether this reference token contains relative column or row
     * references only.
     *
     * @param {Boolean} columns
     *  Whether to check the column references (true), or the row references
     *  (false).
     *
     * @returns {Boolean}
     *  Whether this reference token contains relative column or row references
     *  only.
     */
    ReferenceToken.prototype._isSingleRelative = function (columns) {
        // absolute index in first cell reference: this reference cannot be relative
        if (this._cell1Ref.isAbs(columns)) { return false; }
        // no second cell reference: this reference is single and relative
        if (!this._cell2Ref) { return true; }
        // second cell reference must be relative, and the indexes must be equal
        return !this._cell2Ref.isAbs(columns) && (this._cell1Ref.getIndex(columns) === this._cell2Ref.getIndex(columns));
    };

    /**
     * Returns copies of the own cell reference objects that have been
     * relocated according to the passed settings.
     *
     * @returns {Object}
     *  The resulting cell references, in the properties 'r1' and 'r2'.
     */
    ReferenceToken.prototype._resolveCellRefs = function (options) {

        // reference and target address for relocation
        var refAddress = Utils.getOption(options, 'refAddress', Address.A1);
        var targetAddress = Utils.getOption(options, 'targetAddress', Address.A1);

        // performance: return original cell references, if nothing to relocate
        if (refAddress.equals(targetAddress)) { return { r1: this._cell1Ref, r2: this._cell2Ref }; }

        // the result object returned by this method: start with clones of the cell references
        var cellRefs = {
            r1: this._cell1Ref ? this._cell1Ref.clone() : null,
            r2: this._cell2Ref ? this._cell2Ref.clone() : null
        };
        // whether to wrap the references at sheet borders
        var wrapReferences = Utils.getBooleanOption(options, 'wrapReferences', false);

        // if relocating first reference is not successful, remove it from the result
        if (cellRefs.r1 && !cellRefs.r1.relocate(this._docModel, refAddress, targetAddress, wrapReferences)) {
            cellRefs.r1 = null;
        }

        // if relocating second reference is not successful, remove both cell references from the result
        if (cellRefs.r1 && cellRefs.r2 && !cellRefs.r2.relocate(this._docModel, refAddress, targetAddress, wrapReferences)) {
            cellRefs.r1 = cellRefs.r2 = null;
        }

        return cellRefs;
    };

    // public methods ---------------------------------------------------------

    /**
     * Returns the address of the cell range represented by this token,
     * including sheet indexes, or null if the range address is invalid.
     *
     * @param {Object} [options]
     *  Optional parameters:
     *  @param {Number} [options.refSheet]
     *      The index of the reference sheet used to resolve missing sheet
     *      references. If omitted, and this token does not contain its own
     *      sheet reference, this method will return null.
     *  @param {Address} [options.refAddress]
     *      The address of the original reference cell this token is related
     *      to. If omitted, cell A1 will be used instead. If this address is
     *      different to the address resulting from the option 'targetAddress',
     *      the returned range will be relocated according to these two
     *      addresses.
     *  @param {Address} [options.targetAddress]
     *      The address of the target cell for relocation. If omitted, cell A1
     *      will be used instead. If this address is different to the address
     *      resulting from the option 'refAddress', the returned range will be
     *      relocated according to these two addresses.
     *  @param {Boolean} [options.wrapReferences=false]
     *      If set to true, and if the relocated range is located outside the
     *      sheet, the range will be wrapped at the sheet borders.
     *
     * @returns {Range3D|Null}
     *  The address of the cell range referenced by this reference token; or
     *  null, if the token does not refer to a valid range.
     */
    ReferenceToken.prototype.getRange3D = function (options) {

        // resolve sheet references, return null on error
        var sheetInterval = this._resolveSheetInterval(options);
        if (!sheetInterval) { return null; }

        // return immediately, if cell1Ref cannot be relocated
        var cellRefs = this._resolveCellRefs(options);
        if (!cellRefs.r1) { return null; }

        // create a 3D range with ordered column/row indexes
        var address1 = cellRefs.r1.toAddress();
        var address2 = cellRefs.r2 ? cellRefs.r2.toAddress() : null;
        return address2 ?
            Range3D.createFromAddresses(address1, address2, sheetInterval.first, sheetInterval.last) :
            Range3D.createFromAddress(address1, sheetInterval.first, sheetInterval.last);
    };

    /**
     * Returns the addresses of the cell ranges that will be built from the
     * cell range represented by this token, including sheet indexes, but
     * expanded relative to a reference address according to the specified
     * target ranges.
     *
     * @param {Address} refAddress
     *  The address of the original reference cell this token is related to.
     *  The resulting ranges returned by this method will be expanded according
     *  to the distance of the target ranges, and this reference address.
     *
     * @param {RangeArray|Range} targetRanges
     *  The target ranges to expand the relative components of this reference
     *  to.
     *
     * @param {Object} [options]
     *  Optional parameters:
     *  @param {Number} [options.refSheet]
     *      The index of the reference sheet used to resolve missing sheet
     *      references. If omitted, and this token does not contain its own
     *      sheet reference, this method will return null.
     *  @param {Boolean} [options.wrapReferences=false]
     *      If set to true, the resulting range will be wrapped at the sheet
     *      borders.
     *
     * @returns {Range3DArray}
     *  The addresses of the resulting expanded cell ranges (may be multiple
     *  ranges per target range in some situations in wrapping mode).
     */
    ReferenceToken.prototype.getExpandedRanges = function (refAddress, targetRanges, options) {

        // nothing to do for reference errors
        if (!this._cell1Ref) { return null; }

        // resolve sheet references, return null on error
        var sheetInterval = this._resolveSheetInterval(options);
        if (!sheetInterval) { return null; }

        // whether to wrap the references at sheet borders
        var wrapReferences = Utils.getBooleanOption(options, 'wrapReferences', false);

        // returns the adjusted index of the passed cell reference, according to reference address, and target range
        function getMovedIndex(cellRef, targetRange, columns, leading) {
            var index = cellRef.getIndex(columns);
            if (cellRef.isAbs(columns)) { return index; }
            var targetIndex = (leading ? targetRange.start : targetRange.end).get(columns);
            return index + targetIndex - refAddress.get(columns);
        }

        // returns the expanded column or row intervals of this token, according to the reference address and target range
        function getIntervals(targetRange, columns) {

            // the moved index of the resulting range for the start address of the target range
            var index1 = getMovedIndex(this._cell1Ref, targetRange, columns, true);
            if (this._cell2Ref) { index1 = Math.min(index1, getMovedIndex(this._cell2Ref, targetRange, columns, true)); }

            // the moved index of the resulting range for the end address of the target range
            var index2 = getMovedIndex(this._cell1Ref, targetRange, columns, false);
            if (this._cell2Ref) { index2 = Math.max(index2, getMovedIndex(this._cell2Ref, targetRange, columns, false)); }

            // the largest valid column/row index in a sheet
            var maxIndex = this._docModel.getMaxIndex(columns);

            // wrapping mode: detect how to expand the resulting intervals
            if (wrapReferences) {

                // Normalize indexes outside the sheet area: If the start index is negative, move it so that the interval
                // will be located inside the sheet area, or will wrap at its end. If the start index is too large, move
                // it back so that the interval becomes part of the sheet area. Afterwards, the interval will either be
                // completely inside the sheet area, or will exceed it at its end.
                //
                if (index1 < 0) {
                    index1 += maxIndex + 1;
                    index2 += maxIndex + 1;
                } else if (index1 > maxIndex) {
                    index1 -= maxIndex + 1;
                    index2 -= maxIndex + 1;
                }

                // Create the resulting column/row intervals.
                //
                // Case 1: If the end index is inside the sheet, the interval was either not wrapped, or was wrapped
                // completely without covering the sheet boundaries. In this case, a single interval will be created.
                //
                // Case 2: The resulting interval covers the sheet boundaries, and this reference token represents a
                // single relative column or row (e.g. the rows in the reference C2:D2). This means that the target
                // range has caused to expand the single column/row to an interval which needs to be split into two
                // parts: One at the leading border of the sheet, and one at the trailing border. Example: The
                // reference C2:D2 with reference address A3, and target range A1:A3, will be expanded to the ranges
                // C2:D2 (for target address A3), C1:D1 (for target address A2), and C0:D0 (for target address A1).
                // The latter range will be wrapped to C1048576:D1048576. Therefore, the resulting row intervals
                // returned by this function will be 1:2 and 1048576:1048576.
                //
                // Case 3: The resulting interval covers the sheet boundaries, and this reference token is not a single
                // relative column or row. This means that wrapping causes to swap the start/end index of the resulting
                // range with the effect that the entire interval of the sheet area will be covered. Example: The
                // reference C2:D4 with reference address A3, and target range A1:A3, will be expanded to the ranges
                // C2:D4 (for target address A3), C1:D3 (for target address A2), and C0:D2 (for target address A1).
                // The latter range will be wrapped and swapped to C2:D1048576. Therefore, all rows in the sheet will
                // be covered by the resulting ranges, and the row interval returned by this function will be 1:1048576.
                //
                var intervals = new IntervalArray();
                if (index2 <= maxIndex) { // case 1
                    intervals.push(new Interval(index1, index2));
                } else if (this._isSingleRelative(columns)) { // case 2
                    intervals.push(new Interval(0, index2 - maxIndex - 1));
                    intervals.push(new Interval(index1, maxIndex));
                } else { // case 3
                    intervals.push(new Interval(0, maxIndex));
                }
                return intervals;
            }

            // non-wrapping mode: try to crop the indexes to the valid sheet area, return null otherwise
            index1 = Math.max(index1, 0);
            index2 = Math.min(index2, maxIndex);
            return (index1 <= index2) ? new IntervalArray(new Interval(index1, index2)) : null;
        }

        return Range3DArray.map(targetRanges, function (targetRange) {
            var colIntervals = getIntervals.call(this, targetRange, true);
            var rowIntervals = colIntervals ? getIntervals.call(this, targetRange, false) : null;
            return rowIntervals ? Range3DArray.createFromIntervals(colIntervals, rowIntervals, sheetInterval.first, sheetInterval.last) : null;
        }, this);
    };

    /**
     * Returns whether this reference token contains at least one relative
     * column reference, e.g. the C in C$4:$D$5.
     *
     * @returns {Boolean}
     *  Whether this reference token contains at least one relative column
     *  reference.
     */
    ReferenceToken.prototype.hasRelCol = function () {

        // reference errors (missing first cell reference) cannot be relative
        if (!this._cell1Ref) { return false; }

        // immediately return if the first cell reference contains a relative column
        if (!this._cell1Ref.absCol) { return true; }

        // otherwise check the column reference of the second cell reference if existing
        return !!this._cell2Ref && !this._cell2Ref.absCol;
    };

    /**
     * Returns whether this reference token contains at least one relative row
     * reference, e.g. the 4 in $C4:$D$5.
     *
     * @returns {Boolean}
     *  Whether this reference token contains at least one relative row
     *  reference.
     */
    ReferenceToken.prototype.hasRelRow = function () {

        // reference errors (missing first cell reference) cannot be relative
        if (!this._cell1Ref) { return false; }

        // immediately return if the first cell reference contains a relative row
        if (!this._cell1Ref.absRow) { return true; }

        // otherwise check the row reference of the second cell reference if existing
        return !!this._cell2Ref && !this._cell2Ref.absRow;
    };

    /**
     * Changes the current cell range of this token, keeps all absolute flags
     * and sheet references intact.
     *
     * @param {Range} range
     *  The new position of this reference token.
     *
     * @returns {Boolean}
     *  Whether the token has been changed.
     */
    ReferenceToken.prototype.setRange = function (range) {

        // whether the token changes (either start or end address)
        var changed = false;

        // create first cell reference, if missing
        if (!this._cell1Ref) {
            this._cell1Ref = new CellRef(0, 0, true, true);
            changed = true;
        }

        // create second cell reference, if missing (and if the passed range is not a single cell)
        if (!this._cell2Ref && !range.single()) {
            this._cell2Ref = this._cell1Ref.clone(); // copy the absolute flags of first reference
            changed = true;
        }

        // set the new address indexes for the first cell reference, update the changed flag
        if (this._cell1Ref.col !== range.start[0]) { this._cell1Ref.col = range.start[0]; changed = true; }
        if (this._cell1Ref.row !== range.start[1]) { this._cell1Ref.row = range.start[1]; changed = true; }

        // set the new address indexes for the second cell reference, update the changed flag
        if (this._cell2Ref) {
            if (this._cell2Ref.col !== range.end[0]) { this._cell2Ref.col = range.end[0]; changed = true; }
            if (this._cell2Ref.row !== range.end[1]) { this._cell2Ref.row = range.end[1]; changed = true; }
            // reset cell2Ref if address and absolute flags are equal to cell1Ref
            if (this._cell1Ref.equals(this._cell2Ref)) { this._cell2Ref = null; changed = true; }
        }

        return changed;
    };

    /**
     * Relocates this reference token to a new cell position.
     *
     * @param {Address} refAddress
     *  The source address to relocate the reference token from.
     *
     * @param {Address} targetAddress
     *  The target address to relocate the reference token to.
     *
     * @param {Boolean} wrapReferences
     *  Whether to wrap the addresses at the sheet boundaries.
     *
     * @returns {Boolean}
     *  Whether the token has been changed.
     */
    ReferenceToken.prototype.relocateRange = function (refAddress, targetAddress, wrapReferences) {

        // do not touch tokens representing a reference error (missing first cell reference)
        if (!this._cell1Ref || refAddress.equals(targetAddress)) { return false; }

        // relocate first cell reference, bail out immediately on error
        if (!this._cell1Ref.relocate(this._docModel, refAddress, targetAddress, wrapReferences)) {
            this._cell1Ref = this._cell2Ref = null;
            return true;
        }

        // relocate second cell reference if existing
        if (this._cell2Ref && !this._cell2Ref.relocate(this._docModel, refAddress, targetAddress, wrapReferences)) {
            this._cell1Ref = this._cell2Ref = null;
        }
        return true;
    };

    /**
     * Generates the text representation of this token.
     *
     * @param {FormulaGrammar} formulaGrammar
     *  The formula grammar containing specific settings for formula tokens.
     *
     * @param {Object} [options]
     *  Optional parameters for relocation of range addresses. See method
     *  ReferenceToken.getRange3D() for details. Additionally, the following
     *  options are supported:
     *  @param {SheetRef|Null} [options.sheet1Ref]
     *      A sheet reference for the first sheet to be used instead of the own
     *      first sheet reference.
     *  @param {SheetRef|Null} [options.sheet2Ref]
     *      A sheet reference for the second sheet to be used instead of the
     *      own second sheet reference.
     */
    ReferenceToken.prototype.getText = function (formulaGrammar, options) {
        var sheetRefs = this._resolveSheetRefs(options);
        var cellRefs = this._resolveCellRefs(options);
        var targetAddress = Utils.getOption(options, 'targetAddress', Address.A1);
        return formulaGrammar.formatReference(this._docModel, targetAddress, sheetRefs.s1, sheetRefs.s2, cellRefs.r1, cellRefs.r2);
    };

    /**
     * Returns a text description of this token for debug logging.
     */
    ReferenceToken.prototype.debugText = function () {
        var result = this.debugSheetPrefix();
        result += this._cell1Ref ? this._cell1Ref : ErrorCode.REF;
        result += (this._cell1Ref && this._cell2Ref) ? (':' + this._cell2Ref) : '';
        return result;
    };

    // 'virtual' methods ------------------------------------------------------

    /**
     * Resolves the new text representation of this token, before cell ranges
     * will be moved in a specific sheet of the spreadsheet document.
     *
     * @param {FormulaGrammar} formulaGrammar
     *  The formula grammar containing specific settings for formula tokens.
     *
     * @param {Number} sheet
     *  The zero-based index of the sheet containing the moved cell ranges.
     *
     * @param {Array<MoveDescriptor>} moveDescs
     *  An array of move descriptors created while generating the move document
     *  operations.
     *
     * @param {Object} [options]
     *  Optional parameters. Will be passed to the method getRange3D().
     *
     * @returns {String|Null}
     *  The string representation of the moved cell reference, if it has been
     *  changed; otherwise null.
     */
    ReferenceToken.prototype.resolveMovedCells = function (formulaGrammar, sheet, moveDescs, options) {

        // the original range, with sheet indexes
        var oldRange3d = this.getRange3D(options);

        // nothing to do, if this token represents a reference error, or points
        // to another sheet or a sheet range, or sheet reference is invalid
        if (!oldRange3d || !oldRange3d.isSheet(sheet)) { return null; }

        // special behavior if relative references will not be transformed
        var freezeRelRefs = Utils.getBooleanOption(options, 'freezeRelRefs', false);

        // convert the range address, nothing to do, if the range does not change (compare only the columns/rows, not the sheets)
        var oldRange = oldRange3d.toRange();
        var newRange = oldRange.clone();
        moveDescs.every(function (moveDesc) {

            // special behavior if relative references will not be transformed
            if (freezeRelRefs) {

                // do not transform the range at all, if both reference components are frozen
                var isFrozen1 = !this._cell1Ref.isAbs(moveDesc.columns);
                var isFrozen2 = !this._cell2Ref || !this._cell2Ref.isAbs(moveDesc.columns);
                if (isFrozen1 && isFrozen2) { return newRange; }

                // transform only the end address of the range, if the start address is frozen
                if (isFrozen1) {
                    var endAddress = moveDesc.transformAddress(newRange.end);
                    return (newRange = endAddress ? Range.createFromAddresses(newRange.start, endAddress) : null);
                }

                // transform only the start address of the range, if the end address is frozen
                if (isFrozen2 && this._cell2Ref) {
                    var startAddress = moveDesc.transformAddress(newRange.start);
                    return (newRange = startAddress ? Range.createFromAddresses(startAddress, newRange.end) : null);
                }
            }

            return (newRange = moveDesc.transformRange(newRange));
        }, this);

        // nothing changes, if the transformed range equals the original range
        if (newRange && newRange.equals(oldRange)) { return null; }

        // create copies of the cell references that will be modified to generate the text representation
        var cell1Ref = newRange && CellRef.create(newRange.start, this._cell1Ref.absCol, this._cell1Ref.absRow);
        var cell2Ref = newRange && this._cell2Ref && CellRef.create(newRange.end, this._cell2Ref.absCol, this._cell2Ref.absRow);
        var targetAddress = Utils.getOption(options, 'targetAddress', Address.A1);
        return formulaGrammar.formatReference(this._docModel, targetAddress, this._sheet1Ref, this._sheet2Ref, cell1Ref, cell2Ref);
    };

    /**
     * Resolves the new text representation of this token, after cell ranges
     * have been cut/paste in a specific sheet of the spreadsheet document.
     *
     * @param {FormulaGrammar} formulaGrammar
     *  The formula grammar containing specific settings for formula tokens.
     *
     * @param {Range} sourceRange
     *  The range from which the content was cutted off.
     *
     * @param {Range} targetRange
     *  The range to whom the content will be pasted.
     *
     * @param {Object} [options]
     *  Optional parameters. Will be passed to the method getRange3D() amongst
     *  other things.
     *
     * @returns {String|Null}
     *  Null, if the string representation of the pasted cell reference does
     *  not change; otherwise the new string representation of the pasted cell
     *  reference.
     */
    ReferenceToken.prototype.resolveCutPasteCells = function (formulaGrammar, sourceRange, targetRange, options) {
        // the original range, with sheet indexes
        var sheet       = options.sheet,

            bothExist   = this._cell1Ref && this._cell2Ref,

            oldRange3d  = this.getRange3D(options),
            sameCol     = (sourceRange.start[0] === targetRange.start[0]),
            sameRow     = (sourceRange.start[1] === targetRange.start[1]),
            columns     = sameCol ? true : sameRow ? false : null,
            forward     = sourceRange.start.get(!columns) < targetRange.start.get(!columns),
            oldRange    = oldRange3d.toRange(),
            overlaps    = oldRange.overlaps(targetRange),

            newRange    = null,
            cell1Ref    = null,
            cell2Ref    = null;

        // nothing to do, if this token represents a reference error, or points
        // to another sheet or a sheet range, or sheet reference is invalid
        if (!oldRange3d || !oldRange3d.isSheet(sheet)) { return null; }

        if (bothExist ?
            sourceRange.contains(new Range(this._cell1Ref.toAddress(), this._cell2Ref.toAddress())) :
            (this._cell1Ref && sourceRange.containsAddress(this._cell1Ref.toAddress()))
        ) {

            var backupCell1Ref = this._cell1Ref ? this._cell1Ref.clone() : null;
            var backupCell2Ref = this._cell2Ref ? this._cell2Ref.clone() : null;

            this.relocateRange(sourceRange.start, targetRange.start, false);
            cell1Ref = this._cell1Ref ? this._cell1Ref.clone() : null;
            cell2Ref = this._cell2Ref ? this._cell2Ref.clone() : null;

            this._cell1Ref = backupCell1Ref;
            this._cell2Ref = backupCell2Ref;

            return formulaGrammar.formatReference(this._docModel, targetRange.start, this._sheet1Ref, this._sheet2Ref, cell1Ref, cell2Ref);
        }

        // one end (top, right, bottom or left) completly covered by sourceRange
        if (bothExist && (this._cell1Ref.getIndex(columns) >= sourceRange.start.get(columns)) && (this._cell2Ref.getIndex(columns) <= sourceRange.end.get(columns))) {

            if (!sameCol && !sameRow) { return null; }

            var newRefRange = new RangeArray(oldRange).difference(sourceRange).first(); // oldRange without sourceRange
            var cutRefRange = sourceRange.intersect(oldRange);
            if (!cutRefRange) { return null; }

            // current indexes
            var col1 = this._cell1Ref.col;
            var row1 = this._cell1Ref.row;
            var col2 = this._cell2Ref.col;
            var row2 = this._cell2Ref.row;

            var startCutted = sourceRange.contains(columns ? oldRange.headerRow() : oldRange.leadingCol());
            var endCutted   = sourceRange.contains(columns ? oldRange.footerRow() : oldRange.trailingCol());
            var startPasted = targetRange.contains(columns ? oldRange.headerRow() : oldRange.leadingCol());
            var endPasted   = targetRange.contains(columns ? oldRange.footerRow() : oldRange.trailingCol());

            var xOffset = cutRefRange.start[0] - sourceRange.start[0];
            var yOffset = cutRefRange.start[1] - sourceRange.start[1];
            var xOver   = sourceRange.cols() - xOffset - cutRefRange.cols();
            var yOver   = sourceRange.rows() - yOffset - cutRefRange.rows();

            if (startCutted) {
                if (endPasted) {
                    col1 = newRefRange.start[0];
                    row1 = newRefRange.start[1];
                    col2 = targetRange.start[0] + xOffset + cutRefRange.cols() - 1;
                    row2 = targetRange.start[1] + yOffset + cutRefRange.rows() - 1;
                } else if (overlaps || !forward) {
                    col1 = (xOffset > 0 || !forward) ? (targetRange.start[0] + xOffset) : newRefRange.start[0];
                    row1 = (yOffset > 0 || !forward) ? (targetRange.start[1] + yOffset) : newRefRange.start[1];
                    col2 = Math.max(cutRefRange.cols(), newRefRange.end[0]);
                    row2 = Math.max(cutRefRange.rows(), newRefRange.end[1]);
                }
            } else if (endCutted) {
                if (startPasted) {
                    col1 = Math.min(newRefRange.start[0], targetRange.start[0] + xOffset);
                    row1 = Math.min(newRefRange.start[1], targetRange.start[1] + yOffset);
                    col2 = (xOver > 0 && forward) ? targetRange.start[0] + xOffset + cutRefRange.cols() - 1 : newRefRange.end[0];
                    row2 = (yOver > 0 && forward) ? targetRange.start[1] + yOffset + cutRefRange.rows() - 1 : newRefRange.end[1];
                } else if (overlaps || forward) {
                    col1 = Math.min(newRefRange.start[0], targetRange.start[0] + xOffset);
                    row1 = Math.min(newRefRange.start[1], targetRange.start[1] + yOffset);
                    col2 = (xOver > 0 || forward) ? targetRange.start[0] + xOffset + cutRefRange.cols() - 1 : newRefRange.end[0];
                    row2 = (yOver > 0 || forward) ? targetRange.start[1] + yOffset + cutRefRange.rows() - 1 : newRefRange.end[1];
                }
            }

            // create new range from calculated values
            newRange = Range.create(col1, row1, col2, row2);

            // nothing changes, if the transformed range equals the original range
            if (newRange && newRange.equals(oldRange)) { return null; }

            // create copies of the cell references that will be modified to generate the text representation
            cell1Ref = newRange && CellRef.create(newRange.start, this._cell1Ref.absCol, this._cell1Ref.absRow);
            cell2Ref = newRange && this._cell2Ref && CellRef.create(newRange.end, this._cell2Ref.absCol, this._cell2Ref.absRow);

            return formulaGrammar.formatReference(this._docModel, targetRange.start, this._sheet1Ref, this._sheet2Ref, cell1Ref, cell2Ref);
        }

        return null;
    };

    // class NameToken ========================================================

    /**
     * A formula token that represents a defined name in a formula.
     *
     * @constructor
     *
     * @extends SheetRefToken
     *
     * @param {SpreadsheetModel} docModel
     *  The spreadsheet document model containing this token instance.
     *
     * @param {String} label
     *  The label of the defined name.
     *
     * @param {SheetRef|Null} sheetRef
     *  The sheet reference structure (e.g. the particle 'Sheet1' in the
     *  formula Sheet1!my_name). If null or undefined, the token represents a
     *  local name in the passed sheet, or a global name.
     *
     * @param {Number|Null} extDocRef
     *  The index of the external document, e.g. the number in the brackets in
     *  the formula [1]!extname. The value zero represents a globally defined
     *  name with document reference in the own document, e.g. the zero in
     *  [0]!globalname. If null or undefined, the token represents a name in
     *  the own document without document reference (e.g. globalname).
     */
    var NameToken = SheetRefToken.extend({ constructor: function (docModel, label, sheetRef, extDocRef) {

        // base constructor
        SheetRefToken.call(this, 'name', docModel, sheetRef, null, extDocRef);

        // private properties
        this._value = label;

    } }); // class NameToken

    // private methods --------------------------------------------------------

    /**
     * Returns the model of a defined name from the specified sheet.
     */
    NameToken.prototype._getNameModelFromSheet = function (sheet, label) {
        var sheetModel = (typeof sheet === 'number') ? this._docModel.getSheetModel(sheet) : null;
        return sheetModel ? sheetModel.getNameCollection().getName(label) : null;
    };

    /**
     * Returns the model of the specified defined name.
     *
     * @param {String} label
     *  The label of the defined name to be returned.
     *
     * @param {Number|Null} refSheet
     *  The index of the reference sheet used to resolve sheet-local names
     *  without sheet reference. If set to null, and this token does not
     *  contain its own sheet reference, this method looks for globally defined
     *  names only.
     *
     * @returns {NameModel|Null}
     *  The model of the defined name, if existing; otherwise null.
     */
    NameToken.prototype._resolveNameModel = function (label, refSheet) {

        // TODO: support for external names
        if (!this.isOwnDoc()) { return null; }

        // sheet reference exists (e.g.: Sheet2!name): resolve from specified sheet, do not try global names
        if (this._sheet1Ref) { return this._getNameModelFromSheet(this._sheet1Ref.sheet, label); }

        // document reference without sheet reference (e.g.: [0]!name) refers to global names
        if (this._extDocRef === 0) { return this._docModel.getNameCollection().getName(label); }

        // no sheet reference: try sheet-local names first, then global names
        return this._getNameModelFromSheet(refSheet, label) || this._docModel.getNameCollection().getName(label);
    };

    /**
     * Returns the model of the specified table range.
     *
     * @param {String} tableName
     *  The name of the table range to be returned.
     *
     * @returns {TableModel|Null}
     *  The model of the table range, if existing; otherwise null.
     */
    NameToken.prototype._resolveTableModel = function (tableName) {

        // TODO: support for external tables
        if (!this.isOwnDoc()) { return null; }

        // sheet reference exists (e.g.: Sheet2!Table1): resolve from specified sheet only
        if (this._sheet1Ref) {
            var sheet = this._sheet1Ref.sheet;
            var sheetModel = (typeof sheet === 'number') ? this._docModel.getSheetModel(sheet) : null;
            return sheetModel ? sheetModel.getTableCollection().getTable(tableName) : null;
        }

        // no sheet reference: search for a table range in entire document
        return this._docModel.getTable(tableName);
    };

    // public methods ---------------------------------------------------------

    /**
     * Returns the label of the defined name, as passed to the constructor.
     *
     * @returns {String}
     *  The label of the defined name.
     */
    NameToken.prototype.getValue = function () {
        return this._value;
    };

    /**
     * Returns the model of the defined name referred by this token.
     *
     * @param {Number|Null} refSheet
     *  The index of the reference sheet used to resolve sheet-local names
     *  without sheet reference. If set to null, and this token does not
     *  contain its own sheet reference, this method looks for globally defined
     *  names only.
     *
     * @returns {NameModel|Null}
     *  The model of the defined name, if existing; otherwise null.
     */
    NameToken.prototype.resolveNameModel = function (refSheet) {
        return this._resolveNameModel(this._value, refSheet);
    };

    /**
     * Returns the model of the table range referred by this token.
     *
     * @returns {TableModel|Null}
     *  The model of the table range, if existing; otherwise null.
     */
    NameToken.prototype.resolveTableModel = function () {
        return this._resolveTableModel(this._value);
    };

    /**
     * Generates the text representation of this token.
     *
     * @param {FormulaGrammar} formulaGrammar
     *  The formula grammar containing specific settings for formula tokens.
     *
     * @param {Object} [options]
     *  Optional parameters:
     *  @param {SheetRef|Null} [options.sheet1Ref]
     *      An arbitrary sheet reference to be used instead of the own sheet
     *      reference.
     *  @param {String} [options.label]
     *      An arbitrary label to be used to generate the text representation.
     */
    NameToken.prototype.getText = function (formulaGrammar, options) {
        var sheetRefs = this._resolveSheetRefs(options);
        var label = Utils.getStringOption(options, 'label', this._value);
        return formulaGrammar.formatName(this._docModel, this._extDocRef, sheetRefs.s1, label);
    };

    /**
     * Returns a text description of this token for debug logging.
     */
    NameToken.prototype.debugText = function () {
        return this.debugSheetPrefix() + this._value;
    };

    // 'virtual' methods ------------------------------------------------------

    /**
     * Resolves the new text representation of this token, after the label of a
     * defined name has been changed in the spreadsheet document.
     *
     * @param {FormulaGrammar} formulaGrammar
     *  The formula grammar containing specific settings for formula tokens.
     *
     * @param {Number|Null} sheet
     *  The zero-based index of the sheet containing the changed name, or null,
     *  if a globally defined name has been changed.
     *
     * @param {String} oldLabel
     *  The old label of the defined name.
     *
     * @param {String} newLabel
     *  The new label of the defined name.
     *
     * @param {Number|Null} refSheet
     *  The index of the reference sheet used to resolve sheet-local names.
     *
     * @returns {String|Null}
     *  The text representation of this name token, if it would be changed;
     *  otherwise null.
     */
    NameToken.prototype.resolveRelabeledName = function (formulaGrammar, sheet, oldLabel, newLabel, refSheet) {

        // nothing to do, if the old label of the relabeled name does not match
        if (oldLabel.toLowerCase() !== this._value.toLowerCase()) { return null; }

        // try to get the name model from the document (nothing to do, if the name cannot be found)
        var nameModel = this._resolveNameModel(newLabel, refSheet);
        if (!nameModel) { return; }

        // check if this token would resolve to the relabed name, according to the parent sheet of the changed name
        var sheetModel = (typeof sheet === 'number') ? this._docModel.getSheetModel(sheet) : null;
        if (sheetModel !== nameModel.getSheetModel()) { return null; }

        // generate the text representation for the new label
        return this.getText(formulaGrammar, { label: newLabel });
    };

    // class MacroToken =======================================================

    /**
     * A formula token that represents a macro function call in a formula.
     *
     * @constructor
     *
     * @extends SheetRefToken
     *
     * @param {SpreadsheetModel} docModel
     *  The spreadsheet document model containing this token instance.
     *
     * @param {String} label
     *  The name of the macro function.
     *
     * @param {SheetRef|Null} sheetRef
     *  The sheet reference structure (e.g. the particle 'Module1' in the
     *  formula Module1!myFunc() of a macro call). If null or undefined, the
     *  token represents a globally available macro function.
     */
    var MacroToken = SheetRefToken.extend({ constructor: function (docModel, label, sheetRef) {

        // base constructor
        SheetRefToken.call(this, 'macro', docModel, sheetRef, null, null);

        // private properties
        this._value = label;

    } }); // class MacroToken

    // public methods ---------------------------------------------------------

    /**
     * Returns the name of the macro function, as passed to the constructor.
     *
     * @returns {String}
     *  The name of the macro function.
     */
    MacroToken.prototype.getValue = function () {
        return this._value;
    };

    /**
     * Generates the text representation of this token.
     *
     * @param {FormulaGrammar} formulaGrammar
     *  The formula grammar containing specific settings for formula tokens.
     *
     * @param {Object} [options]
     *  Optional parameters:
     *  @param {SheetRef|Null} [options.sheet1Ref]
     *      An arbitrary sheet reference to be used instead of the own sheet
     *      reference.
     *  @param {String} [options.label]
     *      An arbitrary label to be used to generate the text representation.
     */
    MacroToken.prototype.getText = function (formulaGrammar, options) {
        var sheetRefs = this._resolveSheetRefs(options);
        var label = Utils.getStringOption(options, 'label', this._value);
        return formulaGrammar.formatMacro(this._docModel, this._extDocRef, sheetRefs.s1, label);
    };

    /**
     * Returns a text description of this token for debug logging.
     */
    MacroToken.prototype.debugText = function () {
        return this.debugSheetPrefix() + this._value;
    };

    // class TableToken =======================================================

    /**
     * A formula token that represents a structured table reference in a
     * formula.
     *
     * @constructor
     *
     * @extends BaseToken
     *
     * @param {SpreadsheetModel} docModel
     *  The spreadsheet document model containing this token instance.
     *
     * @param {String} tableName
     *  The name of the table referred by this token.
     *
     * @param {Object} [initOptions]
     *  Optional parameters:
     *  @param {String} [initOptions.regionKey]
     *      The resource key of a table region. If omitted, the default region
     *      'DATA' will be referred by the token. Additionally, the combined
     *      regions 'HEADER,DATA' and 'DATA,TOTALS' will be accepted.
     *  @param {String} [initOptions.col1Name]
     *      The name of a single table column referred by this token, or the
     *      name of the first column in a column range. If omitted, the token
     *      will refer to all columns of the table.
     *  @param {String} [initOptions.col2Name]
     *      The name of the second column in a column range. If omitted, the
     *      token will refer to a single column, or all columns of the table.
     *  @param {Boolean} [initOptions.openWs=false]
     *      Whether to add a space characters after the opening bracket when
     *      generating the text representation of this token.
     *  @param {Boolean} [initOptions.closeWs=false]
     *      Whether to add a space characters before the closing bracket when
     *      generating the text representation of this token.
     *  @param {Boolean} [initOptions.sepWs=false]
     *      Whether to add a space characters after each separator character
     *      inside a complex table reference when generating the text
     *      representation of this token.
     */
    var TableToken = BaseToken.extend({ constructor: function (docModel, tableName, initOptions) {

        // base constructor
        BaseToken.call(this, 'table');

        // private properties
        this._docModel = docModel;
        this._tableName = tableName;
        this._regionKey = Utils.getStringOption(initOptions, 'regionKey', null);
        this._regionKeys = this._regionKey ? this._regionKey.split(',') : null;
        this._col1Name = Utils.getStringOption(initOptions, 'col1Name', null);
        this._col2Name = this._col1Name ? Utils.getStringOption(initOptions, 'col2Name', null) : null;
        this._openWs = Utils.getBooleanOption(initOptions, 'openWs', false);
        this._closeWs = Utils.getBooleanOption(initOptions, 'closeWs', false);
        this._sepWs = Utils.getBooleanOption(initOptions, 'sepWs', false);

    } }); // class TableToken

    // private methods --------------------------------------------------------

    /**
     * Returns details about the effective position of the cell ranges referred
     * by this token.
     *
     * @param {Address|Null} targetAddress
     *  The target address needed to resolve the special table region 'ROW'. If
     *  set to null, the 'ROW' region will be resolved to the entire table
     *  range (region type 'ALL').
     *
     * @returns {Object|Null}
     *  If the token cannot be resolved to a valid position, null will be
     *  returned. Otherwise, the return value is a descriptor object with the
     *  following properties:
     *  - {TableModel} tableModel
     *      The model of the table range.
     *  - {Range} tableRange
     *      The position of the cell range referred by this token.
     */
    TableToken.prototype._resolveRangeData = function (targetAddress) {

        // resolve the model of the table range referred by this token
        var tableModel = this.resolveTableModel();
        if (!tableModel) { return null; }

        // resolve the table region (nothing to return, if no table region is available);
        var regionKey = this.getRegionKey();
        // special handling for the 'ThisRow' region, if no target address is given (do not result in error)
        var ignoreRowRef = !targetAddress && (regionKey === 'ROW');
        var regionRange = ignoreRowRef ? tableModel.getRange() : tableModel.getRegionRange(regionKey, targetAddress);
        if (!regionRange) { return null; }

        // resolve column names (nothing to return, if no valid column range is available)
        var columnRange = tableModel.getRange();
        if (this._col1Name) {
            var tableCol1 = tableModel.findColumnByName(this._col1Name);
            if (tableCol1 !== null) {
                if (this._col2Name) {
                    var tableCol2 = tableModel.findColumnByName(this._col2Name);
                    if (tableCol2 !== null) {
                        columnRange = columnRange.colRange(Math.min(tableCol1, tableCol2), Math.abs(tableCol1 - tableCol2) + 1);
                    } else {
                        columnRange = null;
                    }
                } else {
                    columnRange = columnRange.colRange(tableCol1);
                }
            } else {
                columnRange = null;
            }
        }
        if (!columnRange) { return null; }

        // return the resulting information
        return { tableModel: tableModel, tableRange: regionRange.intersect(columnRange) };

    };

    // public methods ---------------------------------------------------------

    /**
     * Returns the name of the table range referred by this token.
     *
     * @returns {String}
     *  The name of the table range referred by this token.
     */
    TableToken.prototype.getTableName = function () {
        return this._tableName;
    };

    /**
     * The resource key of the table region contained in this table token.
     *
     * @returns {String}
     *  The resource key of a single table region, or one of the combined
     *  regions 'HEADER,DATA' or 'DATA,TOTALS'.
     */
    TableToken.prototype.getRegionKey = function () {
        return this._regionKey || 'DATA';
    };

    /**
     * Returns whether this token extracts a specific column, or a range of
     * columns from the table range, e.g. when using Table[Column1].
     *
     * @returns {Boolean}
     *  Whether this token extracts specific columns from the table range.
     */
    TableToken.prototype.hasColumnRange = function () {
        return !!this._col1Name;
    };

    /**
     * Returns the model of the table range referred by this token.
     *
     * @returns {TableModel|Null}
     *  The model of the table range referred by this token, if available;
     *  otherwise null.
     */
    TableToken.prototype.resolveTableModel = function () {
        return this._docModel.getTable(this._tableName);
    };

    /**
     * Returns the effective position of the cell range referred by this token.
     *
     * @param {Address} targetAddress
     *  The target address needed to resolve the special table region 'ROW'.
     *
     * @returns {Range3D|Null}
     *  The effective position of the cell range referred by this token (with
     *  sheet indexes), if available; otherwise null.
     */
    TableToken.prototype.getRange3D = function (targetAddress) {

        // resolve the sub ranges referred by this table token
        var result = this._resolveRangeData(targetAddress);
        if (!result) { return null; }

        // create a 3D range with sheet indexes
        var sheet = result.tableModel.getSheetModel().getIndex();
        return Range3D.createFromRange(result.tableRange, sheet);
    };

    /**
     * Generates the text representation of this token.
     *
     * @param {FormulaGrammar} formulaGrammar
     *  The formula grammar containing specific settings for formula tokens.
     *
     * @param {Object} [options]
     *  Optional parameters:
     *  @param {String} [options.tableName]
     *      An arbitrary table name to be inserted into the text representation
     *      instead of the own table name.
     */
    TableToken.prototype.getText = function (formulaGrammar, options) {

        function generateWs(flag) { return flag ? ' ' : ''; }

        // the table name to be inserted (own table name can be overridden by options)
        var tableName = Utils.getStringOption(options, 'tableName', this._tableName);
        // whether the token contains leading or trailing whitespace characters
        var outerWs = this._openWs || this._closeWs;

        // add empty brackets after table name in OOXML operation mode
        if (!this._regionKeys && !this._col1Name) {
            return tableName + ((!formulaGrammar.UI && (formulaGrammar.REF_SYNTAX === 'ooxml')) ? ('[' + generateWs(outerWs) + ']') : '');
        }

        // the key of a single table region (null for combined regions)
        var regionKey = (this._regionKeys && (this._regionKeys.length === 1)) ? this._regionKeys[0] : null;
        // the string components to be added into a complex table reference
        var components = [];

        // add the at-sign shortcut for '#This Row' in UI grammar (e.g. =Table[@Col1] instead of =Table[[#This Row],[Col1]])
        if (formulaGrammar.UI && (regionKey === 'ROW')) {
            var colRefText = '';
            if (this._col2Name) {
                colRefText = '[' + formulaGrammar.encodeTableColumn(this._col1Name) + ']:[' + formulaGrammar.encodeTableColumn(this._col2Name) + ']';
            } else if (this._col1Name) {
                colRefText = formulaGrammar.isSimpleTableColumn(this._col1Name) ? this._col1Name : ('[' + formulaGrammar.encodeTableColumn(this._col1Name) + ']');
            }
            components.push('@' + colRefText);

        // add a single region specifier with single brackets (e.g. =Table[#All])
        } else if (regionKey && !outerWs && !this._col1Name) {
            components.push(formulaGrammar.getTableRegionName(regionKey));

        // add a simple column name with single brackets (not for columns with whitespace as first or last character, or with leading/trailing whitespace)
        } else if (!this._regionKeys && this._col1Name && !this._col2Name && !outerWs && !/^\s|\s$/.test(this._col1Name)) {
            components.push(formulaGrammar.encodeTableColumn(this._col1Name));

        // create a complex table reference with regions and column names
        } else {

            // add region names if existing
            if (this._regionKeys) {
                this._regionKeys.forEach(function (regionKey) {
                    components.push('[' + formulaGrammar.getTableRegionName(regionKey) + ']');
                });
            }

            // add the table column names
            if (this._col2Name) {
                components.push('[' + formulaGrammar.encodeTableColumn(this._col1Name) + ']:[' + formulaGrammar.encodeTableColumn(this._col2Name) + ']');
            } else if (this._col1Name) {
                components.push('[' + formulaGrammar.encodeTableColumn(this._col1Name) + ']');
            }
        }

        // omit table name in UI grammar, if the formula is located inside the table (but only for
        // table references consisting of a single column, or a this-row reference to a single column)
        var addTableName = true;
        if (formulaGrammar.UI && Utils.getBooleanOption(options, 'unqualifiedTables', false)) {
            if (this._col1Name && !this._col2Name && (!this._regionKeys || (regionKey === 'ROW'))) {
                var tableModel = this.resolveTableModel();
                var refSheet = Utils.getIntegerOption(options, 'refSheet', null);
                if (tableModel && (tableModel.getSheetModel().getIndex() === refSheet)) {
                    var targetAddress = Utils.getOption(options, 'targetAddress', null);
                    addTableName = !targetAddress || !tableModel.containsCell(targetAddress);
                }
            }
        }

        return (addTableName ? tableName : '') +
            '[' + generateWs(this._openWs) +
            components.join(formulaGrammar.SEP + generateWs(this._sepWs)) +
            generateWs(this._closeWs) + ']';
    };

    /**
     * Returns a text description of this token for debug logging.
     */
    TableToken.prototype.debugText = function () {
        var text = this._tableName;
        if (this._regionKeys) { text += this._regionKeys.map(function (regionKey) { return '[#' + regionKey + ']'; }); }
        if (this._col1Name) { text += '[' + this._col1Name + ']'; }
        if (this._col2Name) { text += ':[' + this._col2Name + ']'; }
        return text;
    };

    // 'virtual' methods ------------------------------------------------------

    /**
     * Resolves the new text representation of this token, before a sheet will
     * be deleted in the spreadsheet document.
     *
     * @param {FormulaGrammar} formulaGrammar
     *  The formula grammar containing specific settings for formula tokens.
     *
     * @param {Number} sheet
     *  The zero-based index of the deleted sheet.
     *
     * @returns {String|Null}
     *  The text representation of this sheet reference token, if it has been
     *  changed; otherwise null.
     */
    TableToken.prototype.resolveDeletedSheet = function (formulaGrammar, sheet) {
        // return simple #REF! error, if the table is located in the deleted sheet
        var tableModel = this.resolveTableModel();
        return (!tableModel || (tableModel.getSheetModel().getIndex() === sheet)) ? formulaGrammar.getErrorName(ErrorCode.REF) : null;
    };

    /**
     * Resolves the new text representation of this token, before a sheet will
     * be renamed in the spreadsheet document, or after an existing sheet has
     * been copied.
     *
     * @param {FormulaGrammar} formulaGrammar
     *  The formula grammar containing specific settings for formula tokens.
     *
     * @param {Number} sheet
     *  The zero-based index of the renamed sheet.
     *
     * @param {String} sheetName
     *  The new name of the sheet.
     *
     * @param {Object} [tableNamesMap]
     *  A map that specifies how to change the table names in table references.
     *  Used when copying an existing sheet where the cloned table ranges have
     *  been renamed, and the formulas in the copied sheet should refer to the
     *  new table ranges. The old table names are the map keys, and the new
     *  table names are the map values. If omitted, no table references will be
     *  changed (e.g. when renaming an existing sheet).
     *
     * @returns {String|Null}
     *  The text representation of this table reference token, if it has been
     *  changed; otherwise null.
     */
    TableToken.prototype.resolveRenamedSheet = function (formulaGrammar, sheet, sheetName, tableNamesMap) {

        // nothing to do without a table name map (rename sheet)
        if (!tableNamesMap) { return null; }

        // return simple #REF! error, if the table token is invalid
        var tableModel = this.resolveTableModel();
        if (!tableModel) { return formulaGrammar.getErrorName(ErrorCode.REF); }

        // get the new table name (do not use the property 'this._tableName', it may have different character case)
        var newTableName = tableNamesMap[tableModel.getName()];

        // if the table name is contained in the map, generate the new text representation
        return newTableName ? this.getText(formulaGrammar, { tableName: newTableName }) : null;
    };

    /**
     * Resolves the new text representation of this token, before cell ranges
     * will be moved in a specific sheet of the spreadsheet document.
     *
     * @param {FormulaGrammar} formulaGrammar
     *  The formula grammar containing specific settings for formula tokens.
     *
     * @param {Number} sheet
     *  The zero-based index of the sheet containing the moved cell ranges.
     *
     * @param {Array<MoveDescriptor>} moveDescs
     *  An array of move descriptors created while generating the move document
     *  operations.
     *
     * @returns {String|Null}
     *  The string representation of the moved cell reference, if it has been
     *  changed; otherwise null.
     */
    TableToken.prototype.resolveMovedCells = function (formulaGrammar, sheet, moveDescs) {

        // return simple #REF! error, if the table token is invalid
        var rangeData = this._resolveRangeData(null);
        if (!rangeData) { return formulaGrammar.getErrorName(ErrorCode.REF); }

        // table ranges located on another sheet will not be affected
        if (sheet !== rangeData.tableModel.getSheetModel().getIndex()) { return null; }

        // transform the resulting range; return #REF!, if the range will be deleted
        var origRange = rangeData.tableRange;
        if (!MoveDescriptor.transformRange(origRange, moveDescs)) {
            return formulaGrammar.getErrorName(ErrorCode.REF);
        }

        // if the table token refers to columns, invalidate it if the start or the end
        // column will be deleted (even if some inner columns remain available)
        if (this.hasColumnRange()) {
            var col1Range = MoveDescriptor.transformRange(origRange.leadingCol(), moveDescs);
            var col2Range = MoveDescriptor.transformRange(origRange.trailingCol(), moveDescs);
            if (!col1Range || !col2Range) { return formulaGrammar.getErrorName(ErrorCode.REF); }
        }

        return null;
    };

    // static class Tokens ====================================================

    var Tokens = {
        FixedToken: FixedToken,
        OperatorToken: OperatorToken,
        SeparatorToken: SeparatorToken,
        ParenthesisToken: ParenthesisToken,
        LiteralToken: LiteralToken,
        MatrixToken: MatrixToken,
        FunctionToken: FunctionToken,
        SheetRefToken: SheetRefToken,
        ReferenceToken: ReferenceToken,
        NameToken: NameToken,
        MacroToken: MacroToken,
        TableToken: TableToken
    };

    // exports ================================================================

    return Tokens;

});
