/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/spreadsheet/model/formula/parser/desccollection', [
    'io.ox/office/tk/container/container'
], function (Container) {

    'use strict';

    // private global functions ===============================================

    /**
     * Returns the pattern for a regular expression that maches any of the
     * strings in the passed array.
     *
     * @param {Array<String>} list
     *  The array with strings to be matched by a regular expression.
     *
     * @returns {String}
     *  The pattern for a regular expression that maches any of the strings in
     *  the passed array, enclosed in parentheses. The strings will be sorted
     *  by length in order to create a greedy regular expression. Example: For
     *  the strings 'a', 'b', and 'ab', the pattern '(ab|a|b)' will be created.
     */
    function createListPattern(list) {
        return '(' + list.sort(function (n1, n2) { return n2.length - n1.length; }).map(_.escapeRegExp).join('|') + ')';
    }

    // class Descriptor =======================================================

    /**
     * Base class for descriptors of single formula resource objects. Contains
     * the native name, and the localized name of an arbitrary formula resource
     * object (an operator, a function, etc.), and optionally other custom
     * properties.
     *
     * @constructor
     *
     * @property {String} key
     *  The unique key of the resource object represented by this descriptor.
     *
     * @property {String} nativeName
     *  The native name of the resource object.
     *
     * @property {String} localName
     *  The localized name of the resource object.
     *
     * @property {Array<String>} altNativeNames
     *  Alternative native names that can be used for the resource object.
     *
     * @property {Array<String>} altLocalNames
     *  Alternative localized names that can be used for the resource object.
     */
    function Descriptor(key, nativeName, localName) {

        this.key = key;

        this.nativeName = nativeName;
        this.localName = localName;

        this.altNativeNames = [];
        this.altLocalNames = [];

    } // function Descriptor

    // class DescriptorCollection =============================================

    /**
     * A collection of resource descriptors of a specific type.
     *
     * @constructor
     *
     * @extends Container
     *
     * @param {Object} map
     *  The raw source data map, mapped by the resource keys to be used in the
     *  new collection.
     *
     * @param {Function} generator
     *  The generator callback function that will be invoked for each entry in
     *  the passed data map. Receives the following parameters:
     *  (1) {Object} data
     *      The current entry of the map passed to this method.
     *  (2) {String} key
     *      The resource key of the current entry.
     *  Can return a descriptor (instance of class Descriptor) to be inserted
     *  into the new collection; or null or undefined to skip the map entry.
     *
     * @param {Object} [context]
     *  The calling context for the callback function.
     */
    var DescriptorCollection = Container.extend({ constructor: function (map, generator, context) {

        // base constructor
        Container.call(this);

        // resource keys of all descriptors as map values, mapped by their names
        this._nativeKeys = {};
        this._localKeys = {};

        // native and localized names of all descriptors, as arrays
        this._nativeNames = [];
        this._localNames = [];

        // initialize the key maps from the passed list of descriptors
        Container.forEach(map, function (data, key) {

            // generator callback creates a new descriptor instance, or returns null
            var descriptor = generator.call(context, data, key);
            if (!(descriptor instanceof Descriptor)) { return; }

            this._insert(key, descriptor);
            this._nativeKeys[descriptor.nativeName.toUpperCase()] = this._localKeys[descriptor.localName.toUpperCase()] = key;
            descriptor.altNativeNames.forEach(function (altName) { this._nativeKeys[altName.toUpperCase()] = key; }, this);
            descriptor.altLocalNames.forEach(function (altName) { this._localKeys[altName.toUpperCase()] = key; }, this);
            this._nativeNames.push(descriptor.nativeName);
            this._nativeNames.push.apply(this._nativeNames, descriptor.altNativeNames);
            this._localNames.push(descriptor.localName);
            this._localNames.push.apply(this._localNames, descriptor.altLocalNames);
        }, this);

        // sort the native and localized names of all descriptors
        this._nativeNames.sort();
        this._localNames.sort();

        // regular expression patterns matching any of the supported names (sort by name length to create a greedy pattern)
        this._nativePattern = createListPattern(this._nativeNames);
        this._localPattern = createListPattern(this._localNames);

    } }); // class DescriptorCollection

    // public methods ---------------------------------------------------------

    /**
     * Returns the descriptor for the specified resource key.
     *
     * @param {String} key
     *  The unique resource key of the descriptor.
     *
     * @returns {Descriptor|Null}
     *  A descriptor for the specified resource key; or null, if the passed key
     *  is not valid.
     */
    DescriptorCollection.prototype.get = function (key) {
        return this._get(key) || null;
    };

    /**
     * Returns the value of a specific property from an existing descriptor.
     *
     * @param {String} key
     *  The unique resource key of the descriptor.
     *
     * @param {String} name
     *  The name of the descriptor property.
     *
     * @param {Any} def
     *  The default value returned in case the descriptor, or the property does
     *  not exist.
     *
     * @returns {Any}
     *  The value of the specified descriptor property, or the default value.
     */
    DescriptorCollection.prototype.getProperty = function (key, name, def) {
        var descriptor = this._get(key);
        return (descriptor && (name in descriptor)) ? descriptor[name] : def;
    };

    /**
     * Returns the native or localized name of the descriptor with the passed
     * unique resource key.
     *
     * @param {String} key
     *  The unique resource key of a descriptor.
     *
     * @param {Boolean} localized
     *  Whether to return the localized name according to the UI language
     *  (true), or the native name as used in the file format (false).
     *
     * @returns {String|Null}
     *  The native or localized name for the passed resource key; or null, if
     *  the passed resource key cannot be resolved to the name of a descriptor.
     */
    DescriptorCollection.prototype.getName = function (key, localized) {
        var descriptor = this._get(key);
        return !descriptor ? null : localized ? descriptor.localName : descriptor.nativeName;
    };

    /**
     * Converts the passed native or localized name of a descriptor to its
     * unique resource key.
     *
     * @param {String} name
     *  The native or localized name (case-insensitive).
     *
     * @param {Boolean} localized
     *  Whether the passed name is localized according to the UI language
     *  (true), or native as used in the file format (false).
     *
     * @returns {String|Null}
     *  The resource key for the passed name; or null, if the passed string is
     *  not the name of a descriptor in this collection.
     */
    DescriptorCollection.prototype.getKey = function (name, localized) {
        var keyMap = localized ? this._localKeys : this._nativeKeys;
        return keyMap[name.toUpperCase()] || null;
    };

    /**
     * Returns a sorted list with the native or localized names of all
     * descriptors in this collection.
     *
     * @param {Boolean} localized
     *  Whether to return the localized names according to the UI language
     *  (true), or the native names as used in the file format (false).
     *
     * @returns {Array<String>}
     *  A sorted list with the native or localized names of all descriptors in
     *  this collection.
     */
    DescriptorCollection.prototype.getAllNames = function (localized) {
        return localized ? this._localNames : this._nativeNames;
    };

    /**
     * Returns the pattern for a regular expression matching any of the native
     * or localized names of all descriptors in this collection.
     *
     * @param {Boolean} localized
     *  Whether to return a pattern for the localized names according to the UI
     *  language (true), or for the native names as used in the file format
     *  (false).
     *
     * @returns {String}
     *  The pattern for a regular expression matching any of the native or
     *  localized names of all descriptors in this collection, in the form
     *  '(name1|name2|name3|...)'. The names in the pattern are sorted by
     *  length (longest names first) to result in a greedy pattern.
     */
    DescriptorCollection.prototype.getAllPattern = function (localized) {
        return localized ? this._localPattern : this._nativePattern;
    };

    // sub classes ------------------------------------------------------------

    /**
     * The class constructor of the instances of a descriptor collection.
     *
     * @type Function
     */
    DescriptorCollection.ElementType = Descriptor;

    // exports ================================================================

    return DescriptorCollection;

});
