/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/spreadsheet/model/formula/interpret/formulainterpreter', [
    'io.ox/office/tk/utils',
    'io.ox/office/tk/object/baseobject',
    'io.ox/office/baseframework/model/modelobject',
    'io.ox/office/spreadsheet/utils/sheetutils',
    'io.ox/office/spreadsheet/model/presetformattable',
    'io.ox/office/spreadsheet/model/formula/formulautils',
    'io.ox/office/spreadsheet/model/formula/interpret/operand',
    'io.ox/office/spreadsheet/model/formula/interpret/formulacontext'
], function (Utils, BaseObject, ModelObject, SheetUtils, PresetFormatTable, FormulaUtils, Operand, FormulaContext) {

    'use strict';

    // convenience shortcuts
    var ErrorCode = SheetUtils.ErrorCode;
    var Address = SheetUtils.Address;
    var Range = SheetUtils.Range;
    var Range3DArray = SheetUtils.Range3DArray;
    var Scalar = FormulaUtils.Scalar;
    var Matrix = FormulaUtils.Matrix;
    var Dimension = FormulaUtils.Dimension;
    var FormulaError = FormulaUtils.FormulaError;

    // class OperandResolver ==================================================

    /**
     * A helper class used to store the compiler token nodes for the operands
     * of an operator or function call.
     *
     * Supports resolving the operands from the token nodes on demand, e.g. for
     * lazy operands. Will be passed to instances of class FormulaContext when
     * resolving the result of a specific operator or function, in order to be
     * able to reuse the same formula context instance for all operators and
     * functions of an entire formula.
     *
     * @constructor
     */
    function OperandResolver() {

        this._operands = [];

    } // class OperandResolver

    // public methods ---------------------------------------------------------

    OperandResolver.prototype.push = function (resolveFunc) {
        this._operands.push({ operand: null, resolve: _.once(resolveFunc) });
    };

    OperandResolver.prototype.size = function () {
        return this._operands.length;
    };

    OperandResolver.prototype.get = function (index) {
        var data = this._operands[index];
        return (data && data.operand) || null;
    };

    OperandResolver.prototype.resolve = function (index) {
        var data = this._operands[index];
        return data ? (data.operand = data.resolve()) : null;
    };

    // class InterpreterInstance ==============================================

    /**
     * Resolves a compiled formula token array to the result of the formula.
     * This implementation has been moved outside the class FormulaInterpreter
     * to be able to recursively resolve multiple formulas (e.g. defined
     * names).
     *
     * @constructor
     *
     * @extends BaseObject
     *
     * @param {SpreadsheetModel} docModel
     *  The spreadsheet document model containing this instance.
     *
     * @param {CompilerNode} rootNode
     *  The root node of the compiled token tree, as returned by the method
     *  FormulaCompiler.compileTokens().
     *
     * @param {Object} [initOptions]
     *  Optional parameters. Supports all options also supported by the method
     *  FormulaInterpreter.interpretTokens().
     */
    var InterpreterInstance = BaseObject.extend({ constructor: function (docModel, rootNode, initOptions) {

        // the number formatter of the document
        var numberFormatter = docModel.getNumberFormatter();

        // the standard number format
        var standardFormat = numberFormatter.getStandardFormat();

        // the reference sheet
        var refSheet = Utils.getIntegerOption(initOptions, 'refSheet', null);

        // the target reference address
        var targetAddress = Utils.getOption(initOptions, 'targetAddress', Address.A1);

        // whether to recalculate all dirty cell formulas resolved by cell references
        var recalcDirty = Utils.getOption(initOptions, 'recalcDirty', false);

        // the calling context for the operator resolver callback function
        var formulaContext = new FormulaContext(docModel, initOptions);

        // base constructor ---------------------------------------------------

        BaseObject.call(this, docModel);

        // private methods ----------------------------------------------------

        /**
         * Resolves the passed reference token to its result.
         *
         * @param {ReferenceToken} refToken
         *  The formula token representing a cell range reference. If the token
         *  cannot be resolved to a cell range address, this method returns the
         *  #REF! error code.
         *
         * @returns {Range3DArray|ErrorCode}
         *  The cell range addresses contained in the reference, or a #REF!
         *  error code.
         */
        function resolveReference(refToken) {
            var range = refToken.getRange3D(initOptions);
            return range ? new Range3DArray(range) : ErrorCode.REF;
        }

        /**
         * Resolves the passed name token to its result.
         *
         * @param {NameToken} nameToken
         *  The formula token representing a defined name. If the name does not
         *  exist in the document, this method returns the #NAME? error code.
         *
         * @param {String} contextType
         *  The context type that influences how to resolve values, matrixes,
         *  and references. See method FormulaInterpreter.interpretTokens() for
         *  details.
         *
         * @param {Dimension|Null} matrixDim
         *  The expected dimension of a matrix result. If set to null, the size
         *  of the result matrix will be deduced from the operand.
         *
         * @returns {Any}
         *  The result value of the defined name (may be values, matrixes, or
         *  references).
         */
        function resolveDefinedName(nameToken, contextType, matrixDim) {

            // resolve the model of a defined name from the token (unknown names result in the #NAME? error code)
            var nameModel = nameToken.resolveNameModel(refSheet);
            if (!nameModel) { return ErrorCode.NAME; }

            // resolve the formula in the defined name (defined names are always relative to cell A1)
            var options = { refSheet: refSheet, refAddress: Address.A1, targetAddress: formulaContext.getRefAddress(), recalcDirty: recalcDirty };

            // reconstruct the matrix range
            if (matrixDim) {
                options.matrixRange = Range.createFromAddressAndSize(options.targetAddress, matrixDim.cols, matrixDim.rows);
            }

            return nameModel.getTokenArray().interpretFormula(contextType, options).value;
        }

        /**
         * Resolves the passed table token to its result.
         *
         * @param {TableToken} tableToken
         *  The formula token representing a structured table reference. If the
         *  token cannot be resolved to a range, this method returns the #REF!
         *  error code.
         *
         * @returns {Range3DArray|ErrorCode}
         *  The resulting range referred by thie passed table token, or the
         *  #REF! error code.
         */
        function resolveTableRange(tableToken) {
            var range = tableToken.getRange3D(targetAddress);
            return range ? new Range3DArray(range) : ErrorCode.REF;
        }

        /**
         * Checks that the passed return type, and the context type of the
         * formula or subexpression match. Throws a 'reference' exception, if
         * the context type is 'ref', and the passed type is different from
         * 'ref' or 'any'. Excel rejects such formulas, and even announces a
         * broken file format if such formulas are written into the file.
         * Examples of such invalid formulas are:
         *  =A1:1 (combined number with range operator)
         *  =OFFSET(1,1,1) (passed number to a reference-only parameter)
         *
         * It is possible to use operands and functions with return type
         * 'any' in reference context, e.g.
         *  =A1:name (return type of defined names is 'any')
         *  =A1:IF(TRUE,A1,B1) (return type of IF is 'any').
         * If the defined name or function returns a non-reference value, the
         * formula simply results in the #VALUE! error.
         *
         * @param {String} type
         *  The return type of an operator or function, or the type of a single
         *  operand to be checked. Must be one of:
         *  - 'val': A scalar value (numbers, dates, strings, boolean values,
         *      complex numbers, or error codes).
         *  - 'mat': Constant matrixes, or functions returning matrixes.
         *  - 'ref': Cell references (constant references, reference operators,
         *      functions returning a reference, or the #REF! error code).
         *  - 'any': Any of the above (defined names, a few functions returning
         *      any data type).
         *
         * @param {String} contextType
         *  The context type that influences how to resolve values, matrixes,
         *  and references. See method FormulaInterpreter.interpretTokens() for
         *  details. If the context type is 'ref', only the return types 'ref'
         *  and 'any' will be accepted. All other return types will cause to
         *  throw an internal 'reference' exception that marks the formula
         *  structure to be ill-formed (see description above).
         *
         * @throws {FormulaError}
         *  The special 'reference' exception, if the passed return type is not
         *  supported in reference context.
         */
        function checkContextType(type, contextType) {
            // context type 'ref': accept 'ref' and 'any' only
            if ((contextType === 'ref') && (type !== 'ref') && (type !== 'any')) {
                throw new FormulaError('expected reference token', 'reference');
            }
        }

        /**
         * Creates a new instance of the class Operand. An operand is an object
         * that carries formula result values of any data type.
         *
         * @param {Any} value
         *  The value to be stored as operand. Can be any value supported by
         *  the constructor of the class Operand.
         *
         * @param {Object} [options]
         *  Optional parameters. Supports all options that are supported by the
         *  constructor of the class Operand.
         *
         * @returns {Operand}
         *  A new operand with the passed value and options.
         */
        function createOperand(value, options) {
            // do not create a new instance, if an operand has been passed without additional options
            return (!options && (value instanceof Operand)) ? value : formulaContext.createOperand(value, options);
        }

        /**
         * Creates an operand object for the passed parser token.
         *
         * @param {CompilerNode} operandNode
         *  The compiler token representing an operand (scalar values, constant
         *  matrixes, cell references, defined names).
         *
         * @param {String} contextType
         *  The context type that influences how to resolve values, matrixes,
         *  and references. See methods FormulaInterpreter.interpretTokens()
         *  for details.
         *
         * @param {Dimension|Null} matrixDim
         *  The expected dimension of a matrix result. If set to null, the size
         *  of the result matrix will be deduced from the operand.
         *
         * @returns {Operand}
         *  An operand instance representing the passed token.
         *
         * @throws {ErrorCode}
         *  The special error code UNSUPPORTED_ERROR, if the type of the passed
         *  token is not supported.
         */
        function resolveOperandNode(operandNode, contextType, matrixDim) {

            // the parser token representing the operand value
            var token = operandNode.token;
            // the token type
            var type = null;
            // the value to be inserted into the operand
            var value = null;

            FormulaUtils.log('> resolving operand token ' + token + ' with context type ' + contextType);

            switch (token.getType()) {
                case 'lit': // scalar literals
                    value = token.getValue();
                    type = (value === ErrorCode.REF) ? 'ref' : 'val';
                    break;
                case 'mat': // matrix literals
                    value = token.getMatrix();
                    type = 'mat';
                    break;
                case 'ref': // cell range references
                    value = resolveReference(token);
                    type = 'ref';
                    break;
                case 'name': // defined names
                    value = resolveDefinedName(token, contextType, matrixDim);
                    type = 'any';
                    break;
                case 'table': // table ranges
                    value = resolveTableRange(token);
                    type = 'ref';
                    break;
                default:
                    FormulaUtils.throwInternal('InterpreterInstance.resolveOperandNode(): unknown token type: "' + token.getType() + '"');
            }

            // check context type (will throw 'reference' exception)
            checkContextType(type, contextType);
            return createOperand(value);
        }

        /**
         * Converts a number format category to a parsed number format intended
         * to be returned with the result of an operator or function.
         */
        function getParsedFormat(resolver, category) {

            // resolve the number format from all operands, if 'combine' is specified
            if (category === 'combine') {
                var parsedFormat = standardFormat;
                for (var index = 0; parsedFormat && (index < resolver.size()); index += 1) {
                    var operand = resolver.get(index);
                    var opFormat = (operand && operand.getFormat()) || standardFormat;
                    parsedFormat = FormulaUtils.combineParsedFormats(parsedFormat, opFormat);
                }
                return parsedFormat;
            }

            // convert format category to parsed number format
            var format = null;
            switch (category) {
                case 'percent':
                    format = PresetFormatTable.getPercentId(true); // no decimal places
                    break;
                case 'currency':
                    format = numberFormatter.getCurrencyCode({ red: true });
                    break;
                case 'fraction':
                    format = PresetFormatTable.getFractionId(true); // two-digit fractions
                    break;
                case 'date':
                    format = PresetFormatTable.SYSTEM_DATE_ID;
                    break;
                case 'time':
                    format = PresetFormatTable.getTimeId(null, true); // with seconds
                    break;
                case 'datetime':
                    format = PresetFormatTable.SYSTEM_DATETIME_ID;
                    break;
            }
            return (format === null) ? null : numberFormatter.getParsedFormat(format);
        }

        function getOperand(descriptor, signature, operands, matRow, matCol) {

            // check maximum evaluation time for each operator to prevent freezing browser
            formulaContext.checkEvalTime();

            // catch thrown error codes, return them as result value
            var result = (function () {
                try {

                    // initialize the correct matrix element position, before resolving parameter values
                    formulaContext.setMatrixPosition(matRow, matCol);

                    // resolve the operand values according to the signature (except lazy function parameters)
                    var params = operands.map(function (operand, index) {
                        return operand ? formulaContext.convertOperandToValue(operand, signature[index].typeSpec) : null;
                    });

                    // get the raw result of the operator (convert thrown error codes to scalar values)
                    FormulaUtils.logTokens('\xa0 resolving with parameters', params);
                    return descriptor.resolve.apply(formulaContext, params);

                } catch (error) {
                    // operators may throw formula error codes as dedicated result, but rethrow
                    // interpreter errors (class FormulaError), and other internal JS exceptions
                    if (error instanceof ErrorCode) { return error; }
                    throw error;
                }
            }());

            // ensure to return an Operand instances
            return createOperand(result);
        }

        /**
         * Processes the passed operator or function call. Calculates the
         * required number of operands from the following compiler tokens, and
         * calculates the result of the operator or function.
         *
         * @param {CompilerNode} subNode
         *  The root node of a compiler token sub-tree representing an operator
         *  or function.
         *
         * @param {String} contextType
         *  The context type that influences how to resolve values, matrixes,
         *  and references. See methods FormulaInterpreter.interpretTokens()
         *  for details.
         *
         * @param {Dimension|Null} matrixDim
         *  The expected dimension of a matrix result. Used if the context type
         *  is 'mat'. If set to null, the size of the result matrix will be
         *  deduced from the operands of the passed token node.
         *
         * @returns {Operand}
         *  An operand instance representing the result of the operator or
         *  function call.
         *
         * @throws {FormulaError}
         *  The error code 'missing', if there are not enough tokens available
         *  in the compiled token array.
         */
        function resolveOperatorNode(subNode, contextType, matrixDim) {

            // immediately throw compiler errors without any further processing
            if (subNode.error) { throw subNode.error; }

            // the parser token representing the operator or function
            var token = subNode.token;
            FormulaUtils.log('> processing operator ' + token + ' with ' + subNode.operands.length + ' parameters and context type ' + contextType);

            // the operator/function specification; return #NAME? for unknown functions (external functions, macro calls)
            var descriptor = subNode.descriptor;
            if (!descriptor) { return createOperand(ErrorCode.NAME); }

            // check context type (will throw 'reference' exception)
            checkContextType(descriptor.type, contextType);

            // check missing implementation
            if (!_.isFunction(descriptor.resolve)) {
                FormulaUtils.warn('InterpreterInstance.resolveOperatorNode(): unsupported operator or function "' + token + '"');
                return createOperand(FormulaUtils.UNSUPPORTED_ERROR);
            }

            // whether the matrix range has been passed explicitly
            if (matrixDim && !Matrix.isValidDim(matrixDim)) {
                return createOperand(FormulaUtils.UNSUPPORTED_ERROR);
            }

            // create and fill the operand resolver
            var resolver = new OperandResolver();
            var signature = subNode.signature;
            signature.forEach(function (typeData, index) {

                // the operand node to be pushed into the resolver
                var opNode = subNode.operands[index];
                // whether the parameter is value type in a value operator
                var isValueType = (typeData.baseType === 'val') && (descriptor.type === 'val');
                // the effective context type for the operand (used to resolve the entire subtree)
                var opContextType = (typeData.matSpec === 'force') ? 'mat' : typeData.baseType;

                // If context type for this operator is matrix, and the current parameter is of type value,
                // and this operator returns a value, pass the matrix context type through to the operand
                // (resolve nested cell references to matrixes instead of single values).
                if ((contextType === 'mat') && (isValueType || (typeData.matSpec === 'pass'))) {
                    opContextType = 'mat';
                }

                var forwardMatrix = isValueType && (opContextType === 'mat');
                resolver.push(resolveNode.bind(null, opNode, opContextType, forwardMatrix ? matrixDim : null));
            });

            // resolve the unconverted operands (except lazy function parameters)
            var operands = signature.map(function (typeData, index) {
                return (typeData.typeSpec === 'any:lazy') ? null : resolver.resolve(index);
            });

            // Implicitly expand matrix range for value operators, if all operands are value type, and at least
            // one will receive a matrix. Example: In the formula =SUM({1;2|3;4}+1), the addition operator will
            // add the scalar number 1 to each matrix element, before handing the new matrix to SUM.
            if (!matrixDim && (descriptor.type === 'val')) {

                // in matrix context: any value-type parameter with a matrix or reference;
                // otherwise: all parameters must be value-type, any of them gets a matrix (no reference)
                var matContext = contextType === 'mat';
                if (matContext || signature.every(function (typeData) { return typeData.baseType === 'val'; })) {
                    try {
                        operands.forEach(function (operand, index) {
                            if (operand && (matContext ? ((signature[index].baseType === 'val') && !operand.isScalar()) : operand.isMatrix())) {
                                var opDim = operand.getDimension({ errorCode: ErrorCode.VALUE }); // #VALUE! thrown for complex references
                                matrixDim = Dimension.boundary(matrixDim, opDim);
                            }
                        });
                    } catch (error) {
                        // #VALUE! error code thrown for complex references
                        if (error instanceof ErrorCode) { return createOperand(error); }
                        throw error;
                    }

                    // validate the resulting matrix size early
                    if (matrixDim) {
                        if (!Matrix.isValidDim(matrixDim)) {
                            return createOperand(FormulaUtils.UNSUPPORTED_ERROR);
                        }
                        contextType = 'mat';
                    }
                }
            }

            // calculate the result of the operator
            var operand = FormulaUtils.takeTime(descriptor.key, function () {
                try {

                    // register the operator resolver with the current operands at the formula context, to make the
                    // operands accessible to function implementations via public methods of the context
                    formulaContext.pushOperandResolver(resolver, contextType, matrixDim);

                    // invoke the resolver callback function of the operator
                    var operand = getOperand(descriptor, signature, operands, 0, 0);

                    // allow 'mat'-type operators to return scalars (especially error codes)
                    if ((descriptor.type === 'mat') && operand.isScalar()) {
                        operand = createOperand(Matrix.create(1, 1, operand.getRawValue()));
                    }

                    // exit immediately if the result does not match the specified type
                    if ((descriptor.type !== 'any') && (descriptor.type !== operand.getType()) &&
                        // allow error codes as return values for 'ref'-type operators
                        !((descriptor.type === 'ref') && operand.isScalar() && (operand.getRawValue() instanceof ErrorCode)) &&
                        // in matrix context, 'val'-type functions/operands are allowed to return matrixes
                        !((descriptor.type === 'val') && (contextType === 'mat') && operand.isMatrix())
                    ) {
                        FormulaUtils.throwInternal('InterpreterInstance.resolveOperatorNode(): ' + descriptor.key + ': operator result type does not match operator return type');
                    }

                    // expand scalar operators to matrix results if required
                    if (matrixDim && operand.isScalar()) {
                        FormulaUtils.info('\xa0 expand scalar result to ' + matrixDim + ' matrix');
                        var matrix = Matrix.generate(matrixDim.rows, matrixDim.cols, function (matRow, matCol) {
                            if ((matRow === 0) && (matCol === 0)) { return operand.getRawValue(); }
                            var elementOp = getOperand(descriptor, signature, operands, matRow, matCol);
                            if (elementOp.isScalar()) { return elementOp.getRawValue(); }
                            FormulaUtils.throwInternal('InterpreterInstance.resolveOperatorNode(): invalid matrix element type');
                        });
                        operand = createOperand(matrix);
                    }

                    return operand;

                } finally {
                    // remove the operand resolver from the internal stack of the formula context
                    formulaContext.popOperandResolver();
                }
            });

            // resolve the number format category of the descriptor to a parsed number format
            var parsedFormat = (descriptor.format && (contextType !== 'mat')) ? getParsedFormat(resolver, descriptor.format) : null;
            // create an operand with resulting number format if available
            return parsedFormat ? createOperand(operand, { format: parsedFormat }) : operand;
        }

        /**
         * Returns the result of the specified compiler token. If it is an
         * operand, it will be returned. If it is an operator or function call,
         * its result will be calculated and returned. If the operands of the
         * operator or functions are also operators or functions, their results
         * will be calculated recursively.
         *
         * @param {CompilerNode} tokenNode
         *  A node from a compiled token tree to be processed.
         *
         * @param {String} contextType
         *  The context type that influences how to resolve values, matrixes,
         *  and references. See methods FormulaInterpreter.interpretTokens()
         *  for details.
         *
         * @param {Dimension|Null} matrixDim
         *  The expected dimension of a matrix result. Used if the context type
         *  is 'mat'. If set to null, the size of the result matrix will be
         *  deduced from the operands of an operator or function contained in
         *  the passed token node.
         *
         * @returns {Operand}
         *  The result of the specified compiler token.
         */
        function resolveNode(tokenNode, contextType, matrixDim) {
            var operand = tokenNode.isOperator() ? resolveOperatorNode(tokenNode, contextType, matrixDim) : resolveOperandNode(tokenNode, contextType, matrixDim);
            FormulaUtils.log('< result of ' + tokenNode + ': ' + operand);
            return operand;
        }

        // public methods -----------------------------------------------------

        /**
         * Calculates the result of the formula represented by the token array
         * passed in the constructor.
         *
         * @param {String} contextType
         *  The context type that influences how to resolve values, matrixes,
         *  and references. See method FormulaInterpreter.interpretTokens()
         *  for details.
         *
         * @param {Dimension|Null} matrixDim
         *  The expected dimension of a matrix result. Used if the context type
         *  is 'mat'. If not set, the matrix size will be deduced from the
         *  operands.
         *
         * @returns {Object}
         *  A result object with the following properties:
         *  - {Any} value
         *      The result value of the interpreted formula, according to the
         *      passed context type. May be any internal error codes defined in
         *      FormulaUtils, regardless of the context type.
         *  - {ParsedFormat} [format]
         *      A parsed number format associated with the result value. The
         *      property will be omitted for generic unformatted formula
         *      results.
         *  - {String} [url]
         *      The URL of a hyperlink associated to the formula result, as
         *      provided by the operand passed to this method. The property
         *      will be omitted if no URL is available.
         *
         * @throws {FormulaError}
         *  Any fatal internal interpreter error.
         */
        this.getResult = function (contextType, matrixDim) {

            // get the result operand from the root node of the token tree
            var operand = (function () {
                try {
                    return resolveNode(rootNode, contextType, matrixDim);
                } catch (error) {
                    // operators may throw formula error codes as dedicated result, but rethrow
                    // interpreter errors (class FormulaError), and other internal JS exceptions
                    if (error instanceof ErrorCode) { return createOperand(error); }
                    throw error;
                }
            }());

            // 'any' context: resolve to actual type of the resulting operand
            contextType = (contextType === 'any') ? operand.getType() : contextType;

            // resolve to result value according to context type (internal errors will be thrown)
            var value = (function () {
                try {
                    switch (contextType) {
                        case 'val':
                            return operand.getScalar(null, null);
                        case 'mat':
                            return operand.getMatrix();
                        case 'ref':
                            return operand.getRanges();
                        default:
                            FormulaUtils.throwInternal('InterpreterInstance.getResult(): unknown context type: "' + contextType + '"');
                    }
                } catch (error) {
                    // operand may throw formula error codes as dedicated result, but rethrow
                    // interpreter errors (class FormulaError), and other internal JS exceptions
                    if (error instanceof ErrorCode) { return error; }
                    throw error;
                }
            }());

            // create the result object (optional properties will be added below)
            var result = { value: value };

            // resolve the parsed number format
            var parsedFormat = operand.getFormat();
            if (parsedFormat && !parsedFormat.isStandard()) { result.format = parsedFormat; }

            // resolve the URL of a hyperlink
            var cellUrl = operand.getURL();
            if (cellUrl !== null) { result.url = cellUrl; }

            return result;
        };

        // initialization -----------------------------------------------------

        this.registerDestructor(function () {
            docModel = rootNode = initOptions = numberFormatter = formulaContext = null;
        });

    } }); // class InterpreterInstance

    // class FormulaInterpreter ===============================================

    /**
     * Calculates the result for a compiled tree of compiler token nodes. See
     * class FormulaCompiler for details about compiled token trees.
     *
     * @constructor
     *
     * @extends ModelObject
     *
     * @param {SpreadsheetModel} docModel
     *  The spreadsheet document model containing this instance.
     */
    var FormulaInterpreter = ModelObject.extend({ constructor: function (docModel) {

        // file-format dependent options for value comparison
        var COMPARE_OPTIONS = {
            withCase: docModel.getApp().getFileFormat() === 'odf',
            nullMode: 'convert'
        };

        // base constructor ---------------------------------------------------

        ModelObject.call(this, docModel);

        /**
         * Returns an interpreter result object that represents a warning, or a
         * fatal error.
         */
        function createErrorResult(contextType, type, code, error) {
            var value = (contextType === 'mat') ? Matrix.create(1, 1, error) : error;
            FormulaUtils.withLogging(function () {
                FormulaUtils.info('result: type=' + type + ' code=' + code + ' value=' + Scalar.toString(value));
            });
            return { type: type, code: code, value: value };
        }

        // public methods -----------------------------------------------------

        /**
         * Returns whether the passed scalar values are considered to be equal
         * according to the current file format. For ODF documents, the
         * character case of strings matters.
         *
         * @param {Number|String|Boolean|Null} value1
         *  The first value to be compared.
         *
         * @param {Number|String|Boolean|Null} value2
         *  The second value to be compared.
         *
         * @returns {Boolean}
         *  Whether the passed scalar values are considered to be equal. See
         *  static function Scalar.equal() for more details.
         */
        this.equalScalars = function (value1, value2) {
            return Scalar.equal(value1, value2, COMPARE_OPTIONS);
        };

        /**
         * Compares the passed scalar values according to the current file
         * format. For ODF documents, the character case of strings matters.
         *
         * @param {Number|String|Boolean|Null} value1
         *  The first value to be compared.
         *
         * @param {Number|String|Boolean|Null} value2
         *  The second value to be compared.
         *
         * @returns {Number}
         *  A negative value, if value1 is less than value2; or a positive
         *  value, if value1 is greater than value2; or zero, if both values
         *  are of the same type and are equal. See function Scalar.compare()
         *  for more details.
         */
        this.compareScalars = function (value1, value2) {
            return Scalar.compare(value1, value2, COMPARE_OPTIONS);
        };

        /**
         * Returns an interpreter result object that represents a warning based
         * on an error code.
         *
         * @param {String} contextType
         *  The context type that influences how to create the result value. If
         *  the context type is 'mat', a 1x1 matrix with an error code as
         *  element will be created, otherwise the error code will be set as
         *  value directly. See method FormulaInterpreter.interpretTokens() for
         *  more details.
         *
         * @param {ErrorCode} errorCode
         *  The error code to be inserted into the result. Can be any regular
         *  or internal error code.
         *
         * @returns {Object}
         *  An interpreter result object, containing the property 'type' set to
         *  'warn'; the property 'code' set to the internal key of the passed
         *  error code, and the property 'value' set to the value of the error
         *  code.
         */
        this.createWarnResult = function (contextType, errorCode) {
            return createErrorResult(contextType, 'warn', errorCode.key, ('value' in errorCode) ? errorCode.value : errorCode);
        };

        /**
         * Returns an interpreter result object that represents a fatal error.
         *
         * @param {String} contextType
         *  The context type that influences how to create the result value. If
         *  the context type is 'mat', a 1x1 matrix with the error code #N/A as
         *  element will be created, otherwise the error code #N/A will be set
         *  as value directly. See method FormulaInterpreter.interpretTokens()
         *  for more details.
         *
         * @param {String} code
         *  The error code to be inserted into the result, e.g. 'missing' or
         *  'unexpected'.
         *
         * @returns {Object}
         *  An interpreter result object, containing the property 'type' set to
         *  'error'; the property 'code' set to the passed error code, and the
         *  property 'value' set to the formula error code #N/A.
         */
        this.createErrorResult = function (contextType, code) {
            return createErrorResult(contextType, 'error', code, ErrorCode.NA);
        };

        /**
         * Calculates the result of the formula represented by the passed
         * compiled token array.
         *
         * @param {String} contextType
         *  The context type that influences how to resolve values, matrixes,
         *  and references. The following context types are supported:
         *  - 'val': A single value is expected, e.g. in simple cell formulas,
         *      or operators and functions working with simple values (plus
         *      operator, ABS).
         *  - 'mat': A matrix of values is expected, e.g. in matrix formulas,
         *      or functions working on entire matrixes (MMULT).
         *  - 'ref': An unresolved cell reference is expected, e.g. reference
         *      operators (list, intersection, range), or functions calculating
         *      with range addresses instead of the cell contents (OFFSET).
         *  - 'any': Accepts any of the types mentioned above, with minimum
         *      conversion, e.g. the result of defined names, functions passing
         *      one of their original parameters through (IF, CHOOSE), or
         *      functions that iterate over available values in matrixes and
         *      cell references (SUM, PRODUCT).
         *
         * @param {CompilerNode} rootNode
         *  The root node of the compiled token tree, as returned by the method
         *  FormulaCompiler.compileTokens().
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Number} [options.refSheet]
         *      The index of the reference sheet that will be used to resolve
         *      reference tokens and defined names without explicit sheet
         *      reference. If omitted, reference tokens without sheet indexes
         *      will result in #REF! errors.
         *  @param {Address} [options.refAddress]
         *      The source reference address used to relocate reference tokens
         *      with relative column/row components. If omitted, cell A1 will
         *      be used as source reference cell.
         *  @param {Address} [options.targetAddress]
         *      The target reference address used to relocate reference tokens
         *      with relative column/row components. If omitted, cell A1 will
         *      be used as target reference cell.
         *  @param {Range} [options.matrixRange]
         *      The entire bounding range of a matrix formula. Used only if the
         *      passed context type is 'mat'. If not set, the matrix size will
         *      be deduced from the operands.
         *  @param {Boolean} [options.wrapReferences=false]
         *      If set to true, relocated ranges that are located outside the
         *      sheet will be wrapped at the sheet borders.
         *  @param {Boolean} [options.detectCircular=false]
         *      If set to true, trying to dereference a cell reference covering
         *      the target address leads to a circular dependeny error.
         *  @param {Boolean} [options.recalcDirty=false]
         *      If set to true, all dirty cell formulas referred by the
         *      reference tokens in the passed compiled token tree will be
         *      recalculated recursively using the dependency manager of the
         *      document.
         *
         * @returns {Object}
         *  The result descriptor of the formula interpreter, with the
         *  following properties:
         *  - {String} type
         *      The result type:
         *      - 'valid': A valid result has been calculated. The property
         *          'value' contains the formula result, the property 'format'
         *          may contain a number format category associated to the
         *          result value.
         *      - 'warn': The formula structure is valid, but a formula result
         *          cannnot be calculated. The formula may be inserted into the
         *          document model though. See property 'code' for details.
         *      - 'error': The formula structure is invalid. The formula
         *          expression MUST NOT be inserted into the document model.
         *          See property 'code' for details.
         *  - {String} [code]
         *      A specific code for a warning or fatal error:
         *      - 'circular': A warning that the formula results in a circular
         *          reference (the formula wants to interpret itself again
         *          during calculation, e.g. directly or indirectly via cell
         *          references, or with defined names referring each other).
         *      - 'unsupported': A warning that an unsupported feature,
         *          e.g. a function that has not been implemented yet) was
         *          found in the formula.
         *      - 'internal': A warning that an undetermined internal error has
         *          occurred while interpreting the formula tokens.
         *      - 'missing': A fatal error indicating that something is missing
         *          in the formula structure, e.g. a function was called with
         *          less arguments than required.
         *      - 'unexpected': A fatal error indicating that something in the
         *          formula structure was not expected, e.g. a function was
         *          called with too many arguments.
         *  - {Any} value
         *      The result of the formula. Will be an appropriate error code
         *      (an instance of class ErrorCode) if calculation resulted in a
         *      warning or a fatal error. If the passed context type is 'mat',
         *      such an error code will be embedded into a 1x1 matrix. See
         *      property 'type' for details.
         *  - {Number|String} [format]
         *      The identifier of a number format as integer, or a format code
         *      as string, associated with the result value. The property will
         *      be omitted for generic formula results, or for invalid result
         *      types ('warn' and 'error').
         *  - {String} [url]
         *      The URL of a hyperlink associated to the formula result. The
         *      property will be omitted for invalid result types ('warn' and
         *      'error').
         */
        this.interpretTokens = FormulaUtils.profileMethod('FormulaInterpreter.interpretTokens()', function (contextType, rootNode, options) {
            FormulaUtils.log('contextType=' + contextType);

            // the actual interpreter instance, needed for recursive interpretation of e.g. defined names
            var instance = new InterpreterInstance(docModel, rootNode, options);

            try {

                // calculate the formula reult
                var matrixRange = (contextType === 'mat') ? Utils.getOption(options, 'matrixRange', null) : null;
                var matrixDim = matrixRange ? Dimension.createFromRange(matrixRange) : null;
                var result = instance.getResult(contextType, matrixDim);

                // create a special result object for internal error codes (in catch andler below)
                if (FormulaUtils.isInternalError(result.value)) {
                    return this.createWarnResult(contextType, result.value);
                }

                // add the type marker to a valid formula result
                result.type = 'valid';

                // log result to console and return ('finally' block will clean-up the interpreter instance)
                FormulaUtils.withLogging(function () {
                    FormulaUtils.info('result: type=valid value=' + Scalar.toString(result.value) + ' format=' + result.format + ' url=' + result.url);
                });
                return result;

            } catch (error) {

                // create a special result object for internal error codes
                if (error instanceof ErrorCode) {
                    return this.createWarnResult(contextType, error);
                }

                // internal interpreter errors (invalid formula structure)
                if (error instanceof FormulaError) {
                    FormulaUtils.warn('\xa0 error: ' + error.message);
                    return this.createErrorResult(contextType, error.type);
                }

                throw error; // re-throw internal JS exceptions

            } finally {
                instance.destroy();
            }
        });

        // initialization -----------------------------------------------------

        // destroy all class members on destruction
        this.registerDestructor(function () {
            docModel = null;
        });

    } }); // class FormulaInterpreter

    // exports ================================================================

    return FormulaInterpreter;

});
