/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/spreadsheet/model/formula/funcs/statisticalfuncs', [
    'io.ox/office/tk/utils',
    'io.ox/office/spreadsheet/utils/errorcode',
    'io.ox/office/spreadsheet/model/formula/formulautils',
    'io.ox/office/spreadsheet/model/formula/utils/statutils'
], function (Utils, ErrorCode, FormulaUtils, StatUtils) {

    'use strict';

    /**************************************************************************
     *
     * This module implements all statistical spreadsheet functions.
     *
     * See the README file in this directory for a detailed documentation about
     * the format of function descriptor objects.
     *
     *************************************************************************/

    // convenience shortcuts
    var MathUtils = FormulaUtils.Math;
    var Scalar = FormulaUtils.Scalar;
    var Matrix = FormulaUtils.Matrix;

    // shortcuts to mathematical functions
    var add = MathUtils.add;
    var div = MathUtils.div;

    // standard options for numeric parameter iterators (method FormulaContext.iterateNumbers())
    var SKIP_MAT_SKIP_REF = { // id: CSS5
        valMode: 'convert', // value operands: convert strings and booleans to numbers
        matMode: 'skip', // matrix operands: skip strings and booleans
        refMode: 'skip', // reference operands: skip strings and booleans
        emptyParam: true, // empty parameters count as zero
        complexRef: true // accept multi-range and multi-sheet references
    };

    // standard options for numeric parameter iterators (method FormulaContext.iterateNumbers())
    var RCONVERT_ALL_SKIP_EMPTY = { // id: RRR0
        valMode: 'rconvert', // value operands: convert strings to numbers (but not booleans)
        matMode: 'rconvert', // matrix operands: convert strings to numbers (but not booleans)
        refMode: 'rconvert' // reference operands: convert strings to numbers (but not booleans)
        // emptyParam: false, // skip all empty parameters
        // complexRef: false // do not accept multi-range and multi-sheet references
    };

    // private global functions ===============================================

    /**
     * standard comparator can used for to sort a array of numbers
     */
    function numberComparator(a, b) { return a - b; }

    /**
     * Returns the counts of all numbers in the passed operand, that are less
     * than, equal to, and greater than the specified number.
     *
     * @param {FormulaContext} context
     *  The formula context used to iterate the numbers in the passed operand.
     *
     * @param {Number} number
     *  The number to be compared with the numbers contained in the operand.
     *
     * @param {Any} operand
     *  A function operand of any type supported by the iterator methods of the
     *  formula context.
     *
     * @param {Object} options
     *  Optional parameters passed to the number iterator used internally. See
     *  method FormulaContext.createNumberIterator() for details.
     *
     * @returns {Object}
     *  A result object with the following properties:
     *  - {Number} 'lt': The count of all numbers contained in the operand that
     *      are less than the number passed to this method.
     *  - {Number} 'eq': The count of all numbers contained in the operand that
     *      are equal to the number passed to this method.
     *  - {Number} 'gt': The count of all numbers contained in the operand that
     *      are greater than the number passed to this method.
     */
    function getCompareCounts(context, number, operand, options) {
        return context.aggregateNumbers([operand], { lt: 0, eq: 0, gt: 0 }, function (result, n) {
            switch (Scalar.compareNumbers(n, number)) {
                case -1: result.lt += 1; break;
                case 0:  result.eq += 1; break;
                case 1:  result.gt += 1; break;
            }
            return result;
        }, _.identity, options);
    }

    // Euclidean norm of row index R starting in column index C;
    // matrix A has count N columns.
    function tGetColumnEuclideanNorm(matA, nR, nC, nN) {
        var norm = 0;
        for (var col = nC; col < nN; col++) {
            var elem = matA.get(nR, col);
            norm += elem * elem;
        }
        return Math.sqrt(norm);
    }

    /* Calculates a QR decomposition with Householder reflection.
     * For each NxK matrix A exists a decomposition A=Q*R with an orthogonal
     * NxN matrix Q and a NxK matrix R.
     * Q=H1*H2*...*Hk with Householder matrices H. Such a householder matrix can
     * be build from a vector u by H=I-(2/u'u)*(u u'). This vectors u are returned
     * in the columns of matrix A, overwriting the old content.
     * The matrix R has a quadric upper part KxK with values in the upper right
     * triangle and zeros in all other elements. Here the diagonal elements of R
     * are stored in the vector R and the other upper right elements in the upper
     * right of the matrix A.
     * The function returns false, if calculation breaks. But because of round-off
     * errors singularity is often not detected.
     */
    function calculateQRdecomposition(matA, pVecR, nK, nN) {
        /* block-scoped-var */
        var row = 0;
        var col = 0;
        /* block-scoped-var */

        // ScMatrix matrices are zero based, index access (row,column)
        for (col = 0; col < nK; col++)        {
            // calculate vector u of the householder transformation
            var fScale = getColumnMaximumNorm(matA, col, col, nN);
            if (fScale === 0) {
                // A is singular
                throw ErrorCode.VALUE;
            }
            for (row = col; row < nN; row++) {
                matA.set(row, col, matA.get(row, col) / fScale, col, row);
            }
            var fEuclid = getColumnEuclideanNorm(matA, col, col, nN);
            var fFactor = 1 / fEuclid / (fEuclid + Math.abs(matA.get(col, col)));
            var fSignum = getSign(matA.get(col, col));
            matA.set(col, col, matA.get(col, col) + fSignum * fEuclid);
            pVecR[col] = -fSignum * fScale * fEuclid;

            // apply Householder transformation to A
            for (var c = col + 1; c < nK; c++) {
                var fSum = getColumnSumProduct(matA, col, matA, c, col, nN);
                for (row = col; row < nN; row++) {
                    matA.set(row, c, matA.get(row, c) - fSum * fFactor * matA.get(row, col));
                }
            }
        }
    }

    // same with transposed matrix A, N is count of columns, K count of rows
    function tCalculateQRdecomposition(matA, pVecR, nK, nN) {
        /* block-scoped-var */
        var row = 0;
        var col = 0;
        /* block-scoped-var */

        var fSum;
        // ScMatrix matrices are zero based, index access (row,column)
        for (row = 0; row < nK; row++) {
            // calculate vector u of the householder transformation
            var fScale = tGetColumnMaximumNorm(matA, row, row, nN);
            if (fScale === 0) {
                // A is singular
                throw ErrorCode.VALUE;
            }
            for (col = row; col < nN; col++) {
                matA.set(row, col, matA.get(row, col) / fScale);
            }
            var fEuclid = tGetColumnEuclideanNorm(matA, row, row, nN);
            var fFactor = 1 / fEuclid / (fEuclid + Math.abs(matA.get(row, row)));
            var fSignum = getSign(matA.get(row, row));
            matA.set(row, row, matA.get(row, row) + fSignum * fEuclid);
            pVecR[row] = -fSignum * fScale * fEuclid;

            // apply Householder transformation to A
            for (var r = row + 1; r < nK; r++) {
                fSum = tGetColumnSumProduct(matA, row, matA, r, row, nN);
                for (col = row; col < nN; col++) {
                    matA.set(r, col, matA.get(r, col) - fSum * fFactor * matA.get(row, col));
                }
            }
        }
    }

    /* Applies a Householder transformation to a column vector Y with is given as
     * Nx1 Matrix. The Vektor u, from which the Householder transformation is build,
     * is the column part in matrix A, with column index C, starting with row
     * index C. A is the result of the QR decomposition as obtained from
     * lcl_CaluclateQRdecomposition.
     */
    function applyHouseholderTransformation(matA, nC, matY, nN) {
        // ScMatrix matrices are zero based, index access (row,column)
        var fDenominator = getColumnSumProduct(matA, nC, matA, nC, nC, nN);
        var fNumerator = getColumnSumProduct(matA, nC, matY, 0, nC, nN);
        var fFactor = 2 * (fNumerator / fDenominator);
        for (var row = nC; row < nN; row++) {
            matY.setByIndex(row, matY.getByIndex(row) - fFactor * matA.get(row, nC));
        }
    }

    // Same with transposed matrices A and Y.
    function tApplyHouseholderTransformation(matA, nR, matY, nN) {
        // ScMatrix matrices are zero based, index access (row,column)
        var fDenominator = tGetColumnSumProduct(matA, nR, matA, nR, nR, nN);
        var fNumerator = tGetColumnSumProduct(matA, nR, matY, 0, nR, nN);
        var fFactor = 2 * (fNumerator / fDenominator);
        for (var col = nR; col < nN; col++) {
            matY.setByIndex(col, matY.getByIndex(col) - fFactor * matA.get(nR, col));
        }
    }

    /* Solve for X in R*X=S using back substitution. The solution X overwrites S.
     * Uses R from the result of the QR decomposition of a NxK matrix A.
     * S is a column vector given as matrix, with at least elements on index
     * 0 to K-1; elements on index>=K are ignored. Vector R must not have zero
     * elements, no check is done.
     */
    function solveWithUpperRightTriangle(matA, pVecR, matS, nK, bIsTransposed) {
        // ScMatrix matrices are zero based, index access (row,column)
        /* block-scoped-var */
        var row = 0;
        var col = 0;
        /* block-scoped-var */
        // SCSIZE is never negative, therefore test with rowp1=row+1
        for (var rowp1 = nK; rowp1 > 0; rowp1--) {
            row = rowp1 - 1;
            var fSum = matS.get(row, 0);
            for (col = rowp1; col < nK; col++) {
                if (bIsTransposed) {
                    fSum -= matA.get(col, row) * matS.getByIndex(col);
                } else {
                    fSum -= matA.get(row, col) * matS.getByIndex(col);
                }
            }
            matS.setByIndex(row, fSum / pVecR[row]);
        }
    }

    /* Solve for X in R' * X= T using forward substitution. The solution X
     * overwrites T. Uses R from the result of the QR decomposition of a NxK
     * matrix A. T is a column vectors given as matrix, with at least elements on
     * index 0 to K-1; elements on index>=K are ignored. Vector R must not have
     * zero elements, no check is done.
     */
    function solveWithLowerLeftTriangle(matA, pVecR, matT, nK, bIsTransposed) {
        // ScMatrix matrices are zero based, index access (row,column)
        for (var row = 0; row < nK; row++) {
            var fSum = matT.get(row, 0);
            for (var col = 0; col < row; col++) {
                if (bIsTransposed) {
                    fSum -= matA.get(row, col) * matT.getByIndex(col);
                } else {
                    fSum -= matA.get(col, row) * matT.getByIndex(col);
                }
            }
            matT.setByIndex(row, fSum / pVecR[row]);
        }
    }

    /* Calculates Z = R * B
     * R is given in matrix A and vector VecR as obtained from the QR
     * decomposition in lcl_CalculateQRdecomposition. B and Z are column vectors
     * given as matrix with at least index 0 to K-1; elements on index>=K are
     * not used.
     */
    function applyUpperRightTriangle(matA, pVecR, matB, matZ, nK, bIsTransposed) {
        // ScMatrix matrices are zero based, index access (row,column)
        for (var row = 0; row < nK; row++) {
            var fSum = pVecR[row] * matB.get(row, 0);
            for (var col = row + 1; col < nK; col++) {
                if (bIsTransposed) {
                    fSum += matA.get(col, row) * matB.getByIndex(col);
                } else {
                    fSum += matA.get(row, col) * matB.getByIndex(col);
                }
            }
            matZ.setByIndex(row, fSum);
        }
    }

    function getMeanOverAll(matrix) {
        return matrix.reduce(0, add) / matrix.size();
    }

    function getSumProduct(matA, matB) {
        var sum = 0;
        for (var i = 0, size = matA.size(); i < size; i += 1) {
            sum += matA.getByIndex(i) * matB.getByIndex(i);
        }
        return sum;
    }

    function getSSresid(matX, matY, slope) {
        var sum = 0;
        for (var i = 0, size = matY.size(); i < size; i += 1) {
            var temp = matX.getByIndex(i) - slope * matY.getByIndex(i);
            sum += temp * temp;
        }
        return sum;
    }

    // Calculates means of the columns of matrix X. X is a RxC matrix;
    // ResMat is a 1xC matrix (=row).
    function calculateColumnMeans(pX, resMat, nC, nR) {
        for (var i = 0; i < nC; i++) {
            var fSum = 0;
            for (var r = 0; r < nR; r++) {
                fSum += pX.get(r, i);
            }
            resMat.setByIndex(i, fSum / nR);
        }
    }

    // Calculates means of the rows of matrix X. X is a RxC matrix;
    // ResMat is a Rx1 matrix (=column).
    function calculateRowMeans(pX, resMat, nC, nR) {
        for (var i = 0; i < nR; i++) {
            var fSum = 0;
            for (var c = 0; c < nC; c++) {
                fSum += pX.get(i, c);
            }
            resMat.setByIndex(i, fSum / nC);
        }
    }

    function calculateColumnsDelta(mat, pColumnMeans, nC, nR) {
        for (var i = 0; i < nC; i++) {
            for (var r = 0; r < nR; r++) {
                mat.set(r, i, mat.get(r, i) - pColumnMeans.getByIndex(i));
            }
        }
    }

    function calculateRowsDelta(mat, pRowMeans, nC, nR) {
        for (var k = 0; k < nR; k++) {
            for (var c = 0; c < nC; c++) {
                mat.set(k, c, mat.get(k, c) - pRowMeans.getByIndex(k));
            }
        }
    }

     // <A(Ra);B(Rb)> starting in column index C;
     // Ra and Rb are indices of rows, matrices A and B have count N columns.
    function tGetColumnSumProduct(matA, nRa, matB, nRb, nC, nN) {
        var fResult = 0;
        for (var col = nC; col < nN; col++) {
            fResult += matA.get(nRa, col) * matB.get(nRb, col);
        }
        return fResult;
    }

    //Special version for use within QR decomposition.
    //<A(Ca);B(Cb)> starting in row index R;
    //Ca and Cb are indices of columns, matrices A and B have count N rows.
    function getColumnSumProduct(matA, nCa, matB, nCb, nR, nN) {
        var fResult = 0;
        for (var row = nR; row < nN; row++) {
            fResult += matA.get(row, nCa) * matB.get(row, nCb);
        }

        return fResult;
    }

    //Special version for use within QR decomposition.
    //Maximum norm of column index C starting in row index R;
    //matrix A has count N rows.
    function getColumnMaximumNorm(matA, nC, nR, nN) {
        var fNorm = 0;
        for (var row = nR; row < nN; row++) {
            fNorm = Math.max(fNorm, Math.abs(matA.get(row, nC)));
        }
        return fNorm;
    }

    //Special version for use within QR decomposition.
    //Euclidean norm of column index C starting in row index R;
    //matrix A has count N rows.
    function getColumnEuclideanNorm(matA, nC, nR, nN) {
        var fNorm = 0;
        for (var row = nR; row < nN; row++) {
            fNorm += matA.get(row, nC) * matA.get(row, nC);
        }
        return Math.sqrt(fNorm);
    }

     // no mathematical signum, but used to switch between adding and subtracting
    function getSign(number) {
        return (number >= 0) ? 1 : -1;
    }

    //Maximum norm of row index R starting in col index C;
    //matrix A has count N columns.
    function tGetColumnMaximumNorm(matA, nR, nC, nN) {
        var fNorm = 0;
        for (var col = nC; col < nN; col++) {
            fNorm = Math.max(fNorm, Math.abs(matA.get(nR, col)));
        }
        return fNorm;
    }

    // Fill default values in matrix X, transform Y to log(Y) in case LOGEST|GROWTH,
    // determine sizes of matrices X and Y, determine kind of regression, clone
    // Y in case LOGEST|GROWTH, if constant.
    //
    // call:                     CheckMatrix(     _bRKP,            nCase,         nCX,         nCY,         nRX,         nRY,         K,         N,              matX,              matY))
    //  def: bool ScInterpreter::CheckMatrix(bool _bLOG, sal_uInt8& nCase, SCSIZE& nCX, SCSIZE& nCY, SCSIZE& nRX, SCSIZE& nRY, SCSIZE& M, SCSIZE& N, ScMatrixRef& matX, ScMatrixRef& matY)
    function checkMatrix(_bLOG, matX, matY) {
        var nCase = null;
        var nCY = matY.cols();
        var nRY = matY.rows();
        var nCX = 0;
        var nRX = 0;
        var M = 0;
        var N = 0;

        var nCountY = nCY * nRY;

        matY.forEach(function (value) {
            if (!_.isNumber(value)) {
                throw ErrorCode.NA;
            }
        });

        if (_bLOG) {
            matY.forEach(function (value, row, col) {
                if (value < 0) {
                    throw ErrorCode.NA;
                }
                matY.set(row, col, Math.log(value));
            });
        }

        if (matX) {
            nCX = matX.cols();
            nRX = matX.rows();

            matX.forEach(function (value) {
                if (!_.isNumber(value)) {
                    throw ErrorCode.NA;
                }
            });

            if (nCX === nCY && nRX === nRY) {
                nCase = 1; // simple regression
                M = 1;
                N = nCountY;
            } else if (nCY !== 1 && nRY !== 1) {
                throw ErrorCode.NA;
            } else if (nCY === 1) {
                if (nRX !== nRY) { throw ErrorCode.NA; }
                nCase = 2; // Y is column
                N = nRY;
                M = nCX;
            } else if (nCX !== nCY) {
                throw ErrorCode.NA;
            } else {
                nCase = 3; // Y is row
                N = nCY;
                M = nRX;
            }
        } else {
            matX = Matrix.create(nRY, nCY, 0);
            nCX = nCY;
            nRX = nRY;

            var i = 1;
            matX.forEach(function (value, row, col) {
                matX.set(row, col, i);
                i++;
            });
            nCase = 1;
            N = nCountY;
            M = 1;
        }
        return {
            nCase: nCase,
            M: M,
            N: N,
            matX: matX,
            matY: matY
        };
    }

    function calculateRGPRKP(knownY, knownX, constant, stats, _bRKP) {
        /* block-scoped-var */
        var i = 0;
        var fIntercept = 0;
        var fSSreg = 0;
        var fDegreesFreedom = 0;
        var fSSresid = 0;
        var fFstatistic = 0;
        var fRMSE = 0;
        var fSigmaSlope = 0;
        var fSigmaIntercept = 0;
        var col = 0;
        var row = 0;
        var fR2 = 0;
        var fPart = 0;
        var aVecR = null;
        var matZ = null;
        var means = null;
        var slopes = null;
        var bIsSingular = false;
        /* block-scoped-var */

        var matY = knownY;

        var matX = null;
        if (_.isUndefined(knownX)) {
            matX = null;
        } else {
            matX = knownX;
        }

        var bConstant = null;
        if (_.isUndefined(constant)) {
            bConstant = true;
        } else {
            bConstant = constant;
        }

        var bStats = null;
        if (_.isUndefined(stats)) {
            bStats = false;
        } else {
            bStats = stats;
        }

        //CheckMatrix(_bRKP,nCase,nCX,nCY,nRX,nRY, K,N,matX,matY)
        var checked = checkMatrix(_bRKP, matX, matY);
        // 1 = simple; 2 = multiple with Y as column; 3 = multiple with Y as row
        var nCase = checked.nCase;

        var K = checked.M, N = checked.N; // K=number of variables X, N=number of data samples
        matX = checked.matX;
        matY = checked.matY;

        // Enough data samples?
        /* original code !!
           if ((bConstant && (N < K + 1)) || (!bConstant && (N < K)) || (N < 1) || (K < 1)) {
        */
        // new code to be excel conform
        if ((N < K) || (N < 1) || (K < 1)) {
            throw ErrorCode.NA;
        }

        var matWidth = bStats ? 5 : 1;
        var resMat = Matrix.create(matWidth, K + 1, 0);

        // Fill unused cells in resMat; order (row,column)
        if (bStats) {
            resMat.fill(2, 2, 4, K, ErrorCode.NA);
        }

        // Uses sum(x-MeanX)^2 and not [sum x^2]-N * MeanX^2 in case bConstant.
        // Clone constant matrices, so that Mat = Mat - Mean is possible.
        var fMeanY = 0;
        if (bConstant) {
            // DeltaY is possible here; DeltaX depends on nCase, so later
            fMeanY = getMeanOverAll(matY);
            matY.transform(function (value) { return value - fMeanY; });
        }

        if (nCase === 1) {
            // calculate simple regression
            var fMeanX = 0;
            if (bConstant) {
                // Mat = Mat - Mean
                fMeanX = getMeanOverAll(matX);
                matX.transform(function (value) { return value - fMeanX; });
            }
            var fSumXY = getSumProduct(matX, matY);
            var fSumX2 = getSumProduct(matX, matX);

            /* original code !!
            if (fSumX2 === 0) {
                throwError(ErrorCode.VALUE, 'fSumX2 === 0');
            }
            */

            var fSlope = fSumXY / fSumX2;

            // new code to be excel conform
            if (isNaN(fSlope)) { fSlope = 0; }

            fIntercept = 0;
            if (bConstant) {
                fIntercept = fMeanY - fSlope * fMeanX;
            }
            resMat.set(0, 1, _bRKP ? Math.exp(fIntercept) : fIntercept);
            resMat.set(0, 0, _bRKP ? Math.exp(fSlope) : fSlope);

            if (bStats) {
                fSSreg = fSlope * fSlope * fSumX2;
                resMat.set(4, 0, fSSreg);

                fDegreesFreedom = (bConstant) ? N - 2 : N - 1;
                resMat.set(3, 1, fDegreesFreedom);

                fSSresid  = getSSresid(matX, matY, fSlope);
                resMat.set(4, 1, fSSresid);

                if (fDegreesFreedom === 0 || fSSresid === 0 || fSSreg === 0) {
                    // exact fit; test SSreg too, because SSresid might be
                    // unequal zero due to round of errors
                    resMat.set(4, 1, 0); // SSresid
                    resMat.set(3, 0, ErrorCode.NA); // F
                    resMat.set(2, 1, 0, 1, 2); // RMSE
                    resMat.set(1, 0, 0, 0, 1); // SigmaSlope
                    if (bConstant) {
                        resMat.set(1, 1, 0); //SigmaIntercept
                    } else {
                        resMat.set(1, 1, ErrorCode.NA);
                    }
                    resMat.set(2, 0, 1); // R^2
                } else {
                    fFstatistic = (fSSreg / K) / (fSSresid / fDegreesFreedom);
                    resMat.set(3, 0, fFstatistic);

                    // standard error of estimate
                    fRMSE = Math.sqrt(fSSresid / fDegreesFreedom);
                    resMat.set(2, 1, fRMSE);

                    fSigmaSlope = fRMSE / Math.sqrt(fSumX2);
                    resMat.set(1, 0, fSigmaSlope);

                    if (bConstant) {
                        fSigmaIntercept = fRMSE * Math.sqrt(fMeanX * fMeanX / fSumX2 + 1 / (N));
                        resMat.set(1, 1, fSigmaIntercept);
                    } else {
                        resMat.set(1, 1, ErrorCode.NA);
                    }

                    fR2 = fSSreg / (fSSreg + fSSresid);
                    resMat.set(2, 0, fR2);
                }
            }
            return resMat;
        } else { // calculate multiple regression;

            // Uses a QR decomposition X = QR. The solution B = (X'X)^(-1) * X' * Y
            // becomes B = R^(-1) * Q' * Y
            if (nCase === 2) { // Y is column

                aVecR = _.times(N, _.constant(0)); // for QR decomposition
                // Enough memory for needed matrices?
                means = Matrix.create(1, K, 0); // mean of each column
                // for Q' * Y , inter alia
                if (bStats) {
                    matZ = matY.clone(); // Y is used in statistic, keep it
                } else {
                    matZ = matY; // Y can be overwritten
                }
                slopes = Matrix.create(K, 1, 0); // from b1 to bK

                if (bConstant) {
                    calculateColumnMeans(matX, means, K, N);
                    calculateColumnsDelta(matX, means, K, N);
                }
                calculateQRdecomposition(matX, aVecR, K, N);

                // Later on we will divide by elements of aVecR, so make sure
                // that they aren't zero.
                bIsSingular = false;
                for (row = 0; row < K && !bIsSingular; row++) {
                    bIsSingular = bIsSingular || aVecR[row] === 0;
                }
                if (bIsSingular) { throw ErrorCode.VALUE; }
                // Z = Q' Y;
                for (col = 0; col < K; col++) {
                    applyHouseholderTransformation(matX, col, matZ, N);
                }
                // B = R^(-1) * Q' * Y <=> B = R^(-1) * Z <=> R * B = Z
                // result Z should have zeros for index>=K; if not, ignore values
                for (i = 0; i < K; i++) {
                    slopes.setByIndex(i, matZ.getByIndex(i));
                }
                solveWithUpperRightTriangle(matX, aVecR, slopes, K, false);
                fIntercept = 0;
                if (bConstant) {
                    fIntercept = fMeanY - getSumProduct(means, slopes);
                }

                // Fill first line in result matrix
                resMat.set(0, K, _bRKP ? Math.exp(fIntercept) : fIntercept);
                for (i = 0; i < K; i++) {
                    resMat.set(0, K - 1 - i, _bRKP ? Math.exp(slopes.getByIndex(i)) : slopes.getByIndex(i));
                }

                if (bStats) {
                    fSSreg = 0;
                    fSSresid  = 0;
                    // re-use memory of Z;
                    matZ.fill(0, 0, N - 1, 0, 0);
                    // Z = R * Slopes
                    applyUpperRightTriangle(matX, aVecR, slopes, matZ, K, false);
                    // Z = Q * Z, that is Q * R * Slopes = X * Slopes
                    for (var colp1 = K; colp1 > 0; colp1--) {
                        applyHouseholderTransformation(matX, colp1 - 1, matZ, N);
                    }
                    fSSreg = getSumProduct(matZ, matZ);
                    // re-use Y for residuals, Y = Y-Z
                    for (i = 0; i < N; i++) {
                        matY.setByIndex(i, matY.getByIndex(i) - matZ.getByIndex(i));
                    }
                    fSSresid = getSumProduct(matY, matY);
                    resMat.set(4, 0, fSSreg);
                    resMat.set(4, 1, fSSresid);

                    fDegreesFreedom = ((bConstant) ? N - K - 1 : N - K);
                    resMat.set(3, 1, fDegreesFreedom);

                    if (fDegreesFreedom === 0 || fSSresid === 0 || fSSreg === 0) {
                        // exact fit; incl. observed values Y are identical
                        resMat.set(4, 1, 0); // SSresid
                        // F = (SSreg/K) / (SSresid/df) = #DIV/0!
                        resMat.set(3, 0, ErrorCode.NA); // F
                        // RMSE = Math.sqrt(SSresid / df) = Math.sqrt(0 / df) = 0
                        resMat.set(2, 1, 0); // RMSE
                        // SigmaSlope[i] = RMSE * Math.sqrt(matrix[i,i]) = 0 * Math.sqrt(...) = 0
                        for (i = 0; i < K; i++) {
                            resMat.set(1, K - 1 - i, 0);
                        }
                        // SigmaIntercept = RMSE * Math.sqrt(...) = 0
                        if (bConstant) {
                            resMat.set(1, K, 0); //SigmaIntercept
                        } else {
                            resMat.set(1, K, ErrorCode.NA);
                        }
                        //  R^2 = SSreg / (SSreg + SSresid) = 1
                        resMat.set(2, 0, 1); // R^2
                    } else {
                        fFstatistic = (fSSreg /  K) / (fSSresid / fDegreesFreedom);
                        resMat.set(3, 0, fFstatistic);

                        // standard error of estimate = root mean SSE
                        fRMSE = Math.sqrt(fSSresid / fDegreesFreedom);
                        resMat.set(2, 1, fRMSE);

                        // standard error of slopes
                        // = RMSE * Math.sqrt(diagonal element of (R' R)^(-1) )
                        // standard error of intercept
                        // = RMSE * Math.sqrt( Xmean * (R' R)^(-1) * Xmean' + 1/N)
                        // (R' R)^(-1) = R^(-1) * (R')^(-1). Do not calculate it as
                        // a whole matrix, but iterate over unit vectors.
                        fSigmaIntercept = 0;

                        for (i = 0; i < K; i++) {
                            //re-use memory of MatZ
                            matZ.fill(0, 0, K - 1, 0, 0); // Z = unit vector e
                            matZ.setByIndex(i, 1);
                            //Solve R' * Z = e
                            solveWithLowerLeftTriangle(matX, aVecR, matZ, K, false);
                            // Solve R * Znew = Zold
                            solveWithUpperRightTriangle(matX, aVecR, matZ, K, false);
                            // now Z is column col in (R' R)^(-1)
                            fSigmaSlope = fRMSE * Math.sqrt(matZ.getByIndex(i));
                            resMat.set(1, K - 1 - i, fSigmaSlope);
                            // (R' R) ^(-1) is symmetric
                            if (bConstant) {
                                fPart = getSumProduct(means, matZ);
                                fSigmaIntercept += fPart * means.getByIndex(i);
                            }
                        }
                        if (bConstant) {
                            fSigmaIntercept = fRMSE * Math.sqrt(fSigmaIntercept + 1 / N);
                            resMat.set(1, K, fSigmaIntercept);
                        } else {
                            resMat.set(1, K, ErrorCode.NA);
                        }

                        fR2 = fSSreg / (fSSreg + fSSresid);
                        resMat.set(2, 0, fR2);
                    }
                }
                return resMat;
            } else { // nCase === 3, Y is row, all matrices are transposed
                aVecR = _.times(N, _.constant(0)); // for QR decomposition
                // Enough memory for needed matrices?
                means = Matrix.create(K, 1, 0); // mean of each row
                  // for Q' * Y , inter alia
                if (bStats) {
                    matZ = matY.clone(); // Y is used in statistic, keep it
                } else {
                    matZ = matY; // Y can be overwritten
                }
                slopes =  Matrix.create(1, K, 0); // from b1 to bK
                if (bConstant) {
                    calculateRowMeans(matX, means, N, K);
                    calculateRowsDelta(matX, means, N, K);
                }

                tCalculateQRdecomposition(matX, aVecR, K, N);

                // Later on we will divide by elements of aVecR, so make sure
                // that they aren't zero.
                bIsSingular = false;
                for (row = 0; row < K && !bIsSingular; row++) {
                    bIsSingular = bIsSingular || aVecR[row] === 0;
                }
                if (bIsSingular) { throw ErrorCode.VALUE; }
                // Z = Q' Y
                for (row = 0; row < K; row++) {
                    tApplyHouseholderTransformation(matX, row, matZ, N);
                }
                // B = R^(-1) * Q' * Y <=> B = R^(-1) * Z <=> R * B = Z
                // result Z should have zeros for index>=K; if not, ignore values
                for (col = 0; col < K; col++) {
                    slopes.set(0, col, matZ.get(0, col));
                }
                solveWithUpperRightTriangle(matX, aVecR, slopes, K, true);
                fIntercept = 0;
                if (bConstant) {
                    fIntercept = fMeanY - getSumProduct(means, slopes);
                }
                // Fill first line in result matrix
                resMat.set(0, K, _bRKP ? Math.exp(fIntercept) : fIntercept);
                for (i = 0; i < K; i++) {
                    resMat.set(0, K - 1 - i, _bRKP ? Math.exp(slopes.getByIndex(i)) : slopes.getByIndex(i));
                }

                if (bStats) {
                    fSSreg = 0;
                    fSSresid  = 0;

                    matZ.fill(0, 0, 0, N - 1, 0); // re-use memory of Z;

                    // Z = R * Slopes
                    applyUpperRightTriangle(matX, aVecR, slopes, matZ, K, true);
                    // Z = Q * Z, that is Q * R * Slopes = X * Slopes
                    for (var rowp1 = K; rowp1 > 0; rowp1--) {
                        tApplyHouseholderTransformation(matX, rowp1 - 1, matZ, N);
                    }
                    fSSreg = getSumProduct(matZ, matZ);
                    // re-use Y for residuals, Y = Y-Z
                    for (i = 0; i < N; i++) {
                        matY.setByIndex(i, matY.getByIndex(i) - matZ.getByIndex(i));
                    }
                    fSSresid = getSumProduct(matY, matY);
                    resMat.set(4, 0, fSSreg);
                    resMat.set(4, 1, fSSresid);

                    fDegreesFreedom = ((bConstant) ? N - K - 1 : N - K);
                    resMat.set(3, 1, fDegreesFreedom);

                    if (fDegreesFreedom === 0 || fSSresid === 0 || fSSreg === 0) {
                        // exact fit; incl. case observed values Y are identical
                        resMat.set(4, 1, 0); // SSresid
                        // F = (SSreg/K) / (SSresid/df) = #DIV/0!
                        resMat.set(3, 0, ErrorCode.NA); // F
                        // RMSE = Math.sqrt(SSresid / df) = Math.sqrt(0 / df) = 0
                        resMat.set(2, 1, 0); // RMSE
                        // SigmaSlope[i] = RMSE * Math.sqrt(matrix[i,i]) = 0 * Math.sqrt(...) = 0
                        for (i = 0; i < K; i++) {
                            resMat.set(1, K - 1 - i, 0);
                        }

                        // SigmaIntercept = RMSE * Math.sqrt(...) = 0
                        if (bConstant) {
                            resMat.set(1, K, 0); //SigmaIntercept
                        } else {
                            resMat.set(1, K, ErrorCode.NA);
                        }
                        //  R^2 = SSreg / (SSreg + SSresid) = 1
                        resMat.set(2, 0, 1); // R^2
                    } else {
                        fFstatistic = (fSSreg /  K) / (fSSresid / fDegreesFreedom);
                        resMat.set(3, 0, fFstatistic);

                        // standard error of estimate = root mean SSE
                        fRMSE = Math.sqrt(fSSresid / fDegreesFreedom);
                        resMat.set(2, 1, fRMSE);

                        // standard error of slopes
                        // = RMSE * Math.sqrt(diagonal element of (R' R)^(-1) )
                        // standard error of intercept
                        // = RMSE * Math.sqrt( Xmean * (R' R)^(-1) * Xmean' + 1/N)
                        // (R' R)^(-1) = R^(-1) * (R')^(-1). Do not calculate it as
                        // a whole matrix, but iterate over unit vectors.
                        // (R' R) ^(-1) is symmetric
                        fSigmaIntercept = 0;

                        for (row = 0; row < K; row++) {
                            //re-use memory of MatZ
                            matZ.fill(0, 0, 0, K - 1, 0); // Z = unit vector e
                            matZ.setByIndex(row, 1);
                            //Solve R' * Z = e
                            solveWithLowerLeftTriangle(matX, aVecR, matZ, K, true);
                            // Solve R * Znew = Zold
                            solveWithUpperRightTriangle(matX, aVecR, matZ, K, true);
                            // now Z is column col in (R' R)^(-1)
                            fSigmaSlope = fRMSE * Math.sqrt(matZ.getByIndex(row));
                            resMat.set(1, K - 1 - row, fSigmaSlope);
                            if (bConstant) {
                                fPart = getSumProduct(means, matZ);
                                fSigmaIntercept += fPart * means.getByIndex(row);
                            }
                        }
                        if (bConstant) {
                            fSigmaIntercept = fRMSE * Math.sqrt(fSigmaIntercept + 1 /  N);
                            resMat.set(1, K, fSigmaIntercept);
                        } else {
                            resMat.set(1, K, ErrorCode.NA);
                        }

                        fR2 = fSSreg / (fSSreg + fSSresid);
                        resMat.set(2, 0, fR2);
                    }
                }
                return resMat;
            }
        }
    }

    // exports ================================================================

    return {

        AVEDEV: {
            category: 'statistical',
            minParams: 1,
            type: 'val:num',
            signature: 'any|mat:pass',
            resolve: function () {

                // default result (no numbers found in parameters) is the #NUM! error code (not #DIV/0!)
                function finalize(numbers, sum) {
                    if (numbers.length === 0) { throw ErrorCode.NUM; }
                    var mean = sum / numbers.length;
                    return numbers.reduce(function (result, number) { return result + Math.abs(number - mean); }, 0) / numbers.length;
                }

                // empty parameters count as zero: AVEDEV(1,) = 0.5
                return this.aggregateNumbersAsArray(arguments, finalize, SKIP_MAT_SKIP_REF);
            }
        },

        AVERAGE: {
            category: 'statistical',
            minParams: 1,
            type: 'val:num',
            signature: 'any|mat:pass',
            resolve: function () { return StatUtils.groupAverage(this, arguments); }
        },

        AVERAGEA: {
            category: 'statistical',
            minParams: 1,
            type: 'val:num',
            signature: 'any|mat:pass',
            resolve: function () { return StatUtils.groupAverage(this, arguments, { matMode: 'zero', refMode: 'zero' }); }
        },

        AVERAGEIF: {
            category: 'statistical',
            minParams: 2,
            maxParams: 3,
            type: 'val:num',
            signature: 'ref:val val:any ref:val',
            resolve: function () {
                return this.aggregateFilteredCells('single', 0, add, div);
            }
        },

        AVERAGEIFS: {
            category: 'statistical',
            minParams: 3,
            repeatParams: 2,
            type: 'val:num',
            signature: 'ref:val ref:val val:any',
            resolve: function () {
                return this.aggregateFilteredCells('multi', 0, add, div);
            }
        },

        'BETA.DIST': {
            category: 'statistical',
            name: { ooxml: '_xlfn.BETA.DIST', odf: 'COM.MICROSOFT.BETA.DIST' },
            minParams: 4,
            maxParams: 6,
            type: 'val:num',
            signature: 'val:num val:num val:num val:bool val:num val:num',
            resolve: function (x, alpha, beta, cumulative, a, b) {
                if (alpha <= 0 || beta <= 0) { throw ErrorCode.NUM; }
                if (_.isUndefined(a)) { a = 0; }
                if (_.isUndefined(b)) { b = 1; }
                var scale = b - a;
                if (this.fileFormat === 'odf') {
                    if (cumulative) {
                        if (x < a) { return 0; }
                        if (x > b) { return 1; }
                    } else if (x < a || x > b) {
                        return 0;
                    }
                    if (scale <= 0) { throw ErrorCode.NUM; }
                } else if (x < a || x > b || a === b) { throw ErrorCode.NUM; }

                x = (x - a) / scale; // convert to standard form
                return cumulative ?
                    StatUtils.betaDist(x, alpha, beta) :
                    StatUtils.betaDistPDF(x, alpha, beta) / scale;
            }
        },

        'BETA.INV': {
            category: 'statistical',
            name: { ooxml: '_xlfn.BETA.INV', odf: 'COM.MICROSOFT.BETA.INV' },
            minParams: 3,
            maxParams: 5,
            type: 'val:num',
            signature: 'val:num val:num val:num val:num val:num val:num',
            resolve: StatUtils.betaInv
        },

        BETADIST: {
            category: 'statistical',
            minParams: 3,
            maxParams: { ooxml: 5, odf: 6 },
            type: 'val:num',
            signature: 'val:num val:num val:num val:num val:num val:bool',
            resolve: function (x, alpha, beta, a, b, cumulative) {
                if (alpha <= 0 || beta <= 0) { throw ErrorCode.NUM; }
                if (_.isUndefined(a)) { a = 0; }
                if (_.isUndefined(b)) { b = 1; }
                var scale = b - a;
                if (this.fileFormat === 'odf') {
                    if (x < a) { return 0; }
                    if (x > b) { return 1; }

                    if (scale <= 0) { throw ErrorCode.NUM; }
                } else if (x < a || x > b || a === b) { throw ErrorCode.NUM; }

                x = (x - a) / scale; // convert to standard form
                return cumulative ?
                    StatUtils.betaDist(x, alpha, beta) :
                    StatUtils.betaDistPDF(x, alpha, beta) / scale;
            }
        },

        BETAINV: {
            category: 'statistical',
            minParams: 3,
            maxParams: 5,
            type: 'val:num',
            signature: 'val:num val:num val:num val:num val:num val:num',
            resolve: 'BETA.INV'
        },

        'BINOM.DIST': {
            category: 'statistical',
            name: { ooxml: '_xlfn.BINOM.DIST', odf: 'COM.MICRSOFT.BINOM.DIST' },
            minParams: 4,
            maxParams: 4,
            type: 'val:num',
            signature: 'val:num val:num val:num val:bool',
            resolve: StatUtils.getBinomDist
        },

        'BINOM.DIST.RANGE': {
            category: 'statistical',
            // AOO 4.4 writes 'B' (https://bz.apache.org/ooo/show_bug.cgi?id=126519)
            name: { ooxml: '_xlfn.BINOM.DIST.RANGE', odf: ['BINOM.DIST.RANGE', 'B'] },
            minParams: 3,
            maxParams: 4,
            type: 'val:num',
            signature: 'val:num val:num val:num val:num',
            resolve: StatUtils.binomDistRange
        },

        'BINOM.INV': {
            category: 'statistical',
            name: { ooxml: '_xlfn.BINOM.INV', odf: 'COM.MICRSOFT.BINOM.INV' },
            minParams: 3,
            maxParams: 3,
            type: 'val:num',
            signature: 'val:num val:num val:num',
            resolve: StatUtils.binomInv
        },

        BINOMDIST: {
            category: 'statistical',
            minParams: 4,
            maxParams: 4,
            type: 'val:num',
            signature: 'val:num val:num val:num val:bool',
            resolve: 'BINOM.DIST'
        },

        CHIDIST: {
            category: 'statistical',
            name: { odf: 'LEGACY.CHIDIST' },
            minParams: 2,
            maxParams: 2,
            type: 'val:num',
            signature: 'val:num val:num',
            resolve: 'CHISQ.DIST.RT'
        },

        CHIINV: {
            category: 'statistical',
            name: { odf: 'LEGACY.CHIINV' },
            minParams: 2,
            maxParams: 2,
            type: 'val:num',
            signature: 'val:num val:num',
            resolve: 'CHISQ.INV.RT'
        },

        'CHISQ.DIST': {
            category: 'statistical',
            name: { ooxml: '_xlfn.CHISQ.DIST', odf: 'COM.MICROSOFT.CHISQ.DIST' },
            minParams: 3,
            maxParams: 3,
            type: 'val:num',
            signature: 'val:num val:num val:bool',
            resolve: StatUtils.chiSqDist
        },

        'CHISQ.DIST.RT': {
            category: 'statistical',
            name: { ooxml: '_xlfn.CHISQ.DIST.RT', odf: 'COM.MICROSOFT.CHISQ.DIST.RT' },
            minParams: 2,
            maxParams: 2,
            type: 'val:num',
            signature: 'val:num val:num',
            resolve: StatUtils.getChiDist
        },

        'CHISQ.INV': {
            category: 'statistical',
            name: { ooxml: '_xlfn.CHISQ.INV', odf: 'COM.MICROSOFT.CHISQ.INV' },
            minParams: 2,
            maxParams: 2,
            type: 'val:num',
            signature: 'val:num val:num',
            resolve: StatUtils.chiSqInv
        },

        'CHISQ.INV.RT': {
            category: 'statistical',
            name: { ooxml: '_xlfn.CHISQ.INV.RT', odf: 'COM.MICROSOFT.CHISQ.INV.RT' },
            minParams: 2,
            maxParams: 2,
            type: 'val:num',
            signature: 'val:num val:num',
            resolve: StatUtils.chiInv
        },

        'CHISQ.TEST': {
            category: 'statistical',
            name: { ooxml: '_xlfn.CHISQ.TEST', odf: 'COM.MICROSOFT.CHISQ.TEST' },
            minParams: 2,
            maxParams: 2,
            type: 'val:num',
            signature: 'mat:num mat:num',
            resolve: StatUtils.chiSqTest
        },

        CHISQDIST: {
            category: 'statistical',
            name: { ooxml: null },
            minParams: 2,
            maxParams: 3,
            type: 'val:num',
            signature: 'val:num val:num val:bool',
            resolve: StatUtils.chiSqDist
        },

        CHISQINV: {
            category: 'statistical',
            name: { ooxml: null },
            minParams: 2,
            maxParams: 2,
            type: 'val:num',
            signature: 'val:num val:num',
            resolve: 'CHISQ.INV'
        },

        CHITEST: {
            category: 'statistical',
            name: { odf: 'LEGACY.CHITEST' },
            minParams: 2,
            maxParams: 2,
            type: 'val:num',
            signature: 'mat:num mat:num',
            resolve: 'CHISQ.TEST'
        },

        CONFIDENCE: {
            category: 'statistical',
            minParams: 3,
            maxParams: 3,
            type: 'val:num',
            signature: 'val:num val:num val:num',
            resolve: 'CONFIDENCE.NORM'
        },

        'CONFIDENCE.NORM': {
            category: 'statistical',
            name: { ooxml: '_xlfn.CONFIDENCE.NORM', odf: 'COM.MICROSOFT.CONFIDENCE.NORM' },
            minParams: 3,
            maxParams: 3,
            type: 'val:num',
            signature: 'val:num val:num val:num',
            resolve: StatUtils.confidence
        },

        'CONFIDENCE.T': {
            category: 'statistical',
            name: { ooxml: '_xlfn.CONFIDENCE.T', odf: 'COM.MICROSOFT.CONFIDENCE.T' },
            minParams: 3,
            maxParams: 3,
            type: 'val:num',
            signature: 'val:num val:num val:num',
            resolve: StatUtils.confidenceT
        },

        CORREL: {
            category: 'statistical',
            minParams: 2,
            maxParams: 2,
            type: 'val:num',
            signature: 'any|mat:pass any|mat:pass',
            resolve: 'PEARSON'
        },

        COUNT: {
            category: 'statistical',
            minParams: 1,
            type: 'val:num',
            signature: 'any|mat:pass',
            resolve: function () { return StatUtils.groupCount(this, arguments); }
        },

        COUNTA: {
            category: 'statistical',
            minParams: 1,
            type: 'val:num',
            signature: 'any|mat:pass',
            resolve: function () { return StatUtils.groupCountA(this, arguments); }
        },

        COUNTBLANK: {
            category: 'statistical',
            minParams: 1,
            maxParams: 1,
            type: 'val:num',
            signature: 'ref:val', // single range only (throws #VALUE! for complex references)
            resolve: function (range) {
                // empty strings (formula results) are counted as blank in Excel
                return this.countBlankCells(range, { emptyStr: this.fileFormat === 'ooxml' });
            }
        },

        COUNTIF: {
            category: 'statistical',
            minParams: 2,
            maxParams: 2,
            type: 'val:num',
            signature: 'ref:val val:any', // single range only (throws #VALUE! for complex reference)
            resolve: function () {
                return this.countFilteredCells(arguments);
            }
        },

        COUNTIFS: {
            category: 'statistical',
            minParams: 2,
            repeatParams: 2,
            type: 'val:num',
            signature: 'ref:val val:any', // single ranges only (throws #VALUE! for complex references)
            resolve: function () {
                return this.countFilteredCells(arguments);
            }
        },

        COVAR: {
            category: 'statistical',
            minParams: 2,
            maxParams: 2,
            type: 'val:num',
            signature: 'any|mat:pass any|mat:pass',
            resolve: 'COVARIANCE.P'
        },

        'COVARIANCE.P': {
            category: 'statistical',
            name: { ooxml: '_xlfn.COVARIANCE.P', odf: 'COM.MICROSOFT.COVARIANCE.P' },
            minParams: 2,
            maxParams: 2,
            type: 'val:num',
            signature: 'any|mat:pass any|mat:pass',
            resolve: function (valuesY, valuesX) {
                var data = StatUtils.getForecastData(this, valuesY, valuesX);
                return data.sumDYX / data.count;
            }
        },

        'COVARIANCE.S': {
            category: 'statistical',
            name: { ooxml: '_xlfn.COVARIANCE.S', odf: 'COM.MICROSOFT.COVARIANCE.S' },
            minParams: 2,
            maxParams: 2,
            type: 'val:num',
            signature: 'any|mat:pass any|mat:pass',
            resolve: function (valuesY, valuesX) {
                var data = StatUtils.getForecastData(this, valuesY, valuesX);
                return div(data.sumDYX, data.count - 1);
            }
        },

        CRITBINOM: {
            category: 'statistical',
            minParams: 3,
            maxParams: 3,
            type: 'val:num',
            signature: 'val:num val:num val:num',
            resolve: 'BINOM.INV'
        },

        DEVSQ: {
            category: 'statistical',
            minParams: 1,
            type: 'val:num',
            signature: 'any|mat:pass',
            resolve: function () {

                // default result (no numbers found in parameters) is the #NUM! error code (not #DIV/0!)
                function finalize(numbers, sum) {
                    // this function does not return #DIV/0! in case of missing numbers
                    if (numbers.length === 0) { throw ErrorCode.NUM; }
                    return MathUtils.devSq(numbers, sum);
                }

                // empty parameters count as zero: DEVSQ(1,) = 0.5
                return this.aggregateNumbersAsArray(arguments, finalize, SKIP_MAT_SKIP_REF);
            }
        },

        'EXPON.DIST': {
            category: 'statistical',
            name: { ooxml: '_xlfn.EXPON.DIST', odf: 'COM.MICROSOFT.EXPON.DIST' },
            minParams: 3,
            maxParams: 3,
            type: 'val:num',
            signature: 'val:num val:num val:bool',
            resolve: StatUtils.exponDist
        },

        EXPONDIST: {
            category: 'statistical',
            minParams: 3,
            maxParams: 3,
            type: 'val:num',
            signature: 'val:num val:num val:bool',
            resolve: 'EXPON.DIST'
        },

        'F.DIST': {
            category: 'statistical',
            // accept 'COM.MICROSOFT.F.DIST' in ODF (http://opengrok.libreoffice.org/xref/core/sc/source/core/tool/compiler.cxx#2628)
            name: { ooxml: '_xlfn.F.DIST', odf: ['FDIST', 'COM.MICROSOFT.F.DIST'] },
            minParams: 4,
            maxParams: 4,
            type: 'val:num',
            signature: 'val:num val:num val:num val:bool',
            resolve: StatUtils.fDistLeftTail
        },

        'F.DIST.RT': {
            category: 'statistical',
            name: { ooxml: '_xlfn.F.DIST.RT', odf: 'COM.MICROSOFT.F.DIST.RT' },
            minParams: 3,
            maxParams: 3,
            type: 'val:num',
            signature: 'val:num val:num val:num',
            resolve: StatUtils.fDistRightTail

        },

        'F.INV': {
            category: 'statistical',
            // LO 5.0 writes 'COM.MICROSOFT.F.INV' (https://bugs.documentfoundation.org/show_bug.cgi?id=94214)
            name: { ooxml: '_xlfn.F.INV', odf: ['FINV', 'COM.MICROSOFT.F.INV'] },
            minParams: 3,
            maxParams: 3,
            type: 'val:num',
            signature: 'val:num val:num val:num',
            resolve: StatUtils.fInvLeftTail
        },

        'F.INV.RT': {
            category: 'statistical',
            name: { ooxml: '_xlfn.F.INV.RT', odf: 'COM.MICROSOFT.F.INV.RT' },
            minParams: 3,
            maxParams: 3,
            type: 'val:num',
            signature: 'val:num val:num val:num',
            resolve: StatUtils.fInv
        },

        'F.TEST': {
            category: 'statistical',
            name: { ooxml: '_xlfn.F.TEST', odf: 'COM.MICROSOFT.F.TEST' },
            minParams: 2,
            maxParams: 2,
            type: 'val:num',
            signature: 'mat:num mat:num',
            resolve: StatUtils.fTest
        },

        FDIST: {
            category: 'statistical',
            name: { odf: 'LEGACY.FDIST' },
            minParams: 3,
            maxParams: 3,
            type: 'val:num',
            signature: 'val:num val:num val:num',
            resolve: 'F.DIST.RT'
        },

        FINV: {
            category: 'statistical',
            name: { odf: 'LEGACY.FINV' },
            minParams: 3,
            maxParams: 3,
            type: 'val:num',
            signature: 'val:num val:num val:num',
            resolve: 'F.INV.RT'
        },

        FISHER: {
            category: 'statistical',
            minParams: 1,
            maxParams: 1,
            type: 'val:num',
            signature: 'val:num',
            resolve: function (number) {
                if (Math.abs(number) >= 1) { throw ErrorCode.NUM; }
                return MathUtils.atanh(number);
            }
        },

        FISHERINV: {
            category: 'statistical',
            minParams: 1,
            maxParams: 1,
            type: 'val:num',
            signature: 'val:num',
            resolve: 'TANH'
        },

        FORECAST: {
            category: 'statistical',
            minParams: 3,
            maxParams: 3,
            type: 'val:num',
            signature: 'val:num any|mat:force any|mat:force',
            resolve: 'FORECAST.LINEAR'
        },

        'FORECAST.ETS': {
            category: 'statistical',
            name: { ooxml: '_xlfn.FORECAST.ETS', odf: null },
            hidden: true,
            minParams: 3,
            maxParams: 6,
            type: 'val:num'
        },

        'FORECAST.ETS.CONFINT': {
            category: 'statistical',
            name: { ooxml: '_xlfn.FORECAST.ETS.CONFINT', odf: null },
            hidden: true,
            minParams: 3,
            maxParams: 7,
            type: 'val:num'
        },

        'FORECAST.ETS.SEASONALITY': {
            category: 'statistical',
            name: { ooxml: '_xlfn.FORECAST.ETS.SEASONALITY', odf: null },
            hidden: true,
            minParams: 2,
            maxParams: 4,
            type: 'val:num'
        },

        'FORECAST.ETS.STAT': {
            category: 'statistical',
            name: { ooxml: '_xlfn.FORECAST.ETS.STAT', odf: null },
            hidden: true,
            minParams: 3,
            maxParams: 6,
            type: 'val:num'
        },

        'FORECAST.LINEAR': {
            category: 'statistical',
            name: { ooxml: '_xlfn.FORECAST.LINEAR', odf: null },
            minParams: 3,
            maxParams: 3,
            type: 'val:num',
            signature: 'val:num any|mat:force any|mat:force',
            resolve: function (x, valuesY, valuesX) {
                return StatUtils.getForecastY(this, x, valuesY, valuesX);
            }
        },

        FREQUENCY: {
            category: 'statistical',
            minParams: 2,
            maxParams: 2,
            type: 'mat',
            signature: 'any|mat:force any|mat:force',
            resolve: function (data, binsParam) {
                var ITERATETYPE = { valMode: 'skip', matMode: 'skip', refMode: 'skip', emptyParam: false, complexRef: false };

                var allNumbers = {};
                this.iterateNumbers(data, function (dataEntry) {
                    var current = allNumbers[dataEntry] || 0;
                    allNumbers[dataEntry] = current + 1;
                }, ITERATETYPE);

                var sortedNumbers = [];
                _.each(allNumbers, function (count, num) {
                    sortedNumbers.push({ count: count, num: num });
                });

                sortedNumbers = _.sortBy(sortedNumbers, 'num');

                allNumbers = null;

                var matrix = [];
                var bins = this.getNumbersAsArray(binsParam, ITERATETYPE);
                bins.push(Number.POSITIVE_INFINITY);
                var sortedBins = _.sortBy(bins);
                var lastBinEntry = Number.NEGATIVE_INFINITY;

                function collect(binEntry) {
                    var count = 0;
                    _.find(sortedNumbers, function (numInfo) {
                        var num = numInfo.num;
                        if (num > binEntry) { return true; }
                        if (num > lastBinEntry && num <= binEntry) { count += numInfo.count; }
                    });

                    matrix[bins.indexOf(binEntry)] = [count];
                    lastBinEntry = binEntry;
                }

                _.each(sortedBins, collect);

                return new Matrix(matrix);
            }
        },

        FTEST: {
            category: 'statistical',
            minParams: 2,
            maxParams: 2,
            type: 'val:num',
            signature: 'mat:num mat:num',
            resolve: 'F.TEST'
        },

        GAMMA: {
            category: 'statistical',
            name: { ooxml: '_xlfn.GAMMA' },
            minParams: 1,
            maxParams: 1,
            type: 'val:num',
            signature: 'val:num',
            resolve: function (x) {
                if (x <= 0 && x === Math.floor(x)) { throw ErrorCode.NUM; }
                return StatUtils.getGamma(x);
            }
        },

        'GAMMA.DIST': {
            category: 'statistical',
            name: { ooxml: '_xlfn.GAMMA.DIST', odf: 'COM.MICROSOFT.GAMMA.DIST' },
            minParams: 4,
            maxParams: 4,
            type: 'val:num',
            signature: 'val:num val:num val:num val:bool',
            resolve: StatUtils.gammaDist
        },

        'GAMMA.INV': {
            category: 'statistical',
            name: { ooxml: '_xlfn.GAMMA.INV', odf: 'COM.MICROSOFT.GAMMA.INV' },
            minParams: 3,
            maxParams: 3,
            type: 'val:num',
            signature: 'val:num val:num val:num',
            resolve: StatUtils.gammaInv
        },

        GAMMADIST: {
            category: 'statistical',
            minParams: 4,
            maxParams: 4,
            type: 'val:num',
            signature: 'val:num val:num val:num val:bool',
            resolve: 'GAMMA.DIST'
        },

        GAMMAINV: {
            category: 'statistical',
            minParams: 3,
            maxParams: 3,
            type: 'val:num',
            signature: 'val:num val:num val:num',
            resolve: 'GAMMA.INV'
        },

        GAMMALN: {
            category: 'statistical',
            minParams: 1,
            maxParams: 1,
            type: 'val:num',
            signature: 'val:num',
            resolve: 'GAMMALN.PRECISE'
        },

        'GAMMALN.PRECISE': {
            category: 'statistical',
            name: { ooxml: '_xlfn.GAMMALN.PRECISE', odf: 'COM.MICROSOFT.GAMMALN.PRECISE' },
            minParams: 1,
            maxParams: 1,
            type: 'val:num',
            signature: 'val:num',
            resolve: function (x) {
                if (x <= 0) { throw ErrorCode.NUM; }
                return StatUtils.getLogGamma(x);
            }
        },

        GAUSS: {
            category: 'statistical',
            name: { ooxml: '_xlfn.GAUSS' },
            minParams: 1,
            maxParams: 1,
            type: 'val:num',
            signature: 'val:num',
            resolve: function (x) {
                return StatUtils.gauss(x);
            }
        },

        GEOMEAN: {
            category: 'statistical',
            minParams: 1,
            type: 'val:num',
            signature: 'any|mat:pass',
            resolve: function () {

                // default result (no numbers found in parameters) is the #NUM! error code
                // a single 0 in the parameters results in the #NUM! error code
                function finalize(result, count) {
                    if ((result === 0) || (count === 0)) { throw ErrorCode.NUM; }
                    return Math.pow(result, 1 / count);
                }

                // empty parameters count as zero: GEOMEAN(1,) = #NUM!
                return this.aggregateNumbers(arguments, 1, MathUtils.mul, finalize, SKIP_MAT_SKIP_REF);
            }
        },

        GROWTH: {
            category: 'statistical',
            minParams: 1,
            maxParams: 4,
            type: 'mat',
            signature: 'any|mat:force any|mat:force any|mat:force val:bool',
            resolve: function (knownY, knownX, newX, constant) {
                var result = StatUtils.calculateTrendGrowth.call(this, knownY, knownX, newX, constant, true);
                return new Matrix(result);
            }
        },

        HARMEAN: {
            category: 'statistical',
            minParams: 1,
            type: 'val:num',
            signature: 'any|mat:pass',
            resolve: function () {

                // adds reciprocal of the passed number to the intermediate result
                function aggregate(result, number) {
                    // a single 0 in the parameters results in the #NUM! error code
                    if (number === 0) { throw ErrorCode.NUM; }
                    return result + 1 / number;
                }

                // default result (no numbers found in parameters) is the #N/A error code
                function finalize(result, count) {
                    if (count === 0) { throw ErrorCode.NA; }
                    return count / result;
                }

                // empty parameters count as zero: GEOMEAN(1,) = #NUM!
                return this.aggregateNumbers(arguments, 0, aggregate, finalize, SKIP_MAT_SKIP_REF);
            }
        },

        'HYPGEOM.DIST': {
            category: 'statistical',
            name: { ooxml: '_xlfn.HYPGEOM.DIST', odf: 'COM.MICROSOFT.HYPGEOM.DIST' },
            minParams: 5,
            maxParams: 5,
            type: 'val:num',
            signature: 'val:int val:int val:int val:int val:bool',
            resolve: function (x, n, M, N, isCumulative) {
                var val;

                if ((x < 0) || (n < x) || (M < x) || (N < n) || (N < M) || (x < n - N + M)) {
                    throw ErrorCode.NUM;
                }

                if (isCumulative) {
                    val = 0;
                    for (var i = 0; i <= x; i++) {
                        val += StatUtils.getHypGeomDist(i, n, M, N);
                    }
                    return val;
                } else {
                    return StatUtils.getHypGeomDist(x, n, M, N);
                }
            }
        },

        HYPGEOMDIST: {
            category: 'statistical',
            minParams: 4,
            maxParams: 4,
            type: 'val:num',
            signature: 'val:int val:int val:int val:int',
            resolve: StatUtils.getHypGeomDist
        },

        INTERCEPT: {
            category: 'statistical',
            minParams: 2,
            maxParams: 2,
            type: 'val:num',
            signature: 'any|mat:force any|mat:force',
            resolve: function (valuesY, valuesX) {
                return StatUtils.getForecastY(this, 0, valuesY, valuesX);
            }
        },

        KURT: {
            category: 'statistical',
            minParams: 1,
            type: 'val:num',
            signature: 'any|mat:pass',
            resolve: function () {

                function finalize(numbers) {
                    var sum = 0,
                        vSum = 0,
                        dx = 0,
                        xpower4 = 0,
                        stdDev,
                        mean,
                        duplicatedPart, leftPart, rightPart, // parts of equation
                        count;

                    if (!numbers.length) { throw ErrorCode.DIV0; }
                    count = numbers.length;

                    sum = numbers.reduce(function (memo, num) { return memo + num; });
                    mean = sum / count;
                    _.each(numbers, function (number) {
                        vSum += (number - mean) * (number - mean);
                    });

                    stdDev = Math.sqrt(vSum / (count - 1));
                    if (stdDev === 0) { throw ErrorCode.DIV0; }
                    _.each(numbers, function (number) {
                        dx = (number - mean) / stdDev;
                        xpower4 += Math.pow(dx, 4);
                    });

                    duplicatedPart = (count - 2) * (count - 3);
                    leftPart = count * (count + 1) / ((count - 1) * duplicatedPart);
                    rightPart = 3 * (count - 1) * (count - 1) / duplicatedPart;

                    return xpower4 * leftPart - rightPart;
                }

                return this.aggregateNumbersAsArray(arguments, finalize, SKIP_MAT_SKIP_REF);
            }
        },

        LARGE: {
            category: 'statistical',
            minParams: 2,
            maxParams: 2,
            type: 'val:num',
            signature: 'any|mat:force val:int',
            resolve: function (operand, k) {
                return StatUtils.groupLarge(this, operand, k);
            }
        },

        LINEST: {
            category: 'statistical',
            minParams: 1,
            maxParams: 4,
            type: 'mat',
            signature: 'mat mat val:bool val:bool',
            resolve: function (knownY, knownX, constant, stats) {
                return calculateRGPRKP(knownY, knownX, constant, stats, false);
            }
        },

        LOGEST: {
            category: 'statistical',
            minParams: 1,
            maxParams: 4,
            type: 'mat',
            signature: 'mat mat val:bool val:bool',
            resolve: function (knownY, knownX, constant, stats) {
                return calculateRGPRKP(knownY, knownX, constant, stats, true);
            }
        },

        LOGINV: {
            category: 'statistical',
            minParams: 3,
            maxParams: 3,
            type: 'val:num',
            signature: 'val:num val:num val:num',
            resolve: 'LOGNORM.INV'
        },

        'LOGNORM.DIST': {
            category: 'statistical',
            name: { ooxml: '_xlfn.LOGNORM.DIST', odf: 'COM.MICROSOFT.LOGNORM.DIST' },
            minParams: 4,
            maxParams: 4,
            type: 'val:num',
            signature: 'val:num val:num val:num val:bool',
            resolve: function (x, mean, sigma, isCumulative) {
                mean = mean || 0;
                sigma = _.isUndefined(sigma) ? 1 : sigma;
                isCumulative = _.isUndefined(isCumulative) ? true : isCumulative;

                if (sigma <= 0) {
                    throw ErrorCode.NUM;
                } if (isCumulative) {
                    if (x <= 0) {
                        return 0;
                    } else {
                        return (StatUtils.integralPhi((Math.log(x) - mean) / sigma));
                    }
                } else { // density
                    if (x <= 0) {
                        throw ErrorCode.NUM;
                    } else {
                        return (StatUtils.phi((Math.log(x) - mean) / sigma) / sigma / x);
                    }
                }
            }
        },

        'LOGNORM.INV': {
            category: 'statistical',
            name: { ooxml: '_xlfn.LOGNORM.INV', odf: 'COM.MICROSOFT.LOGNORM.INV' },
            minParams: 3,
            maxParams: 3,
            type: 'val:num',
            signature: 'val:num val:num val:num',
            resolve: function (p, mean, sigma) {
                if (sigma <= 0 || p <= 0 || p >= 1) {
                    throw ErrorCode.NUM;
                } else {
                    return (Math.exp(mean + sigma * StatUtils.gaussInv(p)));
                }
            }
        },

        LOGNORMDIST: {
            category: 'statistical',
            minParams: 3,
            maxParams: 3,
            type: 'val:num',
            signature: 'val:num val:num val:num',
            resolve: 'LOGNORM.DIST'
        },

        MAX: {
            category: 'statistical',
            minParams: 1,
            type: 'val:num',
            signature: 'any|mat:pass',
            resolve: function () { return StatUtils.groupMax(this, arguments); }
        },

        MAXA: {
            category: 'statistical',
            minParams: 1,
            type: 'val:num',
            signature: 'any|mat:pass',
            resolve: function () { return StatUtils.groupMax(this, arguments, { refMode: 'zero' }); }
        },

        MAXIFS: {
            category: 'statistical',
            name: { ooxml: '_xlfn.MAXIFS', odf: null },
            minParams: 3,
            repeatParams: 2,
            type: 'val:num',
            signature: 'ref:val ref:val val:any',
            resolve: function () {

                // aggregate the maximum of all numbers
                function aggregate(a, b) { return Math.max(a, b); }
                // default result (no numbers found in parameters) is zero (no #NUM! error code from infinity)
                function finalize(result) { return isFinite(result) ? result : 0; }

                return this.aggregateFilteredCells('multi', Number.NEGATIVE_INFINITY, aggregate, finalize);
            }
        },

        MEDIAN: {
            category: 'statistical',
            minParams: 1,
            type: 'val:num',
            signature: 'any|mat:pass',
            resolve: function () { return StatUtils.groupMedian(this, arguments); }
        },

        MIN: {
            category: 'statistical',
            minParams: 1,
            type: 'val:num',
            signature: 'any|mat:pass',
            resolve: function () { return StatUtils.groupMin(this, arguments); }
        },

        MINA: {
            category: 'statistical',
            minParams: 1,
            type: 'val:num',
            signature: 'any|mat:pass',
            resolve: function () { return StatUtils.groupMin(this, arguments, { refMode: 'zero' }); }
        },

        MINIFS: {
            category: 'statistical',
            name: { ooxml: '_xlfn.MINIFS', odf: null },
            minParams: 3,
            repeatParams: 2,
            type: 'val:num',
            signature: 'ref:val ref:val val:any',
            resolve: function () {

                // aggregate the minimum of all numbers
                function aggregate(a, b) { return Math.min(a, b); }
                // default result (no numbers found in parameters) is zero (no #NUM! error code from infinity)
                function finalize(result) { return isFinite(result) ? result : 0; }

                return this.aggregateFilteredCells('multi', Number.POSITIVE_INFINITY, aggregate, finalize);
            }
        },

        MODE: {
            category: 'statistical',
            minParams: 1,
            type: 'val:num',
            signature: 'any|mat:pass',
            resolve: 'MODE.SNGL'
        },

        'MODE.SNGL': {
            category: 'statistical',
            name: { ooxml: '_xlfn.MODE.SNGL', odf: 'COM.MICROSOFT.MODE.SNGL' },
            minParams: 1,
            type: 'val:num',
            signature: 'any|mat:pass',
            resolve: function () { return StatUtils.groupModeSngl(this, arguments); }
        },

        'MODE.MULT': {
            category: 'statistical',
            name: { ooxml: '_xlfn.MODE.MULT', odf: 'COM.MICROSOFT.MODE.MULT' },
            minParams: 1,
            type: 'mat',
            signature: 'any|mat:pass',
            resolve: function () { return StatUtils.groupModeMult(this, arguments); }
        },

        'NEGBINOM.DIST': {
            category: 'statistical',
            name: { ooxml: '_xlfn.NEGBINOM.DIST', odf: 'COM.MICROSOFT.NEGBINOM.DIST' },
            minParams: 4,
            maxParams: 4,
            type: 'val:num',
            signature: 'val:num val:num val:num val:bool',
            resolve: function (x, r, p, isCumulative) {
                if (r < 0 || x < 0 || p < 0 || p > 1) { throw ErrorCode.NUM; }
                if (isCumulative) {
                    return 1 - StatUtils.betaDist(1 - p, x + 1, r);
                } else {
                    return StatUtils.getNegBinomDist(x, r, p);
                }
            }
        },

        NEGBINOMDIST: {
            category: 'statistical',
            minParams: 3,
            maxParams: 3,
            type: 'val:num',
            signature: 'val:num val:num val:num',
            resolve: StatUtils.getNegBinomDist
        },

        'NORM.DIST': {
            category: 'statistical',
            name: { ooxml: '_xlfn.NORM.DIST', odf: 'COM.MICROSOFT.NORM.DIST' },
            minParams: 4,
            maxParams: 4,
            type: 'val:num',
            signature: 'val:num val:num val:num val:bool',
            resolve: function (x, mean, stDev, cumulative) {
                if (stDev <= 0) { throw ErrorCode.NUM; }
                if (cumulative) {
                    return StatUtils.integralPhi((x - mean) / stDev);
                } else {
                    return StatUtils.phi((x - mean) / stDev) / stDev;
                }
            }
        },

        'NORM.INV': {
            category: 'statistical',
            name: { ooxml: '_xlfn.NORM.INV', odf: 'COM.MICROSOFT.NORM.INV' },
            minParams: 3,
            maxParams: 3,
            type: 'val:num',
            signature: 'val:num val:num val:num',
            resolve: function (x, mean, sigma) {
                if (sigma <= 0 || x <= 0 || x >= 1) { throw ErrorCode.NUM; }
                return StatUtils.gaussInv(x) * sigma + mean;
            }
        },

        'NORM.S.DIST': {
            category: 'statistical',
            name: { ooxml: '_xlfn.NORM.S.DIST', odf: 'COM.MICROSOFT.NORM.S.DIST' },
            minParams: 2,
            maxParams: 2,
            type: 'val:num',
            signature: 'val:num val:bool',
            resolve: function (x, isCumulative) {
                if (isCumulative) { return StatUtils.integralPhi(x); }
                return Math.exp(-Math.pow(x, 2) / 2) / Math.sqrt(2 * Math.PI);
            }
        },

        'NORM.S.INV': {
            category: 'statistical',
            name: { ooxml: '_xlfn.NORM.S.INV', odf: 'COM.MICROSOFT.NORM.S.INV' },
            minParams: 1,
            maxParams: 1,
            type: 'val:num',
            signature: 'val:num',
            resolve: function (x) {
                if (x <= 0 || x >= 1) { throw ErrorCode.NUM; }
                return StatUtils.gaussInv(x);
            }
        },

        NORMDIST: {
            category: 'statistical',
            minParams: 4,
            maxParams: 4,
            type: 'val:num',
            signature: 'val:num val:num val:num val:bool',
            resolve: 'NORM.DIST'
        },

        NORMINV: {
            category: 'statistical',
            minParams: 3,
            maxParams: 3,
            type: 'val:num',
            signature: 'val:num val:num val:num',
            resolve: 'NORM.INV'
        },

        NORMSDIST: {
            category: 'statistical',
            minParams: 1,
            maxParams: 1,
            type: 'val:num',
            signature: 'val:num',
            resolve: StatUtils.integralPhi
        },

        NORMSINV: {
            category: 'statistical',
            minParams: 1,
            maxParams: 1,
            type: 'val:num',
            signature: 'val:num',
            resolve: 'NORM.S.INV'
        },

        PEARSON: {
            category: 'statistical',
            minParams: 2,
            maxParams: 2,
            type: 'val:num',
            signature: 'any|mat:force any|mat:force',
            resolve: function (valuesY, valuesX) {
                return StatUtils.getPearsonCoeff(this, valuesY, valuesX);
            }
        },

        PERCENTILE: {
            category: 'statistical',
            minParams: 2,
            maxParams: 2,
            type: 'val:num',
            signature: 'any|mat:pass val:num',
            resolve: 'PERCENTILE.INC'
        },

        'PERCENTILE.EXC': {
            category: 'statistical',
            name: { ooxml: '_xlfn.PERCENTILE.EXC', odf: 'COM.MICROSOFT.PERCENTILE.EXC' },
            minParams: 2,
            maxParams: 2,
            type: 'val:num',
            signature: 'any|mat:pass val:num',
            resolve: function (operand, p) {
                return StatUtils.groupPercentileExc(this, operand, p);
            }
        },

        'PERCENTILE.INC': {
            category: 'statistical',
            name: { ooxml: '_xlfn.PERCENTILE.INC', odf: 'COM.MICROSOFT.PERCENTILE.INC' },
            minParams: 2,
            maxParams: 2,
            type: 'val:num',
            signature: 'any|mat:pass val:num',
            resolve: function (operand, p) {
                return StatUtils.groupPercentileInc(this, operand, p);
            }
        },

        PERCENTRANK: {
            category: 'statistical',
            minParams: 2,
            maxParams: { ooxml: 3, odf: 2 },
            type: 'val:num',
            signature: 'any|mat:pass val:num val:int',
            resolve: 'PERCENTRANK.INC'
        },

        'PERCENTRANK.EXC': {
            category: 'statistical',
            name: { ooxml: '_xlfn.PERCENTRANK.EXC', odf: 'COM.MICROSOFT.PERCENTRANK.EXC' },
            minParams: 2,
            maxParams: 3,
            type: 'val:num',
            signature: 'any|mat:pass val:num val:int',
            resolve: function (operand, p, significance) {
                return StatUtils.groupPercentRankExc(this, operand, p, significance);
            }
        },

        'PERCENTRANK.INC': {
            category: 'statistical',
            name: { ooxml: '_xlfn.PERCENTRANK.INC', odf: 'COM.MICROSOFT.PERCENTRANK.INC' },
            minParams: 2,
            maxParams: 3,
            type: 'val:num',
            signature: 'any|mat:pass val:num val:int',
            resolve: function (operand, p, significance) {
                return StatUtils.groupPercentRankInc(this, operand, p, significance);
            }
        },

        PERMUT: {
            category: 'statistical',
            minParams: 2,
            maxParams: 2,
            type: 'val:num',
            signature: 'val:int val:int',
            resolve: function (number, chosen) {
                if ((chosen <= 0) || (number < chosen)) { throw ErrorCode.NUM; }
                return MathUtils.factorial(number) / MathUtils.factorial(number - chosen);
            }
        },

        PERMUTATIONA: {
            category: 'statistical',
            name: { ooxml: '_xlfn.PERMUTATIONA' },
            minParams: 2,
            maxParams: 2,
            type: 'val:num',
            signature: 'val:int val:int',
            resolve: function (base, exp) {
                if ((base < 0) || (exp < 0)) { throw ErrorCode.NUM; }
                return Math.pow(base, exp);
            }
        },

        PHI: {
            category: 'statistical',
            name: { ooxml: '_xlfn.PHI' },
            minParams: 1,
            maxParams: 1,
            type: 'val:num',
            signature: 'val:num',
            resolve: StatUtils.phi
        },

        POISSON: {
            category: 'statistical',
            minParams: 3,
            maxParams: 3,
            type: 'val:num',
            signature: 'val:num val:int val:bool',
            resolve: 'POISSON.DIST'
        },

        'POISSON.DIST': {
            category: 'statistical',
            name: { ooxml: '_xlfn.POISSON.DIST', odf: 'COM.MICROSOFT.POISSON.DIST' },
            minParams: 3,
            maxParams: 3,
            type: 'val:num',
            signature: 'val:num val:int val:bool',
            resolve: StatUtils.poisson
        },

        PROB: {
            category: 'statistical',
            minParams: 3,
            maxParams: 4,
            type: 'val:num',
            signature: 'mat:num mat:num val:num val:num',
            resolve: function (matW, matP, upper, lower) {
                var numCol1 = matP.cols(),
                    numCol2 = matW.cols(),
                    numRows1 = matP.rows(),
                    numRows2 = matW.rows(),
                    sum = 0,
                    res = 0,
                    bStop = false,
                    p, w, temp;

                if (_.isUndefined(lower)) {
                    lower = upper;
                }
                if (lower > upper) { // switch values
                    temp = lower;
                    lower = upper;
                    upper = temp;
                }
                if (numCol1 !== numCol2 || numRows1 !== numRows2 || numCol1 === 0 || numRows1 === 0 || numCol2 === 0 || numRows2 === 0) {
                    throw ErrorCode.VALUE;
                }
                for (var i = 0; i < numCol1 && !bStop; i++) {
                    for (var j = 0; j < numRows1 && !bStop; j++) {
                        p = matP.get(j, i);
                        w = matW.get(j, i);
                        if (p < 0 || p > 1) {
                            bStop = true;
                        } else {
                            sum += p;
                            if (w >= lower && w <= upper) {
                                res += p;
                            }
                        }
                    }
                }
                if (bStop || Math.abs(sum - 1) > 1E-7) {
                    throw ErrorCode.VALUE;
                }
                return res;
            }
        },

        QUARTILE: {
            category: 'statistical',
            minParams: 2,
            maxParams: 2,
            type: 'val:num',
            signature: 'any|mat:pass val:int',
            resolve: 'QUARTILE.INC'
        },

        'QUARTILE.EXC': {
            category: 'statistical',
            name: { ooxml: '_xlfn.QUARTILE.EXC', odf: 'COM.MICROSOFT.QUARTILE.EXC' },
            minParams: 2,
            maxParams: 2,
            type: 'val:num',
            signature: 'any|mat:pass val:int',
            resolve: function (operand, q) {
                return StatUtils.groupQuartileExc(this, operand, q);
            }
        },

        'QUARTILE.INC': {
            category: 'statistical',
            name: { ooxml: '_xlfn.QUARTILE.INC', odf: 'COM.MICROSOFT.QUARTILE.INC' },
            minParams: 2,
            maxParams: 2,
            type: 'val:num',
            signature: 'any|mat:pass val:int',
            resolve: function (operand, q) {
                return StatUtils.groupQuartileInc(this, operand, q);
            }
        },

        RANK: {
            category: 'statistical',
            minParams: 2,
            maxParams: 3,
            type: 'val:num',
            signature: 'val:num ref val:bool',
            resolve: 'RANK.EQ'
        },

        'RANK.AVG': {
            category: 'statistical',
            name: { ooxml: '_xlfn.RANK.AVG', odf: 'COM.MICROSOFT.RANK.AVG' },
            minParams: 2,
            maxParams: 3,
            type: 'val:num',
            signature: 'val:num ref val:bool',
            resolve: function (number, ref, ascending) {

                // the count of numbers in the source data less than, equal to, or greater than the passed number
                var counts = getCompareCounts(this, number, ref, SKIP_MAT_SKIP_REF);

                // throw #N/A error code, if the passed number does not exist in the source data
                if (counts.eq === 0) { throw ErrorCode.NA; }

                // the average rank of the number is the average of the individual ranks of each found number
                return (ascending ? counts.lt : counts.gt) + (counts.eq + 1) / 2;
            }
        },

        'RANK.EQ': {
            category: 'statistical',
            name: { ooxml: '_xlfn.RANK.EQ', odf: 'COM.MICROSOFT.RANK.EQ' },
            minParams: 2,
            maxParams: 3,
            type: 'val:num',
            signature: 'val:num ref val:bool',
            resolve: function (number, ref, ascending) {

                // the count of numbers in the source data less than, equal to, or greater than the passed number
                var counts = getCompareCounts(this, number, ref, SKIP_MAT_SKIP_REF);

                // throw #N/A error code, if the passed number does not exist in the source data
                if (counts.eq === 0) { throw ErrorCode.NA; }

                // the rank of the number is the count of predecessors (reverse mode: successors) for the number,
                // plus 1 (it counts itself once, regardless how often it exists in the source data)
                return (ascending ? counts.lt : counts.gt) + 1;
            }
        },

        RSQ: {
            category: 'statistical',
            minParams: 2,
            maxParams: 2,
            type: 'val:num',
            signature: 'any|mat:force any|mat:force',
            resolve: function (valuesY, valuesX) {
                var pearson = StatUtils.getPearsonCoeff(this, valuesY, valuesX);
                return pearson * pearson;
            }
        },

        SKEW: {
            category: 'statistical',
            minParams: 1,
            type: 'val:num',
            signature: 'any|mat:pass',
            resolve: function () {

                function finalize(numbers, sum) {
                    return StatUtils.calculateSkews(numbers, sum, false);
                }

                return this.aggregateNumbersAsArray(arguments, finalize, SKIP_MAT_SKIP_REF);
            }
        },

        'SKEW.P': {
            category: 'statistical',
            name: { ooxml: '_xlfn.SKEW.P', odf: 'SKEWP' },
            minParams: 1,
            type: 'val:num',
            signature: 'any|mat:pass',
            resolve: function () {

                function finalize(numbers, sum) {
                    return StatUtils.calculateSkews(numbers, sum, true);
                }

                return this.aggregateNumbersAsArray(arguments, finalize, SKIP_MAT_SKIP_REF);
            }
        },

        SLOPE: {
            category: 'statistical',
            minParams: 2,
            maxParams: 2,
            type: 'val:num',
            signature: 'any|mat:force any|mat:force',
            resolve: function (valuesY, valuesX) {
                var data = StatUtils.getForecastData(this, valuesY, valuesX);
                return div(data.sumDYX, data.sumDX2);
            }
        },

        SMALL: {
            category: 'statistical',
            minParams: 2,
            maxParams: 2,
            type: 'val:num',
            signature: 'any|mat:force val:int',
            resolve: function (operand, k) {
                return StatUtils.groupSmall(this, operand, k);
            }
        },

        STANDARDIZE: {
            category: 'statistical',
            minParams: 3,
            maxParams: 3,
            type: 'val:num',
            signature: 'val:num val:num val:num',
            resolve: function (x, mean, sigma) {
                if (sigma <= 0) { throw ErrorCode.NUM; }
                return (x - mean) / sigma;
            }
        },

        STDEV: {
            category: 'statistical',
            minParams: 1,
            type: 'val:num',
            signature: 'any|mat:pass',
            resolve: 'STDEV.S'
        },

        'STDEV.P': {
            category: 'statistical',
            name: { ooxml: '_xlfn.STDEV.P', odf: 'COM.MICROSOFT.STDEV.P' },
            minParams: 1,
            type: 'val:num',
            signature: 'any|mat:pass',
            resolve: function () { return StatUtils.groupStdDevP(this, arguments); }
        },

        'STDEV.S': {
            category: 'statistical',
            name: { ooxml: '_xlfn.STDEV.S', odf: 'COM.MICROSOFT.STDEV.S' },
            minParams: 1,
            type: 'val:num',
            signature: 'any|mat:pass',
            resolve: function () { return StatUtils.groupStdDevS(this, arguments); }
        },

        STDEVA: {
            category: 'statistical',
            minParams: 1,
            type: 'val:num',
            signature: 'any|mat:pass',
            resolve: function () { return StatUtils.groupStdDevS(this, arguments, { refMode: 'zero' }); }
        },

        STDEVP: {
            category: 'statistical',
            minParams: 1,
            type: 'val:num',
            signature: 'any|mat:pass',
            resolve: 'STDEV.P'
        },

        STDEVPA: {
            category: 'statistical',
            minParams: 1,
            type: 'val:num',
            signature: 'any|mat:pass',
            resolve: function () { return StatUtils.groupStdDevP(this, arguments, { refMode: 'zero' }); }
        },

        STEYX: {
            category: 'statistical',
            minParams: 2,
            maxParams: 2,
            type: 'val:num',
            signature: 'any|mat:force any|mat:force',
            resolve: function (valuesY, valuesX) {
                var data = StatUtils.getForecastData(this, valuesY, valuesX);
                var diff = data.sumDY2 - div(data.sumDYX * data.sumDYX, data.sumDX2);
                return Math.sqrt(div(diff, Math.max(0, data.count - 2)));
            }
        },

        'T.DIST': {
            category: 'statistical',
            name: { ooxml: '_xlfn.T.DIST', odf: 'COM.MICROSOFT.T.DIST' },
            minParams: 3,
            maxParams: 3,
            type: 'val:num',
            signature: 'val:num val:int val:bool',
            resolve: function (T, DF, isCumulative) {
                if (DF < 1) { throw ErrorCode.NUM; }
                return StatUtils.getTDist(T, DF, isCumulative ? 4 : 3);
            }
        },

        'T.DIST.2T': {
            category: 'statistical',
            name: { ooxml: '_xlfn.T.DIST.2T', odf: 'COM.MICROSOFT.T.DIST.2T' },
            minParams: 2,
            maxParams: 2,
            type: 'val:num',
            signature: 'val:num val:int',
            resolve: function (T, DF) {
                if (DF < 1 || T < 0) { throw ErrorCode.NUM; }
                return StatUtils.getTDist(T, DF, 2);
            }
        },

        'T.DIST.RT': {
            category: 'statistical',
            name: { ooxml: '_xlfn.T.DIST.RT', odf: 'COM.MICROSOFT.T.DIST.RT' },
            minParams: 2,
            maxParams: 2,
            type: 'val:num',
            signature: 'val:num val:int',
            resolve: function (T, DF) {
                if (DF < 1) { throw ErrorCode.NUM; }
                return StatUtils.getTDist(T, DF, 1);
            }
        },

        'T.INV': {
            category: 'statistical',
            name: { ooxml: '_xlfn.T.INV', odf: 'COM.MICROSOFT.T.INV' },
            minParams: 2,
            maxParams: 2,
            type: 'val:num',
            signature: 'val:num val:int',
            resolve: function (p, DF) {
                if (DF < 1 || p <= 0 || p > 1) { throw ErrorCode.NUM; }
                return (p < 0.5) ? -StatUtils.getTInv(1 - p, DF, 4) : StatUtils.getTInv(p, DF, 4);
            }
        },

        'T.INV.2T': {
            category: 'statistical',
            name: { ooxml: '_xlfn.T.INV.2T', odf: 'COM.MICROSOFT.T.INV.2T' },
            minParams: 2,
            maxParams: 2,
            type: 'val:num',
            signature: 'val:num val:int',
            resolve: function (p, DF) {
                if (DF < 1 || p <= 0 || p > 1) { throw ErrorCode.NUM; }
                return (p < 0.5) ? -StatUtils.getTInv(1 - p, DF, 2) : StatUtils.getTInv(p, DF, 2);
            }
        },

        'T.TEST': {
            category: 'statistical',
            name: { ooxml: '_xlfn.T.TEST', odf: 'COM.MICROSOFT.T.TEST' },
            minParams: 4,
            maxParams: 4,
            type: 'val:num',
            signature: 'any|mat:pass any|mat:pass val:int val:int',
            resolve: function (data1, data2, tails, type) {
                var count1;
                var count2;
                var S1 = 0;
                var S2 = 0;
                var sum1 = 0;
                var sum2 = 0;
                var sumSqr1 = 0;
                var sumSqr2 = 0;
                var t;
                var f;
                // standard options for numeric parameter iterators (method FormulaContext.iterateNumbers())
                var TTESTPARAM = { // id: ESS
                    valMode: 'exact', // value operands: must be numbers
                    matMode: 'skip', // matrix operands: skip strings and booleans
                    refMode: 'skip', // reference operands: skip strings and booleans
                    emptyParam: true // empty parameters count as zero
                };

                function finalize(result) {
                    var sum1 = result[0];
                    var sum2 = result[1];
                    var sumSqrD = result[2];
                    var count = result[3];
                    var diff = Math.abs(sum1 - sum2);

                    t = Math.sqrt(count - 1) * diff / Math.sqrt(count * sumSqrD - diff * diff);
                    f = count - 1;
                }

                function agregate(result, num) {
                    result[0] += num;
                    result[1] += num * num;
                    return result;
                }

                function finalize2(result, count) {
                    sum1 = result[0];
                    sumSqr1 = result[1];
                    count1  = count;
                }

                function finalize3(result, count) {
                    sum2 = result[0];
                    sumSqr2 = result[1];
                    count2  = count;
                }

                if (tails !== 1 && tails !== 2) {
                    throw ErrorCode.NUM;
                }
                if (type === 1) {
                    this.aggregateNumbersParallel([data1, data2], [0, 0, 0, 0], function (result, nums) {
                        result[0] += nums[0];
                        result[1] += nums[1];
                        result[2] += (nums[0] - nums[1]) * (nums[0] - nums[1]);
                        result[3] += 1;

                        return result;
                    }, finalize, TTESTPARAM);
                } else {
                    if (type !== 2 && type !== 3) {
                        throw ErrorCode.NUM;
                    }

                    this.aggregateNumbers([data1], [0, 0], agregate, finalize2, TTESTPARAM);
                    this.aggregateNumbers([data2], [0, 0], agregate, finalize3, TTESTPARAM);

                    if (type === 2) {
                        S1 = (sumSqr1 - sum1 * sum1 / count1) / (count1 - 1);
                        S2 = (sumSqr2 - sum2 * sum2 / count2) / (count2 - 1);
                        t = Math.abs(sum1 / count1 - sum2 / count2) /
                            Math.sqrt((count1 - 1) * S1 + (count2 - 1) * S2) *
                            Math.sqrt(count1 * count2 * (count1 + count2 - 2) / (count1 + count2));
                        f = count1 + count2 - 2;
                    } else {
                        S1 = (sumSqr1 - sum1 * sum1 / count1) / (count1 - 1) / count1;
                        S2 = (sumSqr2 - sum2 * sum2 / count2) / (count2 - 1) / count2;
                        if (S1 + S2 === 0) {
                            throw ErrorCode.VALUE;
                        }
                        t = Math.abs(sum1 / count1 - sum2 / count2) / Math.sqrt(S1 + S2);
                        var c = S1 / (S1 + S2);
                        f = 1 / (c * c / (count1 - 1) + (1 - c) * (1 - c) / (count2 - 1));
                    }
                }
                return StatUtils.getTDist(t, f, tails);
            }
        },

        TDIST: {
            category: 'statistical',
            // AOO 4.4 writes 'TDIST' (https://bz.apache.org/ooo/show_bug.cgi?id=126519)
            name: { odf: ['LEGACY.TDIST', 'TDIST'] },
            minParams: 3,
            maxParams: 3,
            type: 'val:num',
            signature: 'val:num val:int val:int',
            resolve: function (T, DF, tails) {
                if (DF < 1 || T < 0) { throw ErrorCode.NUM; }
                switch (tails) {
                    case 1: return StatUtils.getTDist(T, DF, 1);
                    case 2: StatUtils.getTDist(T, DF, 2);
                }
                throw ErrorCode.NUM;
            }
        },

        TINV: {
            category: 'statistical',
            minParams: 2,
            maxParams: 2,
            type: 'val:num',
            signature: 'val:num val:int',
            resolve: 'T.INV.2T'
        },

        TREND: {
            category: 'statistical',
            minParams: 1,
            maxParams: 4,
            type: 'mat',
            signature: 'any|mat:force any|mat:force any|mat:force val:bool',
            resolve: function (knownY, knownX, newX, constant) {
                return new Matrix(StatUtils.calculateTrendGrowth.call(this, knownY, knownX, newX, constant, false));
            }
        },

        TRIMMEAN: {
            category: 'statistical',
            minParams: 2,
            maxParams: 2,
            type: 'val:num',
            signature: 'any|mat:pass val:num',
            resolve: function (set, alpha) {
                var numbers, count, numIndex, sum = 0;

                if (alpha < 0 || alpha >= 1) {
                    throw ErrorCode.NUM;
                }

                numbers = this.getNumbersAsArray(set, RCONVERT_ALL_SKIP_EMPTY);
                count = numbers.length;
                numbers.sort(numberComparator);

                if (!_.isArray(numbers)  || _.isEmpty(numbers) || count === 0) {
                    throw ErrorCode.VALUE;
                } else {
                    numIndex = Math.floor(alpha * count);

                    if (numIndex % 2 !== 0) {
                        numIndex--;
                    }
                    numIndex /= 2;

                    if (numIndex > count) {
                        Utils.error('statisticalfuncs.js TRIMMEAN: wrong index!');
                    }

                    for (var i = numIndex; i < count - numIndex; i++) {
                        sum += numbers[i];
                    }

                    return (sum / (count - 2 * numIndex));
                }
            }
        },

        TTEST: {
            category: 'statistical',
            minParams: 4,
            maxParams: 4,
            type: 'val:num',
            signature: 'any|mat:pass any|mat:pass val:int val:int',
            resolve: 'T.TEST'
        },

        VAR: {
            category: 'statistical',
            minParams: 1,
            type: 'val:num',
            signature: 'any|mat:pass',
            resolve: 'VAR.S'
        },

        'VAR.P': {
            category: 'statistical',
            name: { ooxml: '_xlfn.VAR.P', odf: 'COM.MICROSOFT.VAR.P' },
            minParams: 1,
            type: 'val:num',
            signature: 'any|mat:pass',
            resolve: function () { return StatUtils.groupVarP(this, arguments); }
        },

        'VAR.S': {
            category: 'statistical',
            name: { ooxml: '_xlfn.VAR.S', odf: 'COM.MICROSOFT.VAR.S' },
            minParams: 1,
            type: 'val:num',
            signature: 'any|mat:pass',
            resolve: function () { return StatUtils.groupVarS(this, arguments); }
        },

        VARA: {
            category: 'statistical',
            minParams: 1,
            type: 'val:num',
            signature: 'any|mat:pass',
            resolve: function () { return StatUtils.groupVarS(this, arguments, { refMode: 'zero' }); }
        },

        VARP: {
            category: 'statistical',
            minParams: 1,
            type: 'val:num',
            signature: 'any|mat:pass',
            resolve: 'VAR.P'
        },

        VARPA: {
            category: 'statistical',
            minParams: 1,
            type: 'val:num',
            signature: 'any|mat:pass',
            resolve: function () { return StatUtils.groupVarP(this, arguments, { refMode: 'zero' }); }
        },

        WEIBULL: {
            category: 'statistical',
            minParams: 4,
            maxParams: 4,
            type: 'val:num',
            signature: 'val:num val:num val:num',
            resolve: 'WEIBULL.DIST'
        },

        'WEIBULL.DIST': {
            category: 'statistical',
            name: { ooxml: '_xlfn.WEIBULL.DIST', odf: 'COM.MICROSOFT.WEIBULL.DIST' },
            minParams: 4,
            maxParams: 4,
            type: 'val:num',
            signature: 'val:num val:num val:num val:bool',
            resolve: function (x, alpha, beta, cumulative) {
                if (alpha <= 0 || beta <= 0 || x < 0) {
                    throw ErrorCode.NUM;
                } else if (!cumulative) {
                    return (alpha / Math.pow(beta, alpha) * Math.pow(x, alpha - 1) * Math.exp(-Math.pow(x / beta, alpha)));
                } else {
                    return (1 - Math.exp(-Math.pow(x / beta, alpha)));
                }
            }
        },

        'Z.TEST': {
            category: 'statistical',
            name: { ooxml: '_xlfn.Z.TEST', odf: 'COM.MICROSOFT.Z.TEST' },
            minParams: 2,
            maxParams: 3,
            type: 'val:num',
            signature: 'any|mat:pass val:num val:num',
            resolve: function (mat, x, sigma) {
                var numbers,
                    sum = 0,
                    sumSqr = 0,
                    count, m;
                if (!_.isUndefined(sigma) && sigma <= 0) {
                    throw ErrorCode.NUM;
                }
                numbers = this.getNumbersAsArray(mat, RCONVERT_ALL_SKIP_EMPTY);
                count = numbers.length;
                if (count <= 1) {
                    throw ErrorCode.DIV0;
                }
                _.each(numbers, function (number) {
                    sum += number;
                    sumSqr += number * number;
                });
                m = sum / count;

                if (_.isUndefined(sigma)) {
                    sigma = (sumSqr - sum * sum / count) / (count - 1);
                    return (0.5 - StatUtils.gauss((m - x) / Math.sqrt(sigma / count)));
                } else {
                    return (0.5 - StatUtils.gauss((m - x) * Math.sqrt(count) / sigma));
                }
            }
        },

        ZTEST: {
            category: 'statistical',
            minParams: 2,
            maxParams: 3,
            type: 'val:num',
            signature: 'any|mat:pass val:num val:num',
            resolve: 'Z.TEST'
        }
    };

});
