/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/spreadsheet/model/formula/funcs/matrixfuncs', [
    'io.ox/office/spreadsheet/utils/errorcode',
    'io.ox/office/spreadsheet/model/formula/formulautils'
], function (ErrorCode, FormulaUtils) {

    'use strict';

    /**************************************************************************
     *
     * This module implements all spreadsheet functions dealing with matrixes.
     *
     * See the README file in this directory for a detailed documentation about
     * the format of function descriptor objects.
     *
     * In spreadsheet documents, matrixes can be given as literals in formulas,
     * or can be extracted from rectangular cell ranges. The client-side
     * formula engine provides the function signature type 'mat' (and its sub
     * types such as 'mat:num'), and the class Matrix whose instances contain
     * the elements of a matrix in a two-dimensional array.
     *
     *************************************************************************/

    // convenience shortcuts
    var MathUtils = FormulaUtils.Math;
    var Matrix = FormulaUtils.Matrix;

    // aggregation options for the SUMPRODUCT function
    var SUMPRODUCT_OPTIONS = {
        valMode: 'exact', // value operands: must be a number
        matMode: 'skip', // matrix operands: skip strings and booleans
        refMode: 'skip', // reference operands: skip strings and booleans
        emptyParam: true, // empty parameter will lead to #VALUE! (due to 'exact' mode)
        // complexRef: false, // do not accept multi-range and multi-sheet references
        checkSize: 'exact' // operands must have equal width and height
    };

    // aggregation options for other matrix sum functions (SUMX2MY2 etc.)
    var SUM_XY_OPTIONS = {
        valMode: 'exact', // value operands: must be numbers
        matMode: 'skip', // matrix operands: skip strings and booleans
        refMode: 'skip', // reference operands: skip strings and booleans
        emptyParam: true, // empty parameter will lead to #VALUE! (due to 'exact' mode)
        // complexRef: false, // do not accept multi-range and multi-sheet references
        checkSize: 'size', // operands must have same element count, but ignore width/height (e.g. 2x2 and 1x4)
        sizeMismatchError: ErrorCode.NA // throw #N/A error code if the sizes do not match
    };

    // private global functions ===============================================

    /**
     * Throws an error code, if the passed matrix is not a square matrix (i.e.,
     * its width and height are different).
     *
     * @param {Matrix} matrix
     *  The matrix to be checked.
     *
     * @returns {Number}
     *  The size of a square matrix.
     *
     * @throws {ErrorCode}
     *  The #VALUE! error code, if the passed matrix is not a square matrix.
     */
    function ensureSquareMatrix(matrix) {
        var dim = matrix.cols();
        if (dim !== matrix.rows()) { throw ErrorCode.VALUE; }
        return dim;
    }

    // exports ================================================================

    return {

        MDETERM: {
            category: 'matrix',
            minParams: 1,
            maxParams: 1,
            type: 'val:num',
            signature: 'mat:num',
            resolve: function (matrix) {

                // only square matrixes have a determinant
                var dim = ensureSquareMatrix(matrix);

                // shortcut: use compact formulas for small matrixes
                var vals = matrix.values();
                switch (dim) {
                    case 1: return vals[0][0];
                    case 2: return vals[0][0] * vals[1][1] - vals[1][0] * vals[0][1];
                    case 3: return vals[0][0] * vals[1][1] * vals[2][2] + vals[0][1] * vals[1][2] * vals[2][0] + vals[0][2] * vals[1][0] * vals[2][1]
                                - vals[2][0] * vals[1][1] * vals[0][2] - vals[2][1] * vals[1][2] * vals[0][0] - vals[2][2] * vals[1][0] * vals[0][1];
                }

                // clone the matrix for in-place LU decomposition
                var luMat = matrix.clone();
                var luVals = luMat.values();

                // the sign of the determinant changes with every row-swap
                var sign = 1;

                // in-place LU decomposition: matrix L and U will be stored in the same matrix,
                // the diagonal elements of L (all one) will not be stored
                for (var i = 0, row = 0; i < dim; i += 1) {

                    // bug 48482: escape if algorithm runs too long
                    this.checkEvalTime();

                    // find the first row with non-zero element in column 'i' (skip leading rows)
                    for (row = i; row < dim; row += 1) {
                        if (luVals[row][i] !== 0) { break; }
                    }

                    // if column 'i' contains zeros only, the matrix is not invertible (determinant is zero)
                    if (row === dim) { return 0; }

                    // swap rows to move the row with non-zero column element to position 'i'
                    if (i !== row) {
                        luMat.swapRows(i, row);
                        sign = -sign;
                    }

                    // the diagonal element
                    var diag = luVals[i][i];

                    // update the remaining (following) rows in the matrix
                    for (row = i + 1; row < dim; row += 1) {
                        var rowVec = luVals[row];
                        // calculate the elements of the L matrix
                        rowVec[i] /= diag;
                        // calculate the elements of the U matrix
                        var factor = -rowVec[i];
                        for (var col = i + 1; col < dim; col += 1) {
                            rowVec[col] += factor * luVals[i][col];
                        }
                    }
                }

                // multiply the resulting diagonal elements of the U matrix
                return luVals.reduce(function (det, rowVec, row) {
                    return det * rowVec[row];
                }, sign);
            }
        },

        MINVERSE: {
            category: 'matrix',
            minParams: 1,
            maxParams: 1,
            type: 'mat',
            signature: 'mat:num',
            resolve: function (matrix) {

                // only square matrixes are invertible
                var dim = ensureSquareMatrix(matrix);

                // left: original matrix, will be transformed into identity matrix
                var lMat = matrix.clone();
                var lVals = lMat.values();

                // right: identity matrix, will be transformed into inverse matrix
                var rMat = Matrix.identity(dim);

                // process all rows in both matrixes
                for (var i = 0, row = 0; i < dim; i += 1) {

                    // bug 48482: escape if algorithm runs too long
                    this.checkEvalTime();

                    // find the first row with non-zero element in column 'i' (skip leading rows)
                    for (row = i; row < dim; row += 1) {
                        if (lVals[row][i] !== 0) { break; }
                    }

                    // if column 'i' contains zeros only, the matrix is not invertible
                    if (row === dim) { throw ErrorCode.NUM; }

                    // swap rows to move the row with non-zero column element to position 'i'
                    if (i !== row) {
                        lMat.swapRows(i, row);
                        rMat.swapRows(i, row);
                    }

                    // scale the row to have the number 1 on the diagonal of the original matrix
                    var factor = 1 / lVals[i][i];
                    lMat.scaleRow(i, factor);
                    rMat.scaleRow(i, factor);

                    // subtract row 'i' from all other rows (scaled accordingly) to convert
                    // all elements of column 'i' to zero (skip row 'i' of course)
                    for (row = 0; row < dim; row += 1) {
                        if (row !== i) {
                            factor = -lVals[row][i];
                            lMat.addScaledRow(row, i, factor);
                            rMat.addScaledRow(row, i, factor);
                        }
                    }
                }

                // this is the inverse matrix now
                return rMat;
            }
        },

        MMULT: {
            category: 'matrix',
            minParams: 2,
            maxParams: 2,
            type: 'mat',
            signature: 'mat:num mat:num',
            resolve: function (matrix1, matrix2) {

                // width of first matrix must match height of second matrix
                if (matrix1.cols() !== matrix2.rows()) { throw ErrorCode.VALUE; }

                // calculate all elements of the result matrix
                return Matrix.generate(matrix1.rows(), matrix2.cols(), function (row, col) {
                    var elem = 0;
                    for (var i = 0, l = matrix1.cols(); i < l; i += 1) {
                        elem += (matrix1.get(row, i) * matrix2.get(i, col));
                    }
                    return elem;
                });
            }
        },

        MUNIT: {
            category: 'matrix',
            name: { ooxml: '_xlfn.MUNIT' },
            minParams: 1,
            maxParams: 1,
            type: 'mat',
            signature: 'val:int',
            resolve: function (size) {
                if (size < 1) { throw ErrorCode.VALUE; }
                FormulaUtils.ensureMatrixDim(size, size);
                return Matrix.identity(size);
            }
        },

        SUMPRODUCT: {
            category: 'matrix math',
            minParams: 1,
            type: 'val:num',
            signature: 'any|mat:force', // bug 47280: forced matrix context also in cell formulas
            resolve: function () {
                function aggregate(result, numbers) { return result + numbers.reduce(MathUtils.mul, 1); }
                return this.aggregateNumbersParallel(arguments, 0, aggregate, _.identity, SUMPRODUCT_OPTIONS);
            }
        },

        SUMX2MY2: {
            category: 'matrix math',
            minParams: 2,
            maxParams: 2,
            type: 'val:num',
            signature: 'any|mat:force any|mat:force',
            resolve: function () {
                function aggregate(result, nums) { return result + nums[0] * nums[0] - nums[1] * nums[1]; }
                return this.aggregateNumbersParallel(arguments, 0, aggregate, _.identity, SUM_XY_OPTIONS);
            }
        },

        SUMX2PY2: {
            category: 'matrix math',
            minParams: 2,
            maxParams: 2,
            type: 'val:num',
            signature: 'any|mat:force any|mat:force',
            resolve: function () {
                function aggregate(result, nums) { return result + nums[0] * nums[0] + nums[1] * nums[1]; }
                return this.aggregateNumbersParallel(arguments, 0, aggregate, _.identity, SUM_XY_OPTIONS);
            }
        },

        SUMXMY2: {
            category: 'matrix math',
            minParams: 2,
            maxParams: 2,
            type: 'val:num',
            signature: 'any|mat:force any|mat:force',
            resolve: function () {
                function aggregate(result, nums) { var diff = nums[0] - nums[1]; return result + diff * diff; }
                return this.aggregateNumbersParallel(arguments, 0, aggregate, _.identity, SUM_XY_OPTIONS);
            }
        },

        TRANSPOSE: {
            category: 'matrix',
            minParams: 1,
            maxParams: 1,
            type: 'mat',
            signature: 'mat:any',
            resolve: function (matrix) { return matrix.transpose(); }
        }
    };

});
