/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/spreadsheet/model/formula/funcs/conversionfuncs', [
    'io.ox/office/tk/utils',
    'io.ox/office/tk/locale/parser',
    'io.ox/office/tk/locale/formatter',
    'io.ox/office/spreadsheet/utils/errorcode'
], function (Utils, Parser, Formatter, ErrorCode) {

    'use strict';

    /**************************************************************************
     *
     * This module implements all spreadsheet functions dealing with numeric
     * conversion, e.g. number systems such as binary, octal, hexadecimal, or
     * Roman numbers; and conversion between different measurement units.
     *
     * See the README file in this directory for a detailed documentation about
     * the format of function descriptor objects.
     *
     *************************************************************************/

    // string with zero characters for expansion of converted numbers
    var ZEROS = Utils.repeatString('0', 255);

    // private global functions ===============================================

    /**
     * Expands the passed string (binary, octal, or hexadecimal number) with
     * leading zero characters, according to the passed length.
     *
     * @param {String} number
     *  A number as string (binary, octal, or hexadecimal).
     *
     * @param {Number|Undefined} length
     *  If specified, the target length of the passed number. If omitted, the
     *  passed number will be returned unmodified.
     *
     * @param {Number} maxLength
     *  The maximum allowed length. Larger values will result in throwing a
     *  #NUM! error code.
     *
     * @param {Boolean} [ensureMin=false]
     *  If set to true, and the passed number is longer than the value passed
     *  in the 'length' parameter, a #NUM! error code will be thrown. By
     *  default, larger numbers will be returned as passed in this case.
     *
     * @returns {String}
     *  The resulting number with leading zero characters.
     *
     * @throws {ErrorCode}
     *  The #NUM! error code, if the passed length is invalid.
     */
    function expandWithZeros(number, length, maxLength, ensureMin) {

        // always throw, if the passed number is larger than the specified maximum
        if (number.length > maxLength) { throw ErrorCode.NUM; }

        // check and expand the number with leading zeros
        if (_.isNumber(length)) {
            // length parameter must be valid
            if ((ensureMin && (length < number.length)) || (length > maxLength)) {
                throw ErrorCode.NUM;
            }
            // expand, but do not truncate
            if (number.length < length) {
                number = (ZEROS + number).slice(-length);
            }
        }

        return number;
    }

    /**
     * Converts the string representing of a binary number to a decimal number.
     *
     * @param {String} value
     *  A binary number string to be converted to a number if it is not longer
     *  than 10 characters.
     *
     * @returns {Number}
     *  The resulting decimal number.
     *
     * @throws {ErrorCode}
     *  If the passed value is not a valid binary number, or if it is too long
     *  (more than 10 digits), a #NUM! error will be thrown.
     */
    function convertBinToDec(value) {

        // check input value, maximum length of the input string is 10 digits
        if (!/^[01]{0,10}$/.test(value)) {
            throw ErrorCode.NUM;
        }

        var // the parsed result (empty string is interpreted as zero)
            result = (value.length > 0) ? parseInt(value, 2) : 0;

        // negative number, if 10th digit is 1
        if ((value.length === 10) && /^1/.test(value)) {
            result -= 0x400;
        }

        return result;
    }

    /**
     * Converts the string representing of an octal number to a decimal number.
     *
     * @param {String} value
     *  An octal number string to be converted to a number if it is not longer
     *  than 10 characters.
     *
     * @returns {Number}
     *  The resulting decimal number.
     *
     * @throws {ErrorCode}
     *  If the passed value is not a valid octal number, or if it is too long
     *  (more than 10 digits), a #NUM! error will be thrown.
     */
    function convertOctToDec(value) {

        // check input value, maximum length of the input string is 10 digits
        if (!/^[0-7]{0,10}$/.test(value)) {
            throw ErrorCode.NUM;
        }

        var // the parsed result (empty string is interpreted as zero)
            result = (value.length > 0) ? parseInt(value, 8) : 0;

        // negative number, if 10th digit is greater than 3
        if ((value.length === 10) && /^[4-7]/.test(value)) {
            result -= 0x40000000;
        }

        return result;
    }

    /**
     * Converts the string representing of a hexadecimal number to a decimal
     * number.
     *
     * @param {String} value
     *  A hexadecimal number string to be converted to a number if it is not
     *  longer than 10 characters.
     *
     * @returns {Number}
     *  The resulting decimal number.
     *
     * @throws {ErrorCode}
     *  If the passed value is not a valid hexadecimal number, or if it is too
     *  long (more than 10 digits), a #NUM! error will be thrown.
     */
    function convertHexToDec(value) {

        // check input value, maximum length of the input string is 10 digits
        if (!/^[0-9a-f]{0,10}$/i.test(value)) {
            throw ErrorCode.NUM;
        }

        var // the parsed result (empty string is interpreted as zero)
            result = (value.length > 0) ? parseInt(value, 16) : 0;

        // negative number, if 10th digit is greater than 7
        if ((value.length === 10) && /^[89a-f]/i.test(value)) {
            result -= 0x10000000000;
        }

        return result;
    }

    /**
     * Converts the passed decimal number to its binary representation. Inserts
     * leading zero characters if specified.
     *
     * @param {Number} number
     *  The number to be converted to its binary representation.
     *
     * @param {Number} [length]
     *  If specified, the target length of the passed number. See function
     *  expandWithZeros() for details.
     *
     * @returns {String}
     *  The resulting binary number.
     *
     * @throws {ErrorCode}
     *  The #NUM! error code, if the passed number results in more than 10
     *  binary digits, or if the passed length is invalid.
     */
    function convertDecToBin(number, length) {
        // restrict passed number to signed 10-digit binary
        if ((-number > 0x200) || (number > 0x1FF)) { throw ErrorCode.NUM; }
        // convert negative numbers to positive resulting in a 10-digit binary number
        if (number < 0) { number += 0x400; }
        // convert to hexadecimal number, expand to specified length
        return expandWithZeros(number.toString(2), length, 10, true);
    }

    /**
     * Converts the passed decimal number to its octal representation. Inserts
     * leading zero characters if specified.
     *
     * @param {Number} number
     *  The number to be converted to its octal representation.
     *
     * @param {Number} [length]
     *  If specified, the target length of the passed number. See function
     *  expandWithZeros() for details.
     *
     * @returns {String}
     *  The resulting octal number.
     *
     * @throws {ErrorCode}
     *  The #NUM! error code, if the passed number results in more than 10
     *  octal digits, or if the passed length is invalid.
     */
    function convertDecToOct(number, length) {
        // restrict passed number to signed 10-digit octal
        if ((-number > 0x20000000) || (number > 0x1FFFFFFF)) { throw ErrorCode.NUM; }
        // convert negative numbers to positive resulting in a 10-digit octal number
        if (number < 0) { number += 0x40000000; }
        // convert to octal number, expand to specified length
        return expandWithZeros(number.toString(8), length, 10, true);
    }

    /**
     * Converts the passed decimal number to its hexadecimal representation.
     * Inserts leading zero characters if specified.
     *
     * @param {Number} number
     *  The number to be converted to its hexadecimal representation.
     *
     * @param {Number} [length]
     *  If specified, the target length of the passed number. See function
     *  expandWithZeros() for details.
     *
     * @returns {String}
     *  The resulting hexadecimal number.
     *
     * @throws {ErrorCode}
     *  The #NUM! error code, if the passed number results in more than 10
     *  hexadecimal digits, or if the passed length is invalid.
     */
    function convertDecToHex(number, length) {
        // restrict passed number to signed 10-digit hexadecimal
        if ((-number > 0x8000000000) || (number > 0x7FFFFFFFFF)) { throw ErrorCode.NUM; }
        // convert negative numbers to positive resulting in a 10-digit hex number
        if (number < 0) { number += 0x10000000000; }
        // convert to hexadecimal number, expand to specified length
        return expandWithZeros(number.toString(16).toUpperCase(), length, 10, true);
    }

    /**
     * converts assigned number from one unit to another unit
     * 1 km is 1000 meter
     * (1, 'km', 'm' ) = 1000
     *
     * @param {Number} number
     *  The number to be converted
     *
     * @param {String} fromUnit
     *  the unit of the assigned number
     *
     * @param {String} toUnit
     *  the unit of the result number
     *
     * @throws {ErrorCode}
     *  The #NA! error code, if the passed units are not from same type
     *  (length, area, volumne, time, pressure, force, power etc)
     *
     * @return {Number}
     *  the converted Number in the expected to-nit
     */
    var convert = (function () {

        var CDC_Mass = 'mass';
        var CDC_Length = 'length';
        var CDC_Time = 'time';
        var CDC_Pressure = 'pressure';
        var CDC_Force = 'force';
        var CDC_Energy = 'energy';
        var CDC_Power = 'power';
        var CDC_Magnetism = 'magnetism';
        var CDC_Temperature = 'temperature';
        var CDC_Volume = 'volume';
        var CDC_Area = 'area';
        var CDC_Speed = 'speed';
        var CDC_Information = 'information';

        var ALLTYPES = {};
        var ALLDEFAULTS = {};
        var ALLMULTIS = [];
        //https://github.com/LibreOffice/core/blob/db17d3c17c40d6b0e92392cf3c6e343d1d17b771/scaddins/source/analysis/analysishelper.cxx

        function addType(key, unit, cl, offset, prefixSupport) {
            ALLTYPES[key] = { key: key, unit: 1 / unit, cl: cl, prefixSupport: prefixSupport, offset: offset };
            if (unit === 1.0000000000000000E00 && !ALLDEFAULTS[cl]) {
                ALLDEFAULTS[cl] = key;
            }
        }

        function newD(key, unit, cl) {
            addType(key, unit, cl, undefined, false);
        }

        function newDP(key, unit, cl) {
            addType(key, unit, cl, undefined, true);
        }

        function newL(key, unit, offset, cl) {
            addType(key, unit, cl, offset, false);
        }

        function newLP(key, unit, offset, cl) {
            addType(key, unit, cl, offset, true);
        }

        function newMulti(key, multi, cl) {
            ALLMULTIS.push({ key: key, multi: multi, cl: cl });
        }

        // MASS: 1 Gram is...
        newDP('g',         1.0000000000000000E00,  CDC_Mass); // Gram
        newD('sg',         6.8522050005347800E-05, CDC_Mass); // Pieces
        newD('lbm',        2.2046229146913400E-03, CDC_Mass); // Pound (commercial weight)
        newDP('u',         6.0221370000000000E23,  CDC_Mass); // U (atomic mass)
        newD('ozm',        3.5273971800362700E-02, CDC_Mass); // Ounce (commercial weight)
        newD('stone',      1.574730e-04,           CDC_Mass); // *** Stone
        newD('ton',        1.102311e-06,           CDC_Mass); // *** Ton
        newD('grain',      1.543236E01,            CDC_Mass); // *** Grain
        newD('pweight',    7.054792E-01,           CDC_Mass); // *** Pennyweight
        newD('hweight',    1.968413E-05,           CDC_Mass); // *** Hundredweight
        newD('shweight',   2.204623E-05,           CDC_Mass); // *** Shorthundredweight
        newD('brton',      9.842065E-07,           CDC_Mass); // *** Gross Registered Ton
        newD('cwt',        2.2046226218487758E-05, CDC_Mass); // U.S. (short) hundredweight
        newD('shweight',   2.2046226218487758E-05, CDC_Mass); // U.S. (short) hundredweight also
        newD('uk_cwt',     1.9684130552221213E-05, CDC_Mass); // Imperial hundredweight
        newD('lcwt',       1.9684130552221213E-05, CDC_Mass); // Imperial hundredweight also
        newD('hweight',    1.9684130552221213E-05, CDC_Mass); // Imperial hundredweight also
        newD('uk_ton',     9.8420652761106063E-07, CDC_Mass); // Imperial ton
        newD('LTON',       9.8420652761106063E-07, CDC_Mass); // Imperial ton also

        // LENGTH: 1 Meter is...
        newDP('m',         1.0000000000000000E00,       CDC_Length); // Meter
        newD('mi',         6.2137119223733397E-04,      CDC_Length); // Britsh Mile        6,21371192237333969617434184363e-4
        newD('Nmi',        5.3995680345572354E-04,      CDC_Length); // Nautical Mile      5,39956803455723542116630669546e-4
        newD('in',         3.9370078740157480E01,       CDC_Length); // Inch               39,37007874015748031496062992126
        newD('ft',         3.2808398950131234E00,       CDC_Length); // Foot               3,2808398950131233595800524934383
        newD('yd',         1.0936132983377078E00,       CDC_Length); // Yard               1,0936132983377077865266841644794
        newDP('ang',       1.0000000000000000E10,       CDC_Length); // Angstroem
        newD('Pica',       9 / 0.003175,                CDC_Length); // Pica (1/72 Inch)   2834,6456692913385826771653543307
        newD('ell',        1 / 1.143,                   CDC_Length); // *** Ell
        newDP('parsec',    1 / 3.08567758128155E+16,    CDC_Length); // *** Parsec
        newDP('pc',        1 / 3.08567758128155E+16,    CDC_Length); // *** Parsec also
        newDP('ly',        1 / 9.4607304725808E+15,     CDC_Length); // *** Light Year also
        newD('survey_mi',  1 / 1609.34721869444,      CDC_Length); // U.S. survey mile

        // TIME: 1 Second is...
        newD('yr',     3.1688087814028950E-08, CDC_Time); // Year
        newD('day',    1.1574074074074074E-05, CDC_Time); // Day
        newD('d',      1.1574074074074074E-05, CDC_Time); // Day also
        newD('hr',     2.7777777777777778E-04, CDC_Time); // Hour
        newD('mn',     1.6666666666666667E-02, CDC_Time); // Minute
        newD('min',    1.6666666666666667E-02, CDC_Time); // Minute also
        newDP('sec',   1.0000000000000000E00,  CDC_Time); // Second
        newDP('s',     1.0000000000000000E00,  CDC_Time); // Second also

        // PRESSURE: 1 Pascal is...
        newDP('Pa',    1.0000000000000000E00,  CDC_Pressure); // Pascal
        newDP('atm',   9.8692329999819300E-06, CDC_Pressure); // Atmosphere
        newDP('at',    9.8692329999819300E-06, CDC_Pressure); // Atmosphere also
        newDP('mmHg',  7.5006170799862700E-03, CDC_Pressure); // mm Hg (Mercury)
        newD('Torr',   7.5006380000000000E-03, CDC_Pressure); // *** Torr
        newD('psi',    1.4503770000000000E-04, CDC_Pressure); // *** Psi

        // FORCE: 1 Newton is...
        newDP('N',     1.0000000000000000E00,  CDC_Force); // Newton
        newDP('dyn',   1.0000000000000000E05,  CDC_Force); // Dyn
        newDP('dy',    1.0000000000000000E05,  CDC_Force); // Dyn also
        newD('lbf',    2.24808923655339E-01,   CDC_Force); // Pound-Force
        newDP('pond',  1.019716E02,            CDC_Force); // *** Pond

        // ENERGY: 1 Joule is...
        newDP('J',     1.0000000000000000E00,  CDC_Energy); // Joule
        newDP('e',     1.0000000000000000E07,  CDC_Energy); // Erg  -> https://en.wikipedia.org/wiki/Erg
        newDP('c',     2.3900624947346700E-01, CDC_Energy); // Thermodynamical Calorie
        newDP('cal',   2.3884619064201700E-01, CDC_Energy); // Calorie
        newDP('eV',    6.2414570000000000E18,  CDC_Energy); // Electronvolt
        newDP('ev',    6.2414570000000000E18,  CDC_Energy); // Electronvolt also
        newD('HPh',    3.7250611111111111E-07, CDC_Energy); // Horsepower Hours
        newD('hh',     3.7250611111111111E-07, CDC_Energy); // Horsepower Hours also
        newDP('Wh',    2.7777777777777778E-04, CDC_Energy); // Watt Hours
        newDP('wh',    2.7777777777777778E-04, CDC_Energy); // Watt Hours also
        newD('flb',    2.37304222192651E01,    CDC_Energy); // Foot Pound
        newD('BTU',    9.4781506734901500E-04, CDC_Energy); // British Thermal Unit
        newD('btu',    9.4781506734901500E-04, CDC_Energy); // British Thermal Unit also

        // POWER: 1 Watt is...
        newDP('W',     1.0000000000000000E00,  CDC_Power); // Watt
        newDP('w',     1.0000000000000000E00,  CDC_Power); // Watt also
        newD('HP',     1.341022E-03,           CDC_Power); // Horsepower
        newD('h',      1.341022E-03,           CDC_Power); // Horsepower also
        newD('PS',     1.359622E-03,           CDC_Power); // *** German Pferdestaerke

        // MAGNETISM: 1 Tesla is...
        newDP('T',     1.0000000000000000E00,  CDC_Magnetism); // Tesla
        newDP('ga',    1.0000000000000000E04,  CDC_Magnetism); // Gauss

        // TEMPERATURE: 1 Kelvin is...
        newL('C',      1.0000000000000000E00,  -2.7315000000000000E02, CDC_Temperature); // Celsius
        newL('cel',    1.0000000000000000E00,  -2.7315000000000000E02, CDC_Temperature); // Celsius also
        newL('F',      1.8000000000000000E00,  -2.5537222222222222E02, CDC_Temperature); // Fahrenheit
        newL('fah',    1.8000000000000000E00,  -2.5537222222222222E02, CDC_Temperature); // Fahrenheit also
        newLP('K',     1.0000000000000000E00,  +0.0000000000000000E00, CDC_Temperature); // Kelvin
        newLP('kel',   1.0000000000000000E00,  +0.0000000000000000E00, CDC_Temperature); // Kelvin also
        newL('Reau',   8.0000000000000000E-01, -2.7315000000000000E02, CDC_Temperature); // *** Reaumur
        newL('Rank',   1.8000000000000000E00,  +0.0000000000000000E00, CDC_Temperature); // *** Rankine

        // VOLUME: 1 Liter is...
        newD('tsp',        2.0288413621105798E02,  CDC_Volume); // US teaspoon            1/768 gallon
        newD('tbs',        6.7628045403685994E01,  CDC_Volume); // US tablespoon          1/256 gallon
        newD('oz',         3.3814022701842997E01,  CDC_Volume); // Ounce Liquid           1/128 gallon
        newD('cup',        4.2267528377303746E00,  CDC_Volume); // Cup                    1/16 gallon
        newD('pt',         2.1133764188651873E00,  CDC_Volume); // US Pint                1/8 gallon
        newD('us_pt',      2.1133764188651873E00,  CDC_Volume); // US Pint also
        newD('uk_pt',      1.7597539863927023E00,  CDC_Volume); // UK Pint                1/8 imperial gallon
        newD('qt',         1.0566882094325937E00,  CDC_Volume); // Quart                  1/4 gallon
        newD('gal',        2.6417205235814842E-01, CDC_Volume); // Gallon                 1/3.785411784
        newDP('l',         1.0000000000000000E00,  CDC_Volume); // Liter
        newDP('L',         1.0000000000000000E00,  CDC_Volume); // Liter also
        newDP('lt',        1.0000000000000000E00,  CDC_Volume); // Liter also
        newDP('m3',        1.0000000000000000E-03, CDC_Volume); // *** Cubic Meter
        newD('mi3',        1 / 4.16818182544058E+12,    CDC_Volume); // *** Cubic Britsh Mile
        newD('Nmi3',       1 / 6.352182208E+12,         CDC_Volume); // *** Cubic Nautical Mile
        newD('in3',        6.1023744094732284E01,  CDC_Volume); // *** Cubic Inch
        newD('ft3',        1 / 28.316846592,            CDC_Volume); // *** Cubic Foot
        newD('yd3',        1 / 764.554857984,           CDC_Volume); // *** Cubic Yard
        newDP('ang3',      1.0000000000000000E27,  CDC_Volume); // *** Cubic Angstroem
        newD('Pica3',      1 / 4.39039566186557E-08,    CDC_Volume); // *** Cubic Pica
        newD('barrel',     6.2898107704321051E-03, CDC_Volume); // *** Barrel (=42gal)
        newD('bushel',     2.837759E-02,           CDC_Volume); // *** Bushel
        newD('regton',     3.531467E-04,           CDC_Volume); // *** Register ton
        newD('GRT',        3.531467E-04,           CDC_Volume); // *** Register ton also
        newD('Schooner',   2.3529411764705882E00,  CDC_Volume); // *** austr. Schooner
        newD('Middy',      3.5087719298245614E00,  CDC_Volume); // *** austr. Middy
        newD('Glass',      5.0000000000000000E00,  CDC_Volume); // *** austr. Glass
        newD('ly3',        1 / 8.46786664623715E+50,    CDC_Volume); // *** Cubic light-year
        newD('MTON',       1.4125866688595436E00,  CDC_Volume); // *** Measurement ton
        newD('tspm',       2.0000000000000000E02,  CDC_Volume); // *** Modern teaspoon
        newD('uk_gal',     2.1996924829908779E-01,  CDC_Volume); // U.K. / Imperial gallon        1/4.54609
        newD('uk_qt',      8.7987699319635115E-01,  CDC_Volume); // U.K. / Imperial quart         1/4 imperial gallon

        // 1 Square Meter is...
        newDP('m2',        1.0000000000000000E00,  CDC_Area); // *** Square Meter
        newD('mi2',        1 / 2589988.110336,     CDC_Area); // *** Square Britsh Mile
        newD('Nmi2',       2.9155334959812286E-07, CDC_Area); // *** Square Nautical Mile
        newD('in2',        1.5500031000062000E03,  CDC_Area); // *** Square Inch
        newD('ft2',        1.0763910416709722E01,  CDC_Area); // *** Square Foot
        newD('yd2',        1.1959900463010803E00,  CDC_Area); // *** Square Yard
        newDP('ang2',      1.0000000000000000E20,  CDC_Area); // *** Square Angstroem
        newD('Pica2',      1 / 1.24452160493827E-07,    CDC_Area); // *** Square Pica
        newD('Morgen',     4.0000000000000000E-04, CDC_Area); // *** Morgen
        newDP('ar',        1.000000E-02,           CDC_Area); // *** Ar
        newD('acre',       2.471053815E-04,        CDC_Area); // *** Acre
        newD('uk_acre',    2.4710538146716534E-04, CDC_Area); // *** International acre
        newD('us_acre',    2.4710439304662790E-04, CDC_Area); // *** U.S. survey/statute acre
        newD('ly2',        1 / 8.95054210748189E+31, CDC_Area); // *** Square Light-year
        newD('ha',         1.000000E-04,           CDC_Area); // *** Hectare

        // SPEED: 1 Meter per Second is...
        newDP('m/s',   1.0000000000000000E00,  CDC_Speed); // *** Meters per Second
        newDP('m/sec', 1.0000000000000000E00,  CDC_Speed); // *** Meters per Second also
        newDP('m/h',   3.6000000000000000E03,  CDC_Speed); // *** Meters per Hour
        newDP('m/hr',  3.6000000000000000E03,  CDC_Speed); // *** Meters per Hour also
        newD('mph',    2.2369362920544023E00,  CDC_Speed); // *** Britsh Miles per Hour
        newD('kn',     1.9438444924406048E00,  CDC_Speed); // *** Knot = Nautical Miles per Hour
        newD('admkn',  1.9438446603753486E00,  CDC_Speed); // *** Admiralty Knot

        // INFORMATION: 1 Bit is...
        newDP('bit',   1.00E00,  CDC_Information); // *** Bit
        newDP('byte',  1.25E-01, CDC_Information); // *** Byte

        newMulti('Yi', Math.pow(2, 80), CDC_Information);
        newMulti('Zi', Math.pow(2, 70), CDC_Information);
        newMulti('Ei', Math.pow(2, 60), CDC_Information);
        newMulti('Pi', Math.pow(2, 50), CDC_Information);
        newMulti('Ti', Math.pow(2, 30), CDC_Information);
        newMulti('Gi', Math.pow(2, 30), CDC_Information);
        newMulti('Mi', Math.pow(2, 30), CDC_Information);
        newMulti('ki', Math.pow(2, 30), CDC_Information);

        newMulti('da', 1e01);
        newMulti('Y', 1e+24);
        newMulti('Z', 1e+21);
        newMulti('E', 1e+18);
        newMulti('p', 1e+15);
        newMulti('T', 1e+12);
        newMulti('G', 1e+09);
        newMulti('M', 1e+06);
        newMulti('k', 1e+03);
        newMulti('h', 1e+02);
        newMulti('e', 1e+01);
        newMulti('d', 1e-01);
        newMulti('c', 1e-02);
        newMulti('m', 1e-03);
        newMulti('u', 1e-06);
        newMulti('n', 1e-09);
        newMulti('p', 1e-12);
        newMulti('f', 1e-15);
        newMulti('a', 1e-18);
        newMulti('z', 1e-21);
        newMulti('y', 1e-24);

        function convertToBase(value, group) {
            return value * group.unit;
        }

        function convertFromBase(value, group) {
            return value / group.unit;
        }

        function convertToBaseOffset(value, group) {
            value *= group.unit;
            value -= group.offset;
            return value;
        }

        function convertFromBaseOffset(value, group) {
            value += group.offset;
            value /=  group.unit;
            return value;
        }

        function getMatchGroup(unit) {
            var last = _.last(unit);
            if (unit[unit.length - 2] === '^') {
                unit = unit.substring(unit, unit.length - 2) + last;
            }
            var group = ALLTYPES[unit];
            if (!group) {
                for (var key in ALLMULTIS) {
                    var multi = ALLMULTIS[key];
                    if (multi.key.length < unit.length && unit.indexOf(multi.key) === 0) {
                        var multiGroup = getMatchGroup(unit.substring(multi.key.length));
                        if (multiGroup.prefixSupport) {
                            if (multi.cl && multi.cl !== multiGroup.cl) {
                                group = {
                                    error: 'group found, but multiplikator is not allowed for that group type "' + unit + '" found "' + multiGroup.key + '" !!!'
                                };
                            } else {
                                var exp = 1;
                                if (last === '2' || last === '3') {
                                    exp = parseInt(last, 10);
                                }
                                group = _.clone(multiGroup);
                                group.unit *= Math.pow(multi.multi, exp);
                                group.key = unit;
                            }
                        } else {
                            group = {
                                error: 'group found, but has no prefix support "' + unit + '" found "' + multiGroup.key + '" !!!'
                            };
                        }
                        break;
                    }
                }

                if (!group) {
                    group = {
                        error: 'cant find any fitting group for "' + unit + '" !!!'
                    };
                }
                ALLTYPES[unit] = group;
            }
            if (group.error) {
                //console.warn('group.error', group.error);
                throw ErrorCode.NA;
            }
            return group;
        }

        return function (number, fromUnit, toUnit) {
            //console.warn('resolve', number, fromUnit, toUnit);

            var fromGroup = getMatchGroup(fromUnit);
            var toGroup = getMatchGroup(toUnit);

            if (!fromGroup || !toGroup || fromGroup.cl !== toGroup.cl) { throw ErrorCode.NA; }
            if (fromGroup.error || toGroup.error) { throw ErrorCode.NA; }

            var base = null;
            var result = null;
            if (_.isUndefined(fromGroup.offset)) {
                base = convertToBase(number, fromGroup);
            } else {
                base = convertToBaseOffset(number, fromGroup);
            }
            if (_.isUndefined(fromGroup.offset)) {
                result = convertFromBase(base, toGroup);
            } else {
                result = convertFromBaseOffset(base, toGroup);
            }
            //console.warn('convert number ' + number + ' ' + fromUnit + ' base ' + base + ' ' + ALLDEFAULTS[fromGroup.cl] + ' result ' + result + ' ' + toUnit);
            return result;
        };
    }());

    // exports ================================================================

    return {

        ARABIC: {
            category: 'conversion',
            name: { ooxml: '_xlfn.ARABIC' },
            minParams: 1,
            maxParams: 1,
            type: 'val:num',
            signature: 'val:str',
            resolve: function (roman) {

                // trim space characters; roman number must not be longer than 255 characters (including minus sign)
                roman = roman.replace(/^ +| +$/g, '');
                if (roman.length > 255) { throw ErrorCode.VALUE; }

                // roman number may contain a leading minus sign
                var sign = 1;
                if (roman[0] === '-') {
                    sign = -1;
                    roman = roman.substr(1);
                }

                // use number formatter to parse the roman number
                var result = Parser.parseRoman(roman);

                // throw #VALUE! error, if the text cannot be parsed
                if (result === null) { throw ErrorCode.VALUE; }
                return sign * result;
            }
        },

        BASE: {
            category: 'conversion',
            name: { ooxml: '_xlfn.BASE' },
            minParams: 2,
            maxParams: 3,
            type: 'val:str',
            signature: 'val:int val:int val:int',
            resolve: function (number, radix, length) {
                if ((number < 0) || (number > 0x1FFFFFFFFFFFFF)) { throw ErrorCode.NUM; }
                if ((radix < 2) || (radix > 36)) { throw ErrorCode.NUM; }
                if (_.isNumber(length) && ((length < 0) || (length > 255))) { throw ErrorCode.NUM; }
                return expandWithZeros(number.toString(radix).toUpperCase(), length, 255);
            }
        },

        BIN2DEC: {
            category: 'conversion',
            minParams: 1,
            maxParams: 1,
            type: 'val:num',
            signature: 'val:str',
            resolve: convertBinToDec
        },

        BIN2HEX: {
            category: 'conversion',
            minParams: 1,
            maxParams: 2,
            type: 'val:str',
            signature: 'val:str val:int',
            resolve: function (number, length) {
                return convertDecToHex(convertBinToDec(number), length);
            }
        },

        BIN2OCT: {
            category: 'conversion',
            minParams: 1,
            maxParams: 2,
            type: 'val:str',
            signature: 'val:str val:int',
            resolve: function (number, length) {
                return convertDecToOct(convertBinToDec(number), length);
            }
        },

        COLOR: {
            category: 'conversion',
            name: { ooxml: null, odf: 'ORG.LIBREOFFICE.COLOR' },
            minParams: 3,
            maxParams: 4,
            type: 'val:num',
            signature: 'val:int val:int val:int val:int',
            resolve: function (r, g, b, a) {

                if ((r < 0) || (r > 255)) { throw ErrorCode.VALUE; }
                if ((g < 0) || (g > 255)) { throw ErrorCode.VALUE; }
                if ((b < 0) || (b > 255)) { throw ErrorCode.VALUE; }

                if (this.isMissingOperand(3)) {
                    a = 0;
                } else if ((a < 0) || (a > 255)) {
                    throw ErrorCode.VALUE;
                }

                return a * 0x1000000 + r * 0x10000 + g * 0x100 + b;
            }
        },

        CONVERT: {
            category: 'conversion',
            minParams: 3,
            maxParams: 3,
            type: 'val:num',
            signature: 'val:num val:str val:str',
            resolve: function (number, fromUnit, toUnit) {
                return convert(number, fromUnit, toUnit);
            }
        },

        'CONVERT.ODF': {
            category: 'conversion',
            name: { ooxml: null, odf: 'ORG.OPENOFFICE.CONVERT' },
            hidden: true,
            minParams: 3,
            maxParams: 3,
            type: 'val:num'
        },

        DEC2BIN: {
            category: 'conversion',
            minParams: 1,
            maxParams: 2,
            type: 'val:str',
            signature: 'val:int val:int',
            resolve: convertDecToBin
        },

        DEC2HEX: {
            category: 'conversion',
            minParams: 1,
            maxParams: 2,
            type: 'val:str',
            signature: 'val:int val:int',
            resolve: convertDecToHex
        },

        DEC2OCT: {
            category: 'conversion',
            minParams: 1,
            maxParams: 2,
            type: 'val:str',
            signature: 'val:int val:int',
            resolve: convertDecToOct
        },

        DECIMAL: {
            category: 'conversion',
            name: { ooxml: '_xlfn.DECIMAL' },
            minParams: 2,
            maxParams: 2,
            type: 'val:num',
            signature: 'val:str val:int',
            resolve: (function () {
                var getTestRE = _.memoize(function (radix) {

                    // for radices 2-10: decimal digits only; otherwise all digits and additional letters from A to radix
                    var charClass = (radix <= 10) ? ('0-' + (radix - 1)) : ('0-9A-' + String.fromCharCode(54 + radix));

                    // create the regular expression
                    return new RegExp('^[' + charClass + ']+$', 'i');
                });

                return function (input, radix) {

                    if ((radix < 2) || (radix > 36)) { throw ErrorCode.NUM; }

                    // test the input value
                    if (!getTestRE(radix).test(input)) { throw ErrorCode.NUM; }

                    var decimal = parseInt(input, radix);
                    if (!isFinite(decimal) || (decimal < 0)) { throw ErrorCode.NUM; }
                    return decimal;
                };
            }())

        },

        HEX2BIN: {
            category: 'conversion',
            minParams: 1,
            maxParams: 2,
            type: 'val:str',
            signature: 'val:str val:int',
            resolve: function (number, length) {
                return convertDecToBin(convertHexToDec(number), length);
            }
        },

        HEX2DEC: {
            category: 'conversion',
            minParams: 1,
            maxParams: 1,
            type: 'val:num',
            signature: 'val:str',
            resolve: convertHexToDec
        },

        HEX2OCT: {
            category: 'conversion',
            minParams: 1,
            maxParams: 2,
            type: 'val:str',
            signature: 'val:str val:int',
            resolve: function (number, length) {
                return convertDecToOct(convertHexToDec(number), length);
            }
        },

        OCT2BIN: {
            category: 'conversion',
            minParams: 1,
            maxParams: 2,
            type: 'val:str',
            signature: 'val:str val:int',
            resolve: function (number, length) {
                return convertDecToBin(convertOctToDec(number), length);
            }
        },

        OCT2DEC: {
            category: 'conversion',
            minParams: 1,
            maxParams: 1,
            type: 'val:num',
            signature: 'val:str',
            resolve: convertOctToDec
        },

        OCT2HEX: {
            category: 'conversion',
            minParams: 1,
            maxParams: 2,
            type: 'val:str',
            signature: 'val:str val:int',
            resolve: function (number, length) {
                return convertDecToHex(convertOctToDec(number), length);
            }
        },

        ROMAN: {
            category: 'conversion',
            minParams: 1,
            maxParams: 2,
            type: 'val:str',
            signature: 'val:int val:int',
            resolve: function (number, depth) {

                // default is depth 0 (classic mode)
                if (!_.isNumber(depth)) { depth = 0; }

                // check the input parameters
                if ((depth < 0) || (depth >= 5) || (number < 0) || (number > 3999)) { throw ErrorCode.VALUE; }

                // use number formatter to format to a roman number
                return Formatter.formatRoman(number, depth);
            }
        }
    };

});
