/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/spreadsheet/model/formula/deps/pendingcollection', [
    'io.ox/office/tk/utils/iteratorutils',
    'io.ox/office/tk/container/valueset',
    'io.ox/office/tk/container/valuemap',
    'io.ox/office/spreadsheet/utils/sheetutils'
], function (IteratorUtils, ValueSet, ValueMap, SheetUtils) {

    'use strict';

    // convenience shortcuts
    var AddressArray = SheetUtils.AddressArray;
    var RangeArray = SheetUtils.RangeArray;

    // class PendingSheetData =================================================

    /**
     * Container for addresses of dirty cells and ranges in a single sheet.
     *
     * @constructor
     *
     * @property {SheetModel} sheetModel
     *  The model of the sheet this instance is related to.
     *
     * @property {RangeArray} dirtyRanges
     *  The addresses of all cell ranges where cells have been moved and thus
     *  whose formula cells need to be updated completely.
     *
     * @property {AddressArray} dirtyValueCells
     *  The addresses of all dirty value cells (cells with changed values) in
     *  the sheet. The cell addresses will be collected while processing cell
     *  change events received from the document, and will cause to update all
     *  formulas known by the dependency manager that depend directly or
     *  indirectly on these cells.
     *
     * @property {AddressArray} dirtyFormulaCells
     *  The addresses of all dirty formula cells (cells with changed formula
     *  expression) in the sheet. The cell addresses will be collected while
     *  processing cell change events received from the document, and will
     *  cause to refresh the reference dependencies.
     *
     * @property {AddressArray} recalcFormulaCells
     *  The addresses of all formula cells in the sheet with outdated result
     *  values. The cell addresses will be collected, and will cause to
     *  recalculate all formulas, in order to get the correct results e.g.
     *  after generating new cell formulas without the results.
     *
     * @property {ValueSet<RuleModel|SourceLinkMixin>} dirtyModelSet
     *  A set with all model objects with embedded token arrays that have been
     *  inserted into the sheet, or that have been changed. The models will be
     *  collected while processing change events received from the document,
     *  and will cause to update all their dependency settings.
     *
     * @property {Boolean} initialUpdate
     *  A property that marks a new sheet inserted into the pending collection.
     *  The next update cycle of the dependency manager will fetch all formulas
     *  from the new sheet, instead of just processing the changed formulas.
     */
    function PendingSheetData(sheetModel) {

        this.sheetModel = sheetModel;
        this.dirtyRanges = new RangeArray();
        this.dirtyValueCells = new AddressArray();
        this.dirtyFormulaCells = new AddressArray();
        this.recalcFormulaCells = new AddressArray();
        this.dirtyModelSet = new ValueSet('getUid()');
        this.initialUpdate = false;

    } // class PendingSheetData

    // public methods ---------------------------------------------------------

    /**
     * Inserts the passed model with embedded token arrays into this instance.
     *
     * @param {RuleModel|SourceLinkMixin} tokenModel
     *  The new or changed model object with embedded token arrays.
     *
     * @returns {PendingSheetData}
     *  A reference to this instance.
     */
    PendingSheetData.prototype.registerTokenModel = function (tokenModel) {
        this.dirtyModelSet.insert(tokenModel);
        return this;
    };

    /**
     * Removes the passed model with embedded token arrays from this instance.
     *
     * @param {RuleModel|SourceLinkMixin} tokenModel
     *  The model object with embedded token arrays to be deleted.
     *
     * @returns {PendingSheetData}
     *  A reference to this instance.
     */
    PendingSheetData.prototype.unregisterTokenModel = function (tokenModel) {
        this.dirtyModelSet.remove(tokenModel);
        return this;
    };

    // class PendingCollection ================================================

    /**
     * Encapsulates all data that will be collected by a dependency manager
     * when processing document change events. Will be used to find and update
     * all dirty formulas depending on the data collected in an instance of
     * this class.
     *
     * @constructor
     *
     * @property {Boolean} recalcAll
     *  Specifies whether the next update cycle has to recalculate all existing
     *  formulas, instead of just the dirty formulas according to the collected
     *  data in this instance.
     *
     * @property {Boolean} recalcVolatile
     *  Specifies whether the next update cycle has to recalculate all volatile
     *  formulas, also if they do not depend on any changed cell.
     *
     * @property {ValueSet<NameModel>} changedNamesSet
     *  The model instances of all dirty defined names. After inserting a new
     *  defined name, or changing its formula expression, all depending
     *  formulas need to be recalculated.
     *
     * @property {ValueSet<NameModel>} changedLabelsSet
     *  The model instances of all defined names with changed label. After
     *  inserting a new defined name, or changing its label, all formulas that
     *  contain unresolved names need to be recalculated in order to ged rid of
     *  #NAME? errors.
     *
     * @property {ValueSet<TableModel>} changedTablesSet
     *  The model instances of all dirty table ranges. After changing the
     *  target range of a table range, all depending formulas need to be
     *  recalculated, even if the contents of the table did not change (e.g.
     *  when using the functions COLUMNS or ROWS).
     *
     * @property {ValueMap<PendingSheetData>} changedSheetMap
     *  A map with all data of changed cell contents in a sheet, mapped by
     *  sheet UIDs.
     */
    function PendingCollection() {

        this.recalcAll = false;
        this.recalcVolatile = true;
        this.changedNamesSet = new ValueSet('getUid()');
        this.changedLabelsSet = new ValueSet('getUid()');
        this.changedTablesSet = new ValueSet('getUid()');
        this.changedSheetMap = new ValueMap();

    } // class PendingCollection

    // public methods ---------------------------------------------------------

    /**
     * Registers a defined name in this collection that has been inserted into
     * the spreadsheet document, or that has been changed (either the formula
     * expression, or the label).
     *
     * @param {NameModel} nameModel
     *  The model instance of the new or changed defined name.
     *
     * @param {Boolean} formulaChanged
     *  Whether the formula expression of the defined name has been changed.
     *  MUST be set to true for new defined names inserted into the document.
     *
     * @param {Boolean} labelChanged
     *  Whether the label of the defined name has been changed. MUST be set to
     *  true for new defined names inserted into the document.
     *
     * @returns {PendingCollection}
     *  A reference to this instance.
     */
    PendingCollection.prototype.registerNameModel = function (nameModel, formulaChanged, labelChanged) {
        if (formulaChanged) { this.changedNamesSet.insert(nameModel); }
        if (labelChanged) { this.changedLabelsSet.insert(nameModel); }
        return this;
    };

    /**
     * Unregisters a defined name from this collection that will be deleted
     * from the spreadsheet document.
     *
     * @param {NameModel} nameModel
     *  The model instance of the defined name that is about to be deleted.
     *
     * @returns {PendingCollection}
     *  A reference to this instance.
     */
    PendingCollection.prototype.unregisterNameModel = function (nameModel) {
        this.changedNamesSet.insert(nameModel);
        this.changedLabelsSet.remove(nameModel);
        return this;
    };

    /**
     * Registers a table range with a changed target range in this collection.
     *
     * @param {TableModel} tableModel
     *  The model instance of the changed table range.
     *
     * @returns {PendingCollection}
     *  A reference to this instance.
     */
    PendingCollection.prototype.registerTableModel = function (tableModel) {
        this.changedTablesSet.insert(tableModel);
        return this;
    };

    /**
     * Unregisters a table range from this collection that will be deleted from
     * the spreadsheet document.
     *
     * @param {TableModel} tableModel
     *  The model instance of the table range that is about to be deleted.
     *
     * @returns {PendingCollection}
     *  A reference to this instance.
     */
    PendingCollection.prototype.unregisterTableModel = function (tableModel) {
        this.changedTablesSet.remove(tableModel);
        return this;
    };

    /**
     * Returns an existing container for dirty addresses for the specified
     * sheet, or creates a new container on demand.
     *
     * @param {SheetModel} sheetModel
     *  The model instance of a sheet.
     *
     * @returns {PendingSheetData}
     *  The collection of dirty cell addresses for the specified sheet.
     */
    PendingCollection.prototype.createPendingSheetData = function (sheetModel) {
        return this.changedSheetMap.getOrConstruct(sheetModel.getUid(), PendingSheetData, sheetModel);
    };

    /**
     * Returns an existing container for dirty addresses for the specified
     * sheet, or creates a new container on demand.
     *
     * @param {SheetModel} sheetModel
     *  The model instance of a sheet.
     *
     * @returns {PendingSheetData}
     *  The collection of dirty cell addresses for the specified sheet.
     */
    PendingCollection.prototype.removePendingSheetData = function (sheetModel) {
        this.changedSheetMap.remove(sheetModel.getUid());
        return this;
    };

    /**
     * Returns an existing container for dirty addresses for the specified
     * sheet.
     *
     * @param {SheetModel} sheetModel
     *  The model instance of a sheet.
     *
     * @returns {PendingSheetData|Null}
     *  The collection of dirty cell addresses for the specified sheet, if
     *  existing; otherwise null.
     */
    PendingCollection.prototype.getPendingSheetData = function (sheetModel) {
        return this.changedSheetMap.get(sheetModel.getUid(), null);
    };

    /**
     * Adds initial settings for a new sheet into this collection.
     *
     * @param {SheetModel} sheetModel
     *  The model instance of the new sheet.
     *
     * @returns {PendingCollection}
     *  A reference to this instance.
     */
    PendingCollection.prototype.registerSheetModel = function (sheetModel) {
        this.createPendingSheetData(sheetModel).initialUpdate = true;
        return this;
    };

    /**
     * Removes all settings for a deleted sheet from this collection: The range
     * arrays of moved cells, all sheet-locally defined names, and the table
     * ranges.
     *
     * @param {SheetModel} sheetModel
     *  The model instance of the deleted sheet.
     *
     * @returns {PendingCollection}
     *  A reference to this instance.
     */
    PendingCollection.prototype.unregisterSheetModel = function (sheetModel) {

        // remove all dirty addresses for the deleted sheet
        this.removePendingSheetData(sheetModel);

        // remove all pending sheet-local names contained in the deleted sheet
        var nameIterator = sheetModel.getNameCollection().createModelIterator();
        IteratorUtils.forEach(nameIterator, this.unregisterNameModel, this);

        // remove all pending table models contained in the deleted sheet
        var tableIterator = sheetModel.getTableCollection().createModelIterator();
        IteratorUtils.forEach(tableIterator, this.unregisterTableModel, this);

        return this;
    };

    /**
     * Appends the addresses of dirty cells in a sheet according to the passed
     * cell change descriptor.
     *
     * @param {SheetModel} sheetModel
     *  The model instance of the changed sheet.
     *
     * @param {ChangeDescriptor} changeDesc
     *  The descriptor with the addresses of all changed value and formula
     *  cells, as received from a 'change:cells' document event.
     *
     * @returns {Boolean}
     *  Whether any cell addresses have been registered.
     */
    PendingCollection.prototype.registerDirtyCells = function (sheetModel, changeDesc) {
        var sheetData = this.createPendingSheetData(sheetModel);
        sheetData.dirtyValueCells.append(changeDesc.valueCells);
        sheetData.dirtyFormulaCells.append(changeDesc.formulaCells);
        return !changeDesc.valueCells.empty() || !changeDesc.formulaCells.empty();
    };

    /**
     * Moves all cached addresses of dirty cells in a sheet according to the
     * passed move descriptor.
     *
     * @param {SheetModel} sheetModel
     *  The model instance of the changed sheet.
     *
     * @param {MoveDescriptor} moveDesc
     *  The descriptor with the positions of all moved cells, as received from
     *  a 'move:cells' document event.
     *
     * @returns {PendingCollection}
     *  A reference to this instance.
     */
    PendingCollection.prototype.moveDirtyCells = function (sheetModel, moveDesc) {
        var sheetData = this.createPendingSheetData(sheetModel);
        sheetData.dirtyRanges.push(moveDesc.dirtyRange);
        sheetData.dirtyValueCells = moveDesc.transformAddresses(sheetData.dirtyValueCells);
        sheetData.dirtyFormulaCells = moveDesc.transformAddresses(sheetData.dirtyFormulaCells);
        sheetData.recalcFormulaCells = moveDesc.transformAddresses(sheetData.recalcFormulaCells);
        return this;
    };

    /**
     * Inserts the passed model object with embedded token arrays into this
     * collection.
     *
     * @param {RuleModel|SourceLinkMixin} tokenModel
     *  The new or changed model object with embedded token arrays.
     *
     * @returns {PendingCollection}
     *  A reference to this instance.
     */
    PendingCollection.prototype.registerTokenModel = function (tokenModel) {
        var sheetData = this.createPendingSheetData(tokenModel.getSheetModel());
        sheetData.registerTokenModel(tokenModel);
        return this;
    };

    /**
     * Removes the passed model object with embedded token arrays from this
     * collection.
     *
     * @param {RuleModel|SourceLinkMixin} tokenModel
     *  The model object with embedded token arrays to be deleted.
     *
     * @returns {PendingCollection}
     *  A reference to this instance.
     */
    PendingCollection.prototype.unregisterTokenModel = function (tokenModel) {
        this.changedSheetMap.with(tokenModel.getSheetModel().getUid(), function (sheetData) {
            sheetData.unregisterTokenModel(tokenModel);
        });
        return this;
    };

    // exports ================================================================

    return PendingCollection;

});
