/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Stefan Eckert <stefan.eckert@open-xchange.com>
 */

define('io.ox/office/spreadsheet/model/drawing/manualcolorchart', [
    'io.ox/office/tk/utils',
    'io.ox/office/drawinglayer/view/chartstyleutil'
], function (Utils, ChartStyleUtil) {

    'use strict';

    var BACKGROUNDTRANSFORMATION = ChartStyleUtil.getBackgroundTransformation();

    var STYLESET = ChartStyleUtil.getStyleSet();

    var COLORSET = ChartStyleUtil.getColorSet();

    var DUMMY_SHAPE = { type: 'solid', color: { type: 'rgb', value: '0000ff', width: 36 } };

    // class ManualColorHandler ===============================================

    function ManualColorHandler(chartModel, data) {

        // the document model
        var docModel = chartModel.getDocModel();

        // private methods ----------------------------------------------------

        /*
         * Apply tint/shade depending on the cycle index. The colors of leading
         * series are darkened (color shade), the colors of trailing series are
         * lightened (color tint). Shade/tint is applied in an exclusive range of
         * -70% to 70%.
         *
         * Example 1: 3 data series using single-color shading with accent color 1
         * (e.g. automatic chart style #3). Shade/tint is applied per series.
         * Shade/tint changes in steps of 140%/(<series_count+1) = 140%/4 = 35%,
         * starting at -70%:
         *  Step 1: -70% -> Not used.
         *  Step 2: -35% -> Series 1 has 35% shade of accent color 1.
         *  Step 3:   0% -> Series 2 has pure accent color 1.
         *  Step 4:  35% -> Series 3 has 35% tint of accent color 1.
         *  Step 5:  70% -> Not used.
         *
         * Example 2: 20 data series using accent color pattern (e.g. automatic
         * chart style #2). Each color cycle has a size of 6 series (accent colors
         * 1 to 6). Shade/tint is applied per color cycle.
         *  Cycle #1: Series 1...6 are based on accent colors 1 to 6.
         *  Cycle #2: Series 7...12 are based on accent colors 1 to 6.
         *  Cycle #3: Series 13...18 are based on accent colors 1 to 6.
         *  Cycle #4: Series 19...20 are based on accent colors 1 to 2.
         * Shade/tint changes in steps of 140%/(cycle_count+1) = 140%/5 = 28%,
         * starting at -70%:
         *  Step 1: -70% -> Not used.
         *  Step 2: -42% -> Cycle #1 has 42% shade of accent colors 1...6
         *  Step 3: -14% -> Cycle #2 has 14% shade of accent colors 1...6
         *  Step 4:  14% -> Cycle #3 has 14% tint of accent colors 1...6
         *  Step 5:  42% -> Cycle #4 has 42% tint of accent colors 1...6
         *  Step 6:  70% -> Not used.
         *
         * 843 sal_Int32 nPhClr = maColorPattern[ static_cast< size_t >( nSeriesIdx % maColorPattern.size() ) ];
         * 845 size_t nCycleIdx = static_cast< size_t >( nSeriesIdx / maColorPattern.size() );
         * 846 size_t nMaxCycleIdx = static_cast< size_t >( mrData.mnMaxSeriesIdx / maColorPattern.size() );
         * 847 double fShadeTint = static_cast< double >( nCycleIdx + 1 ) / (nMaxCycleIdx + 2) * 1.4 - 0.7;
         * 848 if( fShadeTint != 0.0 )
         * 849 {
         * 850     Color aColor;
         * 851     aColor.setSrgbClr( nPhClr );
         * 852     aColor.addChartTintTransformation( fShadeTint );
         * 853     nPhClr = aColor.getColor( mrData.mrFilter.getGraphicHelper() );
         * 854 }
         * 856 return nPhClr;
         */

        function getColor(index, type, count) {
            var attributes = chartModel.getMergedAttributeSet(true);

            var colors = attributes.chart.chartColors;

            var result = null;

            if (colors) {

                result = ChartStyleUtil.getColorOfPattern(colors.meth, 'group', index, colors.schemeClr, count, docModel);

            } else if (attributes.chart.chartStyleId) {
                var colorSet = COLORSET[ChartStyleUtil.toColorSetIndex(attributes.chart.chartStyleId)];

                var res = ChartStyleUtil.getColorOfPattern('cycle', colorSet.type, index, colorSet.colors, count, docModel);
                result = {
                    color: res,
                    type: type
                };
            } else {
                result = DUMMY_SHAPE;
            }
            return result;
        }

        function getStyleSet(dataSeries, color) {
            var attributes = chartModel.getMergedAttributeSet(true);

            var result = color;
            var styleSet = null;

            if (attributes.chart.chartStyleId) {
                styleSet = STYLESET[ChartStyleUtil.toStyleSetIndex(attributes.chart.chartStyleId)];
                dataSeries.bevelEnabled = styleSet.bevelEnabled;
            }

            if (dataSeries.seriesIndex === 0) {
                data.cssBackgroundColor = null;
                if (attributes.chart.chartStyleId && result) {

                    styleSet = STYLESET[ChartStyleUtil.toStyleSetIndex(attributes.chart.chartStyleId)];

                    var givenStyle = styleSet.bg;
                    if (!givenStyle) {
                        givenStyle = { type: result.color.type, value: result.color.value, transformations: BACKGROUNDTRANSFORMATION };
                    }

                    data.cssBackgroundColor = docModel.getCssColor(givenStyle, 'fill');
                }
            }
        }

        function getVaryColor() {
            if (chartModel.isVaryColorEnabled()) {
                var attributes = chartModel.getMergedAttributeSet(true);
                return attributes.chart.varyColors;
            }
            return false;
        }

        function applyColor(target, shape, targetKey) {
            var cssColor;

            if (shape.type === 'none') {
                cssColor = 'transparent';
            } else {
                var color = shape.color;
                if (shape.gradient && shape.gradient.colorStops && (!color || ChartStyleUtil.isAutoColor(color))) {
                    color = shape.gradient.colorStops[0].color;
                }
                cssColor = docModel.getCssColor(color, 'fill');
            }

            target[targetKey] = cssColor;
        }

        // public functions ---------------------------------------------------

        this.handleColor = function (sourceKey, dataSeries, targetKey, count) {

            var attrs = dataSeries.model.getMergedAttributeSet(true);
            var shape = attrs[sourceKey];

            var index = dataSeries.seriesIndex;

            _.each(dataSeries.dps, function (dataPoint) {
                dataPoint[targetKey] = null;
            });
            dataSeries[targetKey] = null;

            if (ChartStyleUtil.isAutoShape(shape)) {
                // shape = getColor(index, shape.type, count);
                if (getVaryColor()) {
                    dataSeries.color = null;
                    _(dataSeries.dps).each(function (dataPoint, pi) {
                        var colorHolder = getColor(pi, shape.type, dataSeries.dps.length);
                        if (pi === 0) {
                            getStyleSet(dataSeries, colorHolder);
                            if (colorHolder.type === 'none') {
                                //fallback for CanvasJS bug when series color is null and point color is set
                                dataSeries[targetKey] = 'transparent';
                            }
                        }
                        applyColor(dataPoint, colorHolder, targetKey);
                    });
                } else {
                    shape = getColor(index, shape.type, count);
                }
            }

            if (!ChartStyleUtil.isAutoShape(shape)) {
                applyColor(dataSeries, shape, targetKey);
            }

            getStyleSet(dataSeries, shape);

            var chartAttrs = chartModel.getMergedAttributeSet(true);
            if (index === 0 && chartModel.getExplicitAttributeSet(true).fill) {
                var fill = chartAttrs.fill;
                if (ChartStyleUtil.isAutoShape(fill) && chartAttrs.chart.chartStyleId) {
                    data.cssBackgroundColor = '#' + this.getBackgroundColor(chartAttrs.chart.chartStyleId).fallbackValue;
                } else {
                    applyColor(data, fill, 'cssBackgroundColor');
                }
            }

            var dataPointsAttrs = attrs.series.dataPoints;
            if (dataPointsAttrs && dataPointsAttrs.length) {
                _.each(dataSeries.dps, function (dataPoint, pi) {
                    var dataPointAttrs = dataPointsAttrs[pi % dataPointsAttrs.length];
                    if (dataPointAttrs) {
                        applyColor(dataPoint, dataPointAttrs[sourceKey], targetKey);
                    }
                });
            }
        };

        this.getBackgroundColor = function (chartStyleId) {
            var styleSet = STYLESET[ChartStyleUtil.toStyleSetIndex(chartStyleId)];

            var bgColor = styleSet && styleSet.bg;
            if (!bgColor) {
                var colorSet = COLORSET[ChartStyleUtil.toColorSetIndex(chartStyleId)];
                if (colorSet) {
                    var color = ChartStyleUtil.getColorOfPattern('cycle', 'single', 1, colorSet.colors, 3, docModel);
                    bgColor = { type: color.type, value: color.value, transformations: BACKGROUNDTRANSFORMATION };
                } else {
                    bgColor = { type: 'auto' };
                }
            }
            if (!bgColor.fallbackValue) {
                bgColor.fallbackValue = docModel.parseAndResolveColor(bgColor, 'fill').hex;
            }
            return bgColor;
        };

        // initialization -----------------------------------------------------

        data.colorSet = null;

    } // class ManualColorHandler

    // exports ================================================================

    return ManualColorHandler;

});
