/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Stefan Eckert <stefan.eckert@open-xchange.com>
 */

define('io.ox/office/spreadsheet/model/drawing/legendmodel', [
    'io.ox/office/drawinglayer/view/chartstyleutil',
    'io.ox/office/editframework/model/attributedmodel',
    'io.ox/office/spreadsheet/utils/operations'
], function (ChartStyleUtil, AttributedModel, Operations) {

    'use strict';

    // class LegendModel ======================================================

    /**
     * Representation of a legend in a chart drawing object.
     *
     * @constructor
     *
     * @extends AttributedModel
     */
    function LegendModel(chartModel, attrs, dataLegend) {

        var self = this;

        // base constructor ---------------------------------------------------

        AttributedModel.call(this, chartModel.getDocModel(), attrs, { families: 'legend character' });

        // public methods -----------------------------------------------------

        this.refreshInfo = function () {
            var attributeSet = self.getMergedAttributeSet(true);
            var legendPos = attributeSet.legend.pos;
            switch (legendPos) {
                case 'bottom':
                case 'top':
                    dataLegend.verticalAlign = legendPos;
                    dataLegend.horizontalAlign = 'center';
                    break;
                case 'left':
                case 'right':
                    dataLegend.verticalAlign = 'center';
                    dataLegend.horizontalAlign = legendPos;
                    break;
                case 'topRight':
                    dataLegend.verticalAlign = 'center';
                    dataLegend.horizontalAlign = 'right';
                    break;
            }

            if (legendPos !== 'off') {
                ChartStyleUtil.handleCharacterProps(chartModel, attributeSet.character, dataLegend);
            }
        };

        // operation generators -----------------------------------------------

        /**
         * Generates the undo operations needed to restore this chart legend.
         *
         * @param {SheetOperationGenerator} generator
         *  The operations generator to be filled with the undo operations.
         *
         * @param {Array<Number>} position
         *  The position of the parent chart object in the sheet, as expected
         *  by the method SheetOperationGenerator.generateDrawingOperation().
         *
         * @returns {LegendModel}
         *  A reference to this instance.
         */
        this.generateRestoreOperations = function (generator, position) {
            if (this.hasExplicitAttributes()) {
                var properties = { attrs: this.getExplicitAttributeSet(true) };
                generator.generateDrawingOperation(Operations.SET_CHART_LEGEND_ATTRIBUTES, position, properties, { undo: true });
            }
            return this;
        };

    } // class LegendModel

    // exports ================================================================

    return AttributedModel.extend({ constructor: LegendModel });

});
