/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Stefan Eckert <stefan.eckert@open-xchange.com>
 */

define('io.ox/office/spreadsheet/model/drawing/gridlinemodel', [
    'io.ox/office/editframework/model/attributedmodel',
    'io.ox/office/drawinglayer/view/chartstyleutil',
    'io.ox/office/spreadsheet/utils/operations'
], function (AttributedModel, ChartStyleUtil, Operations) {

    'use strict';

    // class GridlineModel ====================================================

    /**
     * Representation of the grid lines attached to an axis in a chart drawing
     * object.
     *
     * @constructor
     *
     * @extends AttributedModel
     */
    var GridlineModel = AttributedModel.extend({ constructor: function (chartModel, axisId, attrs, dataAxis) {

        // base constructor ---------------------------------------------------

        AttributedModel.call(this, chartModel.getDocModel(), attrs, { families: 'line' });

        // public methods -----------------------------------------------------

        this.refreshInfo = function () {

            if (!dataAxis) { return; }

            var lineAttrs = this.getMergedAttributeSet(true).line;
            ChartStyleUtil.handleLineProps(chartModel, lineAttrs, dataAxis, 'grid');
        };

        // operation generators -----------------------------------------------

        /**
         * Generates the undo operations needed to restore the grid lines.
         *
         * @param {SheetOperationGenerator} generator
         *  The operations generator to be filled with the undo operations.
         *
         * @param {Array<Number>} position
         *  The position of the parent chart object in the sheet, as expected
         *  by the method SheetOperationGenerator.generateDrawingOperation().
         *
         * @returns {GridlineModel}
         *  A reference to this instance.
         */
        this.generateRestoreOperations = function (generator, position) {
            if (this.hasExplicitAttributes()) {
                var properties = { axis: axisId, attrs: this.getExplicitAttributeSet(true) };
                generator.generateDrawingOperation(Operations.SET_CHART_GRIDLINE_ATTRIBUTES, position, properties, { undo: true });
            }
            return this;
        };

    } }); // class GridlineModel

    // exports ================================================================

    return GridlineModel;

});
