/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/spreadsheet/model/drawing/drawingcollection', [
    'io.ox/office/tk/utils',
    'io.ox/office/tk/utils/iteratorutils',
    'io.ox/office/tk/container/valuemap',
    'io.ox/office/tk/render/rectangle',
    'io.ox/office/editframework/utils/attributeutils',
    'io.ox/office/drawinglayer/utils/drawingutils',
    'io.ox/office/drawinglayer/model/indexedcollection',
    'io.ox/office/drawinglayer/view/drawingframe',
    'io.ox/office/spreadsheet/utils/operations',
    'io.ox/office/spreadsheet/utils/sheetutils',
    'io.ox/office/spreadsheet/model/drawing/framenodemanager',
    'io.ox/office/spreadsheet/model/drawing/drawingmodel',
    'io.ox/office/spreadsheet/model/drawing/shapemodel',
    'io.ox/office/spreadsheet/model/drawing/connectormodel',
    'io.ox/office/spreadsheet/model/drawing/imagemodel',
    'io.ox/office/spreadsheet/model/drawing/chartmodel',
    'io.ox/office/spreadsheet/model/drawing/groupmodel',
    'io.ox/office/spreadsheet/model/drawing/text/textframeutils'
], function (Utils, IteratorUtils, ValueMap, Rectangle, AttributeUtils, DrawingUtils, IndexedCollection, DrawingFrame, Operations, SheetUtils, FrameNodeManager, SheetDrawingModel, SheetShapeModel, SheetConnectorModel, SheetImageModel, SheetChartModel, SheetGroupModel, TextFrameUtils) {

    'use strict';

    // convenience shortcuts
    var Interval = SheetUtils.Interval;
    var Address = SheetUtils.Address;

    // maps 'anchorType' attribute values to the internal anchor mode used for anchor attributes
    var INTERNAL_ANCHOR_MODES = {
        twoCell: 'twoCell',
        oneCell: 'oneCell',
        absolute: 'absolute',
        twoCellAsOneCell: 'twoCell',
        twoCellAsAbsolute: 'twoCell'
    };

    // maps 'anchorType' attribute values to display anchor mode defining runtime behavior
    var DISPLAY_ANCHOR_MODES = {
        twoCell: 'twoCell',
        oneCell: 'oneCell',
        absolute: 'absolute',
        twoCellAsOneCell: 'oneCell',
        twoCellAsAbsolute: 'absolute'
    };

    // names of generic (cell independent) anchor attributes
    var GENERIC_ANCHOR_ATTR_NAMES = ['left', 'top', 'width', 'height'];

    // class CellAnchor =======================================================

    /**
     * Represents the position of a single corner of a drawing object that is
     * anchored to a specific spreadsheet cell.
     *
     * @constructor
     *
     * @property {Address} address
     *  The address of the spreadsheet cell containing the corner of the
     *  drawing object.
     *
     * @property {Number} colOffset
     *  The relative offset inside the column, in 1/100 mm.
     *
     * @property {Number} rowOffset
     *  The relative offset inside the row, in 1/100 mm.
     */
    function CellAnchor(address, colOffset, rowOffset) {

        this.address = address.clone();
        this.colOffset = colOffset;
        this.rowOffset = rowOffset;

    } // class CellAnchor

    // class SheetDrawingCollection ===========================================

    /**
     * Represents the drawing collection of a single sheet in a spreadsheet
     * document.
     *
     * @constructor
     *
     * @extends IndexedCollection
     *
     * @param {SheetModel} sheetModel
     *  The sheet model instance containing this collection.
     */
    var SheetDrawingCollection = IndexedCollection.extend({ constructor: function (sheetModel) {

        // self reference
        var self = this;

        // the document model
        var docModel = sheetModel.getDocModel();

        // the application instance
        var app = docModel.getApp();

        // the manager for the DOM drawing frames of the drawing models in this collection
        var frameNodeManager = new FrameNodeManager(docModel);

        // cached settings for drawing model frames, mapped by drawing model UIDs
        var modelFrameCache = new ValueMap();

        // the column and row collections (size and formatting of all columns/rows)
        var colCollection = sheetModel.getColCollection();
        var rowCollection = sheetModel.getRowCollection();

        // special behavior in OOXML documents
        var ooxml = app.isOOXML();

        // base constructor ---------------------------------------------------

        IndexedCollection.call(this, docModel, modelFactory);

        // private methods ----------------------------------------------------

        /**
         * Constructs a drawing model instance for the specified drawing type.
         *
         * @param {DrawingCollection} parentCollection
         *  The parent drawing collection that will contain the new drawing
         *  object.
         *
         * @param {String} drawingType
         *  The type of the drawing model to be created.
         *
         * @param {Object|Null} attributeSet
         *  Initial formatting attributes for the drawing model.
         *
         * @returns {DrawingModel|Null}
         *  The new drawing model instance, if the passed type is supported.
         */
        function modelFactory(parentCollection, drawingType, attributeSet) {

            // create an explicit model for supported drawing types
            switch (drawingType) {
                case 'shape':
                    return new SheetShapeModel(sheetModel, parentCollection, attributeSet);
                case 'connector':
                    return new SheetConnectorModel(sheetModel, parentCollection, attributeSet);
                case 'image':
                    return new SheetImageModel(sheetModel, parentCollection, attributeSet);
                case 'chart':
                    return new SheetChartModel(sheetModel, parentCollection, attributeSet);
                case 'group':
                    return new SheetGroupModel(sheetModel, parentCollection, attributeSet);
                case 'diagram':
                case 'ole':
                case 'undefined':
                    // create a placeholder model for known but unsupported drawing types
                    return new SheetDrawingModel(sheetModel, parentCollection, drawingType, attributeSet);
            }

            return null;
        }

        /**
         * Returns all available information about the position of a drawing
         * object, according to the passed drawing attributes.
         *
         * @param {Object} drawingAttrs
         *  The values of the drawing anchor attributes, and the anchor type.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  - {Boolean} [options.displayAnchor=false]
         *      If set to true, uses the runtime behavior defined by the
         *      'anchorType' formatting attribute as anchor mode, instead of
         *      the internal anchor mode normally used.
         *
         * @returns {Object}
         *  The result object with position information for the passed drawing
         *  attribute set, in the following properties:
         *  - {String} anchorMode
         *      The effective anchor mode (one of 'absolute', 'oneCell', or
         *      'twoCell'), according to the anchor type attribute, and the
         *      passed options.
         *  - {Rectangle} rectHmm
         *      The absolute location of the drawing object in the sheet, in
         *      1/100 of millimeters, independent from the current sheet zoom
         *      factor.
         *  - {Rectangle} rectPx
         *      The absolute location of the drawing object in the sheet, in
         *      pixels, according to the current sheet zoom factor.
         *  - {CellAnchor} startAnchor
         *      The location of the top-left corner of the drawing object, as
         *      cell anchor position.
         *  - {CellAnchor} endAnchor
         *      The location of the bottom-right corner of the drawing object,
         *      as cell anchor position.
         *  - {Boolean} hiddenCols
         *      Whether the drawing object is located entirely in a range of
         *      hidden columns.
         *  - {Boolean} hiddenRows
         *      Whether the drawing object is located entirely in a range of
         *      hidden rows.
         */
        function getCellAnchorDescriptor(drawingAttrs, options) {

            // whether to use the display anchor instead of the anchor type
            var displayAnchor = Utils.getBooleanOption(options, 'displayAnchor', false);
            // the actual anchor mode used to find inactive anchor attributes
            var anchorMode = (displayAnchor ? DISPLAY_ANCHOR_MODES : INTERNAL_ANCHOR_MODES)[drawingAttrs.anchorType] || 'twoCell';
            // the location of the drawing object, in 1/100 mm
            var rectHmm = new Rectangle(0, 0, 0, 0);
            // the location of the drawing object, in pixels
            var rectPx = new Rectangle(0, 0, 0, 0);
            // the location of the top-left corner, as cell anchor
            var startAnchor = new CellAnchor(Address.A1, 0, 0);
            // the location of the bottom-right corner, as cell anchor
            var endAnchor = new CellAnchor(Address.A1, 0, 0);
            // the resulting cell anchor descripor
            var anchorDesc = { anchorMode: anchorMode, rectHmm: rectHmm, rectPx: rectPx, startAnchor: startAnchor, endAnchor: endAnchor };
            // collection entry of the column/row collection
            var entryData = null;

            // get start position of the drawing object
            switch (anchorMode) {
                case 'twoCell':
                case 'oneCell':
                    // left position in 1/100 mm and pixels, start column index and offset
                    rectHmm.left = colCollection.getEntryOffsetHmm(drawingAttrs.startCol, drawingAttrs.startColOffset);
                    rectPx.left = colCollection.getEntryOffset(drawingAttrs.startCol, drawingAttrs.startColOffset);
                    startAnchor.address[0] = drawingAttrs.startCol;
                    startAnchor.colOffset = drawingAttrs.startColOffset;

                    // top position in 1/100 mm and pixels, start row index and offset
                    rectHmm.top = rowCollection.getEntryOffsetHmm(drawingAttrs.startRow, drawingAttrs.startRowOffset);
                    rectPx.top = rowCollection.getEntryOffset(drawingAttrs.startRow, drawingAttrs.startRowOffset);
                    startAnchor.address[1] = drawingAttrs.startRow;
                    startAnchor.rowOffset = drawingAttrs.startRowOffset;
                    break;

                case 'absolute':
                    // left position in 1/100 mm and pixels, start column index and offset
                    entryData = colCollection.getEntryByOffset(drawingAttrs.left);
                    rectHmm.left = entryData.offsetHmm + entryData.relOffsetHmm;
                    rectPx.left = entryData.offset + entryData.relOffset;
                    startAnchor.address[0] = entryData.index;
                    startAnchor.colOffset = entryData.relOffsetHmm;

                    // top position in 1/100 mm and pixels, start row index and offset
                    entryData = rowCollection.getEntryByOffset(drawingAttrs.top);
                    rectHmm.top = entryData.offsetHmm + entryData.relOffsetHmm;
                    rectPx.top = entryData.offset + entryData.relOffset;
                    startAnchor.address[1] = entryData.index;
                    startAnchor.rowOffset = entryData.relOffsetHmm;
                    break;
            }

            // get effective size of the drawing object
            switch (anchorMode) {
                case 'twoCell':
                    // width in 1/100 mm and pixels; end column index and offset
                    rectHmm.width = colCollection.getEntryOffsetHmm(drawingAttrs.endCol, drawingAttrs.endColOffset) - rectHmm.left;
                    rectPx.width = colCollection.getEntryOffset(drawingAttrs.endCol, drawingAttrs.endColOffset) - rectPx.left;
                    endAnchor.address[0] = drawingAttrs.endCol;
                    endAnchor.colOffset = drawingAttrs.endColOffset;

                    // height in 1/100 mm and pixels, end row index and offset
                    rectHmm.height = rowCollection.getEntryOffsetHmm(drawingAttrs.endRow, drawingAttrs.endRowOffset) - rectHmm.top;
                    rectPx.height = rowCollection.getEntryOffset(drawingAttrs.endRow, drawingAttrs.endRowOffset) - rectPx.top;
                    endAnchor.address[1] = drawingAttrs.endRow;
                    endAnchor.rowOffset = drawingAttrs.endRowOffset;
                    break;

                case 'oneCell':
                case 'absolute':
                    // width in 1/100 mm and pixels; end column index and offset
                    entryData = colCollection.getEntryByOffset(rectHmm.left + drawingAttrs.width);
                    rectHmm.width = entryData.offsetHmm + entryData.relOffsetHmm - rectHmm.left;
                    rectPx.width = entryData.offset + entryData.relOffset - rectPx.left;
                    endAnchor.address[0] = entryData.index;
                    endAnchor.colOffset = entryData.relOffsetHmm;

                    // height in 1/100 mm and pixels, end row index and offset
                    entryData = rowCollection.getEntryByOffset(rectHmm.top + drawingAttrs.height);
                    rectHmm.height = entryData.offsetHmm + entryData.relOffsetHmm - rectHmm.top;
                    rectPx.height = entryData.offset + entryData.relOffset - rectPx.top;
                    endAnchor.address[1] = entryData.index;
                    endAnchor.rowOffset = entryData.relOffsetHmm;
                    break;
            }

            // ooxml has special behavior to swap width/height values of the drawing if certan rotation/flip conditions are met.
            // For proper display, those values should be reverted back.
            if (ooxml && DrawingUtils.hasSwappedDimensions(drawingAttrs)) {
                anchorDesc.rectHmm.rotateSelf();
                anchorDesc.rectPx.rotateSelf();
            }

            // whether the drawing is located entirely in hidden columns or rows
            anchorDesc.hiddenCols = colCollection.isIntervalHidden(new Interval(startAnchor.address[0], endAnchor.address[0]));
            anchorDesc.hiddenRows = rowCollection.isIntervalHidden(new Interval(startAnchor.address[1], endAnchor.address[1]));

            return anchorDesc;
        }

        function findAllIntersections(sourceDrawingModel, compareDrawingModels, targetList) {
            var sourceRect = sourceDrawingModel.getRectangleHmm();

            _.each(compareDrawingModels, function (compareDrawingModel) {
                if (compareDrawingModel === sourceDrawingModel) { return; }

                if (sourceRect.overlaps(compareDrawingModel.getRectangleHmm())) {
                    targetList.push(compareDrawingModel);
                }
            });
        }

        // protected methods --------------------------------------------------

        /**
         * Callback handler for the document operation 'copySheet'. Clones all
         * drawing objects from the passed collection into this collection.
         *
         * @param {SheetOperationContext} context
         *  A wrapper representing the 'copySheet' document operation.
         *
         * @param {SheetDrawingCollection} collection
         *  The source collection whose contents will be cloned into this
         *  collection.
         *
         * @throws {OperationError}
         *  If applying the operation fails, e.g. if a required property is
         *  missing in the operation.
         */
        this.applyCopySheetOperation = function (context, collection) {

            // clone the contents of the source collection
            var iterator = collection.createModelIterator();
            IteratorUtils.forEach(iterator, function (drawingModel, result) {

                // clone the source drawing model and insert it into this collection
                // (the embedded drawing models of group objects will be cloned automatically)
                var newDrawingModel = drawingModel.clone(sheetModel, this);
                this.implInsertModel([result.index], newDrawingModel);

                // clone the DOM text contents of the drawing model, and all embedded child models
                (function copyTextFrames(sourceModel) {

                    // copy the DOM text frame of the specified model
                    var sourceFrame = collection.getDrawingFrameForModel(sourceModel);
                    var targetFrame = self.getDrawingFrame(sourceModel.getPosition());
                    TextFrameUtils.copyTextFrame(sourceFrame, targetFrame);

                    // copy the DOM text frames of the embedded models
                    sourceModel.forEachChildModel(copyTextFrames);

                }(drawingModel));
            }, this);
        };

        /**
         * Callback handler for the document operation 'insertDrawing'.
         *
         * @param {SheetOperationContext} context
         *  A wrapper representing the 'insertDrawing' document operation.
         *
         * @param {Array<Number>} position
         *  The insertion position of the new drawing object in this collection
         *  (without leading sheet index).
         */
        this.applyInsertDrawingOperation = function (context, position) {
            var result = this.insertModel(position, context.getStr('type'), context.getOptObj('attrs'));
            context.ensure(result, 'cannot insert drawing');
        };

        /**
         * Callback handler for the document operation 'deleteDrawing'.
         *
         * @param {SheetOperationContext} context
         *  A wrapper representing the 'deleteDrawing' document operation.
         *
         * @param {Array<Number>} position
         *  The position of the drawing object to be deleted in this collection
         *  (without leading sheet index).
         */
        this.applyDeleteDrawingOperation = function (context, position) {
            var result = this.deleteModel(position);
            context.ensure(result, 'cannot delete drawing');
        };

        /**
         * Callback handler for the document operation 'moveDrawing'.
         *
         * @param {SheetOperationContext} context
         *  A wrapper representing the 'moveDrawing' document operation.
         *
         * @param {Array<Number>} position
         *  The position of the drawing object to be moved in this collection
         *  (without leading sheet index).
         */
        this.applyMoveDrawingOperation = function (context, position) {

            // get the target position (must be located in the same sheet for now)
            var targetPos = context.getPos('to');
            context.ensure(targetPos[0] === sheetModel.getIndex(), 'cannot move drawing to another sheet');
            targetPos = targetPos.slice(1);

            // target position must be located in the same parent drawing object
            context.ensure(_.isEqual(position.slice(0, -1), targetPos.slice(0, -1)), 'invalid target position');
            var result = this.moveModel(position, _.last(targetPos));
            context.ensure(result, 'cannot move drawing');
        };

        // public methods -----------------------------------------------------

        /**
         * Returns the DOM container node for the DOM models of all drawing
         * objects in this collection.
         *
         * @returns {HTMLElement}
         *  The DOM container node for the DOM models of all drawing objects in
         *  this collection.
         */
        this.getContainerNode = function () {
            return frameNodeManager.getRootNode();
        };

        /**
         * Returns the DOM drawing frame that represents the drawing model at
         * the specified document position. The DOM drawing frame is used to
         * store the text contents of the drawing objects.
         *
         * @param {Array<Number>} position
         *  The document position of a drawing model to return the DOM drawing
         *  frame for.
         *
         * @returns {jQuery|Null}
         *  The DOM drawing frame for the specified drawing object; or null, if
         *  no drawing frame could be found.
         */
        this.getDrawingFrame = function (position) {
            var drawingModel = this.getModel(position);
            return drawingModel ? frameNodeManager.getDrawingFrame(drawingModel) : null;
        };

        /**
         * Returns the DOM drawing frame that represents the passed drawing
         * model. The DOM drawing frame is used to store the text contents of
         * the drawing objects.
         *
         * @param {DrawingModel} drawingModel
         *  The drawing model instance to return the DOM drawing frame for.
         *
         * @returns {jQuery|Null}
         *  The DOM drawing frame for the passed drawing model; or null, if no
         *  drawing frame could be found.
         */
        this.getDrawingFrameForModel = function (drawingModel) {
            return frameNodeManager.getDrawingFrame(drawingModel);
        };

        /**
         * Refreshes the settings of the DOM drawing frame associated with the
         * passed drawing model, and returns the cached rendering settings for
         * the drawing model.
         *
         * @param {DrawingModel} drawingModel
         *  The drawing model whose DOM drawing frame will be refreshed.
         *
         * @returns {Object|Null}
         *  A descriptor with rendering data for the passed drawing model; or
         *  null, if no data is available for the drawing model. The descriptor
         *  will contain the following properties:
         *  - {jQuery} drawingFrame
         *      The DOM drawing frame associated with the drawing model. This
         *      drawing frame serves as model container for the text contents
         *      of shape objects.
         *  - {Object} explicitAttrSet
         *      A deep copy of the explicit attribute set of the drawing model.
         *  - {Object|Null} textPosition
         *      The position of the text frame inside the drawing frame at a
         *      fixed zoom level of 100%, as object with the properties 'left',
         *      'top', 'right'm, 'bottom', 'width', and 'height', in pixels. If
         *      the drawing model does not contain a text frame, this property
         *      will be null.
         */
        this.refreshDrawingFrameForModel = function (drawingModel) {

            // directly return existing cached settings for the passed drawing model
            var renderSettings = modelFrameCache.get(drawingModel.getUid(), null);
            if (renderSettings) { return renderSettings; }

            // resolve the DOM drawing frame, do nothing if the frame does not exist yet
            var drawingFrame = drawingModel.getModelFrame();
            if (!drawingFrame) { return null; }

            // create a new entry in the drawing model cache
            renderSettings = modelFrameCache.insert(drawingModel.getUid(), { drawingFrame: drawingFrame, textPosition: null });

            // copy explicit formatting attributes (default character formatting)
            renderSettings.explicitAttrSet = drawingModel.getExplicitAttributeSet();
            AttributeUtils.setExplicitAttributes(drawingFrame, renderSettings.explicitAttrSet);

            // set the pixel size of the drawing frame (always for 100% zoom factor to get the original padding!)
            var rectangle = drawingModel.getRectangleHmm();
            if (ooxml && drawingModel.hasSwappedDimensions()) { rectangle.rotateSelf(); }
            var width = Utils.convertHmmToLength(rectangle.width, 'px', 1);
            var height = Utils.convertHmmToLength(rectangle.height, 'px', 1);
            drawingFrame.css({ width: width, height: height });

            // additional processing for the embedded text frame
            TextFrameUtils.withTextFrame(drawingFrame, function (textFrame) {

                // update the text frame position, but not the entire formatting (especially, do not create canvas elements!)
                var mergedAttrSet = drawingModel.getMergedAttributeSet(true);
                DrawingFrame.updateTextFramePosition(app, drawingFrame, mergedAttrSet);

                // reset CSS transformations before measuring DOM positions
                var drawingTransform = drawingFrame.css('transform');
                var textTransform = textFrame.css('transform');
                drawingFrame.css('transform', '');
                textFrame.css('transform', '');

                // store the padding of the text frame to its enclosing drawing frame (to be used by renderers)
                renderSettings.textPosition = Utils.getChildNodePositionInNode(drawingFrame, textFrame);

                // restore the CSS transformations
                drawingFrame.css('transform', drawingTransform);
                textFrame.css('transform', textTransform);
            });

            return renderSettings;
        };

        /**
         * Updates the CSS text formatting of the text contents in the drawing
         * frame associated to the passed drawing model.
         *
         * @param {DrawingModel} drawingModel
         *  The model of the drawing objects whose text formatting will be
         *  updated.
         *
         * @returns {SheetDrawingCollection}
         *  A reference to this instance.
         */
        this.updateTextFormatting = function (drawingModel) {
            frameNodeManager.updateTextFormatting(drawingModel);
            return this;
        };

        /**
         * Returns the inactive cell anchor attributes of the passed attribute
         * set, according to the anchor mode and anchor attributes.
         *
         * @param {Object} drawingAttrs
         *  The values of the drawing anchor attributes, and the anchor type.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  - {Boolean} [options.displayAnchor=false]
         *      If set to true, uses the runtime behavior defined by the
         *      'anchorType' formatting attribute as anchor mode, instead of
         *      the internal anchor mode normally used.
         *
         * @returns {Object}
         *  The current values of the inactive anchor attributes, according to
         *  the anchor mode.
         */
        this.getInactiveCellAnchorAttributes = function (drawingAttrs, options) {

            // current position of the drawing object
            var anchorDesc = getCellAnchorDescriptor(drawingAttrs, options);
            // the location of the drawing object, in 1/100 mm
            var rectHmm = anchorDesc.rectHmm;
            // the resulting inactive anchor attributes
            var anchorAttrs = {};

            // add absolute position, if anchor starts in a cell; otherwise add the start cell address
            switch (anchorDesc.anchorMode) {
                case 'twoCell':
                case 'oneCell':
                    anchorAttrs.left = rectHmm.left;
                    anchorAttrs.top = rectHmm.top;
                    break;
                case 'absolute':
                    var startAnchor = anchorDesc.startAnchor;
                    anchorAttrs.startCol = startAnchor.address[0];
                    anchorAttrs.startColOffset = startAnchor.colOffset;
                    anchorAttrs.startRow = startAnchor.address[1];
                    anchorAttrs.startRowOffset = startAnchor.rowOffset;
                    break;
            }

            // add absolute size, if anchor is two-cell; otherwise add the end cell address
            switch (anchorDesc.anchorMode) {
                case 'twoCell':
                    anchorAttrs.width = rectHmm.width;
                    anchorAttrs.height = rectHmm.height;
                    break;
                case 'oneCell':
                case 'absolute':
                    var endAnchor = anchorDesc.endAnchor;
                    anchorAttrs.endCol = endAnchor.address[0];
                    anchorAttrs.endColOffset = endAnchor.colOffset;
                    anchorAttrs.endRow = endAnchor.address[1];
                    anchorAttrs.endRowOffset = endAnchor.rowOffset;
                    break;
            }

            return anchorAttrs;
        };

        /**
         * Returns the sheet rectangle in pixels covered by the passed drawing
         * object.
         *
         * @param {DrawingModel} drawingModel
         *  The drawing object to return the rectangle for.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  - {Boolean} [options.absolute=false]
         *      If set to true, this method will return the absolute sheet
         *      position also for drawing objects embedded into other drawing
         *      objects. By default, this method returns the location of the
         *      specified drawing object relative to its parent object.
         *
         * @returns {Rectangle|Null}
         *  The location of the specified drawing object in pixels according to
         *  the current sheet zoom factor, either relative to its parent object
         *  (sheet area for top-level drawing objects), or always absolute in
         *  the sheet area (according to the passed options); or null, if the
         *  drawing object is hidden by itself, or is located in a hidden group
         *  object, or is located entirely in hidden columns or rows and thus
         *  is invisible.
         */
        this.getRectangleForModel = function (drawingModel, options) {

            // always return null for invisible drawing objects
            if (!drawingModel.isEffectivelyVisible()) { return null; }

            // the top-level drawing model containing the passed drawing model
            var rootModel = drawingModel.getRootModel();
            // the complete cell anchor descriptor of the top-level drawing
            var anchorDesc = getCellAnchorDescriptor(rootModel.getMergedAttributeSet(true).drawing);
            // return null for all drawing objects (top-level and embedded) in hidden columns/rows
            if (anchorDesc.hiddenCols || anchorDesc.hiddenRows) { return null; }

            // the resulting rectangle in pixels
            var rectangle = null;
            // the direct parent of the passed drawing model (null for top-level drawings)
            var parentModel = drawingModel.getParentModel();
            // whether to convert to effective absolute coordinates
            var absolute = Utils.getBooleanOption(options, 'absolute', false);

            if (parentModel) {

                // the exact location of the drawing model in 1/100 mm
                var childRectHmm = absolute ? drawingModel.getEffectiveRectangleHmm() : drawingModel.getRectangleHmm();
                // the exact location of the parant drawing model in 1/100 mm
                var parentRectHmm = absolute ? anchorDesc.rectHmm.rotatedBoundingBox(rootModel.getRotationRad()) : parentModel.getRectangleHmm();
                // the exact location of the parant drawing model in pixels
                var parentRectPx = absolute ? anchorDesc.rectPx.rotatedBoundingBox(rootModel.getRotationRad()) : this.getRectangleForModel(parentModel);

                // conversion factor for left position and width
                var widthRatio = (parentRectHmm.width > 0) ? (parentRectPx.width / parentRectHmm.width) : 0;
                // conversion factor for top position and height
                var heightRatio = (parentRectHmm.height > 0) ? (parentRectPx.height / parentRectHmm.height) : 0;

                // the location of the drawing object, relative to its parent object
                rectangle = childRectHmm.clone().scaleSelf(widthRatio, heightRatio).roundSelf();
            } else {
                // position of top-level drawings is contained in the anchor descriptor
                rectangle = absolute ? anchorDesc.rectPx.rotatedBoundingBox(rootModel.getRotationRad()) : anchorDesc.rectPx;
            }

            // keep minimum size of 5 pixels for shapes, or 1 pixel for connectors
            var minSize = drawingModel.getMinSize();
            if (rectangle.width < minSize) {
                rectangle.left = Math.max(0, rectangle.left - Math.round((minSize - rectangle.width) / 2));
                rectangle.width = minSize;
            }
            if (rectangle.height < minSize) {
                rectangle.top = Math.max(0, rectangle.top - Math.round((minSize - rectangle.height) / 2));
                rectangle.height = minSize;
            }

            return rectangle;
        };

        /**
         * Returns a drawing attribute set containing the anchor attributes
         * needed to set the position of a drawing object to the passed sheet
         * rectangle.
         *
         * @param {Rectangle} rectangle
         *  The position of a drawing object in the sheet in pixels, according
         *  to the current sheet zoom factor.
         *
         * @returns {Object}
         *  A drawing attribute set with the anchor attributes (contained in
         *  the sub-object 'drawing' of the resulting attribute set) for the
         *  new position of the drawing object.
         */
        this.getAttributeSetForRectangle = function (rectangle) {

            // collection entry of the column/row collection
            var entryData = null;
            // the new anchor attributes
            var anchorAttrs = {};

            // horizontal start position
            entryData = colCollection.getEntryByOffset(rectangle.left, { pixel: true });
            anchorAttrs.left = entryData.offsetHmm + entryData.relOffsetHmm;
            anchorAttrs.startCol = entryData.index;
            anchorAttrs.startColOffset = entryData.relOffsetHmm;

            // object width and horizontal end position
            entryData = colCollection.getEntryByOffset(rectangle.right(), { pixel: true });
            anchorAttrs.width = entryData.offsetHmm + entryData.relOffsetHmm - anchorAttrs.left;
            anchorAttrs.endCol = entryData.index;
            anchorAttrs.endColOffset = entryData.relOffsetHmm;

            // vertical start position
            entryData = rowCollection.getEntryByOffset(rectangle.top, { pixel: true });
            anchorAttrs.top = entryData.offsetHmm + entryData.relOffsetHmm;
            anchorAttrs.startRow = entryData.index;
            anchorAttrs.startRowOffset = entryData.relOffsetHmm;

            // object height and vertical end position
            entryData = rowCollection.getEntryByOffset(rectangle.bottom(), { pixel: true });
            anchorAttrs.height = entryData.offsetHmm + entryData.relOffsetHmm - anchorAttrs.top;
            anchorAttrs.endRow = entryData.index;
            anchorAttrs.endRowOffset = entryData.relOffsetHmm;

            return { drawing: anchorAttrs };
        };

        // operation generators -----------------------------------------------

        /**
         * Generates the operations, and the undo operations, to insert new
         * drawing objects into this collection.
         *
         * @param {SheetOperationGenerator} generator
         *  The operations generator to be filled with the operations.
         *
         * @param {Array<Number>} position
         *  The position of the first new drawing object. All drawing objects
         *  will be inserted subsequently.
         *
         * @param {Array<Object>} drawingDescs
         *  An array of descriptors with the types and attribute sets for the
         *  drawing objects to be inserted. Each array element MUST be an
         *  object with the following properties:
         *  - {String} drawingDesc.type
         *      The type identifier of the drawing object.
         *  - {Object} drawingDesc.attrs
         *      Initial attributes for the drawing object, especially the
         *      anchor attributes specifying the physical position of the
         *      drawing object in the sheet.
         *
         * @param {Function} [callback]
         *  A callback function that will be invoked every time after a new
         *  'insertDrawing' operation has been generated for a drawing object.
         *  Allows to create additional operations for the drawing objects.
         *  Receives the following parameters:
         *  (1) {SheetOperationGenerator} generator
         *      The operations generator (already containing the initial
         *      'insertDrawing' operation of the current drawing).
         *  (2) {Number} sheet
         *      The zero-based index of this sheet.
         *  (3) {Array<Number>} position
         *      The effective document position of the current drawing object
         *      in the sheet, as inserted into the 'insertDrawing' operation.
         *  (4) {Object} data
         *      The data element from the array parameter 'drawingData' that is
         *      currently processed.
         *  The callback function may return a promise to defer processing of
         *  subsequent drawing objects.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved when the operations have been
         *  generated successfully, or that will be rejected on any error.
         */
        this.generateInsertDrawingOperations = function (generator, position, drawingDescs, callback) {

            // the index of the sheet containing this collection
            var sheet = sheetModel.getIndex();

            // generate all operations for the new drawing objects, and the undo operations
            return this.iterateArraySliced(drawingDescs, function (drawingDesc) {

                // generate the undo operations that delete the new drawings (in reversed order)
                generator.generateDrawingOperation(Operations.DELETE_DRAWING, position, null, { undo: true, prepend: true });

                // create a deep copy of the original attributes
                var attributeSet = _.copy(drawingDesc.attrs, true);

                // add the generic anchor attributes of top-level drawing objects to the attribute set
                if (position.length === 1) {

                    // the merged attribute set (with default values for all missing attributes)
                    var mergedAttributeSet = docModel.getDrawingStyles().getMergedAttributes(drawingDesc.attrs);
                    // the inactive anchor attributes
                    var anchorAttrs = self.getInactiveCellAnchorAttributes(mergedAttributeSet.drawing);
                    // the drawing attributes in the attribute set (create on demand)
                    var drawingAttrs = attributeSet.drawing || (attributeSet.drawing = {});

                    GENERIC_ANCHOR_ATTR_NAMES.forEach(function (attrName) {
                        if (attrName in anchorAttrs) {
                            drawingAttrs[attrName] = anchorAttrs[attrName];
                        }
                    });
                }

                // create the 'insertDrawing' operation at the current position
                generator.generateDrawingOperation(Operations.INSERT_DRAWING, position, { type: drawingDesc.type, attrs: attributeSet });
                // invoke callback function for more operations (may return a promise)
                var result = callback ? callback.call(self, generator, sheet, position.slice(), drawingDesc) : null;

                // adjust document position for the next drawing object
                position = position.slice();
                position[position.length - 1] += 1;

                // callback may want to defer processing the following drawings
                return result;
            }, 'SheetDrawingCollection.generateInsertDrawingOperations');
        };

        /**
         * Generates the operations, and the undo operations, to delete the
         * specified drawing objects from this collection.
         *
         * @param {SheetOperationGenerator} generator
         *  The operations generator to be filled with the operations.
         *
         * @param {Array<Array<Number>>} positions
         *  An array with positions of the drawing objects to be deleted.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved when the operations have been
         *  generated successfully, or that will be rejected on any error.
         */
        this.generateDeleteDrawingOperations = function (generator, positions) {

            // sort by position, remove positions of embedded drawings whose parents will be deleted too
            DrawingUtils.optimizeDrawingPositions(positions);

            // create the delete operations (in reversed order), and the according undo operations
            return this.iterateArraySliced(positions, function (position) {

                // the drawing model to be deleted
                var drawingModel = self.getModel(position);
                // a local generator to be able to prepend all undo operations at once
                var generator2 = sheetModel.createOperationGenerator();

                // generate the operations, and insert them into the passed generator
                drawingModel.generateDeleteOperations(generator2);
                generator.appendOperations(generator2);
                generator.prependOperations(generator2, { undo: true });

            }, 'SheetDrawingCollection.generateDeleteDrawingOperations', { reverse: true });
        };

        /**
         * Generates the operations, and the undo operations, to change the
         * formatting attributes of the specified drawing objects individually.
         *
         * @param {SheetOperationGenerator} generator
         *  The operations generator to be filled with the operations.
         *
         * @param {Array<Object>} drawingDescs
         *  An array of descriptors with the positions and attribute sets for
         *  the drawing objects to be changed. Each descriptor MUST contain the
         *  following properties:
         *  - {Array<Number>} drawingDesc.position
         *      The document position of the drawing object to be changed.
         *  - {Object} drawingDesc.attrs
         *      The (incomplete) attribute set to be applied at the drawing
         *      object.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved when the operations have been
         *  generated successfully.
         */
        this.generateChangeDrawingOperations = function (generator, drawingDescs) {
            return this.iterateArraySliced(drawingDescs, function (drawingDesc) {
                var drawingModel = self.getModel(drawingDesc.position);
                drawingModel.generateChangeOperations(generator, drawingDesc.attrs);
            }, 'SheetDrawingCollection.generateChangeDrawingOperations');
        };

        /**
         * Generates the operations, and the undo operations, to apply the same
         * formatting attributes to multiple drawing objects.
         *
         * @param {SheetOperationGenerator} generator
         *  The operations generator to be filled with the operations.
         *
         * @param {Array<Array<Number>>} positions
         *  An array with positions of the drawing objects to be modified.
         *
         * @param {Object} attributeSet
         *  An (incomplete) attribute set to be applied at all drawing objects.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved when the operations have been
         *  generated successfully.
         */
        this.generateFormatDrawingOperations = function (generator, positions, attributeSet) {
            return this.iterateArraySliced(positions, function (position) {
                var drawingModel = self.getModel(position);
                drawingModel.generateChangeOperations(generator, attributeSet);
            }, 'SheetDrawingCollection.generateFormatDrawingOperations');
        };

        /**
         * Generates the operations and undo operations to update and restore
         * the inactive anchor attributes of the top-level drawing objects,
         * after the size or visibility of the columns or rows in the sheet has
         * been changed.
         *
         * @param {SheetOperationGenerator} generator
         *  The operations generator to be filled with the operations.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved when all operations have been
         *  generated.
         */
        this.generateUpdateAnchorOperations = function (generator) {
            return this.iterateSliced(this.createModelIterator(), function (drawingModel) {
                drawingModel.generateUpdateAnchorOperations(generator);
            }, 'SheetDrawingCollection.generateUpdateAnchorOperations');
        };

        /**
         * Generates the operations and undo operations to update and restore
         * the formula expressions of the source links in all drawing objects
         * that refer to some cells in this document.
         *
         * @param {SheetOperationGenerator} generator
         *  The operations generator to be filled with the operations.
         *
         * @param {Object} changeDesc
         *  The properties describing the document change. The properties that
         *  are expected in this descriptor depend on the change type in its
         *  'type' property. See method TokenArray.resolveOperation() for more
         *  details.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved when all operations have been
         *  generated.
         */
        this.generateUpdateFormulaOperations = function (generator, changeDesc) {

            // iterate in reversed order, moving cells in a sheet (including inserting/deleting columns or rows) may delete drawings
            var iterator = this.createModelIterator({ deep: true, reverse: true });
            return this.iterateSliced(iterator, function (drawingModel) {
                // a local generator to be able to prepend all undo operations at once (for deleted drawings)
                var generator2 = sheetModel.createOperationGenerator();
                // generate the operations, and insert them into the passed generator
                drawingModel.generateUpdateFormulaOperations(generator2, changeDesc);
                generator.appendOperations(generator2);
                generator.prependOperations(generator2, { undo: true });
            }, 'SheetDrawingCollection.generateUpdateFormulaOperations');
        };

        this.generateReorderDrawingOperations = function (generator, positions, moveType) {

            var drawings = positions.map(this.getModel, this);
            var modelCount = self.getModelCount();
            var others = [];
            var intersections;

            _.times(modelCount, function (i) {
                var otherModel = self.getModel([i]);
                if (!_.contains(drawings, otherModel)) {
                    others.push(otherModel);
                }
            });

            if (moveType === 'backward' || moveType === 'forward') {
                intersections = [];

                drawings.forEach(function (drawing) {
                    findAllIntersections(drawing, others, intersections);
                    others = _.difference(others, intersections);
                });
            }

            if (!intersections || !intersections.length) {
                intersections = others;
            }

            if (intersections.length) {
                var afterInsertDiff;
                var i;
                var currentIndex;
                var insertIndex;

                if (moveType === 'front' || moveType === 'forward') {
                    drawings.reverse();

                    afterInsertDiff = 0;

                    if (moveType === 'front') {
                        insertIndex = null;
                    } else {
                        currentIndex = positions[0][0];
                        insertIndex = null;
                        for (i = 0; i < intersections.length; i++) {
                            if (intersections[i].getPosition()[0] > currentIndex) {
                                insertIndex = i;
                                break;
                            }
                        }

                    }
                } else {
                    afterInsertDiff = 1;

                    if (moveType === 'back') {
                        insertIndex = -1;
                    } else {
                        currentIndex = positions[0][0];
                        insertIndex = -1;
                        for (i = intersections.length - 1; i >= 0; i--) {
                            if (intersections[i].getPosition()[0] < currentIndex) {
                                insertIndex = i;
                                break;
                            }
                        }
                    }
                }

                var toIndex = 0;
                if (insertIndex === null) {
                    toIndex = modelCount - 1;
                } else if (insertIndex >= 0) {
                    toIndex = intersections[insertIndex].getPosition()[0];
                }

                positions.forEach(function (position) {
                    var fromIndex = position[0];
                    if (fromIndex !== toIndex) {
                        generator.generateMoveDrawingOperation([fromIndex], [toIndex]);
                        generator.generateMoveDrawingOperation([toIndex], [fromIndex], { undo: true, prepend: true });
                    }
                    toIndex += afterInsertDiff;
                });
            }
            return $.when();
        };

        // initialization -----------------------------------------------------

        // create the DOM drawing frame for a new drawing model
        this.on('insert:drawing', function (event, drawingModel) {
            frameNodeManager.createDrawingFrame(drawingModel, { internal: true });
        });

        // remove the DOM drawing frame of a deleted drawing model
        this.on('delete:drawing', function (event, drawingModel) {
            frameNodeManager.removeDrawingFrame(drawingModel);
            modelFrameCache.remove(drawingModel.getUid());
        });

        // move the DOM drawing frame of a drawing model to its new position (Z order)
        this.on('move:drawing', function (event, drawingModel) {
            frameNodeManager.moveDrawingFrame(drawingModel);
        });

        // remove cached model frame settings if formatting has changed
        this.on('change:drawing', function (event, drawingModel, changeType) {
            if (changeType === 'attributes') {
                modelFrameCache.remove(drawingModel.getUid());
            }
        });

        // remove cached model frame settings if text contents have changed
        this.on('change:drawing:text', function (event, drawingModel) {
            modelFrameCache.remove(drawingModel.getUid());
        });

        // destroy all class members
        this.registerDestructor(function () {
            // drawing models are owned by the base class DrawingCollection, do not destroy them here!
            frameNodeManager.destroy();
            modelFrameCache.clear();
            self = app = docModel = null;
            sheetModel = colCollection = rowCollection = null;
            frameNodeManager = modelFrameCache = null;
        });

    } }); // class SheetDrawingCollection

    // exports ================================================================

    return SheetDrawingCollection;

});
