/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Stefan Eckert <stefan.eckert@open-xchange.com>
 */

define('io.ox/office/spreadsheet/model/drawing/dataseriesmodel', [
    'io.ox/office/editframework/model/attributedmodel',
    'io.ox/office/spreadsheet/utils/operations',
    'io.ox/office/spreadsheet/utils/sheetutils',
    'io.ox/office/spreadsheet/model/drawing/sourcelinkmixin'
], function (AttributedModel, Operations, SheetUtils, SourceLinkMixin) {

    'use strict';

    // convenience shortcuts
    var MoveDescriptor = SheetUtils.MoveDescriptor;

    // class DataSeriesModel ==================================================

    /**
     * Representation of a single data series in a chart object.
     * model.
     *
     * @constructor
     *
     * @extends AttributedModel
     * @extends SourceLinkMixin
     *
     * @param {SheetModel} sheetModel
     *  The sheet model that contains the chart object with this data series.
     */
    var DataSeriesModel = AttributedModel.extend({ constructor: function (sheetModel, initAttributes) {

        var docModel = sheetModel.getDocModel();

        // base constructors --------------------------------------------------

        AttributedModel.call(this, docModel, initAttributes, { families: 'series fill line' });
        SourceLinkMixin.call(this, sheetModel, 'series.title series.names series.values series.bubbles');

        // operation generators -----------------------------------------------

        /**
         * Generates the undo operations needed to restore this data series.
         *
         * @param {SheetOperationGenerator} generator
         *  The operations generator to be filled with the undo operations.
         *
         * @param {Array<Number>} position
         *  The position of the parent chart object in the sheet, as expected
         *  by the method SheetOperationGenerator.generateDrawingOperation().
         *
         * @param {Number} index
         *  The index of this data series in the collection of all data series.
         *
         * @returns {DataSeriesModel}
         *  A reference to this instance.
         */
        this.generateRestoreOperations = function (generator, position, index) {
            var properties = { series: index, attrs: this.getExplicitAttributeSet() };
            generator.generateDrawingOperation(Operations.INSERT_CHART_DATASERIES, position, properties, { undo: true });
            return this;
        };

        /**
         * Generates the operations and undo operations needed to change the
         * formatting attributes of this data series.
         *
         * @param {SheetOperationGenerator} generator
         *  The operations generator to be filled with the operations.
         *
         * @param {Array<Number>} position
         *  The position of the parent chart object in the sheet, as expected
         *  by the method SheetOperationGenerator.generateDrawingOperation().
         *
         * @param {Number} index
         *  The index of this data series in the collection of all data series.
         *
         * @param {Object} attributeSet
         *  The incomplete attribute set with all formatting attributes to be
         *  changed.
         *
         * @returns {DataSeriesModel}
         *  A reference to this instance.
         */
        this.generateChangeOperations = function (generator, position, index, attributeSet) {
            var undoAttrSet = this.getUndoAttributeSet(attributeSet);
            if (!_.isEmpty(undoAttrSet)) {
                generator.generateDrawingOperation(Operations.SET_CHART_DATASERIES_ATTRIBUTES, position, { series: index, attrs: undoAttrSet }, { undo: true });
                generator.generateDrawingOperation(Operations.SET_CHART_DATASERIES_ATTRIBUTES, position, { series: index, attrs: attributeSet });
            }
            return this;
        };

        /**
         * Generates the operations and undo operations to update or restore
         * the formula expressions of the source links of this data series.
         *
         * @param {SheetOperationGenerator} generator
         *  The operations generator to be filled with the operations.
         *
         * @param {Array<Number>} position
         *  The position of the parent chart object in the sheet, as expected
         *  by the method SheetOperationGenerator.generateDrawingOperation().
         *
         * @param {Number} index
         *  The index of this data series in the collection of all data series.
         *
         * @param {Object} changeDesc
         *  The properties describing the document change. The properties that
         *  are expected in this descriptor depend on the change type in its
         *  'type' property. See method TokenArray.resolveOperation() for more
         *  details.
         */
        this.generateUpdateFormulaOperations = function (generator, position, index, changeDesc) {

            // when the value range will be deleted (e.g. while deleting a column), remove the entire data series from the chart
            if (changeDesc.type === 'moveCells') {
                var valueRanges = this.resolveRanges('series.values');
                if (valueRanges && !valueRanges.empty() && MoveDescriptor.transformRanges(valueRanges, changeDesc.moveDescs).empty()) {
                    this.generateRestoreOperations(generator, position, index);
                    generator.generateDrawingOperation(Operations.DELETE_CHART_DATASERIES, position, { series: index });
                    return;
                }
            }

            // mix-in class generates the necessary change operations
            return this.implGenerateUpdateFormulaOperations(generator, Operations.SET_CHART_DATASERIES_ATTRIBUTES, position, changeDesc, { series: index });
        };

        // initialization -----------------------------------------------------

        // destroy all class members on destruction
        this.registerDestructor(function () {
            docModel = sheetModel = initAttributes = null;
        });

    } }); // class DataSeriesModel

    // exports ================================================================

    return DataSeriesModel;

});
