/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Carsten Driesner <carsten.driesner@open-xchange.com>
 * @author Stefan Eckert <stefan.eckert@open-xchange.com>
 */

define('io.ox/office/spreadsheet/model/drawing/chartmodel', [
    'io.ox/office/tk/utils',
    'io.ox/office/tk/locale/localedata',
    'io.ox/office/drawinglayer/model/chartmodel',
    'io.ox/office/drawinglayer/view/chartstyleutil',
    'io.ox/office/spreadsheet/model/drawing/chartformatter',
    'io.ox/office/spreadsheet/utils/sheetutils',
    'io.ox/office/spreadsheet/model/formula/tokenarray',
    'io.ox/office/spreadsheet/model/drawing/drawingmodelmixin',
    'io.ox/office/spreadsheet/model/drawing/dataseriesmodel',
    'io.ox/office/spreadsheet/model/drawing/titlemodel',
    'io.ox/office/spreadsheet/model/drawing/axismodel',
    'io.ox/office/spreadsheet/model/drawing/legendmodel',
    'io.ox/office/spreadsheet/model/drawing/manualcolorchart',
    'io.ox/office/drawinglayer/lib/canvasjs.min'
], function (Utils, LocaleData, ChartModel, ChartStyleUtil, ChartFormatter, SheetUtils, TokenArray, DrawingModelMixin, DataSeriesModel, TitleModel, AxisModel, LegendModel, ManualColorHandler, CanvasJS) {

    'use strict';

    // convenience shortcuts
    var Range3D = SheetUtils.Range3D;
    var RangeArray = SheetUtils.RangeArray;
    var Range3DArray = SheetUtils.Range3DArray;

    var SUPPORTED_COMBINED = /^(column|area|line|scatter)/;

    var SUPPORTED = /^(column|bar|area|line|scatter|bubble|pie|donut)/;

    var VALID = /^(column|bar|area|line|scatter|bubble|pie|donut|sunburst|ofPie)/;

    var TYPES = {
        'column standard': 'column',
        'column clustered': 'column',
        'column stacked': 'stackedColumn',
        'column percentStacked': 'stackedColumn100',
        'bar standard': 'bar',
        'bar clustered': 'bar',
        'bar stacked': 'stackedBar',
        'bar percentStacked': 'stackedBar100',
        'area standard': 'area',
        'area stacked': 'stackedArea',
        'area percentStacked': 'stackedArea100',
        'line standard': 'line',
        'line clustered': 'line',
        'line stacked': 'line',
        'line percentStacked': 'spline',
        'line standard curved': 'spline',
        'line clustered curved': 'spline',
        'line stacked curved': 'spline',
        'line percentStacked curved': 'spline',
        'scatter standard': 'line',
        'scatter standard curved': 'spline',
        'bubble standard': 'bubble',
        'pie standard': 'pie',
        'donut standard': 'doughnut',

        'sunburst standard': 'doughnut',
        'ofPie standard': 'pie'
    };

    var MARKER_TYPES = {
        circle: 'circle',
        dot: 'circle',
        square: 'square',
        triangle: 'triangle',
        x: 'cross',
        none: 'none'
    };

    var MARKER_LIST = ['circle', 'square', 'triangle', 'cross'];

    function parseChartType(type, stacking, curved) {
        type = type.replace('2d', '').replace('3d', '');

        if (stacking.length) {
            type = type + ' ' + stacking;
        }
        if (curved && /(line|scatter)/.test(type)) {
            type += ' curved';
        }
        return type;
    }

    function parseCJSType(type, stacking, curved) {
        var t = parseChartType(type, stacking, curved);
        var res = TYPES[t];

        if (!res) {
            Utils.warn('no correct charttype found', t);
            if (!type) {
                res = 'column';
            } else if (!type.indexOf('radar') || !type.indexOf('boxWhisker')) {
                res = 'line';
            } else {
                res = 'column';
            }
        }

        if (res === 'line' && curved === true) {
            res = 'spline';
        }
        return res;
    }

    function isLineType(seriesType) {
        if (seriesType.indexOf('line') >= 0 || seriesType.indexOf('scatter') >= 0) {
            return true;
        }
        return false;
    }

    // class SheetChartModel ==================================================

    /**
     * A class that implements a chart model to be inserted into a sheet of a
     * spreadsheet document.
     *
     * @constructor
     *
     * @extends ChartModel
     * @extends DrawingModelMixin
     *
     * @param {SheetModel} sheetModel
     *  The model instance of the sheet that contains this chart object.
     *
     * @param {DrawingCollection} parentCollection
     *  The parent drawing collection that will contain this drawing object.
     *
     * @param {Object} [initAttributes]
     *  Initial formatting attribute set for this chart model.
     */
    var SheetChartModel = ChartModel.extend({ constructor: function (sheetModel, parentCollection, initAttributes) {

        var self = this;

        var docModel = sheetModel.getDocModel();

        var highlightTokenArray = new TokenArray(sheetModel, 'link');

        var data = {
            animationEnabled: false,
            culture:  'en',
            bg: 'white',
            backgroundColor: 'transparent',
            axisX: { stripLines: [], labelAutoFit: true, labelAngle: 0 },
            axisY: { stripLines: [], labelAutoFit: true, labelAngle: 0 },
            axisX2: { stripLines: [], labelAutoFit: true, labelAngle: 0 },
            axisY2: { stripLines: [], labelAutoFit: true, labelAngle: 0 },
            axisZ: { stripLines: [] },
            creditHref: '',
            creditText: '',
            title: { text: '' },
            legend: {},
            data: [],
            series: [],
            toolTip: {
                backgroundColor: 'black',
                borderColor: 'black',
                fontColor: 'white',
                cornerRadius: 3,
                borderThickness: 4,
                fontStyle: 'normal',
                contentFormatter: function (e) {
                    var content = ' ';
                    for (var i = 0; i < e.entries.length; i++) {
                        if (e.entries[i].dataPoint.label) {
                            content += e.entries[i].dataPoint.label;
                        } else {
                            content += e.entries[i].dataPoint.x;
                        }
                        content += ': ';
                        if (e.entries[i].dataPoint.name) {
                            content += e.entries[i].dataPoint.name;
                        } else {
                            content += e.entries[i].dataPoint.y;
                        }
                    }

                    return Utils.escapeHTML(content);
                }
            }
        };

        var indexLabelAttrs = {
            fontSize: 10,
            color: { type: 'auto' },
            fontName: 'Arial'
        };

        var colorHandler = null;
        var chartFormatter = null;

        // base constructors --------------------------------------------------

        ChartModel.call(this, parentCollection, initAttributes);
        DrawingModelMixin.call(this, sheetModel, cloneConstructor, {
            generateRestoreOperations: generateRestoreOperations,
            generateUpdateFormulaOperations: generateUpdateFormulaOperations
        });

        /**
         * this is only called by the Drawingframe to initialize the REAL CanvasJS-chart
         */
        this.getModelData = function () {
            return data;
        };

        var axisModelMap = {
            x: new AxisModel(this, 'x'),
            y: new AxisModel(this, 'y'),
            x2: new AxisModel(this, 'x2'),
            y2: new AxisModel(this, 'y2'),
            z: new AxisModel(this, 'z')
        };

        var mainTitleModel = new TitleModel(this, 'main', data.title, 'text');

        var legendModel = new LegendModel(this, {}, data.legend);

        // private methods ----------------------------------------------------

        /**
         * Returns a clone of this chart model for the specified sheet. Used by
         * the implementation of the public clone() method defined by the class
         * DrawingModelMixin.
         *
         * @param {SheetModel} targetModel
         *  The sheet model that will contain the cloned chart model.
         *
         * @param {DrawingCollection} targetCollection
         *  The drawing collection that will contain the cloned chart model.
         *
         * @returns {SheetChartModel}
         *  A clone of this chart model, initialized for ownership by the
         *  passed target drawing collection.
         */
        function cloneConstructor(targetModel, targetCollection) {
            // pass private clone data as hidden argument to the constructor
            return new SheetChartModel(targetModel, targetCollection, this.getExplicitAttributeSet(true), this.getCloneData());
        }

        /**
         * Generates additional undo operations needed to restore the contents
         * of this chart object, after it has been restored with an initial
         * 'insertDrawing' operation.
         *
         * @param {SheetOperationGenerator} generator
         *  The operations generator to be filled with the undo operations.
         *
         * @param {Array<Number>} position
         *  The position of this chart object in the sheet, as expected by the
         *  method SheetOperationGenerator.generateDrawingOperation().
         */
        function generateRestoreOperations(generator, position) {

            // restore all data series of the chart
            data.series.forEach(function (seriesData, index) {
                seriesData.model.generateRestoreOperations(generator, position, index);
            });

            // restore all axes of the chart
            _.each(axisModelMap, function (axisModel, axisId) {
                // workaround for Bug 46966
                if (axisId === 'z') { return; }

                axisModel.generateRestoreOperations(generator, position);
            });

            // restore the main title and legend
            mainTitleModel.generateRestoreOperations(generator, position);
            legendModel.generateRestoreOperations(generator, position);
        }

        /**
         * Generates the operations and undo operations to update or restore
         * the formula expressions of the source links in this chart object.
         *
         * @param {SheetOperationGenerator} generator
         *  The operations generator to be filled with the operations.
         *
         * @param {Array<Number>} position
         *  The position of this drawing object in the sheet, as expected by
         *  the method SheetOperationGenerator.generateDrawingOperation().
         *
         * @param {Object} changeDesc
         *  The properties describing the document change. The properties that
         *  are expected in this descriptor depend on the change type in its
         *  'type' property. See method TokenArray.resolveOperation() for more
         *  details.
         */
        function generateUpdateFormulaOperations(generator, position, changeDesc) {
            // generate the change operations for the source links in all data series
            // (in reverse order, to get the correct indexes when deleting entire data series)
            Utils.iterateArray(data.series, function (seriesData, index) {
                var generator2 = sheetModel.createOperationGenerator();
                seriesData.model.generateUpdateFormulaOperations(generator2, position, index, changeDesc);
                generator.appendOperations(generator2);
                // prepend undo operations to restore the data series (needed for correct series array index)
                generator.prependOperations(generator2, { undo: true });
            }, { reverse: true });

            // update the source links of the axis titles
            _.each(axisModelMap, function (axisModel) {
                axisModel.generateUpdateFormulaOperations(generator, position, changeDesc);
            });

            // update the source link of the main title
            mainTitleModel.generateUpdateFormulaOperations(generator, position, changeDesc);
        }

        /**
         * Debounced version of the public method SheetChartModel.refresh()
         * with increased delay time.
         */
        var refreshChartDebounced = this.createDebouncedMethod('SheetChartModel.refreshChartDebounced', null, innerRefresh, { delay: 500 });

        /**
         * Updates the visibility of data points, after columns or rows have
         * been shown or hidden in a sheet.
         */
        function updateColRowVisibility(sheet, interval, columns) {

            // the 3D cell range with changed visibility
            var range = Range3D.createFromRange(docModel.makeFullRange(interval, columns), sheet);

            // check passed ranges if they overlap with the source ranges of the data series
            var changed = false;
            data.series.forEach(function (seriesData) {
                if (seriesData.needUpdate) {
                    changed = true;
                } else if (seriesData.model.rangesOverlap(range)) {
                    changed = true;
                    seriesData.needUpdate = true;
                }
            });
            if (mainTitleModel.rangesOverlap(range)) { changed = true; }

            if (changed) { refreshChartDebounced(); }
        }

        function isXYType() {
            var chartType = self.getChartType();
            return chartType.indexOf('bubble') === 0 || chartType.indexOf('scatter') === 0;
        }

        function updateFormatting(subType) {
            var lAttrs = legendModel.getMergedAttributeSet(true);
            var attributes = self.getMergedAttributeSet(true);
            var chartAttrs = self.getMergedAttributeSet(true);
            var pieDonut = self.isPieOrDonut();

            data.series.forEach(function (seriesData, seriesIndex) {
                var attrs = seriesData.model.getMergedAttributeSet(true);

                seriesData.type = parseCJSType(attrs.series.type, chartAttrs.chart.stacking, chartAttrs.chart.curved);

                if (seriesData.needUpdate) {
                    return;
                }
                var lineType = isLineType(attrs.series.type);
                if (subType === 'series') {
                    colorHandler.handleColor('fill', seriesData, 'markerColor', data.series.length);
                    if (lineType) {
                        colorHandler.handleColor('line', seriesData, 'color', data.series.length);
                    } else {
                        if (Utils.getStringOption(attrs.fill, 'type', 'none') === 'none' && Utils.getStringOption(attrs.line, 'type', 'none') !== 'none') {
                            // use line color for fill, workaround for Bug 46848
                            colorHandler.handleColor('line', seriesData, 'color', data.series.length);
                        } else {
                            colorHandler.handleColor('fill', seriesData, 'color', data.series.length);
                        }
                    }
                }
                if (pieDonut) {
                    seriesData.startAngle = attributes.chart.rotation - 90;
                }
                if (lAttrs.legend.pos === 'off') {
                    seriesData.showInLegend = false;
                } else {
                    seriesData.showInLegend = true;
                    if (pieDonut) {
                        //we only see the first series
                        //canvasjs has rendering problems with to big legend-sizes
                        if ((seriesIndex > 0) || (seriesData.dps.length > 50)) {
                            seriesData.showInLegend = false;
                        }
                    }
                }

                if (seriesData.showInLegend && (!seriesData.dps || !seriesData.dps.length)) {
                    seriesData.showInLegend = false;
                }

                ChartStyleUtil.handleCharacterProps(self, indexLabelAttrs, seriesData, 'indexLabel');

                if (attrs.series.dataLabel) {
                    seriesData.indexLabel = '{name}';
                    seriesData.indexLabelLineColor = null;
                    seriesData.indexLabelPlacement = 'inside';
                } else {
                    seriesData.indexLabel = ' ';
                    seriesData.indexLabelLineColor = 'transparent';
                    seriesData.indexLabelPlacement = null;
                }

                _.each(seriesData.dps, function (dataPoint, dataPointIndex) {
                    if (lineType) {
                        if (data.series.length > 1) {
                            dataPoint.markerType = MARKER_LIST[seriesIndex % MARKER_LIST.length];
                        } else {
                            dataPoint.markerType = MARKER_LIST[dataPointIndex % MARKER_LIST.length];
                        }
                        dataPoint.markerSize = sheetModel.getEffectiveZoom() * 7;

                        seriesData.legendMarkerType = dataPoint.markerType;
                    } else {
                        dataPoint.markerType = null;
                        dataPoint.markerSize = 0;

                        seriesData.legendMarkerType = 'square';
                    }
                });
            });

            chartFormatter.format();

            legendModel.refreshInfo();

            mainTitleModel.refreshInfo();

            refreshAxis();

            updateMaxDataPointSize();
        }

        /**
         * Inserts a new data series model into this chart model.
         */
        function insertDataSeries(index, attrs) {

            // workarounds for Bug 41988 (black charts from mac-excel)
            if (attrs.fill && !attrs.fill.color && !attrs.fill.gradient) {
                attrs.fill.color = { type: 'auto' };
            }

            if (!isLineType(attrs.series.type) && attrs.line && attrs.line.width) {
                //TODO: needed?
                delete attrs.line.width;
            }

            var chartAttrs = self.getMergedAttributeSet(true);
            var cjsType = parseCJSType(attrs.series.type, chartAttrs.chart.stacking, chartAttrs.chart.curved);

            var seriesData = {
                type: cjsType,
                fillOpacity: 1,
                marker: MARKER_TYPES[attrs.series.marker],
                model: new DataSeriesModel(sheetModel, attrs),
                seriesIndex: index,
                needUpdate: true,
                dps: [],
                name: '',
                markerColor: 'transparent',
                color: 'transparent',
                indexLabelFontColor: 'transparent'
            };

            data.series.splice(index, 0, seriesData);

            self.listenTo(seriesData.model, 'refresh:formulas', function () {
                seriesData.needUpdate = true;
                refreshChartDebounced();
            });

            seriesData.axisXType = attrs.series.axisXIndex ? 'secondary' : 'primary';
            seriesData.axisYType = attrs.series.axisYIndex ? 'secondary' : 'primary';

            self.trigger('change:drawing');

            updateHighlightTokenArray();

            return true;
        }

        function compareFormats(group, other) {

            if (group.format === other.format) {
                group.display = null;
                group.result = (group.result || group.value) + ' ' + (other.result || other.value);
                return;
            }

            if (group.format.isAnyDateTime() && other.format.isAnyDateTime()) {
                group.display = null;
                group.result = (group.result || group.value) + (other.result || other.value);

                var groupDate = group.format.isAnyDate();
                var otherDate = other.format.isAnyDate();
                var groupTime = group.format.isAnyTime();
                var otherTime = other.format.isAnyTime();

                //take org format
                if ((groupDate && otherDate) || (groupTime && otherTime)) { return; }

                // use combined date/time format
                if ((groupDate && otherTime) || (groupTime && otherDate)) {
                    group.format = chartFormatter.parseFormatCode(LocaleData.SHORT_DATE + ' ' + LocaleData.LONG_TIME);
                }
            }
        }

        function reduceBigRange(source, index, name) {

            var ranges = data.series[index].model.resolveRanges(name);
            var range = ranges ? ranges.first() : null;
            if (!range) { return source; }

            var cols = range.cols();
            var rows = range.rows();
            if (cols > rows) {
                Utils.warn('chart with 2d ranges as "' + name + '" reduceBigRange() must be implemented!');
                return source;
            } else if (rows > cols) {
                var newList = [];
                var numberFormatter = docModel.getNumberFormatter();
                for (var row = 0; row < rows; ++row) {
                    var group = source[row * cols];
                    for (var col = 1; col < cols; ++col) {
                        var other = source[row * cols + col];
                        compareFormats(group, other);
                    }
                    group.display = numberFormatter.formatValue(group.format, group.result || group.value);
                    newList.push(group);
                }
                return newList;
            }
        }

        function updateMaxDataPointSize() {
            if (self.getSeriesCount() === 0) { return; }
            var rect = self.getRectangle();
            if (!rect) { return; }

            delete data.dataPointMaxWidth;
            delete data.dataPointMinWidth;

            var attrs = self.getMergedAttributeSet(true);
            var stacking = attrs.chart.stacking;
            var chartType = self.getChartType();
            var dataPointMaxWidth = null;

            if (stacking !== 'percentStacked' && stacking !== 'stacked' && data.series[0].dps.length === 1) {
                var multi = data.series.length;
                if (chartType.indexOf('bar') === 0) {
                    dataPointMaxWidth = Math.floor(rect.height / multi);
                } else if (chartType.indexOf('column') === 0) {
                    dataPointMaxWidth = Math.floor((rect.width * 0.9) / multi);
                }
                if (dataPointMaxWidth) {
                    data.dataPointMaxWidth = dataPointMaxWidth * 1.5;
                    data.dataPointMinWidth = dataPointMaxWidth * 0.5;
                }
            }

            ///////////////////////////////////

            var legendPos = legendModel.getMergedAttributeSet(true).legend.pos;
            var maxWidth = rect.width;

            if (legendPos === 'left' || legendPos === 'right' || legendPos === 'topRight') {
                maxWidth = rect.width / 4;
            }

            data.legend.maxWidth = maxWidth;
            data.legend.itemWrap = true;
            data.legend.maxHeight = rect.height;
        }

        function seriesAttrsToCellArray(array) {
            var result = [];
            if (_.isArray(array)) {
                array.forEach(function (value) {
                    var res = { display: value };
                    if (Utils.isFiniteNumber(value)) {
                        res.value = value;
                    }
                    result.push(res);
                });
            }
            return result;
        }

        /**
         * Refreshes the source values of the data series of this chart.
         */
        function innerRefresh() {

            // nothing to do during import of the document
            if (!self.isImportFinished()) { return; }

            // all ranges to be querried, as array of arrays of range addresses
            var sourceRanges = [];
            // maps series indexes and source link types to array elements in 'sourceRanges'
            var collectInfo = [];

            var constantData = false;

            data.series.forEach(function (seriesData, seriesIndex) {
                if (!seriesData.needUpdate) { return; }

                var info = { series: seriesIndex, indexes: {} };

                var tokenCount = 0;

                seriesData.model.iterateTokenArrays(function (tokenArray, linkKey, attrName) {
                    tokenCount++;
                    var entryRanges = seriesData.model.resolveRanges(linkKey);
                    if (!entryRanges.empty()) {
                        info.indexes[attrName] = sourceRanges.length;
                        sourceRanges.push(entryRanges);
                    }
                });

                if (tokenCount) {
                    collectInfo.push(info);
                } else {
                    constantData = true;
                    var sAttrs = seriesData.model.getExplicitAttributeSet(true).series;
                    chartFormatter.update(seriesIndex, seriesAttrsToCellArray(sAttrs.values), seriesAttrsToCellArray(sAttrs.title), seriesAttrsToCellArray(sAttrs.names), seriesAttrsToCellArray(sAttrs.bubbles));
                }
                seriesData.needUpdate = false;
            });

            if (!sourceRanges.length) {
                if (mainTitleModel.refreshInfo() || constantData) {
                    // fix for Bug 48095 & Bug 51282
                    self.trigger('change:drawing', 'series');
                } else {
                    updateFormatting();
                }
                return;
            }

            if (sourceRanges.length > 0) {

                // query contents of visible cells for the data series (be nice and use a time-sliced loop)
                var allContents = [];
                var promise = self.iterateArraySliced(sourceRanges, function (ranges) {
                    allContents.push(docModel.getRangeContents(ranges, { blanks: true, visible: docModel.getApp().isOOXML(), attributes: true, display: true, maxCount: 1000 }));
                }, 'ChartModel.innerRefresh');

                // do not call the handler function, if the chart has been deleted in the meantime
                self.waitForSuccess(promise, function () {

                    collectInfo.forEach(function (info) {

                        // returns the cell contents for the specified source link type
                        function getCellContents(key) {
                            return _.isNumber(info.indexes[key]) ? allContents[info.indexes[key]] : [];
                        }

                        // series values must exist
                        var valuesSource = getCellContents('values');
                        var titleSource = getCellContents('title');
                        var namesSource = getCellContents('names');
                        var bubblesSource = getCellContents('bubbles');

                        if (namesSource.length && namesSource.length >= valuesSource.length * 2) {
                            // workaround for TOO BIG ranges (2d range)
                            namesSource = reduceBigRange(namesSource, info.series, 'series.names');
                        }
                        if (titleSource.length && titleSource.length >= valuesSource.length * 2) {
                            // workaround for TOO BIG ranges (2d range)
                            titleSource = reduceBigRange(titleSource, info.series, 'series.title');
                        }
                        chartFormatter.update(info.series, valuesSource, titleSource, namesSource, bubblesSource);

                    });
                    refreshAxis();
                    self.trigger('change:drawing', 'series');
                });
            } else {
                _(data.series).each(function (seriesData) {
                    seriesData.needUpdate = false;
                });
                self.trigger('change:drawing', 'series');
            }
        }

        function changeAllSeriesAttrs(attrs) {
            return sheetModel.createAndApplyOperations(function (generator) {
                var position = self.getPosition();
                data.series.forEach(function (seriesData, seriesIndex) {
                    seriesData.model.generateChangeOperations(generator, position, seriesIndex, attrs);
                });
                return $.when();
            }, { storeSelection: true });
        }

        // protected methods --------------------------------------------------

        /**
         * Handler for the document operation 'insertChartDataSeries'.
         *
         * @param {OperationContext} context
         *  A wrapper representing the 'insertChartDataSeries' operation.
         */
        this.applyInsertSeriesOperation = function (context) {
            var index = context.getOptInt('series', data.series.length);
            insertDataSeries(index, context.getOptObj('attrs'));
        };

        /**
         * Handler for the document operation 'deleteChartDataSeries'.
         *
         * @param {OperationContext} context
         *  A wrapper representing the 'deleteChartDataSeries' operation.
         */
        this.applyDeleteSeriesOperation = function (context) {

            var index = context.getInt('series'),
                seriesData = data.series[index];
            context.ensure(seriesData, 'invalid series index');

            seriesData.model.destroy();

            //workaround for Bug 43958, changing chart type to Bubble chart, dataseries count is cut to half
            //so we make a copy of the array, that the array in CanvasJS has still the same length
            data.series = data.series.slice(0);

            data.series.splice(index, 1);
            this.trigger('change:drawing', 'series');

            updateHighlightTokenArray();
        };

        /**
         * Handler for the document operation 'setChartDataSeriesAttributes'.
         *
         * @param {OperationContext} context
         *  A wrapper representing the 'setChartDataSeriesAttributes' operation.
         */
        this.applyChangeSeriesOperation = function (context) {

            var seriesData = data.series[context.getInt('series')];
            context.ensure(seriesData, 'invalid series index');

            seriesData.model.setAttributes(context.getObj('attrs'));
            seriesData.needUpdate = true;
            this.trigger('change:drawing', 'series');

            updateHighlightTokenArray();
        };

        /**
         * Handler for the document operation 'setChartAxisAttributes'.
         *
         * @param {OperationContext} context
         *  A wrapper representing the 'setChartAxisAttributes' operation.
         */
        this.applyChangeAxisOperation = function (context) {

            var axisModel = axisModelMap[context.getStr('axis')];
            context.ensure(axisModel, 'invalid axis identifier');

            axisModel.setAttributes(context.getObj('attrs'));
            this.trigger('change:drawing');
        };

        /**
         * Handler for the document operation 'setChartGridlineAttributes'.
         *
         * @param {OperationContext} context
         *  A wrapper representing the 'setChartGridlineAttributes' operation.
         */
        this.applyChangeGridOperation = function (context) {

            var axisModel = axisModelMap[context.getStr('axis')];
            context.ensure(axisModel, 'invalid axis identifier');

            var gridModel = axisModel.getGrid();
            context.ensure(gridModel, 'missing grid line model');

            gridModel.setAttributes(context.getObj('attrs'));
            this.trigger('change:drawing');
        };

        /**
         * Handler for the document operation 'setChartTitleAttributes'.
         *
         * @param {OperationContext} context
         *  A wrapper representing the 'setChartTitleAttributes' operation.
         */
        this.applyChangeTitleOperation = function (context) {

            var axisId = context.getStr('axis');
            var titleModel = (axisId === 'main') ? mainTitleModel : (axisId in axisModelMap) ? axisModelMap[axisId].getTitle() : null;
            context.ensure(titleModel, 'invalid axis identifier');

            titleModel.setAttributes(context.getObj('attrs'));
            this.trigger('change:drawing');
        };

        /**
         * Handler for the document operation 'setChartLegendAttributes'.
         *
         * @param {OperationContext} context
         *  A wrapper representing the 'setChartLegendAttributes' operation.
         */
        this.applyChangeLegendOperation = function (context) {
            legendModel.setAttributes(context.getObj('attrs'));
            this.trigger('change:drawing', 'series');
        };

        // public methods -----------------------------------------------------

        this.setMainName = function (name) {
            mainTitleModel.setName(name);
        };

        /**
         * is called by the DrawingController for a selected chart
         */
        this.getChartType = function () {
            if (!data.series.length) {
                return 'bar';
            }
            var series = self.getSeriesModel(0).getMergedAttributeSet(true).series;
            var attrs = self.getMergedAttributeSet(true).chart;
            return parseChartType(series.type, attrs.stacking, attrs.curved);
        };

        this.getChartTypeForGui = function () {
            var chartType = self.getChartType();
            chartType = chartType.replace('clustered', 'standard');
            if (self.isMarkerOnly()) {
                chartType = chartType.replace(' curved', '');
                chartType += ' marker';
            }
            return chartType;
        };

        /**
         * chart CONTENT can be visible
         * (normal and fallbacks)
         */
        this.isValidChartType = function () {
            if (!data.series.length) {
                // no fallback for Bug 53212
                return true;
            }

            var firstType = self.getSeriesModel(0).getMergedAttributeSet(true).series.type;
            firstType = firstType.replace('2d', '').replace('3d', '');

            if (!VALID.test(firstType)) {
                return false;
            }

            var lastType = self.getSeriesModel(data.series.length - 1).getMergedAttributeSet(true).series.type;
            lastType = lastType.replace('2d', '').replace('3d', '');
            if (lastType !== firstType && !(SUPPORTED_COMBINED.test(firstType) && SUPPORTED_COMBINED.test(lastType))) {
                return false;
            }
            return true;
        };

        /**
         * chart BUTTONS can be usable
         * undo redo is supported
         */
        this.isSupported = function () {
            if (!data.series.length) {
                return false;
            }

            var firstType = self.getSeriesModel(0).getMergedAttributeSet(true).series.type;
            var lastType = self.getSeriesModel(data.series.length - 1).getMergedAttributeSet(true).series.type;
            if (lastType !== firstType) {
                return false;
            }
            if (!SUPPORTED.test(firstType)) {
                return false;
            }
            return true;
        };

        /**
         * Refreshes the source values of the data series of this chart.
         */
        this.refresh = this.createDebouncedMethod('SheetChartModel.refresh', null, innerRefresh, { delay: 50, maxDelay: 250 });

        this.getSeriesModel = function (index) {
            return data.series[index].model;
        };

        this.getLegendModel = function () {
            return legendModel;
        };

        this.getAxisModel = function (axisId) {
            return axisModelMap[axisId];
        };

        this.getTitleModel = function (axisId) {
            return (axisId === 'main') ? mainTitleModel : this.getAxisModel(axisId).getTitle();
        };

        /**
         *  Adds formula tokens to the given containing the source ranges of all data
         *  series of this chart object.
         */
        this.getHighlightTokenArray = function () {
            updateHighlightTokenArray();
            return highlightTokenArray;
        };

        /**
         * Invokes the passed callback function for all data series.
         *
         * @param {Function} callback
         *  (1) {Number} seriesIndex the index of the data series
         *  (2) {Object} tokens the object with the token name(key) and tokenArray(value)
         */
        this.iterateTokenArrays = function (callback) {
            data.series.forEach(function (seriesData) {
                var tokens = {};
                seriesData.model.iterateTokenArrays(function (tokenArray, linkKey, attrName) {
                    tokens[attrName] = tokenArray;
                });
                callback(seriesData.seriesIndex, tokens);
            });
        };

        function updateHighlightTokenArray() {

            var sourceRanges = new Range3DArray();
            data.series.forEach(function (seriesData) {
                seriesData.model.iterateTokenArrays(function (tokenArray, linkKey) {
                    sourceRanges.append(seriesData.model.resolveRanges(linkKey));
                });
            });

            var sheet = sheetModel.getIndex();
            sourceRanges = RangeArray.map(sourceRanges, function (range3d) {
                return range3d.isSheet(sheet) ? range3d.toRange() : null;
            });

            highlightTokenArray.clearTokens().appendRangeList(sourceRanges, { abs: true });
        }

        function refreshAxis() {
            if (!data.series.length) {
                return;
            }
            _.each(axisModelMap, function (axisModel, axisId) {
                if (axisId === 'z') { return; }
                axisModel.refreshInfo();
            });
        }

        function getStyleId() {
            return self.getMergedAttributeSet(true).chart.chartStyleId;
        }

        function sourceExistsComplete(name) {
            return (data.series.length > 0) && data.series.every(function (seriesData) {
                return !_.isEmpty(seriesData.model.getMergedAttributeSet(true).series[name]);
            });
        }

        function isSingleSeries() {
            return self.isPieOrDonut() || data.series.length === 1;
        }

        this.getBackgroundColor = function () {
            var fillAttrs = this.getMergedAttributeSet(true).fill;
            return ChartStyleUtil.isAutoShape(fillAttrs) ? colorHandler.getBackgroundColor(getStyleId()) : fillAttrs.color;
        };

        this.isXYType = isXYType;

        this.isAxesEnabled = function () {
            return !self.isPieOrDonut();
        };

        this.isPieOrDonut = function () {
            return /^(pie|donut|doughnut|sunburst)/.test(self.getChartType());
        };

        this.isAreaOrLine = function () {
            return /^(area|line|scatter)/.test(self.getChartType());
        };

        this.isVaryColorEnabled = function () {
            return isSingleSeries();
        };

        this.isVaryColor = function () {

            if (!isSingleSeries()) {
                return false;
            }

            var attrs = self.getExplicitAttributeSet(true);
            if (attrs && attrs.chart && attrs.chart.varyColors) {
                return true;
            }

            var colorInfo = self.getSeriesColorInfo();
            if (colorInfo.length <= 1) { return false; }

            return !_.isEqual(colorInfo[0], colorInfo[1]);
        };

        /**
         * it generates a unique-color-ID out of the internal chartStyleId
         * for describing the color patterns
         * return null, if one or more series have a unique fill color
         * @returns {String|null}
         */
        this.getColorSet = function () {
            return 'cs' + ChartStyleUtil.toColorSetIndex(getStyleId());
        };

        /**
         * it generates a unique-style-ID out of the internal chartStyleId
         * for describing the background and the highlight of the chart
         * it ignores unique fill colors of the series
         * @returns {String}
         */
        this.getStyleSet = function () {
            return 'ss' + ChartStyleUtil.toStyleSetIndex(getStyleId());
        };

        function generateChangeStyleIdOperations(generator, chartStyleId) {

            var styleSet = ChartStyleUtil.getStyleSet()[ChartStyleUtil.toStyleSetIndex(chartStyleId)];

            var bgColor = styleSet.bg;
            if (!bgColor) {
                var color = ChartStyleUtil.getColorOfPattern('cycle', 'single', 1, ChartStyleUtil.getColorSet()[ChartStyleUtil.toColorSetIndex(chartStyleId)].colors, 3, docModel);
                bgColor = { type: color.type, value: color.value, transformations: ChartStyleUtil.getBackgroundTransformation() };
            }
            if (!bgColor.fallbackValue) {
                bgColor.fallbackValue = docModel.parseAndResolveColor(bgColor, 'fill').hex;
            }

            var attrSet = { fill: { type: 'solid', color: bgColor }, chart: { chartStyleId: chartStyleId } };
            return self.generateChangeOperations(generator, attrSet);
        }

        this.changeVaryColors = function (state) {
            return sheetModel.createAndApplyOperations(function (generator) {

                var chartStyleId = getStyleId();

                generateChangeStyleIdOperations(generator, chartStyleId);

                var position = self.getPosition();
                var colors = ChartStyleUtil.getColorSet()[ChartStyleUtil.toColorSetIndex(chartStyleId)];
                var count = 1;

                if (self.isVaryColorEnabled()) {
                    self.generateChangeOperations(generator, { chart: { varyColors: state } });

                    var seriesData = data.series[0];
                    var dataPoints = [];
                    var seriesAttrs = seriesData.model.getMergedAttributeSet(true);
                    count = seriesData.dps.length;

                    seriesData.dps.forEach(function (dataPoint, index) {
                        var color = null;
                        if (state) {
                            color = ChartStyleUtil.getColorOfPattern('cycle', colors.type, index, colors.colors, count, docModel);
                        } else {
                            color = ChartStyleUtil.getColorOfPattern('cycle', colors.type, 0, colors.colors, 3, docModel);
                        }
                        dataPoints.push({ fill: { type: seriesAttrs.fill.type, color: color }, line: { type: seriesAttrs.line.type, color: color } });
                    });

                    seriesData.model.generateChangeOperations(generator, position, 0, { series: { dataPoints: dataPoints } });
                } else {
                    count = data.series.length;
                    data.series.forEach(function (seriesData, seriesIndex) {
                        var color = null;
                        if (state) {
                            color = ChartStyleUtil.getColorOfPattern('cycle', colors.type, seriesIndex, colors.colors, count, docModel);
                        } else {
                            color = ChartStyleUtil.getColorOfPattern('cycle', colors.type, 0, colors.colors, 3, docModel);
                        }

                        seriesData.model.generateChangeOperations(generator, position, seriesIndex, { fill: { color: color }, line: { color: color } });
                    });
                }

                return $.when();
            }, { storeSelection: true });
        };

        /**
         * updates the interal chartStyleId by the unique ID
         * deletes all series colors if existing
         *
         * @param {String} colorSet
         */
        this.changeColorSet = function (colorSet) {
            return sheetModel.createAndApplyOperations(function (generator) {

                var chartStyleId = getStyleId();
                var csId = colorSet.replace('cs', '') | 0;
                var styleSet = ChartStyleUtil.toStyleSetIndex(chartStyleId);

                chartStyleId = ChartStyleUtil.toChartStyleId(csId, styleSet);

                generateChangeStyleIdOperations(generator, chartStyleId);

                var position = self.getPosition();
                var colors = ChartStyleUtil.getColorSet()[ChartStyleUtil.toColorSetIndex(chartStyleId)];
                var count = 1;

                var seriesData = data.series[0];
                var seriesAttrs = seriesData.model.getMergedAttributeSet(true);

                if (self.isVaryColorEnabled()) {
                    var dataPoints = [];
                    count = seriesData.dps.length;

                    seriesData.dps.forEach(function (dataPoint, index) {
                        var color = ChartStyleUtil.getColorOfPattern('cycle', colors.type, index, colors.colors, count, docModel);
                        dataPoints.push({ fill: { type: seriesAttrs.fill.type, color: color }, line: { type: seriesAttrs.line.type, color: color } });
                    });

                    seriesData.model.generateChangeOperations(generator, position, 0, { series: { dataPoints: dataPoints } });
                } else {
                    count = data.series.length;
                    data.series.forEach(function (seriesData, seriesIndex) {
                        var color = ChartStyleUtil.getColorOfPattern('cycle', colors.type, seriesIndex, colors.colors, count, docModel);

                        seriesData.model.generateChangeOperations(generator, position, seriesIndex, { fill: { type: seriesAttrs.fill.type, color: color }, line: { type: seriesAttrs.line.type, color: color } });
                    });
                }

                return $.when();
            }, { storeSelection: true });
        };

        /**
         * updates the interal chartStyleId by the unique ID
         * does not touch the series colors
         *
         * @param {String} colorSet
         */
        this.changeStyleSet = function (styleSet) {
            return sheetModel.createAndApplyOperations(function (generator) {

                var chartStyleId = getStyleId();
                var ssId = styleSet.replace('ss', '') | 0;
                var colorSet = ChartStyleUtil.toColorSetIndex(chartStyleId);
                chartStyleId = ChartStyleUtil.toChartStyleId(colorSet, ssId);

                generateChangeStyleIdOperations(generator, chartStyleId);

                return $.when();
            }, { storeSelection: true });
        };

        this.getSeriesCount = function () {
            return data.series.length;
        };

        this.getFirstPointsCount = function () {
            try {
                if (data.series.length) {
                    return data.series[0].dps.length;
                }
            } catch (e) {
                Utils.warn('error while calling chartmodel.getFirstPointsCount()', e);
            }
            return 0;
        };

        /**
         * must be called for having correct behavior mit "varyColors" in combination with more than one series
         * FIXME: still needed?
         */
        this.firstInit = function () {
            self.trigger('change:drawing');
        };

        /**
         * must be called for having correct behavior mit "new CanvasJS.Chart" is called, because of a crazy bug inside there
         */
        this.resetData = function () {
            data.axisX._oldOptions = null;
            data.axisY._oldOptions = null;
        };

        this.hasDataPoints = function () {
            for (var i = 0; i < data.series.length; i++) {
                if (!_.isEmpty(data.series[i].dps)) { return true; }
            }
            return false;
        };

        /**
         * normally the assigned axisId (x or y) will be returned
         * except chart is a bar-type, then x & y are interchanged
         *
         * @param {String} axisId
         *
         * @returns {String}
         */
        this.getAxisIdForDrawing = function (axisId) {
            var chartType = self.getChartType();
            if (chartType.indexOf('bar') === 0) {
                if (axisId === 'x') { return 'y'; }
                if (axisId === 'y') { return 'x'; }
            }
            return axisId;
        };

        /**
         * transfers model-data to CanvasJs-data
         */
        this.updateRenderInfo = function () {
            updateFormatting();
            self.refresh();
        };

        /**
         * iterates all sources and calculates min and max positions,
         * direction of the series-values and the sheetindex
         *
         * @returns {Object}
         *  A descriptor with several properties of the data source:
         *  - {String} [warn]
         *      If existing, a specific warning code for invalid source data:
         *      - 'nodata': Source ranges are not available at all.
         *      - 'sheets': Source ranges are not on the same sheet.
         *      - 'directions': Source ranges are not in the same direction.
         * - {Range} range
         *      The bounding range of all source ranges in the targeted sheet.
         * - {Number} sheet
         *      Sheet index of all source ranges.
         * - {Number} axis
         *      Identifier of the main axis of the source ranges.
         */
        this.getDataSourceInfo = function () {

            var ranges = new RangeArray();
            var mainAxis = null;
            var sheet = null;
            var warn = null;
            var lastValues = null;

            data.series.forEach(function (seriesData) {
                seriesData.model.iterateTokenArrays(function (tokenArray, linkKey, attrName) {
                    var rangeList = seriesData.model.resolveRanges(linkKey);
                    if (rangeList.empty()) { return; }
                    var range = rangeList.first();
                    if (sheet === null) {
                        sheet = range.sheet1;
                    }
                    if (!range.isSheet(sheet)) {
                        warn = 'sheets';
                    }
                    if (attrName === 'values') {
                        var axis = null;
                        var col = range.cols();
                        var row = range.rows();
                        if (col > row) {
                            axis = 1;
                        } else if (row > 1) {
                            axis = 0;
                        } else if (lastValues) {
                            if (lastValues.start[0] !== range.start[0]) {
                                axis = 0;
                            } else {
                                axis = 1;
                            }
                        }
                        if (_.isNumber(axis)) {
                            if (mainAxis === null) {
                                mainAxis = axis;
                            } else if (mainAxis !== axis) {
                                warn = 'directions';
                            }
                        }
                        lastValues = range;
                    }
                    // convert the 3D range (with sheet indexes) to a 2D range (without sheet indexes)
                    ranges.push(range.toRange());
                });
            });

            if (warn) { return { warn: warn }; }
            if (sheet === null || ranges.empty()) { return { warn: 'nodata' }; }
            return { range: ranges.boundary(), axis: mainAxis, sheet: sheet };
        };

        this.getSeriesColorInfo = function () {

            function addColorInfo(attrs, points) {
                if (!attrs) { return; }

                if (!attrs.fill || attrs.fill.type === 'none') {
                    if (!attrs.line) { /* TODO: special ODF charts, we ignore it... */ return; }
                    points.push(attrs.line.color);
                } else {
                    points.push(attrs.fill.color);
                }
            }

            var points = [];
            if (isSingleSeries()) {
                var dataPoints = data.series[0].model.getExplicitAttributeSet(true).series.dataPoints;
                if (dataPoints) {
                    dataPoints.forEach(function (pointAttrs) {
                        addColorInfo(pointAttrs, points);
                    });
                } else {
                    Utils.warn('ChartModel.getSeriesColorInfo(): no dataPoints found in series');
                }
            }
            if (!points.length) {
                data.series.forEach(function (seriesData) {
                    addColorInfo(seriesData.model.getExplicitAttributeSet(true), points);
                });
            }
            return points;
        };

        /**
         * checks if all dataseries have a sourcelink 'title'
         * which is the first row in a normal chart
         */
        this.isTitleLabel = function () {
            return sourceExistsComplete('title');
        };

        /**
         * checks if all dataseries have a sourcelink 'names'
         * which is the first column in a normal chart
         */
        this.isNamesLabel = function () {
            return sourceExistsComplete('names');
        };

        /**
         *
         * @returns {Boolean}
         */
        this.isMarkerOnly = function () {
            var marker = false;
            _.find(data.series, function (dataSeries) {
                var att = dataSeries.model.getMergedAttributeSet(true);
                if (isLineType(att.series.type) && att.line.type === 'none') {
                    marker = true;
                }
                return true;
            });
            return marker;
        };

        this.getDataLabel = function () {
            var dataLabel = null;
            _.each(data.series, function (dataSeries) {
                var att = dataSeries.model.getMergedAttributeSet(true);
                dataLabel = dataLabel || att.series.dataLabel;
            });
            return dataLabel;
        };

        /**
         * Creates an image replacement from this model with the given
         * extent and image mimeType (either image/png or image/jpeg).
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {String} [options.mimeType]
         *   If specified, the given image mime type will
         *   be used for image creation. Allowed values ATM
         *   are image/png (default) and image/jpeg
         *  @param {Object} [options.size]
         *   If specified, the chart replacement will be
         *   rendered with the given size, otherwise, its
         *   size is retrieved from the model's drawing
         *   attributes.
         *
         * @returns {jQuery|Null}
         *  If successful, a floating jquery img node, containg the image data
         *  as src dataUrl and with the requested width and height set
         *  accordingly.
         */
        this.getReplacementNode = function (options) {

            var drawingRectPx = this.getRectangle();
            if (!drawingRectPx) { return null; }

            var jqImgNode = null,
                modelData = this.getModelData(),
                chartId = 'io-ox-documents-chart-frame' + _.uniqueId(),
                widthHeightAttr = '"width="' + drawingRectPx.width + 'px" height="' + drawingRectPx.height + 'px"',
                jqDrawingDiv = $('<div class="chartnode chartholder" id="' + chartId + '" ' + widthHeightAttr + '>').css({
                    position: 'absolute',
                    left: -drawingRectPx.width,
                    top: -drawingRectPx.height,
                    width: drawingRectPx.width,
                    height: drawingRectPx.height
                });

            $(':root').append(jqDrawingDiv);

            var chartRenderer = new CanvasJS.Chart(chartId, _.extend(modelData, { backgroundColor: modelData.cssBackgroundColor })),
                jqCanvas = $('#' + chartId + ' .canvasjs-chart-canvas'),
                canvas = jqCanvas.get(0);

            try {
                // render chart
                chartRenderer.render();

                var dataUrl = canvas.toDataURL(Utils.getStringOption(options, 'mimeType', 'image/png'));

                if (dataUrl) {
                    jqImgNode = $('<img>').attr({
                        width: drawingRectPx.width,
                        height: drawingRectPx.height,
                        src: dataUrl
                    });

                    jqCanvas = dataUrl = chartRenderer = null;
                }
            } catch (ex) {
                Utils.exception(ex, 'replacement chart image rendering error');
            }

            jqDrawingDiv.remove();
            jqDrawingDiv = null;

            return jqImgNode;
        };

        /**
         * @param {Boolean} state
         */
        this.setMarkerOnly = function (state) {
            var noneShape = { type: 'none' };
            var solidShape = { type: 'solid' };

            var attrs = null;
            if (state) {
                attrs = { line: noneShape, fill: solidShape };
            } else {
                attrs = { line: solidShape, fill: solidShape };
            }

            return changeAllSeriesAttrs(attrs);
        };

        this.setDataLabel = function (state) {
            var attrs = { series: { dataLabel: state ? 'value' : '' } };
            return changeAllSeriesAttrs(attrs);
        };

        this.getCloneData = function () {

            // a data object passed as hidden parameter to the constructor of the clone
            var cloneData = { series: [], axes: {}, legend: legendModel.getExplicitAttributeSet(), title: mainTitleModel.getExplicitAttributeSet() };

            _.each(axisModelMap, function (axisModel, axisId) {
                // workaround for Bug 46966
                if (axisId === 'z') { return; }

                var aClone = {
                    axis: axisModel.getExplicitAttributeSet()
                };
                if (axisModel.getGrid()) {
                    aClone.grid = axisModel.getGrid().getExplicitAttributeSet();
                }
                if (axisModel.getTitle()) {
                    aClone.title = axisModel.getTitle().getExplicitAttributeSet();
                }
                cloneData.axes[axisId] = aClone;
            });

            _.each(data.series, function (dataSeries) {
                var nsp = dataSeries.model.getExplicitAttributeSet();

                dataSeries.model.iterateTokenArrays(function (tokenArray, linkKey, attrName) {
                    nsp.series[attrName] = tokenArray.getFormula('op');
                });

                cloneData.series.push(nsp);
            });

            return cloneData;
        };

        this.getCanvasJSData = function () {

            var cvsData = {
                axis: {},
                series: []
            };
            _.each(data.series, function (series) {
                var points = [];
                _.each(series.dps, function (dataPoint) {
                    points.push({
                        color: dataPoint.color,
                        label: dataPoint.label,
                        x: dataPoint.x,
                        y: dataPoint.y,
                        z: dataPoint.z
                    });
                });

                cvsData.series.push({
                    bevelEnabled:       series.bevelEnabled,
                    color:              series.color,
                    fillOpacity:        series.fillOpacity,
                    indexLabel:         series.indexLabel,
                    legendMarkerType:   series.legendMarkerType,
                    marker:             series.marker,
                    markerColor:        series.markerColor,
                    name:               series.name,
                    showInLegend:       series.showInLegend,
                    type:               series.type,
                    points:             points
                });

            });
            _.each(axisModelMap, function (axisModel, axisId) {
                var dataAxis = data['axis' + axisId.toUpperCase()];
                cvsData.axis[axisId] = {
                    label:     dataAxis.label,
                    grid:      dataAxis.grid,
                    labelFont: dataAxis.labelFont,
                    line:      dataAxis.line,
                    tick:      dataAxis.tick,
                    titleFont: dataAxis.titleFont
                };
            });
            cvsData.legend = {
                font: data.legend.font,
                verticalAlign: data.legend.verticalAlign,
                horizontalAlign: data.legend.horizontalAlign
            };
            cvsData.title = data.title.textFont;

            return cvsData;
        };

        // initialization -----------------------------------------------------

        colorHandler = new ManualColorHandler(this, data);
        chartFormatter = new ChartFormatter(docModel, this);

        this.on('change:attributes', function () {
            updateFormatting('series');
        });

        this.on('change:drawing', function (event, subType) {
            updateFormatting(subType);
        });

        // additional processing and event handling after the document has been imported
        this.waitForImportSuccess(function (alreadyImported) {

            // refresh the chart after the visibility of columns or rows has changed
            this.listenTo(docModel, 'change:columns', function (event, sheet, interval, changeInfo) {
                if (changeInfo.visibilityChanged) { updateColRowVisibility(sheet, interval, true); }
            });
            this.listenTo(docModel, 'change:rows', function (event, sheet, interval, changeInfo) {
                if (changeInfo.visibilityChanged) { updateColRowVisibility(sheet, interval, false); }
            });

            if (!alreadyImported) {
                self.executeDelayed(function () {
                    //small hack for fastcalc-loading
                    if (sheetModel.getIndex() === docModel.getActiveSheet()) {
                        self.refresh();
                    }
                }, 'SheetChartModel.waitForImportSucess');
            }

        }, this);

        // clone private data passed as hidden argument to the c'tor
        (function (args) {
            var cloneData = args[SheetChartModel.length];
            if (_.isObject(cloneData)) {
                _.each(cloneData.axes, function (axisData, axisId) {
                    var targetAxisModel = axisModelMap[axisId];
                    targetAxisModel.setAttributes(axisData.axis);
                    if (!_.isEmpty(axisData.grid)) {
                        targetAxisModel.getGrid().setAttributes(axisData.grid);
                    }
                    if (!_.isEmpty(axisData.title)) {
                        targetAxisModel.getTitle().setAttributes(axisData.title);
                    }
                });
                _.each(cloneData.series, function (attrs, index) {
                    insertDataSeries(index, attrs);
                });
                mainTitleModel.setAttributes(cloneData.title);
                legendModel.setAttributes(cloneData.legend);
            }
        }(arguments));

        // destroy all class members on destruction
        this.registerDestructor(function () {
            _.invoke(axisModelMap, 'destroy');
            mainTitleModel.destroy();
            legendModel.destroy();
            chartFormatter.destroy();

            _.each(data.series, function (seriesData) {
                seriesData.model.destroy();
                // must delete, otherwise cansjs has still reference on this
                delete seriesData.model;
            });

            self = docModel = sheetModel = parentCollection = null;
            mainTitleModel = legendModel = data = axisModelMap = colorHandler = chartFormatter = null;
        });

    } }); // class SheetChartModel

    // exports ================================================================

    return SheetChartModel;

});
