/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/spreadsheet/model/cellautostylecollection', [
    'io.ox/office/tk/utils',
    'io.ox/office/tk/container/valuemap',
    'io.ox/office/editframework/utils/border',
    'io.ox/office/editframework/utils/attributeutils',
    'io.ox/office/editframework/model/autostylecollection',
    'io.ox/office/spreadsheet/utils/sheetutils'
], function (Utils, ValueMap, Border, AttributeUtils, AutoStyleCollection, SheetUtils) {

    'use strict';

    // class CellAutoStyleCollection ==========================================

    /**
     * Contains the auto-styles with cell/character formatting attributes for
     * all cells, columns, and rows of a spreadsheet document.
     *
     * @constructor
     *
     * @extends AutoStyleCollection
     *
     * @param {SpreadsheetModel} docModel
     *  The document model containing this instance.
     */
    function CellAutoStyleCollection(docModel) {

        // self reference
        var self = this;

        // special behavior for OOXML files
        var ooxml = docModel.getApp().isOOXML();

        // special behavior for ODF files
        var odf = docModel.getApp().isODF();

        // the collection of cell style sheets this instance is based on
        var styleSheets = docModel.getCellStyles();

        // the attribute pool for cell attributes
        var attributePool = styleSheets.getAttributePool();

        // the names of all supported attribute families, as flag set
        var supportedFamilySet = styleSheets.getSupportedFamilies();

        // the number formatter of the document
        var numberFormatter = docModel.getNumberFormatter();

        // parsed format code for unknown auto-styles
        var standardFormat = numberFormatter.getStandardFormat();

        // a cache for parsed number formats for all existing auto-styles
        var parsedFormatCache = new ValueMap();

        // a cache for visible border maps for all existing auto-styles
        var borderMapCache = new ValueMap();

        // base constructor ---------------------------------------------------

        AutoStyleCollection.call(this, docModel, 'cell', {
            standardPrefix: ooxml ? 'a' : 'ce',
            strictMode: ooxml,
            keepComplete: ooxml,
            updateHandler: updateAutoStyleHandler
        });

        // private methods ----------------------------------------------------

        /**
         * Extends auto-styles of this collection with specific properties for
         * spreadsheet cells.
         */
        function updateAutoStyleHandler(styleId, type) {

            // when deleting an auto-style, remove the additional data from the cache
            if (type === 'delete') {
                parsedFormatCache.remove(styleId);
                borderMapCache.remove(styleId);
                return;
            }

            // the effective merged attributes of the auto-style
            var mergedAttributeSet = self.getMergedAttributeSet(styleId);
            var cellAttrs = mergedAttributeSet.cell;

            // resolve the parsed number format for fast access
            var parsedFormat = numberFormatter.getParsedFormatForAttributes(cellAttrs);
            parsedFormatCache.insert(styleId, parsedFormat);

            // cache the visible border attributes in a map with short border keys
            borderMapCache.insert(styleId, Utils.makeObject('tblrdu', function (borderKey) {
                var border = cellAttrs[SheetUtils.getBorderName(borderKey)];
                return Border.isVisibleBorder(border) ? border : undefined;
            }));
        }

        // public methods -----------------------------------------------------

        /**
         * Returns the parsed number format instance of an auto-style.
         *
         * @param {String} styleId
         *  The unique identifier of an auto-style. The empty string can be
         *  used for the default auto-style of this collection.
         *
         * @returns {ParsedFormat}
         *  The parsed number format instance of the auto-style.
         */
        this.getParsedFormat = function (styleId) {
            return parsedFormatCache.get(styleId, standardFormat);
        };

        /**
         * Returns the visible border attributes of an auto-style in a map with
         * short single-character border keys.
         *
         * @param {String} styleId
         *  The unique identifier of an auto-style. The empty string can be
         *  used for the default auto-style of this collection.
         *
         * @returns {Object}
         *  The visible border attributes of the auto-style, as (incomplete)
         *  map. Each visible border is mapped by the one-character key for the
         *  border ('t', 'b', 'l', 'r', 'd', and 'u'). Invisible borders are
         *  not contained in the map (the map object may be empty).
         */
        this.getBorderMap = function (styleId) {
            return borderMapCache.get(styleId, {});
        };

        /**
         * Returns the identifier of an auto-style that contains the formatting
         * attributes of an existing auto-style, merged with the passed
         * attribute set. If the auto-style does not exist yet, it will be
         * created, and the appropriate remote operations will be inserted into
         * the passed generators.
         *
         * @param {SheetOperationGenerator} generator
         *  The operations generator to be filled with the operation.
         *
         * @param {String} styleId
         *  The unique identifier of an auto-style. The empty string can be
         *  used for the default auto-style of this collection.
         *
         * @param {Object} contents
         *  A contents descriptor used in various operation generators for
         *  columns, rows, and cells. This method uses the optional properties
         *  'a' (an incomplete attribute set), and 'format' (the identifier of
         *  a number format, or an explicit format code). The attribute set
         *  MUST NOT contain explicit apply flags (any attributes from the
         *  attribute family 'apply'). These flags will be calculated
         *  automatically from the other explicit attributes contained in the
         *  attribute set, and/or from a new cell style sheet.
         *
         * @param {Function} [processAttributes]
         *  A callback function that allows to process the attribute set
         *  according to the current attributes of the auto-style. Receives the
         *  following parameters:
         *  (1) {Object} attributeSet
         *      A deep copy of the attribute set passed to this method. Shall
         *      be modified in-place by the callback function.
         *  (2) {Object} mergedAttributeSet
         *      The current merged attribute set of the auto-style. MUST NOT be
         *      changed by the callback function.
         *
         * @returns {String}
         *  The identifier of the resulting auto-style containing the merged
         *  formatting attributes.
         */
        this.generateAutoStyleOperation = _.wrap(this.generateAutoStyleOperation, function (generateAutoStyleOperation, generator, styleId, contents, processAttributes) {

            // the attribute set with the new attributes to be inserted into the auto-style
            // (work on a two-level deep copy to not modify the object passed by the caller)
            var attributeSet = _.isObject(contents.a) ? _.mapObject(contents.a, _.clone) : {};

            // add the number format passed in the contents object (OOXML: as identifier; ODF: as format code)
            if ('format' in contents) {
                if (!attributeSet.cell) { attributeSet.cell = {}; }
                _.extend(attributeSet.cell, numberFormatter.generateNumberFormatOperations(generator, contents.format));
            }

            // the explicit attribute set of the auto-style
            var explicitAttributeSet = this.getExplicitAttributeSet(styleId);
            // the merged attribute set of the auto-style
            var mergedAttributeSet = this.getMergedAttributeSet(styleId);
            // the merged attribute set of the new style sheet
            var styleAttributeSet = _.isString(attributeSet.styleId) ? styleSheets.getStyleAttributeSet(attributeSet.styleId) : null;

            // callback function wants to preprocess the attributes
            if (processAttributes) { processAttributes(attributeSet, mergedAttributeSet); }

            // apply flags cannot be set by outer code (will be calculated below)
            delete attributeSet.apply;

            // Update the 'apply' flags in the explicit attribute set for OOXML files. The 'apply' flags
            // mark entire groups of attributes that have been set explicitly, and override the attributes
            // of the cell style sheet. In ODF files, the 'apply' flags are not used at all.
            //
            if (ooxml) {
                var applyFlags = {};
                _.each(attributeSet, function (attributes, family) {
                    var definitionMap = attributePool.getRegisteredAttributes(family);
                    if (definitionMap && (family in supportedFamilySet)) {
                        definitionMap.forEachSupported(attributes, function (value, name, definition) {
                            if (definition.apply) { applyFlags[definition.apply] = true; }
                        });
                    }
                });
                attributeSet.apply = applyFlags;
            }

            // Apply a new style sheet. In OOXML files, the 'apply' flags of the style sheet specify which explicit
            // attributes of the auto-style have to be removed in order to let the style sheet attributes take effect.
            // In ODF files, all explicit attributes will be removed, and the explicit attributes of the new style
            // sheet will take effect only.
            //
            if (styleAttributeSet) {
                if (ooxml) {

                    // copy all formatting attributes with active 'apply' flag from the style sheet to the attribute set
                    _.each(styleAttributeSet, function (styleAttributes, family) {
                        if (family !== 'apply') {
                            var definitionMap = attributePool.getRegisteredAttributes(family);
                            if (definitionMap) {
                                definitionMap.forEach(function (definition, name) {
                                    if (definition.apply && styleAttributeSet.apply[definition.apply]) {
                                        AttributeUtils.insertAttribute(attributeSet, family, name, styleAttributes[name]);
                                    }
                                });
                            }
                        }
                    });

                    // reset the explicit 'apply' flags of the auto-style that are set in the style sheet
                    _.each(styleAttributeSet.apply, function (applyStyleAttrs, name) {
                        if (applyStyleAttrs && mergedAttributeSet.apply[name]) {
                            // keep the existing flags added above for new explicit attributes
                            AttributeUtils.insertAttribute(attributeSet, 'apply', name, false, true);
                        }
                    });

                } else if (odf) {

                    // ODF: clear all existing explicit attributes (set all missing attributes to null)
                    _.each(explicitAttributeSet, function (autoStyleAttributes, family) {
                        if ((family in supportedFamilySet) && _.isObject(autoStyleAttributes)) {
                            _.each(autoStyleAttributes, function (value, name) {
                                AttributeUtils.insertAttribute(attributeSet, family, name, null, true);
                            });
                        }
                    });

                }
                // create 'insertStyleSheet' operations for the built-in style sheets available in Excel
                if (typeof attributeSet.styleId === 'string') {
                    styleSheets.generateMissingStyleSheetOperations(generator, attributeSet.styleId);
                }
            } else {
                delete attributeSet.styleId;
            }

            // invoke the base class method to create the auto-style
            return generateAutoStyleOperation.call(this, generator, styleId, attributeSet);
        });

        // initialization -----------------------------------------------------

        // destroy class members on destruction
        this.registerDestructor(function () {
            self = docModel = styleSheets = attributePool = null;
            numberFormatter = standardFormat = null;
            parsedFormatCache = borderMapCache = null;
        });

    } // class CellAutoStyleCollection

    // exports ================================================================

    return AutoStyleCollection.extend({ constructor: CellAutoStyleCollection });

});
