/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/spreadsheet/controller/viewmixin', [
    'io.ox/office/tk/utils/iteratorutils',
    'io.ox/office/spreadsheet/utils/config',
    'io.ox/office/spreadsheet/utils/sheetutils',
    'io.ox/office/spreadsheet/utils/paneutils'
], function (IteratorUtils, Config, SheetUtils, PaneUtils) {

    'use strict';

    // convenience shortcuts
    var SplitMode = SheetUtils.SplitMode;
    var Interval = SheetUtils.Interval;

    // mix-in class ViewMixin =================================================

    /**
     * Implementations of all controller items for manipulating view settings
     * in the active sheet, intended to be mixed into a document controller
     * instance.
     *
     * @constructor
     *
     * @param {SpreadsheetView} docView
     *  The document view providing view settings such as zoom and split.
     */
    function ViewMixin(docView) {

        // self reference (the controller)
        var self = this;

        // the applicxation instance
        var app = this.getApp();

        // the document model
        var docModel = docView.getDocModel();

        // the model of the active sheet
        var sheetModel = null;

        // private methods ----------------------------------------------------

        /**
         * Toggles the dynamic view split mode according to the current view
         * split mode, and the current selection. Enabling dynamic split mode
         * in a frozen split view will thaw the frozen panes. Otherwise,
         * enabling the dynamic split mode will split the view at the position
         * left of and above the active cell. Disabling the dynamic split mode
         * will remove the split also if the view is in frozen split mode.
         *
         * @param {Boolean} state
         *  The new state of the dynamic view split mode.
         */
        function setDynamicSplit(state) {

            // whether the view is currently split
            var hasSplit = sheetModel.hasSplit();

            // disable dynamic or frozen split mode
            if (!state && hasSplit) {
                sheetModel.clearSplit();

            // convert frozen split to dynamic split
            } else if (state && sheetModel.hasFrozenSplit()) {
                sheetModel.setDynamicSplit(sheetModel.getSplitWidthHmm(), sheetModel.getSplitHeightHmm());

            // set dynamic split at active cell
            } else if (state && !hasSplit) {

                // the address of the active cell in the current selection
                var activeAddress = docView.getActiveCell();
                // additional scroll anchor attributes
                var attributes = {};

                // get position of active cell and visible area of the grid pane
                var cellRect = docView.getCellRectangle(activeAddress, { expandMerged: true });
                var visibleRect = docView.getActiveGridPane().getVisibleRectangle();

                // do not split vertically, if the active cell is shown at the left border
                var splitWidth = cellRect.left - visibleRect.left;
                if ((-cellRect.width < splitWidth) && (splitWidth < PaneUtils.MIN_PANE_SIZE)) {
                    splitWidth = 0;
                }

                // do not split horizontally, if the active cell is shown at the top border
                var splitHeight = cellRect.top - visibleRect.top;
                if ((-cellRect.height < splitHeight) && (splitHeight < PaneUtils.MIN_PANE_SIZE)) {
                    splitHeight = 0;
                }

                // split in the middle of the grid pane, if active cell is outside the visible area
                if (((splitWidth === 0) && (splitHeight === 0)) ||
                    (splitWidth < 0) || (splitWidth > visibleRect.width - PaneUtils.MIN_PANE_SIZE) ||
                    (splitHeight < 0) || (splitHeight > visibleRect.height - PaneUtils.MIN_PANE_SIZE)
                ) {
                    splitWidth = Math.floor(visibleRect.width / 2);
                    splitHeight = Math.floor(visibleRect.height / 2);
                    attributes.anchorRight = docView.getColCollection().getScrollAnchorByOffset(visibleRect.left + splitWidth, { pixel: true });
                    attributes.anchorBottom = docView.getRowCollection().getScrollAnchorByOffset(visibleRect.top + splitHeight, { pixel: true });
                    attributes.activePane = 'topLeft';
                } else {
                    if (splitWidth > 0) { attributes.anchorRight = activeAddress[0]; }
                    if (splitHeight > 0) { attributes.anchorBottom = activeAddress[1]; }
                }

                // activate the dynamic split view
                splitWidth = sheetModel.convertPixelToHmm(splitWidth);
                splitHeight = sheetModel.convertPixelToHmm(splitHeight);
                sheetModel.setDynamicSplit(splitWidth, splitHeight, attributes);
            }
        }

        /**
         * Toggles the frozen view split mode according to the current view
         * split mode, and the current selection. Enabling frozen split mode
         * in a dynamic split view will freeze the panes with their current
         * size (split mode 'frozenSplit'). Otherwise, enabling the frozen
         * split mode will split the view at the position left of and above the
         * active cell. Disabling the frozen split mode will either return to
         * dynamic split mode (if split mode was 'frozenSplit'), or will remove
         * the frozen split at all (if split mode was 'frozen').
         *
         * @param {Boolean|String} state
         *  The new state of the frozen view split mode (as boolean), or the
         *  string 'toggle' to toggle the current frozen split mode.
         */
        function setFrozenSplit(state) {

            if (state === 'toggle') { state = !sheetModel.hasFrozenSplit(); }

            // whether the view is currently split
            var hasSplit = sheetModel.hasSplit();

            // convert frozen split to dynamic split
            if (!state && hasSplit && (sheetModel.getSplitMode() === SplitMode.FROZEN_SPLIT)) {
                sheetModel.setDynamicSplit(sheetModel.getSplitWidthHmm(), sheetModel.getSplitHeightHmm());

            // disable split mode completely
            } else if (!state && hasSplit) {
                sheetModel.clearSplit();

            // convert dynamic split to frozen split
            } else if (state && hasSplit && !sheetModel.hasFrozenSplit()) {
                sheetModel.setFrozenSplit(sheetModel.getSplitColInterval(), sheetModel.getSplitRowInterval());

            // enable frozen split mode
            } else if (state && !hasSplit) {

                // the address of the active cell in the current selection
                var activeAddress = docView.getActiveCell();

                // calculate frozen column interval (must result in at least one frozen column)
                var colAnchor = sheetModel.getViewAttribute('anchorRight');
                var colInterval = new Interval(Math.round(colAnchor), activeAddress[0] - 1);
                if (colInterval.first > colInterval.last) { colInterval = null; }

                // calculate frozen row interval (must result in at least one frozen row)
                var rowAnchor = sheetModel.getViewAttribute('anchorBottom');
                var rowInterval = new Interval(Math.round(rowAnchor), activeAddress[1] - 1);
                if (rowInterval.first > rowInterval.last) { rowInterval = null; }

                // activate the frozen split view
                sheetModel.setFrozenSplit(colInterval, rowInterval);
            }
        }

        /**
         * Freezes the passed number of columns and rows in the active sheet,
         * regardless of the current split settings.
         *
         * @param {Number} cols
         *  The number of columns to freeze. The value zero will not freeze any
         *  column.
         *
         * @param {Number} rows
         *  The number of rows to freeze. The value zero will not freeze any
         *  row.
         */
        function setFixedFrozenSplit(cols, rows) {

            // the scroll anchors for the first visible column/row
            var colAnchor = sheetModel.getViewAttribute(sheetModel.hasColSplit() ? 'anchorLeft' : 'anchorRight');
            var rowAnchor = sheetModel.getViewAttribute(sheetModel.hasRowSplit() ? 'anchorTop' : 'anchorBottom');
            // the first visible column and row in the split view (reduce to be able to freeze the passed number of columns/rows)
            var col = Math.max(0, Math.min(Math.round(colAnchor), docModel.getMaxCol() - cols + 1));
            var row = Math.max(0, Math.min(Math.round(rowAnchor), docModel.getMaxRow() - rows + 1));

            // activate the frozen split view
            sheetModel.setFrozenSplit((cols > 0) ? new Interval(col, col + cols - 1) : null, (rows > 0) ? new Interval(row, row + rows - 1) : null);
        }

        // public methods -----------------------------------------------------

        /**
         * Generates and sends operations for all view attributes that have
         * been changed while the document was open.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved when all operations have been
         *  created.
         */
        this.sendChangedViewAttributes = function () {

            // Only send changed view attributes when user has edit rights, and has modified
            // the document locally, or if 'important' sheet view settings have been changed.
            // This is needed to prevent creating new versions of the document after selecting
            // cells or scrolling around, but without editing anything.
            if (!docView.isEditable()) { return $.when(); }

            // whether the document is modified, and view settings can be sent
            var modified = app.isLocallyModified();

            // detect if any sheet contains 'important' changed view attributes
            if (!modified) {
                modified = IteratorUtils.some(docModel.createSheetIterator(), function (sheetModel2) {
                    return sheetModel2.hasChangedViewAttributes();
                });
            }

            // nothing to do (e.g. only selection has been changed, which will not be saved then)
            if (!modified) { return $.when(); }

            // generate and apply the operations for the document, and all affected sheets
            return docModel.createAndApplySheetOperations(function (generator) {

                // create attribute operation for the active sheet index
                var activeSheet = sheetModel.getIndex();
                if (activeSheet !== docModel.getDocumentAttribute('activeSheet')) {
                    generator.generateDocAttrsOperation({ activeSheet: activeSheet });
                }

                // create operations for the changed view attributes of all sheets
                return docModel.generateOperationsForAllSheets(generator, function (sheetModel2) {
                    var sheetAttributes = sheetModel2.getChangedViewAttributes();
                    if (!_.isEmpty(sheetAttributes)) {
                        return sheetModel2.generateChangeSheetOperations(generator, { sheet: sheetAttributes });
                    }
                });
            }, { undoMode: 'skip' }); // do not create an undo action for view settings
        };

        // item registration --------------------------------------------------

        // register all controller items
        this.registerDefinitions({

            // parent item, enabled if the cell selection mode is active (no drawing object selected)
            // other document states do not care here (read-only mode, text edit mode, etc.)
            'view/selection/cell': {
                parent: 'app/valid',
                enable: function () { return !docView.hasDrawingSelection(); }
            },

            // parent item, enabled if the drawing selection mode is active
            // other document states do not care here (read-only mode, text edit mode, etc.)
            'view/selection/drawing': {
                parent: 'app/valid',
                enable: function () { return docView.hasDrawingSelection(); }
            },

            // parent item, enabled if the text edit mode is NOT active
            'view/editmode/off': {
                parent: 'app/valid',
                enable: function () { return !docView.isTextEditMode(); }
            },

            // parent item, enabled if any text edit mode is active
            'view/editmode/on': {
                parent: 'app/valid',
                enable: function () { return docView.isTextEditMode(); }
            },

            // parent item, enabled if the cell edit mode is active
            'view/editmode/cell': {
                parent: 'view/selection/cell',
                enable: function () { return docView.isTextEditMode('cell'); }
            },

            // parent item, enabled if the drawing edit mode is active
            'view/editmode/drawing': {
                parent: 'view/selection/drawing',
                enable: function () { return docView.isTextEditMode('drawing'); }
            },

            // represents the visibility state of the formula pane (setter expects a boolean state)
            'view/formulapane/show': {
                parent: 'document/worksheet',
                get: function () { return docView.getFormulaPane().isVisible(); },
                set: function (state) {
                    app.setUserSettingsValue('showFormulaBar', state);
                    docView.getFormulaPane().toggle(state);
                }
            },

            // toggles the height of the formula pane (single text line vs. multiple text lines)
            'view/formulapane/toggle/height': {
                parent: 'app/valid',
                set: function () { return docView.getFormulaPane().toggleHeight(); }
            },

            // represents the visibility state of the status pane (setter expects a boolean state)
            'view/statuspane/show': {
                parent: 'app/valid',
                get: function () { return docView.getStatusPane().isVisible(); },
                set: function (state) {
                    app.setUserSettingsValue('showStatusBar', state);
                    docView.getStatusPane().toggle(state);
                }
            },

            // item value is the current status label to be shown in the status bar
            'view/status/label': {
                parent: 'app/valid',
                enable: function () { return docView.getStatusLabel() !== null; },
                get: function () { return docView.getStatusLabel(); }
            },

            // item value is the type of the subtotals result (e.g. 'sum', 'average') to be shown in the status pane
            'view/status/subtotals': {
                parent: 'document/worksheet',
                enable: function () {
                    var subtotals = docView.getSubtotalResult();
                    return (subtotals.cells > 1) && (this.getValue() in subtotals);
                },
                get: function () { return app.getUserSettingsValue('subtotalType', 'sum'); },
                set: function (subtotalType) { app.setUserSettingsValue('subtotalType', subtotalType); }
            },

            // item value is the formula expression of the active cell, with leading equality sign
            'view/status/formula': {
                parent: ['document/worksheet', 'view/selection/cell', 'view/editmode/off'],
                enable: function () { return docView.isSingleCellSelection() && (this.getValue() !== null); },
                get: function () { return docView.getCellCollection().getEditFormula(docView.getActiveCell()); }
            },

            'view/zoom': {
                parent: 'app/valid',
                get: function () { return sheetModel.getZoom(); },
                set: function (zoom) { sheetModel.setZoom(zoom); }
            },

            'view/zoom/dec': {
                parent: 'view/zoom',
                enable: function (zoom) { return zoom > PaneUtils.MIN_ZOOM; },
                set: function () {
                    var prevZoom = PaneUtils.getPrevPresetZoom(sheetModel.getZoom());
                    if (prevZoom) { sheetModel.setZoom(prevZoom); }
                }
            },

            'view/zoom/inc': {
                parent: 'view/zoom',
                enable: function (zoom) { return zoom < PaneUtils.MAX_ZOOM; },
                set: function () {
                    var nextZoom = PaneUtils.getNextPresetZoom(sheetModel.getZoom());
                    if (nextZoom) { sheetModel.setZoom(nextZoom); }
                }
            },

            'view/grid/show': {
                parent: 'document/worksheet',
                get: function () { return sheetModel.isCellType() && sheetModel.getViewAttribute('showGrid'); },
                set: function (state) { sheetModel.setViewAttribute('showGrid', state); }
            },

            'view/split/dynamic': {
                parent: ['document/worksheet', 'view/editmode/off'],
                get: function () { return sheetModel.hasDynamicSplit(); },
                set: setDynamicSplit
            },

            'view/split/frozen': {
                parent: ['document/worksheet', 'view/editmode/off'],
                get: function () { return sheetModel.hasFrozenSplit(); },
                set: setFrozenSplit
            },

            'view/split/frozen/fixed': {
                parent: ['document/worksheet', 'view/editmode/off'],
                set: function (value) { setFixedFrozenSplit(value.cols, value.rows); }
            },

            // tool bars and tool bar tabs ------------------------------------

            // enabled if the 'Format' toolbar tab shall be visible in the top pane (either in cell mode, or with selected text shapes)
            'view/toolbartab/format/visible': {
                parent: 'document/editable/worksheet',
                enable: function () { return self.isItemEnabled('view/selection/cell') || (!Config.COMBINED_TOOL_PANES && self.isItemEnabled('drawing/operation/text')); }
            },

            // enabled if the 'Data' toolbar tab shall be visible in the top pane
            'view/toolbartab/data/visible': {
                parent: 'document/editable/cell'
            },

            // enabled if the 'Insert' toolbar tab shall be visible in the top pane
            'view/toolbartab/insert/visible': {
                parent: 'document/editable/worksheet'
            },

            // enabled if the 'Columns/Rows' toolbar tab shall be visible in the top pane
            'view/toolbartab/colrow/visible': {
                parent: 'document/editable/cell'
            },

            // enabled if the 'Table' toolbar tab shall be visible in the top pane
            'view/toolbartab/table/visible': {
                parent: ['sheet/realtable', 'document/ooxml'],
                enable: function (tableModel) { return tableModel !== null; }
            },

            // enabled if the 'Drawing' toolbar tab shall be visible in the top pane
            'view/toolbartab/drawing/visible': {
                parent: 'document/editable/drawing'
            },

            // tool bars ------------------------------------------------------

            // visibility item for the 'Connector' toolbar for drawing objects
            // (at least one selected connector object)
            'view/toolbar/drawing/connector/visible': {
                parent: 'drawing/operation/connector'
            },

            // visibility item for the 'Border' toolbar for drawing objects
            // (at least one selected drawing object that supports line attributes, but no connectors are selected)
            'view/toolbar/drawing/border/visible': {
                parent: 'drawing/operation/line',
                enable: function () { return !self.isItemEnabled('view/toolbar/drawing/connector/visible'); }
            },

            // visibility item for the 'Fill' toolbar for drawing objects
            // (at least one selected drawing object that supports fill attributes)
            'view/toolbar/drawing/fill/visible': {
                parent: 'drawing/operation/fill'
            },

            // debugging ------------------------------------------------------

            'debug/highlight/formulas': {
                parent: 'debug/enabled',
                get: function () { return docView.getViewAttribute('highlightFormulas'); },
                set: function (enable) { docView.setViewAttribute('highlightFormulas', enable); }
            }
        });

        // initialization -----------------------------------------------------

        // update GUI after changed selection, after receiving view updates, and when switching text edit modes
        this._updateOnEvent(docView, 'change:doc:viewattributes change:sheet:viewattributes update:selection:data textedit:enter textedit:change textedit:leave');

        // initialize sheet-dependent class members according to the active sheet
        this.listenTo(docView, 'change:activesheet', function () {
            sheetModel = docView.getSheetModel();
        });

        // destroy all class members on destruction
        this.registerDestructor(function () {
            app = docView = docModel = sheetModel = null;
        });

    } // class ViewMixin

    // exports ================================================================

    return ViewMixin;

});
