/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 * @author Edy Haryono <edy.haryono@open-xchange.com>
 */

define('io.ox/office/spreadsheet/controller/searchmixin', [
    'io.ox/office/tk/utils',
    'io.ox/office/tk/utils/iteratorutils',
    'io.ox/office/spreadsheet/utils/sheetutils',
    'gettext!io.ox/office/spreadsheet/main'
], function (Utils, IteratorUtils, SheetUtils, gt) {

    'use strict';

    // convenience shortcuts
    var RangeArray = SheetUtils.RangeArray;

    // mix-in class SearchMixin ===============================================

    /**
     * Implementation of the search handler as required by the constructor of
     * the class EditController, intended to be mixed into a document
     * controller instance.
     *
     * @constructor
     *
     * @param {SpreadsheetView} docView
     *  The document view providing view settings such as the current sheet
     *  selection.
     */
    function SearchMixin(docView) {

        // self reference (the document controller)
        var self = this;

        // the model and collections of the active sheet
        var sheetModel = null;
        var cellCollection = null;
        var mergeCollection = null;

        // private methods ----------------------------------------------------

        /**
         * Selects the cell contained in the passed result object of an address
         * iterator.
         */
        function selectIteratorCell(iterResult, singleCell) {
            var address = iterResult.value;
            if (singleCell) {
                docView.selectCell(address, { active: iterResult.index });
            } else {
                docView.changeActiveCell(iterResult.index, address);
            }
            docView.scrollToCell(address);
        }

        /**
         * Returns the text to be matched at the specified address while
         * searching and replacing. For formula cells, the formula expression
         * will be returned. For all other cells, the display string will be
         * returned.
         *
         * @param {Address} address
         *  The address of a cell in the active sheet.
         *
         * @returns {String}
         *  The matching text at the specified cell address.
         */
        function getSearchText(address) {
            // bug 46987: convert search text to lower-case characters
            return cellCollection.getEditString(address).toLowerCase();
        }

        function searchPredicate(address, query) {
            // 'bla'.indexOf('') is always zero
            return !!query && (getSearchText(address).indexOf(query) >= 0);
        }

        // returns the replacement text for the passed cell descriptor
        function getReplaceContents(address, query, replace) {
            var searchText = getSearchText(address);
            var replaceText = searchText.replace(new RegExp(_.escapeRegExp(query), 'gi'), replace);
            var contents = cellCollection.parseCellValue(replaceText, 'val', address);
            return { address: address, contents: contents };
        }

        /**
         * Iterates through the current search results.
         *
         * @param {String} direction
         *  The search direction, either 'next', or 'prev'.
         *
         * @param {String} [query]
         *  The current string to be searched in the active sheet, as contained
         *  in the respective GUI element. If this value is different from the
         *  initial query string, a new search request will be sent to the
         *  server. If omitted, the current query string will be used again.
         */
        function searchNext(direction, query) {

            var selection = docView.getSelection();
            var singleCell = docView.isSingleCellSelection();
            var ranges = singleCell ? new RangeArray(cellCollection.getUsedRange()) : selection.ranges;

            var iterator = cellCollection.createAddressIterator(ranges, { startAddress: selection.address, startIndex: selection.active, skipStart: true, wrap: true, reverse: (direction === 'prev'), type: 'value', visible: false });
            iterator = IteratorUtils.createFilterIterator(iterator, function (address) {
                var isMerged = mergeCollection.isMergedCell(address);
                var isVisible = isMerged ? !mergeCollection.isHiddenCell(address) : cellCollection.isVisibleCell(address);
                return isVisible && searchPredicate(address, query);
            });

            var iterResult = iterator.next();
            if (iterResult.done) {
                docView.yell({ type: 'info', message: gt('Nothing found.') });
            } else {
                selectIteratorCell(iterResult, singleCell);
            }

            return null;
        }

        /**
         * Replaces the value of the current selection with the replacement
         * value.
         *
         * @param {String} query
         *  The current string to be searched in the active sheet, as contained
         *  in the respective GUI element. If this value is different from the
         *  initial query string, a new search request will be sent to the
         *  server.
         *
         * @param {String} replace
         *  The replacement text.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Boolean} [options.all=false]
         *      If set to true, all occurrences of the search text in the
         *      active sheet will be replaced.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved when the text has been replaced.
         */
        function searchReplace(query, replace, options) {

            // whether to replace all occurrences of the search text at once
            var replaceAll = Utils.getBooleanOption(options, 'all', false);

            // no replace functionality in locked cells (TODO: replace in unlocked cells?);
            // additional checks for search query and replacement text
            var promise = docView.ensureUnlockedSheet().then(function () {
                if ((query === '') || (query === replace)) {
                    return SheetUtils.makeRejected('replace:nothing');
                }
            });

            // replace the next occurence, or all occurrences, according to the passed options
            promise = promise.then(function () {

                var selection = docView.getSelection();
                var singleCell = docView.isSingleCellSelection();
                var ranges = singleCell ? new RangeArray(cellCollection.getUsedRange()) : selection.ranges;

                // search the next cell if the current cell value not contains the query
                if (!searchPredicate(docView.getActiveCell(), query)) {
                    searchNext('next', query);
                    return;
                }

                // the cell iterator: visit all value cells and formula cells
                var iterator = cellCollection.createAddressIterator(ranges, { startAddress: selection.address, wrap: true, type: 'value', visible: true });
                // filter for cell that match the search query
                iterator = IteratorUtils.createFilterIterator(iterator, function (address) { return searchPredicate(address, query); });

                // replace all occurrences of the search text at once
                if (replaceAll) {

                    // the array of all cell content descriptors, with addresses
                    var contentsArray = [];

                    // generate the cell contents for all matching cells
                    var promise2 = self.iterateSliced(iterator, function (address) {
                        contentsArray.push(getReplaceContents(address, query, replace));
                    }, 'SearchMixin.searchReplace');

                    // generate the cell operations
                    promise2 = promise2.then(function () {
                        return docView.setCellContentsArray(contentsArray);
                    });

                    return promise2;
                }

                // exit silently, if no more search results are available
                var iterResult = iterator.next();
                if (iterResult.done) { return $.when(); }

                // generate the cell operations for the next matching cell only
                var address = iterResult.value;
                var promise3 = docView.setCellContentsArray([getReplaceContents(address, query, replace)]);

                // select the next search result (or show a message, if no more results are available)
                return promise3.done(function () {
                    iterResult = iterator.next();
                    if (iterResult.done) {
                        docView.yellMessage('search:finished', true);
                    } else {
                        selectIteratorCell(iterResult, singleCell);
                    }
                });
            });

            return docView.yellOnFailure(promise);
        }

        // public methods -----------------------------------------------------

        /**
         * Implementation for search/replace operations. See the description of
         * the constructor parameter 'searchHandler' of the class
         * EditController for more details.
         *
         * @param {String} command
         *  The search command to be executed.
         *
         * @param {Object} settings
         *  The current search settings.
         *
         * @returns {jQuery.Promise|Null}
         *  A promise that will be resolved after executing the specified
         *  command successfully, or rejected on any error; or null for a
         *  synchronous operation.
         */
        this.executeSearchCommand = function (command, settings) {

            var query = settings.query;
            if (query) {
                // fix for Bug 46987
                query = query.toLowerCase();
            } else {
                // TODO: just return here?
            }

            switch (command) {
                case 'search:start':
                    return searchNext('next', query);
                case 'search:next':
                    return searchNext('next', query);
                case 'search:prev':
                    return searchNext('prev', query);
                case 'search:end':
                    return null;
                case 'replace:next':
                    return searchReplace(query, settings.replace);
                case 'replace:all':
                    return searchReplace(query, settings.replace, { all: true });
            }

            Utils.error('SearchMixin.executeCommand(): unsupported search command "' + command + '"');
            return $.Deferred().reject('internal');
        };

        // initialization -----------------------------------------------------

        // initialize sheet-dependent class members according to the active sheet
        this.listenTo(docView, 'change:activesheet', function () {
            sheetModel = docView.getSheetModel();
            cellCollection = sheetModel.getCellCollection();
            mergeCollection = sheetModel.getMergeCollection();
        });

        // destroy all class members on destruction
        this.registerDestructor(function () {
            self = docView = null;
            sheetModel = cellCollection = mergeCollection = null;
        });

    } // class SearchMixin

    // exports ================================================================

    return SearchMixin;

});
