/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/spreadsheet/controller/drawingmixin', [
    'io.ox/office/tk/utils',
    'io.ox/office/editframework/utils/color',
    'io.ox/office/drawinglayer/utils/drawingutils',
    'io.ox/office/drawinglayer/view/imageutil',
    'io.ox/office/drawinglayer/view/drawingframe',
    'io.ox/office/spreadsheet/utils/operations',
    'io.ox/office/spreadsheet/utils/sheetutils',
    'io.ox/office/spreadsheet/model/drawing/text/textframeutils',
    'io.ox/office/spreadsheet/view/labels',
    'io.ox/office/spreadsheet/view/dialogs',
    'gettext!io.ox/office/spreadsheet/main'
], function (Utils, Color, DrawingUtils, ImageUtils, DrawingFrame, Operations, SheetUtils, TextFrameUtils, Labels, Dialogs, gt) {

    'use strict';

    // maximum size of an image in any direction, in 1/100 mm
    var MAX_IMAGE_SIZE = 21000;

    // private global functions ===============================================

    /**
     * Returns whether the passed drawing model can be flipped and rotated.
     *
     * @param {DrawingModel} drawingModel
     *  The drawing model to be checked.
     *
     * @returns {Boolean}
     *  Whether the passed drawing model can be flipped and rotated.
     */
    function isRotatableModel(drawingModel) {
        return drawingModel.isDeepRotatable();
    }

    /**
     * Returns whether the passed drawing model supports line formatting.
     *
     * @param {DrawingModel} drawingModel
     *  The drawing model to be checked.
     *
     * @returns {Boolean}
     *  Whether the passed drawing model supports line formatting.
     */
    function isLineFormatModel(drawingModel) {
        // TODO: allow changing the border of chart objects
        return drawingModel.supportsFamily('line') && (drawingModel.getType() !== 'chart');
    }

    /**
     * Returns whether the passed drawing model supports fill formatting.
     *
     * @param {DrawingModel} drawingModel
     *  The drawing model to be checked.
     *
     * @returns {Boolean}
     *  Whether the passed drawing model supports fill formatting.
     */
    function isFillFormatModel(drawingModel) {
        // TODO: allow changing the background of chart objects
        return drawingModel.supportsFamily('fill') && !isConnectorModel(drawingModel) && (drawingModel.getType() !== 'chart');
    }

    /**
     * Returns whether the passed drawing model is a connector shape.
     *
     * @param {DrawingModel} drawingModel
     *  The drawing model to be checked.
     *
     * @returns {Boolean}
     *  Whether the passed drawing model is a connector shape.
     */
    function isConnectorModel(drawingModel) {
        var attrs = drawingModel.getExplicitAttributeSet(true);
        return drawingModel.supportsFamily('geometry') && attrs.geometry && Labels.isPresetConnector(attrs.geometry.presetShape);
    }

    /**
     * Returns whether the passed drawing model contains a DOM text frame.
     *
     * @param {DrawingModel} drawingModel
     *  The drawing model to be checked.
     *
     * @returns {Boolean}
     *  Whether the passed drawing model contains a DOM text frame.
     */
    function isTextFrameModel(drawingModel) {
        return drawingModel.supportsFamily('shape') && (TextFrameUtils.getTextFrame(drawingModel.getModelFrame()) !== null);
    }

    // mix-in class DrawingMixin ==============================================

    /**
     * Implementations of all controller items for manipulating drawing objects
     * in the active sheet, intended to be mixed into a document controller
     * instance.
     *
     * @constructor
     *
     * @param {SpreadsheetView} docView
     *  The document view providing view settings such as the current drawing
     *  selection.
     */
    function DrawingMixin(docView) {

        // self reference
        var self = this;

        // the application instance containing this controller
        var app = this.getApp();
        var ooxml = app.isOOXML();
        var odf = app.isODF();

        // the document model
        var docModel = docView.getDocModel();

        // the attribute pool for drawing attribute handling
        var attributePool = docModel.getDrawingAttributePool();

        // the model and collections of the active sheet
        var sheetModel = null;
        var drawingCollection = null;

        // private methods ----------------------------------------------------

        /**
         * Generates and applies the operations needed to change the formatting
         * attributes for all drawing objects, and shows an error alert box, if
         * this fails.
         *
         * @param {Array<DrawingModel>} drawingModels
         *  An array with all drawing models to be changed.
         *
         * @param {Function|Object} callback
         *  A callback function that will be invoked for each drawing object,
         *  and that must return the attribute set to be applied for that
         *  drawing object. Receives the following parameters:
         *  (1) {DrawingModel} drawingModel
         *      The model of the drawing object to be changed.
         *  (2) {Object} mergedAttrSet
         *      The merged attribute set of the drawing object.
         *  Alternatively, a constant attribute set can be passed instead that
         *  will be applied at all drawing objects.
         *
         * @param {Object} [context]
         *  The calling context for the callback function.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved when all operations have been
         *  created and applied successfully.
         */
        function implChangeDrawings(drawingModels, callback, context) {

            // allow to pass a constant attribute set instead of a callback function
            if (typeof callback !== 'function') { callback = _.constant(callback); }

            // build the new attributes for all selected drawing objects
            var drawingDescs = drawingModels.map(function (drawingModel) {

                // the current merged attribute set
                var oldAttrSet = drawingModel.getMergedAttributeSet(true);
                // the (incomplete) attribute set with the new attributes
                var newAttrSet = callback.call(context, drawingModel, oldAttrSet);

                // special adjustments for transformations (rotation and flipping)
                if ('drawing' in newAttrSet) {

                    // the old and new drawing attributes (with flipping and rotation)
                    var oldAttrs = oldAttrSet.drawing;
                    var newAttrs = newAttrSet.drawing;

                    // normalize the new rotation angle
                    if ('rotation' in newAttrs) {
                        newAttrs.rotation = Utils.mod(Utils.round(newAttrs.rotation, 0.1), 360);
                    }

                    // special anchor adjustment for rotated drawings in OOXML
                    if (ooxml && (('flipH' in newAttrs) || ('flipV' in newAttrs) || ('rotation' in newAttrs))) {

                        // the original (unrotated) location of the drawing model (nothing to do for squares)
                        var rectangle = drawingModel.getRectangle();
                        if (!rectangle.square()) {

                            // the old and new state of width/height swapping according to the transformation attributes
                            var oldSwappedDims = DrawingUtils.hasSwappedDimensions(oldAttrs);
                            var newSwappedDims = DrawingUtils.hasSwappedDimensions(_.extend({}, oldAttrs, newAttrs));

                            // if the width/height swapping changes due to new rotation angle, add the new anchor attributes
                            if (oldSwappedDims !== newSwappedDims) {
                                if (newSwappedDims) { rectangle.rotateSelf(); }
                                attributePool.extendAttributeSet(newAttrSet, drawingCollection.getAttributeSetForRectangle(rectangle));
                            }
                        }
                    }
                }

                return { position: drawingModel.getPosition(), attrs: newAttrSet };
            });

            // change the selected drawing objects
            var promise = self.execChangeDrawings(drawingDescs);

            // show warning messages if needed
            return docView.yellOnFailure(promise);
        }

        // public methods -----------------------------------------------------

        /**
         * Inserts new drawing sobject into the active sheet, and selects them
         * afterwards. This method is an implementation helper that does not
         * show any error alerts on failure.
         *
         * @param {Array<Object>} drawingDescs
         *  An array of descriptors with the types and the formatting attribute
         *  sets for the drawing objects to be inserted. Each descriptor MUST
         *  contain the following properties:
         *  - {String} drawingDesc.type
         *      The type identifier of the drawing object.
         *  - {Object} drawingDesc.attrs
         *      Initial attribute set for the drawing object, especially the
         *      anchor attributes specifying the physical position of the
         *      drawing object in the sheet.
         *
         * @param {Function} [callback]
         *  A callback function that will be invoked everytime after an
         *  'insertDrawing' operation has been generated for a drawing object.
         *  Allows to create additional operations for the drawing objects,
         *  before all these operations will be applied at once. Receives the
         *  following parameters:
         *  (1) {SheetOperationGenerator} generator
         *      The operations generator (already containing the initial
         *      'insertDrawing' operation of the current drawing).
         *  (2) {Number} sheet
         *      The zero-based index of this sheet.
         *  (3) {Array<Number>} position
         *      The effective document position of the current drawing object
         *      in the sheet, as inserted into the 'insertDrawing' operation.
         *  (4) {Object} drawingDesc
         *      The descriptor element from the array parameter 'drawingDescs'
         *      that is currently processed.
         *  The callback function may return a promise to defer processing of
         *  subsequent drawing objects.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved when all drawing objects have been
         *  created successfully, or that will be rejected on any error.
         */
        this.execInsertDrawings = function (drawingDescs, callback) {

            // leave text edit mode before inserting a drawing object
            var promise = docView.leaveTextEditMode();

            // check that the drawing objects in the active sheet are not locked
            promise = promise.then(function () {
                return docView.ensureUnlockedDrawings({ errorCode: 'drawing:insert:locked' });
            });

            // generate and apply the drawing operations
            return promise.then(function () {
                return sheetModel.createAndApplyOperations(function (generator) {

                    // the position of the first new drawing object
                    var firstPosition = [drawingCollection.getModelCount()];
                    // the positions of all drawing objects to be selected later
                    var allPositions = [];

                    // generate and apply the document operations
                    var promise = drawingCollection.generateInsertDrawingOperations(generator, firstPosition, drawingDescs, function (generator, sheet, position, drawingDesc) {
                        allPositions.push(position);
                        return _.isFunction(callback) ? callback.call(self, generator, sheet, position, drawingDesc) : null;
                    });

                    // select the drawing objects on success
                    return promise.done(function () { docView.setDrawingSelection(allPositions); });

                }, { storeSelection: true });
            });
        };

        /**
         * Inserts new image objects into the active sheet, and selects them
         * afterwards. This method is an implementation helper that does not
         * show any error alerts on failure.
         *
         * @param {Object|Array<Object>} imageDescs
         *  A descriptor for the new image object, containing the properties
         *  'url' and 'name', or an array of such image descriptors to be able
         *  to insert multiple images at once.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved when the image objects have been
         *  created and inserted into the active sheet, or that will be
         *  rejected on any error.
         */
        this.execInsertImages = function (imageDescs) {

            // create the image DOM nodes for all images
            var promises = _.getArray(imageDescs).map(function (imageDesc) {

                // create the image DOM node, the promise waits for it to load the image data
                var promise = app.createImageNode(ImageUtils.getFileUrl(app, imageDesc.url));

                // fail with empty resolved promise to be able to use $.when() which would
                // otherwise abort immediately if ANY of the promises fails
                return promise.then(function (imgNode) {
                    return { node: imgNode[0], url: imageDesc.url, name: imageDesc.name };
                }, _.constant($.when()));
            });

            // wait for all images nodes, create and apply the 'insertDrawing' operations
            var promise = $.when.apply($, promises).then(function () {

                // $.when() passes the results as function arguments, filter out invalid images
                var results = _.filter(arguments, _.identity);
                if (results.length === 0) { return $.Deferred().reject(); }

                // the address of teh active cell
                var activeAddress = docView.getActiveCell();
                // all image nodes collected in a jQuery object
                var imgNodes = $();

                // the drawing attributes to be passed to execInsertDrawings()
                var drawingDescs = results.map(function (result) {

                    // current width of the image, in 1/100 mm
                    var width = Utils.convertLengthToHmm(result.node.width, 'px');
                    // current height of the image, in 1/100 mm
                    var height = Utils.convertLengthToHmm(result.node.height, 'px');
                    // the generic drawing attributes (anchor position)
                    var drawingAttrs = {
                        anchorType: 'oneCell',
                        startCol: activeAddress[0],
                        startRow: activeAddress[1],
                        name: result.name,
                        aspectLocked: true
                    };
                    // additional specific image attributes
                    var imageAttrs = { imageUrl: result.url };

                    // restrict size to fixed limit
                    if (width > height) {
                        if (width > MAX_IMAGE_SIZE) {
                            height = Math.round(height * MAX_IMAGE_SIZE / width);
                            width = MAX_IMAGE_SIZE;
                        }
                    } else {
                        if (height > MAX_IMAGE_SIZE) {
                            width = Math.round(width * MAX_IMAGE_SIZE / height);
                            height = MAX_IMAGE_SIZE;
                        }
                    }
                    drawingAttrs.width = width;
                    drawingAttrs.height = height;

                    // collect all image nodes for clean-up
                    imgNodes = imgNodes.add(result.node);

                    // create the drawing descriptor for execInsertDrawings()
                    return { type: 'image', attrs: { drawing: drawingAttrs, image: imageAttrs } };
                });

                // destroy the temporary image nodes
                app.destroyImageNodes(imgNodes);

                // insert the image objects into the active sheet
                return self.execInsertDrawings(drawingDescs);
            });

            // on any error, return a promise rejected with a specific error code
            return promise.then(null, function () {
                return SheetUtils.makeRejected('image:insert');
            });
        };

        /**
         * Inserts a new shape object into the active sheet, and selects it
         * afterwards. This method is an implementation helper that does not
         * show any error alerts on failure.
         *
         * @param {String} presetId
         *  The identifier of a predefined shape type.
         *
         * @param {Rectangle} rectangle
         *  The location of the shape. If the size of the rectangle is zero,
         *  the default size will be used.
         *
         * @param {Object} [options]
         *  Optional parameters for the default formatting attributes. Will be
         *  passed to the method DrawingUtils.createDefaultShapeAttributeSet().
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved when the shape object has been
         *  created and inserted into the active sheet, or that will be
         *  rejected on any error.
         */
        this.execInsertShape = function (presetId, rectangle, options) {

            // predefined formatting attributes for the shape object
            var attrSet = DrawingUtils.createDefaultShapeAttributeSet(docModel, presetId, options);

            // calculate default size, if the passed rectangle is empty
            if (rectangle.area() === 0) {

                // default shape size is 1 inch (convert from 1/100 mm to pixels)
                var defSize = docView.convertHmmToPixel(2450);
                // the effective shape size, according to the shape type (may become portrait or landscape format)
                var shapeSize = DrawingUtils.createDefaultSizeForShapeId(presetId, defSize, defSize);

                // copy effective shape size into the rectangle
                rectangle = rectangle.clone();
                rectangle.width = shapeSize.width;
                rectangle.height = shapeSize.height;
            }

            // add the anchor position attributes
            attributePool.extendAttributeSet(attrSet, drawingCollection.getAttributeSetForRectangle(rectangle));
            attrSet.drawing.anchorType = 'twoCell';

            // insert the image objects into the active sheet
            var drawingType = Labels.getPresetShape(presetId).type;
            return self.execInsertDrawings([{ type: drawingType, attrs: attrSet }], function (generator, sheet, position) {
                if (drawingType === 'shape') {
                    attrSet = attributePool.extendAttributeSet(attrSet, DrawingUtils.getParagraphAttrsForDefaultShape(docModel), { clone: true });
                    generator.generateDrawingOperation(Operations.PARA_INSERT, position.concat(0), { attrs: attrSet });
                }
            });
        };

        /**
         * Deletes the specified drawing objects in the active sheet.
         *
         * @param {Array<Array<Number>>} positions
         *  The document positions of all drawing objects to be deleted.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved when all drawing objects have been
         *  deleted successfully, or that will be rejected on any error.
         */
        this.execDeleteDrawings = function (positions) {

            // leave text edit mode before deleting a drawing object
            var promise = docView.leaveTextEditMode();

            // check that the drawing objects in the active sheet are not locked
            promise = promise.then(function () {
                return docView.ensureUnlockedDrawings({ errorCode: 'drawing:delete:locked' });
            });

            // ask user when unsupported drawing objects are going to be deleted
            promise = promise.then(function () {

                // bug 47948: check whether any unsupported (not undoable) drawing object is selected
                var unsupported = positions.some(function (position) {
                    return !drawingCollection.getModel(position).isDeepSupported();
                });
                if (!unsupported) { return false; }

                // ask user whether to delete drawing objects that cannot be restored
                var dialog = new Dialogs.ModalQueryDialog(docView, Labels.DELETE_CONTENTS_QUERY, { title: Labels.DELETE_CONTENTS_TITLE });

                // close dialog automatically after losing edit rights
                docView.closeDialogOnReadOnlyMode(dialog);

                // show dialog, promise resolves with value true, or rejects
                return dialog.show();
            });

            // generate and apply the drawing operations (back to cell selection before deleting the drawing objects)
            return promise.then(function (unsupported) {
                return sheetModel.createAndApplyOperations(function (generator) {
                    docView.removeDrawingSelection();
                    return drawingCollection.generateDeleteDrawingOperations(generator, positions);
                }, { storeSelection: true, undoMode: unsupported ? 'clear' : 'generate' });
            });
        };

        /**
         * Changes the relative order of the specified drawing objects in the
         * active sheet.
         *
         * @param {Array<Array<Number>>} positions
         *  The document positions of all drawing objects to be reordered.
         *
         * @param {String} direction
         *  The moved direction used to reorder the drawing objects. Must be
         *  one of 'front' (pull to front), 'forward' (one step towards
         *  foreground), 'backward' (one step towards background), or 'back'
         *  (push to background).
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved when all drawing objects have been
         *  reordered successfully, or that will be rejected on any error.
         */
        this.execReorderDrawings = function (positions, direction) {

            // leave text edit mode before moving a drawing object (document position changes)
            var promise = docView.leaveTextEditMode();

            // check that the drawing objects in the active sheet are not locked
            promise = promise.then(function () {
                return docView.ensureUnlockedDrawings();
            });

            // generate and apply the drawing operations
            return promise.then(function () {
                return sheetModel.createAndApplyOperations(function (generator) {
                    return drawingCollection.generateReorderDrawingOperations(generator, positions, direction);
                }, { storeSelection: true });
            });
        };

        /**
         * Changes the formatting attributes of the specified drawing objects.
         *
         * @param {Array<Object>} drawingDescs
         *  An array of descriptors with the positions and the formatting
         *  attribute sets for the drawing objects to be changed. Each
         *  descriptor MUST contain the following properties:
         *  - {Array<Number>} drawingDesc.position
         *      The document position of the drawing object to be changed.
         *  - {Object} drawingDesc.attrs
         *      The (incomplete) attribute set to be applied at the drawing
         *      object.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved when all drawing objects have been
         *  modified successfully, or that will be rejected on any error.
         */
        this.execChangeDrawings = function (drawingDescs) {

            // check that the drawing objects in the active sheet are not locked
            var promise = docView.ensureUnlockedDrawings();

            // generate and apply the drawing operations
            return promise.then(function () {
                return sheetModel.createAndApplyOperations(function (generator) {
                    return drawingCollection.generateChangeDrawingOperations(generator, drawingDescs);
                }, { storeSelection: true });
            });
        };

        // item registration --------------------------------------------------

        // register all controller items
        this.registerDefinitions({

            // parent item that is enabled if it is possible to insert drawing objects
            'drawing/insert/enable': {
                parent: 'document/editable/worksheet',
                enable: function () { return !docView.isSheetLocked(); }
            },

            // dialog to insert an image from file system, from Drive, or by URL
            'image/insert/dialog': {
                parent: 'drawing/insert/enable',
                set: function (dialogType) {
                    // leave text edit mode before showing the dialog
                    var promise = docView.leaveTextEditMode();
                    // show the 'Insert Image' dialog
                    promise = promise.then(function () { return ImageUtils.showInsertImageDialogByType(app, dialogType); });
                    // handle the dialog result
                    promise = promise.then(function (imageDesc) { return self.execInsertImages([imageDesc]); });
                    // show warning messages if needed
                    return docView.yellOnFailure(promise);
                }
            },

            // insert a shape object (via frame selection mode)
            'shape/insert': {
                parent: 'drawing/insert/enable',
                // return whether frame selection mode is active (as expected by ShapeTypePicker)
                get: function () { return docView.isFrameSelectionMode('shape'); },
                set: function (presetId, options) {

                    // insert a shape directly on touch devices, or when using the keyboard in the GUI control
                    if (Utils.TOUCHDEVICE || (Utils.getStringOption(options, 'sourceType') !== 'click')) {
                        // insert a shape object into the active cell, show warning messages if needed
                        var rectangle = docView.getCellRectangle(docView.getActiveCell());
                        return docView.yellOnFailure(self.execInsertShape(presetId, rectangle));
                    }

                    // special behavior for line/connector shapes
                    var twoPointMode = Labels.isPresetConnector(presetId);
                    // custom aspect ratio for special shapes
                    var aspectRatio = Labels.getPresetAspectRatio(presetId);

                    // returns the flipping options for connector/line shapes
                    function getFlippingOptions(frameRect) {
                        return twoPointMode ? { flipH: frameRect.reverseX, flipV: frameRect.reverseY } : null;
                    }

                    // renders the preview shape into the frame node (callback for SpresdeheetView.enterFrameSelectionMode)
                    function renderPreviewShape(frameNode, frameRect) {
                        var options = _.extend({ transparent: true }, getFlippingOptions(frameRect));
                        DrawingFrame.drawPreviewShape(app, frameNode, frameRect, presetId, options);
                    }

                    // start frame selection mode
                    var promise = docView.enterFrameSelectionMode('shape', {
                        statusLabel: gt('Insert a shape'),
                        renderer: renderPreviewShape,
                        trackingOptions: { twoPointMode: twoPointMode, aspectRatio: aspectRatio }
                    });

                    // insert a shape into the resulting rectangle
                    promise = promise.then(function (frameRect) {
                        return self.execInsertShape(presetId, frameRect, getFlippingOptions(frameRect));
                    });

                    // show warning messages if needed; but do not return the promise
                    // (GUI must not be blocked while frame selection mode is active)
                    docView.yellOnFailure(promise);
                }
            },

            // Parent item that is enabled if at least one drawing object is selected.
            // The getter returns an array with all selected drawing models.
            'drawing/models': {
                parent: 'document/editable/drawing',
                get: function (positions) { return positions.map(drawingCollection.getModel, drawingCollection).filter(_.identity); }
            },

            // returns the type identifiers of the selected drawing objects as flag set
            'drawing/types': {
                parent: 'drawing/models', // returns the selected drawing models
                get: function (drawingModels) { return Utils.makeSet(_.invoke(drawingModels, 'getType')); }
            },

            // enabled if the selected drawing objects contain a shape or a connector
            'drawing/type/shape': {
                parent: 'drawing/types', // the type identifiers of the selected drawing models, as flag set
                enable: function (typeSet) { return ('shape' in typeSet) || ('connector' in typeSet); }
            },

            // returns the title label for the drawing toolbar, according to the selected drawing objects
            'drawing/type/label': {
                parent: 'drawing/types', // the type identifiers of the selected drawing models, as flag set
                get: function (typeSet) {
                    var mixedType = Utils.hasSingleProperty(typeSet) ? Utils.getAnyPropertyKey(typeSet) : null;
                    return Labels.getDrawingTypeLabel(mixedType);
                }
            },

            // Parent item that is enabled if at least one drawing object is selected, AND the active
            // sheet is unlocked. The getter returns an array with all selected drawing models.
            'drawing/operation': {
                parent: 'drawing/models',
                enable: function () { return !docView.isSheetLocked(); }
            },

            'drawing/delete': {
                parent: 'drawing/operation',
                set: function () {
                    // delete the selected drawing objects
                    var promise = self.execDeleteDrawings(docView.getSelectedDrawings());
                    // show warning messages if needed
                    return docView.yellOnFailure(promise);
                }
            },

            'drawing/order': {
                parent: 'drawing/operation',
                set: function (direction) {
                    // reorder the specified drawing objects
                    var promise = self.execReorderDrawings(docView.getSelectedDrawings(), direction);
                    // show warning messages if needed
                    return docView.yellOnFailure(promise);
                }
            },

            'drawing/change/explicit': {
                parent: 'drawing/operation',
                set: function (drawingDescs) {
                    // change the specified drawing objects
                    var promise = self.execChangeDrawings(drawingDescs);
                    // show warning messages if needed
                    return docView.yellOnFailure(promise);
                }
            },

            // flipping and rotation ------------------------------------------

            // Parent item for drawing objects that can be flipped and rotated. The getter
            // returns an array with all drawing models contained in the selection that can
            // be flipped.
            'drawing/operation/transform': {
                parent: 'drawing/operation',
                enable: function () { return this.getValue().length > 0; },
                get: function (drawingModels) { return drawingModels.filter(isRotatableModel); }
            },

            // (stateless command) flips the selected drawing objects horizontally
            'drawing/transform/fliph': {
                parent: 'drawing/operation/transform',
                set: function () {
                    return implChangeDrawings(this.getParentValue(), function (drawingModel) {
                        return { drawing: { flipH: !drawingModel.isFlippedH(), rotation: -drawingModel.getRotationDeg() } };
                    });
                }
            },

            // (stateless command) flips the selected drawing objects vertically
            'drawing/transform/flipv': {
                parent: 'drawing/operation/transform',
                set: function () {
                    return implChangeDrawings(this.getParentValue(), function (drawingModel) {
                        return { drawing: { flipV: !drawingModel.isFlippedV(), rotation: -drawingModel.getRotationDeg() } };
                    });
                }
            },

            // (stateless command) rotates the selected drawing objects by the passed angle
            'drawing/transform/rotate': {
                parent: 'drawing/operation/transform',
                enable: function () { return !odf; }, // bug 52930: rotation disabled due to bugs in LO when displaying rotated shapes
                set: function (angle) {
                    return implChangeDrawings(this.getParentValue(), function (drawingModel) {
                        return { drawing: { rotation: drawingModel.getRotationDeg() + angle } };
                    });
                }
            },

            // getter returns the current rotation angle of all rotatable drawings;
            // setter rotates the selected drawing objects to a specific rotation angle
            'drawing/transform/angle': {
                parent: 'drawing/operation/transform',
                enable: function () { return !odf; }, // bug 52930: rotation disabled due to bugs in LO when displaying rotated shapes
                get: function (drawingModels) { return DrawingUtils.getMixedAttributeValue(drawingModels, 'drawing', 'rotation'); },
                set: function (angle) { return implChangeDrawings(this.getParentValue(), { drawing: { rotation: angle } }); }
            },

            // line attributes ------------------------------------------------

            // Parent item for drawing objects whose line attributes can be modified. The getter returns an
            // array with all drawing models contained in the selection that support line attributes, either
            // selected directly, or the children in selected group objects.
            'drawing/operation/line': {
                parent: 'drawing/operation',
                enable: function () { return this.getValue().length > 0; },
                get: function (drawingModels) { return DrawingUtils.resolveModelsDeeply(drawingModels, isLineFormatModel); }
            },

            // preset border style of all selected drawings
            'drawing/border/style': {
                parent: 'drawing/operation/line',
                get: DrawingUtils.getMixedPresetBorder,
                set: function (preset) {
                    // create the line attributes for the passed preset border, clear color when hiding the line
                    var defLineAttrs = DrawingUtils.resolvePresetBorder(preset);
                    if (defLineAttrs.type === 'none') { defLineAttrs.color = null; }
                    // generate and apply the operations for all selected drawing objects
                    return implChangeDrawings(this.getParentValue(), function (drawingModel, mergedAttrSet) {
                        var attrSet = { line: _.clone(defLineAttrs) };
                        // replace automatic color with black
                        if ((defLineAttrs.type !== 'none') && (mergedAttrSet.line.color.type === 'auto')) {
                            attrSet.line.color = Color.BLACK;
                        }
                        return attrSet;
                    });
                }
            },

            // border color of all selected drawings
            'drawing/border/color': {
                parent: 'drawing/operation/line',
                get: DrawingUtils.getMixedBorderColor,
                set: function (color) {
                    return implChangeDrawings(this.getParentValue(), function (drawingModel, mergedAttrSet) {
                        var attrSet = { line: { color: color } };
                        // change invisible borders to solid borders
                        if (!DrawingUtils.hasVisibleBorder(mergedAttrSet.line)) { attrSet.line.type = 'solid'; }
                        return attrSet;
                    });
                }
            },

            // fill attributes ------------------------------------------------

            // Parent item for drawing objects whose fill attributes can be modified. The getter returns an
            // array with all drawing models contained in the selection that support fill attributes, either
            // selected directly, or the children in selected group objects.
            'drawing/operation/fill': {
                parent: 'drawing/operation',
                enable: function () { return this.getValue().length > 0; },
                get: function (drawingModels) { return DrawingUtils.resolveModelsDeeply(drawingModels, isFillFormatModel); }
            },

            // fill color of all selected drawings
            'drawing/fill/color': {
                parent: 'drawing/operation/fill',
                get: DrawingUtils.getMixedFillColor,
                set: function (color) {
                    var type = (color.type === 'auto') ? 'none' : 'solid';
                    return implChangeDrawings(this.getParentValue(), { fill: { type: type, color: color } });
                }
            },

            // connector attributes -------------------------------------------

            // Parent item for connector line objects. The getter returns an array with all connector models
            // contained in the selection, either selected directly, or the children in selected group objects.
            'drawing/operation/connector': {
                parent: 'drawing/operation',
                enable: function () { return this.getValue().length > 0; },
                get: function (drawingModels) { return DrawingUtils.resolveModelsDeeply(drawingModels, isConnectorModel); }
            },

            // arrow types (head and tail) of all selected connector lines
            'drawing/connector/arrows': {
                parent: 'drawing/operation/connector',
                get: function (drawingModels) {
                    var arrowHead = DrawingUtils.getMixedAttributeValue(drawingModels, 'line', 'headEndType');
                    var arrowTail = DrawingUtils.getMixedAttributeValue(drawingModels, 'line', 'tailEndType');
                    return (arrowHead && arrowTail) ? (arrowHead + ':' + arrowTail) : null;
                },
                set: function (presetArrows) {
                    var parts = presetArrows.split(':');
                    return implChangeDrawings(this.getParentValue(), { line: { headEndType: parts[0], tailEndType: parts[1] } });
                }
            },

            // text content attributes ----------------------------------------

            // Parent item for drawing objects with text contents. The getter returns an array with all
            // drawing models contained in the selection that suuport text contents, either selected
            // directly, or the children in selected group objects.
            'drawing/operation/text': {
                parent: 'drawing/operation',
                enable: function () { return this.getValue().length > 0; },
                get: function (drawingModels) { return DrawingUtils.resolveModelsDeeply(drawingModels, isTextFrameModel); }
            },

            // fixed item key, as expected by the text framework
            'drawing/verticalalignment': {
                parent: 'drawing/operation/text',
                get: function (drawingModels) { return DrawingUtils.getMixedAttributeValue(drawingModels, 'shape', 'anchor'); },
                set: function (anchorMode) { return implChangeDrawings(this.getParentValue(), { shape: { anchor: anchorMode } }); }
            }
        });

        // initialization -----------------------------------------------------

        // initialize sheet-dependent class members according to the active sheet
        this.listenTo(docView, 'change:activesheet', function () {
            sheetModel = docView.getSheetModel();
            drawingCollection = sheetModel.getDrawingCollection();
        });

        // destroy all class members on destruction
        this.registerDestructor(function () {
            self = app = docView = docModel = attributePool = null;
            sheetModel = drawingCollection = null;
        });

    } // class DrawingMixin

    // exports ================================================================

    return DrawingMixin;

});
