/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author York Richter <york.richter@open-xchange.com>
 */

define('io.ox/office/settings/userdictionary', [
    'io.ox/core/event',
    'io.ox/office/tk/config',
    'io.ox/office/tk/utils',
    'io.ox/office/tk/keycodes',
    'io.ox/office/tk/dialogs',
    'gettext!io.ox/office/settings/main'
], function (Event, Config, Utils, KeyCodes, Dialogs, gt) {

    'use strict';

    var
        // the key for save/get the user dictionary to the user settings
        SETTINGS_KEY = 'userdictionary',
        // for sorting the elements in the dictionary
        wordSortIteratee = function (word) {
            return word.toLowerCase();
        },
        // the user dictionary
        dictionary,
        // the min length of the word
        MIN_WORD_LENGTH = 1;

    function UserDictionary() {

        // public methods -----------------------------------------------------
        this.refresh = function (settings) {

            settings.append($('<div style="display: block; height: 0; margin-top: -15px">'));
            settings.appendButton(/*#. Button label for open the user dictionary dialog */gt('Edit user dictionary'), new Dialog().show).attr('data-action', 'edituserdic');
            settings.append($('<div style="display: block; height: 0; margin-bottom: 20px">'));
        };
    }

    // static functions -------------------------------------------------------

    /**
     * Return the Dictionary.
     * @returns {String[]} the dictionary
     */
    UserDictionary.getDictionary = function () {

        if (!dictionary) {
            dictionary = Config.get(SETTINGS_KEY, []);
        }

        return dictionary;
    };

    /**
     * Add a word to the User Dictionary.
     * @param {String} word to add
     */
    UserDictionary.addWord = function (word) {
        var wordToAdd = getTrimmedWord(word);

        if (!existWord(dictionary, wordToAdd)) {
            saveDictionary([wordToAdd]);
        }
    };

    /**
     * Remove a word from the User Dictionary.
     * @param {String} word for remove
     */
    UserDictionary.removeWord = function (word) {
        var wordToAdd = getTrimmedWord(word);

        if (existWord(dictionary, wordToAdd)) {
            saveDictionary([], [wordToAdd]);
        }
    };

    // private functions ------------------------------------------------------

    /**
     * Save the Dictionary to the User Settings an triger the 'save' event.
     * @param {type} newWords the new words added to the dictionary.
     * @param {type} removedWords the words removed from the dictionary.
     */
    function saveDictionary(newWords, removedWords) {
        var userDicChanges = false;
        if (!_.isEmpty(newWords) || !_.isEmpty(removedWords)) {

            _.each(newWords, function (newWord) {
                var word = getTrimmedWord(newWord);
                if (!existWord(dictionary, word)) {
                    dictionary.push(word);
                    userDicChanges = true;
                }
            });

            _.each(removedWords, function (removedWord) {
                var word = getTrimmedWord(removedWord);
                if (existWord(dictionary, word)) {
                    dictionary = _.without(dictionary, word);
                    userDicChanges = true;
                }
            });
        }

        if (userDicChanges) {
            Config.set(SETTINGS_KEY, dictionary);
            UserDictionary.trigger('save', _.isArray(newWords) ? newWords : [], _.isArray(removedWords) ? removedWords : []);
        }
    }

    /**
     * Check if the list contains the word and if the word length is valid.
     * @param {String[]} the list to check if the word exist
     * @param {String} word to check
     * @returns {Boolean} true if the word exist, otherwise false.
     */
    function existWord(list, word) {
        return word.length >= MIN_WORD_LENGTH && _.contains(list, word);
    }

    /**
     * Remove spaces from Word and trim the word.
     * @param {String} word to trim
     * @returns {String} the trimmed word.
     */
    function getTrimmedWord(word) {
        return word ? word.trim() : '';
    }

    /**
     * The User Dictionary Edit Dialog which is visible if the user presses the setting's edit button.
     */
    function Dialog() {

        var list,
            wordInput,
            addButton,
            dictionaryClone;

        /**
         * If the user press the save button, the dictionary will be updated with the new word list and a save event will be fired.
         */
        function onPressSaveButton() {
            var newWords = _.difference(dictionaryClone, dictionary),
                removedWords = _.difference(dictionary, dictionaryClone);

            saveDictionary(newWords, removedWords);
        }

        /**
         * The value of the word input field will be splitted by ' ' and add the word\s to dialog word list and the ui.
         */
        function onClickAddButton() {
            var words = getWordValue().split(' ');
            _.each(words, addWord);
            // clean input
            wordInput.val('');
            toggleAddButton();
            Utils.setFocus(wordInput);
        }

        /**
         * If it is a new word add it to the word list and the UI.
         * @param {String} word to add
         */
        function addWord(word) {
            var wordToAdd = getTrimmedWord(word);
            if (wordToAdd && !existWord(dictionaryClone, wordToAdd)) {
                dictionaryClone.push(wordToAdd);
                addWordToListElement(wordToAdd, true);
            }
        }

        /**
         * Remove the selected word.
         */
        function removeSelectedWord() {
            var selectItem;
            _.each(list.children(), function (item) {
                item = $(item);
                if (item.is(':focus')) {
                    selectItem = getItemToSelect(item);
                    removeWord(item);
                }
            });
            if (selectItem && selectItem.length === 1) {
                Utils.setFocus(selectItem);
            } else {
                Utils.setFocus(wordInput);
            }
        }

        /**
         * Remove the word from the list and UI.
         * @param {jQuery} listItem to remove from the UI.
         */
        function removeWord(listItem) {
            var word = listItem.data('word');

            listItem.remove();
            if (existWord(dictionaryClone, word)) {
                dictionaryClone = _.without(dictionaryClone, word);
            }
        }

        function getItemToSelect(listItem) {
            var item = listItem.next();
            if (!item || item.length === 0) {
                item = listItem.prev();
            }
            return item;
        }

        /**
         * Event handler to the words from the wordInput if the user press 'Enter'
         * or delete selected word if the user click 'Delete' or 'Backspace'.
         */
        function keyDownHandler(event) {
            if (KeyCodes.matchKeyCode(event, KeyCodes.ENTER)) {
                if (isValidWordValueLength()) {
                    onClickAddButton();
                    event.stopPropagation();
                }
            } else if (KeyCodes.matchKeyCode(event, KeyCodes.BACKSPACE) || KeyCodes.matchKeyCode(event, KeyCodes.DELETE)) {
                removeSelectedWord();
            }
        }

        /**
         * @returns {String} the value of the input element.
         */
        function getWordValue() {
            return getTrimmedWord(wordInput.val());
        }

        /**
         * Check if the word length in the input field is valid.
         * @returns {Boolean} true if the length is valid, otherwise false.
         */
        function isValidWordValueLength() {
            return getWordValue().length >= MIN_WORD_LENGTH;
        }

 /**
         * Create a list element for the word and add it to the list (UI).
         * @param {String} word the word to add.
         * @param {Boolean} if true insert the element as the first element
         *                  of the list, otherwise add the word element to the end of the list
         */
        function addWordToListElement(word, firstElement) {
            var wordLabel = $('<div>')
                .css({
                    whiteSpace: 'nowrap',
                    width: 'calc(100% - 15px)',
                    textOverflow: 'ellipsis',
                    overflow: 'hidden',
                    float: 'left',
                    paddingRight: 10
                })
                .text(word),
                deleteButton = $('<div class="fa fa-trash-o">')
                .css({
                    visibility: 'hidden',
                    lineHeight: 'inherit',
                    'aria-label': /*#. Label for delete a word from the User Dictionary. */ gt('Delete word')
                }),
                listItem = $('<a href="#" class="list-group-item">')
                .css({
                    borderRightWidth: 0,
                    borderLeftWidth: 0,
                    paddingTop: 4,
                    paddingBottom: 4
                })
                .attr('tabindex', 0);

            listItem.append(wordLabel).append(deleteButton);

            if (firstElement) {
                listItem.prependTo(list);
            } else {
                list.append(listItem);
            }

            listItem.on('mouseover focus', function () {
                deleteButton.css('visibility', 'visible');
            });

            listItem.on('mouseout', function () {
                if (!listItem.is(':focus')) {
                    deleteButton.css('visibility', 'hidden');
                }
            });

            listItem.on('blur', function () {
                deleteButton.css('visibility', 'hidden');
            });

            listItem.data('word', word);

            deleteButton.on('click', function () {
                removeWord(listItem);
            });
        }

        function toggleAddButton() {
            addButton.toggleClass('disabled', !isValidWordValueLength());
        }

        /**
         * Fill the UI list with the words from the dictionary.
         */
        function fillList() {
            _.each(_.sortBy(UserDictionary.getDictionary(), wordSortIteratee), function (word) {
                addWordToListElement(word, false);
            });
        }

        /**
         * Show the User Dictionary Dialog.
         */
        this.show = function () {

            var dialog = new Dialogs.ModalDialog('io-ox-office-settings', {
                okLabel: /*#. Button label for the save button for the user dictionary dialog. */gt('Save'),
                title: /*#. Title of the user dictionary dialog. */gt('User dictionary'),
                validator: onPressSaveButton
            });
            var placeholdertext = /*#. Placeholder text for the input field for add words to the user dictionary */ gt('Add a new word');

            wordInput = $('<input>', {
                tabindex: 0,
                role: 'textbox',
                'aria-label': placeholdertext,
                placeholder: placeholdertext,
                type: 'text'
            }).addClass('form-control');

            wordInput.on('input', toggleAddButton);

            addButton = $('<a class="input-group-addon btn disabled" tabindex="0">');
            addButton.text(/*#. Button label for a add word to the user dictionary */ gt('Add')).on('click', onClickAddButton);

            dialog.append(($('<div class="input-group form-group">').append(wordInput, addButton)));

            list = $('<div class="list-group">').css({
                height: _.device('smartphone') ? $(window).height() - 300 : 307,
                overflowY: 'auto',
                minHeight: 60
            });

            dialog.append($('<div class="panel panel-default">').append(list));

            wordInput.on('keydown', keyDownHandler);
            addButton.on('keydown', keyDownHandler);
            $(document).on('keydown', keyDownHandler);

            dialog.show().always(function () {
                // Clean-up after closing the dialog
                $(document).off('keydown', keyDownHandler);
                wordInput.on('keydown', keyDownHandler);
            });

            dictionaryClone = _.clone(UserDictionary.getDictionary());
            fillList();
        };

    }

    // initialization -----------------------------------------------------

    Event.extend(UserDictionary);

    return UserDictionary;

});

