/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Peter Seliger <peter.seliger@open-xchange.com>
 */

define('io.ox/office/settings/spellchecklanguagenotification', [
    'io.ox/core/event',
    'io.ox/office/tk/config',
    'io.ox/office/tk/dialogs',
    'io.ox/office/tk/locale/localedata',
    'gettext!io.ox/office/settings/main',
    'less!io.ox/office/settings/spellchecklanguagenotification'
], function (Event, Config, Dialogs, LocaleData, gt) {

    'use strict';

    var
        // the keys for save/get the user's spell-check language blacklist/whitelist to the user settings
        blacklistSettingsKey = 'module/unsupportedspellchecklanguageblacklist',
        whitelistSettingsKey = 'module/unsupportedspellchecklanguagewhitelist',

        // the user's language blacklist/whitelist
        languageBlacklist,
        languageWhitelist,

        editSettingsActionRegistry = {};

    // constructor ------------------------------------------------------------

    function SpellcheckNotification() {

        // public methods -----------------------------------------------------
        this.refresh = function (settings) {
            if (SpellcheckNotification.doesLanguageListExist()) {

                settings.append($('<div style="display: block; height: 0; margin-top: -15px">'));
                settings.appendButton(/*#. Button label for open the user spell-check languages dialog */gt('Set language notification'), (new Dialog()).show).attr('data-action', 'editspellchecknotification');
                settings.append($('<div style="display: block; height: 0; margin-bottom: 20px">'));
            }
        };
    }

    // static functions -------------------------------------------------------

    /**
     * does dynamically at runtime register an action (related to an action link)
     * once per application and keeps track on each application specific registration.
     * @param app
     */
    SpellcheckNotification.registerEditSettingsAction = function (app) {
        var
            applicationId = app.cid,
            registryItem  = editSettingsActionRegistry[applicationId];

        if (!registryItem || !registryItem.isRegistered) {

            app.getController().registerDefinition('document/settings/editspellchecknotification', {
                set: triggerEditSpellcheckNotification
            });

            editSettingsActionRegistry[applicationId] = {
                isRegistered: true
            };
        }
    };

    SpellcheckNotification.saveToLanguageWhitelist = function (languageList) {
        languageList      = languageList.filter(isValidLanguageKey); // sanitize received data - see BUG 50606 [https://bugs.open-xchange.com/show_bug.cgi?id=50606]
        languageWhitelist = _.unique(languageList.concat(SpellcheckNotification.getLanguageWhitelist()));

        Config.set(whitelistSettingsKey, languageWhitelist);
    };

    /**
     * Return the Whitelist of Spell-check Languages.
     * @returns {String[]} the whitelist of spell-check languages
     */
    SpellcheckNotification.getLanguageWhitelist = function () {
        if (!languageWhitelist) {
            languageWhitelist = Config.get(whitelistSettingsKey, []).filter(isValidLanguageKey); // sanitize list - see BUG 50606 [https://bugs.open-xchange.com/show_bug.cgi?id=50606]
        }
        return languageWhitelist;
    };

    /**
     * Return the Blacklist of Spell-check Languages.
     * @returns {String[]} the blacklist of spell-check languages
     */
    SpellcheckNotification.getLanguageBlacklist = function () {
        if (!languageBlacklist) {
            languageBlacklist = Config.get(blacklistSettingsKey, []).filter(isValidLanguageKey); // sanitize list - see BUG 50606 [https://bugs.open-xchange.com/show_bug.cgi?id=50606]
        }
        return languageBlacklist;
    };

    /**
     * Does return whether or not there are any spell-check language items listed specifically under a user's settings.
     * @returns {boolean}
     * whether or not there are any user specific spell-check language items already listed/available.
     */
    SpellcheckNotification.doesLanguageListExist = function () {
        var
            currentBlacklist        = _.clone(SpellcheckNotification.getLanguageBlacklist()),
            currentWhitelist        = _.clone(SpellcheckNotification.getLanguageWhitelist()),

            unsupportedLanguageList = _.unique(currentBlacklist.concat(currentWhitelist)),
            isLanguageListDoesExist = (unsupportedLanguageList.length >= 1);

        return isLanguageListDoesExist;
    };

    // private functions ------------------------------------------------------

    function isValidLanguageKey(languageKey) {
        return (_.isString(languageKey)/* && (languageKey !== '')*/ && !!$.trim(languageKey) && (languageKey !== 'none'));
    }

    function triggerEditSpellcheckNotification() {
        if (SpellcheckNotification.doesLanguageListExist()) {

            (new Dialog()).show();
        }
    }

    /**
     * The Language Notification Edit Dialog which is visible if the user presses the setting's edit button.
     */
    function Dialog() {

        var
            $groupItemBlueprint = $('<div class="checkbox">'),
            $labelBlueprint     = $('<label class="control-label">'),
            $checkboxBlueprint  = $('<input type="checkbox" tabindex="1">'),

            $fieldset,
            $languageList,

            currentBlacklist,
            currentWhitelist,

            checkboxRegistry = {};

        /**
         * If the user press the save button, the dictionary will be updated with the new word list and a save event will be fired.
         */
        function onPressSaveButton() {
            var
                updatedBlacklist = [],
                updatedWhitelist = [];

            Object.keys(checkboxRegistry).forEach(function (languageKey) {
                var
                    elmCheckbox = checkboxRegistry[languageKey];

                if (elmCheckbox.checked) {
                    updatedBlacklist.push(languageKey);
                  //updatedWhitelist.push(languageKey);
                } else {
                  //updatedBlacklist.push(languageKey);
                    updatedWhitelist.push(languageKey);
                }
            });
            languageBlacklist = updatedBlacklist;
            languageWhitelist = updatedWhitelist;

            Config.set(blacklistSettingsKey, updatedBlacklist);
            Config.set(whitelistSettingsKey, updatedWhitelist);
        }

        /**
         * Fill the UI list with the words from the dictionary.
         */
        function renderCheckboxList(languageList) {

            languageList.map(function (languageKey) {
                return {
                    key: languageKey,
                    name: LocaleData.getLanguageName(languageKey.replace('-', '_'))
                };
            }).filter(function (languageTuple) { // sanitize list - see BUG 50606 [https://bugs.open-xchange.com/show_bug.cgi?id=50606]

                return (!!languageTuple.name && isValidLanguageKey(languageTuple.key));

            }).sort(function (a, b) {
                var
                    a_name = a.name.toLowerCase(),
                    b_name = b.name.toLowerCase();

                return (((a_name < b_name) && -1) || ((a_name > b_name) && 1) || 0);

            }).forEach(function (languageTuple) {
                var
                    languageKey = languageTuple.key,
                    languageName = languageTuple.name,

                    elmCheckbox = checkboxRegistry[languageKey] = ($checkboxBlueprint.clone())[0]; // native dom node.

                elmCheckbox.name = languageKey;
                elmCheckbox.checked = false;
              //elmCheckbox.checked = true;

                $languageList.append(
                    $groupItemBlueprint.clone().append(
                        $labelBlueprint.clone().append(
                            elmCheckbox,
                            $('<span>' + languageName + '</span>')
                        )
                    )
                );
            });
            currentBlacklist.forEach(function (languageKey) {
                var
                    elmCheckbox = checkboxRegistry[languageKey];

                if (elmCheckbox) {
                    elmCheckbox.checked = true;
                  //elmCheckbox.checked = false;
                }
            });
        }

        /**
         * Show the edit dialog of how to handle notification(s) about unsupported spell-check language(s).
         */
        this.show = function () {
            var
                dialog = new Dialogs.ModalDialog('io-ox-office-settings', {
                    okLabel: /*#. Button label for the save button for the user dictionary dialog. */gt('Save'),
                    title: /*#. Title of the user's spell-check language blacklist dialog. */gt('Unsupported spell checker notification'),
                    validator: onPressSaveButton
                }),
                copyEditList  = /*#. Text that describes a black-list of languages, a user wishes to be informed about, in case each of it is unsupported by spellchecking */ gt('Suppress missing spell checker warnings for the following languages:'),

                unsupportedLanguageList;

            currentBlacklist = _.clone(SpellcheckNotification.getLanguageBlacklist());
            currentWhitelist = _.clone(SpellcheckNotification.getLanguageWhitelist());

            unsupportedLanguageList = _.unique(currentBlacklist.concat(currentWhitelist));

            $fieldset = $([

                '<fieldset>',
                '<legend class="sectiontitle expertmode">',

                copyEditList,

                '</legend>',
                '</fieldset>'

            ].join(''));

            $languageList = $('<div class="form-group expertmode">')/*.css({
                height: _.device('smartphone') ? $(window).height() - 300 : 307,
                overflowY: 'auto',
                minHeight: 60
            })*/;

            renderCheckboxList(unsupportedLanguageList);

            dialog.append($fieldset.append($languageList));

            dialog.getBody().addClass('settings-details');

            dialog.show();
        };
    }

    // initialization ---------------------------------------------------------

    Event.extend(SpellcheckNotification);

    return SpellcheckNotification;

});
