/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Mario Schroeder <mario.schroeder@open-xchange.com>
 */

define('io.ox/office/presenter/errormessages', [
    'io.ox/office/tk/utils',
    'gettext!io.ox/office/presenter/main'
], function (Utils, gt) {

    'use strict';

    /**
     * Default error messages
     */
    var GENERAL_ERROR              = gt('A general error occurred. Please try to reload the document. In case this does not help, please contact your system administrator.');
    var GENERAL_NETWORK_ERROR      = gt('A general network error occurred. Please try to reload the document. In case this does not help, please contact your system administrator.');
    var GENERAL_FILE_STORAGE_ERROR = gt('A general file storage error occurred. Please try to reload the document. In case this does not help, please contact your system administrator.');

    /**
     * Drive error messages
     */
    var FILE_STORAGE = {
        'FILE_STORAGE-0008': gt('The requested document does not exist.'),
        'FILE_STORAGE-0017': gt('The presentation cannot be started. Please check the URL or contact the presenter.'),
        'FILE_STORAGE-0057': gt('This document does not have any content.'),
        'FILE_STORAGE-0058': gt('You do not have the appropriate permissions to read the document.')
    };
    FILE_STORAGE['FILE_STORAGE-0018'] = FILE_STORAGE['FILE_STORAGE-0028'] = FILE_STORAGE['FILE_STORAGE-0047'] = FILE_STORAGE['FILE_STORAGE-0017'];
    FILE_STORAGE['FILE_STORAGE-0026'] = FILE_STORAGE['FILE_STORAGE-0055'] = FILE_STORAGE['FILE_STORAGE-0008'];
    FILE_STORAGE['FILE_STORAGE-0062'] = FILE_STORAGE['FILE_STORAGE-0058'];

    var IFO = {
        'IFO-0300': FILE_STORAGE['FILE_STORAGE-0008'],
        'IFO-0400': FILE_STORAGE['FILE_STORAGE-0058'],
        'IFO-0438': FILE_STORAGE['FILE_STORAGE-0008'],
        'IFO-0500': FILE_STORAGE['FILE_STORAGE-0057']
    };

    var DRIVE_ERROR_MESSAGES = _.extend({}, FILE_STORAGE, IFO);

    /**
     * Realtime error messages
     */
    var SVL = {
        'SVL-0002': FILE_STORAGE['FILE_STORAGE-0017'],
        'SVL-0004': FILE_STORAGE['FILE_STORAGE-0017'],
        'SVL-0010': FILE_STORAGE['FILE_STORAGE-0017'],
        'SVL-0015': gt('A general network error occurred. Please contact your system administrator.'),
        'SVL-0030': FILE_STORAGE['FILE_STORAGE-0017']
    };
    SVL['SVL-0016'] = SVL['SVL-0015'];

    var RT_STANZA = {
        'RT_STANZA-1015': gt('You can\'t present the same document more than once.')
    };

    var RT_ERROR_MESSAGES = _.extend({}, SVL, RT_STANZA);

    /**
     * Document conversion error messages
     */
    var DOC_CONVERTER_ERROR_MESSAGES = {
        importError: gt('An error occurred loading the document so it cannot be displayed.'),
        filterError: gt('An error occurred converting the document so it cannot be displayed.'),
        passwordProtected: gt('This document is password protected and cannot be displayed.')
    };

    /**
     * A central class that maps the file storage, realtime and document conversion error codes to user readable error messages.
     */
    var ErrorMessages = {

        /**
         * Creates a user readable error message from error objects provided
         * by Realtime, Drive and the Document Converter.
         * An unknown error code will be translated to a general error message.
         *
         * @param {Object} error
         *  The error object to get a user-interface error message for.
         *
         * @param {Object} [options]
         *  @param {String} options.category
         *      The error category to provide more specific messages for unknown error codes.
         *      Supported categories are 'rt', 'drive' and 'conversion'.
         *
         * @returns {String}
         *  A user readable error message.
         */
        getErrorMessage: function (error, options) {
            // the error code provided by Drive and Realtime
            var errorCode  = (_.isObject(error) && _.isString(error.error) && error.code) || null;
            // the error cause provided by document conversion
            var errorCause = Utils.getStringOption(error, 'cause', null);
            // the error category
            var category   = Utils.getStringOption(options, 'category', null);
            // the resulting message
            var message    = DRIVE_ERROR_MESSAGES[errorCode] || RT_ERROR_MESSAGES[errorCode] || DOC_CONVERTER_ERROR_MESSAGES[errorCause];

            if (!message) {
                switch (category) {
                    case 'conversion':
                        message = DOC_CONVERTER_ERROR_MESSAGES.importError;
                        break;

                    case 'drive':
                        message = GENERAL_FILE_STORAGE_ERROR;
                        break;

                    case 'rt':
                        message = GENERAL_NETWORK_ERROR;
                        break;

                    default:
                        message = GENERAL_ERROR;
                        break;
                }
            }

            return message;
        },

        /**
         * Creates a user readable error message from the provided Realtime error object.
         * An unknown error code will be translated to a general Realtime error message.
         *
         * @param {Object} error
         *  The error object to get a user-interface error message for.
         */
        getRealtimeErrorMessage: function (error) {
            return this.getErrorMessage(error, { category: 'rt' });
        },

        /**
         * Creates a user readable error message from the provided Drive error object.
         * An unknown error code will be translated to a general Drive error message.
         *
         * @param {Object} error
         *  The error object to get a user-interface error message for.
         */
        getDriveErrorMessage: function (error) {
            return this.getErrorMessage(error, { category: 'drive' });
        },

        /**
         * Creates a user readable error message from the provided document conversion error object.
         * An unknown error code will be translated to a document conversion import error message.
         *
         * @param {Object} error
         *  The error object to get a user-interface error message for.
         */
        getConversionErrorMessage: function (error) {
            return this.getErrorMessage(error, { category: 'conversion' });
        }
    };

    return ErrorMessages;

});
