/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Ingo Schmidt-Rosbiegal <ingo.schmidt-rosbiegal@open-xchange.com>
 */
define('io.ox/office/presentation/utils/presentationutils', [
    'io.ox/office/tk/utils',
    'io.ox/office/editframework/utils/attributeutils',
    'io.ox/office/textframework/utils/dom',
    'gettext!io.ox/office/presentation/main'
], function (Utils, AttributeUtils, DOM, gt) {

    'use strict';

    // static class PresentationUtils =========================================

    var PresentationUtils = {};

    PresentationUtils.PLACEHOLDER_DRAWING_IMPORTANT_ATTRS = ['left', 'top', 'width', 'height', 'rotation', 'flipH', 'flipV'];

    /**
     * The default type for place holders in presentation attributes.
     *
     * @constant
     */
    PresentationUtils.DEFAULT_PLACEHOLDER_TYPE = 'body';

    /**
     * The table place holder type.
     *
     * @constant
     */
    PresentationUtils.TABLE_PLACEHOLDER_TYPE = 'tbl';

    /**
     * The image place holder type.
     *
     * @constant
     */
    PresentationUtils.IMAGE_PLACEHOLDER_TYPE = 'pic';

    /**
     * A name used to differentiate between the two place holder types 'body'.
     * There is a content body, if 'body' is not set a explicit attribute and
     * a text body, if the phType is explicitely set to 'body'. This marker
     * can be used to mark content body place holders.
     *
     * @constant
     */
    PresentationUtils.CONTENTBODY = 'contentbody';

    /**
     * The title types for place holders title drawings.
     */
    PresentationUtils.TITLE_PLACEHOLDER_TYPES = {
        title: 1,
        ctrTitle: 1
    };

    PresentationUtils.TITLE_OR_SUBTITLE_PLACEHOLDER_TYPES = {
        title: 1,
        ctrTitle: 1,
        subTitle: 1
    };

    /**
     * The place holder types, whose content cannot be modified or that cannot be
     * deleted on master slides in ODF.
     */
    PresentationUtils.ODF_MASTER_READONLY_TYPES = {
        title: 1,
        body: 1
    };

    /**
     * The place holder drawing types, in that no text content can be inserted.
     */
    PresentationUtils.NO_TEXT_PLACEHOLDER_TYPE = {
        media: 1,
        clipArt: 1,
        chart: 1,
        dgm: 1,
        tbl: 1,
        pic: 1
    };

    /**
     * Defining the name of that attributes family that must be defined as explicit
     * attribute to a place holder drawing that cannot contain text.
     * If a place holder drawing of type 'tbl' contains the 'table' family in its
     * explicit attributes, it cannot be empty (in this case the template images for
     * inserting content must not be inserted). The same is true for a place holder
     * drawing of type 'pic', if the explicit attributes contain the image family.
     */
    PresentationUtils.NO_TEXT_PLACEHOLDER_CONTENT_PROPERTY = {
        media: null,
        clipArt: 'image',
        chart: null,
        dgm: null,
        tbl: 'table',
        pic: 'image'
    };

    /**
     * The default texts that are inserted, if the place holder drawing is empty.
     */
    PresentationUtils.PLACEHOLDER_TEMPLATE_TEXTS = {
        body: gt('Click to add text'),
        title: gt('Click to add title'),
        ctrTitle: gt('Click to add title'),
        subTitle: gt('Click to add subtitle'),
        media: gt('Media clip'),
        clipArt: gt('Online image'),
        chart: gt('Chart'),
        dgm: gt('Smart Art'),
        tbl: gt('Click to add table'),
        pic: gt('Click to add image'),
        sldNum: gt('Slide Number'),
        ftr: gt('Footer'),
        dt: gt('Date')
    };

    /**
     * The default texts that are inserted, if the place holder drawing is empty (ODF format).
     */
    PresentationUtils.PLACEHOLDER_TEMPLATE_TEXTS_ODF = {
        body: gt('Click to add text'),
        title: gt('Click to add title'),
        ctrTitle: gt('Click to add title'),
        subTitle: gt('Click to add text'),
        media: gt('Media clip'),
        clipArt: gt('Online image'),
        chart: gt('Chart'),
        dgm: gt('Smart Art'),
        tbl: gt('Click to add table'),
        pic: gt('Click to add image'),
        sldNum: gt('Slide Number'),
        ftr: gt('Footer'),
        dt: gt('Date')
    };

    /**
     * The default texts that are inserted, if the place holder drawing with click button is empty.
     * Info: These strings need to be used by the filter.
     */
    PresentationUtils.PLACEHOLDER_TEMPLATE_TEXTS_CLICK = {
        pic: gt('Click icon to add image'),
        tbl: gt('Click icon to add table'),
        chart: gt('Click icon to add chart'),
        dgm: gt('Click icon to add smart art graphic'),
        clipArt: gt('Click icon to add clip art'),
        media: gt('Click icon to add media')
    };

    /**
     * The supported buttons inside a place holder drawing.
     */
    PresentationUtils.PLACEHOLDER_TEMPLATE_BUTTONS = {
        contentBody: ['picture', 'table'],
        body: null,
        title: null,
        ctrTitle: null,
        subTitle: null,
        media: ['media'],
        clipArt: ['clipart'],
        chart: ['chart'],
        dgm: ['smartart'],
        tbl: ['table'],
        pic: ['picture'],
        sldNum: null,
        ftr: null,
        dt: null
    };

    /**
     * The supported buttons inside a place holder drawing in ODF format.
     */
    PresentationUtils.PLACEHOLDER_TEMPLATE_BUTTONS_ODF = {
        contentBody: ['picture', 'table'],
        body: ['picture', 'table'],
        title: null,
        ctrTitle: null,
        subTitle: null,
        media: ['media'],
        clipArt: ['clipart'],
        chart: ['chart'],
        dgm: ['smartart'],
        tbl: ['table'],
        pic: ['picture'],
        sldNum: null,
        ftr: null,
        dt: null
    };

    /**
     * Supported names for inserting custom drawing placeholders.
     */
    PresentationUtils.PLACEHOLDER_DRAWING_NAMES = {
        content:    gt('Content placeholder'),
        contentv:   gt('Vertical content placeholder'),
        text:       gt('Text placeholder'),
        textv:      gt('Vertical text placeholder'),
        picture:    gt('Image placeholder'),
        table:      gt('Table placeholder')
    };

    /**
     * The template text inserted into supported placeholder drawings.
     */
    PresentationUtils.PLACEHOLDER_TEXT_CONTENT = [
        gt('Edit Master text styles'),
        gt('Second level'),
        gt('Third level'),
        gt('Fourth level'),
        gt('Fifth level')
    ];

    /**
     * The template text inserted into title placeholder drawing.
     */
    PresentationUtils.PLACEHOLDER_TITLE_CONTENT = gt('Click to edit Master title style');

    /**
     * The default index for place holders in presentation attributes.
     *
     * @constant
     */
    PresentationUtils.DEFAULT_PLACEHOLDER_INDEX = 0;

    // methods ----------------------------------------------------------------

    /**
     * Check, whether the specified attribute set (explicit attributes from a drawing) contain the presentation
     * place holder attributes. This means, that the phType is set, or that the phIndex is specified. It is possible,
     * that only the phIndex is specified. In this case the phType is defaulted to 'body'.
     *
     * @param {Object} attrs
     *  The attribute object. This should be the explicit attributes from a drawing.
     *
     * @returns {Boolean}
     *  Whether the specified attribute set contains presentation place holder attributes.
     */
    PresentationUtils.isPlaceHolderAttributeSet = function (attrs) {
        return !!(attrs && attrs.presentation && (attrs.presentation.phType || _.isNumber(attrs.presentation.phIndex)));
    };

    /**
     * Check, whether the specified attribute set (explicit attributes from a drawing) contain the presentation
     * place holder attributes of a title place holder drawing. This means, that the phType is set to 'title' or
     * 'ctrTitle'.
     *
     * @param {Object} attrs
     *  The attribute object. This should be the explicit attributes from a drawing.
     *
     * @returns {Boolean}
     *  Whether the specified attribute set contains presentation place holder attributes of 'title' drawing.
     */
    PresentationUtils.isTitlePlaceHolderAttributeSet = function (attrs) {
        return !!(attrs && attrs.presentation && attrs.presentation.phType && PresentationUtils.TITLE_PLACEHOLDER_TYPES[attrs.presentation.phType]);
    };

    /**
     * Check, whether the specified attribute set (explicit attributes from a drawing) contain the presentation
     * place holder attributes of a title or subtitle place holder drawing. This means, that the phType is set to
     * 'title', 'ctrTitle' or 'subTitle'.
     *
     * @param {Object} attrs
     *  The attribute object. This should be the explicit attributes from a drawing.
     *
     * @returns {Boolean}
     *  Whether the specified attribute set contains presentation place holder attributes of 'title' or 'subtitle' drawing.
     */
    PresentationUtils.isTitleOrSubtitlePlaceHolderAttributeSet = function (attrs) {
        return !!(attrs && attrs.presentation && attrs.presentation.phType && PresentationUtils.TITLE_OR_SUBTITLE_PLACEHOLDER_TYPES[attrs.presentation.phType]);
    };

    /**
     * Check, whether the specified attribute set (explicit attributes from a drawing) contain the presentation
     * place holder attributes of a body place holder drawing. This means, that the phType is set to 'body' or
     * that it is not defined.
     *
     * @param {Object} attrs
     *  The attribute object. This should be the explicit attributes from a drawing.
     *
     * @returns {Boolean}
     *  Whether the specified attribute set contains presentation place holder attributes of 'body' drawing.
     */
    PresentationUtils.isBodyPlaceHolderAttributeSet = function (attrs) {
        return !!(attrs && attrs.presentation && _.isNumber(attrs.presentation.phIndex) && (attrs.presentation.phType === PresentationUtils.DEFAULT_PLACEHOLDER_TYPE || !attrs.presentation.phType));
    };

    /**
     * Check, whether the specified attribute set (explicit attributes from a drawing) contain the presentation
     * place holder attributes of a body place holder drawing for inserting text only. This means, that the
     * phType is set explicitely to 'body'.
     * Info: In ODF file format this is quite different, because there is no place holder of type body, that
     *       accepts only text input.
     *
     * @param {Object} attrs
     *  The attribute object. This should be the explicit attributes from a drawing.
     *
     * @param {Boolean} [isODF=false]
     *  Whether this is an ODF application. If not specified, the value is set to 'false'.
     *
     * @returns {Boolean}
     *  Whether the specified attribute set contains presentation place holder attributes of 'body' drawing
     *  that allows only text insertion.
     */
    PresentationUtils.isTextBodyPlaceHolderAttributeSet = function (attrs, isODF) {
        var odf = isODF || false;
        return !!(!odf && attrs && attrs.presentation && _.isNumber(attrs.presentation.phIndex) && (attrs.presentation.phType === PresentationUtils.DEFAULT_PLACEHOLDER_TYPE));
    };

    /**
     * Check, whether the specified attribute set (explicit attributes from a drawing) contain the presentation
     * place holder attributes of a body place holder drawing that allows any content (text, table, picture, ...).
     * This means, that the phType is explicitely undefined and the phIndex is explicitely defined.
     * Info: In ODF file format this is quite different. The phType is set to 'body' and it is not necessary,
     *       that a phIndex is specified.
     *
     * @param {Object} attrs
     *  The attribute object. This should be the explicit attributes from a drawing.
     *
     * @param {Boolean} [isODF=false]
     *  Whether this is an ODF application. If not specified, the value is set to 'false'.
     *
     * @returns {Boolean}
     *  Whether the specified attribute set contains presentation place holder attributes of 'body' drawing
     *  that allows input of any content (text, table, picture, ...).
     */
    PresentationUtils.isContentBodyPlaceHolderAttributeSet = function (attrs, isODF) {
        var odf = isODF || false;
        return !!(attrs && attrs.presentation && (_.isNumber(attrs.presentation.phIndex) || odf) && (!attrs.presentation.phType || (odf && attrs.presentation.phType === PresentationUtils.DEFAULT_PLACEHOLDER_TYPE)));
    };

    /**
     * Check, whether the specified attribute set (explicit attributes from a drawing) describes a place holder drawing
     * in which no text can be inserted.
     *
     * @param {Object} attrs
     *  The attribute object. This should be the explicit attributes from a drawing.
     *
     * @returns {Boolean}
     *  Whether the specified attribute set contains presentation place holder attributes of a place holder drawing, in
     *  that no text can be inserted.
     */
    PresentationUtils.isPlaceHolderWithoutTextAttributeSet = function (attrs) {
        return !!(attrs && attrs.presentation && attrs.presentation.phType && PresentationUtils.NO_TEXT_PLACEHOLDER_TYPE[attrs.presentation.phType]);
    };

    /**
     * Check, whether the specified attribute set describes a place holder drawing with user defined template text.
     *
     * @param {Object} attrs
     *  The attribute object. This might be the explicit attributes from a drawing.
     *
     * @returns {Boolean}
     *  Whether the specified attribute set contains presentation place holder attributes of a place holder drawing with
     *  user defined template text.
     */
    PresentationUtils.isCustomPromptPlaceHolderAttributeSet = function (attrs) {
        return !!(attrs && attrs.presentation && attrs.presentation.customPrompt);
    };

    /**
     * Check, whether the specified attribute set (explicit attributes from a drawing) contain the presentation
     * place holder attributes of a place holder drawing in that in ODF in the master view the content cannot be
     * modified.
     *
     * @param {Object} attrs
     *  The attribute object. This should be the explicit attributes from a drawing.
     *
     * @returns {Boolean}
     *  Whether the specified attribute set contains presentation place holder attributes of a drawing that cannot
     *  be deleted or whose content cannot be modified in ODF masterview.
     */
    PresentationUtils.isODFReadOnlyPlaceHolderAttributeSet = function (attrs) {
        return !!(attrs && attrs.presentation && attrs.presentation.phType && PresentationUtils.ODF_MASTER_READONLY_TYPES[attrs.presentation.phType]);
    };

    /**
     * Check, whether the specified attribute set (explicit attributes from a drawing) contain the presentation
     * place holder attributes of a body place holder drawing without specified index. This means, that the phType
     * is set to 'body' and the phIndex is not specified (used for ODF).
     *
     * @param {Object} attrs
     *  The attribute object. This should be the explicit attributes from a drawing.
     *
     * @returns {Boolean}
     *  Whether the specified attribute set contains presentation place holder attributes of 'body' drawing without
     *  specified index.
     */
    PresentationUtils.isBodyPlaceHolderAttributeSetWithoutIndex = function (attrs) {
        return !!(attrs && attrs.presentation && !_.isNumber(attrs.presentation.phIndex) && (attrs.presentation.phType === PresentationUtils.DEFAULT_PLACEHOLDER_TYPE));
    };

    /**
     * Checking, whether the specified attribute set (explicit attributes from a drawing) specifies
     * a place holder drawing whose size and/or position was modified by the user.
     * This check is important for inheritance from master slide in ODP documents.
     *
     * @param {Object} attrs
     *  The attribute object. This should be the explicit attributes from a drawing.
     *
     * @returns {Boolean}
     *  Whether the specified attribute set specifies a place holder drawing whose size
     *  and/or position was modified by the user..
     */
    PresentationUtils.isUserModifiedPlaceHolderAttributeSet = function (attrs) {
        return !!((attrs && attrs.presentation && attrs.presentation.userTransformed) || false);
    };

    /**
     * Check, whether the specified drawing is a drawing that contains place holder attributes. This means, that
     * the phType is set, or that the phIndex is specified. It is possible, that only the phIndex is specified. In
     * this case the phType is defaulted to 'body'.
     *
     * @param {HTMLElement|jQuery} drawing
     *  The drawing node.
     *
     * @returns {Boolean}
     *  Whether the specified drawing contains presentation place holder attributes.
     */
    PresentationUtils.isPlaceHolderDrawing = function (drawing) {
        return PresentationUtils.isPlaceHolderAttributeSet(AttributeUtils.getExplicitAttributes(drawing));
    };

    /**
     * Helper function to get the place holder type from a specified drawing. If it cannot be determined,
     * null is returned.
     *
     * @param {HTMLElement|jQuery} drawing
     *  The drawing node.
     *
     * @param {Object} [drawingAttrs]
     *  The explicit drawing attributes. They can be specified for performance reasons.
     *
     * @returns {String|Null}
     *  The place holder type or null if it cannot be determined.
     */
    PresentationUtils.getPlaceHolderDrawingType = function (drawing, drawingAttrs) {

        var // the explicit drawing attributes
            attrs = drawingAttrs || AttributeUtils.getExplicitAttributes(drawing);

        return (attrs && attrs.presentation && attrs.presentation.phType) ? attrs.presentation.phType : null;
    };

    /**
     * Helper function to get the place holder index from a specified drawing. If it cannot be determined,
     * null is returned.
     *
     * @param {HTMLElement|jQuery} drawing
     *  The drawing node.
     *
     * @param {Object} [drawingAttrs]
     *  The explicit drawing attributes. They can be specified for performance reasons.
     *
     * @returns {Number|Null}
     *  The place holder index or null if it cannot be determined.
     */
    PresentationUtils.getPlaceHolderDrawingIndex = function (drawing, drawingAttrs) {

        var // the explicit drawing attributes
            attrs = drawingAttrs || AttributeUtils.getExplicitAttributes(drawing);

        return (attrs && attrs.presentation && _.isNumber(attrs.presentation.phIndex)) ? attrs.presentation.phIndex : null;
    };

    /**
     * Check, whether the specified drawing is a title drawing that contains place holder attributes. This means, that
     * the phType is set to 'title' or 'ctrTitle'.
     *
     * @param {HTMLElement|jQuery} drawing
     *  The drawing node.
     *
     * @returns {Boolean}
     *  Whether the specified drawing contains presentation place holder attributes of type 'title'.
     */
    PresentationUtils.isTitlePlaceHolderDrawing = function (drawing) {
        return PresentationUtils.isTitlePlaceHolderAttributeSet(AttributeUtils.getExplicitAttributes(drawing));
    };

    /**
     * Check, whether the specified drawing is a title or a subtitle drawing that contains place holder attributes.
     * This means, that the phType is set to 'title', 'ctrTitle' or 'subTitle'.
     *
     * @param {HTMLElement|jQuery} drawing
     *  The drawing node.
     *
     * @returns {Boolean}
     *  Whether the specified drawing contains presentation place holder attributes of type 'title' or 'subTitle'.
     */
    PresentationUtils.isTitleOrSubtitlePlaceHolderDrawing = function (drawing) {
        return PresentationUtils.isTitleOrSubtitlePlaceHolderAttributeSet(AttributeUtils.getExplicitAttributes(drawing));
    };

    /**
     * Check, whether the specified drawing is a body drawing that contains place holder attributes. This means, that
     * the phType is set to 'body or is not specified and the the phIndex is specified.
     *
     * @param {HTMLElement|jQuery} drawing
     *  The drawing node.
     *
     * @returns {Boolean}
     *  Whether the specified drawing contains presentation place holder attributes of type 'body'.
     */
    PresentationUtils.isBodyPlaceHolderDrawing = function (drawing) {
        return PresentationUtils.isBodyPlaceHolderAttributeSet(AttributeUtils.getExplicitAttributes(drawing));
    };

    /**
     * Check, whether the specified drawing is a body drawing that contains place holder attributes. This means, that
     * the phType is set explicitely to 'body'. In this case, only text content can be inserted into the
     * place holder drawing.
     *
     * @param {HTMLElement|jQuery} drawing
     *  The drawing node.
     *
     * @returns {Boolean}
     *  Whether the specified drawing contains presentation place holder attributes of type 'body' that allows
     *  only text insertion.
     */
    PresentationUtils.isTextBodyPlaceHolderDrawing = function (drawing) {
        return PresentationUtils.isTextBodyPlaceHolderAttributeSet(AttributeUtils.getExplicitAttributes(drawing));
    };

    /**
     * Check, whether the specified drawing is a body drawing that contains place holder attributes. This means, that
     * the phType is explicitely undefined. In this case, any content can be inserted into the place holder drawing
     * (text, table, picture, ...).
     *
     * @param {HTMLElement|jQuery} drawing
     *  The drawing node.
     *
     * @returns {Boolean}
     *  Whether the specified drawing contains presentation place holder attributes of type 'body' that allow input
     *  of any content into the 'body'.
     */
    PresentationUtils.isContentBodyPlaceHolderDrawing = function (drawing) {
        return PresentationUtils.isContentBodyPlaceHolderAttributeSet(AttributeUtils.getExplicitAttributes(drawing));
    };

    /**
     * Check, whether the specified drawing is a place holder drawing, in that no text can be inserted. In this
     * case it contains no paragraphs (in master/layout view a paragraph is inserted, but not in the document
     * view). Instead a button is available that can be used to insert a table or an image.
     *
     * @param {HTMLElement|jQuery} drawing
     *  The drawing node.
     *
     * @returns {Boolean}
     *  Whether the specified drawing is a place holder drawing, that does not support text input.
     */
    PresentationUtils.isPlaceHolderWithoutTextDrawing = function (drawing) {
        return PresentationUtils.isPlaceHolderWithoutTextAttributeSet(AttributeUtils.getExplicitAttributes(drawing));
    };

    /**
     * Check, whether the specified drawing is an empty place holder drawing. For all types of place
     * holders it is checked, if the place holder contains the possible content.
     *
     * @param {HTMLElement|jQuery} drawing
     *  The drawing node.
     *
     * @returns {Boolean}
     *  Whether the specified drawing is an empty place holder drawing.
     */
    PresentationUtils.isEmptyPlaceHolderDrawing = function (drawing) {

        // the explicit attributes of the drawing
        var attrs = AttributeUtils.getExplicitAttributes(drawing);
        // whether this is a place holder that does not contain text and that contains no content
        var isEmpty = false;
        // the selector to find a node that only exists, if the place holder is not empty
        var attributesProperty = null;

        // First check: Has the place holder a valid type
        if (PresentationUtils.isPlaceHolderWithoutTextAttributeSet(attrs)) {

            if ($(drawing).attr('data-type') !== 'undefined') {
                // type: 'undefined' : handling place holder drawings with unknown content -> they are never empty
                // -> otherwise checking the explicit attributes to find out, if the drawing contains content like a table or an image
                attributesProperty = PresentationUtils.NO_TEXT_PLACEHOLDER_CONTENT_PROPERTY[attrs.presentation.phType];

                // ... and is the place holder empty (or contains only the template text
                if (!attrs || !attributesProperty || !attrs[attributesProperty]) {
                    isEmpty = true;
                }

            }
        } else if (PresentationUtils.isPlaceHolderAttributeSet(attrs)) {
            // handling of those place holders, that allow text input (text body place holders and content body place holders)
            // but also title, ctrTitle, ...
            isEmpty = DOM.isEmptyTextframe(drawing, { ignoreTemplateText: true });
        }

        return isEmpty;
    };

    /**
     * Check, whether the specified drawing is a place holder drawing that contains a user
     * specified template text.
     *
     * @param {HTMLElement|jQuery} drawing
     *  The drawing node.
     *
     * @returns {Boolean}
     *  Whether the specified drawing is a place holder drawing with user defined template text.
     */
    PresentationUtils.isCustomPromptPlaceHolderDrawing = function (drawing) {
        return PresentationUtils.isCustomPromptPlaceHolderAttributeSet(AttributeUtils.getExplicitAttributes(drawing));
    };

    /**
     * Check, whether the specified drawing is a place holder drawing that cannot be deleted or whose content
     * cannot be modified in master view in ODF files.
     *
     * @param {HTMLElement|jQuery} drawing
     *  The drawing node.
     *
     * @returns {Boolean}
     *  Whether the specified drawing is a place holder drawing that cannot be deleted or whose content
     *  cannot be modified in master view in ODF files.
     */
    PresentationUtils.isODFReadOnlyPlaceHolderDrawing = function (drawing) {
        return PresentationUtils.isODFReadOnlyPlaceHolderAttributeSet(AttributeUtils.getExplicitAttributes(drawing));
    };

    /**
     * Check, whether the specified drawing is a place holder drawing with explicitely set phType to 'body' and
     * undefined 'phIndex' (used for ODF).
     *
     * @param {HTMLElement|jQuery} drawing
     *  The drawing node.
     *
     * @returns {Boolean}
     *  Whether the specified drawing is a place holder drawing with explicitely set phType to 'body' and
     *  undefined 'phIndex'.
     */
    PresentationUtils.isBodyPlaceHolderWithoutIndexDrawing = function (drawing) {
        return PresentationUtils.isBodyPlaceHolderAttributeSetWithoutIndex(AttributeUtils.getExplicitAttributes(drawing));
    };

    /**
     * Checking, whether the specified drawing is a place holder drawing whose size
     * and/or position was modified by the user.
     * This check is important for inheritance from master slide in ODP documents.
     *
     * @param {HTMLElement|jQuery} drawing
     *  The drawing node.
     *
     * @returns {Boolean}
     *  Whether the specified drawing is a place holder drawing whose size
     *  and/or position was modified by the user.
     */
    PresentationUtils.isUserModifiedPlaceHolderDrawing = function (drawing) {
        return PresentationUtils.isUserModifiedPlaceHolderAttributeSet(AttributeUtils.getExplicitAttributes(drawing));
    };

    /**
     * Checking, whether the specified drawing requires the property 'userTransformed'.
     * This is the case of place holder drawings in ODF, that are moved or resized
     * by the user.
     * This check is important for inheritance from master slide in ODP documents.
     * This check can be used after a drawing was moved or resized by the user, so
     * that a valid operation can be generated.
     *
     * @param {HTMLElement|jQuery} drawing
     *  The drawing node.
     *
     * @returns {Boolean}
     *  Whether the specified drawing requires the property 'userTransformed'. This is
     *  the case, if it is a place holder drawing whose size and/or position was modified
     *  by the user.
     */
    PresentationUtils.requiresUserTransformedProperty = function (drawing) {
        var attrs = AttributeUtils.getExplicitAttributes(drawing);
        return PresentationUtils.isPlaceHolderAttributeSet(attrs) && !PresentationUtils.isUserModifiedPlaceHolderAttributeSet(attrs);
    };

    /**
     * Getting an object containing the keys 'target', 'type' and 'index' for a specified drawing. This
     * object can be used to get attributes and list styles from the model for the drawing
     *
     * Info: Place holder drawings cannot be grouped, they are always direct children of a slide.
     *
     * @param {HTMLElement|jQuery} drawing
     *  The drawing node.
     *
     * @argument {Object} options
     *  Optional parameters:
     *  @param {Boolean} [useDefaultValues=true]
     *      If set to true, the default values for drawing place holder type and index are used.
     *      Otherwise the explicit attributes for type and index are inserted into the return object,
     *      but this might not be defined.
     *
     * @returns {Object}
     *  An object describing 'target' (the ID of the slide), 'type' (the type of the drawing) and
     *  'index' of a specified drawing.
     */
    PresentationUtils.getDrawingModelDescriptor = function (drawing, options) {

        var // the explicit attributes set at the drawing
            drawingAttrs = AttributeUtils.getExplicitAttributes(drawing),
            // whether the default values for type and index shall be returned
            useDefaultValues = Utils.getBooleanOption(options, 'useDefaultValues', true),
            // the return object containing keys 'target', 'type' and 'index'
            drawingObj = null;

        if (PresentationUtils.isPlaceHolderAttributeSet(drawingAttrs)) {

            drawingObj = {};

            drawingObj.target = DOM.getTargetContainerId($(drawing).parent()); // place holder drawings are direct children of slide
            drawingObj.type = drawingAttrs.presentation.phType;
            drawingObj.index = drawingAttrs.presentation.phIndex;

            if (useDefaultValues) {
                if (drawingObj.type && !_.isNumber(drawingObj.index)) { drawingObj.index = PresentationUtils.DEFAULT_PLACEHOLDER_INDEX; }
                if (_.isNumber(drawingObj.index) && !drawingObj.type) { drawingObj.type = PresentationUtils.DEFAULT_PLACEHOLDER_TYPE; }
            }
        }

        return drawingObj;
    };

    /**
     * Receiving a valid set for the properties 'type' and 'index', so that the model can be accessed.
     * Typically the specified attributes parameter is the set of explicit drawing attributes.
     *
     * @param {Object} attrs
     *  The attributes object. This are typically the explicit attributes set at a drawing.
     *
     * @returns {Object|Null}
     *  An object containing the properties 'type' and 'index'. 'type' is a string, that describes
     *  the place holder type, 'index' is a number describing the index specified for the given
     *  place holder type.
     *  If the specified attribute does not contain a 'presentation' property or inside this
     *  'presentation' property neither 'phType' nor 'phIndex' are defined, null is returned.
     */
    PresentationUtils.getValidTypeIndexSet = function (attrs) {

        // it is necessary that at least one of phType or phIndex is defined in the attributes
        if (!attrs || !attrs.presentation || !('phType' in attrs.presentation || 'phIndex' in attrs.presentation)) {
            return null;
        }

        return {
            type: Utils.getStringOption(attrs.presentation, 'phType', PresentationUtils.DEFAULT_PLACEHOLDER_TYPE),
            index: Utils.getIntegerOption(attrs.presentation, 'phIndex', PresentationUtils.DEFAULT_PLACEHOLDER_INDEX)
        };
    };

    /**
     * Getting the default text for a place holder drawing. This text is dependent from the place holder type.
     *
     * @param {HTMLElement|jQuery} drawing
     *  The drawing node.
     *
     * @param {Boolean} [isODF=false]
     *  Whether this is an ODF application. If not specified, the value is set to 'false'.
     *
     * @returns {String|Null}
     *  The default text for a place holder drawing. Or null, if it cannot be determined.
     */
    PresentationUtils.getPlaceHolderTemplateText = function (drawing, isODF) {

        var // the place holder type
            type = PresentationUtils.getPlaceHolderDrawingType(drawing),
            // whether this is an ODF application.
            odf = isODF || false;

        if (!type && _.isNumber(PresentationUtils.getPlaceHolderDrawingIndex(drawing))) {
            type = PresentationUtils.DEFAULT_PLACEHOLDER_TYPE;
        }

        return type ? (odf ? PresentationUtils.PLACEHOLDER_TEMPLATE_TEXTS_ODF[type] : PresentationUtils.PLACEHOLDER_TEMPLATE_TEXTS[type]) : null;
    };

    /**
     * Getting a list of strings, that describe the required buttons for a specified place holder type.
     * If the type is the empty string, it is defaulted to the place holder content body.
     *
     * @param {type} String
     *  The string describing the place holder type. The content body type must use the empty string.
     *
     * @param {Boolean} [isODF=false]
     *  Whether this is an ODF application. If not specified, the value is set to 'false'.
     *
     * @returns {Array|Null}
     *  An array containing strings for the required buttons for the specified place holder type.
     */
    PresentationUtils.getPlaceHolderTemplateButtonsList = function (type, isODF) {
        var buttonMap = isODF ? PresentationUtils.PLACEHOLDER_TEMPLATE_BUTTONS_ODF : PresentationUtils.PLACEHOLDER_TEMPLATE_BUTTONS;
        return buttonMap[type || 'contentBody']; // using special name for content body place holders
    };

    /**
     * Getting the default representation string for fields in footer place holder types in ODF.
     *
     * @param {type} String
     *  The string describing the place holder type.
     *
     * @returns {String}
     *  The default representation string for fields in footer place holder types in ODF. Or an
     *  empty string, if it cannot be determined.
     */
    PresentationUtils.getODFFooterRepresentation = function (type) {
        var representationContent = PresentationUtils.PLACEHOLDER_TEMPLATE_TEXTS_ODF[type];
        return representationContent ? ('<' + representationContent + '>') : '';
    };

    /**
     * This function converts the properties of a rectangle object ('top', 'left',
     * 'width' and 'height'). Specified is a rectangle that has values in pixel
     * relative to the app-content-root node. This is for example created by the
     * selectionBox. All properties are converted into 1/100 mm relative to the
     * slide. Therefore the returned values can directly be used for the creation
     * of operations. The zoom level is also taken into account.
     *
     * @param {Application} app
     *  The application containing this instance.
     *
     * @param {Object} box
     *  The rectangle with the properties 'top', 'left', 'width' and 'height'.
     *  The values are given in pixel relative to the app-content-root node.
     *
     * @param {Boolean} [notScaleBoxSize=false]
     *  Optional flag to prevent the scaling from the box size. The default is false,
     *  so when it's not provided the box is not scaled. This is useful when a box is
     *  not based on the px size from the selection box (created with the mouse), but
     *  with fixed values. These fixed values are not depended on the scaling and
     *  therefore must not be scaled.
     *
     *
     * @returns {Object}
     *  The rectangle with the properties 'top', 'left', 'width' and 'height'.
     *  The values are given in 1/100 mm relative to the slide.
     */
    PresentationUtils.convertAppContentBoxToOperationBox = function (app, box, notScaleBoxSize) {

        var // the current zoom factor
            zoomFactor = app.getView().getZoomFactor(),
            // the active slide node
            activeSlide = app.getModel().getSlideById(app.getModel().getActiveSlideId()),
            // the offset of the active slide inside the content root node
            slideOffset = activeSlide.offset(), // the slide is positioned absolutely
            // the content root node, the base for the position calculation
            contentRootNode = app.getView().getContentRootNode(),
            // the position of the content root node
            pos = contentRootNode.offset(),
            // the horizontal scroll shift
            scrollLeft = contentRootNode.scrollLeft(),
            // the vertical scroll shift
            scrollTop = contentRootNode.scrollTop(),
            // the new box object
            operationBox = {},
            // flag if the width/height from the box should be scaled by the zoomFactor or not
            notScaleBoxSizeFlag = _.isBoolean(notScaleBoxSize) ? notScaleBoxSize : false;

        // the left position of the user defined box relative to the slide (can be negative)
        operationBox.left = Utils.round((box.left - (slideOffset.left - pos.left) - scrollLeft) / zoomFactor, 1);
        // the top position of the user defined box relative to the slide (can be negative)
        operationBox.top = Utils.round((box.top - (slideOffset.top - pos.top) - scrollTop) / zoomFactor, 1);
        // the width of the box
        operationBox.width = notScaleBoxSizeFlag ? box.width : Utils.round(box.width / zoomFactor, 1);
        // the height of the box
        operationBox.height = notScaleBoxSizeFlag ? box.height : Utils.round(box.height / zoomFactor, 1);

        // converting to 1/100 mm
        operationBox.left = Utils.convertLengthToHmm(operationBox.left, 'px');
        operationBox.top = Utils.convertLengthToHmm(operationBox.top, 'px');
        operationBox.width = Utils.convertLengthToHmm(operationBox.width, 'px');
        operationBox.height = Utils.convertLengthToHmm(operationBox.height, 'px');

        //workaround for Bug 52977, TODO: opBox is broken when zoomed out!!!
        if (operationBox.left < 0) { operationBox.left = 0; }

        return operationBox;
    };

    /**
     * Helper function, for extending operation with the drawing-attributes listed
     * in PresentationUtils.PLACEHOLDER_DRAWING_IMPORTANT_ATTRS.
     *
     * @param {Application} app
     *  The application containing this instance.
     *
     * @param {HTMLElement|jQuery} drawingNode
     *  The drawing node to be selected, as DOM node or jQuery object.
     *
     * @param {Object} operationProperties
     *  the operation properties to generate operations. 'attrs.drawing' must be set
     */
    PresentationUtils.extendPlaceholderProp = function (app, drawingNode, operationProperties) {
        if (operationProperties && operationProperties.attrs && operationProperties.attrs.drawing) {
            var expAttrs = AttributeUtils.getExplicitAttributes(drawingNode);
            if (PresentationUtils.isPlaceHolderAttributeSet(expAttrs)) {
                var mergedDrawing = app.getModel().getDrawingStyles().getElementAttributes(drawingNode).drawing;
                _.each(mergedDrawing, function (mAttr, mKey) {

                    if (!(mKey in operationProperties.attrs.drawing) &&
                        _.contains(PresentationUtils.PLACEHOLDER_DRAWING_IMPORTANT_ATTRS, mKey) &&
                        !(mKey in expAttrs.drawing)) {
                        operationProperties.attrs.drawing[mKey] = mAttr;
                    }

                });
            }
        }
    };

    // exports ================================================================

    return PresentationUtils;

});
