/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Stefan Eckert <stefan.eckert@open-xchange.com>
 */

define('io.ox/office/presentation/model/dynfontsizemixin', [
    'io.ox/office/tk/utils',
    'io.ox/office/drawinglayer/view/drawingframe',
    'io.ox/office/editframework/utils/attributeutils',
    'io.ox/office/textframework/utils/dom',
    'io.ox/office/textframework/utils/position',
    'io.ox/office/textframework/utils/operations',
    'io.ox/office/presentation/utils/presentationutils'
], function (Utils, DrawingFrame, AttributeUtils, DOM, Position, Operations, PresentationUtils) {

    'use strict';

    var DEFAULT_FONT_SIZE = 16;
    var MIN_FONT_SIZE = 4;

    var INDEX_ARRAY = [0, 1, 2, 3, 4, 5];

    // mix-in class DynFontSizeMixin ==========================================

    function DynFontSizeMixin(app) {

        // self reference (the document model)
        var self = this;

        // all drawings nodes that need to be updated
        var drawingQueue = $();

        // private methods ----------------------------------------------------

        function updateStep(textframecontent, textframe, maxHeight, lastHeight, i) {
            var currentFontSize = parseFloat(textframecontent.css('font-size'));
            var lineReduction = parseFloat(textframecontent.attr('linereduction')) || 0;

            // dont use jquery postion, because it is cached!!!
            // var currentHeight = lastP.offsetTop + lastP.offsetHeight;

            var currentHeight = 0;
            var paraChildren = textframe.children().get();
            paraChildren.forEach(function (para, index) {
                para = $(para);
                // fix for Bug 47960
                if (index === paraChildren.length - 1 && (para.hasClass(DOM.PARAGRAPH_NODE_LIST_EMPTY_CLASS) || para.find('>span:empty').length)) {
                    return;
                } else {
                    currentHeight += para.outerHeight(true);
                }
            });

            /*
            console.log('iter', i, 'currentHeight', currentHeight, 'maxHeight', maxHeight, 'currentFontSize', currentFontSize);
            if (i === 6) {
                return null;
            }
            */
            if (currentHeight < maxHeight && currentFontSize === DEFAULT_FONT_SIZE && lineReduction === 0) {
                //all is fine
                // console.log('all is fine');
                return null;
            } else if (currentHeight < maxHeight * 1.1 && currentHeight > maxHeight * 0.90) {
                // all is scaled but fine in 90% - 110%
                // console.log('all is scaled but fine');
                return null;
            } else {
                var diff = maxHeight / currentHeight;

                if (currentFontSize <= 4 && diff <= 1) {
                    // console.log('early cut out');
                    return null;
                } else {

                    if (diff > 1) {
                        diff = diff * 0.5 + 0.5;
                    } else if (lastHeight === currentHeight && diff > 0.5) {
                        diff = diff * 0.5 + 0.25;
                    }
                    // console.log('have to scale', diff, i);

                    var newFont = diff * currentFontSize;
                    var newLineReduction = lineReduction;
                    if (diff < 1) {
                        newLineReduction += 8;
                        newLineReduction = Math.min(20, newLineReduction);
                    }

                    if (i === 0 && (currentFontSize !== DEFAULT_FONT_SIZE || lineReduction !== 0)) {
                        newFont = DEFAULT_FONT_SIZE;
                        newLineReduction = 0;
                    }

                    newFont = Math.min(newFont, DEFAULT_FONT_SIZE);
                    newFont = Math.max(newFont, MIN_FONT_SIZE);
                    newLineReduction = Math.min(newLineReduction, 20);

                    textframecontent.css('font-size', newFont + 'px');
                    textframecontent.attr('font-scale', newFont / DEFAULT_FONT_SIZE);
                    textframecontent.attr('linereduction', newLineReduction);
                }
            }
            return currentHeight;
        }

        function updateDrawing(drawing) {
            var $drawing = $(drawing);
            var textframecontent = $drawing.find('>.textframecontent');
            var initialFontSize = parseFloat(textframecontent.css('font-size'));
            var initialLineReduction = parseFloat(textframecontent.attr('linereduction'));
            var textFrame = textframecontent.find('>.textframe');
            var drawingHeight = null;
            var currentHeight = null;
            var drawingWidth = null;
            var currentWidth = null;
            var drawingAttrs = null;
            var angle = DrawingFrame.getDrawingRotationAngle(self, drawing);

            // local method to get left position value for drawing with auto resize and no wrap word,
            // depending on alignment of first paragraph child
            function getLeftPostion() {
                var pAlignment = null;
                var leftPos = null;

                if (_.isNumber(angle) && angle % 360 !== 0) {
                    // if rotated, treat like center alignment
                    leftPos = parseFloat(drawing.style.left) - (currentWidth - drawingWidth) / 2;
                } else {
                    pAlignment = AttributeUtils.getExplicitAttributes($drawing.find('.p').first());
                    pAlignment = pAlignment && pAlignment.paragraph && pAlignment.paragraph.alignment;

                    switch (pAlignment) {
                        case 'left':
                            leftPos = parseFloat(drawing.style.left);
                            break;
                        case 'center':
                        case 'justify':
                            leftPos = parseFloat(drawing.style.left) - (currentWidth - drawingWidth) / 2;
                            break;
                        case 'right':
                            leftPos = parseFloat(drawing.style.left) - (currentWidth - drawingWidth);
                            break;
                        default:
                            leftPos = parseFloat(drawing.style.left);
                    }
                }

                return Utils.convertLengthToHmm(leftPos, 'px');
            }

            // local method to get top position value for drawing with auto resize and wrap word,
            // depending on shape vertical alignment
            function getTopPostion() {
                var vAlignment = null;
                var topPos = null;

                if (_.isNumber(angle) && angle % 360 !== 0) {
                    // if rotated, treat like centered vertical alignment
                    topPos = parseFloat(drawing.style.top) - (currentHeight - drawingHeight) / 2;
                } else {
                    vAlignment = AttributeUtils.getExplicitAttributes($drawing);
                    vAlignment = vAlignment && vAlignment.shape && vAlignment.shape.anchor;

                    switch (vAlignment) {
                        case 'top':
                            topPos = parseFloat(drawing.style.top);
                            break;
                        case 'centered':
                            topPos = parseFloat(drawing.style.top) - (currentHeight - drawingHeight) / 2;
                            break;
                        case 'bottom':
                            topPos = parseFloat(drawing.style.top) - (currentHeight - drawingHeight);
                            break;
                        default:
                            topPos = parseFloat(drawing.style.top);
                    }
                }

                return Utils.convertLengthToHmm(topPos, 'px');
            }

            if (!textFrame.children().length) {
                //empty drawing
                return;
            }

            // console.log('updateDrawing', textframecontent.attr('class'));
            if (textframecontent.hasClass('autoresizeheight') || !textframecontent.hasClass('autoresizetext')) {

                if (initialFontSize !== DEFAULT_FONT_SIZE || initialLineReduction !== 0) {

                    textframecontent.css('font-size', DEFAULT_FONT_SIZE + 'px');
                    textframecontent.attr('font-scale', null);
                    textframecontent.attr('linereduction', null);
                }
                if (textframecontent.hasClass('autoresizeheight')) {

                    drawingHeight = $drawing.data('drawingHeight') || 0;
                    currentHeight = $drawing.height();
                    drawingWidth = $drawing.data('drawingWidth') || 0;
                    currentWidth = $drawing.width();

                    if (textframecontent.hasClass(DrawingFrame.NO_WORDWRAP_CLASS) && Math.abs(drawingWidth - currentWidth) > 1) {
                        $drawing.data('drawingWidth', currentWidth);

                        drawingAttrs = {
                            height: Utils.convertLengthToHmm(currentHeight, 'px'),
                            width: Utils.convertLengthToHmm(currentWidth, 'px'),
                            left: getLeftPostion(),
                            top: Utils.convertLengthToHmm(parseFloat(drawing.style.top), 'px')
                        };
                        return $.when({ attrs: { drawing: drawingAttrs }, drawing: drawing });
                    } else if (Math.abs(drawingHeight - currentHeight) > 1) {

                        $drawing.data('drawingHeight', currentHeight);

                        drawingAttrs = {
                            height: Utils.convertLengthToHmm(currentHeight, 'px'),
                            width: Utils.convertLengthToHmm($drawing.width(), 'px'),
                            left: Utils.convertLengthToHmm(parseFloat(drawing.style.left), 'px'),
                            top: getTopPostion()
                        };
                        return $.when({ attrs: { drawing: drawingAttrs }, drawing: drawing });
                    }
                }
                return;
            }
            // fix for Bug 48665
            var maxHeight = $drawing.data('widthHeightRev') ? $drawing.width() : $drawing.height();
            maxHeight = maxHeight -/* (parseFloat(textFrame[0].style.top) || 0) -*/ (parseFloat(textFrame[0].style.paddingBottom) || 0);

            if (!maxHeight) {
                return;
            }
            var lastHeight = -1;

            function innerUpdate(i) {
                if (lastHeight === null) {
                    return;
                }
                lastHeight = updateStep(textframecontent, textFrame, maxHeight, lastHeight, i);
            }

            function returnFontSize() {
                var newFontScale = parseFloat(textframecontent.css('font-size'));
                var newLineReduction = parseFloat(textframecontent.attr('linereduction'));

                if (Math.abs(newFontScale - initialFontSize) < 0.001 && Math.abs(newLineReduction - initialLineReduction) < 0.001) {
                    return;
                }
                return {
                    attrs: {
                        shape: {
                            fontScale: newFontScale / DEFAULT_FONT_SIZE,
                            lineReduction: newLineReduction / 100,
                            autoResizeText: true,
                            autoResizeHeight: false,
                            noAutoResize: false
                        }
                    },
                    drawing: drawing
                };
            }

            var def = null;
            if (_.browser.IE) {
                def = self.iterateArraySliced(INDEX_ARRAY, innerUpdate, 'DynFontSizeMixin.updateDrawing', { slice: 1 });
            } else {
                _.each(INDEX_ARRAY, innerUpdate);
                def = $.when(true);
            }

            return def.then(returnFontSize);
        }

        function handleContainerVisibility(slide, invisibleWithReflow) {
            var makeVisible = !slide.hasClass('invisibleslide');
            self.handleContainerVisibility(slide, { invisibleWithReflow: invisibleWithReflow, makeVisible: makeVisible });
        }

        /**
         * Updating some specified drawings in a specified slide. During loading a document
         * the slides are formatted at all. In that context also the font size is set and it is not
         * necessary to specify the drawings, because all of them are updated.
         *
         * After the document is loaded, it is only necessary to update those drawings, who got new
         * attributes or whose content has changed. In this case, it is necessary to specify the
         * drawings, so that not all drawings of a slide need to be updated.
         *
         * @param {jQuery} slide
         *  The slide of the drawing
         *
         * @param {String} slideID
         *  The slideID of the slide
         *
         * @param {jQuery} [drawings]
         *  this drawings will be updated.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved or rejected after all drawings are updated.
         */
        function updateSlide(slide, slideID, drawings) {

            // ignoring layout or master slides
            if (!self.isStandardSlideId(slideID)) {
                return;
            }

            var defs = [];

            handleContainerVisibility(slide, true);

            _.each(drawings, function (drawing) {
                var def = updateDrawing(drawing);
                if (def) { defs.push(def); }
            });

            if (defs.length) {
                return $.when.apply($, defs).then(function () {
                    handleContainerVisibility(slide, false);
                    return arguments;
                });
            } else {
                handleContainerVisibility(slide, false);
            }
        }

        // The drawing element as DOM node or jQuery object.
        function registerDrawing(newDrawing) {
            // store the new drawing(s) in the collection (jQuery keeps the collection unique)
            // -> but not storing images or tables
            if (newDrawing && (DrawingFrame.isShapeDrawingFrame(newDrawing) || DrawingFrame.isGroupDrawingFrame(newDrawing)) && !DrawingFrame.isTableDrawingFrame(newDrawing)) {
                drawingQueue = drawingQueue.add(newDrawing);
            }
            // TODO: Remove asap check for tables (problem in DrawingFrame.isShapeDrawingFrame)
            // if (newDrawing && (DrawingFrame.isShapeDrawingFrame(newDrawing) || DrawingFrame.isGroupDrawingFrame(newDrawing))) { drawingQueue = drawingQueue.add(newDrawing); }
        }

        function updateDynFontSizeInDrawings() {
            var promise = updateDynFontSizeForDrawings(drawingQueue);
            drawingQueue = $();
            return promise;
        }

        // updating the specified drawings
        //
        // Info: This function handles only drawings on document slides, not on layout
        //       or master slides. Therefore the 'target' is always NOT defined.
        //
        //  Because of the 'target'
        function updateDynFontSizeForDrawings(drawings) {
            var defs = [];
            var slideID = null;
            // iterating over all drawings
            _.each(drawings, function (drawing) {
                // TODO: operate correct on grouped drawings (width/height are different to ungrouped drawings)
                // Also no text resize for 'title' place holders
                if ($(drawing).is('.grouped') || PresentationUtils.isTitlePlaceHolderDrawing(drawing)) { return; }

                var slide = $(drawing).closest(DOM.SLIDE_SELECTOR);
                slideID = self.getSlideId(slide);

                self.appendSlideToDom(slideID); // attaching the slide node to the DOM

                // skip slides if they are not formatted at all -> asking slide format manager
                if (!self.getSlideFormatManager().isUnformattedSlide(slideID)) {
                    var change = updateSlide(slide, slideID, $(drawing));
                    if (change) { defs.push(change); }
                }
            });

            if (defs.length) {
                return $.when.apply($, defs).then(function () {
                    //controller update must be called for changed font-size in the toolbar
                    app.getController().update();

                    if (!app.isEditable()) {
                        return;
                    }
                    var ops = [];

                    _.flatten(arguments).forEach(function (change) {
                        if (!change) {
                            return;
                        }

                        var position = Position.getOxoPosition(self.getRootNode(slideID), change.drawing);
                        ops.push({
                            noUndo: true,
                            name: Operations.SET_ATTRIBUTES,
                            start: position,
                            attrs: change.attrs,
                            target: self.getOperationTargetBlocker()
                        });

                        // Info: Adding the operationsTargetBlocker is necessary because the operation is not
                        //       executed on the active slide. Therefore the target must not be added automatically
                        //       by the finalizeOperations handler.
                        //       Because all affected slides are document slides, it is never necessary to add any
                        //       target. If this is required, this target must be set explicitely to the operation.
                    });
                    self.applyOperations(ops);

                });
            } else {
                return $.when();
            }
        }

        // public methods -----------------------------------------------------

        /**
         * Handling the property 'autoResizeHeight' of a selected text frame node.
         *
         * @param {Boolean} autoResizeHeight
         *  Whether the property 'autoResizeHeight' of a selected text frame shall be
         *  enabled or disabled.
         */
        self.handleTextFrameAutoFit = function (autoResizeHeight, autoResizeText) {

            var selection = self.getSelection();
            // the operations generator
            var generator = self.createOperationGenerator();
            // the options for the setAttributes operation
            var operationOptions = {};
            // a selected text frame node or the text frame containing the selection
            var textFrame = null;
            // a collector for all affected drawings in a multi drawing selection
            var textFrameCollector = null;
            // a selected drawing node
            var selectedDrawing = null;

            /**
             * Helper function to handle all shapes inside a specified drawing group node.
             */
            function handleShapesInGroup(groupNode) {

                _.each(DrawingFrame.getAllGroupDrawingChildren(groupNode), function (child) {

                    if (DrawingFrame.isTextFrameShapeDrawingFrame(child)) {

                        textFrame = $(child);

                        // collecting the attributes for the operation
                        operationOptions.attrs =  {};
                        operationOptions.attrs.shape = {};

                        if (!app.isODF()) { operationOptions.attrs.shape.autoResizeText   = autoResizeText || false; }
                        operationOptions.attrs.shape.autoResizeHeight = autoResizeHeight || false;
                        operationOptions.attrs.shape.noAutoResize = (!autoResizeText && !autoResizeHeight);

                        operationOptions.start = Position.getOxoPosition(self.getNode(), textFrame, 0);

                        if (autoResizeText || autoResizeHeight) { // Bug 48714 fix for undo
                            operationOptions.attrs.drawing = {};
                            PresentationUtils.extendPlaceholderProp(app, textFrame, operationOptions);
                        }

                        // generate the 'setAttributes' operation
                        generator.generateOperation(Operations.SET_ATTRIBUTES, operationOptions);

                        // collecting the affected drawings
                        textFrameCollector = textFrameCollector || [];
                        textFrameCollector.push(textFrame);
                    }
                });
            }

            // multiselection
            if (selection.isMultiSelectionSupported() && selection.isMultiSelection()) {

                _.each(selection.getMultiSelection(), function (drawingItem) {

                    var drawingNode = selection.getDrawingNodeFromMultiSelection(drawingItem);

                    // filter to modify only text frames
                    if (DrawingFrame.isTextFrameShapeDrawingFrame(drawingNode)) { // selected drawing text frame in a multi drawing selection

                        textFrame = drawingNode;

                        // collecting the attributes for the operation
                        operationOptions.attrs =  {};
                        operationOptions.attrs.shape = {};

                        if (!app.isODF()) { operationOptions.attrs.shape.autoResizeText   = autoResizeText || false; }
                        operationOptions.attrs.shape.autoResizeHeight = autoResizeHeight || false;
                        operationOptions.attrs.shape.noAutoResize = (!autoResizeText && !autoResizeHeight);
                        if (app.isODF()) { operationOptions.attrs.shape.wordWrap = true; } // odf requires wordWrap value, later this might be (autoResizeHeight && !autoResizeText)

                        operationOptions.start = drawingItem.startPosition;

                        if (autoResizeText || autoResizeHeight) { // Bug 48714 fix for undo
                            operationOptions.attrs.drawing = {};
                            PresentationUtils.extendPlaceholderProp(app, textFrame, operationOptions);
                        }

                        // generate the 'setAttributes' operation
                        generator.generateOperation(Operations.SET_ATTRIBUTES, operationOptions);

                        // collecting the affected drawings
                        textFrameCollector = textFrameCollector || [];
                        textFrameCollector.push(textFrame);

                    } else if (DrawingFrame.isGroupDrawingFrameWithShape(drawingNode)) { // selected drawing groups in a multi drawing selection
                        handleShapesInGroup(drawingNode);
                    }
                });

            } else if (selection.isAnyDrawingSelection()) {

                textFrame = selection.getAnyTextFrameDrawing({ forceTextFrame: true });

                // collecting the attributes for the operation
                operationOptions.attrs =  {};
                operationOptions.attrs.shape = {};

                if (!app.isODF()) { operationOptions.attrs.shape.autoResizeText   = autoResizeText || false; }
                operationOptions.attrs.shape.autoResizeHeight = autoResizeHeight || false;
                operationOptions.attrs.shape.noAutoResize = (!autoResizeText && !autoResizeHeight);
                if (app.isODF()) { operationOptions.attrs.shape.wordWrap = true; } // odf requires wordWrap value, later this might be (autoResizeHeight && !autoResizeText)

                if (selection.isAdditionalTextframeSelection()) {  // a text inside a text frame is selected
                    operationOptions.start = Position.getOxoPosition(self.getNode(), textFrame, 0);
                    if (autoResizeText || autoResizeHeight) { // Bug 48714 fix for undo
                        operationOptions.attrs.drawing = {};
                        PresentationUtils.extendPlaceholderProp(app, textFrame, operationOptions);
                    }
                    generator.generateOperation(Operations.SET_ATTRIBUTES, operationOptions); // generate the 'setAttributes' operation
                } else {
                    if (textFrame && textFrame.length > 0) { // the text frame itself is selected
                        operationOptions.start = selection.getStartPosition();
                        if (autoResizeText || autoResizeHeight) { // Bug 48714 fix for undo
                            operationOptions.attrs.drawing = {};
                            PresentationUtils.extendPlaceholderProp(app, textFrame, operationOptions);
                        }
                        generator.generateOperation(Operations.SET_ATTRIBUTES, operationOptions); // generate the 'setAttributes' operation
                    } else {
                        selectedDrawing = selection.getSelectedDrawing(); // this might be a group drawing, that is selected
                        if (DrawingFrame.isGroupDrawingFrameWithShape(selectedDrawing)) {
                            handleShapesInGroup(selectedDrawing);
                        }
                    }
                }

            }

            // apply all collected operations
            self.applyOperations(generator);

            if (textFrameCollector) {
                _.each(textFrameCollector, function (oneTextFrame) {
                    self.updateDynFontSizeDebounced(oneTextFrame);
                });
            } else {
                self.updateDynFontSizeDebounced(textFrame);
            }
        };

        self.updateDynFontSizeDebounced =  app.isODF() ? function () { return $.when(); } : self.createDebouncedMethod('DynfontSizeMixin.updateDynFontSizeDebounced', registerDrawing, updateDynFontSizeInDrawings, { delay: 500 });

        self.getPreFlushDocumentPromise = app.isODF() ? function () { return $.when(); } : function () { return updateDynFontSizeInDrawings(); };

        // initialization -----------------------------------------------------

    } // mixin DynFontSizeMixin

    // exports ================================================================

    return DynFontSizeMixin;
});
